/**************************************************************************//**
 * \file       clSDS_ReadSmsList.cpp
 *
 * See .cpp file for description.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "clSDS_ReadSmsList.h"
#include "application/clSDS_StringVarList.h"
#include "application/clSDS_SDSStatus.h"
#include "application/StringUtils.h"
#include "clSDS_ReadTextListObserver.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_ReadSmsList.cpp.trc.h"
#endif


#define MAX_SIZE_OF_SMS_LIST 100


using namespace MOST_Msg_FI;
using namespace most_Msg_fi_types;
using namespace clock_main_fi;
using namespace most_BTSet_fi_types_Extended;


/***********************************************************************//**
 *
 ***************************************************************************/
clSDS_ReadSmsList::clSDS_ReadSmsList(
   ::boost::shared_ptr< MOST_Msg_FIProxy > pSds2MsgProxy,
   ::boost::shared_ptr< ::MOST_PhonBk_FI::MOST_PhonBk_FIProxy> phonebookProxy,
   clSDS_FormatTimeDate* pFormatTimeDate,
   clSDS_SDSStatus* pSdsStatus)
   : _messageProxy(pSds2MsgProxy)
   , _phonebookProxy(phonebookProxy)
   , _pFormatTimeDate(pFormatTimeDate)
   , _smsListInfo()
   , _index(0)
   , _listHandle(0)
   , _listSize(0)
   , _deviceHandle(0)
   , _pReadTextListObserver(NULL)
   , _pSdsStatus(pSdsStatus)
{
   _pSdsStatus->vRegisterObserver(this);
}


/***********************************************************************//**
 *
 ***************************************************************************/
clSDS_ReadSmsList::~clSDS_ReadSmsList()
{
   _pFormatTimeDate = NULL;
   _pReadTextListObserver = NULL;
   _pSdsStatus = NULL;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::setTextListObserver(clSDS_ReadTextListObserver* obs)
{
   if (obs != NULL)
   {
      _pReadTextListObserver = obs;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _messageProxy)
   {
      _messageProxy->sendMessagingDeviceConnectionUpReg(*this);
      _messageProxy->sendMessageListChangeUpReg(*this);
   }

   if (proxy == _phonebookProxy)
   {
      _phonebookProxy->sendPreferredPhoneBookSortOrderUpReg(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _messageProxy)
   {
      _messageProxy->sendMessagingDeviceConnectionRelUpRegAll();
      _messageProxy->sendMessageListChangeRelUpRegAll();
   }

   if (proxy == _phonebookProxy)
   {
      _phonebookProxy->sendPreferredPhoneBookSortOrderRelUpRegAll();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onMessagingDeviceConnectionError(
   const ::boost::shared_ptr< MOST_Msg_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MessagingDeviceConnectionError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onMessagingDeviceConnectionStatus(
   const ::boost::shared_ptr< MOST_Msg_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MessagingDeviceConnectionStatus >& status)
{
   if (_deviceHandle == status->getU8DeviceHandle())
   {
      if (status->getBSMSSupport())
      {
         createMessageList();
      }
   }
   _deviceIDtoSMSAvailabilityMap[status->getU8DeviceHandle()] = status->getBSMSSupport();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::createMessageList()
{
   if (_messageProxy->isAvailable())
   {
      _messageProxy->sendCreateMessageListStart(*this, _deviceHandle, ::T_e8_MsgFolderType__e8MSG_FOLDER_INBOX, 0,
            ::T_e8_MsgMessageListType__e8MSG_LIST_SMS_MMS, ::T_e8_MsgMessageListSortType__e8MSG_LIST_SORT_TIMESTAMP,
            ::T_e8_MsgMessageListFilterType__e8MSG_LIST_FILTER_CURRENT_MAP_SESSION);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onCreateMessageListError(
   const ::boost::shared_ptr< MOST_Msg_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< ::CreateMessageListError >& /*error*/)
{
   _smsListInfo.clear();
   notifyListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onCreateMessageListResult(
   const ::boost::shared_ptr< MOST_Msg_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< ::CreateMessageListResult >& result)
{
   _listHandle = result->getU16ListHandle();
   _listSize = result->getU16ListLength() > MAX_SIZE_OF_SMS_LIST ? MAX_SIZE_OF_SMS_LIST : result->getU16ListLength();
   // update when change in list size
   if (_pReadTextListObserver != NULL)
   {
      _pReadTextListObserver->textListChanged(_listSize);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::requestSliceMessageList()
{
   if (_listHandle != 0)
   {
      _messageProxy->sendRequestSliceMessageListStart(*this, _listHandle, 0, _listSize);
   }
   else
   {
      notifyListObserver();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onRequestSliceMessageListError(
   const ::boost::shared_ptr< MOST_Msg_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< ::RequestSliceMessageListError >& /*error*/)
{
   _smsListInfo.clear();
   notifyListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onRequestSliceMessageListResult(
   const ::boost::shared_ptr< MOST_Msg_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< ::RequestSliceMessageListResult >& result)
{
   _smsListInfo.clear();
   const ::T_MsgMessageListSliceResult& resultList = result->getOMessageListSliceResult();
   struct smsInfo smsItem;
   for (std::vector< ::T_MsgMessageListSliceResultItem >::const_iterator itr = resultList.begin(); itr != resultList.end(); ++itr)
   {
      std::string formattedString("");
      smsItem.messageHandle.setU8DeviceHandle(itr->getOMessageHandle().getU8DeviceHandle());
      smsItem.messageHandle.setU32MsgHandleUpper(itr->getOMessageHandle().getU32MsgHandleUpper());
      smsItem.messageHandle.setU32MsgHandleLower(itr->getOMessageHandle().getU32MsgHandleLower());
      smsItem.readstatus = itr->getBReadStatusFlag();
      smsItem.subject = itr->getSSubject();
      _pFormatTimeDate->formatTimeDate(itr->getOMessageDateTime().getOMessageTime().getSHours(),
                                       itr->getOMessageDateTime().getOMessageTime().getSMinutes(),
                                       itr->getOMessageDateTime().getOMessageDate().getSCldrDay(),
                                       itr->getOMessageDateTime().getOMessageDate().getSCldrMonth(),
                                       itr->getOMessageDateTime().getOMessageDate().getSCldrYear(), formattedString);

      smsItem.formattedTimeDate = formattedString;
      std::string contactName = "";
      if (_phonebookProxy->hasPreferredPhoneBookSortOrder()
            && _phonebookProxy->getPreferredPhoneBookSortOrder().getE8PreferredPhoneBookSortOrder() == most_PhonBk_fi_types::T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_LASTNAME)
      {
         contactName = itr->getSLastName() + " " + itr->getSFirstName();
      }
      else
      {
         contactName = itr->getSFirstName() + " " + itr->getSLastName();
      }
      smsItem.contactName = StringUtils::trim(contactName);
      smsItem.phoneNumber = itr->getSPhoneNumber();
      _smsListInfo.push_back(smsItem);
   }
   notifyListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onReleaseMessageListError(
   const ::boost::shared_ptr< MOST_Msg_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< ReleaseMessageListError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onReleaseMessageListResult(
   const ::boost::shared_ptr< MOST_Msg_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< ReleaseMessageListResult >& /*result*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onMessageListChangeError(
   const ::boost::shared_ptr< MOST_Msg_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MessageListChangeError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onMessageListChangeStatus(
   const ::boost::shared_ptr< MOST_Msg_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MessageListChangeStatus >& status)
{
   if (status->getU16ListHandle() == _listHandle)
   {
      if ((status->getE8ListChangeType() == T_e8_MsgListChangeType__e8LCH_ITEMS_ADDED) ||
            (status->getE8ListChangeType() == T_e8_MsgListChangeType__e8LCH_ITEMS_REMOVED) ||
            (status->getE8ListChangeType() == T_e8_MsgListChangeType__e8LCH_CONTENT_CHANGED))
      {
         _listSize = status->getU16ListLength() > MAX_SIZE_OF_SMS_LIST ? MAX_SIZE_OF_SMS_LIST : status->getU16ListLength();
         if (_pReadTextListObserver != NULL)
         {
            _pReadTextListObserver->textListChanged(_listSize);
         }
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onPreferredPhoneBookSortOrderError(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::PreferredPhoneBookSortOrderError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::onPreferredPhoneBookSortOrderStatus(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::PreferredPhoneBookSortOrderStatus >& /*status*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
tU32 clSDS_ReadSmsList::u32GetSize()
{
   ETG_TRACE_USR1(("clSDS_ReadSmsList::u32GetSize(): %d", _smsListInfo.size()));
   return _smsListInfo.size();
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<clSDS_ListItems> clSDS_ReadSmsList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;
   for (tU32 u32Index = u32StartIndex; u32Index < u32EndIndex; u32Index++)
   {
      clSDS_ListItems oListItem;
      oListItem.oDescription.szString = oGetItem(u32Index);
      oListItem.oTime.szString = oGetDate(u32Index);
      oListItem.oMessage.szString = oGetMessage(u32Index);
      oListItem.oDirectionSymbol.szString = oGetMessageReadStatus(u32Index);
      oListItems.push_back(oListItem);
   }
   return oListItems;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_ReadSmsList::oGetDate(tU32 u32Index)
{
   std::string smsListElement = "";
   if (u32Index < _smsListInfo.size())
   {
      smsListElement =  _smsListInfo[u32Index].formattedTimeDate;
   }
   return smsListElement;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_ReadSmsList::oGetItem(tU32 u32Index)
{
   std::string smsListElement = "Unknown";
   if (u32Index < _smsListInfo.size())
   {
      if (!_smsListInfo[u32Index].contactName.empty())
      {
         smsListElement = _smsListInfo[u32Index].contactName;
      }
      else if (!_smsListInfo[u32Index].phoneNumber.empty())
      {
         smsListElement = _smsListInfo[u32Index].phoneNumber;
      }
   }
   ETG_TRACE_USR1(("clSDS_ReadSmsList::oListItem.oDesc: %s", smsListElement.c_str()));
   return smsListElement;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_ReadSmsList::oGetMessage(tU32 u32Index)
{
   std::string smsListElement = "";
   if (u32Index < _smsListInfo.size())
   {
      if (!_smsListInfo[u32Index].subject.empty())
      {
         smsListElement = _smsListInfo[u32Index].subject;
      }
      else
      {
         smsListElement = "  ";
      }
   }
   return smsListElement;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_ReadSmsList::oGetMessageReadStatus(tU32 u32Index)
{
   std::string smsListElement = "";
   if (u32Index < _smsListInfo.size())
   {
      tChar szTempString[3] = "";
      OSALUTIL_s32SaveNPrintFormat(szTempString, sizeof(szTempString), "%d", _smsListInfo[u32Index].readstatus);
      smsListElement = szTempString;
   }
   return smsListElement;
}


/***********************************************************************//**
 *
 ***************************************************************************/
tBool clSDS_ReadSmsList::bSelectElement(tU32 u32SelectedIndex)
{
   if (u32SelectedIndex > 0 && u32SelectedIndex <= _smsListInfo.size())
   {
      _index = u32SelectedIndex - 1;
      clSDS_StringVarList::vSetVariable("$(PredefinedMessageText)", Sds_TextDB_vGetText(SDS_TEXT_DRIVING_CAN_T_TEXT));
      return true;
   }
   else
   {
      return false;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_ReadSmsList::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType listType)
{
   if (listType == sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_PHONE_RECEIVED_SMS)
   {
      // reset index to zero
      _index = 0;
      requestSliceMessageList();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
unsigned int clSDS_ReadSmsList::getSelectedSmsIndex() const
{
   if (_index < _smsListInfo.size())
   {
      return (_index + 1);
   }
   return 0;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_ReadSmsList::getSelectedPhoneNumber() const
{
   if (_index < _smsListInfo.size())
   {
      return _smsListInfo[_index].phoneNumber;
   }
   return "";
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_ReadSmsList::getSelectedContactName() const
{
   if (_index < _smsListInfo.size())
   {
      return _smsListInfo[_index].contactName;
   }
   return "";
}


/***********************************************************************//**
 *
 ***************************************************************************/
T_MsgMessageHandle clSDS_ReadSmsList::getMessageHandle() const
{
   T_MsgMessageHandle messageHandle;
   if (_index < _smsListInfo.size())
   {
      return _smsListInfo[_index].messageHandle;
   }
   return messageHandle;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::selectNextMessage()
{
   if (_index < (_smsListInfo.size() - 1))
   {
      ++_index;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::selectPreviousMessage()
{
   if (_index > 0)
   {
      --_index;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_ReadSmsList::getTimeDate() const
{
   if (_index < _smsListInfo.size())
   {
      return _smsListInfo[_index].formattedTimeDate;
   }
   return "";
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::phoneStatusChanged(uint8 deviceHandle, most_BTSet_fi_types_Extended::T_e8_BTConnectionStatus status)
{
   if (status == T_e8_BTConnectionStatus__e8STATUS_CONNECTED)
   {
      ETG_TRACE_USR1(("clSDS_ReadSmsList::DeviceListExtendedStatus : T_e8_BTConnectionStatus__e8STATUS_CONNECTED"));
      if (_deviceHandle != deviceHandle)
      {
         _deviceHandle = deviceHandle;
         _listHandle = 0;
         _listSize = 0;
         std::map<uint16, bool>::iterator itr = _deviceIDtoSMSAvailabilityMap.find(_deviceHandle);
         if ((itr != _deviceIDtoSMSAvailabilityMap.end()) && (itr->second))
         {
            createMessageList();
         }
      }
   }
   if (status  == T_e8_BTConnectionStatus__e8STATUS_DISCONNECTED)
   {
      ETG_TRACE_USR1(("clSDS_ReadSmsList::DeviceListExtendedStatus : T_e8_BTConnectionStatus__e8STATUS_DISCONNECTED"));
      if (_messageProxy->isAvailable() && _listHandle)
      {
         _messageProxy->sendReleaseMessageListStart(*this, _listHandle);
         _deviceHandle = 0;
         _smsListInfo.clear();
         _listSize = 0;
         _listHandle = 0;
         if (_pReadTextListObserver != NULL)
         {
            _pReadTextListObserver->textListChanged(_listSize);
         }
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_ReadSmsList::vSDSStatusChanged()
{
   if (_pSdsStatus->getStatus() == clSDS_SDSStatus::EN_IDLE && _smsListInfo.size())
   {
      ETG_TRACE_USR1(("clSDS_ReadSmsList::clSDS_SDSStatus::EN_IDLE; clear list"));
      _smsListInfo.clear();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescription> clSDS_ReadSmsList::getHmiElementDescription(unsigned int index)
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> hmiElementDescriptionList;
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   if ((index > 0) && ((index - 1) < _smsListInfo.size()))
   {
      if (!_smsListInfo[index - 1].contactName.empty())
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_CONTACTNAME;
         hmiElementDescription.DescriptorValue = _smsListInfo[index - 1].contactName.c_str();
         hmiElementDescriptionList.push_back(hmiElementDescription);
      }
      else if (!_smsListInfo[index - 1].phoneNumber.empty())
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_PHONENUMBER;
         hmiElementDescription.DescriptorValue = _smsListInfo[index - 1].phoneNumber.c_str();
         hmiElementDescriptionList.push_back(hmiElementDescription);
      }
      else
      {
         ETG_TRACE_USR1(("clSDS_ReadSmsList::getHmiElementDescription : no ContactInfo"));
      }
   }
   return hmiElementDescriptionList;
}
