/**************************************************************************//**
 * \file       clSDS_POIList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_POIList.h"
#include "application/clSDS_StringVarList.h"
#include "view_db/Sds_ViewDB.h"


using namespace org::bosch::cm::navigation::NavigationService;


#define NUM_OF_DIGITS_IN_DIRECTION_VALUE 5
#define FIRST_CHARACTER_OF_STRING_IS_HOUSE_NUMBER 1


/**************************************************************************//**
*Destructor
******************************************************************************/
clSDS_POIList::~clSDS_POIList()
{
   _poiList.clear();
}


/**************************************************************************//**
*Constructor
******************************************************************************/
clSDS_POIList::clSDS_POIList(::boost::shared_ptr< NavigationServiceProxy > naviProxy): _naviProxy(naviProxy)
{
   _poiList.clear();
   _restoreHMIListResponse = true;
   _POIListViaHaptic = false;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_POIList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendSdsPOICategoryShortcutsRegister(*this);
      _naviProxy->sendSdsPOICategoryShortcutsGet(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_POIList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendSdsPOICategoryShortcutsDeregisterAll();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_POIList::onSdsPOICategoryShortcutsError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::SdsPOICategoryShortcutsError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_POIList::onSdsPOICategoryShortcutsUpdate(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::SdsPOICategoryShortcutsUpdate >& update)
{
   _sdsPOICategory.clear();
   _sdsPOICategory = update->getSdsPOICategoryShortcuts();
}


/***********************************************************************//**
 *
 ***************************************************************************/

void clSDS_POIList::setPOIList(const std::vector< AddressListElement >& oPOIList)
{
   _poiList = oPOIList;
}


/**************************************************************************//**
*
******************************************************************************/
tU32 clSDS_POIList::u32GetSize()
{
   return _poiList.size();
}


/**************************************************************************//**
*
******************************************************************************/
std::vector<clSDS_ListItems> clSDS_POIList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;
   for (tU32 u32Index = u32StartIndex; u32Index < std::min(u32EndIndex, u32GetSize()); u32Index++)
   {
      oListItems.push_back(oGetListItem(u32Index));
   }
   return oListItems;
}


/**************************************************************************//**
*
******************************************************************************/
clSDS_ListItems clSDS_POIList::oGetListItem(tU32 u32Index)
{
   clSDS_ListItems oListItem;
   oListItem.oDescription.szString = oGetItem(u32Index);
   oListItem.oDistance.szString = oGetDisatance(u32Index);

   tChar szTempString[NUM_OF_DIGITS_IN_DIRECTION_VALUE] = "";
   OSALUTIL_s32SaveNPrintFormat(szTempString, sizeof(szTempString), "%d", oGetDirectionFromMapping(u32Index));
   oListItem.oDirectionSymbol.szString = szTempString;

   return oListItem;
}


/**************************************************************************//**
*
******************************************************************************/
tBool clSDS_POIList::bSelectElement(tU32 u32SelectedIndex)
{
   if (u32SelectedIndex > 0)
   {
      clSDS_StringVarList::vSetVariable("$(POI_address)", oGetItem(u32SelectedIndex - 1));
      return TRUE;
   }
   return FALSE;
}


/**************************************************************************//**
*
******************************************************************************/
std::string clSDS_POIList::oGetItem(tU32 u32Index)
{
   if (u32Index < _poiList.size())
   {
      if (clSDS_ConfigurationFlags::getPOIAddressInfo())
      {
         std::string data = _poiList[u32Index].getData();
         std::string address = _poiList[u32Index].getAddress();
         if (std::isdigit(address[FIRST_CHARACTER_OF_STRING_IS_HOUSE_NUMBER]))
         {
            return ((data).append(" ").append(address));
         }
         return ((data).append((address)));
      }
      else
      {
         return _poiList[u32Index].getData();
      }
   }
   else
   {
      return "";
   }
}


/**************************************************************************//**
*
******************************************************************************/
std::string clSDS_POIList::oGetDisatance(tU32 u32Index)
{
   if (u32Index < _poiList.size())
   {
      return _poiList[u32Index].getDistance();
   }
   else
   {
      return "";
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
signed char clSDS_POIList::oGetDirectionFromMapping(tU32 u32Index)const
{
   if (_poiList[u32Index].getDirection() == DirectionDescription__NORTH ||
         _poiList[u32Index].getDirection()  == DirectionDescription__NORTH_NORTH_WEST)
   {
      return (EN_HEADING_INDEX_NORTH);
   }
   else if (_poiList[u32Index].getDirection()  == DirectionDescription__NORTH_NORTH_EAST ||
            _poiList[u32Index].getDirection()  == DirectionDescription__NORTH_EAST)
   {
      return (EN_HEADING_INDEX_NORTH_EAST);
   }
   else if (_poiList[u32Index].getDirection()  == DirectionDescription__EAST_NORTH_EAST ||
            _poiList[u32Index].getDirection()  == DirectionDescription__EAST)
   {
      return (EN_HEADING_INDEX_EAST);
   }
   else if (_poiList[u32Index].getDirection()  == DirectionDescription__EAST_SOUTH_EAST ||
            _poiList[u32Index].getDirection()  == DirectionDescription__SOUTH_EAST)
   {
      return (EN_HEADING_INDEX_SOUTH_EAST);
   }
   else if (_poiList[u32Index].getDirection() == DirectionDescription__SOUTH_SOUTH_EAST ||
            _poiList[u32Index].getDirection()  == DirectionDescription__SOUTH)
   {
      return (EN_HEADING_INDEX_SOUTH);
   }
   else if (_poiList[u32Index].getDirection()  == DirectionDescription__SOUTH_SOUTH_WEST ||
            _poiList[u32Index].getDirection() == DirectionDescription__SOUTH_WEST)
   {
      return (EN_HEADING_INDEX_SOUTH_WEST);
   }
   else if (_poiList[u32Index].getDirection()  == DirectionDescription__WEST_SOUTH_WEST ||
            _poiList[u32Index].getDirection()  == DirectionDescription__WEST)
   {
      return (EN_HEADING_INDEX_WEST);
   }
   else if (_poiList[u32Index].getDirection()  == DirectionDescription__WEST_NORTH_WEST ||
            _poiList[u32Index].getDirection()  == DirectionDescription__NORTH_WEST)
   {
      return (EN_HEADING_INDEX_NORTH_WEST);
   }
   else if (_poiList[u32Index].getDirection()  == DirectionDescription__INVALID)
   {
      return (EN_HEADING_INDEX_UNKNOWN);
   }
   else
   {
      return (EN_HEADING_INDEX_UNKNOWN);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_POIList::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType /*listType*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescription> clSDS_POIList::getHmiElementDescription(unsigned int index)
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> hmiElementDescriptionList;
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   if ((index <= _poiList.size()) && (index > 0))
   {
      if (!_poiList[index - 1].getAddress().empty() || !_poiList[index - 1].getData().empty())
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_PLAIN_TEXT;
         hmiElementDescription.DescriptorValue.bSet(oGetItem(index - 1).c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);

         hmiElementDescriptionList.push_back(hmiElementDescription);
      }
   }
   return hmiElementDescriptionList;
}


/***********************************************************************//**
 *
 ***************************************************************************/
tU32 clSDS_POIList::getRefinementID(tU32 absoluteValue)
{
   if (absoluteValue > 0 && absoluteValue <= _poiList.size())
   {
      return (_poiList.at(absoluteValue - 1).getId());
   }
   return 0;
}


/***********************************************************************//**
 *
 ***************************************************************************/

void clSDS_POIList::setHandOverPOIList(const std::vector< org::bosch::cm::navigation::NavigationService::AddressListElement >& oPOIList)
{
   _poiList = oPOIList;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_POIList::vRestoreHMIListData()
{
   if (_POIListViaHaptic)
   {
      _naviProxy->sendUndoSetLocationWithFixedPOICategoryInputRequest(*this);
   }
   else
   {
      _naviProxy->sendUndoFreeTextSearchElementSelectionRequest(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_POIList::onUndoFreeTextSearchElementSelectionError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::UndoFreeTextSearchElementSelectionError >& /*error*/)
{
   _restoreHMIListResponse = false;
   notifyRestoreHMIListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_POIList::onUndoFreeTextSearchElementSelectionResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::UndoFreeTextSearchElementSelectionResponse >& /*response*/)
{
   _restoreHMIListResponse = true;
   notifyRestoreHMIListObserver();
}


/***********************************************************************//**

 ***************************************************************************/
void clSDS_POIList::onUndoSetLocationWithFixedPOICategoryInputError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< UndoSetLocationWithFixedPOICategoryInputError >& /*error*/)
{
   _restoreHMIListResponse = false;
   notifyRestoreHMIListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_POIList::onUndoSetLocationWithFixedPOICategoryInputResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< UndoSetLocationWithFixedPOICategoryInputResponse >& /*response*/)
{
   _restoreHMIListResponse = true;
   notifyRestoreHMIListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
bool clSDS_POIList::getResultType()
{
   return _restoreHMIListResponse;
}


/***********************************************************************//**
 *
 ***************************************************************************/
bool clSDS_POIList::isPOIListViaHaptic()
{
   return _POIListViaHaptic;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_POIList::setPOIListViaHaptic()
{
   _POIListViaHaptic = true;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_POIList::resetPOIListViaHaptic()
{
   _POIListViaHaptic = false;
}
