/**************************************************************************//**
 * \file       clSDS_MultipleDestinationsList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_MultipleDestinationsList.h"
#include "application/clSDS_NaviAmbiguitylistObserver.h"
#include "application/clSDS_StringVarList.h"
#include "external/sds2hmi_fi.h"

#include "SdsAdapter_Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_MultipleDestinationsList.cpp.trc.h"
#endif

#define NUM_OF_DIGITS_IN_DIRECTION_VALUE 5
using namespace org::bosch::cm::navigation::NavigationService;

//#define DIST_VALUE_COLUMN   2
//#define DIRECTION_VALUE_COLUMN   4
//#define ADDRESS_COLUMN    1
//#define NUM_OF_DIGITS_IN_DIRECTION_VALUE 5


/**************************************************************************//**
*Destructor
******************************************************************************/
clSDS_MultipleDestinationsList::~clSDS_MultipleDestinationsList()
{
}


/**************************************************************************//**
*Constructor
******************************************************************************/
clSDS_MultipleDestinationsList::clSDS_MultipleDestinationsList(::boost::shared_ptr<NavigationServiceProxy> pNaviProxy)
   : _navigationProxy(pNaviProxy)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_MultipleDestinationsList::onSdsGetRefinementListError(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsGetRefinementListError >& /*error*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_MultipleDestinationsList::onSdsGetRefinementListResponse(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsGetRefinementListResponse >& oSdsRefinementListresponse)
{
   ETG_TRACE_USR1(("onSdsGetRefinementListResponse"));

   _addressList = oSdsRefinementListresponse->getRefinements();

   vNotifyAmbiguitylistObserver();
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_MultipleDestinationsList::vRequestAmbigListToNavi(org::bosch::cm::navigation::NavigationService::SDSAddressElementType ambigousElement)
{
   ETG_TRACE_USR1(("vRequestAmbigListToNavi"));
   _addressList.clear();
   _navigationProxy->sendSdsGetRefinementListRequest(*this, ambigousElement);
}


/**************************************************************************//**
*
******************************************************************************/
tU32 clSDS_MultipleDestinationsList::getRefinementID(tU32 absoluteValue)
{
   if (absoluteValue > 0 && absoluteValue <= _addressList.size())
   {
      return (_addressList.at(absoluteValue - 1).getId());
   }
   return 0;
}


/**************************************************************************//**
*
******************************************************************************/
tU32 clSDS_MultipleDestinationsList::u32GetSize()
{
   return _addressList.size();
}


/**************************************************************************//**
*
******************************************************************************/
std::vector<clSDS_ListItems> clSDS_MultipleDestinationsList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;

   for (tU32 u32Index = u32StartIndex; u32Index < std::min(u32EndIndex, u32GetSize()); u32Index++)
   {
      oListItems.push_back(oGetListItem(u32Index));
   }
   return oListItems;
}


/**************************************************************************//**
*
******************************************************************************/
clSDS_ListItems clSDS_MultipleDestinationsList::oGetListItem(tU32 u32Index)
{
   clSDS_ListItems oListItem;
   oListItem.oDescription.szString = oGetItem(u32Index);
   oListItem.oDistance.szString = oGetDistance(u32Index);
   oListItem.oIcon.szString = NO_ICON;
   oListItem.oIcon2.szString = NO_ICON;
   tChar szTempString[NUM_OF_DIGITS_IN_DIRECTION_VALUE] = "";
   OSALUTIL_s32SaveNPrintFormat(szTempString, sizeof(szTempString), "%d", oGetDirectionFromMapping(u32Index));
   oListItem.oDirectionSymbol.szString = szTempString;

   return oListItem;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_MultipleDestinationsList::oGetItem(tU32 u32Index)
{
   if (u32Index < u32GetSize())
   {
      return _addressList[u32Index].getData();
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_MultipleDestinationsList::oGetDistance(tU32 u32Index)
{
   if (u32Index < u32GetSize())
   {
      return _addressList[u32Index].getDistance();
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
signed char clSDS_MultipleDestinationsList::oGetDirectionFromMapping(tU32 u32Index)const
{
   if (u32Index < _addressList.size())
   {
      switch (_addressList[u32Index].getDirection())
      {
         case DirectionDescription__NORTH:
         case DirectionDescription__NORTH_NORTH_WEST:
            return EN_HEADING_INDEX_NORTH;

         case DirectionDescription__NORTH_NORTH_EAST:
         case DirectionDescription__NORTH_EAST:
            return EN_HEADING_INDEX_NORTH_EAST;

         case DirectionDescription__EAST_NORTH_EAST:
         case DirectionDescription__EAST:
            return EN_HEADING_INDEX_EAST;

         case DirectionDescription__EAST_SOUTH_EAST:
         case DirectionDescription__SOUTH_EAST:
            return (EN_HEADING_INDEX_SOUTH_EAST);

         case DirectionDescription__SOUTH_SOUTH_EAST:
         case DirectionDescription__SOUTH:
            return (EN_HEADING_INDEX_SOUTH);

         case DirectionDescription__SOUTH_SOUTH_WEST:
         case DirectionDescription__SOUTH_WEST:
            return EN_HEADING_INDEX_SOUTH_WEST;

         case DirectionDescription__WEST_SOUTH_WEST:
         case DirectionDescription__WEST:
            return EN_HEADING_INDEX_WEST;

         case DirectionDescription__WEST_NORTH_WEST:
         case DirectionDescription__NORTH_WEST:
            return EN_HEADING_INDEX_NORTH_WEST;

         case DirectionDescription__INVALID:
            return EN_HEADING_INDEX_UNKNOWN;

         default:
            break;
      }
   }
   return EN_HEADING_INDEX_UNKNOWN;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_MultipleDestinationsList::bSelectElement(tU32 u32SelectedIndex)
{
   if (u32SelectedIndex > 0)
   {
      clSDS_StringVarList::vSetVariable("$(Address)", oGetItem(u32SelectedIndex - 1));
      return TRUE;
   }
   return FALSE;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_MultipleDestinationsList::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType /*listType*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_MultipleDestinationsList::vRegisterAmbiguityListObserver(clSDS_NaviAmbiguitylistObserver* oAmbiguitylistObserver)
{
   _oAmbiguityObserverlist.push_back(oAmbiguitylistObserver);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_MultipleDestinationsList::vNotifyAmbiguitylistObserver()
{
   ETG_TRACE_USR1(("vNotifyAmbiguitylistObserver"));
   std::vector<clSDS_NaviAmbiguitylistObserver*>::iterator iter = _oAmbiguityObserverlist.begin();
   while (iter != _oAmbiguityObserverlist.end())
   {
      if (*iter != NULL)
      {
         (*iter)->vAmbiguityListResolved();
      }
      ++iter;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/

std::vector<sds2hmi_fi_tcl_HMIElementDescription> clSDS_MultipleDestinationsList::getHmiElementDescription(unsigned int index)
{
   ETG_TRACE_USR1(("getHmiElementDescription"));

   std::vector<sds2hmi_fi_tcl_HMIElementDescription> hmiElementDescriptionList;
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   if ((index <= _addressList.size()) && (index > 0))
   {
      if (!_addressList[index - 1].getData().empty())
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_PLAIN_TEXT;
         hmiElementDescription.DescriptorValue.bSet(oGetItem(index - 1).c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
         hmiElementDescriptionList.push_back(hmiElementDescription);
      }
   }
   return hmiElementDescriptionList;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_MultipleDestinationsList::setHandOverList(std::vector<org::bosch::cm::navigation::NavigationService::RefinementListElement > handOverList)
{
   _addressList = handOverList;
}
