/**************************************************************************//**
 * \file       clSDS_MenuManager.cpp
 *
 * clSDS_MenuManager class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_MenuManager.h"
#include "application/clSDS_Display.h"
#include "application/clSDS_ListScreen.h"
#include "application/clSDS_SdsControl.h"
#include "application/clSDS_ScreenData.h"
#include "application/clSDS_SessionControl.h"
#include "application/clSDS_View.h"
#include "application/GuiService.h"
#include "application/PopUpService.h"
#include "SdsAdapter_Trace.h"

#define G_GLO_SETTINGS_ID 2010
#define G_GLO_HELP_ID 2000
#define INVALID_FOCUS_INDEX 50000

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_MenuManager.cpp.trc.h"
#endif


using namespace sds_gui_fi::SdsGuiService;


/**************************************************************************//**
* Destructor
******************************************************************************/
clSDS_MenuManager::~clSDS_MenuManager()
{
   _pDisplay = NULL;
   _pSdsControl = NULL;
   _pListScreen = NULL;
   _pSessionControl = NULL;
}


/**************************************************************************//**
* Constructor
******************************************************************************/
clSDS_MenuManager::clSDS_MenuManager(
   clSDS_Display* pDisplay,
   clSDS_SdsControl* pSdsControl,
   clSDS_ListScreen* pListScreen,
   clSDS_SessionControl* pSessionControl)

   : _pDisplay(pDisplay)
   , _pSdsControl(pSdsControl)
   , _pListScreen(pListScreen)
   , _pSessionControl(pSessionControl)
   , _listIndex(0)
   , _defaultFocusIndex(INVALID_FOCUS_INDEX)
{
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_MenuManager::vShowSDSView(clSDS_ScreenData& oScreenData, tU32 pageNumber, tU32 numberOfPages, tU8 nextPreviousOptions, unsigned int maxItemsperPage, sds2hmi_fi_tcl_e8_PopUp_UpdateType screenUpdateType)
{
   if (_pDisplay)
   {
      _pDisplay->vShowSDSView(oScreenData, pageNumber, numberOfPages, nextPreviousOptions, maxItemsperPage, screenUpdateType);
   }
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_MenuManager::vCloseGUI(tBool bError) const
{
   if (_pDisplay)
   {
      _pDisplay->vCloseView(bError);
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_MenuManager::onElementSelected(unsigned int indexOnPage)
{
   if (_pSdsControl && _pDisplay)
   {
      const uint32 listIndex = getOverallListIndex(indexOnPage);
      _promptId = Sds_ViewDB_getPromptId(_pDisplay->getScreenId(), listIndex, _pDisplay->getCombinedScreenVariableData());
      if (_pListScreen->bIsListScreen(_pDisplay->getScreenId()))
      {
         _pSdsControl->sendListSelectRequest(convertToSdsListIndex(indexOnPage));
      }
      else if (_pDisplay->hasCommandList())
      {
         _pSdsControl->sendSelectRequest(getGrammarIdOfCommand(listIndex));
      }
   }
}


void clSDS_MenuManager::onElementSelectedHL(unsigned int indexOnPage)
{
   if (_pSdsControl && _pDisplay)
   {
      const uint32 listIndex = indexOnPage;
      _promptId = Sds_ViewDB_getPromptId(_pDisplay->getScreenId(), listIndex, _pDisplay->getCombinedScreenVariableData());
      if (_pDisplay->hasCommandList())
      {
         _pSdsControl->sendSelectRequest(getGrammarIdOfCommand(listIndex));
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_MenuManager::onCursorMoved(unsigned int indexOnPage)
{
   _listIndex = indexOnPage;
   if (_pSdsControl && _pDisplay && _pSessionControl)
   {
      _pDisplay->cursorMoved(indexOnPage);
      unsigned int listIndex = getOverallListIndex(indexOnPage);
      _promptId = Sds_ViewDB_getPromptId(_pDisplay->getScreenId(), listIndex, _pDisplay->getCombinedScreenVariableData());
      _pSessionControl->sendEnterManualMode();
      if (_pDisplay->hasCommandList())
      {
         _pSdsControl->sendFocusMoved(getGrammarIdOfCommand(listIndex));
      }
      else
      {
         _pSdsControl->sendFocusMoved(convertToSdsListIndex(indexOnPage));
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
unsigned int clSDS_MenuManager::getOverallListIndex(unsigned int indexOnPage) const
{
   if (_pDisplay)
   {
      return indexOnPage + (_pDisplay->getPageNumber() * _pDisplay->getMaxItemsPerPage());
   }
   return 0;
}


/**************************************************************************//**
 *
 ******************************************************************************/
unsigned int clSDS_MenuManager::getGrammarIdOfCommand(unsigned int listIndex) const
{
   if (_pDisplay)
   {
      const std::string& grammarId = Sds_ViewDB_getGrammarId(_pDisplay->getScreenId(), listIndex, _pDisplay->getCombinedScreenVariableData());
      return (unsigned int) strtol(grammarId.c_str(), NULL, 10);
   }
   return 0;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_MenuManager::onRequestNextPage()
{
   if (_pSdsControl)
   {
      _pSdsControl->sendNextPageRequest();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_MenuManager::onRequestPreviousPage()
{
   if (_pSdsControl)
   {
      _pSdsControl->sendPreviousPageRequest();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_MenuManager::onSettingsRequest()
{
   if (_pSdsControl)
   {
      _pSdsControl->sendSelectRequest(G_GLO_SETTINGS_ID);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_MenuManager::onHelpRequest()
{
   if (_pSdsControl)
   {
      _pSdsControl->sendSelectRequest(G_GLO_HELP_ID);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_MenuManager::getPromptId() const
{
   return _promptId;
}


/**************************************************************************//**
 *
 ******************************************************************************/
unsigned int clSDS_MenuManager::convertToSdsListIndex(unsigned int absoluteIndex) const
{
   return (absoluteIndex + 1);
}


/**************************************************************************//**
 *
 ******************************************************************************/

unsigned int clSDS_MenuManager::getListIndex()
{
   return _listIndex + (_pDisplay->getPageNumber() * _pDisplay->getMaxItemsPerPage());
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_MenuManager::onCursorIndexChanged(unsigned int focusedIndex)
{
   if (_pDisplay)
   {
      _pDisplay->cursorMoved(focusedIndex);
   }
   _defaultFocusIndex = focusedIndex;
}


unsigned int clSDS_MenuManager::getDefaultFocusIndex()
{
   return _defaultFocusIndex;
}
