/**************************************************************************//**
 * \file       clSDS_FuelPriceList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_FuelPriceList.h"
#include "application/clSDS_KDSConfiguration.h"
#include "application/clSDS_StringVarList.h"
#include "application/GuiService.h"
#include "application/StringUtils.h"
#include "view_db/Sds_TextDB.h"

#define FUEL_TYPE_PETROL 0x00
#define FUEL_TYPE_DIESEL  0x01


using namespace org::bosch::cm::navigation::NavigationService;
using namespace org::bosch::cm::navigation::NavigationSDSService;
using namespace sds_gui_fi::SdsGuiService;
using namespace sxm_canadian_fuel_main_fi;
using namespace sxm_canadian_fuel_main_fi_types;
using namespace sxm_fuel_main_fi;
using namespace sxm_fuel_main_fi_types;
using namespace sxm_main_fi_types;
using namespace VEHICLE_MAIN_FI;
using namespace vehicle_main_fi_types;


/**************************************************************************//**
 *Destructor
 ******************************************************************************/
clSDS_FuelPriceList::~clSDS_FuelPriceList()
{
   _pGuiService = NULL;
}


/**************************************************************************//**
 *Constructor
 ******************************************************************************/
clSDS_FuelPriceList::clSDS_FuelPriceList(
   boost::shared_ptr< Sxm_canadian_fuel_main_fiProxy > sxmCanadianFuelProxy,
   boost::shared_ptr< Sxm_fuel_main_fiProxy > sxmFuelProxy,
   boost::shared_ptr< NavigationServiceProxy > naviProxy,
   boost::shared_ptr< NavigationSDSServiceProxy > naviSDSProxy,
   boost::shared_ptr< VEHICLE_MAIN_FIProxy > vehicleFiProxy,
   GuiService* pGuiService)

   : _sxmCanadianFuelProxy(sxmCanadianFuelProxy)
   , _sxmFuelProxy(sxmFuelProxy)
   , _naviProxy(naviProxy)
   , _naviSDSProxy(naviSDSProxy)
   , _vehicleFiProxy(vehicleFiProxy)
   , _pGuiService(pGuiService)
   , _startvehicleheading(0.0)
   , _selectElementIndex(0)
   , _vehicleDistanceUnits(DistanceDirectionUtility::EN_DISTANCE_MILES)
   , _enCountryCode(EN_COUNTRY_CODE_UNKNOWN)
{
   _vehicleEngineType = clSDS_KDSConfiguration::getVehicleEngineType();
   // set default location
   _naviGeoPosition.setLatitude(0.0);
   _naviGeoPosition.setLongitude(0.0);
}


/**************************************************************************//**
 *
 ******************************************************************************/
unsigned int clSDS_FuelPriceList::u32GetSize()
{
   return _fuelListItems.size();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviSDSProxy)
   {
      _naviSDSProxy->sendPositionStatusInfoDeregisterAll();
   }

   if (proxy == _vehicleFiProxy)
   {
      _vehicleFiProxy->sendDistanceUnitRelUpRegAll();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviSDSProxy)
   {
      _naviSDSProxy->sendPositionStatusInfoRegister(*this);
      _naviSDSProxy->sendPositionStatusInfoGet(*this);
   }

   if (proxy == _vehicleFiProxy)
   {
      _vehicleFiProxy->sendDistanceUnitUpReg(*this);
      _vehicleFiProxy->sendDistanceUnitGet(*this);
   }
}


/***********************************************************************//**

 ***************************************************************************/

void clSDS_FuelPriceList::onPositionStatusInfoError(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< PositionStatusInfoError >& /*error*/)
{
}


/***********************************************************************//**

 ***************************************************************************/

void clSDS_FuelPriceList::onPositionStatusInfoUpdate(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< PositionStatusInfoUpdate >& update)
{
   _enCountryCode = EN_COUNTRY_CODE_UNKNOWN;

   if (update->hasPositionStatusInfo())
   {
      //Store position information.
      _naviGeoPosition.setLatitude(update->getPositionStatusInfo().getPositionWGS84().getLatitude());
      _naviGeoPosition.setLongitude(update->getPositionStatusInfo().getPositionWGS84().getLongitude());
      _startvehicleheading = static_cast<double>(update->getPositionStatusInfo().getVehicleHeading());

      if (update->getPositionStatusInfo().getCountryCode().compare("USA") == 0)
      {
         _enCountryCode = EN_COUNTRY_CODE_USA;
      }
      else if (update->getPositionStatusInfo().getCountryCode().compare("CAN") == 0)
      {
         _enCountryCode = EN_COUNTRY_CODE_CAN;
      }
      else
      {
         _enCountryCode = EN_COUNTRY_CODE_UNKNOWN;
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onDistanceUnitError(
   const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DistanceUnitError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onDistanceUnitStatus(
   const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DistanceUnitStatus >& status)
{
   if (status->hasE8DistanceUnit())
   {
      switch (status->getE8DistanceUnit())
      {
         case::T_e8_Vehicle_DistanceUnit__Unknown:
            _vehicleDistanceUnits = DistanceDirectionUtility::EN_DISTANCE_UNKNOWN;
            break;
         case::T_e8_Vehicle_DistanceUnit__Miles:
            _vehicleDistanceUnits = DistanceDirectionUtility::EN_DISTANCE_MILES;
            break;
         case::T_e8_Vehicle_DistanceUnit__Km:
            _vehicleDistanceUnits = DistanceDirectionUtility::EN_DISTANCE_KM;
            break;
         default:
            _vehicleDistanceUnits = DistanceDirectionUtility::EN_DISTANCE_MILES;
            break;
      }
   }
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_FuelPriceList::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType listType)
{
   switch (listType)
   {
      case sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_FUEL_PRICES:
         onFuelSortByPriceorDistance(T_e8_SortType__FUEL_SORT_BY_PRICE, T_e8_CanFuelSortType__CAN_FUEL_SORT_BY_PRICE);
         break;
      case sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_FUEL_DISTANCES:
         onFuelSortByPriceorDistance(T_e8_SortType__FUEL_SORT_BY_DISTANCE, T_e8_CanFuelSortType__CAN_FUEL_SORT_BY_DISTANCE);
         break;

      default:
         _fuelListItems.clear();
         notifyListObserver();
         break;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onFuelSortByPriceorDistance(T_e8_SortType sortTypeUSA, T_e8_CanFuelSortType sortTypeCAN)
{
   switch (_enCountryCode)
   {
      case EN_COUNTRY_CODE_USA:
         retrieveUSAFuelPriceList(sortTypeUSA);
         break;
      case EN_COUNTRY_CODE_CAN:
         retrieveCanadianFuelPriceList(sortTypeCAN);
         break;
      default:
         _fuelListItems.clear();
         notifyListObserver();
         break;
   }
}


/***********************************************************************//**
 * Method Start to SXM Fuel service for USA region
 ***************************************************************************/
void clSDS_FuelPriceList::retrieveUSAFuelPriceList(sxm_fuel_main_fi_types::T_e8_SortType sortType)
{
   T_FuelTypeInfo lFuelTypeInfo;

   switch (_vehicleEngineType)
   {
      case FUEL_TYPE_PETROL:
      {
         lFuelTypeInfo.setFuelName("Regular");
         lFuelTypeInfo.setFuelType(T_e8_FuelTypeEnum__FUEL_TYPE_REGULAR);
      }
      break;
      case FUEL_TYPE_DIESEL:
      {
         lFuelTypeInfo.setFuelName("Diesel");
         lFuelTypeInfo.setFuelType(T_e8_FuelTypeEnum__FUEL_TYPE_DIESEL);
      }
      break;
      default :
      {
         lFuelTypeInfo.setFuelName("Regular");
         lFuelTypeInfo.setFuelType(T_e8_FuelTypeEnum__FUEL_TYPE_REGULAR);
      }
      break;
   }
   _sxmFuelProxy->sendGetFuelInfoListStart(*this,
                                           T_e8_SxmPredefinedListIds__NEAR_BY,
                                           T_e8_SxmListOperation__GET,
                                           "All Brands",
                                           lFuelTypeInfo,
                                           sortType);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onGetFuelInfoListError(
   const ::boost::shared_ptr< Sxm_fuel_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetFuelInfoListError >& /*error*/)
{
   _fuelListItems.clear();
   notifyListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onGetFuelInfoListResult(const ::boost::shared_ptr< Sxm_fuel_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetFuelInfoListResult >& result)
{
   _fuelListItems.clear();
   unsigned int  listID = static_cast<unsigned int >(T_e8_SxmPredefinedListIds__NEAR_BY);

   if ((result->hasListInfo() && result->getListID() == listID) && (result->getFuelInfoList().size()))
   {
      setUSAFuelInfoList(result->getFuelInfoList());
   }
   notifyListObserver();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FuelPriceList::setUSAFuelInfoList(const sxm_fuel_main_fi_types::T_FuelInfoList& fuelInfoList)
{
   for (size_t index = 0; index < fuelInfoList.size(); ++index)
   {
      const  T_FuelInfo& fuelInfo = fuelInfoList[index];
      float fuelDistance = _vehicleDistanceUnits == (DistanceDirectionUtility::EN_DISTANCE_MILES) ? fuelInfo.getFuelDistMile() : fuelInfo.getFuelDistKm();

      DistanceDirectionUtility::stGeoCoordinates fuelGeoCoordinates;
      fuelGeoCoordinates.dLatitude = DistanceDirectionUtility::vConvertLatitudeLongitudeTodouble(fuelInfo.getFuelLoc().getLat());
      fuelGeoCoordinates.dLongitude = DistanceDirectionUtility::vConvertLatitudeLongitudeTodouble(fuelInfo.getFuelLoc().getLon());
      fuelGeoCoordinates.dVehicleHeading = 0.0;

      DistanceDirectionUtility::stGeoCoordinates naviGeoCoordinates;
      naviGeoCoordinates.dLatitude =  _naviGeoPosition.getLatitude();
      naviGeoCoordinates.dLongitude = _naviGeoPosition.getLongitude();
      naviGeoCoordinates.dVehicleHeading =  _startvehicleheading;

      stFuelListItems fuelListItems;
      fuelListItems.szDistanceWithUnits = DistanceDirectionUtility::getFormattedDistance(fuelDistance, _vehicleDistanceUnits);
      fuelListItems.szBrandName = fuelInfo.getBrand();
      fuelListItems.szStationName = fuelInfo.getStationName();
      fuelListItems.szFuelPriceValue = fuelInfo.getFuelPrice();
      fuelListItems.u8fuelAge = fuelInfo.getFuelAge();
      fuelListItems.longitude = fuelGeoCoordinates.dLongitude;
      fuelListItems.latitude = fuelGeoCoordinates.dLatitude;
      fuelListItems.direction = static_cast<int>(DistanceDirectionUtility::oGetDirection(fuelGeoCoordinates, naviGeoCoordinates)) ;
      fuelListItems.fuelLocationID = fuelInfo.getFuel_LOCID();
      _fuelListItems.push_back(fuelListItems);
   }
}


/***********************************************************************//**

***************************************************************************/
void clSDS_FuelPriceList::onGetFuelStationInfoError(
   const ::boost::shared_ptr<Sxm_fuel_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetFuelStationInfoError >& /*error*/)
{
}


/***********************************************************************//**

***************************************************************************/
void clSDS_FuelPriceList::onGetFuelStationInfoResult(
   const ::boost::shared_ptr< Sxm_fuel_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr<  GetFuelStationInfoResult >& result)
{
   if (result->hasFuelStationInfo() && T_e8_SxmPredefinedListIds__NEAR_BY == result->getListID())
   {
      _fuelStationInfo.szCityName = result->getFuelStationInfo().getCity();
      _fuelStationInfo.szStateName = result->getFuelStationInfo().getState();
      _fuelStationInfo.szStationNameBrandName = result->getFuelStationInfo().getBrand();
      _fuelStationInfo.szPhoneNumber =  result->getFuelStationInfo().getPhone();
      _fuelStationInfo.szFuelStationAddress = result->getFuelStationInfo().getStreetName() + ", "
                                              + result->getFuelStationInfo().getCity() + ", "
                                              + result->getFuelStationInfo().getState() + ", "
                                              + result->getFuelStationInfo().getZipCode();

      if (result->getFuelStationInfo().getBrand().empty())
      {
         _fuelStationInfo.szStationNameBrandName = result->getFuelStationInfo().getStationName();
      }

      clSDS_StringVarList::vSetVariable("$(POI)", _fuelStationInfo.szStationNameBrandName);
      clSDS_StringVarList::vSetVariable("$(POI_address)", _fuelStationInfo.szFuelStationAddress);
      setFuelStationLocationDetailsToNavi(_selectElementIndex);
   }
}


/***********************************************************************//**
 Method Start to SXM Fuel service for Canada region
 ***************************************************************************/
void clSDS_FuelPriceList::retrieveCanadianFuelPriceList(T_e8_CanFuelSortType sortType)
{
   T_CanFuelTypeInfo canadianFuelType;

   switch (_vehicleEngineType)
   {
      case FUEL_TYPE_PETROL:
      {
         canadianFuelType.setCanFuelName("Regular");
         canadianFuelType.setCanFuelType(T_e8_CanFuelTypeEnum__CAN_FUEL_TYPE_REGULAR);
      }
      break;
      case FUEL_TYPE_DIESEL:
      {
         canadianFuelType.setCanFuelName("Diesel");
         canadianFuelType.setCanFuelType(T_e8_CanFuelTypeEnum__CAN_FUEL_TYPE_DIESEL);
      }
      break;
      default :
      {
         canadianFuelType.setCanFuelName("Regular");
         canadianFuelType.setCanFuelType(T_e8_CanFuelTypeEnum__CAN_FUEL_TYPE_REGULAR);
      }
      break;
   }
   _sxmCanadianFuelProxy->sendGetCanadianFuelInfoListStart(*this,
         T_e8_SxmPredefinedListIds__NEAR_BY,
         T_e8_SxmListOperation__GET,
         "All Brands",
         canadianFuelType,
         sortType);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onGetCanadianFuelInfoListError(
   const ::boost::shared_ptr< Sxm_canadian_fuel_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetCanadianFuelInfoListError >& /* error*/)
{
   _fuelListItems.clear();
   notifyListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/

void clSDS_FuelPriceList::onGetCanadianFuelInfoListResult(
   const ::boost::shared_ptr< Sxm_canadian_fuel_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr<GetCanadianFuelInfoListResult >& result)
{
   _fuelListItems.clear();

   if ((result->hasListInfo() && result->getCanadianFuelInfoList().size() > 0) &&
         (result->hasListID() && T_e8_SxmPredefinedListIds__NEAR_BY == result->getListID()))
   {
      setCanadianFuelInfoList(result->getCanadianFuelInfoList());
   }
   notifyListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void  clSDS_FuelPriceList::setCanadianFuelInfoList(const T_CanFuelInfoList& canadianFuelInfoList)
{
   for (size_t i = 0 ; i < canadianFuelInfoList.size() ; ++i)
   {
      const T_CanFuelInfo& fuelInfo = canadianFuelInfoList[i];
      float fuelDistance = _vehicleDistanceUnits == (DistanceDirectionUtility::EN_DISTANCE_MILES) ? fuelInfo.getCanFuelDistMile() : fuelInfo.getCanFuelDistKm();

      DistanceDirectionUtility::stGeoCoordinates fuelGeoCoordinates;
      fuelGeoCoordinates.dLatitude = DistanceDirectionUtility::vConvertLatitudeLongitudeTodouble(fuelInfo.getCanFuelLoc().getLat());
      fuelGeoCoordinates.dLongitude = DistanceDirectionUtility::vConvertLatitudeLongitudeTodouble(fuelInfo.getCanFuelLoc().getLon());
      fuelGeoCoordinates.dVehicleHeading = 0.0;

      DistanceDirectionUtility::stGeoCoordinates naviGeoCoordinates;
      naviGeoCoordinates.dLatitude =  _naviGeoPosition.getLatitude();
      naviGeoCoordinates.dLongitude = _naviGeoPosition.getLongitude();
      naviGeoCoordinates.dVehicleHeading =  _startvehicleheading;

      stFuelListItems fuelListItems;
      fuelListItems.szDistanceWithUnits = DistanceDirectionUtility::getFormattedDistance(fuelDistance, _vehicleDistanceUnits);
      fuelListItems.szBrandName = fuelInfo.getBrand();
      fuelListItems.szStationName = fuelInfo.getStationName();
      fuelListItems.szFuelPriceValue = fuelInfo.getCanFuelPrice();
      fuelListItems.u8fuelAge = fuelInfo.getCanFuelAge();
      fuelListItems.longitude = fuelGeoCoordinates.dLongitude;
      fuelListItems.latitude = fuelGeoCoordinates.dLatitude ;
      fuelListItems.direction = DistanceDirectionUtility::oGetDirection(fuelGeoCoordinates, naviGeoCoordinates);
      fuelListItems.fuelLocationID = fuelInfo.getCanFuel_LOCID();
      _fuelListItems.push_back(fuelListItems);
   }
}


/***********************************************************************//**

***************************************************************************/
void clSDS_FuelPriceList::onGetCanadianFuelStationInfoError(
   const ::boost::shared_ptr< Sxm_canadian_fuel_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetCanadianFuelStationInfoError >& /*error*/)
{
}


/***********************************************************************//**

***************************************************************************/
void clSDS_FuelPriceList::onGetCanadianFuelStationInfoResult(
   const ::boost::shared_ptr< sxm_canadian_fuel_main_fi::Sxm_canadian_fuel_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< sxm_canadian_fuel_main_fi::GetCanadianFuelStationInfoResult >& result)
{
   if (result->hasCanadianFuelStationInfo() && T_e8_SxmPredefinedListIds__NEAR_BY == result->getListID())
   {
      _fuelStationInfo.szCityName =  result->getCanadianFuelStationInfo().getCity();
      _fuelStationInfo.szStateName = result->getCanadianFuelStationInfo().getState();
      _fuelStationInfo.szStationNameBrandName = result->getCanadianFuelStationInfo().getBrand();
      _fuelStationInfo.szFuelStationAddress = result->getCanadianFuelStationInfo().getStreetName() + ", "
                                              + result->getCanadianFuelStationInfo().getCity() + ", "
                                              + result->getCanadianFuelStationInfo().getState() + ", "
                                              + result->getCanadianFuelStationInfo().getZipCode();

      if (result->getCanadianFuelStationInfo().getBrand().empty())
      {
         _fuelStationInfo.szStationNameBrandName = result->getCanadianFuelStationInfo().getStationName();
      }

      clSDS_StringVarList::vSetVariable("$(POI)", _fuelStationInfo.szStationNameBrandName);
      clSDS_StringVarList::vSetVariable("$(POI_address)", _fuelStationInfo.szFuelStationAddress);
      setFuelStationLocationDetailsToNavi(_selectElementIndex);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::vector<clSDS_ListItems> clSDS_FuelPriceList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;

   for (unsigned int u32Index = u32StartIndex; u32Index < ::std::min(u32EndIndex, u32GetSize()); u32Index++)
   {
      oListItems.push_back(getListItem(u32Index));
   }
   return oListItems;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_FuelPriceList::getFuelPriceValue(tU32 u32Index) const
{
   if (u32Index < _fuelListItems.size())
   {
      return _fuelListItems[u32Index].szFuelPriceValue;
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_FuelPriceList::getFuelAge(tU8 u8FuelAge) const
{
   std::string oFuelAgeStr;

   switch (u8FuelAge)
   {
      case TODAY:
         oFuelAgeStr = Sds_TextDB_vGetText(SDS_TEXT_TODAY);
         break;

      case YESTERDAY:
         oFuelAgeStr = Sds_TextDB_vGetText(SDS_TEXT_ONE_DAY);
         break;

      case TWO_DAY:
         oFuelAgeStr = Sds_TextDB_vGetText(SDS_TEXT_TWO_DAYS);
         break;

      default:
         oFuelAgeStr = Sds_TextDB_vGetText(SDS_TEXT_THREE_DAYS_PLUS);
         break;
   }
   return oFuelAgeStr;
}


/**************************************************************************//**
 *
 ******************************************************************************/
clSDS_ListItems::tenColorofText clSDS_FuelPriceList::enGetFuelAgeColor(tU8 u8FuelAge) const
{
   switch (u8FuelAge)
   {
      case TODAY:
         return clSDS_ListItems::PRICE_AGE_CURRENT;

      case YESTERDAY:
         return clSDS_ListItems::PRICE_AGE_1_DAY;

      case TWO_DAY:
         return clSDS_ListItems::PRICE_AGE_2_DAYS;

      default:
         return clSDS_ListItems::PRICE_AGE_3_DAYS_PLUS;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
clSDS_ListItems clSDS_FuelPriceList::getListItem(tU32 u32Index) const
{
   clSDS_ListItems listItems;
   listItems.oPrice.szString = getFuelPriceValue(u32Index);
   listItems.oLastPriceUpdate.szString = getLastPriceUpdate(u32Index);
   listItems.oLastPriceUpdate.enTextColor = enGetLastPriceTextColor(u32Index);
   listItems.oDescription.szString = getBrandOrFuelStationName(u32Index);
   listItems.oDistance.szString = getDistance(u32Index);
   listItems.oDirectionSymbol.szString = StringUtils::toString(static_cast<unsigned int>(getDirection(u32Index)));;
   return listItems;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_FuelPriceList::getBrandOrFuelStationName(tU32 u32Index) const
{
   if (u32Index < _fuelListItems.size())
   {
      const std::string oBrandStr = _fuelListItems[u32Index].szBrandName;
      const std::string oStationNameStr = _fuelListItems[u32Index].szStationName;

      if (oBrandStr.empty())
      {
         return oStationNameStr;
      }
      return oBrandStr;
   }
   return  "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_FuelPriceList::getDistance(tU32 u32Index) const
{
   if (u32Index < _fuelListItems.size())
   {
      return  _fuelListItems[u32Index].szDistanceWithUnits;
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_FuelPriceList::getLastPriceUpdate(tU32 u32Index) const
{
   if (u32Index < _fuelListItems.size())
   {
      return getFuelAge(_fuelListItems[u32Index].u8fuelAge);
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
clSDS_ListItems::tenColorofText clSDS_FuelPriceList::enGetLastPriceTextColor(tU32 u32Index) const
{
   if (u32Index < _fuelListItems.size())
   {
      return enGetFuelAgeColor(_fuelListItems[u32Index].u8fuelAge);
   }
   return clSDS_ListItems::NORMAL;
}


/**************************************************************************//**
 *
 ******************************************************************************/
int clSDS_FuelPriceList::getDirection(tU32 u32Index) const
{
   if (u32Index < _fuelListItems.size())
   {
      return _fuelListItems[u32Index].direction;
   }
   return static_cast<int>(DistanceDirectionUtility::EN_HEADING_INDEX_UNKNOWN);
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_FuelPriceList::bSelectElement(tU32 u32SelectedIndex)
{
   if ((u32SelectedIndex > 0) && (u32SelectedIndex <= _fuelListItems.size()))
   {
      _selectElementIndex = u32SelectedIndex - 1;
      reteriveFuelStationInfoOnSelectElement(_selectElementIndex);
      return TRUE;
   }
   return FALSE;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::vStartGuidance()
{
   _naviProxy->sendStartGuidanceRequest(*this);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onStartGuidanceError(
   const ::boost::shared_ptr<NavigationServiceProxy>& /*proxy*/,
   const ::boost::shared_ptr<StartGuidanceError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onStartGuidanceResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< StartGuidanceResponse >& /*response*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onSetLocationWithDetailsError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithDetailsError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onSetLocationWithDetailsResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithDetailsResponse >& /*response*/)
{
   _naviProxy->sendRequestDetailsForSelectedLocationRequest(*this);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onRequestDetailsForSelectedLocationError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestDetailsForSelectedLocationError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_FuelPriceList::onRequestDetailsForSelectedLocationResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestDetailsForSelectedLocationResponse >& /*response*/)
{
   if (_pGuiService)
   {
      _pGuiService->sendEventSignal(Event__SPEECH_DIALOG_GADGET_ACTIVATION);
   }
}


/***********************************************************************//**

***************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescription> clSDS_FuelPriceList::getHmiElementDescription(unsigned int index)
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> hmiElementDescriptionList;
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   if ((index  > 0) && ((index - 1) < _fuelListItems.size()))
   {
      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POINAME;
      hmiElementDescription.DescriptorValue = getBrandOrFuelStationName(index - 1).c_str();
      hmiElementDescriptionList.push_back(hmiElementDescription);

      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_CITY;
      hmiElementDescription.DescriptorValue = _fuelStationInfo.szCityName.c_str();
      hmiElementDescriptionList.push_back(hmiElementDescription);

      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STATE;
      hmiElementDescription.DescriptorValue = _fuelStationInfo.szStateName.c_str();
      hmiElementDescriptionList.push_back(hmiElementDescription);
   }
   return hmiElementDescriptionList;
}


/***********************************************************************//**

***************************************************************************/
void clSDS_FuelPriceList::reteriveFuelStationInfoOnSelectElement(unsigned long index)
{
   if (index < _fuelListItems.size())
   {
      switch (_enCountryCode)
      {
         case EN_COUNTRY_CODE_USA:
            _sxmFuelProxy->sendGetFuelStationInfoStart(*this, _fuelListItems[index].fuelLocationID, T_e8_SxmPredefinedListIds__NEAR_BY);
            break;
         case EN_COUNTRY_CODE_CAN:
            _sxmCanadianFuelProxy->sendGetCanadianFuelStationInfoStart(*this, _fuelListItems[index].fuelLocationID, T_e8_SxmPredefinedListIds__NEAR_BY);
            break;
         default:
            break;
      }
   }
}


/***********************************************************************//**

***************************************************************************/
void clSDS_FuelPriceList::setFuelStationLocationDetailsToNavi(unsigned long index)
{
   if (index < _fuelListItems.size())
   {
      POILocation poiLocation;
      org::bosch::cm::navigation::NavigationService::PositionWGS84 position;
      position.setLatitude(_fuelListItems[index].latitude);
      position.setLongitude(_fuelListItems[index].longitude);
      Location location;
      location.setPositionWGS84(position);
      location.setName(_fuelStationInfo.szStationNameBrandName);
      location.setValid(true);
      poiLocation.setLocation(location);
      poiLocation.setAddress(_fuelStationInfo.szFuelStationAddress);
      poiLocation.setPhoneNumber(_fuelStationInfo.szPhoneNumber);
      _naviProxy->sendSetLocationWithDetailsRequest(*this, poiLocation);
   }
}
