/**************************************************************************//**
 * \file       clSDS_ConfigurationFlags.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_ConfigurationFlags.h"
#include "application/clSDS_KDSConfiguration.h"
#include "application/clSDS_StringVarList.h"
#include "application/StringUtils.h"
#include "SdsAdapter_Trace.h"
#include <fstream>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_ConfigurationFlags.cpp.trc.h"
#endif


#define CONFIG_FILE_PATH "/opt/bosch/sds/bin/config.dat"


std::map<std::string, std::string> clSDS_ConfigurationFlags::dynamicVariables;
std::map< std::string, std::string >clSDS_ConfigurationFlags::configDataFromFile;


/**************************************************************************//**
 *
 ******************************************************************************/
clSDS_ConfigurationFlags::clSDS_ConfigurationFlags()
{
}


/**************************************************************************//**
 *
 ******************************************************************************/

clSDS_ConfigurationFlags::~clSDS_ConfigurationFlags()
{
}


/**************************************************************************//**
 *3.x: getNavKey() : this method Return B_NAV flag
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getNavKey()
{
   if (isDisableNavigationForVDEUnavailable())
   {
      if (getDomainKey() && isVDEAvailble() && getMapUpdateStatus())
      {
         return true ;
      }
      return false ;
   }
   else
   {
      return (getMapUpdateStatus() && getDomainKey());
   }
}


/*********************************************************************************
 *
 *******************************************************************************/
bool clSDS_ConfigurationFlags::getDomainKey()
{
   switch (clSDS_KDSConfiguration::getSystemVariant())
   {
      case clSDS_KDSConfiguration::EN_MID_NAVIGATION:
      case clSDS_KDSConfiguration::EN_GENERIC_NAVIGATION:
      case clSDS_KDSConfiguration::EN_EVOBUS_NAVIGATION:
         return true;
      case clSDS_KDSConfiguration::EN_ENTRY_NAVIGATION:
         return isVDEForEntryNaviEnable();
      case clSDS_KDSConfiguration::EN_ENTRY_DISPLAY_AUDIO:
      case clSDS_KDSConfiguration::EN_MID_DISPLAY_AUDIO:

         return false;
      default:
         return false;
   }
}


/***********************************************************************************
 * IsVDEAvailble: it gives info about VDEAvailability
 *********************************************************************************/
bool clSDS_ConfigurationFlags::isVDEAvailble()
{
   if (dynamicVariables.find("VDEAvailable") != dynamicVariables.end()
         && (dynamicVariables["VDEAvailable"] == "2" || dynamicVariables["VDEAvailable"] == "0"))
   {
      return true;
   }

   return false;
}


bool clSDS_ConfigurationFlags::getMapUpdateStatus()
{
   if (clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::JPN ||
         clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::KOR)  //navDatasetInfo received as 0in JAPAN
   {
      return true;
   }
   if (dynamicVariables.find("MapUpdateStatus") != dynamicVariables.end()
         && (dynamicVariables["MapUpdateStatus"] == "1"))
   {
      return true;
   }

   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getNavAdrsKey()
{
   return getDomainKey();
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getPOIKey()
{
   return getDomainKey();
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getPOICatKey()
{
   return getDomainKey();
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getNavIntersectionKey()
{
   switch (clSDS_KDSConfiguration::getMarketRegionType())
   {
      case clSDS_KDSConfiguration::USA:
      case clSDS_KDSConfiguration::CAN:
      case clSDS_KDSConfiguration::MEX:
      case clSDS_KDSConfiguration::PRC:
      case clSDS_KDSConfiguration::HKG_MACAU:
         return true;
      default:
         return false;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getNavMultiLingualVDE()
{
   if (getDomainKey() && getSDNRegion() == SDN_EUR)
   {
      return true; //available only for European countries
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getWhereAmIKey()
{
   switch (clSDS_KDSConfiguration::getMarketRegionType())
   {
      case clSDS_KDSConfiguration::JPN:
      case clSDS_KDSConfiguration::PRC:
         return false;

      default:
         return true;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
POIRegion clSDS_ConfigurationFlags::getNaviPOIRegion()
{
   switch (getSDNRegion())
   {
      case SDN_NAM:
         return POI_NAM;

      case SDN_EUR:
      case SDN_TKY:
      case SDN_RUS:
         return POI_EUR;

      case SDN_GCC:
         return POI_GCC;

      case SDN_PRC:
      case SDN_TWN:
         return POI_PRC;

      case SDN_JPN:
         return POI_JPN;

      case SDN_LAC:
         return POI_LAC;

      case SDN_ASR:
      case SDN_SAF:
      case SDN_IND:
      case SDN_KOR:
         return POI_OTH;

      default:
         return MAX_POI_REGIONS;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/

SDNRegion clSDS_ConfigurationFlags::getSDNRegion()
{
   switch (clSDS_KDSConfiguration::getMarketRegionType())
   {
      case clSDS_KDSConfiguration::USA:
      case clSDS_KDSConfiguration::CAN:
      case clSDS_KDSConfiguration::MEX:
         return SDN_NAM;

      case clSDS_KDSConfiguration::AGT:
      case clSDS_KDSConfiguration::BRA:
      case clSDS_KDSConfiguration::OTHER_LAC:
         return SDN_LAC;

      case clSDS_KDSConfiguration::UK:
      case clSDS_KDSConfiguration::OTHER_EUR:
         return SDN_EUR;

      case clSDS_KDSConfiguration::TKY:
         return SDN_TKY;

      case clSDS_KDSConfiguration::RUS:
         return SDN_RUS;

      case clSDS_KDSConfiguration::ASR_NZR:
         return SDN_ASR;

      case clSDS_KDSConfiguration::SAF:
         return SDN_SAF;

      case clSDS_KDSConfiguration::GCC:
      case clSDS_KDSConfiguration::EGP:
      case clSDS_KDSConfiguration::ISR:
         return SDN_GCC;

      case clSDS_KDSConfiguration::IND:
         return SDN_IND;

      case clSDS_KDSConfiguration::PRC:
      case clSDS_KDSConfiguration::HKG_MACAU:
      case clSDS_KDSConfiguration::INN:
         return SDN_PRC;

      case clSDS_KDSConfiguration::TWN:
         return SDN_TWN;

      case clSDS_KDSConfiguration::JPN:
         return SDN_JPN;

      case clSDS_KDSConfiguration::KOR:
         return SDN_KOR;

      case clSDS_KDSConfiguration::SGP:
      case clSDS_KDSConfiguration::MLY:
      case clSDS_KDSConfiguration::PHL:
      case clSDS_KDSConfiguration::THI:
      case clSDS_KDSConfiguration::VNM:
      case clSDS_KDSConfiguration::OTHER_GOM:
         return MAX_SDN_REGIONS;

      default:
         return MAX_SDN_REGIONS;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getNavStateAvailable()
{
   switch (clSDS_KDSConfiguration::getMarketRegionType())
   {
      case clSDS_KDSConfiguration::USA:
      case clSDS_KDSConfiguration::CAN:
      case clSDS_KDSConfiguration::MEX:
      case clSDS_KDSConfiguration::PRC:
      case clSDS_KDSConfiguration::HKG_MACAU:
      case clSDS_KDSConfiguration::JPN:
         return true;

      default:
         return false;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getNavCountryAvailable()
{
   switch (clSDS_KDSConfiguration::getMarketRegionType())
   {
      case clSDS_KDSConfiguration::UK:
      case clSDS_KDSConfiguration::TKY:
      case clSDS_KDSConfiguration::RUS:
      case clSDS_KDSConfiguration::OTHER_EUR:
      case clSDS_KDSConfiguration::GCC:
      case clSDS_KDSConfiguration::ASR_NZR:
         return true;

      default:
         return false;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/

bool clSDS_ConfigurationFlags::getNavJapaneseAddressInput()
{
   if ((clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::JPN)
         && (clSDS_KDSConfiguration::getSystemVariant() == clSDS_KDSConfiguration::EN_MID_NAVIGATION))
   {
      return true;
   }

   else
   {
      return false;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getAudHDKey()
{
   // Vehicle Information: DestinationRegion == US
   return (clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::USA);
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getAudRDSKey()
{
   // AMFM_Tuner_Parameter: “RDS audio tuner 1” == on and “PI ignore” == off.
   return ((clSDS_KDSConfiguration::getRDSAudioTuner1() == 1 && clSDS_KDSConfiguration::getPIIgnore() == 0)
           || (clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::JPN));
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getAudDABKey()
{
   return (clSDS_KDSConfiguration::getDABMounted() && clSDS_KDSConfiguration::getDABEnabled());
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getAudSXMKey()
{
   return (clSDS_KDSConfiguration::getXMTunerMounted() == 1);
}


bool clSDS_ConfigurationFlags::getAudTVKey()
{
   return (clSDS_KDSConfiguration::getDTVMounted() == 1);
}


bool clSDS_ConfigurationFlags::getVINFMaintenancekey()
{
   return (clSDS_KDSConfiguration::getMaintenanceInfo() == 1);
}


bool clSDS_ConfigurationFlags::getVINFFuelEconomykey()
{
   return (clSDS_KDSConfiguration::getFuelConsumptionInfo() == 1 ||
           clSDS_KDSConfiguration::getFuelConsumptionInfo() == 2);
}


unsigned int clSDS_ConfigurationFlags::getAudFreqTypeKey()
{
   switch (clSDS_KDSConfiguration::getMarketRegionType())
   {
      case clSDS_KDSConfiguration::USA:
      case clSDS_KDSConfiguration::CAN:
      case clSDS_KDSConfiguration::MEX:
         return 0;

      case clSDS_KDSConfiguration::UK:
      case clSDS_KDSConfiguration::TKY:
      case clSDS_KDSConfiguration::RUS:
      case clSDS_KDSConfiguration::OTHER_EUR:
         return 1;

      case clSDS_KDSConfiguration::JPN:
         return 5;

      default:
         return 2; //TODO raa8hi - other values (2-4) have be checked with tuner colleagues
   }

   // ------cases from previous project-------------------
   //      case KOR:
   //      case SAF:
   //      case THI://What about THI and JPN value for frequency
   //      case HKG_MACAU://      case PRC:

   //      case JPN:
   //      case TWN:
   //      case IND:
   //      case GCC:
   //      case PRC://      case PRC:

   //          return 2;
   //
   //      case BRA:
   //      case OTHER_LAC:
   //          return 3;
   //
   //      case ASR_NZR:
   //          return 4;
   //
   //      default:
   //         return 0;
   //----------------------------------------------------------
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getInfTrafficKey()
{
   switch (clSDS_KDSConfiguration::getMarketRegionType())
   {
      case clSDS_KDSConfiguration::USA:
      case clSDS_KDSConfiguration::CAN:
      case clSDS_KDSConfiguration::UK:
      case clSDS_KDSConfiguration::TKY:
      case clSDS_KDSConfiguration::RUS:
      case clSDS_KDSConfiguration::OTHER_EUR:
      case clSDS_KDSConfiguration::PRC:
      case clSDS_KDSConfiguration::ASR_NZR:
      case clSDS_KDSConfiguration::JPN:
         return getDomainKey();

      default:
         return false;
   }//Todo : vel5kor Will be replaced with interface provided by Navi HMI
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getInfHEVKey()
{
   return (clSDS_KDSConfiguration::getHEVInformationFunction() == 1);
}


/**************************************************************************//**
 *
 ******************************************************************************/

bool clSDS_ConfigurationFlags::getInfKey()
{
   if (dynamicVariables.find("B_INF") != dynamicVariables.end()
         && dynamicVariables["B_INF"] == "1")
   {
      return true;
   }

   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getPOIAddressInfo()
{
   if (configDataFromFile.find("B_INT_POIAddressCheck") != configDataFromFile.end()
         && configDataFromFile["B_INT_POIAddressCheck"] == "1")
   {
      return true;
   }

   return false;
}


bool clSDS_ConfigurationFlags::getInfFeeds()
{
   if ((configDataFromFile.find("B_INF_InformationFeeds") != configDataFromFile.end()
         && configDataFromFile["B_INF_InformationFeeds"] == "1"))
   {
      return true;
   }

   return false;
}


bool clSDS_ConfigurationFlags::getEveKey()
{
   if (((configDataFromFile.find("B_INT_EveFeature") != configDataFromFile.end()
         && configDataFromFile["B_INT_EveFeature"] == "1")) && (clSDS_KDSConfiguration::getVehicleEngineType() == clSDS_KDSConfiguration::ELECTRICITY)
         && (clSDS_ConfigurationFlags::getDomainKey()))
   {
      return true;
   }

   return false;
}


bool clSDS_ConfigurationFlags::getVehicleInfoKey()
{
   if (configDataFromFile.find("B_VINF") != configDataFromFile.end()
         && configDataFromFile["B_VINF"] == "1")
   {
      return true;
   }

   return false;
}


bool clSDS_ConfigurationFlags::getPhoneInCallFeatureKey()
{
   if (configDataFromFile.find("B_INT_PhoneInCallFeature") != configDataFromFile.end()
         && configDataFromFile["B_INT_PhoneInCallFeature"] == "1")
   {
      return true;
   }

   return false;
}


bool clSDS_ConfigurationFlags::getHapticModeAfterThirdError()
{
   if (configDataFromFile.find("B_GRL_HapticModeAfterThirdError") != configDataFromFile.end()
         && configDataFromFile["B_GRL_HapticModeAfterThirdError"] == "1")
   {
      return true;
   }

   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getSwitchThroughDomainsKey()
{
   if (configDataFromFile.find("B_INT_SwitchThroughDomains") != configDataFromFile.end()
         && configDataFromFile["B_INT_SwitchThroughDomains"] == "1")
   {
      return true;
   }

   return false;
}


bool clSDS_ConfigurationFlags::getPromptOnFocusMovementKey()
{
   if (configDataFromFile.find("B_GRL_PromptOnFocusMovement") != configDataFromFile.end()
         && configDataFromFile["B_GRL_PromptOnFocusMovement"] == "1")
   {
      return true;
   }

   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getElectricVehicleKey()
{
   if (clSDS_KDSConfiguration::getVehicleEngineType() == clSDS_KDSConfiguration::ELECTRICITY)
   {
      return true;
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getInfSXMKey()
{
   return ((clSDS_KDSConfiguration::getXMTunerMounted() == 1) &&
           clSDS_KDSConfiguration::getSystemVariant() == clSDS_KDSConfiguration::EN_MID_NAVIGATION);
}


bool clSDS_ConfigurationFlags::getBluetoothPhoneEnabledKey()
{
   return (clSDS_KDSConfiguration::getBluetoothPhoneEnabled() == 1);
}


bool clSDS_ConfigurationFlags::getTaxiAvailabilityKey()
{
   return (clSDS_KDSConfiguration::getAllianceVehicleFleetType() == 1);
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getCDAvailableKey()
{
   return ((clSDS_KDSConfiguration::getCDMounted() == 1));
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getDVDAvailableKey()
{
   return ((clSDS_KDSConfiguration::getDVDMounted() == 1));
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getMusicBoxAvailableKey()
{
   return ((clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::JPN)
           && (clSDS_KDSConfiguration::getVariantType() == clSDS_KDSConfiguration::TYPE_INFINITI));
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_ConfigurationFlags::setDynamicVariable(const std::string& name, const std::string& value)
{
   dynamicVariables[name] = value;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_ConfigurationFlags::setDynamicVariable(const std::string& name, unsigned int value)
{
   dynamicVariables[name] = StringUtils::toString(value);
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::map<std::string, std::string>  clSDS_ConfigurationFlags::getConfigurationFlags()
{
   std::map<std::string, std::string> variables;
   variables["B_NAV"] = StringUtils::toString(getNavKey());
   variables["NavDomain"] = StringUtils::toString(getDomainKey());//for internal use in view spec
   variables["B_NAV_ADRS"] = StringUtils::toString(getNavAdrsKey());
   variables["B_NAV_POI"] = StringUtils::toString(getPOIKey());
   variables["B_NAV_POICAT"] = StringUtils::toString(getPOICatKey());
   variables["B_NAV_CHNG_STATE"] = StringUtils::toString(getNavStateAvailable()); // for internal use in view spec; no longer a CSDSC flag => not to be sent to SDS_MW
   variables["B_NAV_CHNG_CNTRY"] = StringUtils::toString(getNavCountryAvailable());
   variables["B_NAV_Japanese_Address_Input"] = StringUtils::toString(getNavJapaneseAddressInput());
   variables["B_NAV_Intersection"] = StringUtils::toString(getNavIntersectionKey());

   variables["B_AUD_AM"] = "1"; //chg6kor: This flag is not sent to SDS MW, Its initialize to default value of SDS MW for AM that is 1 .
   // however it is been used for command condition in view spec for AM to be visible
   variables["B_AUD_DAB"] = StringUtils::toString(getAudDABKey());
   variables["B_AUD_SXM"] = StringUtils::toString(getAudSXMKey());
   variables["B_AUD_RDS"] = StringUtils::toString(getAudRDSKey());
   variables["B_AUD_HD"] = StringUtils::toString(getAudHDKey());
   variables["B_AUD_TV"] = StringUtils::toString(getAudTVKey());
   variables["E_AUD_FREQCY_TYPE"] = StringUtils::toString(getAudFreqTypeKey());

   variables["B_INF_FuelEconomy"] = StringUtils::toString(getVINFFuelEconomykey());
   variables["B_INF_Maintenance"] = StringUtils::toString(getVINFMaintenancekey());
   variables["B_INF_TRA"] = StringUtils::toString(getInfTrafficKey());
   variables["B_INF_HEV"] = StringUtils::toString(getInfHEVKey());
   variables["B_INF_SXM"] = StringUtils::toString(getInfSXMKey());
   // used to for command condition in view spec for connect voice menu or connect to Intouch to be visible
   variables["E_INF_VASSIST"] = StringUtils::toString(getTCUVoiceAssitOnVariantType());
   variables["B_INF_WhereAmI"] = StringUtils::toString(getWhereAmIKey());

   variables["B_PHO"] = "1"; //enable phone functionality independent of "Bluetooth function" config in KDS

   variables["TourLocation"] = "0"; // for internal use in view spec
   variables["CD_AVAILABLE"] = StringUtils::toString(getCDAvailableKey());
   variables["DVD_AVAILABLE"] = StringUtils::toString(getDVDAvailableKey());
   variables["MUSICBOX_AVAILABLE"] = StringUtils::toString(getMusicBoxAvailableKey());
   variables["B_INT_ElectricVehicle"] = StringUtils::toString(getElectricVehicleKey());
   variables["B_EVE"] = StringUtils::toString(clSDS_ConfigurationFlags::getEveKey());

   variables.insert(dynamicVariables.begin(), dynamicVariables.end());
   variables.insert(configDataFromFile.begin(), configDataFromFile.end());

   return variables;
}


/**************************************************************************//**
 * Function to read the flag values from config.dat file
 ******************************************************************************/
void clSDS_ConfigurationFlags::readConfigFile()
{
   ETG_TRACE_USR1(("clSDS_ConfigurationFlags::readConfigFile "));

   std::ifstream input(CONFIG_FILE_PATH);
   std::string line;

   while (std::getline(input, line))
   {
      size_t comment = line.find("#");     // remove comments
      if (comment != std::string::npos)
      {
         line.erase(comment);
      }
      size_t equals = line.find("=");
      if (equals != std::string::npos)
      {
         std::string key = line.substr(0, equals);    // part before equals sign
         std::string value = line.substr(equals + 1); // part after equals sign
         StringUtils::trim(key);
         StringUtils::trim(value);
         configDataFromFile[key] = value;
         ETG_TRACE_USR1(("config flag: %s", (key + " = " + value).c_str()));
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_ConfigurationFlags::getConfigDataMap(std::map< std::string, std::string>& configMap)
{
   configMap = configDataFromFile;
}


/**************************************************************************//**
 *
 ******************************************************************************/
sds2hmi_fi_tcl_e8_INFO_TCUType::tenType clSDS_ConfigurationFlags::getTcuInformationOnVariantType()
{
   switch (clSDS_KDSConfiguration::getVariantType())
   {
      case clSDS_KDSConfiguration::TYPE_NISSAN:
         return sds2hmi_fi_tcl_e8_INFO_TCUType::FI_EN_INFO_TCUTYPE_NISSAN;
      case clSDS_KDSConfiguration::TYPE_INFINITI:
         return sds2hmi_fi_tcl_e8_INFO_TCUType::FI_EN_INFO_TCUTYPE_INFINITY;
      default:
         return sds2hmi_fi_tcl_e8_INFO_TCUType::FI_EN_INFO_TCUTYPE_NONE;
   }
}


/**************************************************************************//**
 * SDSA HMI to display TCU items 0:Connect Voice Menu 1:Connect to Touch
 ******************************************************************************/
TCUVoiceAssit clSDS_ConfigurationFlags::getTCUVoiceAssitOnVariantType()
{
   switch (clSDS_KDSConfiguration::getVariantType())
   {
      case clSDS_KDSConfiguration::TYPE_NISSAN:
         return TCU_CONNECT_VOICE_MENU;
      case clSDS_KDSConfiguration::TYPE_INFINITI:
         return TCU_CONNECT_IN_TOUCH;
      default:
         return TCU_CONNECT_VOICE_ASSIT_UNKNOWN; // unknown
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::isCDTrackAvailable()
{
   if (dynamicVariables.find("CDTRACK_AVAILABLE") != dynamicVariables.end()
         && dynamicVariables["CDTRACK_AVAILABLE"] == "1")
   {
      return true;
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::isVoiceBargeInFeatureEnable()
{
   if ((clSDS_KDSConfiguration::getAutoBargeIn() == clSDS_KDSConfiguration::STATE_ON_FEEDBACK_INSIDE_PCB) ||
         ((clSDS_KDSConfiguration::getAutoBargeIn() == clSDS_KDSConfiguration::STATE_ON_FEEDBACK_OUT)))
   {
      return true;
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::isVDEForEntryNaviEnable()
{
   if (configDataFromFile.find("B_INT_VDEForEntryNav") != configDataFromFile.end()
         && configDataFromFile["B_INT_VDEForEntryNav"] == "1")
   {
      return true; // True for Renault variants
   }

   return false; // false for Nissan variants
}


/**************************************************************************************
 *
 * ************************************************************************************/
bool clSDS_ConfigurationFlags::isDisableNavigationForVDEUnavailable()
{
   if (configDataFromFile.find("B_INT_DisableNavigationForVDEUnavailable") != configDataFromFile.end()
         && configDataFromFile["B_INT_DisableNavigationForVDEUnavailable"] == "1")
   {
      return true;
   }

   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::isPOIExtendedSearchLocation()
{
   if (configDataFromFile.find("B_INT_POIExtendedSearchLocation") != configDataFromFile.end()
         && configDataFromFile["B_INT_POIExtendedSearchLocation"] == "1")
   {
      return true;
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getAndroidAuto()
{
   return (clSDS_KDSConfiguration::getAndroidAuto() == 1);
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getCarplay()
{
   return (clSDS_KDSConfiguration::getCarplay() == 1);
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getSIRIStatus()
{
   if ((configDataFromFile.find("B_INT_SPVRFeature") != configDataFromFile.end()
         && (configDataFromFile["B_INT_SPVRFeature"] == "1")) && (clSDS_KDSConfiguration::getSIRI() == 1))
   {
      return true;
   }

   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_ConfigurationFlags::getVoiceAssistant()
{
   if ((configDataFromFile.find("B_INT_SPVRFeature") != configDataFromFile.end()
         && (configDataFromFile["B_INT_SPVRFeature"] == "1")) && (clSDS_KDSConfiguration::getVoiceAssistant() == 1))
   {
      return true;
   }

   return false;
}


/**************************************************************************//**
 * This method is used to switch off TCU related commands for some regions requested by Customer in AIVI-72567
 * For any addition/deletion of region, check with FO.
 ******************************************************************************/
bool clSDS_ConfigurationFlags::disableTCUforCurrentRegion()
{
   ETG_TRACE_USR1(("Marketregion %d", clSDS_KDSConfiguration::getMarketRegionType()));
   switch (clSDS_KDSConfiguration::getMarketRegionType())
   {
      case clSDS_KDSConfiguration::OTHER_EUR:
         return true;
      default:
         return false;
   }
}


/**************************************************************************//**
 * This method is used to know if the the variant is A2S1.
 ******************************************************************************/
bool clSDS_ConfigurationFlags::isA2S1()
{
#ifdef VARIANT_S_FTR_ENABLE_A2S1
   return true;
#else
   return false;
#endif
}
