/**************************************************************************//**
 * \file       clSDS_AmbigNumberList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/

#include "application/clSDS_AmbigNumberList.h"
#include "application/clSDS_StringVarList.h"
#include "application/PhonebookContacts.h"
#include "application/clSDS_TextMsgContent.h"
#include "application/AmbigNumberListObserver.h"
#include "application/PhoneUtils.h"

#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_AmbigNumberList.cpp.trc.h"
#endif

clSDS_AmbigNumberList::clSDS_AmbigNumberList(PhonebookContacts* phonebookContacts)
   : _pPhonebookContacts(phonebookContacts)
{
   if (_pPhonebookContacts)
   {
      _pPhonebookContacts->registerObserver(this);
   }
}


clSDS_AmbigNumberList::~clSDS_AmbigNumberList()
{
   _pPhonebookContacts = NULL;
}


void clSDS_AmbigNumberList::setAmbigNumberList(const std::vector<std::pair <std::string, unsigned int> >& contactNumbers)
{
   _ambigContactNumbers = contactNumbers;
   std::string PhoneNumber = _pPhonebookContacts ->getContactNumber(_pPhonebookContacts->getSelectedIndex());
   clSDS_TextMsgContent ::setPhoneNumber(PhoneNumber);

   if (_ambigContactNumbers.size() == 1)
   {
      notifyAmbigNumberUpdated(_ambigContactNumbers[0]);
   }
}


tU32 clSDS_AmbigNumberList::u32GetSize()
{
   return _ambigContactNumbers.size();
}


std::vector<clSDS_ListItems> clSDS_AmbigNumberList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;
   if (u32StartIndex < u32GetSize())
   {
      for (tU32 u32Index = u32StartIndex; u32Index < std::min(u32EndIndex, u32GetSize()); u32Index++)
      {
         clSDS_ListItems oListItem;
         oListItem.oDescription.szString = oGetItem(u32Index);
         oListItem.oIcon.szString = oGetPhoneNumberType(u32Index);
         oListItem.oIcon2.szString = NO_ICON;
         oListItem.oDistance.szString = " ";
         oListItems.push_back(oListItem);
      }
   }
   return oListItems;
}


tBool clSDS_AmbigNumberList::bSelectElement(tU32 u32SelectedIndex)
{
   if (_ambigContactNumbers.size() > 1 && u32SelectedIndex <= _ambigContactNumbers.size())
   {
      notifyAmbigNumberUpdated(_ambigContactNumbers[u32SelectedIndex - 1]);
   }
   return true;
}


tVoid clSDS_AmbigNumberList::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType listType)
{
   NORMAL_M_ASSERT(listType == sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_CONTACT_AMBIGUITY);

   if (_pPhonebookContacts)
   {
      _pPhonebookContacts->getContactNumbers();
   }
}


std::string clSDS_AmbigNumberList::oGetSelectedItem(tU32 u32Index)
{
   ETG_TRACE_USR4(("clSDS_AmbigNumberList::oGetSelectedItem index = %d", u32Index));
   if (u32Index > 0)
   {
      return oGetItem(u32Index - 1);
   }
   return "";
}


std::string clSDS_AmbigNumberList::oGetItem(tU32 u32Index) const
{
   if (_ambigContactNumbers.size() > u32Index)
   {
      return _ambigContactNumbers[u32Index].first;
   }
   return " ";
}


std::string clSDS_AmbigNumberList::oGetPhoneNumberType(tU32 u32Index) const
{
   std::string numberAmbiguityType = " ";
   if (_ambigContactNumbers.size() > u32Index)
   {
      tChar szTempString[3] = "";
      OSALUTIL_s32SaveNPrintFormat(szTempString, sizeof(szTempString), "%d", _ambigContactNumbers[u32Index].second);
      numberAmbiguityType = szTempString;
      return numberAmbiguityType;
   }
   return " ";
}


void clSDS_AmbigNumberList::contactsDetailsUpdated(std::vector<std::pair <std::string, unsigned int> > contactDetails)
{
   ETG_TRACE_USR4(("clSDS_AmbigNumberList::contactsDetailsUpdated - count = %d", contactDetails.size()));
   setAmbigNumberList(contactDetails);
   notifyListObserver();
}


std::vector<sds2hmi_fi_tcl_HMIElementDescription> clSDS_AmbigNumberList::getHmiElementDescription(unsigned int index)
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> hmiElementDescription;
   if (index > 0 && index <= _ambigContactNumbers.size())
   {
      sds2hmi_fi_tcl_HMIElementDescription element;
      if (!_ambigContactNumbers[index].first.empty())
      {
         element.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_PHONENUMBER;
         element.DescriptorValue = _ambigContactNumbers[index - 1].first.c_str();
         hmiElementDescription.push_back(element);
      }
   }
   return hmiElementDescription;
}


void clSDS_AmbigNumberList::registerObservers(AmbigNumberListObserver* obs)
{
   if (obs != NULL)
   {
      _ambigNumberListObservers.push_back(obs);
   }
}


void clSDS_AmbigNumberList::notifyAmbigNumberUpdated(const std::pair<std::string , unsigned int>& numberDetails)
{
   NumberDetails ambigNumberDetails = {numberDetails.first, PhoneUtils::maptoPhoneBookType((sds2hmi_fi_tcl_e8_PHN_NumberType::tenType)numberDetails.second)};
   for (size_t obsIndex = 0; obsIndex != _ambigNumberListObservers.size(); ++obsIndex)
   {
      _ambigNumberListObservers[obsIndex]->ambigNumbersUpdated(ambigNumberDetails);
   }
}
