/**************************************************************************//**
 * \file       clSDS_AmChannelList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "clSDS_AmChannelList.h"
#include "application/clSDS_TunerStateObserver.h"
#include "application/StringUtils.h"
#include "application/clSDS_KDSConfiguration.h"
#include <ctime>

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif

#include "SdsAdapter_Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_AmChannelList.cpp.trc.h"
#endif


using namespace tuner_main_fi;
using namespace tuner_main_fi_types;

/**************************************************************************//**
 *
 ******************************************************************************/
clSDS_AmChannelList::~clSDS_AmChannelList()
{
   _pTunerStateObserver = NULL;
}


/**************************************************************************//**
 *
 ******************************************************************************/
clSDS_AmChannelList::clSDS_AmChannelList(::boost::shared_ptr< sds_am_fi::SdsAmService::SdsAmServiceProxy > sdsAmProxy,
      ::boost::shared_ptr<Tuner_main_fiProxy > tunerProxy)
   : _sdsAmProxy(sdsAmProxy)
   , _tunerProxy(tunerProxy)
   , _pTunerStateObserver(NULL)


{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AmChannelList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _tunerProxy)
   {
      _tunerProxy->sendFID_TUN_G_SURVIELLANCE_LISTUPDATEUpReg(*this);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AmChannelList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _tunerProxy)
   {
      _tunerProxy->sendFID_TUN_G_SURVIELLANCE_LISTUPDATERelUpRegAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AmChannelList::setAmChannelListObserver(clSDS_TunerStateObserver* pTunerStateObserver)
{
   _pTunerStateObserver = pTunerStateObserver;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AmChannelList::onStoreHDChannelNamesError(
   const ::boost::shared_ptr< sds_am_fi::SdsAmService::SdsAmServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_am_fi::SdsAmService::StoreHDChannelNamesError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AmChannelList::onStoreHDChannelNamesResponse(
   const ::boost::shared_ptr< sds_am_fi::SdsAmService::SdsAmServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_am_fi::SdsAmService::StoreHDChannelNamesResponse >& /*response*/)
{
   // Notify SDS MW on tuner status as READY when DB updating is completed
   if (_pTunerStateObserver != NULL)
   {
      _pTunerStateObserver->vTunerStatusChanged(sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_HD_AM, sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY);
   }
   ETG_TRACE_USR1(("clSDS_AmChannelList::onStoreHDChannelNamesResponse : tuner status ready"));
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AmChannelList::onFID_TUN_G_SURVIELLANCE_LISTUPDATEError(const ::boost::shared_ptr< tuner_main_fi::Tuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< tuner_main_fi::FID_TUN_G_SURVIELLANCE_LISTUPDATEError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AmChannelList::onFID_TUN_G_SURVIELLANCE_LISTUPDATEStatus(const ::boost::shared_ptr< tuner_main_fi::Tuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< tuner_main_fi::FID_TUN_G_SURVIELLANCE_LISTUPDATEStatus >& status)
{
   sds_am_fi::SdsAmService::AMChannelItem storeAmChannelElement;
   const T_TunerConfigListElementList& resultConfigListElements = status->getTun_ConfigListElements();
   if (resultConfigListElements.size())
   {
      _amChannelList.clear();
      std::string stationName = "";
      for (::std::vector< T_TunerConfigListElement >::const_iterator itr = resultConfigListElements.begin(); itr != resultConfigListElements.end(); ++itr)
      {
         if ((itr->getSPSName().size()) && (itr->getE8ModulationType() == T_e8_Tun_ModulationType__TUN_MODULATION_HD) &&
               (itr->getE8ConfigListID() == T_e8_Tun_ConfigListID__TUN_TUN_CONFIG_LIST_MW))
         {
            stationName = itr->getSPSName();
            storeAmChannelElement.setFrequency(itr->getU32Frequency());
            if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_MPS())
            {
               stationName.append(" HD1");
               storeAmChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_ONE);
            }
            else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS1())
            {
               stationName.append(" HD2");
               storeAmChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_TWO);
            }
            else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS2())
            {
               stationName.append(" HD3");
               storeAmChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_THREE);
            }
            else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS3())
            {
               stationName.append(" HD4");
               storeAmChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_FOUR);
            }
            else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS4())
            {
               stationName.append(" HD5");
               storeAmChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_FIVE);
            }
            else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS5())
            {
               stationName.append(" HD6");
               storeAmChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_SIX);
            }
            else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS6())
            {
               stationName.append(" HD7");
               storeAmChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_SEVEN);
            }
            else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS7())
            {
               stationName.append(" HD8");
               storeAmChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_EIGHT);
            }
            storeAmChannelElement.setStationName(stationName);
            _amChannelList.push_back(storeAmChannelElement);
         }
      }
      setObjectIdsforHDStationList();
      // check if channel list is updated
      if (isAMHDListUpdated())
      {
         updateAmObjectIDToChannelNameMap();
         // send tuner status to SDS MW as UPDATING
         _pTunerStateObserver->vTunerStatusChanged(sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_HD_AM, sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_UPDATING);
         ETG_TRACE_USR4(("clSDS_AmChannelList::onFID_TUN_G_SURVIELLANCE_LISTUPDATEStatus : tuner status updating"));
         // send trigger to sds_am to store channel list in AM DB
         _sdsAmProxy->sendStoreHDChannelNamesRequest(*this, _amChannelList);
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_AmChannelList::isAMHDListUpdated()
{
   bool stationFound = false;
   for (std::vector<sds_am_fi::SdsAmService::AMChannelItem>::const_iterator it = _amChannelList.begin(); it != _amChannelList.end(); ++it)
   {
      stationFound = false;
      for (std::map<uint32 , std::string>::const_iterator itr_map = _amObjectIDToChannelNameMap.begin(); itr_map != _amObjectIDToChannelNameMap.end(); ++itr_map)
      {
         if (it->getStationName() == itr_map->second)
         {
            stationFound = true;
            break;
         }
      }
      if (!stationFound)
      {
         return true;
      }
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AmChannelList::setObjectIdsforHDStationList()
{
   // generate random numbers and set them as object IDs for HD stations
   time_t seconds;
   //Get value from system clock and place in seconds variable.
   time(&seconds);
   srand((unsigned int) seconds);
   for (std::vector<sds_am_fi::SdsAmService::AMChannelItem>::iterator it = _amChannelList.begin(); it != _amChannelList.end(); ++it)
   {
      it->setObjectID(rand());
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AmChannelList::updateAmObjectIDToChannelNameMap()
{
   _amObjectIDToChannelNameMap.clear();
   for (std::vector<sds_am_fi::SdsAmService::AMChannelItem>::const_iterator it = _amChannelList.begin(); it != _amChannelList.end(); ++it)
   {
      _amObjectIDToChannelNameMap.insert(std::pair<uint32, std::string>(it->getObjectID(), it->getStationName()));
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
uint32 clSDS_AmChannelList::getAudioProgramForObjectID(uint32 objectID)
{
   for (std::vector<sds_am_fi::SdsAmService::AMChannelItem>::const_iterator it = _amChannelList.begin(); it != _amChannelList.end(); ++it)
   {
      if (it->getObjectID() == objectID)
      {
         return it->getAudioProgram();
      }
   }
   return 0;
}


/**************************************************************************//**
 *
 ******************************************************************************/
uint32 clSDS_AmChannelList::getFrequencyForObjectID(uint32 objectID)
{
   for (std::vector<sds_am_fi::SdsAmService::AMChannelItem>::const_iterator it = _amChannelList.begin(); it != _amChannelList.end(); ++it)
   {
      if (it->getObjectID() == objectID)
      {
         return it->getFrequency();
      }
   }
   return 0;
}
