/**************************************************************************//**
 * \file       clSDS_AddressBookList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_AddressBookList.h"
#include "application/clSDS_StringVarList.h"
#include "SdsAdapter_Trace.h"
#include "application/GuiService.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_AddressBookList.cpp.trc.h"
#endif


using namespace org::bosch::cm::navigation::NavigationService;
using namespace sds_gui_fi::SdsGuiService;


/*************************************************************************//**
 *Destructor
 ******************************************************************************/
clSDS_AddressBookList::~clSDS_AddressBookList()
{
   _pGuiService = NULL;
}


/**************************************************************************//**
 *Constructor
 ******************************************************************************/
clSDS_AddressBookList::clSDS_AddressBookList(
   ::boost::shared_ptr< NavigationServiceProxy > naviProxy,
   NaviAddressFormatter* pNaviAddressFormatter,
   GuiService* pGuiService)
   : _naviProxy(naviProxy)
   , _pNaviAddressFormatter(pNaviAddressFormatter)
   , _pGuiService(pGuiService)
   , _selectedIndex(0)
   , _isAddressBookChanged(false)
{
   _isInitialRequest = true;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_AddressBookList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendIsAddressBookChangedRegister(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_AddressBookList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendIsAddressBookChangedDeregisterAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/


void clSDS_AddressBookList::onRequestDetailsForSelectedLocationError(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::RequestDetailsForSelectedLocationError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::onRequestDetailsForSelectedLocationResponse(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::RequestDetailsForSelectedLocationResponse >& /*response*/)
{
   if (_pGuiService)
   {
      _pGuiService->sendEventSignal(Event__SPEECH_DIALOG_GADGET_ACTIVATION);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/


tBool clSDS_AddressBookList::bSelectElement(tU32 u32SelectedIndex)
{
   if (u32SelectedIndex > 0)
   {
      _selectedIndex = u32SelectedIndex - 1;
      clSDS_StringVarList::vSetVariable("$(Address)", oGetItem(_selectedIndex));
      sendDestinationMemoryEntryIdToNavi();
      return TRUE;
   }
   return FALSE;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_AddressBookList::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType listType)
{
   if (listType == sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_ADDRESSBOOK)
   {
      if (_isInitialRequest)
      {
         _naviProxy->sendRequestDestinationMemoryEntriesRequest(*this, DestinationMemoryCategoryType__ADDRESS_BOOK);
         _isInitialRequest = false;
      }
      else if (_isAddressBookChanged)
      {
         _naviProxy->sendRequestDestinationMemoryEntriesRequest(*this, DestinationMemoryCategoryType__ADDRESS_BOOK);
      }
      else
      {
         notifyListObserver();
      }
   }

   else
   {
      vClear();
      notifyListObserver();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::onRequestDestinationMemoryEntriesError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /* proxy*/,
   const ::boost::shared_ptr< RequestDestinationMemoryEntriesError >& /*error*/)
{
   vClear();
   notifyListObserver();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::onRequestDestinationMemoryEntriesResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestDestinationMemoryEntriesResponse >& response)
{
   vClear();
   if (!response->getDestinationMemoryEntries().empty())
   {
      _addressBook = response->getDestinationMemoryEntries();
   }
   notifyListObserver();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::onIsAddressBookChangedError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< IsAddressBookChangedError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::onIsAddressBookChangedUpdate(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< IsAddressBookChangedUpdate >& update)
{
   _isAddressBookChanged = update->getIsAddressBookChanged();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::onSetLocationWithDestinationMemoryEntryError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithDestinationMemoryEntryError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::onSetLocationWithDestinationMemoryEntryResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithDestinationMemoryEntryResponse >& /*response*/)
{
   _naviProxy->sendRequestDetailsForSelectedLocationRequest(*this);
}


/**************************************************************************//**
 *
 ******************************************************************************/

void clSDS_AddressBookList::vClear()
{
   _addressBook.clear();
   _selectedIndex = 0;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_AddressBookList::oGetItem(tU32 u32Index)
{
   if (u32Index < _addressBook.size())
   {
      DestinationMemoryEntry _entry;
      _entry = _addressBook[u32Index].getEntry();
      return _entry.getName();
      // return "";
   }
   else
   {
      return "";
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::vector<clSDS_ListItems> clSDS_AddressBookList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;
   for (tU32 u32Index = u32StartIndex; u32Index < std::min(u32EndIndex, u32GetSize()); u32Index++)
   {
      clSDS_ListItems oListItem;
      oListItem.oDescription.szString = oGetItem(u32Index);
      oListItems.push_back(oListItem);
   }
   return oListItems;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tU32 clSDS_AddressBookList::u32GetSize()
{
   return _addressBook.size();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::vStartGuidance()
{
   _naviProxy->sendStartGuidanceRequest(*this);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::onStartGuidanceError(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< StartGuidanceError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::onStartGuidanceResponse(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr<StartGuidanceResponse >& /*response*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/

std::string clSDS_AddressBookList::oGetHMIListDescriptionItems(tU32 u32Index)
{
   return oGetItem(u32Index);
}


/**************************************************************************//**
 *
 ******************************************************************************/

bool clSDS_AddressBookList::isUserWordAddress(const org::bosch::cm::navigation::NavigationService::DestinationMemoryDetailListElement& address)
{
   std::vector <std::string> currentCountryAddressFormat;
   std::string addressString = "";

   std::map < std::string, std::vector <std::string> >::const_iterator itr;

   for (itr = _countryAddressFormat.begin(); itr != _countryAddressFormat.end(); ++itr)
   {
      if (itr->first == _addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY_CODE])
      {
         currentCountryAddressFormat = itr->second;
      }
   }

   addressString = getFormattedAddress(currentCountryAddressFormat);

   if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].empty())
   {
      addressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].c_str());
   }

   else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].empty())
   {
      addressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].c_str());
   }
   else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].empty())
   {
      addressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].c_str());
   }

   if (!addressString.compare(address.getEntry().getName()))
   {
      return false;
   }
   else
   {
      //userword
      return true;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_AddressBookList::getFormattedAddress(const std::vector <std::string> currentCountryAddressFormat)
{
   std::string formattedAddressString = "";

   for (size_t strIterator = 0; strIterator < currentCountryAddressFormat.size(); strIterator++)
   {
      if (currentCountryAddressFormat[strIterator] == "HNR" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].empty()))
      {
         if (currentCountryAddressFormat[strIterator + 1] == "ROAD")
         {
            formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str()).append(" ");
         }
         else
         {
            formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str()).append(", ");
         }
      }
      if (currentCountryAddressFormat[strIterator] == "ROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].empty()))
      {
         if (currentCountryAddressFormat[strIterator + 1] == "HNR")
         {
            formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str()).append(" ");
         }
         else
         {
            formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str()).append(", ");
         }
      }
      if (currentCountryAddressFormat[strIterator] == "CROSSROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].empty()))
      {
         formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].c_str());
      }
      if (currentCountryAddressFormat[strIterator] == "PLACE" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].empty()))
      {
         formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].c_str()).append(", ");
      }
      if (currentCountryAddressFormat[strIterator] == "SUBCOUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].empty()))
      {
         formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].c_str()).append(", ");
      }
      if (currentCountryAddressFormat[strIterator] == "COUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].empty()))
      {
         formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].c_str());
      }
   }
   return formattedAddressString;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> clSDS_AddressBookList::getHmiListDescription()
{
   _countryAddressFormat = _pNaviAddressFormatter->getAddressFormat();
   std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> addressBookDetails;

   for (size_t i = 0; i < _addressBook.size(); ++i)
   {
      sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
      sds2hmi_fi_tcl_HMIElementDescrptionList hmiElementDescriptionList;

      std::vector<DestinationMemoryEntryItem> destinationMemoryEntryItemList = _addressBook[i].getDetails().getSdsDestinationMemoryEntries();
      setAddressDetails(destinationMemoryEntryItemList);

      if (isUserWordAddress(_addressBook[i]))
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_PLAIN_TEXT;
         hmiElementDescription.DescriptorValue.bSet(_addressBook[i].getEntry().getName().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);

         hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
      }
      else
      {
         if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POINAME;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].c_str());

            hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
         }
         else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].c_str());

            hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
         }

         else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POIBRAND;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].c_str());

            hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
         }
         else
         {
            std::map <std::string, std::vector <std::string> >::iterator itr;

            for (itr = _countryAddressFormat.begin(); itr != _countryAddressFormat.end(); ++itr)
            {
               if ((itr->first) == _addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY_CODE])
               {
                  std::vector <std::string>::iterator strIterator;

                  for (strIterator = itr->second.begin(); strIterator != itr->second.end(); ++strIterator)
                  {
                     if (*strIterator == "HNR" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_HOUSENUM;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "ROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }
                     if (*strIterator == "CROSSROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET2;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }
                     if (*strIterator == "PLACE" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_CITY;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }
                     if (*strIterator == "SUBCOUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STATE;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "COUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_COUNTRY;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }
                  }
               }
            }
         }
      }
      addressBookDetails.push_back(hmiElementDescriptionList);
   }
   return addressBookDetails;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescription>  clSDS_AddressBookList::getHmiElementDescription(unsigned int index)
{
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> _hmiElementDescriptionList;
   _hmiElementDescriptionList.clear();
   if ((index <= _addressBook.size()) && (index > 0))
   {
      std::vector<DestinationMemoryEntryItem> destinationMemoryEntryItemList = _addressBook[index - 1].getDetails().getSdsDestinationMemoryEntries();
      setAddressDetails(destinationMemoryEntryItemList);

      if (isUserWordAddress(_addressBook[index - 1]))
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_PLAIN_TEXT;
         hmiElementDescription.DescriptorValue.bSet(_addressBook[index - 1].getEntry().getName().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);

         _hmiElementDescriptionList.push_back(hmiElementDescription);
      }
      else
      {
         if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POINAME;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].c_str());

            _hmiElementDescriptionList.push_back(hmiElementDescription);
         }

         else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].c_str());

            _hmiElementDescriptionList.push_back(hmiElementDescription);
         }

         else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POIBRAND;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].c_str());

            _hmiElementDescriptionList.push_back(hmiElementDescription);
         }
         else
         {
            std::map <std::string, std::vector <std::string> >::iterator itr;

            for (itr = _countryAddressFormat.begin(); itr != _countryAddressFormat.end(); ++itr)
            {
               if ((itr->first) == _addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY_CODE])
               {
                  std::vector <std::string>::iterator strIterator;

                  for (strIterator = itr->second.begin(); strIterator != itr->second.end(); ++strIterator)
                  {
                     if (*strIterator == "HNR" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].empty()))

                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_HOUSENUM;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "ROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "CROSSROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET2;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "PLACE" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_CITY;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "SUBCOUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STATE;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "COUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_COUNTRY;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }
                  }
               }
            }
         }
      }
   }
   return _hmiElementDescriptionList;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::sendDestinationMemoryEntryIdToNavi()
{
   if (_selectedIndex < _addressBook.size())
   {
      _naviProxy->sendSetLocationWithDestinationMemoryEntryRequest(*this, (uint32)_addressBook[_selectedIndex].getEntry().getEntryID());
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_AddressBookList::setAddressDetails(const std::vector<DestinationMemoryEntryItem>& destinationMemoryEntryItemList)
{
   std::vector<DestinationMemoryEntryItem>::const_iterator entryIt;

   for (entryIt = destinationMemoryEntryItemList.begin(); entryIt != destinationMemoryEntryItemList.end(); ++entryIt)
   {
      _addressDetailsItemStringList[entryIt->getAddressType()] = entryIt->getItemString();
      _addressDetailsItemPhonemeList[entryIt->getAddressType()] = entryIt->getItemPhoneme();
   }
}
