/**************************************************************************//**
 * \file       SxmDataServices.cpp
 *
 * SxmDataService class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/SxmDataServices.h"
#include "application/SxmDataServicesObserver.h"

using namespace sxm_audio_main_fi;
using namespace sxm_audio_main_fi_types;
using namespace sxm_canadian_fuel_main_fi;
using namespace sxm_fuel_main_fi;
using namespace sxm_main_fi_types;
using namespace sxm_movies_main_fi;
using namespace sxm_parking_main_fi;
using namespace sxm_sports_main_fi;
using namespace sxm_stocks_main_fi;
using namespace sxm_sports_main_fi_types;
using namespace sxm_tabweather_main_fi;
using namespace org::bosch::cm::navigation::NavigationSDSService;


/**************************************************************************
 *
 ***************************************************************************/
SxmDataServices::~SxmDataServices()
{
   _pDataService = NULL;
}


/**************************************************************************
 *
 ***************************************************************************/
SxmDataServices::SxmDataServices(
   boost::shared_ptr< sxm_audio_main_fi::Sxm_audio_main_fiProxy > sxmAudioProxy,
   boost::shared_ptr< sxm_canadian_fuel_main_fi::Sxm_canadian_fuel_main_fiProxy > sxmCanadianFuelProxy,
   boost::shared_ptr< sxm_fuel_main_fi::Sxm_fuel_main_fiProxy > sxmFuelProxy,
   boost::shared_ptr< sxm_parking_main_fi::Sxm_parking_main_fiProxy > sxmParkingProxy,
   boost::shared_ptr< sxm_movies_main_fi::Sxm_movies_main_fiProxy > sxmMovieProxy,
   boost::shared_ptr< sxm_sports_main_fi::Sxm_sports_main_fiProxy > sxmSportsProxy,
   boost::shared_ptr< sxm_stocks_main_fi::Sxm_stocks_main_fiProxy > sxmStocksProxy,
   boost::shared_ptr< sxm_tabweather_main_fi::Sxm_tabweather_main_fiProxy > sxmTabWeatherProxy,
   boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy > naviSDSProxy)

   : _sxmAudioProxy(sxmAudioProxy)
   , _sxmCanadianFuelProxy(sxmCanadianFuelProxy)
   , _sxmFuelProxy(sxmFuelProxy)
   , _sxmParkingProxy(sxmParkingProxy)
   , _sxmMovieProxy(sxmMovieProxy)
   , _sxmSportsProxy(sxmSportsProxy)
   , _sxmStocksProxy(sxmStocksProxy)
   , _sxmTabWeatherProxy(sxmTabWeatherProxy)
   , _naviSDSProxy(naviSDSProxy)
   , _pDataService(NULL)
   , _isSxmSignalStatus(false)
   , _issxmAntennaStatus(false)
   , _currentCountryCode(Enum_COUNTRY_CODE_USA)

{
   initializeAdvisoryStatus();
   initializeListStatus();
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::initializeAdvisoryStatus()
{
   _advisoryStatusMap[T_e8_SxmDataServiceType__CANADIAN_FUEL] = ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN;
   _advisoryStatusMap[T_e8_SxmDataServiceType__FUEL] = ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN;
   _advisoryStatusMap[T_e8_SxmDataServiceType__PARKING] = ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN;
   _advisoryStatusMap[T_e8_SxmDataServiceType__MOVIES] = ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN;
   _advisoryStatusMap[T_e8_SxmDataServiceType__STOCKS] = ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN;
   _advisoryStatusMap[T_e8_SxmDataServiceType__SPORTS] = ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN;
   _advisoryStatusMap[T_e8_SxmDataServiceType__TABWEATHER] = ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN;
   _advisoryStatusMap[T_e8_SxmDataServiceType__AGW] = ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN; //AGW:weather Map in SDS
}


void SxmDataServices::initializeListStatus()
{
   _listStatusMap[T_e8_SxmDataServiceType__CANADIAN_FUEL] = ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;
   _listStatusMap[T_e8_SxmDataServiceType__FUEL] = ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;
   _listStatusMap[T_e8_SxmDataServiceType__PARKING] = ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;
   _listStatusMap[T_e8_SxmDataServiceType__MOVIES] = ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;
   _listStatusMap[T_e8_SxmDataServiceType__STOCKS] = ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;
   _listStatusMap[T_e8_SxmDataServiceType__SPORTS] = ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;
   _listStatusMap[T_e8_SxmDataServiceType__TABWEATHER] = ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;
   _listStatusMap[T_e8_SxmDataServiceType__AGW] = ENUM_SXM_DATA_LIST_STATUS_READY; // AGW:weather Map in SDS
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _sxmAudioProxy)
   {
      _sxmAudioProxy->sendSxmDiagSignalQualityUpReg(*this);
      _sxmAudioProxy->sendSxmDataServiceStatusUpReg(*this);
      _sxmAudioProxy->sendSxmDataServiceAntennaStatusUpReg(*this);
   }
   if (proxy == _sxmCanadianFuelProxy)
   {
      _sxmCanadianFuelProxy->sendCanadianFuelInfoListsStatusUpReg(*this);
   }

   if (proxy == _sxmParkingProxy)
   {
      _sxmParkingProxy->sendParkingInfoListsStatusUpReg(*this);
   }

   if (proxy == _sxmFuelProxy)
   {
      _sxmFuelProxy->sendFuelInfoListsStatusUpReg(*this);
   }

   if (proxy == _sxmMovieProxy)
   {
      _sxmMovieProxy->sendTheatreListsStatusUpReg(*this);
   }

   if (proxy == _sxmSportsProxy)
   {
      _sxmSportsProxy->sendSportsServiceDataStatusUpReg(*this);
   }

   if (proxy == _sxmStocksProxy)
   {
      _sxmStocksProxy->sendStockStatusUpReg(*this);
   }

   if (proxy == _sxmTabWeatherProxy)
   {
      _sxmTabWeatherProxy->sendTabWeatherInfoListStatusUpReg(*this);
   }

   if (proxy == _naviSDSProxy)
   {
      _naviSDSProxy->sendPositionStatusInfoRegister(*this);
      _naviSDSProxy->sendPositionStatusInfoGet(*this);
   }
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _sxmAudioProxy)
   {
      _sxmAudioProxy->sendSxmDiagSignalQualityRelUpRegAll();
      _sxmAudioProxy->sendSxmDataServiceStatusRelUpRegAll();
      _sxmAudioProxy->sendSxmDataServiceAntennaStatusRelUpRegAll();
   }

   if (proxy == _sxmCanadianFuelProxy)
   {
      _sxmCanadianFuelProxy->sendCanadianFuelInfoListsStatusRelUpRegAll();
   }
   if (proxy == _sxmFuelProxy)
   {
      _sxmFuelProxy->sendFuelInfoListsStatusRelUpRegAll();
   }
   if (proxy == _sxmParkingProxy)
   {
      _sxmParkingProxy->sendParkingInfoListsStatusRelUpRegAll();
   }

   if (proxy == _sxmMovieProxy)
   {
      _sxmMovieProxy->sendTheatreListsStatusRelUpRegAll();
   }

   if (proxy == _sxmSportsProxy)
   {
      _sxmSportsProxy->sendSportsServiceDataStatusRelUpRegAll();
   }

   if (proxy == _sxmStocksProxy)
   {
      _sxmStocksProxy->sendStockStatusRelUpRegAll();
   }
   if (proxy == _sxmTabWeatherProxy)
   {
      _sxmTabWeatherProxy->sendTabWeatherInfoListStatusRelUpRegAll();
   }
   if (proxy == _naviSDSProxy)
   {
      _naviSDSProxy->sendPositionStatusInfoDeregisterAll();
   }
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::addSxmDataServiceObserver(SxmDataServicesObserver* pDataService)
{
   if (pDataService != NULL)
   {
      _pDataService = pDataService;
   }
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onSxmDiagSignalQualityError(
   const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr<SxmDiagSignalQualityError >& /*error*/)
{
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onSxmDiagSignalQualityStatus(
   const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< SxmDiagSignalQualityStatus >& status)
{
   _isSxmSignalStatus = false;

   if ((status->hasSignalQualityStatus()) &&
         (status->getSignalQualityStatus() != T_e8_Sxm_SignalQuality__FC_SXM_DIAG_SQ_NO_SIGNAL &&
          status->getSignalQualityStatus() != T_e8_Sxm_SignalQuality__FC_SXM_DIAG_SQ_INVALID_SIGNAL))
   {
      _isSxmSignalStatus = true;
   }

   onSxmAntennaAndSignalChangeStatus();
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onSxmDataServiceStatusError(
   const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< sxm_audio_main_fi::SxmDataServiceStatusError >& /*error*/)
{
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onSxmDataServiceStatusStatus(
   const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr<sxm_audio_main_fi::SxmDataServiceStatusStatus >& status)
{
   if (status->hasList() && status->getList().size())
   {
      for (size_t idx = 0; idx < status->getList().size(); ++idx)
      {
         if (isSxmServiceValidInInformationDomain(status->getList()[idx].getType()))
         {
            _advisoryStatusMap[status->getList()[idx].getType()] = getAdvisoryStatusOnSxmDataService(status->getList()[idx].getStatus().getStatus());
            updateSdsInfoMenuSxmServiceType(status->getList()[idx].getType());
         }
      }
      notifyInfoMenuServiceStatus();
   }
}


/**************************************************************************
 *
 ***************************************************************************/

void SxmDataServices::onSxmDataServiceAntennaStatusError(
   const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< SxmDataServiceAntennaStatusError >& /* error*/)
{
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onSxmDataServiceAntennaStatusStatus(
   const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< SxmDataServiceAntennaStatusStatus >& status)
{
   _issxmAntennaStatus = false;

   if (status->hasIsAntennaConnected() && status->getIsAntennaConnected())
   {
      _issxmAntennaStatus = true;
   }
   onSxmAntennaAndSignalChangeStatus();
}


/**************************************************************************
*
***************************************************************************/
void SxmDataServices::onCanadianFuelInfoListsStatusError(
   const ::boost::shared_ptr< Sxm_canadian_fuel_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr<  CanadianFuelInfoListsStatusError >& /*error*/)
{
}


/**************************************************************************
*
***************************************************************************/
void SxmDataServices::onCanadianFuelInfoListsStatusStatus(
   const ::boost::shared_ptr< Sxm_canadian_fuel_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< CanadianFuelInfoListsStatusStatus >& status)
{
   if (_currentCountryCode == Enum_COUNTRY_CODE_CAN)
   {
      if (status->hasInfoList() && status->getInfoList().getListInfo().size())
      {
         for (size_t index = 0 ; index < status->getInfoList().getListInfo().size(); ++index)
         {
            const T_SxmListInfo& sxmListInfo =  status->getInfoList().getListInfo()[index];

            if (sxmListInfo.getListType() == T_e8_SxmListType__NEAR_BY)
            {
               EnumSxmDataListStatus enPreviousListStatus = getListStatusOnRequestedServiceType(T_e8_SxmDataServiceType__CANADIAN_FUEL);
               EnumSxmDataListStatus enCurrentListStatus = getSxmDataListStatus(sxmListInfo.getStatus());

               if (enCurrentListStatus != enPreviousListStatus)
               {
                  _listStatusMap[T_e8_SxmDataServiceType__CANADIAN_FUEL] = enCurrentListStatus;
                  updateSdsInfoMenuSxmServiceType(T_e8_SxmDataServiceType__CANADIAN_FUEL);
                  notifyInfoMenuServiceStatus();
               }
            }
         }
      }
   }
}


/**************************************************************************
*
***************************************************************************/
void SxmDataServices::onFuelInfoListsStatusError(
   const ::boost::shared_ptr< Sxm_fuel_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< FuelInfoListsStatusError >& /*error*/)
{
}


/**************************************************************************
*
***************************************************************************/
void SxmDataServices::onFuelInfoListsStatusStatus(
   const ::boost::shared_ptr< Sxm_fuel_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< FuelInfoListsStatusStatus >& status)
{
   if (_currentCountryCode == Enum_COUNTRY_CODE_USA)
   {
      if (status->hasInfoList() && status->getInfoList().getListInfo().size())
      {
         for (size_t index = 0 ; index < status->getInfoList().getListInfo().size(); ++index)
         {
            const T_SxmListInfo& sxmListInfo = status->getInfoList().getListInfo()[index];

            if (sxmListInfo.getListType() == T_e8_SxmListType__NEAR_BY)
            {
               EnumSxmDataListStatus enPreviousListStatus = getListStatusOnRequestedServiceType(T_e8_SxmDataServiceType__FUEL);
               EnumSxmDataListStatus enCurrentListStatus = getSxmDataListStatus(sxmListInfo.getStatus());

               if (enCurrentListStatus != enPreviousListStatus)
               {
                  _listStatusMap[T_e8_SxmDataServiceType__FUEL] = enCurrentListStatus;
                  updateSdsInfoMenuSxmServiceType(T_e8_SxmDataServiceType__FUEL);
                  notifyInfoMenuServiceStatus();
               }
            }
         }
      }
   }
}


/**************************************************************************
*
***************************************************************************/
void SxmDataServices::onParkingInfoListsStatusError(
   const ::boost::shared_ptr< Sxm_parking_main_fiProxy >& /* proxy*/,
   const ::boost::shared_ptr< ParkingInfoListsStatusError >& /*error*/)
{
}


/**************************************************************************
*
***************************************************************************/
void SxmDataServices::onParkingInfoListsStatusStatus(
   const ::boost::shared_ptr< Sxm_parking_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< ParkingInfoListsStatusStatus >& status)
{
   if (status->hasInfoList() && status->getInfoList().getListInfo().size())
   {
      for (size_t index = 0; index < status->getInfoList().getListInfo().size(); index++)
      {
         const T_SxmListInfo& sxmListInfo = status->getInfoList().getListInfo()[index];

         if (sxmListInfo.getListType() == T_e8_SxmListType__NEAR_BY)
         {
            EnumSxmDataListStatus enPreviousListStatus = getListStatusOnRequestedServiceType(T_e8_SxmDataServiceType__PARKING);
            EnumSxmDataListStatus enCurrentListStatus = getSxmDataListStatus(sxmListInfo.getStatus());

            if (enCurrentListStatus != enPreviousListStatus)
            {
               _listStatusMap[T_e8_SxmDataServiceType__PARKING] = enCurrentListStatus;
               updateSdsInfoMenuSxmServiceType(T_e8_SxmDataServiceType__PARKING);
               notifyInfoMenuServiceStatus();
            }
         }
      }
   }
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onSportsServiceDataStatusError(
   const ::boost::shared_ptr< Sxm_sports_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< SportsServiceDataStatusError >& /*error*/)
{
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onSportsServiceDataStatusStatus(
   const ::boost::shared_ptr< sxm_sports_main_fi::Sxm_sports_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< sxm_sports_main_fi::SportsServiceDataStatusStatus >& status)
{
   EnumSxmDataListStatus enCurrentListStatus = ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;

   if (status->hasSportsStatus() &&
         status->getSportsStatus() == T_e8SportsServiceDataStatus__SPORTS_SERVICE_DATA_UPDATED)
   {
      enCurrentListStatus = ENUM_SXM_DATA_LIST_STATUS_READY;
   }

   EnumSxmDataListStatus enPreviousListStatus = getListStatusOnRequestedServiceType(T_e8_SxmDataServiceType__SPORTS);

   if (enCurrentListStatus != enPreviousListStatus)
   {
      _listStatusMap[T_e8_SxmDataServiceType__SPORTS] = enCurrentListStatus;
      updateSdsInfoMenuSxmServiceType(T_e8_SxmDataServiceType__SPORTS);
      notifyInfoMenuServiceStatus();
   }
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onStockStatusError(
   const ::boost::shared_ptr< sxm_stocks_main_fi::Sxm_stocks_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< sxm_stocks_main_fi::StockStatusError >& /*error*/)
{
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onStockStatusStatus(
   const ::boost::shared_ptr< Sxm_stocks_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< StockStatusStatus >& status)
{
   if (status->hasStockStatus() && status->getStockStatus().getListInfo().size())
   {
      EnumSxmDataListStatus enCurrentListStatus = ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;

      for (size_t index = 0; index < status->getStockStatus().getListInfo().size(); index++)
      {
         const T_SxmListInfo& sxmListInfo = status->getStockStatus().getListInfo()[index];

         if (sxmListInfo.getListType() == T_e8_SxmListType__FAVORITE)
         {
            enCurrentListStatus  = getSxmDataListStatus(sxmListInfo.getStatus());
            EnumSxmDataListStatus enPreviousListStatus = getListStatusOnRequestedServiceType(T_e8_SxmDataServiceType__STOCKS);

            if (enCurrentListStatus != enPreviousListStatus)
            {
               _listStatusMap[T_e8_SxmDataServiceType__STOCKS] = enCurrentListStatus;
               updateSdsInfoMenuSxmServiceType(T_e8_SxmDataServiceType__STOCKS);
               notifyInfoMenuServiceStatus();
            }
         }
      }
   }
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onTabWeatherInfoListStatusError(
   const ::boost::shared_ptr< Sxm_tabweather_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< TabWeatherInfoListStatusError >& /*error*/)
{
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onTabWeatherInfoListStatusStatus(
   const ::boost::shared_ptr< Sxm_tabweather_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< TabWeatherInfoListStatusStatus >& status)
{
   if (status->hasTabweatherListStatus() && status->getTabweatherListStatus().getListInfo().size())
   {
      std::map<sxm_main_fi_types::T_e8_SxmListType, sxm_main_fi_types::T_e8_SxmListStatus> lSxmTabWeatherList;

      for (size_t index = 0; index < status->getTabweatherListStatus().getListInfo().size(); index++)
      {
         const T_SxmListInfo& sxmListInfo = status->getTabweatherListStatus().getListInfo()[index];
         lSxmTabWeatherList[sxmListInfo.getListType()] = sxmListInfo.getStatus();
      }

      EnumSxmDataListStatus enPreviousListStatus = getListStatusOnRequestedServiceType(T_e8_SxmDataServiceType__TABWEATHER);
      EnumSxmDataListStatus enCurrentListStatus = getSxmTabWeatherAreMovieListstatus(lSxmTabWeatherList);

      if (enPreviousListStatus != enCurrentListStatus)
      {
         _listStatusMap[T_e8_SxmDataServiceType__TABWEATHER] = enCurrentListStatus;
         updateSdsInfoMenuSxmServiceType(T_e8_SxmDataServiceType__TABWEATHER);
         notifyInfoMenuServiceStatus();
      }
   }
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onTheatreListsStatusError(
   const ::boost::shared_ptr< Sxm_movies_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< TheatreListsStatusError >& /*error*/)
{
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::onTheatreListsStatusStatus(
   const ::boost::shared_ptr< Sxm_movies_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< TheatreListsStatusStatus >& status)
{
   if ((status->hasTheatreListsStatus()) && (status ->getTheatreListsStatus().hasListInfo()))
   {
      std::map<sxm_main_fi_types::T_e8_SxmListType, sxm_main_fi_types::T_e8_SxmListStatus> lSxmMoveList;
      std::vector <T_SxmListInfo> listInfo = status->getTheatreListsStatus().getListInfo();
      std::vector< T_SxmListInfo >::const_iterator itrListInfo;

      for (itrListInfo = listInfo.begin(); itrListInfo != listInfo.end(); ++itrListInfo)
      {
         T_e8_SxmListStatus listStatus = itrListInfo->getStatus();
         T_e8_SxmListType enType = itrListInfo->getListType();
         lSxmMoveList[enType] = listStatus;
      }

      EnumSxmDataListStatus enPreviousListStatus = getListStatusOnRequestedServiceType(T_e8_SxmDataServiceType__MOVIES);
      EnumSxmDataListStatus enCurrentListStatus = getSxmTabWeatherAreMovieListstatus(lSxmMoveList);

      if (enCurrentListStatus != enPreviousListStatus)
      {
         _listStatusMap[T_e8_SxmDataServiceType__MOVIES] = enCurrentListStatus;
         updateSdsInfoMenuSxmServiceType(T_e8_SxmDataServiceType__MOVIES);
         notifyInfoMenuServiceStatus();
      }
   }
}


/***********************************************************************//**

 ***************************************************************************/
void SxmDataServices::onPositionStatusInfoError(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< PositionStatusInfoError >& /*error*/)
{
}


/***********************************************************************//**

 ***************************************************************************/
void SxmDataServices::onPositionStatusInfoUpdate(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< PositionStatusInfoUpdate >& update)
{
   _currentCountryCode = Enum_COUNTRY_CODE_UNKNOWN;

   if (update->hasPositionStatusInfo())
   {
      if (update->getPositionStatusInfo().getCountryCode().compare("USA") == 0)
      {
         _currentCountryCode = Enum_COUNTRY_CODE_USA;
      }
      else if (update->getPositionStatusInfo().getCountryCode().compare("CAN") == 0)
      {
         _currentCountryCode = Enum_COUNTRY_CODE_CAN;
      }
      else
      {
         _currentCountryCode = Enum_COUNTRY_CODE_UNKNOWN;
      }
   }
}


/**************************************************************************
 *
 ***************************************************************************/
SxmDataServices::EnumSxmDataListStatus SxmDataServices::getSxmTabWeatherAreMovieListstatus(
   const std::map < T_e8_SxmListType,
   T_e8_SxmListStatus > & sxmListReady) const
{
   std::map<T_e8_SxmListType, T_e8_SxmListStatus> ::const_iterator iter = sxmListReady.begin();

   while (iter != sxmListReady.end())
   {
      if ((*iter).first == T_e8_SxmListType__NEAR_BY  ||
            (*iter).first == T_e8_SxmListType__FAVORITE ||
            (*iter).first == T_e8_SxmListType__NEAR_DESTINATION)
      {
         if (getSxmDataListStatus((*iter).second) == ENUM_SXM_DATA_LIST_STATUS_READY)
         {
            return ENUM_SXM_DATA_LIST_STATUS_READY;
         }
      }
      ++iter;
   }
   return ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;
}


/**************************************************************************
 *
 ***************************************************************************/
SxmDataServices::EnumSxmDataListStatus SxmDataServices::getSxmDataListStatus(T_e8_SxmListStatus enStatus) const
{
   switch (enStatus)
   {
      case T_e8_SxmListStatus__READY:
      case T_e8_SxmListStatus__CHANGED:
      case T_e8_SxmListStatus__FROZEN:
      case T_e8_SxmListStatus__ACTIVATED:
         return ENUM_SXM_DATA_LIST_STATUS_READY;
      case T_e8_SxmListStatus__INVALID:
      case T_e8_SxmListStatus__UPDATING:
      case T_e8_SxmListStatus__ERROR:
         return ENUM_SXM_DATA_LIST_STATUS_ERROR;
      default:
         return ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;
   }
}


/**************************************************************************
 *
 ***************************************************************************/
SxmDataServices::EnumSxmServiceAdvisoryStatus SxmDataServices::getAdvisoryStatusOnRequestedServiceType(T_e8_SxmDataServiceType enType) const
{
   EnumSxmServiceAdvisoryStatus advisoryStatus = ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN;

   if (_advisoryStatusMap.find(enType) != _advisoryStatusMap.end())
   {
      advisoryStatus =  _advisoryStatusMap.find(enType)->second;
   }

   if (advisoryStatus == ENUM_SXM_SERVICE_ADVISORY_STATUS_NOT_SUBSCRIBED)
   {
      //  subscribed status is a high priority
   }
   else if (!_issxmAntennaStatus)
   {
      advisoryStatus = ENUM_SXM_SERVICE_ADVISORY_STATUS_CHECK_ANTENNA;
   }
   else if ((!_isSxmSignalStatus) && (advisoryStatus == ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN))
   {
      advisoryStatus = ENUM_SXM_SERVICE_ADVISORY_STATUS_NO_SIGNAL;
   }
   else if ((!_isSxmSignalStatus) && (advisoryStatus != ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN))
   {
      advisoryStatus = ENUM_SXM_SERVICE_ADVISORY_STATUS_READY_NO_SIGNAL;//service subscribed even signal lost
   }

   return  advisoryStatus;
}


/**************************************************************************
 *
 ***************************************************************************/
SxmDataServices::EnumSxmServiceAdvisoryStatus SxmDataServices::getAdvisoryStatusOnSxmDataService(T_e8_SxmDataServiceStatus serviceStatus)const
{
   switch (serviceStatus)
   {
      case T_e8_SxmDataServiceStatus__UNKNOWN :
         return ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN;
      case T_e8_SxmDataServiceStatus__SUBSCRIBED:
         return ENUM_SXM_SERVICE_ADVISORY_STATUS_SUBSCRIBED;
      case T_e8_SxmDataServiceStatus__NOT_SUBSCRIBED:
         return ENUM_SXM_SERVICE_ADVISORY_STATUS_NOT_SUBSCRIBED;
      case T_e8_SxmDataServiceStatus__NOT_AVAILABLE:
         return ENUM_SXM_SERVICE_ADVISORY_STATUS_NOT_AVAILABLE;
      default:
         return ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN;
   }
}


/**************************************************************************
 *
 ***************************************************************************/
SxmDataServices::EnumSxmDataListStatus SxmDataServices::getListStatusOnRequestedServiceType(T_e8_SxmDataServiceType enType) const
{
   EnumSxmDataListStatus enTypeList  = ENUM_SXM_DATA_LIST_STATUS_UNKNOWN;

   if (_listStatusMap.find(enType) != _listStatusMap.end())
   {
      enTypeList =  _listStatusMap.find(enType)->second;
   }
   return enTypeList;
}


/**************************************************************************
 *
 ***************************************************************************/
bool SxmDataServices::isSxmServiceValidInInformationDomain(T_e8_SxmDataServiceType enType)
{
   switch (enType)
   {
      case T_e8_SxmDataServiceType__MOVIES:
      case T_e8_SxmDataServiceType__AGW:
      case T_e8_SxmDataServiceType__TABWEATHER:
      case T_e8_SxmDataServiceType__FUEL:
      case T_e8_SxmDataServiceType__CANADIAN_FUEL:
      case T_e8_SxmDataServiceType__STOCKS:
      case T_e8_SxmDataServiceType__SPORTS:
      case T_e8_SxmDataServiceType__PARKING:
         return true;
      default:
         return false;
   }
}


/**************************************************************************
 *
 ***************************************************************************/
sds2hmi_fi_tcl_e8_Info_Menu_Srvc_Status::tenType SxmDataServices::getInfoMenuServiceStatus(T_e8_SxmDataServiceType enType) const
{
   EnumSxmServiceAdvisoryStatus advisoryStatus = getAdvisoryStatusOnRequestedServiceType(enType);

   if ((advisoryStatus == ENUM_SXM_SERVICE_ADVISORY_STATUS_SUBSCRIBED ||
         advisoryStatus == ENUM_SXM_SERVICE_ADVISORY_STATUS_READY_NO_SIGNAL) &&
         getListStatusOnRequestedServiceType(enType) == ENUM_SXM_DATA_LIST_STATUS_READY)
   {
      return sds2hmi_fi_tcl_e8_Info_Menu_Srvc_Status::FI_EN_AVAILABLE;
   }
   else if (advisoryStatus == ENUM_SXM_SERVICE_ADVISORY_STATUS_NOT_SUBSCRIBED)
   {
      return sds2hmi_fi_tcl_e8_Info_Menu_Srvc_Status::FI_EN_SERVICE_NOT_SUBSCRIBED;
   }
   else if (advisoryStatus == ENUM_SXM_SERVICE_ADVISORY_STATUS_CHECK_ANTENNA)
   {
      return sds2hmi_fi_tcl_e8_Info_Menu_Srvc_Status::FI_EN_ANTENA_NOT_CONNECTED;
   }
   else if (advisoryStatus == ENUM_SXM_SERVICE_ADVISORY_STATUS_NO_SIGNAL)
   {
      return sds2hmi_fi_tcl_e8_Info_Menu_Srvc_Status::FI_EN_NO_SIGNAL;
   }
   else if (advisoryStatus == ENUM_SXM_SERVICE_ADVISORY_STATUS_UNKNOWN)
   {
      return sds2hmi_fi_tcl_e8_Info_Menu_Srvc_Status::FI_EN_SERVICE_NOT_SUBSCRIBED;// service unknown consider as service not subscribed.
   }
   return sds2hmi_fi_tcl_e8_Info_Menu_Srvc_Status::FI_EN_SERVICE_NOT_SUBSCRIBED;;
}


/**************************************************************************
 *
 ***************************************************************************/
void  SxmDataServices::onSxmAntennaAndSignalChangeStatus()
{
   // Sxm Antenna status change "connected" to "disconnected"
   // Sxm Signal status change "Signal" to "No signal", update sxm Service info to Property_InfoServiceStatus

   std::map<T_e8_SxmDataServiceType, EnumSxmServiceAdvisoryStatus> ::const_iterator iter = _advisoryStatusMap.begin();

   while (iter != _advisoryStatusMap.end())
   {
      updateSdsInfoMenuSxmServiceType((*iter).first);
      ++iter;
   }

   notifyInfoMenuServiceStatus();
}


/**************************************************************************
 *
 ***************************************************************************/
void  SxmDataServices::updateSdsInfoMenuSxmServiceType(T_e8_SxmDataServiceType enType)
{
   switch (enType)
   {
      case T_e8_SxmDataServiceType__MOVIES:
         _sdsInfoSxmServiceMap[sds2hmi_fi_tcl_e8_Info_SXM_Services::FI_EN_MOVIES] = getInfoMenuServiceStatus(T_e8_SxmDataServiceType__MOVIES);
         break;

      case T_e8_SxmDataServiceType__AGW: //weather Map
         _sdsInfoSxmServiceMap[sds2hmi_fi_tcl_e8_Info_SXM_Services::FI_EN_WEATHER_MAP] = getInfoMenuServiceStatus(T_e8_SxmDataServiceType__AGW);
         break;

      case T_e8_SxmDataServiceType__STOCKS:
         _sdsInfoSxmServiceMap[sds2hmi_fi_tcl_e8_Info_SXM_Services::FI_EN_STOCKS] = getInfoMenuServiceStatus(T_e8_SxmDataServiceType__STOCKS);
         break;

      case T_e8_SxmDataServiceType__SPORTS:
         _sdsInfoSxmServiceMap[sds2hmi_fi_tcl_e8_Info_SXM_Services::FI_EN_SPORTS] = getInfoMenuServiceStatus(T_e8_SxmDataServiceType__SPORTS);
         break;

      case T_e8_SxmDataServiceType__PARKING:
         _sdsInfoSxmServiceMap[sds2hmi_fi_tcl_e8_Info_SXM_Services::FI_EN_PARKING] = getInfoMenuServiceStatus(T_e8_SxmDataServiceType__PARKING);
         break;

      case T_e8_SxmDataServiceType__CANADIAN_FUEL:
      case T_e8_SxmDataServiceType__FUEL:
         updateFuelService();
         break;

      case T_e8_SxmDataServiceType__TABWEATHER:
         updateTabWeatherService();
         break;

      default:
         break;
   }
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::updateFuelService()
{
   switch (_currentCountryCode)
   {
      case Enum_COUNTRY_CODE_CAN:
         _sdsInfoSxmServiceMap[sds2hmi_fi_tcl_e8_Info_SXM_Services::FI_EN_FUEL_PRICE] = getInfoMenuServiceStatus(T_e8_SxmDataServiceType__CANADIAN_FUEL);
         break;
      case Enum_COUNTRY_CODE_USA:
         _sdsInfoSxmServiceMap[sds2hmi_fi_tcl_e8_Info_SXM_Services::FI_EN_FUEL_PRICE] = getInfoMenuServiceStatus(T_e8_SxmDataServiceType__FUEL);
         break;
      default:
         _sdsInfoSxmServiceMap[sds2hmi_fi_tcl_e8_Info_SXM_Services::FI_EN_FUEL_PRICE] = sds2hmi_fi_tcl_e8_Info_Menu_Srvc_Status::FI_EN_UNKNOWN;
         break;
   }
}


/**************************************************************************
 *
 ***************************************************************************/
void SxmDataServices::updateTabWeatherService()
{
   sds2hmi_fi_tcl_e8_Info_Menu_Srvc_Status::tenType enStatus  = getInfoMenuServiceStatus(T_e8_SxmDataServiceType__TABWEATHER);
   _sdsInfoSxmServiceMap[sds2hmi_fi_tcl_e8_Info_SXM_Services::FI_EN_CURRENT_WEATHER] = enStatus;
   _sdsInfoSxmServiceMap[sds2hmi_fi_tcl_e8_Info_SXM_Services::FI_EN_SIX_HOUR_FORECAST] = enStatus;
   _sdsInfoSxmServiceMap[sds2hmi_fi_tcl_e8_Info_SXM_Services::FI_EN_FIVE_DAY_FORECAST] = enStatus;
}


/**************************************************************************
 *
 ***************************************************************************/
void  SxmDataServices::notifyInfoMenuServiceStatus()
{
   if (_pDataService)
   {
      _pDataService->notifyInfoMenuServiceStatus(_sdsInfoSxmServiceMap);
   }
}
