/**************************************************************************//**
 * \file       PhonebookContacts.cpp
 *
 * PhonebookContacts class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/

#include <application/PhonebookObserver.h>
#include "application/PhonebookContacts.h"
#include "application/StringUtils.h"

using namespace most_PhonBk_fi_types;
using namespace most_PhonBk_fi_types_Extended;
using namespace MOST_Tel_FI;
using namespace most_Tel_fi_types;


PhonebookContacts::PhonebookContacts(boost::shared_ptr<MOST_PhonBk_FI::MOST_PhonBk_FIProxy> phonebookProxy,
                                     boost::shared_ptr< MOST_Tel_FI::MOST_Tel_FIProxy > telephoneProxy)
   : _phonebookProxy(phonebookProxy)
   , _selectedIndex(0)
   , _telephoneProxy(telephoneProxy)
{
}


PhonebookContacts::~PhonebookContacts()
{
}


void PhonebookContacts::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _phonebookProxy)
   {
      _phonebookProxy->sendPreferredPhoneBookSortOrderUpReg(*this);
   }
}


void PhonebookContacts::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _phonebookProxy)
   {
      _phonebookProxy->sendPreferredPhoneBookSortOrderRelUpRegAll();
   }
}


void PhonebookContacts::onCreateContactListError(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::CreateContactListError >& /*error*/)
{
   notifyContactListObserver();
}


void PhonebookContacts::onCreateContactListResult(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::CreateContactListResult >& result)
{
   const unsigned int listHandle = result->getU16ListHandle();
   const unsigned int listLength = result->getU16ListLength();

   if (listLength)
   {
      _phonebookProxy->sendRequestPhoneBookListSliceStart(*this, listHandle, 0, listLength);
   }
   else
   {
      notifyContactListObserver();
   }
}


void PhonebookContacts::onRequestPhoneBookListSliceError(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::RequestPhoneBookListSliceError >& /*error*/)
{
   notifyContactListObserver();
}


void PhonebookContacts::onRequestPhoneBookListSliceResult(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::RequestPhoneBookListSliceResult >& result)
{
   _contactsList = result->getOPhoneBookListSliceResult();
   notifyContactListObserver();
}


void PhonebookContacts::onGetContactDetailsExtendedError(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::GetContactDetailsExtendedError >& /*error*/)
{
   notifyContactDetailsObserver(); // numbers
}


void PhonebookContacts::onGetContactDetailsExtendedResult(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::GetContactDetailsExtendedResult >& result)
{
   _contactDetails = result->getOContactDetailsExtended();
   notifyContactDetailsObserver();
}


void PhonebookContacts::onPreferredPhoneBookSortOrderError(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::PreferredPhoneBookSortOrderError >& /*error*/)
{
}


void PhonebookContacts::onPreferredPhoneBookSortOrderStatus(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::PreferredPhoneBookSortOrderStatus >& /*status*/)
{
}


std::vector<std::string> PhonebookContacts::getContactsNames()
{
   std::vector<std::string> contactsNames;
   for (unsigned int i = 0; i < _contactsList.size(); i++)
   {
      contactsNames.push_back(getContactName(_contactsList[i]));
   }
   return contactsNames;
}


std::string PhonebookContacts::getContactName(T_PhonBkPhoneBookListSliceResultItem contactDetails) const
{
   std::string fullName = "";

   if (_phonebookProxy->hasPreferredPhoneBookSortOrder()
         && _phonebookProxy->getPreferredPhoneBookSortOrder().getE8PreferredPhoneBookSortOrder() == T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_LASTNAME)
   {
      fullName = contactDetails.getSLastName() + " " + contactDetails.getSFirstName();
   }
   else
   {
      fullName = contactDetails.getSFirstName() + " " + contactDetails.getSLastName();
   }

   return StringUtils::trim(fullName);
}


std::vector<std::pair <std::string, unsigned int> > PhonebookContacts::getContactDetails()
{
   std::vector<std::pair <std::string, unsigned int> > contactNumberDetails;
   std::pair <std::string, unsigned int> contactNumber;

   if (_contactDetails.getOPhoneNumber1().getSNumber() != "")
   {
      contactNumber.first = _contactDetails.getOPhoneNumber1().getSNumber();
      contactNumber.second = getPhoneNumberType(_contactDetails.getOPhoneNumber1().getE8Type());
      addNewContact(contactNumberDetails, contactNumber);
   }
   if (_contactDetails.getOPhoneNumber2().getSNumber() != "")
   {
      contactNumber.first = _contactDetails.getOPhoneNumber2().getSNumber();
      contactNumber.second = getPhoneNumberType(_contactDetails.getOPhoneNumber2().getE8Type());
      addNewContact(contactNumberDetails, contactNumber);
   }
   if (_contactDetails.getOPhoneNumber3().getSNumber() != "")
   {
      contactNumber.first = _contactDetails.getOPhoneNumber3().getSNumber();
      contactNumber.second = getPhoneNumberType(_contactDetails.getOPhoneNumber3().getE8Type());
      addNewContact(contactNumberDetails, contactNumber);
   }
   if (_contactDetails.getOPhoneNumber4().getSNumber() != "")
   {
      contactNumber.first = _contactDetails.getOPhoneNumber4().getSNumber();
      contactNumber.second = getPhoneNumberType(_contactDetails.getOPhoneNumber4().getE8Type());
      addNewContact(contactNumberDetails, contactNumber);
   }
   if (_contactDetails.getOPhoneNumber5().getSNumber() != "")
   {
      contactNumber.first = _contactDetails.getOPhoneNumber5().getSNumber();
      contactNumber.second = getPhoneNumberType(_contactDetails.getOPhoneNumber5().getE8Type());
      addNewContact(contactNumberDetails, contactNumber);
   }
   if (_contactDetails.getOPhoneNumber6().getSNumber() != "")
   {
      contactNumber.first = _contactDetails.getOPhoneNumber6().getSNumber();
      contactNumber.second = getPhoneNumberType(_contactDetails.getOPhoneNumber6().getE8Type());
      addNewContact(contactNumberDetails, contactNumber);
   }
   if (_contactDetails.getOPhoneNumber7().getSNumber() != "")
   {
      contactNumber.first = _contactDetails.getOPhoneNumber7().getSNumber();
      contactNumber.second = getPhoneNumberType(_contactDetails.getOPhoneNumber7().getE8Type());
      addNewContact(contactNumberDetails, contactNumber);
   }
   if (_contactDetails.getOPhoneNumber8().getSNumber() != "")
   {
      contactNumber.first = _contactDetails.getOPhoneNumber8().getSNumber();
      contactNumber.second = getPhoneNumberType(_contactDetails.getOPhoneNumber8().getE8Type());
      addNewContact(contactNumberDetails, contactNumber);
   }

   return contactNumberDetails;
}


void PhonebookContacts::retrieveContactsList(sds2hmi_fi_tcl_e8_HMI_ListType::tenType hmiListType)
{
   _contactsList.clear();

   if ((_phonebookProxy->isAvailable()) && (_telephoneProxy->hasActivePhoneDevice()))
   {
      _phonebookProxy->sendCreateContactListStart(*this, _telephoneProxy->getActivePhoneDevice().getU8DeviceHandle(),
            getPhonebookListType(hmiListType),
            getPhonebookListSortType(hmiListType));
   }
   else
   {
      notifyContactListObserver();
   }
}


most_PhonBk_fi_types_Extended::T_e8_PhonBkContactSortType PhonebookContacts::getPhonebookListSortType(sds2hmi_fi_tcl_e8_HMI_ListType::tenType hmiListType)
{
   if (hmiListType == sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_PHONE_VB_ALL_ENTRIES)
   {
      return T_e8_PhonBkContactSortType__e8PB_LIST_SORT_POSITION;
   }
   else
   {
      if (_phonebookProxy->hasPreferredPhoneBookSortOrder()
            && _phonebookProxy->getPreferredPhoneBookSortOrder() == T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_LASTNAME)
      {
         return T_e8_PhonBkContactSortType__e8PB_LIST_SORT_NUMBER_LASTNAME;
      }

      return T_e8_PhonBkContactSortType__e8PB_LIST_SORT_NUMBER_FIRSTNAME;
   }
}


most_PhonBk_fi_types_Extended::T_e8_PhonBkContactListType PhonebookContacts::getPhonebookListType(sds2hmi_fi_tcl_e8_HMI_ListType::tenType hmiListType)
{
   if (hmiListType == sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_PHONE_VB_ALL_ENTRIES)
   {
      return most_PhonBk_fi_types_Extended::T_e8_PhonBkContactListType__VEHICLE;
   }
   else //hmiListType == sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_PHONE_PB_ALL_ENTRIES
   {
      return most_PhonBk_fi_types_Extended::T_e8_PhonBkContactListType__PHONEBOOK;
   }
}


void PhonebookContacts::getContactNumbers()
{
   _contactDetails.clear();

   if (_selectedIndex < _contactsList.size())
   {
      unsigned int contactHandle = _contactsList.at(_selectedIndex).getU32ContactHandle();
      _phonebookProxy->sendGetContactDetailsExtendedStart(*this, contactHandle);
   }
   else
   {
      notifyContactDetailsObserver(); //numbers
   }
}


std::string PhonebookContacts::getContactNumber(unsigned int selectedIndex)
{
   std::vector<std::pair <std::string, unsigned int> > contactNumbers = getContactDetails();

   if (contactNumbers.size() > 0)
   {
      if (selectedIndex > 0 && selectedIndex <= contactNumbers.size())
      {
         return contactNumbers[selectedIndex - 1].first;
      }
      else
      {
         return contactNumbers[0].first;
      }
   }

   return "";
}


void PhonebookContacts::setSelectedIndex(unsigned int selectedIndex)
{
   _selectedIndex = selectedIndex;
}


void PhonebookContacts::registerObserver(PhonebookObserver* obs)
{
   _phonebookObservers.push_back(obs);
}


void PhonebookContacts::notifyContactListObserver()
{
   for (unsigned int i = 0; i < _phonebookObservers.size(); i++)
   {
      if (_phonebookObservers[i])
      {
         _phonebookObservers[i]->contactsListUpdated(getContactsNames());
      }
   }
}


void PhonebookContacts::notifyContactDetailsObserver()
{
   for (unsigned int i = 0; i < _phonebookObservers.size(); i++)
   {
      if (_phonebookObservers[i])
      {
         _phonebookObservers[i]->contactsDetailsUpdated(getContactDetails());
      }
   }
}


PhonebookContacts::enPhoneNumberType PhonebookContacts::getPhoneNumberType(most_PhonBk_fi_types_Extended::T_e8_PhonBkNumberType numberType)
{
   switch (numberType)
   {
      case T_e8_PhonBkNumberType__CELL:
         return PhonebookContacts::EN_CELL_1;
      case T_e8_PhonBkNumberType__HOME :
         return PhonebookContacts::EN_HOME_1;
      case T_e8_PhonBkNumberType__WORK :
         return PhonebookContacts::EN_WORK_1;
      case T_e8_PhonBkNumberType__PREFERRED :
         return PhonebookContacts::EN_PREFFERED;
      case T_e8_PhonBkNumberType__OTHER :
      default:
         return PhonebookContacts::EN_OTHERS;
   }
}


unsigned int PhonebookContacts::getSelectedIndex()
{
   return _selectedIndex;
}


void PhonebookContacts::addNewContact(std::vector<std::pair <std::string, unsigned int> >& contactNumberDetails, std::pair <std::string, unsigned int>& contactNumber)
{
   bool duplicateExist = false;
   for (size_t i = 0; i < contactNumberDetails.size(); i++)
   {
      if ((contactNumberDetails[i].second == sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_UNKNOWN) && (contactNumberDetails[i].first == contactNumber.first))
      {
         duplicateExist = true;
         break;
      }
   }
   if (!duplicateExist)
   {
      contactNumberDetails.push_back(contactNumber);
   }
}
