/**************************************************************************//**
 * \file       NdsCountryStateRequestor.cpp
 *
 * clSDS_Method_NavDataGetCountryStateList method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "NdsCountryStateRequestor.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/NdsCountryStateRequestor.cpp.trc.h"
#endif


NdsCountryStateRequestor::NdsCountryStateRequestor
(::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy > pSds2NaviProxy)
   : _sds2NaviProxy(pSds2NaviProxy)
{
   _ndsRequestor = NDSID_UNKNWON;
   _requestedUpdateRegionId = 0;
   _requestedCountryNdsId = 0;
   _requestedsubCountryNdsIdSize = 0;
   _countryStateNameAvailability = false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
NdsCountryStateRequestor::~NdsCountryStateRequestor()
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NdsCountryStateRequestor::getNdsCountryStateIds()
{
   _ndsRequestor = NDSID_FOR_COUNTRY;
   _sds2NaviProxy->sendGetIdListOfCountryAndSubCountryRequest(*this);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NdsCountryStateRequestor::onGetIdListOfCountryAndSubCountryError
(const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& /*proxy*/,
 const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::GetIdListOfCountryAndSubCountryError >& /*error*/)
{
   notifyNdsCountryStateObservers();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NdsCountryStateRequestor::onGetIdListOfCountryAndSubCountryResponse
(const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& /*proxy*/,
 const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::GetIdListOfCountryAndSubCountryResponse >& response)
{
   std::vector< org::bosch::cm::navigation::NavigationSDSService::GlobalNamedObjectAndRegionId > countryIdList;

   countryIdList = response->getCountryIdList();

   _stateIdList = response->getSubCountryIdList();

   getNamesForIds(countryIdList);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NdsCountryStateRequestor::getNamesForIds(std::vector< org::bosch::cm::navigation::NavigationSDSService::GlobalNamedObjectAndRegionId > ndsIds)
{
   std::vector< org::bosch::cm::navigation::NavigationSDSService::LanguageEntry > languageEntryList;
   org::bosch::cm::navigation::NavigationSDSService::NDSFeatureType ndsFeatureType;

   org::bosch::cm::navigation::NavigationSDSService::LanguageEntry languageEntry;

   languageEntry.setISOLanguageCode("");
   languageEntry.setISOCountryCode("");
   languageEntry.setTranscriptCode("latn");
   languageEntryList.push_back(languageEntry);

   ndsFeatureType = org::bosch::cm::navigation::NavigationSDSService::NDSFeatureType__NAMEDOBJECT;

   std::vector< org::bosch::cm::navigation::NavigationSDSService::GlobalNamedObjectId > namedObjectIdList;
   std::vector< org::bosch::cm::navigation::NavigationSDSService::GlobalNamedObjectAndRegionId >::iterator ndsIdsIt;

   for (ndsIdsIt = ndsIds.begin(); ndsIdsIt != ndsIds.end(); ++ndsIdsIt)
   {
      org::bosch::cm::navigation::NavigationSDSService::GlobalNamedObjectId namedObjectId;
      namedObjectId.setGlobalUpdateRegionId(ndsIdsIt->getGlobalUpdateRegionId());
      namedObjectId.setNamedObjectId(ndsIdsIt->getNamedObjectId());
      namedObjectIdList.push_back(namedObjectId);
   }

   _sds2NaviProxy->sendGetNamesForNDSIdsRequest(*this, namedObjectIdList, languageEntryList, ndsFeatureType, false);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NdsCountryStateRequestor::onGetNamesForNDSIdsError
(const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& /*proxy*/,
 const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::GetNamesForNDSIdsError >& error)
{
   notifyNdsCountryStateObservers();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NdsCountryStateRequestor::onGetNamesForNDSIdsResponse
(const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& /*proxy*/,
 const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::GetNamesForNDSIdsResponse >& response)
{
   if (NDSID_FOR_COUNTRY == _ndsRequestor)
   {
      _countryNameList = response->getNamedObjectIdStringsList();

      std::vector< org::bosch::cm::navigation::NavigationSDSService::NamedObjectIdStrings > ::const_iterator itr;
      for (itr = _countryNameList.begin(); itr != _countryNameList.end(); ++itr)
      {
         CountryDetails countryDetail;

         countryDetail.countryNdsId = itr->getGlobalNamedObjectId().getNamedObjectId();
         countryDetail.isoLanguageCode = itr->getLanguageEntry().getISOLanguageCode();
         countryDetail.stringnames = itr->getSdsNamedStringsList();

         _countryNamesList.push_back(countryDetail);
      }

      _ndsRequestor = NDSID_FOR_STATE;

      _requestedsubCountryNdsIdSize = 0;

      requestStateNames(_stateIdList.at(_requestedsubCountryNdsIdSize));
   }
   else if (NDSID_FOR_STATE == _ndsRequestor)
   {
      _stateNameList = response->getNamedObjectIdStringsList();
      std::vector< org::bosch::cm::navigation::NavigationSDSService::NamedObjectIdStrings > ::const_iterator itr;
      for (itr = _stateNameList.begin(); itr != _stateNameList.end(); ++itr)
      {
         stateDetails stateDetail;
         stateDetail.stateNdsId = itr->getGlobalNamedObjectId().getNamedObjectId();
         stateDetail.isoLanguageCode = itr->getLanguageEntry().getISOLanguageCode();
         stateDetail.stringnames = itr->getSdsNamedStringsList();
         stateDetail.countryNdsId = _requestedCountryNdsId;
         _stateNamesList.push_back(stateDetail);
      }

      std::vector< org::bosch::cm::navigation::NavigationSDSService::NamedObjectIdStrings > subCountryNameList;

      subCountryNameList = response->getNamedObjectIdStringsList();

      _requestedsubCountryNdsIdSize++;

      if (_requestedsubCountryNdsIdSize < _stateIdList.size())
      {
         requestStateNames(_stateIdList.at(_requestedsubCountryNdsIdSize));
      }
      else
      {
         _countryStateNameAvailability = true;
         notifyNdsCountryStateObservers();
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NdsCountryStateRequestor::requestStateNames(org::bosch::cm::navigation::NavigationSDSService::SubCountryIds subCountryIds)
{
   std::vector< org::bosch::cm::navigation::NavigationSDSService::GlobalNamedObjectAndRegionId > subCountryIdList;

   subCountryIdList = subCountryIds.getSubCountryIdList();

   _requestedCountryNdsId = subCountryIds.getCountryId().getNamedObjectId();

   getNamesForIds(subCountryIdList);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NdsCountryStateRequestor::updateCountrySubCountryPhonemes(std::vector< org::bosch::cm::navigation::NavigationSDSService::NamedObjectIdStrings > stateList)
{
   _requestedsubCountryNdsIdSize++;
}


/**************************************************************************//**
*
******************************************************************************/
void NdsCountryStateRequestor::addCountryStateObserver(NdsCountryStateObserver* pObserver)
{
   if (pObserver != NULL)
   {
      _countryStateObservers.push_back(pObserver);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void NdsCountryStateRequestor::notifyNdsCountryStateObservers()
{
   std::vector<NdsCountryStateObserver*>::iterator it = _countryStateObservers.begin();

   while (it != _countryStateObservers.end())
   {
      if (*it != NULL)
      {
         (*it)->NdsCountryStateUpdated();
      }
      ++it;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
unsigned long NdsCountryStateRequestor::getNdsCountryListSize()
{
   return _requestedsubCountryNdsIdSize;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string NdsCountryStateRequestor::getNdsStateName(std::string statename, std::string languageId)
{
   unsigned long correctStateNdsId = 0;
   std::vector < stateDetails >::const_iterator itr;
   for (itr = _stateNamesList.begin(); itr != _stateNamesList.end(); ++itr)
   {
      std::vector <std::string>::const_iterator stateNameIterator;
      for (stateNameIterator = itr->stringnames.begin(); stateNameIterator != itr->stringnames.end(); ++stateNameIterator)
      {
         if (statename == *stateNameIterator)
         {
            setCountryNameForState(itr->countryNdsId, languageId);
            correctStateNdsId = itr->stateNdsId;
         }
      }
   }
   std::vector < stateDetails >::const_iterator iterator;
   for (iterator = _stateNamesList.begin(); iterator != _stateNamesList.end(); ++iterator)
   {
      if ((iterator->stateNdsId == correctStateNdsId) && (iterator->isoLanguageCode == languageId))
      {
         return *(iterator->stringnames.begin());
      }
   }
   return "";
}


/***********************************************************************//**
 *
 ***************************************************************************/

void NdsCountryStateRequestor::setCountryNameForState(unsigned long countryNdsId, std::string languageCode)
{
   std::vector < CountryDetails >::const_iterator itr;
   for (itr = _countryNamesList.begin(); itr != _countryNamesList.end(); ++itr)
   {
      if ((countryNdsId == itr->countryNdsId) && (languageCode == itr->isoLanguageCode))
      {
         _correctCountryName = *(itr->stringnames.begin());
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string NdsCountryStateRequestor::getNdsCountryName()
{
   return _correctCountryName;
}


/***********************************************************************//**
 *
 ***************************************************************************/
bool NdsCountryStateRequestor::isCountryStateInfoAvailable() const
{
   return _countryStateNameAvailability;
}
