/**************************************************************************//**
 * \file       NaviPOIIconsList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/NaviPOIIconsList.h"
#include "Sds2HmiServer/functions/clSDS_Property_NaviPOIIconList.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/NaviPOIIconsList.cpp.trc.h"
#endif


using namespace org::bosch::cm::navigation::NavigationSDSService;


/**************************************************************************//**
* Constructor
******************************************************************************/

NaviPOIIconsList::NaviPOIIconsList(
   GuiService* guiService,
   ::boost::shared_ptr<org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy> pNaviSDSProxy,
   clSDS_Property_NaviPOIIconList* pclSDSPropertyNaviPOIIconList)
   : _pGuiService(guiService),
     _naviSDSProxy(pNaviSDSProxy),
     _pclSDSPropertyNaviPOIIconList(pclSDSPropertyNaviPOIIconList)
{
   _showIcon = true;
   _removeIcon = false;
   _showRemoveRequested = false;
}


/**************************************************************************//**
* Destructor
******************************************************************************/
NaviPOIIconsList::~NaviPOIIconsList()
{
}


/***********************************************************************//**
*
***************************************************************************/
void NaviPOIIconsList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviSDSProxy)
   {
      _naviSDSProxy->sendDynamicMapPoiCategoryWithLanguageRegister(*this);
      _naviSDSProxy->sendDynamicMapPoiCategoryWithLanguageGet(*this);
   }
}


/***********************************************************************//**
*
***************************************************************************/
void NaviPOIIconsList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviSDSProxy)
   {
      _naviSDSProxy->sendDynamicMapPoiCategoryWithLanguageDeregisterAll();
   }
}


/***********************************************************************//**
*
***************************************************************************/

void NaviPOIIconsList::onGetDynamicPoiIconCategoriesInMapError(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::GetDynamicPoiIconCategoriesInMapError >& /*error*/)
{
}


/***********************************************************************//**
*
***************************************************************************/

void NaviPOIIconsList::onGetDynamicPoiIconCategoriesInMapResponse(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::GetDynamicPoiIconCategoriesInMapResponse >& response)
{
   if (response->hasDynamicMapPoiCategory())
   {
      _dynamicMapPOICategory.clear();
      _dynamicMapPOICategory = response->getDynamicMapPoiCategory();
   }

   if (_showRemoveRequested)
   {
      _showRemoveRequested = false;
      handlePOIIconRequest(_requestedDisplayAction);
   }
}


/***********************************************************************//**
*
***************************************************************************/

void NaviPOIIconsList::onSetDynamicPoiIconCategoriesInMapError(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::SetDynamicPoiIconCategoriesInMapError >& /*error*/)
{
}


/***********************************************************************//**
*
***************************************************************************/

void NaviPOIIconsList::onSetDynamicPoiIconCategoriesInMapResponse(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::SetDynamicPoiIconCategoriesInMapResponse >& response)
{
}


/***********************************************************************//**
*
***************************************************************************/
void NaviPOIIconsList::onDynamicMapPoiCategoryWithLanguageUpdate(const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& proxy, const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::DynamicMapPoiCategoryWithLanguageUpdate >& update)
{
   if (update->hasDynamicMapPoiCategoryWithLanguage())
   {
      _dynamicMapPOICategory.clear();
      _dynamicMapPOICategory = update->getDynamicMapPoiCategoryWithLanguage().getDynamicMapPoiCategories();
      _languageEntry = update->getDynamicMapPoiCategoryWithLanguage().getPoiCategoriesLanguageEntry();
   }
   handleDynamicMapPoiCategoriesWithLanguage();
}


/***********************************************************************//**
*
***************************************************************************/
void NaviPOIIconsList::onDynamicMapPoiCategoryWithLanguageError(const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& /*proxy*/, const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::DynamicMapPoiCategoryWithLanguageError >& /*error*/)
{
}


/***********************************************************************//**
*
***************************************************************************/
void NaviPOIIconsList::handlePOIIconRequest(sds2hmi_fi_tcl_Nav_IconSetting oNavIconSettings)
{
   switch (oNavIconSettings.DisplayAction.enType)
   {
      case sds2hmi_fi_tcl_e8_NAV_IconDisplayAction::FI_EN_ICONDSPLY_SHOWALL:
         sendShowOrRemoveallPOIIconRequest(_showIcon);
         break;
      case sds2hmi_fi_tcl_e8_NAV_IconDisplayAction::FI_EN_ICONDSPLY_SHOW:
      {
         setNaviPOIIconCategory(oNavIconSettings.CategoryString.szValue, _showIcon);
      }
      break;
      case sds2hmi_fi_tcl_e8_NAV_IconDisplayAction::FI_EN_ICONDSPLY_REMOVEALL:
         sendShowOrRemoveallPOIIconRequest(_removeIcon);
         break;
      case sds2hmi_fi_tcl_e8_NAV_IconDisplayAction::FI_EN_ICONDSPLY_REMOVE:
      {
         setNaviPOIIconCategory(oNavIconSettings.CategoryString.szValue, _removeIcon);
      }
      break;
      default:
         break;
   }
}


/***********************************************************************//**
*
***************************************************************************/
void NaviPOIIconsList::sendShowOrRemoveallPOIIconRequest(bool poiIconShowStatus)
{
   for (size_t itr = 0; itr < _dynamicMapPOICategory.size(); ++itr)
   {
      _dynamicMapPOICategory[itr].setPoiCategoryVisibilityOnMap(poiIconShowStatus);
   }
   _naviSDSProxy->sendSetDynamicPoiIconCategoriesInMapRequest(*this, _dynamicMapPOICategory);
   _pGuiService->sendEventSignal(sds_gui_fi::SdsGuiService::Event__SPEECH_DIALOG_SHOW_NAVI_SOURCECHANGE_ACTIVE);
}


/***********************************************************************//**
*
***************************************************************************/
void NaviPOIIconsList::setNaviPOIIconCategory(const std::string categoryString, bool poiIconShowStatus)
{
   for (size_t itr = 0; itr < _dynamicMapPOICategory.size(); ++itr)
   {
      if (!strcasecmp(_dynamicMapPOICategory[itr].getPoiCategoryName().c_str(), categoryString.c_str()))
      {
         _dynamicMapPOICategory[itr].setPoiCategoryVisibilityOnMap(poiIconShowStatus);
      }
   }
   _naviSDSProxy->sendSetDynamicPoiIconCategoriesInMapRequest(*this, _dynamicMapPOICategory);
   _pGuiService->sendEventSignal(sds_gui_fi::SdsGuiService::Event__SPEECH_DIALOG_SHOW_NAVI_SOURCECHANGE_ACTIVE);
}


/**********************************************************************
*
**************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> NaviPOIIconsList::getHmiListDescription()
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> mapIconsList;

   for (size_t itr = 0; itr < _dynamicMapPOICategory.size(); ++itr)
   {
      sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
      sds2hmi_fi_tcl_HMIElementDescrptionList hmiElementDescriptionList;
      if (!_dynamicMapPOICategory[itr].getPoiCategoryName().empty())
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY;
         hmiElementDescription.DescriptorValue.bSet(_dynamicMapPOICategory[itr].getPoiCategoryName().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
         hmiElementDescription.DestinationIdList.push_back(getDestinationIdList(_dynamicMapPOICategory[itr].getSdsNameObjectId()));

         hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
      }
      mapIconsList.push_back(hmiElementDescriptionList);
   }
   return mapIconsList;
}


/**********************************************************************
*
**************************************************************************/
void NaviPOIIconsList::sendGetDynamicPOIIconCategoriesRequest(sds2hmi_fi_tcl_Nav_IconSetting requestedAction)
{
   _requestedDisplayAction = requestedAction;
   _showRemoveRequested = true;
   _naviSDSProxy->sendGetDynamicPoiIconCategoriesInMapRequest(*this);
}


/**********************************************************************
*
**************************************************************************/
tU32 NaviPOIIconsList::u32GetSize()
{
   return _dynamicMapPOICategory.size();
}


/**********************************************************************
*
**************************************************************************/
std::vector<clSDS_ListItems> NaviPOIIconsList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;
   for (tU32 u32Index = u32StartIndex; u32Index < std::min(u32EndIndex, u32GetSize()); u32Index++)
   {
      clSDS_ListItems oListItem;
      oListItem.oCommand.szString = oGetItem(u32Index);
      oListItem.bIsListScreenWithoutIndex = TRUE;
      oListItems.push_back(oListItem);
   }
   return oListItems;
}


/**************************************************************************//**
*
******************************************************************************/
std::string NaviPOIIconsList::oGetItem(tU32 u32Index)
{
   if (u32Index < (_dynamicMapPOICategory.size()))
   {
      return _dynamicMapPOICategory[u32Index].getPoiCategoryName();
   }
   else
   {
      return "";
   }
}


/**********************************************************************
*
**************************************************************************/
tBool NaviPOIIconsList::bSelectElement(tU32 /*u32SelectedIndex*/)
{
   return true;
}


void NaviPOIIconsList::handleDynamicMapPoiCategoriesWithLanguage()
{
   sds2hmi_sdsfi_tclMsgNaviPOIIconListStatus naviPOIlistInfo;

   for (size_t itr = 0; itr < _dynamicMapPOICategory.size(); ++itr)
   {
      sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;

      if (!_dynamicMapPOICategory[itr].getPoiCategoryName().empty())
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY;
         hmiElementDescription.DescriptorValue.bSet(_dynamicMapPOICategory[itr].getPoiCategoryName().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
         hmiElementDescription.DestinationIdList.push_back(getDestinationIdList(_dynamicMapPOICategory[itr].getSdsNameObjectId()));

         naviPOIlistInfo.DescriptionList.push_back(hmiElementDescription);
      }
   }

   if (!_languageEntry.getISOCountryCode().empty() && !_languageEntry.getISOLanguageCode().empty())
   {
      naviPOIlistInfo.LanguageID.ISO3166_CountryCode.enType = convertNaviStringtoISOCountryCode(_languageEntry.getISOCountryCode());
      naviPOIlistInfo.LanguageID.ISO639_3_SDSLanguageCode.enType = convertNaviStringtoISOLanguageCode(_languageEntry.getISOLanguageCode());
      if (_pclSDSPropertyNaviPOIIconList)
      {
         _pclSDSPropertyNaviPOIIconList->sendNaviPOIIconListToSDS(naviPOIlistInfo);
      }
   }
}


/**********************************************************************
*
**************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescription> NaviPOIIconsList::getHmiElementDescription(unsigned int index)
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> hmiElementDescriptionList;
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   if ((index > 0) && (index <= _dynamicMapPOICategory.size()))
   {
      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY;
      hmiElementDescription.DescriptorValue.bSet(_dynamicMapPOICategory[index - 1].getPoiCategoryName().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
      hmiElementDescription.DestinationIdList.push_back(getDestinationIdList(_dynamicMapPOICategory[index - 1].getSdsNameObjectId()));

      hmiElementDescriptionList.push_back(hmiElementDescription);
   }
   return hmiElementDescriptionList;
}


/***********************************************************************//**

 ***************************************************************************/
sds2hmi_fi_tcl_e16_ISOCountryCode::tenType NaviPOIIconsList::convertNaviStringtoISOCountryCode(const std::string& countryCode)
{
   uint32 value = 0;
   for (size_t i = 0; i < countryCode.size(); i++)
   {
      value <<= 5;
      value += std::toupper(countryCode[i]) - '@';
   }
   ETG_TRACE_USR4(("country code %u = '%s'", value, countryCode.c_str()));
   return (sds2hmi_fi_tcl_e16_ISOCountryCode::tenType) value;
}


/***********************************************************************//**

 ***************************************************************************/
sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::tenType NaviPOIIconsList::convertNaviStringtoISOLanguageCode(const std::string& languageCode)
{
   uint32 value = 0;
   for (size_t i = 0; i < languageCode.size(); i++)
   {
      value <<= 5;
      value += std::toupper(languageCode[i]) - '@';
   }
   ETG_TRACE_USR4(("language code %u = '%s'", value, languageCode.c_str()));
   return (sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::tenType) value;
}


sds2hmi_fi_tcl_NDSID NaviPOIIconsList::getDestinationIdList(GlobalNamedObjectId nameObjectId)
{
   sds2hmi_fi_tcl_NDSID dataNDSId;
   if (nameObjectId.getIsValid())
   {
      dataNDSId.u16UpdateRegionId = nameObjectId.getGlobalUpdateRegionId().getUpdateRegionId();
      dataNDSId.u32ProductId = nameObjectId.getGlobalUpdateRegionId().getProductId();
      dataNDSId.u8SupplierId = nameObjectId.getGlobalUpdateRegionId().getSupplierId();
      dataNDSId.u32NDSId = nameObjectId.getNamedObjectId();
      dataNDSId.eNDSFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_CATEGORY;
   }
   return dataNDSId;
}
