/**************************************************************************//**
 * \file       NaviDetourList.cpp
 *
 * NaviDetourList  class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/NaviDetourList.h"
#include "SdsAdapter_Trace.h"
#include "application/clSDS_StringVarList.h"
#include "application/clSDS_ConfigurationFlags.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/NaviDetourList.cpp.trc.h"
#endif


using namespace org::bosch::cm::navigation::NavigationService;


NaviDetourList::NaviDetourList(
   ::boost::shared_ptr< NavigationServiceProxy> naviProxy,
   clSDS_Property_NaviStatus* naviStatus)

   : _pNaviProxy(naviProxy)
   , _pNaviStatus(naviStatus)
{
   _detourListRequested = false;
   mapNaviUnittoSDSUnit();
}


/***********************************************************************//**
 *
 ***************************************************************************/
NaviDetourList::~NaviDetourList()
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NaviDetourList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _pNaviProxy)
   {
      _pNaviProxy->sendJamAheadListRegister(*this);
      _detourListRequested = true;
      _pNaviProxy->sendRequestJamAheadListRequest(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NaviDetourList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _pNaviProxy)
   {
      _pNaviProxy->sendJamAheadListDeregisterAll();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/

void NaviDetourList::onJamAheadListError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< JamAheadListError >& /*error*/)
{
   notifyListObserverError();
}


/***********************************************************************//**
 *
 ***************************************************************************/

void NaviDetourList::onJamAheadListUpdate(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< JamAheadListUpdate >& update)
{
   if (update->hasJamAheadList())
   {
      if (update->getJamAheadList().getJamAheadActivityStatus() == JamAheadActivityStatus__INACTIVE)
      {
         clSDS_ConfigurationFlags::setDynamicVariable("DetourOn", false);
         _jamAheadList.clear();

         if (update->getJamAheadList().hasJamAheadListElements())
         {
            unsigned int jamAheadElementSize = update->getJamAheadList().getJamAheadListElements().size();

            for (size_t i = 0; i < jamAheadElementSize ; ++i)
            {
               if (update->getJamAheadList().getJamAheadListElements()[i].getIsEnabled())
               {
                  _jamAheadList.push_back(update->getJamAheadList().getJamAheadListElements()[i]);
                  _pNaviStatus->setNaviDetourStatus(false);
               }
            }

            notifyListObserver();
         }
         else
         {
            notifyListObserverError();
         }
      }
      else if (update->getJamAheadList().getJamAheadActivityStatus() == JamAheadActivityStatus__NO_DETOUR_POSSIBLE)
      {
         clSDS_ConfigurationFlags::setDynamicVariable("DetourOn", true);
         _jamAheadList.clear();
         _pNaviStatus->setNaviDetourStatus(true);
         notifyListObserver();
      }
      else if (update->getJamAheadList().getJamAheadActivityStatus() == JamAheadActivityStatus__ACTIVE)
      {
         clSDS_ConfigurationFlags::setDynamicVariable("DetourOn", true);
         notifyListObserverError();
         _pNaviStatus->setNaviDetourStatus(true);
      }
      else
      {
         notifyListObserverError();
      }
   }
   if (_detourListRequested)
   {
      _pNaviProxy->sendReleaseJamAheadListRequest(*this);
      _detourListRequested = false;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NaviDetourList::onRequestJamAheadListResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestJamAheadListResponse >& /*response*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/

void NaviDetourList::onRequestJamAheadListError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestJamAheadListError >& /*error*/)
{
   notifyListObserverError();
}


/***********************************************************************//**
 *
 ***************************************************************************/


void NaviDetourList::onReleaseJamAheadListError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< ReleaseJamAheadListError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/

void NaviDetourList::onReleaseJamAheadListResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< ReleaseJamAheadListResponse >& /*response*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/

tU32 NaviDetourList::u32GetSize()
{
   return _jamAheadList.size();
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid NaviDetourList::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType listType)
{
   if (sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_NAV_DISTANCE_DETOUR == listType)
   {
      _detourListRequested = true;
      _pNaviProxy->sendRequestJamAheadListRequest(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<clSDS_ListItems> NaviDetourList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;
   for (tU32 u32Index = u32StartIndex; u32Index < std::min(u32EndIndex, u32GetSize()); u32Index++)
   {
      clSDS_ListItems oListItem;
      oListItem.oCommand.szString = oGetItem(u32Index);
      oListItem.bIsListScreenWithoutIndex = TRUE;
      oListItems.push_back(oListItem);
   }
   return oListItems;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string NaviDetourList::oGetItem(tU32 u32Index)
{
   if (u32Index < _jamAheadList.size())
   {
      return _jamAheadList[u32Index].getDistance().c_str();
   }
   else
   {
      return "";
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tBool NaviDetourList::bSelectElement(tU32 u32SelectedIndex)
{
   if (u32SelectedIndex > 0)
   {
      return true;
   }
   return false;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescription> NaviDetourList::getHmiElementDescription(unsigned int index)
{
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> hmiElementDescriptionList;
   if ((index <= _jamAheadList.size()) && (index > 0))
   {
      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_GENERIC_NUMBER;
      hmiElementDescription.DescriptorValue.bSet(_jamAheadList[index - 1].getDistanceWithoutUnit().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
      hmiElementDescriptionList.push_back(hmiElementDescription);

      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_DISTANCE_UNIT;
      hmiElementDescription.DescriptorValue.bSet("", sds2hmi_fi_tclString::FI_EN_UTF8);
      hmiElementDescription.DescriptorId = _distanceUnitMap[_jamAheadList[index - 1].getDistanceUnit()];
      hmiElementDescriptionList.push_back(hmiElementDescription);
   }
   return hmiElementDescriptionList;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> NaviDetourList::getHmiListDescription()
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> detourListDetails;
   for (size_t i = 0; i < _jamAheadList.size(); ++i)
   {
      sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
      sds2hmi_fi_tcl_HMIElementDescrptionList hmiElementDescriptionList;
      if (_jamAheadList[i].getIsEnabled())
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_GENERIC_NUMBER;
         hmiElementDescription.DescriptorValue.bSet((_jamAheadList[i].getDistanceWithoutUnit().c_str()), sds2hmi_fi_tclString::FI_EN_UTF8);
         hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);

         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_DISTANCE_UNIT;
         hmiElementDescription.DescriptorValue.bSet("", sds2hmi_fi_tclString::FI_EN_UTF8);
         hmiElementDescription.DescriptorId = _distanceUnitMap[_jamAheadList[i].getDistanceUnit()];
         hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
      }
      detourListDetails.push_back(hmiElementDescriptionList);
   }
   return detourListDetails;
}


/***********************************************************************//**
 *
 ***************************************************************************/
unsigned long NaviDetourList::getIndexForDistance(std::string distance, sds2hmi_fi_tcl_e8_Distance_Unit::tenType unit)
{
   for (size_t index = 0; index < _jamAheadList.size(); ++index)
   {
      std::string distanceWithoutUnit = _jamAheadList[index].getDistanceWithoutUnit().c_str();
      sds2hmi_fi_tcl_e8_Distance_Unit::tenType indexUnit = _distanceUnitMap[_jamAheadList[index].getDistanceUnit()];

      if ((distanceWithoutUnit == distance) && (indexUnit == unit))
      {
         return index;
      }
   }

   return 0;
}


/***********************************************************************//**
 *
 ***************************************************************************/

void NaviDetourList::mapNaviUnittoSDSUnit()
{
   _distanceUnitMap[DistanceUnit__METERS] = sds2hmi_fi_tcl_e8_Distance_Unit::FI_EN_METER;
   _distanceUnitMap[DistanceUnit__KILOMETERS] = sds2hmi_fi_tcl_e8_Distance_Unit::FI_EN_KM;
   _distanceUnitMap[DistanceUnit__FEET] = sds2hmi_fi_tcl_e8_Distance_Unit::FI_EN_FEET;
   _distanceUnitMap[DistanceUnit__YARD] = sds2hmi_fi_tcl_e8_Distance_Unit::FI_EN_YARD;
   _distanceUnitMap[DistanceUnit__MILES] = sds2hmi_fi_tcl_e8_Distance_Unit::FI_EN_MILE;
   _distanceUnitMap[DistanceUnit__INVALID] = sds2hmi_fi_tcl_e8_Distance_Unit::FI_EN_UNKNOWN;
}
