/**************************************************************************//**
 * \file       EarlyStartupPlayer.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/EarlyStartupPlayer.h"
#include "application/clSDS_KDSConfiguration.h"
#include "application/EarlyPromptStatusObserver.h"
#include "application/SdsAudioSource.h"
#include "sds_gui_fi/SettingsServiceConst.h"
#include <string.h>

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_02_if.h"
#include "dp_tclfc_sds_adapter_SpeechSettings.h"
#endif

#include "SdsAdapter_Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/EarlyStartupPlayer.cpp.trc.h"
#endif


using namespace vehicle_main_fi_types;
using namespace sds_gui_fi::SettingsService;


#define ARRAY_SIZE(array)     (sizeof (array) / sizeof (array)[0])


struct EarlyPromptMapping
{
   T_e8_Language_Code vehicleLanguage;
   ActiveSpeakerGender enTypeVoicePreference;
   std::string earlyPromptFile;
};


/**
 * EARLY_PROMPT_SDS() describes on select HMI language and speaker name
 * for which provide  respective early prompt file path
 */
#define EARLY_PROMPT_SDS(vehicle_lang, sds_gender, promptfile) \
   { \
      T_e8_Language_Code__##vehicle_lang, \
      ActiveSpeakerGender__##sds_gender, \
      #promptfile \
   }


/**
 * EarlyPromptMapping language i.e vehicle language code mapped with
 * respective support Lang_CountryCode and speaker name like  Male or Female(Example:ENG_USA_TOM)
 */
static const EarlyPromptMapping g_EarlyPromptMapping[]  =
{
   EARLY_PROMPT_SDS(Arabic,             MALE,     ARA_SAU_Tarik),
   EARLY_PROMPT_SDS(Bulgarian,          FEMALE,   BUL_BGR_Daria),
   EARLY_PROMPT_SDS(Czech,              FEMALE,   CES_CZE_Zuzana),
   EARLY_PROMPT_SDS(Danish,             FEMALE,   DAN_DNK_Ida),
   EARLY_PROMPT_SDS(Dutch,              FEMALE,   NLD_NLD_Claire),
   EARLY_PROMPT_SDS(English_Australian, FEMALE,   ENG_AUS_Karen),
   EARLY_PROMPT_SDS(English_India,      FEMALE,   ENG_IND_Veena),
   EARLY_PROMPT_SDS(English_UK,         FEMALE,   ENG_GBR_Serena),
   EARLY_PROMPT_SDS(English_UK,         MALE,     ENG_GBR_Daniel),
   EARLY_PROMPT_SDS(English_US,         FEMALE,   ENG_USA_Ava),
   EARLY_PROMPT_SDS(English_US,         MALE,     ENG_USA_Tom),
   EARLY_PROMPT_SDS(Finnish,            MALE,     FIN_FIN_Mikko),
   EARLY_PROMPT_SDS(French_Canadian,    FEMALE,   FRA_CAN_Julie),
   EARLY_PROMPT_SDS(French,             FEMALE,   FRA_FRA_Audrey),
   EARLY_PROMPT_SDS(French,             MALE,     FRA_FRA_Thomas),
   EARLY_PROMPT_SDS(German,             FEMALE,   DEU_DEU_Andrea),
   EARLY_PROMPT_SDS(Greek,              MALE,     ELL_GRC_Alexandros),
   EARLY_PROMPT_SDS(Italian,            FEMALE,   ITA_ITA_Silvia),
   EARLY_PROMPT_SDS(Japanese,           FEMALE,   JPN_JPN_Sakura),
   EARLY_PROMPT_SDS(Japanese,           MALE,     JPN_JPN_Daisuke),
   EARLY_PROMPT_SDS(Korean,             FEMALE,   KOR_KOR_Sora),
   EARLY_PROMPT_SDS(Norwegian,          FEMALE,   NOR_NOR_Stine),
   EARLY_PROMPT_SDS(Polish,             FEMALE,   POL_POL_Agata),
   EARLY_PROMPT_SDS(Portuguese_Brazilian, FEMALE, POR_BRA_Luciana),
   EARLY_PROMPT_SDS(Portuguese,          FEMALE,  POR_PRT_Joana),
   EARLY_PROMPT_SDS(Russian,             FEMALE,  RUS_RUS_Milena),
   EARLY_PROMPT_SDS(Russian,             MALE,    RUS_RUS_Yuri),
   EARLY_PROMPT_SDS(Spanish,             FEMALE,  SPA_ESP_Marisol),
   EARLY_PROMPT_SDS(Spanish_Mexican,     FEMALE,  SPA_MEX_Paulina),
   EARLY_PROMPT_SDS(Spanish_Mexican,     FEMALE,  Spanish_Latin_American),
   EARLY_PROMPT_SDS(Swedish,             FEMALE,  SWE_SWE_Alva),
   EARLY_PROMPT_SDS(Turkish,             FEMALE,  TUR_TUR_Aylin),
   EARLY_PROMPT_SDS(Taiwanese,           FEMALE,  CMN_TWN_Mei_Ja),
   EARLY_PROMPT_SDS(Chinese_Mandarin_Simplified_Chinese_character, FEMALE, CMN_CHN_Tian_Tian),
   EARLY_PROMPT_SDS(Chinese_Cantonese_Traditional_Chinese_character, FEMALE, YUE_HKG_Sin_Ji),
};


/**************************************************************************//**
*
******************************************************************************/
EarlyStartupPlayer::EarlyStartupPlayer(
   SdsAudioSource& sdsAudioSource)
   : _sdsAudioSource(sdsAudioSource)
   , _earlyPromptStatusObserver(NULL)
   , _audioChannelRequested(false)
{
   _sdsAudioSource.addObserver(this);
}


/**************************************************************************//**
*
******************************************************************************/
EarlyStartupPlayer::~EarlyStartupPlayer()
{
   _earlyPromptStatusObserver = NULL;
}


/**************************************************************************//**
*
******************************************************************************/
void EarlyStartupPlayer::onAudioSourceStateChanged(arl_tenActivity state)
{
   if (_audioChannelRequested)
   {
      ETG_TRACE_USR4(("EarlyStartupPlayer::onAudioSourceStateChanged state = %d", state));

      if (state == ARL_EN_ISRC_ACT_ON)
      {
         startPlayback();
      }
      else // ARL_EN_ISRC_ACT_OFF or ARL_EN_ISRC_ACT_PAUSE
      {
         requestStopPlayback();
         notifyPlaybackTerminated();
      }
   }
}


/**************************************************************************//**
*
******************************************************************************/
void EarlyStartupPlayer::onExpired(asf::core::Timer& /*timer*/, boost::shared_ptr<asf::core::TimerPayload> /*data*/)
{
   ETG_TRACE_USR4(("EarlyStartupPlayer::onExpired"));

   bool exitFlag = false;

   if (_gstreamerData.bus)
   {
      // Check for End Of Stream or error messages on bus
      // The global exit_flag will be set in case of EOS or error. Exit if the flag is set
      GstMessageType gstMessageType = (GstMessageType)(GST_MESSAGE_EOS | GST_MESSAGE_ERROR);
      _gstreamerData.message = gst_bus_poll(_gstreamerData.bus, gstMessageType, (GstClockTimeDiff)100);

      if (_gstreamerData.message)
      {
         if (GST_MESSAGE_TYPE(_gstreamerData.message))
         {
            exitFlag = checkBusCallback();
         }
         gst_message_unref(_gstreamerData.message);
      }
      if (exitFlag)
      {
         _audioChannelRequested = false;
         stopPlayback();
         notifyPlaybackFinished();

         ETG_TRACE_USR4(("EarlyStartupPlayer::onExpired - playback ended"));
      }
      else
      {
         startTimer();
      }
   }
}


/**************************************************************************//**
*
******************************************************************************/
void EarlyStartupPlayer::requestStartPlayback()
{
   if (_audioChannelRequested == false) // avoid starting playback multiple times
   {
      ETG_TRACE_USR4(("EarlyStartupPlayer::requestStartPlayback"));

      _audioChannelRequested = true;

      _sdsAudioSource.sendAudioRouteRequest(ARL_SRC_VRU, ARL_EN_ISRC_ACT_ON);
   }
};


/**************************************************************************//**
*
******************************************************************************/
void EarlyStartupPlayer::requestStopPlayback()
{
   if (_audioChannelRequested == true)
   {
      ETG_TRACE_USR4(("EarlyStartupPlayer::requestStopPlayback"));

      _audioChannelRequested = false;

      stopPlayback();

      _sdsAudioSource.sendAudioRouteRequest(ARL_SRC_VRU, ARL_EN_ISRC_ACT_OFF);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void EarlyStartupPlayer::startPlayback()
{
   ETG_TRACE_USR4(("EarlyStartupPlayer::startPlayback"));

   std::string filepath = getEarlySDSPromptPath(getPromptLanguage());
   ETG_TRACE_USR4(("EarlyStartupPlayer::startPlayback filepath %s", filepath.c_str()));
   gst_init(NULL, NULL);

   memset(_gstreamerData.filelocation, 0, sizeof(_gstreamerData.filelocation));
   std::strncpy(_gstreamerData.filelocation, filepath.c_str(), sizeof(_gstreamerData.filelocation) - 1);

   if (!createPipeline())
   {
      deletePipeline();
   }

   if (initAudioPlaybackPipeline())
   {
      if (!addBinPlaybackToPipe())
      {
         deletePipeline();
      }

      startPlaybackPipe();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void EarlyStartupPlayer::stopPlayback()
{
   ETG_TRACE_USR4(("EarlyStartupPlayer::stopPlayback"));

   removeBinPlaybackFromPipe();
   deletePipeline();
}


/**************************************************************************//**
*Create the pipeline element
******************************************************************************/
bool EarlyStartupPlayer::createPipeline()
{
   _gstreamerData.pipeline = gst_pipeline_new("audio_pipeline");
   if (_gstreamerData.pipeline == NULL)
   {
      return false;
   }
   gst_element_set_state(_gstreamerData.pipeline, GST_STATE_NULL);
   return true;
}


/**************************************************************************//**
*Callback function for dynamically linking the "wavparse" element and "alsasink" element
******************************************************************************/
void EarlyStartupPlayer::onPadAdded(GstElement* /*src_element*/, GstPad* src_pad, gpointer data)
{
   ETG_TRACE_USR4(("EarlyStartupPlayer::onPadAdded"));

   GstElement* sink_element = (GstElement*) data;
   GstPad* sink_pad = gst_element_get_static_pad(sink_element, "sink");

   gst_pad_link(src_pad, sink_pad);

   gst_object_unref(sink_pad);
}


/**************************************************************************//**
*
******************************************************************************/
void EarlyStartupPlayer::onStreamStatus(GstBus* bus, GstMessage* message, gpointer user_data)
{
   GstStreamStatusType type;
   GstElement* owner;

   gst_message_parse_stream_status(message, &type, &owner);

   switch (type)
   {
      case GST_STREAM_STATUS_TYPE_ENTER:
         pthread_t thId;
         int policy;
         struct sched_param param;
         thId = pthread_self(); //onStreamStatus callback is called from streaming thread itself
         if (0 == pthread_getschedparam(thId, &policy, &param))
         {
            policy = SCHED_FIFO;
            param.sched_priority = sched_get_priority_max(policy);
            if (0 != pthread_setschedparam(thId, policy, &param))
            {
               ETG_TRACE_USR4(("EarlyStartupPlayer::onStreamStatus - GST_STREAM_STATUS_TYPE_ENTER - could not raise streaming thread priority!"));
            }
         }
         else
         {
            ETG_TRACE_USR4(("EarlyStartupPlayer::onStreamStatus - GST_STREAM_STATUS_TYPE_ENTER - could not get current streaming thread priority!"));
         }
         break;
      case GST_STREAM_STATUS_TYPE_LEAVE:
         break;
      default:
         break;
   }
}


/**************************************************************************//**
*Setup the pipeline
******************************************************************************/
bool EarlyStartupPlayer::initAudioPlaybackPipeline()
{
   ETG_TRACE_USR4(("EarlyStartupPlayer::initAudioPlaybackPipeline"));

   _gstreamerData.file_source = gst_element_factory_make("filesrc", "filesource");

   if (strstr(_gstreamerData.filelocation, ".mp3"))
   {
      _gstreamerData.audio_decoder = gst_element_factory_make("decodebin2", "audiomp3decoder");
   }

   if (strstr(_gstreamerData.filelocation, ".wav"))
   {
      _gstreamerData.audio_decoder = gst_element_factory_make("wavparse", "audiowavdecoder");
   }

   _gstreamerData.audioconvert = gst_element_factory_make("audioconvert", "audioconverter");

   _gstreamerData.alsasink = gst_element_factory_make("alsasink", "audiosink");

   if (!_gstreamerData.file_source || !_gstreamerData.audio_decoder || !_gstreamerData.audioconvert || !_gstreamerData.alsasink)
   {
      return false;
   }

   g_object_set(G_OBJECT(_gstreamerData.file_source), "location", _gstreamerData.filelocation, NULL); //lint !e826 Suspicious pointer-to-pointer conversion
   g_object_set(G_OBJECT(_gstreamerData.alsasink), "device", "AdevAcousticoutSpeech", NULL); //lint !e826 Suspicious pointer-to-pointer conversion

   _gstreamerData.bin_playback = gst_bin_new("bin_playback");

   gst_bin_add_many(GST_BIN(_gstreamerData.bin_playback), _gstreamerData.file_source, _gstreamerData.audio_decoder, _gstreamerData.alsasink, NULL); //lint !e826 Suspicious pointer-to-pointer conversion

   if (gst_element_link_many(_gstreamerData.file_source, _gstreamerData.audio_decoder, NULL) != TRUE)
   {
      return false;
   }

   gst_element_link_many(_gstreamerData.audio_decoder, _gstreamerData.alsasink, NULL);

   g_signal_connect(_gstreamerData.audio_decoder, "pad-added", G_CALLBACK(onPadAdded), _gstreamerData.alsasink);

   _gstreamerData.bus = gst_pipeline_get_bus(GST_PIPELINE(_gstreamerData.pipeline));
   gst_bus_enable_sync_message_emission(_gstreamerData.bus);
   g_signal_connect(_gstreamerData.bus, "sync-message::stream-status", G_CALLBACK(onStreamStatus), NULL);

   ETG_TRACE_USR4(("EarlyStartupPlayer::initAudioPlaybackPipeline - SUCCESS"));

   return true;
}


/**************************************************************************//**
*Starts the pipeline
******************************************************************************/
void EarlyStartupPlayer::startPlaybackPipe()
{
   gst_element_set_state(_gstreamerData.pipeline, GST_STATE_PLAYING);
   startTimer();
}


/**************************************************************************//**
*Add the pipeline to the bin
******************************************************************************/
bool EarlyStartupPlayer::addBinPlaybackToPipe()
{
   if ((gst_bin_add(GST_BIN(_gstreamerData.pipeline), _gstreamerData.bin_playback)) != TRUE) //lint !e826 Suspicious pointer-to-pointer conversion
   {
      return false;
   }

   if (gst_element_set_state(_gstreamerData.pipeline, GST_STATE_NULL) == GST_STATE_CHANGE_SUCCESS)
   {
      ETG_TRACE_USR4(("EarlyStartupPlayer::addBinPlaybackToPipe - SUCCESS"));
      return true;
   }
   else
   {
      return false;
   }
}


/**************************************************************************//**
*Disconnect the pipeline and the bin
******************************************************************************/
void EarlyStartupPlayer::removeBinPlaybackFromPipe()
{
   gst_element_set_state(_gstreamerData.pipeline, GST_STATE_NULL);
   gst_element_set_state(_gstreamerData.bin_playback, GST_STATE_NULL);
   gst_bin_remove(GST_BIN(_gstreamerData.pipeline), _gstreamerData.bin_playback); //lint !e826 Suspicious pointer-to-pointer conversion
}


/**************************************************************************//**
*Cleanup
******************************************************************************/
void EarlyStartupPlayer::deletePipeline()
{
   if (_gstreamerData.bus)
   {
      gst_object_unref(_gstreamerData.bus);
      _gstreamerData.bus = NULL;
   }
   if (_gstreamerData.pipeline)
   {
      gst_element_set_state(_gstreamerData.pipeline, GST_STATE_NULL);

      gst_object_unref(_gstreamerData.pipeline);
      _gstreamerData.pipeline = NULL;
   }
}


/**************************************************************************//**
*Function for checking the specific message on bus
 We look for EOS or Error messages
******************************************************************************/
bool EarlyStartupPlayer::checkBusCallback()
{
   GError* err = NULL;
   gchar* dbg = NULL;

   ETG_TRACE_USR4(("EarlyStartupPlayer::checkBusCallback - message = %s", GST_MESSAGE_TYPE_NAME(_gstreamerData.message)));

   switch (GST_MESSAGE_TYPE(_gstreamerData.message))
   {
      case GST_MESSAGE_EOS:
         return true;

      case GST_MESSAGE_ERROR:
         gst_message_parse_error(_gstreamerData.message, &err, &dbg);
         if (err)
         {
            ETG_TRACE_USR4(("EarlyStartupPlayer::checkBusCallback - GST_MESSAGE_ERROR - err = %s", err->message));
            g_error_free(err);
         }
         if (dbg)
         {
            ETG_TRACE_USR4(("EarlyStartupPlayer::checkBusCallback - GST_MESSAGE_ERROR - dbg = %s", dbg));
            g_free(dbg);
         }
         return true;

      default:
         break;
   }

   return false;
}


/**************************************************************************//**
*
******************************************************************************/
void EarlyStartupPlayer::startTimer()
{
   _timer.start(*this, 500);
}


/**************************************************************************//**
*
******************************************************************************/
void EarlyStartupPlayer::notifyPlaybackFinished()
{
   if (_earlyPromptStatusObserver)
   {
      _earlyPromptStatusObserver->onEarlyPromptFinished();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void EarlyStartupPlayer::notifyPlaybackTerminated()
{
   if (_earlyPromptStatusObserver)
   {
      _earlyPromptStatusObserver->onEarlyPromptTerminated();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void EarlyStartupPlayer::registerObserver(EarlyPromptStatusObserver* observer)
{
   _earlyPromptStatusObserver = observer;
}


/**************************************************************************//**
*
******************************************************************************/
T_e8_Language_Code EarlyStartupPlayer::getPromptLanguage()
{
   dp_tclfc_sds_adapter_SpeechSettingsSystemLanguage dpSytemLanguage;
   T_e8_Language_Code dpSystemLanguage  =  static_cast<T_e8_Language_Code>(dpSytemLanguage.tGetData());

   if (dpSystemLanguage == T_e8_Language_Code__Unknown)
   {
      // vehicle language not updated in Data pool. Read Default Language from KDS
      return static_cast<T_e8_Language_Code>(clSDS_KDSConfiguration::getDefaultLanguage());
   }
   return dpSystemLanguage;
}


/**************************************************************************//**
*
******************************************************************************/
std::string EarlyStartupPlayer::getEarlySDSPromptPath(T_e8_Language_Code systemLanguage)
{
   std::string pathPrefix = "/opt/bosch/sds/res/early_sds_prompt_";
   dp_tclfc_sds_adapter_SpeechSettingsVoicePreference dpvoicePrefernce;
   ActiveSpeakerGender enTypeVoicePreference =  static_cast<ActiveSpeakerGender>(dpvoicePrefernce.tGetData());

   // find prompt path with  preferred Gender and support language
   for (size_t i = 0; i < ARRAY_SIZE(g_EarlyPromptMapping); ++i)
   {
      if (g_EarlyPromptMapping[i].vehicleLanguage == systemLanguage &&
            g_EarlyPromptMapping[i].enTypeVoicePreference == enTypeVoicePreference)
      {
         return pathPrefix + g_EarlyPromptMapping[i].earlyPromptFile + ".mp3";
      }
   }

   //find prompt path with disregarding preferred Gender and support language
   for (size_t i = 0; i < ARRAY_SIZE(g_EarlyPromptMapping); ++i)
   {
      if (g_EarlyPromptMapping[i].vehicleLanguage == systemLanguage)
      {
         return pathPrefix + g_EarlyPromptMapping[i].earlyPromptFile + ".mp3";
      }
   }

   return "" ; // TTS support Language or Language not defined in EarlyPromptMapping
}
