/**************************************************************************//**
 * \file       CountryNameRequestor.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/CountryNameRequestor.h"
#include "application/CountryNameObserver.h"
#include "application/StateNameRequestor.h"
#include "application/StringUtils.h"
#include "application/clSDS_SDSStatus.h"


using namespace org::bosch::cm::navigation::NavigationService;

/**************************************************************************//**
 * Destructor
 ******************************************************************************/
CountryNameRequestor::~CountryNameRequestor()
{
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
CountryNameRequestor::CountryNameRequestor(
   ::boost::shared_ptr<org::bosch::cm::navigation::NavigationService::NavigationServiceProxy> naviProxy
   , StateNameRequestor* stateNameRequestor
   , clSDS_SDSStatus* pSDSStatus)
   : _navigationProxy(naviProxy)
   , _stateNameRequestor(stateNameRequestor)
   , _pSDSStatus(pSDSStatus)
{
   _availableCountryList.clear();
   _stateNameRequestor->addStateObserver(this);
   _requestedCountryIndex = 0;
   _pSDSStatus->vRegisterObserver(this);
}


/**************************************************************************//**
*
******************************************************************************/
void CountryNameRequestor::addCountryObserver(CountryNameObserver* pObserver)
{
   if (pObserver != NULL)
   {
      _countryObservers.push_back(pObserver);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void CountryNameRequestor::notifyCountryObservers()
{
   std::vector<CountryNameObserver*>::iterator it = _countryObservers.begin();

   while (it != _countryObservers.end())
   {
      if (*it != NULL)
      {
         (*it)->countriesUpdated();
      }
      ++it;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void CountryNameRequestor::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _navigationProxy)
   {
      _navigationProxy->sendSdsAddressWithOptionsDeregisterAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void CountryNameRequestor::onAvailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
                                       const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _navigationProxy)
   {
      _navigationProxy->sendSdsAddressWithOptionsRegister(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void CountryNameRequestor::sendCheckAddressForCountry()
{
   std::vector< SDSAddressElement > requestCountryDetails;

   _navigationProxy->sendSdsCheckAddressRequest(*this, requestCountryDetails);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void CountryNameRequestor::onSdsCheckAddressError(const ::boost::shared_ptr<NavigationServiceProxy>& /*proxy*/,
      const ::boost::shared_ptr< SdsCheckAddressError >& /*Error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void CountryNameRequestor::onSdsCheckAddressResponse(const ::boost::shared_ptr<NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsCheckAddressResponse >& /*osdsCheckAddressResponse*/)
{
   //Navi hall will send the response of SdsCheckAddress in form of property update.The below function will handle this.
   handleSdsCheckAddressWithCountry();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void CountryNameRequestor::onSdsAddressWithOptionsUpdate(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsAddressWithOptionsUpdate >& /*oSdsAddressWithOptionsUpdate*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void CountryNameRequestor::onSdsAddressWithOptionsError(const ::boost::shared_ptr<NavigationServiceProxy>& /*proxy*/,
      const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::SdsAddressWithOptionsError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void CountryNameRequestor::handleSdsCheckAddressWithCountry()
{
   std::vector<SDSAddressElementType> elementType;
   ::std::vector<SDSAddressElementType>::iterator it;

   if (_navigationProxy->hasSdsAddressWithOptions())
   {
      elementType = _navigationProxy->getSdsAddressWithOptions().getAddressElementType();

      for (it = elementType.begin(); it != elementType.end(); ++it)
      {
         if ((*it) == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY)
         {
            requestCountryNames();
            break;
         }
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void CountryNameRequestor::requestCountryNames()
{
   _navigationProxy->sendSdsGetRefinementListRequest(*this, SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void CountryNameRequestor::onSdsGetRefinementListError(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsGetRefinementListError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void CountryNameRequestor::onSdsGetRefinementListResponse(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsGetRefinementListResponse >& oSdsRefinementListresponse)
{
   ::std::vector< RefinementListElement > countryList;
   ::std::vector< RefinementListElement >:: iterator it;

   countryList = oSdsRefinementListresponse->getRefinements();

   for (it = countryList.begin(); it != countryList.end(); ++it)
   {
      _availableCountryList.push_back(it->getData());
   }

   getStateName();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void CountryNameRequestor::getStateName()
{
   _requestedCountryIndex = 0;

   if (NULL != _stateNameRequestor && (_availableCountryList.size()))
   {
      _stateNameRequestor->getAllStatesforCountry(_availableCountryList.at(_requestedCountryIndex));
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void CountryNameRequestor::statesUpdated()
{
   std::vector <std::string>::iterator it;

   _requestedCountryIndex++;

   if (_requestedCountryIndex < _availableCountryList.size())
   {
      _stateNameRequestor->getAllStatesforCountry(_availableCountryList.at(_requestedCountryIndex));
   }
   else
   {
      notifyCountryObservers();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tU32 CountryNameRequestor::getCountryListSize()
{
   return _availableCountryList.size();
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string CountryNameRequestor::getCountryName(std::string stateName)
{
   return _stateNameRequestor->getCountryForGivenStateName(stateName);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void CountryNameRequestor::vSDSStatusChanged()
{
   //When there is a language change SDS goes to LOADING state during which we have to
   //clear the language list and get the new one for new language.

   if (_pSDSStatus != NULL && _pSDSStatus->getStatus() == clSDS_SDSStatus::EN_LOADING)
   {
      _availableCountryList.clear();
      _requestedCountryIndex = 0;
      _stateNameRequestor->clearStateList();
   }
}
