/**************************************************************************//**
 * \file       clSDS_Property_PhoneStatus.cpp
 *
 * Common Action Request property implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Property_PhoneStatus.h"
#include "application/clSDS_PhoneStatusObserver.h"
#include "application/clSDS_ConfigurationFlags.h"
#include "application/clSDS_Userwords.h"
#include "application/StringUtils.h"
#include "SdsAdapter_Trace.h"

#define DEFAULT_DEVICE_HANDLE 65535

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Property_PhoneStatus.cpp.trc.h"
#endif


using namespace MOST_BTSet_FI;
using namespace most_BTSet_fi_types;
using namespace most_BTSet_fi_types_Extended;
using namespace MOST_Tel_FI;
using namespace most_Tel_fi_types;
using namespace MOST_PhonBk_FI;
using namespace most_PhonBk_fi_types;
using namespace most_PhonBk_fi_types_Extended;
using namespace smartphoneint_main_fi;
using namespace smartphoneint_main_fi_types;

//#define VPB_OFFSET 1000
#define VALID_DEVICE_HANDLE_MIN 2
#define VALID_DEVICE_HANDLE_MAX 11


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_Property_PhoneStatus::~clSDS_Property_PhoneStatus()
{
   for (size_t obsIndex = 0; obsIndex != _phoneStatusObserverList.size(); ++obsIndex)
   {
      _phoneStatusObserverList[obsIndex] = NULL;
   }
   _pUserwords = NULL;
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_Property_PhoneStatus::clSDS_Property_PhoneStatus(
   ahl_tclBaseOneThreadService* pService,
   ::boost::shared_ptr< ::MOST_BTSet_FI::MOST_BTSet_FIProxy > pSds2BtSetProxy,
   ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy > pSds2TelProxy,
   ::boost::shared_ptr <MOST_PhonBk_FIProxy> pPhoneBookProxy,
   ::boost::shared_ptr< Smartphoneint_main_fiProxy > smartphoneProxy,
   clSDS_Userwords* userWords)

   : clServerProperty(SDS2HMI_SDSFI_C_U16_PHONESTATUS, pService)
   , _sds2BtSetProxy(pSds2BtSetProxy)
   , _telephoneProxy(pSds2TelProxy)
   , _phoneBookProxy(pPhoneBookProxy)
   , _smartphoneProxy(smartphoneProxy)
   , _inCallDeviceHandle(1)
   , _hfpDeviceHandle(DEFAULT_DEVICE_HANDLE)
   , _activeHFPDeviceHandle(DEFAULT_DEVICE_HANDLE)
   , _btStatus(false)
   , _callPresent(false)
   , _handsetMode(false)
   , _userConnectedState(false)
   , _androidAuto(false)
   , _firstCallStatus(T_e8_TelCallStatus__e8IDLE)
   , _secondCallStatus(T_e8_TelCallStatus__e8IDLE)
   , _pUserwords(userWords)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_PhoneStatus::vSet(amt_tclServiceData* /*pInMsg*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_PhoneStatus::vGet(amt_tclServiceData* /*pInMsg*/)
{
   vSendPhoneStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_PhoneStatus::vUpreg(amt_tclServiceData* /*pInMsg*/)
{
   vSendPhoneStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::handlePhonebookDownloadStatus(const most_PhonBk_fi_types_Extended::T_PhonBkDownloadStateExtendedStreamItem oDownloadStatus)
{
   // TODO jnd2hi: is it ok to derive connection state from phonebook? what if phonebook is not supported/allowed by connected phone
   switch (oDownloadStatus.getE8PhoneBookDownloadState())
   {
      case T_e8_PhonBkPhoneBookDownloadStateExtended__e8PBDS_COMPLETE:
      case T_e8_PhonBkPhoneBookDownloadStateExtended__e8PBDS_LIMIT_REACHED:
      case T_e8_PhonBkPhoneBookDownloadStateExtended__e8PBDS_ERROR:
      {
         _connectedPhonedeviceStatus.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY;
         vSendPhoneStatus();
      }
      break;

      case T_e8_PhonBkPhoneBookDownloadStateExtended__e8PBDS_CONTACT_INFO:
      case T_e8_PhonBkPhoneBookDownloadStateExtended__e8PBDS_CONTACT_PHOTO:
      {
         _connectedPhonedeviceStatus.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_UPDATING;
         vSendPhoneStatus();
      }
      break;

      default:
         break;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
sds2hmi_fi_tcl_DeviceStatus clSDS_Property_PhoneStatus::getDeviceStatus(const DeviceDescriptor& device) const
{
   sds2hmi_fi_tcl_DeviceStatus deviceStatus;
   deviceStatus.DeviceID = (tU32)device.id;
   deviceStatus.DeviceName.bSet(device.name.c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   deviceStatus.UUID.bSet(device.UUID.c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   if (device.id == _activeHFPDeviceHandle)
   {
      // connected device (actual phone)
      deviceStatus.Status = _connectedPhonedeviceStatus;
   }
   /*else if (device.id == VPB_OFFSET + (uint32)_activeHFPDeviceHandle)
   {
      // device for vehicle phonebook
      deviceStatus.Status = _connectedPhonedeviceStatus;
   }*/
   else if (device.id  == _hfpDeviceHandle)
   {
      // slave Device
      deviceStatus.Status.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY_NOT_ACTIVE;
   }
   else
   {
      deviceStatus.Status.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_DISABLED;
   }
   return deviceStatus;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_PhoneStatus::vSendPhoneStatus()
{
   if ((_sds2BtSetProxy->isAvailable()) && (_sds2BtSetProxy->hasDeviceListExtended()))
   {
      sds2hmi_sdsfi_tclMsgPhoneStatusStatus oMessage;
      for (tU32 u32ListIndex = 0; u32ListIndex < _deviceList.size(); u32ListIndex++)
      {
         oMessage.DeviceList.push_back(getDeviceStatus(_deviceList[u32ListIndex]));
      }
      oMessage.MenuType.enType = sds2hmi_fi_tcl_e8_PHN_MenuType::FI_EN_OTHER;
      oMessage.Status = _phoneStatus;
      oMessage.BTenabled = _btStatus;
      oMessage.TransorderType = _preferredSortingOrder;
      oMessage.CarPlayAvailable = FALSE;
      vStatus(oMessage);
      vTracePhoneStatus(oMessage);
      clSDS_ConfigurationFlags::setDynamicVariable("PhoneReadyToCall", isPhoneReadyToCall(_phoneStatus));
   }
}


bool clSDS_Property_PhoneStatus::isPhoneReadyToCall(const sds2hmi_fi_tcl_e8_PHN_Status& phoneStatus)
{
   return (phoneStatus.enType == sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_IDLE);
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_PhoneStatus::vTracePhoneStatus(const sds2hmi_sdsfi_tclMsgPhoneStatusStatus& /*oMessage*/) const
{
   // TODO jnd2hi rework trace
   //   ET_TRACE_BIN(
   //      TR_CLASS_SDSADP_DETAILS,
   //      TR_LEVEL_HMI_INFO,
   //      ET_EN_T16 _ TRACE_CCA_VALUE _
   //      ET_EN_T16 _ _pService->u16GetServiceId() _
   //      ET_EN_T16 _ u16GetFunctionID() _
   //      ET_EN_T16 _ 0 _
   //      ET_EN_T16 _ oMessage.BTenabled _
   //      ET_EN_T16 _ oMessage.TransorderType.enType _
   //      ET_EN_DONE);
   //
   //   for (tU32 u32Index = 0; u32Index < dataPool.u8GetValue(DPTELEPHONE__PAIRED_DEVICES_COUNT); u32Index++)
   //   {
   //      ET_TRACE_BIN(
   //         TR_CLASS_SDSADP_DETAILS,
   //         TR_LEVEL_HMI_INFO,
   //         ET_EN_T16 _ TRACE_CCA_VALUE _
   //         ET_EN_T16 _ _pService->u16GetServiceId() _
   //         ET_EN_T16 _ u16GetFunctionID() _
   //         ET_EN_T16 _ 1 _
   //         ET_EN_T16 _ oMessage.DeviceList[u32Index].DeviceID _
   //         ET_EN_T16 _ oMessage.DeviceList[u32Index].Status.enType _
   //         ET_EN_STRING _ oMessage.DeviceList[u32Index].DeviceName.szValue _
   //         ET_EN_DONE);
   //   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _sds2BtSetProxy)
   {
      _sds2BtSetProxy->sendBluetoothOnOffUpReg(*this);
      _sds2BtSetProxy->sendDeviceListExtendedUpReg(*this);
   }
   if (proxy == _telephoneProxy)
   {
      _telephoneProxy->sendCallStatusNoticeUpReg(*this);
      _telephoneProxy->sendActivePhoneDeviceUpReg(*this);
   }
   if (proxy == _phoneBookProxy)
   {
      _phoneBookProxy->sendDownloadStateExtendedUpReg(*this);
      _phoneBookProxy->sendPreferredPhoneBookSortOrderUpReg(*this);
      _phoneBookProxy->sendListChangeUpReg(*this);
   }
   if (proxy == _smartphoneProxy)
   {
      _smartphoneProxy->sendSessionStatusInfoUpReg(*this);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _sds2BtSetProxy)
   {
      _sds2BtSetProxy->sendBluetoothOnOffRelUpRegAll();
      _sds2BtSetProxy->sendDeviceListExtendedRelUpRegAll();
   }
   if (proxy == _telephoneProxy)
   {
      _telephoneProxy->sendCallStatusNoticeRelUpRegAll();
      _telephoneProxy->sendActivePhoneDeviceRelUpRegAll();
   }
   if (proxy == _phoneBookProxy)
   {
      _phoneBookProxy->sendDownloadStateExtendedRelUpRegAll();
      _phoneBookProxy->sendPreferredPhoneBookSortOrderRelUpRegAll();
      _phoneBookProxy->sendListChangeRelUpRegAll();
   }
   if (proxy == _smartphoneProxy)
   {
      _smartphoneProxy->sendSessionStatusInfoRelUpRegAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::onBluetoothOnOffError(
   const ::boost::shared_ptr< MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< BluetoothOnOffError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::onBluetoothOnOffStatus(
   const ::boost::shared_ptr< MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< BluetoothOnOffStatus >& status)
{
   _btStatus = status->getBBTOnOff();
   vSendPhoneStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::onCallStatusNoticeError(
   const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< CallStatusNoticeError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::onCallStatusNoticeStatus(
   const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< CallStatusNoticeStatus >& status)
{
   bool handsetMode = _handsetMode;
   bool callPresent = _callPresent;
   T_e8_TelCallStatus  firstCallStatus = _firstCallStatus;
   T_e8_TelCallStatus  secondCallStatus = _secondCallStatus;

   _handsetMode = false;
   _callPresent = false;
   //TODO raa8hi:  initialize _inCallDeviceHandle?
   for (uint8 i = 0 ; i < status->getOCallStatusNoticeStream().size() ; i++)
   {
      if (status->getOCallStatusNoticeStream()[i].getE8CallStatus() != T_e8_TelCallStatus__e8IDLE)
      {
         _callPresent = true;
         if (!(status->getOCallStatusNoticeStream()[i].getBUsingVehicleAudio()))
         {
            _handsetMode = true;
         }
      }

      if (status->getOCallStatusNoticeStream()[i].getE8CallStatus() == T_e8_TelCallStatus__e8ACTIVE)
      {
         _inCallDeviceHandle = status->getOCallStatusNoticeStream()[i].getU8DeviceHandle();
      }
   }
   if (status->getOCallStatusNoticeStream().size() >= 2)
   {
      _firstCallStatus = status->getOCallStatusNoticeStream()[0].getE8CallStatus();
      _secondCallStatus = status->getOCallStatusNoticeStream()[1].getE8CallStatus();
   }

   // notify observers on onCallStatusNoticeStatus
   for (size_t obsIndex = 0; obsIndex != _phoneStatusObserverList.size(); ++obsIndex)
   {
      if (_phoneStatusObserverList[obsIndex] != NULL)
      {
         _phoneStatusObserverList[obsIndex]->callStatusChanged(_firstCallStatus, _secondCallStatus);
      }
   }

   if ((handsetMode != _handsetMode) || (callPresent != _callPresent) || (firstCallStatus != _firstCallStatus) || (secondCallStatus != _secondCallStatus))
   {
      handlePhoneStatus();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::handlePhoneStatus()
{
   ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus"));
   if (_androidAuto)
   {
      ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus _androidAuto"));
      _phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_NOT_AVAILABLE;
   }
   else if (_handsetMode)
   {
      ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus FI_EN_HANDSET_MODE"));
      //_phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_HANDSET_MODE;
      // issue 577713 sending IDLE mode to MW for second phone call working.
      _phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_IDLE;
   }
   else if (_userConnectedState)
   {
      if (_callPresent == false)
      {
         ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus FI_EN_IDLE 1"));
         _phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_IDLE;
      }
      else if (_firstCallStatus == T_e8_TelCallStatus__e8BUSY || _secondCallStatus == T_e8_TelCallStatus__e8BUSY)
      {
         ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus FI_EN_BUSY 1"));
         _phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_BUSY;
      }
      else if ((_firstCallStatus  == T_e8_TelCallStatus__e8ACTIVE && _secondCallStatus == T_e8_TelCallStatus__e8IDLE) ||
               (_firstCallStatus  == T_e8_TelCallStatus__e8IDLE && _secondCallStatus == T_e8_TelCallStatus__e8ACTIVE))
      {
         ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus FI_EN_INCALL 1"));
         _phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_INCALL;
      }
      else if (_firstCallStatus == T_e8_TelCallStatus__e8DIALING || _secondCallStatus == T_e8_TelCallStatus__e8DIALING)
      {
         ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus FI_EN_DIALLING 1"));
         _phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_DIALLING;
      }
      else if (_firstCallStatus == T_e8_TelCallStatus__e8RINGTONE || _secondCallStatus == T_e8_TelCallStatus__e8RINGTONE)
      {
         ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus FI_EN_RINGING 1"));
         _phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_RINGING;
      }
      else if ((_firstCallStatus == T_e8_TelCallStatus__e8ON_HOLD && _secondCallStatus == T_e8_TelCallStatus__e8ACTIVE) ||
               (_firstCallStatus == T_e8_TelCallStatus__e8ACTIVE && _secondCallStatus == T_e8_TelCallStatus__e8ON_HOLD))
      {
         ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus FI_EN_INCALL_HOLD 1"));
         _phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_INCALL_HOLD;
      }
      else if (_firstCallStatus == T_e8_TelCallStatus__e8DISCONNECTING || _secondCallStatus == T_e8_TelCallStatus__e8DISCONNECTING)
      {
         ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus FI_EN_INCALL 2"));
         _phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_INCALL;
      }
      else
      {
         ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus FI_EN_UNKNOWN"));
         _phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_UNKNOWN;
      }
   }
   else if (_deviceList.size())
   {
      ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus FI_EN_PHONE_NOT_READY"));
      _phoneStatus.enType = sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_PHONE_NOT_READY;
   }
   else
   {
      ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: handlePhoneStatus FI_EN_NOT_AVAILABLE"));
      _phoneStatus.enType =  sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_NOT_AVAILABLE;
   }

   vSendPhoneStatus();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::onDownloadStateExtendedError(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::DownloadStateExtendedError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::onDownloadStateExtendedStatus(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::DownloadStateExtendedStatus >& status)
{
   if (status->getODownloadStateExtendedStream().size())
   {
      const T_PhonBkDownloadStateExtendedStreamItem oDownloadStatus = status->getODownloadStateExtendedStream()[0];
      handlePhonebookDownloadStatus(oDownloadStatus);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::onPreferredPhoneBookSortOrderError(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< PreferredPhoneBookSortOrderError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::onPreferredPhoneBookSortOrderStatus(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< PreferredPhoneBookSortOrderStatus >& status)
{
   if (status->getE8PreferredPhoneBookSortOrder() ==  T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_FIRSTNAME)
   {
      _preferredSortingOrder.enType = sds2hmi_fi_tcl_e8_PHN_Transorder::FI_EN_FIRSTNAME_LASTNAME;
   }
   else if (status->getE8PreferredPhoneBookSortOrder() ==  T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_LASTNAME)
   {
      _preferredSortingOrder.enType = sds2hmi_fi_tcl_e8_PHN_Transorder::FI_EN_LASTNAME_FIRSTNAME;
   }
   else
   {
      _preferredSortingOrder.enType = sds2hmi_fi_tcl_e8_PHN_Transorder::FI_EN_UNKNOWN;
   }
   vSendPhoneStatus();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::onCreateContactListError(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< CreateContactListError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::onCreateContactListResult(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< CreateContactListResult >& result)
{
   const uint16 listHandle = result->getU16ListHandle();

   _listHandlesMap[listHandle] = (uint16)_activeHFPDeviceHandle;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::onListChangeError(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< ListChangeError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::onListChangeStatus(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< ListChangeStatus >& status)
{
   const uint16 listHandle = status->getU16ListHandle();
   const T_e8_PhonBkListChangeType listChangeType = status->getE8ListChangeType();

   if (_listHandlesMap.find(listHandle) != _listHandlesMap.end()
         && T_e8_PhonBkListChangeType__e8LCH_CONTENT_CHANGED != listChangeType)
   {
      ETG_TRACE_USR4(("clSDS_Property_PhoneStatus: onListChangeStatus - listHandle = %d", listHandle));
      _listHandlesMap[listHandle] = status->getU16ListLength();

      _connectedPhonedeviceStatus.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_UPDATING;
      vSendPhoneStatus();

      _connectedPhonedeviceStatus.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY;
      vSendPhoneStatus();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::onSessionStatusInfoError(
   const ::boost::shared_ptr< Smartphoneint_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< SessionStatusInfoError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::onSessionStatusInfoStatus(
   const ::boost::shared_ptr< Smartphoneint_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< SessionStatusInfoStatus >& status)
{
   bool androidAuto = _androidAuto;
   if (status->getDeviceCategory() == T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO
         && status->getSessionStatus() == T_e8_SessionStatus__SESSION_ACTIVE)
   {
      _androidAuto = true;
   }
   else
   {
      _androidAuto = false;
   }
   if (_androidAuto != androidAuto)
   {
      handlePhoneStatus();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
uint8 clSDS_Property_PhoneStatus::getDeviceHandle() const
{
   return _inCallDeviceHandle;
}


/***********************************************************************//**
 *
 ***************************************************************************/
sds2hmi_fi_tcl_e8_PHN_Status::tenType clSDS_Property_PhoneStatus::getPhoneStatus() const
{
   return _phoneStatus.enType;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::setPhoneStatusObserver(clSDS_PhoneStatusObserver* obs)
{
   if (obs != NULL)
   {
      ETG_TRACE_USR1(("clSDS_Property_PhoneStatus::setPhoneStatusObserver:NULL"));
      _phoneStatusObserverList.push_back(obs);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::updateContactLists()
{
   std::map<uint16, uint16>::const_iterator itr;
   bool isNewDevice = true;
   for (itr = _listHandlesMap.begin(); itr != _listHandlesMap.end(); ++itr)
   {
      if (itr->second == _activeHFPDeviceHandle)
      {
         isNewDevice = false;
      }
   }

   if (isNewDevice)
   {
      _phoneBookProxy->sendCreateContactListStart(*this, _activeHFPDeviceHandle, T_e8_PhonBkContactListType__PHONEBOOK, T_e8_PhonBkContactSortType__e8PB_LIST_SORT_POSITION);
      _phoneBookProxy->sendCreateContactListStart(*this, _activeHFPDeviceHandle, T_e8_PhonBkContactListType__VEHICLE, T_e8_PhonBkContactSortType__e8PB_LIST_SORT_POSITION);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::onDeviceListExtendedError(
   const ::boost::shared_ptr< MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DeviceListExtendedError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_PhoneStatus::handleDeviceStatusUpdateForUserWords(uint8 deviceHandle, T_e8_BTSetDeviceStatus deviceStatus, uint8 numOfPairedDevices)
{
   for (size_t itr = 0; itr != _deviceList.size(); itr++)
   {
      if (numOfPairedDevices == 0 && _pUserwords)
      {
         _pUserwords->vDeleteAllUserwords(_deviceList[0].id, sds2hmi_fi_tcl_e8_Domain::FI_EN_PHONE);
         _listHandlesMap.clear();
      }
      if (_deviceList[itr].id == deviceHandle && deviceStatus == T_e8_BTSetDeviceStatus__e8DEVICE_DELETED)
      {
         if (_pUserwords != NULL)
         {
            _pUserwords->vDeleteAllUserwords(deviceHandle, sds2hmi_fi_tcl_e8_Domain::FI_EN_PHONE);
         }
         std::map<uint16, uint16>::const_iterator itr;
         for (itr = _listHandlesMap.begin(); itr != _listHandlesMap.end(); ++itr)
         {
            if (itr->second == deviceHandle)
            {
               _listHandlesMap.erase(itr->first);
            }
         }
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::updateObservers(T_e8_BTConnectionStatus status)
{
   if (status == T_e8_BTConnectionStatus__e8STATUS_CONNECTED ||
         status == T_e8_BTConnectionStatus__e8STATUS_DISCONNECTED)
   {
      for (size_t obsIndex = 0; obsIndex != _phoneStatusObserverList.size(); ++obsIndex)
      {
         if (_phoneStatusObserverList[obsIndex] != NULL)
         {
            _phoneStatusObserverList[obsIndex]->phoneStatusChanged(_activeHFPDeviceHandle, status);
         }
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::onDeviceListExtendedStatus(
   const ::boost::shared_ptr< MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DeviceListExtendedStatus >& status)
{
   handleDeviceStatusUpdateForUserWords(status->getODeviceListChange().getU8DeviceHandle(),
                                        status->getODeviceListChange().getE8DeviceStatus(), status->getU8NumPairedDevices());

   uint8 deviceListSize = _deviceList.size();
   uint8 connectedDeviceListSize = _connectedDevice.size();
   if (_telephoneProxy->isAvailable() && _telephoneProxy->hasActivePhoneDevice() &&
         (_activeHFPDeviceHandle == DEFAULT_DEVICE_HANDLE))
   {
      _activeHFPDeviceHandle = _telephoneProxy->getActivePhoneDevice().getU8DeviceHandle();
      updateContactLists();
      updateUSWprofileChanged();
   }

   _deviceList.clear();
   _connectedDevice.clear();
   _userConnectedState = false;
   const T_BTSetDeviceListExtendedResult& odeviceListResult = status->getODeviceListExtendedResult();
   for (size_t i = 0; i != odeviceListResult.size(); ++i)
   {
      const T_BTSetDeviceListExtendedResultItem& resultDevice = odeviceListResult[i];
      DeviceDescriptor device;
      device.id = resultDevice.getU8DeviceHandle();
      device.name = resultDevice.getSDeviceName();
      device.UUID = StringUtils::toString(resultDevice.getU8DeviceHandle());
      if ((resultDevice.getODeviceProfileConnectionStatus().getE8HFP() == T_e8_BTConnectionStatus__e8STATUS_CONNECTED))
      {
         _userConnectedState = true;
         _connectedDevice.push_back(device);
      }
      _deviceList.push_back(device);

      //DeviceDescriptor vehiclePhonebookDevice;
      //vehiclePhonebookDevice.id = device.id + VPB_OFFSET;
      //vehiclePhonebookDevice.name =  "PhoneBook_VPB" + StringUtils::toString(device.id);
      //vehiclePhonebookDevice.UUID = StringUtils::toString(vehiclePhonebookDevice.id);
      //_deviceList.push_back(vehiclePhonebookDevice);

      if (_activeHFPDeviceHandle == resultDevice.getU8DeviceHandle())
      {
         //send update handle call lists and message when HFP device is connected or disconnected
         updateObservers(resultDevice.getODeviceProfileConnectionStatus().getE8HFP());
      }
      if (_hfpDeviceHandle == resultDevice.getU8DeviceHandle() &&
            resultDevice.getODeviceProfileConnectionStatus().getE8HFP() == T_e8_BTConnectionStatus__e8STATUS_DISCONNECTED)
      {
         //clear _activeHFPdeviceHandle when previously connected HFP device is disconnected
         _hfpDeviceHandle = DEFAULT_DEVICE_HANDLE;
      }
   }
   updateMultiPointPairingStatus();
   setHFPDeviceHandle();
   if (!_userConnectedState)
   {
      _connectedDevice.clear();
      _inCallDeviceHandle = 0;
   }
   if (status->getU8NumPairedDevices() == 0)
   {
      _activeHFPDeviceHandle = _hfpDeviceHandle = DEFAULT_DEVICE_HANDLE;
   }
   if ((deviceListSize != _deviceList.size()) || (connectedDeviceListSize != _connectedDevice.size()))
   {
      handlePhoneStatus();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::updateUSWprofileChanged()
{
   for (size_t obsIndex = 0; obsIndex != _phoneStatusObserverList.size(); ++obsIndex)
   {
      if (_phoneStatusObserverList[obsIndex] != NULL)
      {
         _phoneStatusObserverList[obsIndex]->phoneProfileChanged(_activeHFPDeviceHandle);
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::updateMultiPointPairingStatus()
{
   bool multiPointPairing = false;
   if (_hfpDeviceHandle != DEFAULT_DEVICE_HANDLE)
   {
      multiPointPairing = true;
   }

   for (size_t obsIndex = 0; obsIndex != _phoneStatusObserverList.size(); ++obsIndex)
   {
      if (_phoneStatusObserverList[obsIndex] != NULL)
      {
         _phoneStatusObserverList[obsIndex]->multiPointPairingEnabled(multiPointPairing);
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::setHFPDeviceHandle()
{
   for (unsigned int index = 0; index != _connectedDevice.size(); ++index)
   {
      if (_connectedDevice[index].id != _activeHFPDeviceHandle)
      {
         _hfpDeviceHandle = _connectedDevice[index].id;
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::onActivePhoneDeviceError(
   const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< ActivePhoneDeviceError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_PhoneStatus::onActivePhoneDeviceStatus(
   const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< ActivePhoneDeviceStatus >& status)
{
   _activeHFPDeviceHandle = status->getU8DeviceHandle();
   updateContactLists();
   updateUSWprofileChanged();
   setHFPDeviceHandle();
   updateMultiPointPairingStatus();
   updateObservers(T_e8_BTConnectionStatus__e8STATUS_CONNECTED);
   vSendPhoneStatus();
}
