/**************************************************************************//**
 * \file       clSDS_Property_NavDataDataSetInfo.cpp
 *
 * NavDataDataSetInfo property implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Property_NavDataDataSetInfo.h"
#include "application/NavigationAvailabilityObserver.h"
#include "application/clSDS_ConfigurationFlags.h"
#include "SdsAdapter_Trace.h"


using namespace org::bosch::cm::navigation::NavigationSDSService;


/**************************************************************************//**
* Constructor
******************************************************************************/
clSDS_Property_NavDataDataSetInfo::clSDS_Property_NavDataDataSetInfo(
   ahl_tclBaseOneThreadService* pService,
   ::boost::shared_ptr< NavigationSDSServiceProxy> naviSDSProxy)
   : clServerProperty(SDS2HMI_SDSFI_C_U16_NAVDATADATASETINFO, pService)
   , _naviSDSProxy(naviSDSProxy)
   , _dataSetId(0)
{
   mapNaviUnittoSDSUnit();
}


/**************************************************************************//**
* Destructor
******************************************************************************/
clSDS_Property_NavDataDataSetInfo::~clSDS_Property_NavDataDataSetInfo()
{
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Property_NavDataDataSetInfo::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviSDSProxy)
   {
      _naviSDSProxy->sendNavDatasetInfoRegister(*this);
      _naviSDSProxy->sendNavDatasetInfoGet(*this);
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Property_NavDataDataSetInfo::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviSDSProxy)
   {
      _naviSDSProxy->sendNavDatasetInfoDeregisterAll();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Property_NavDataDataSetInfo::onNavDatasetInfoError(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< NavDatasetInfoError >& /*error*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Property_NavDataDataSetInfo::onNavDatasetInfoUpdate(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< NavDatasetInfoUpdate >& update)
{
   if (update->hasNavDatasetInfo())
   {
      sendNavDataSetInfoStatus(update->getNavDatasetInfo());
      setAvailability((update->getNavDatasetInfo()).getDatasetId());
   }
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Property_NavDataDataSetInfo::vSet(amt_tclServiceData* pInMsg)
{
   // read-only property - ignore set
   sendResponse();
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Property_NavDataDataSetInfo::vGet(amt_tclServiceData* /*pInMsg*/)
{
   sendResponse();
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Property_NavDataDataSetInfo::vUpreg(amt_tclServiceData* /*pInMsg*/)
{
   sendResponse();
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Property_NavDataDataSetInfo::sendResponse()
{
   if (_naviSDSProxy->hasNavDatasetInfo())
   {
      sendNavDataSetInfoStatus(_naviSDSProxy->getNavDatasetInfo());
   }
   else
   {
      sendNavDataSetInfoStatus(NavDataSetInfo());
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Property_NavDataDataSetInfo::sendNavDataSetInfoStatus(const NavDataSetInfo& dataSetInfo)
{
   sds2hmi_sdsfi_tclMsgNavDataDataSetInfoStatus oMessage;

   oMessage.DataSetId = dataSetInfo.getDatasetId();
   oMessage.PersistentDataSetId.bSet(dataSetInfo.getPersistentDatasetId().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   oMessage.SharedDataAccessAttributes.SharedDataAccess.enType = _sharedDataAccessMap[dataSetInfo.getSharedDataAccessAttributes().getSharedDataAccess()];
   oMessage.SharedDataAccessAttributes.JournalingMode.enType = _journalingModeMap[dataSetInfo.getSharedDataAccessAttributes().getJournalingMode()];
   oMessage.SharedDataAccessAttributes.LockType.enType = _lockTypeMap[dataSetInfo.getSharedDataAccessAttributes().getLockType()];
   vStatus(oMessage);
}


void clSDS_Property_NavDataDataSetInfo::mapNaviUnittoSDSUnit()
{
   _sharedDataAccessMap[SharedDataAccess__SHARED_DATA_ACCESS__NOT_REQUIRED] = sds2hmi_fi_tcl_e8_SharedDataAccess::FI_EN_NOT_REQUIRED;
   _sharedDataAccessMap[SharedDataAccess__SHARED_DATA_ACCESS__REQUIRED] = sds2hmi_fi_tcl_e8_SharedDataAccess::FI_EN_REQUIRED;
   _sharedDataAccessMap[SharedDataAccess__SHARED_DATA_ACCESS__ENABLE] = sds2hmi_fi_tcl_e8_SharedDataAccess::FI_EN_ENABLE;
   _sharedDataAccessMap[SharedDataAccess__SHARED_DATA_ACCESS__UNKNOWN] = sds2hmi_fi_tcl_e8_SharedDataAccess::FI_EN_UNKNOWN;

   _journalingModeMap[JournalingMode__JOURNALING_MODE__NONE] = sds2hmi_fi_tcl_e8_JournalingMode::FI_EN_NONE;
   _journalingModeMap[JournalingMode__JOURNALING_MODE__WAL] = sds2hmi_fi_tcl_e8_JournalingMode::FI_EN_WAL;
   _journalingModeMap[JournalingMode__JOURNALING_MODE__UNKNOWN] = sds2hmi_fi_tcl_e8_JournalingMode::FI_EN_UNKNOWN;

   _lockTypeMap[LockType__LOCK_TYPE__NONE] = sds2hmi_fi_tcl_e8_LockType::FI_EN_NONE;
   // _lockTypeMap[LockType__LOCK_TYPE__EXCLUSIVE] = sds2hmi_fi_tcl_e8_LockType::FI_EN_EXCLUSIVE;
   _lockTypeMap[LockType__LOCK_TYPE__NORMAL] = sds2hmi_fi_tcl_e8_LockType::FI_EN_NORMAL;
   _lockTypeMap[LockType__LOCK_TYPE__UNKNOWN] = sds2hmi_fi_tcl_e8_LockType::FI_EN_UNKNOWN;
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Property_NavDataDataSetInfo::setAvailability(uint32 dataSetId)
{
   if (_dataSetId != dataSetId)
   {
      _dataSetId = dataSetId;
      notifyNavDataObservers();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Property_NavDataDataSetInfo::addNavDataObserver(NavigationAvailabilityObserver* pObserver)
{
   if (pObserver != NULL)
   {
      _NavDataObservers.push_back(pObserver);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Property_NavDataDataSetInfo::notifyNavDataObservers()
{
   bool mapUpdateStatus = _dataSetId ? true : false;
   clSDS_ConfigurationFlags::setDynamicVariable("MapUpdateStatus", mapUpdateStatus);
   for (size_t i = 0; i < _NavDataObservers.size(); ++i)
   {
      _NavDataObservers.at(i)->onNaviAvailabilityStatus();
   }
}
