/**************************************************************************//**
 * \file       clSDS_Property_MediaStatus.cpp
 *
 * Common Action Request property implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Property_MediaStatus.h"
#include "application/clSDS_ConfigurationFlags.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Property_MediaStatus.cpp.trc.h"
#endif


using namespace MPlay_fi_types;
using namespace mplay_MediaPlayer_FI;
using namespace smartphoneint_main_fi;
using namespace smartphoneint_main_fi_types;


/************************************************************************************//**
 * To find the item in list of devices info with the givendeviceTag
 ***************************************************************************************/
static size_t findItemWithDeviceTag(const T_MPlayDeviceInfo& deviceList, uint8 deviceTag)
{
   size_t i = 0;
   for (; i < deviceList.size(); ++i)
   {
      if (deviceList[i].getU8DeviceTag() == deviceTag)
      {
         break;
      }
   }
   return i;
}


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_Property_MediaStatus::~clSDS_Property_MediaStatus()
{
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_Property_MediaStatus::clSDS_Property_MediaStatus(ahl_tclBaseOneThreadService* pService,
      ::boost::shared_ptr< Mplay_MediaPlayer_FIProxy > pSds2MediaProxy,
      ::boost::shared_ptr< Smartphoneint_main_fiProxy > smartphoneProxy)
   : clServerProperty(SDS2HMI_SDSFI_C_U16_MEDIASTATUS, pService)
   , _sds2MediaProxy(pSds2MediaProxy)
   , _smartphoneProxy(smartphoneProxy)
{
   _carPlayAvailable = false;
   _androidAutoAvailable = false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vSet(amt_tclServiceData* /*pInMsg*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vGet(amt_tclServiceData* /*pInMsg*/)
{
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vUpreg(amt_tclServiceData* /*pInMsg*/)
{
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vSendStatus()
{
   sds2hmi_sdsfi_tclMsgMediaStatusStatus oMessage;
   vUpdateDeviceList(oMessage.DeviceList);
   if (!oMessage.DeviceList.empty())
   {
      vStatus(oMessage);
      vTraceMediaStatus(oMessage);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
sds2hmi_fi_tcl_e8_MPL_SourceType::tenType clSDS_Property_MediaStatus::eGetMediaPlayerSource(size_t idx) const
{
   switch (_oDeviceList[idx].getE8DeviceType())
   {
      case T_e8_MPlayDeviceType__e8DTY_IPOD:
      case T_e8_MPlayDeviceType__e8DTY_IPHONE:
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_IPOD;

      case T_e8_MPlayDeviceType__e8DTY_USB:
      case T_e8_MPlayDeviceType__e8DTY_MTP:
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_USB;

      default:
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_UNKNOWN;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
sds2hmi_fi_tcl_e8_DeviceStatus::tenType clSDS_Property_MediaStatus::eGetDeviceStatus(size_t idx) const
{
   switch (_oDeviceList[idx].getE8DeviceIndexedState())
   {
      case T_e8_MPlayDeviceIndexedState__e8IDS_COMPLETE:
         return sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY;

      case T_e8_MPlayDeviceIndexedState__e8IDS_PARTIAL:
         return sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_UPDATING;

      case T_e8_MPlayDeviceIndexedState__e8IDS_NOT_STARTED:
         return sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_VERIFYING;

      case T_e8_MPlayDeviceIndexedState__e8IDS_NOT_SUPPORTED:
      default:
         return sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_DISABLED;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Property_MediaStatus::bUsbSourceAvailable(size_t idx) const
{
   tU32 noOfAudioFiles = _oDeviceList[idx].getU32NumberOfFiles() - (_oDeviceList[idx].getU32NumberOfImageFiles() + _oDeviceList[idx].getU32NumberOfVideoFiles());
   std::string syspath = _oDeviceList[idx].getSDeviceSysPath();
   if (((_oDeviceList[idx].getE8DeviceType() == T_e8_MPlayDeviceType__e8DTY_USB) ||
         (_oDeviceList[idx].getE8DeviceType() == T_e8_MPlayDeviceType__e8DTY_MTP)) &&
         (noOfAudioFiles > 0) &&
         (_oDeviceList[idx].getBDeviceConnected() == TRUE))
   {
      return TRUE;
   }
   return FALSE;
}


tBool clSDS_Property_MediaStatus::isDeviceConnectedInEvenPortNumber(std::string& sysPath) const
{
   ETG_TRACE_USR4(("isDeviceConnectedInEvenPortNumber Mount Point is %s", sysPath.c_str()));
   tBool status = TRUE;
   ::std::size_t found = ::std::string::npos;
   if (!sysPath.empty())
   {
      found = sysPath.find_last_of(".");
   }
   if (found != ::std::string::npos)
   {
      ETG_TRACE_USR4(("isDeviceConnectedInEvenPortNumber PortNumber Substring is %s", sysPath.substr(found + 1).c_str()));
      int portNumber = atoi(sysPath.substr(found + 1).c_str());
      ETG_TRACE_USR4(("isDeviceConnectedInEvenPortNumber PortNumber is %d", portNumber));
      if (portNumber <= 0) //If device connected Internally(Direct Connection), No External Hub Connection
      {
         status = TRUE;
      }
      else
      {
         if (portNumber % 2)
         {
            status = FALSE; //For Reno only- Device Connected in EVEN Port Number (2nd USB Port number is returned as 3)
         }
         else
         {
            status = TRUE; //For Reno only- Device Connected in ODD Port Number (1st USB Port number is returned as 2)
         }
      }
   }
   return status;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Property_MediaStatus::bIpodSourceAvailable(size_t idx) const
{
   if (((_oDeviceList[idx].getE8DeviceType() == T_e8_MPlayDeviceType__e8DTY_IPOD) ||
         (_oDeviceList[idx].getE8DeviceType() == T_e8_MPlayDeviceType__e8DTY_IPHONE)) && _oDeviceList[idx].getBDeviceConnected() == TRUE)
   {
      return TRUE;
   }
   return FALSE;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Property_MediaStatus::bMediaSourceAvailable(size_t idx) const
{
   switch (eGetMediaPlayerSource(idx))
   {
      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_IPOD:
         return bIpodSourceAvailable(idx);

      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_USB:
         return bUsbSourceAvailable(idx);

      default:
         return FALSE;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Property_MediaStatus::bBTSourceAvailable(size_t idx) const
{
   if ((_oDeviceList[idx].getE8DeviceType() == T_e8_MPlayDeviceType__e8DTY_BLUETOOTH) && _oDeviceList[idx].getBDeviceConnected() == true)
   {
      return true;
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Property_MediaStatus::bIsDeviceSupportCDtracks(size_t idx) const
{
   if ((_oDeviceList[idx].getE8DeviceType() == T_e8_MPlayDeviceType__e8DTY_CDDA)
         || ((_oDeviceList[idx].getE8DeviceType() == T_e8_MPlayDeviceType__e8DTY_DVD_DRIVE)
             && (_oDeviceList[idx].getE8DiscType() == T_e8_MDiscType__e8AUDIO_CD_DISC)))
   {
      return true;
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Property_MediaStatus::bIsCDDeviceSupportSourceChangeOnly(size_t idx) const
{
   if ((_oDeviceList[idx].getE8DeviceType() == T_e8_MPlayDeviceType__e8DTY_CDROM))
   {
      return true;
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Property_MediaStatus::diff(const MPlay_fi_types::T_MPlayDeviceInfo& currentDeviceInfoList, const MPlay_fi_types::T_MPlayDeviceInfo& newDeviceInfoList)
{
   tBool hasChange = (currentDeviceInfoList.size() != newDeviceInfoList.size());
   if (!hasChange)
   {
      for (size_t i = 0; i < newDeviceInfoList.size(); ++i)
      {
         size_t foundIdx = findItemWithDeviceTag(currentDeviceInfoList, newDeviceInfoList[i].getU8DeviceTag());
         if (foundIdx != currentDeviceInfoList.size()) //found in old list
         {
            if (currentDeviceInfoList[foundIdx].getU8DeviceTag()         != newDeviceInfoList[i].getU8DeviceTag()
                  || currentDeviceInfoList[foundIdx].getBDeviceConnected()    != newDeviceInfoList[i].getBDeviceConnected()
                  || currentDeviceInfoList[foundIdx].getE8DeviceIndexedState() != newDeviceInfoList[i].getE8DeviceIndexedState()
                  || currentDeviceInfoList[foundIdx].getSDeviceName()         != newDeviceInfoList[i].getSDeviceName()
                  || currentDeviceInfoList[foundIdx].getE8DeviceType()        != newDeviceInfoList[i].getE8DeviceType()
                  || currentDeviceInfoList[foundIdx].getE8DiscType()          != newDeviceInfoList[i].getE8DiscType())
            {
               hasChange = true;
               break;
            }
         }
         else
         {
            hasChange = true;
            break;
         }
      }

      if (!hasChange)
      {
         for (size_t i = 0; i < currentDeviceInfoList.size(); ++i)
         {
            if (findItemWithDeviceTag(newDeviceInfoList, currentDeviceInfoList[i].getU8DeviceTag()) == newDeviceInfoList.size()) //not found in new list
            {
               hasChange = true;
               break;
            }
         }
      }
   }

   return hasChange;
}


/**************************************************************************//**
 *_sds2PlayUSBProxy
 ******************************************************************************/
tBool clSDS_Property_MediaStatus::bCdAvailable(size_t idx) const
{
   if ((bIsDeviceSupportCDtracks(idx)) && (_oDeviceList[idx].getBDeviceConnected() == true))
   {
      updateCDConnected(true);
      updateCDTrackAvailable(true);
      return true;
   }
   else if ((bIsCDDeviceSupportSourceChangeOnly(idx)) && (_oDeviceList[idx].getBDeviceConnected() == true))
   {
      updateCDConnected(true);
      updateCDTrackAvailable(false);
      return true;
   }
   else
   {
      return false;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Property_MediaStatus::bDvdAvailable(size_t idx) const
{
   if ((_oDeviceList[idx].getBDeviceConnected() == true) && (_oDeviceList[idx].getE8DeviceType() == T_e8_MPlayDeviceType__e8DTY_DVD_DRIVE)
         && ((_oDeviceList[idx].getE8DiscType() == T_e8_MDiscType__e8DVD_AUDIO_DISC)
             || (_oDeviceList[idx].getE8DiscType() == T_e8_MDiscType__e8DVD_VIDEO_DISC) || (_oDeviceList[idx].getE8DiscType() == T_e8_MDiscType__e8AUDIO_CD_DISC)))
   {
      updateDVDConnected(true);
      return true;
   }
   else
   {
      return false;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Property_MediaStatus::bMusicBoxAvailable(size_t idx) const
{
   if ((_oDeviceList[idx].getBDeviceConnected() == true) && (_oDeviceList[idx].getE8DeviceType() == T_e8_MPlayDeviceType__e8DTY_MUSICBOX))
   {
      return true;
   }
   else
   {
      return false;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Property_MediaStatus::bAuxSourceAvailable() const
{
   return TRUE;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vUpdateMediaDeviceStatus(sds2hmi_fi_tcl_DeviceStatus& oDeviceStatus, size_t idx) const
{
   if (_oDeviceList[idx].getBDeviceConnected() == true)
   {
      oDeviceStatus.DeviceID = _oDeviceList[idx].getU8DeviceTag();
      std::string oDeviceInfo;
      oDeviceInfo = _oDeviceList[idx].getSDeviceName();
      oDeviceStatus.DeviceName.bSet(oDeviceInfo.c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
      oDeviceStatus.Status.enType = eGetDeviceStatus(idx);
      oDeviceStatus.Type.enType = eGetMediaPlayerSource(idx);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vUpdateDeviceListWithMedia(std::vector<sds2hmi_fi_tcl_DeviceStatus>& oDeviceList)
{
   if (_sds2MediaProxy->hasMediaPlayerDeviceConnections())
   {
      for (size_t idx = 0; idx < _oDeviceList.size(); ++idx)
      {
         if (bMediaSourceAvailable(idx))
         {
            sds2hmi_fi_tcl_DeviceStatus oDeviceStatus;
            vUpdateMediaDeviceStatus(oDeviceStatus, idx);
            oDeviceList.push_back(oDeviceStatus);
         }
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vUpdateDeviceListWithBTAudio(std::vector<sds2hmi_fi_tcl_DeviceStatus>& oDeviceList) const
{
   if (_sds2MediaProxy->hasMediaPlayerDeviceConnections())
   {
      for (size_t idx = 0; idx < _oDeviceList.size(); ++idx)
      {
         if (bBTSourceAvailable(idx))
         {
            sds2hmi_fi_tcl_DeviceStatus oDeviceStatus;
            oDeviceStatus.Type.enType = sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_BTAUDIO;
            oDeviceStatus.Status.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY;
            oDeviceList.push_back(oDeviceStatus);
         }
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vUpdateDeviceListWithAux(std::vector<sds2hmi_fi_tcl_DeviceStatus>& oDeviceList) const
{
   if (bAuxSourceAvailable())
   {
      sds2hmi_fi_tcl_DeviceStatus oDeviceStatus;
      oDeviceStatus.Type.enType = sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_LINE_IN;
      oDeviceStatus.Status.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY;
      oDeviceList.push_back(oDeviceStatus);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vUpdateDeviceListWithCD(std::vector<sds2hmi_fi_tcl_DeviceStatus>& oDeviceList) const
{
   if (_sds2MediaProxy->hasMediaPlayerDeviceConnections())
   {
      bool cdAvailable = false;

      for (size_t idx = 0; idx < _oDeviceList.size(); ++idx)
      {
         if (bCdAvailable(idx))
         {
            sds2hmi_fi_tcl_DeviceStatus oDeviceStatus;
            oDeviceStatus.Type.enType = sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_CDA;
            oDeviceStatus.Status.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY;
            oDeviceList.push_back(oDeviceStatus);
            cdAvailable = true;
         }
      }
      if (cdAvailable != true)
      {
         updateCDConnected(false);
         updateCDTrackAvailable(false);
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vUpdateDeviceList(std::vector<sds2hmi_fi_tcl_DeviceStatus>& oDeviceList)
{
   vUpdateDeviceListWithMedia(oDeviceList);
   vUpdateDeviceListWithBTAudio(oDeviceList);
   vUpdateDeviceListWithAux(oDeviceList);
   vUpdateDeviceListWithCD(oDeviceList);
   updateDeviceListWithCarPlayAudio(oDeviceList);
   updateDeviceListWithAndroidAutoAudio(oDeviceList);
   vUpdateDeviceListWithDVD(oDeviceList);
   vUpdateDeviceListWithMusicBox(oDeviceList);
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vTraceMediaStatus(const sds2hmi_sdsfi_tclMsgMediaStatusStatus& oMessage) const
{
   for (size_t idx = 0; idx < oMessage.DeviceList.size(); ++idx)
   {
      sds2hmi_fi_tcl_DeviceStatus oDeviceStatus(oMessage.DeviceList[idx]);
      ETG_TRACE_USR4(("DeviceID %d, DeviceStaus %d, DeviceType %d",
                      oDeviceStatus.DeviceID,
                      oDeviceStatus.Status.enType,
                      oDeviceStatus.Type.enType));
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_MediaStatus::onAvailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _sds2MediaProxy)
   {
      _sds2MediaProxy->sendMediaPlayerDeviceConnectionsUpReg(*this);
   }
   if (proxy == _smartphoneProxy)
   {
      _smartphoneProxy->sendSessionStatusInfoUpReg(*this);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_MediaStatus::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _sds2MediaProxy)
   {
      _sds2MediaProxy->sendMediaPlayerDeviceConnectionsRelUpRegAll();
   }
   if (proxy == _smartphoneProxy)
   {
      _smartphoneProxy->sendSessionStatusInfoRelUpRegAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_MediaStatus:: onMediaPlayerDeviceConnectionsError(const ::boost::shared_ptr< Mplay_MediaPlayer_FIProxy >& /*proxy*/,
      const boost::shared_ptr< MediaPlayerDeviceConnectionsError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_MediaStatus::onMediaPlayerDeviceConnectionsStatus(const ::boost::shared_ptr< Mplay_MediaPlayer_FIProxy >& /*proxy*/,
      const boost::shared_ptr< MediaPlayerDeviceConnectionsStatus >& status)
{
   if (status->hasODeviceInfo())
   {
      if (diff(_oDeviceList, status->getODeviceInfo()))
      {
         _oDeviceList = status->getODeviceInfo();
         vSendStatus();
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_MediaStatus::onSessionStatusInfoError(
   const ::boost::shared_ptr< Smartphoneint_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< SessionStatusInfoError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_MediaStatus::onSessionStatusInfoStatus(
   const ::boost::shared_ptr< Smartphoneint_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< SessionStatusInfoStatus >& status)
{
   const bool carPlay =
      (status->getDeviceCategory() == T_e8_DeviceCategory__DEV_TYPE_DIPO
       && status->getSessionStatus() == T_e8_SessionStatus__SESSION_ACTIVE);

   const bool androidAuto =
      (status->getDeviceCategory() == T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO
       && status->getSessionStatus() == T_e8_SessionStatus__SESSION_ACTIVE);

   if (_carPlayAvailable != carPlay
         || _androidAutoAvailable != androidAuto)
   {
      updateCarPlayAndroidAutoStatus(carPlay, androidAuto);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_MediaStatus::updateCarPlayAndroidAutoStatus(bool carPlay, bool androidAuto)
{
   _carPlayAvailable = carPlay;
   _androidAutoAvailable = androidAuto;

   clSDS_ConfigurationFlags::setDynamicVariable("AppleCarPlay", _carPlayAvailable);
   clSDS_ConfigurationFlags::setDynamicVariable("AndroidAutoAudio", _androidAutoAvailable);

   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_MediaStatus::updateDeviceListWithCarPlayAudio(std::vector<sds2hmi_fi_tcl_DeviceStatus>& oDeviceList) const
{
   if (_carPlayAvailable)
   {
      sds2hmi_fi_tcl_DeviceStatus oDeviceStatus;
      oDeviceStatus.Type.enType = sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_CARPLAY_AUD;
      oDeviceStatus.Status.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY;
      oDeviceList.push_back(oDeviceStatus);
   }
   else
   {
      sds2hmi_fi_tcl_DeviceStatus oDeviceStatus;
      oDeviceStatus.Type.enType = sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_CARPLAY_AUD;
      oDeviceStatus.Status.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_DISABLED;
      oDeviceList.push_back(oDeviceStatus);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_MediaStatus::updateDeviceListWithAndroidAutoAudio(std::vector<sds2hmi_fi_tcl_DeviceStatus>& oDeviceList) const
{
   if (_androidAutoAvailable)
   {
      sds2hmi_fi_tcl_DeviceStatus oDeviceStatus;
      oDeviceStatus.Type.enType = sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_ANDROID_AUD;
      oDeviceStatus.Status.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY;
      oDeviceList.push_back(oDeviceStatus);
   }
   else
   {
      sds2hmi_fi_tcl_DeviceStatus oDeviceStatus;
      oDeviceStatus.Type.enType = sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_ANDROID_AUD;
      oDeviceStatus.Status.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_DISABLED;
      oDeviceList.push_back(oDeviceStatus);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_MediaStatus::updateCDConnected(bool cdConnected) const
{
   clSDS_ConfigurationFlags::setDynamicVariable("CD_CONNECTED", cdConnected);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_MediaStatus::updateDVDConnected(bool dvdConnected) const
{
   clSDS_ConfigurationFlags::setDynamicVariable("DVD_CONNECTED", dvdConnected);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_MediaStatus::updateCDTrackAvailable(bool cdtrackAvailable) const
{
   clSDS_ConfigurationFlags::setDynamicVariable("CDTRACK_AVAILABLE", cdtrackAvailable);
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vUpdateDeviceListWithDVD(std::vector<sds2hmi_fi_tcl_DeviceStatus>& oDeviceList) const
{
   bool dvdAvailable = false;
   if (_sds2MediaProxy->hasMediaPlayerDeviceConnections())
   {
      for (size_t idx = 0; idx < _oDeviceList.size(); ++idx)
      {
         if (bDvdAvailable(idx))
         {
            sds2hmi_fi_tcl_DeviceStatus oDeviceStatus;
            oDeviceStatus.Type.enType = sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_DVD;
            oDeviceStatus.Status.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY;
            oDeviceList.push_back(oDeviceStatus);
            dvdAvailable = true;
         }
      }
      if (dvdAvailable != true)
      {
         updateDVDConnected(false);
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_MediaStatus::vUpdateDeviceListWithMusicBox(std::vector<sds2hmi_fi_tcl_DeviceStatus>& oDeviceList) const
{
   if (_sds2MediaProxy->hasMediaPlayerDeviceConnections())
   {
      for (size_t idx = 0; idx < _oDeviceList.size(); ++idx)
      {
         if (bMusicBoxAvailable(idx))
         {
            sds2hmi_fi_tcl_DeviceStatus oDeviceStatus;
            oDeviceStatus.Type.enType = sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_MUSIC_BOX;
            oDeviceStatus.Status.enType = sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY;
            oDeviceList.push_back(oDeviceStatus);
         }
      }
   }
}
