/**************************************************************************//**
 * \file       clSDS_Property_CommonStatus.cpp
 *
 * Common Action Request property implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Property_CommonStatus.h"
#include "application/clSDS_ConfigurationFlags.h"
#include "application/clSDS_IsVehicleMoving.h"
#include "external/sds2hmi_fi.h"
#include "SdsAdapter_Trace.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Property_CommonStatus.cpp.trc.h"
#endif


#define MAX_VEHICLE_SPEED_IN_MILES 5
#define KMPH_TO_MILES_CONVERSION_VALUE 161


using namespace VEHICLE_MAIN_FI;
using namespace tcu_main_fi;


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_Property_CommonStatus::~clSDS_Property_CommonStatus()
{
   _pIsVehicleMoving = NULL;
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_Property_CommonStatus::clSDS_Property_CommonStatus(
   ahl_tclBaseOneThreadService* pService,
   clSDS_IsVehicleMoving* pIsVehicleMoving,
   ::boost::shared_ptr< VEHICLE_MAIN_FIProxy > pVehicleFiProxy,
   ::boost::shared_ptr< Tcu_main_fiProxy > pTcuFiProxy)
   : clServerProperty(SDS2HMI_SDSFI_C_U16_COMMONSTATUS, pService),
     _pIsVehicleMoving(pIsVehicleMoving),
     _vehicleProxy(pVehicleFiProxy),
     _tcuProxy(pTcuFiProxy),
     _tcuConnectionStatus(false),
     _sportsStatus(false)
{
   _pIsVehicleMoving->vRegisterObserver(this);
   _vehicleSpeed = 0;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_CommonStatus::vSet(amt_tclServiceData* /*pInMsg*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_CommonStatus::vGet(amt_tclServiceData* /*pInMsg*/)
{
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_CommonStatus::vUpreg(amt_tclServiceData* /*pInMsg*/)
{
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_CommonStatus::vSendStatus()
{
   sds2hmi_sdsfi_tclMsgCommonStatusStatus oMessage;
   oMessage.IsVehicleMoving = ((_vehicleSpeed > MAX_VEHICLE_SPEED_IN_MILES) ? TRUE : FALSE);
   oMessage.TCUAvialable = _tcuConnectionStatus;
   oMessage.SportsServiceAvailable = false; //Sport status update through property_InfoServiceStatus.
   //no need to set oMessage.FeatureConfiguration - this was only used for lcn2kai

   vStatus(oMessage);
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_CommonStatus::vTCUServiceStatusChanged()
{
   ETG_TRACE_USR1(("TCU Service Status Changed %d", _tcuConnectionStatus));
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Property_CommonStatus::vVehicleSpeedChanged()
{
   ETG_TRACE_USR1(("Vehicle Speed Changed. Speed is %f ", _vehicleSpeed));
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonStatus::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _vehicleProxy)
   {
      _vehicleProxy->sendSpeedUpReg(*this);
   }
   if (proxy == _tcuProxy)
   {
      _tcuProxy->sendConnectionUpReg(*this);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonStatus::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _vehicleProxy)
   {
      _vehicleProxy->sendSpeedRelUpRegAll();
   }
   if (proxy == _tcuProxy)
   {
      _tcuProxy->sendConnectionRelUpRegAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonStatus::onSpeedError(
   const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< SpeedError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonStatus::onSpeedStatus(
   const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< SpeedStatus >& status)
{
   uint16 _vehicleSpeedinKmph = status->getSpeedValue();
   if (_vehicleSpeedinKmph < 65535)
   {
      _vehicleSpeed = _vehicleSpeedinKmph / KMPH_TO_MILES_CONVERSION_VALUE;
      vVehicleSpeedChanged();
   }
   else
   {
      ETG_TRACE_USR1(("Vehicle speed received from Vehicle is invalid"));
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonStatus::onConnectionError(
   const ::boost::shared_ptr< Tcu_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< ConnectionError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonStatus::onConnectionStatus(
   const ::boost::shared_ptr< Tcu_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< ConnectionStatus >& status)
{
   _tcuConnectionStatus = false ;

   if (status->hasBConnected())
   {
      _tcuConnectionStatus = status->getBConnected();
   }

   if (clSDS_ConfigurationFlags::disableTCUforCurrentRegion())
   {
      _tcuConnectionStatus = false;
   }
   vTCUServiceStatusChanged();
}
