/**************************************************************************//**
 * \file       clSDS_Method_PhoneSetPhoneSetting.cpp
 *
 * clSDS_Method_PhoneSetPhoneSetting method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Method_PhoneSetPhoneSetting.h"
#include "Sds2HmiServer/functions/clSDS_Property_PhoneStatus.h"
#include "external/sds2hmi_fi.h"


using namespace MOST_BTSet_FI;
using namespace MOST_Tel_FI;
using namespace most_Tel_fi_types;

using namespace most_BTSet_fi_types_Extended;

/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_Method_PhoneSetPhoneSetting::~clSDS_Method_PhoneSetPhoneSetting()
{
   _pPhoneStatus = NULL;
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_Method_PhoneSetPhoneSetting::clSDS_Method_PhoneSetPhoneSetting(
   ahl_tclBaseOneThreadService* pService,
   ::boost::shared_ptr< ::MOST_BTSet_FI::MOST_BTSet_FIProxy > bluetoothSettingsProxy,
   ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy > telephoneProxy,
   clSDS_Property_PhoneStatus* pPhoneStatus, GuiService& guiService)

   : clServerMethod(SDS2HMI_SDSFI_C_U16_PHONESETPHONESETTING, pService)
   , _guiService(guiService)
   , _bluetoothSettingsProxy(bluetoothSettingsProxy)
   , _telephoneProxy(telephoneProxy)
   , _pPhoneStatus(pPhoneStatus)
   , _btOnOffState(false)
   , _autoConnectSettings(true)
   , _noPairedDevices(0)
{
}


/***********************************************************************//**
 * A timer of 3 sec is introduced to delay the MResult to allow auto connection
 * in the background
 ***************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::setBluetooth()
{
   if (_bluetoothSettingsProxy->isAvailable())
   {
      _bluetoothSettingsProxy->sendSwitchBluetoothOnOffExtendedStart(*this, _btOnOffState, false);
   }
   if (_btOnOffState == true)
   {
      _methodResponseWaitTimer.start(*this, 3000);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onSwitchBluetoothOnOffExtendedError(
   const ::boost::shared_ptr< MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< SwitchBluetoothOnOffExtendedError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onSwitchBluetoothOnOffExtendedResult(
   const ::boost::shared_ptr< MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< SwitchBluetoothOnOffExtendedResult >& /*result*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::setHandsfreeMode()
{
   if (_pPhoneStatus != NULL)
   {
      if (_pPhoneStatus->getPhoneStatus() == sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_HANDSET_MODE)
      {
         _telephoneProxy->sendTransferCallToVehicleStart(*this , _pPhoneStatus->getDeviceHandle());
         _guiService.sendEventSignal(sds_gui_fi::SdsGuiService::Event__SPEECH_DIALOG_SDS_PHONE_HANDSFREE_STATE);
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_PhoneSetPhoneSetting::vMethodStart(amt_tclServiceData* pInMessage)
{
   sds2hmi_sdsfi_tclMsgPhoneSetPhoneSettingMethodStart oMessage;
   vGetDataFromAmt(pInMessage, oMessage);

   switch (oMessage.nPhoneSetting.enType)
   {
      case sds2hmi_fi_tcl_e8_PHN_Setting::FI_EN_BLUETOOTH_ON:
         _btOnOffState = true;
         setBluetooth();
         break;

      case sds2hmi_fi_tcl_e8_PHN_Setting::FI_EN_BLUETOOTH_OFF:
         _btOnOffState = false;
         setBluetooth();
         break;

      case sds2hmi_fi_tcl_e8_PHN_Setting::FI_EN_SWITCH_TO_HANDSFREE:
         setHandsfreeMode();
         postMethodResult();
         break;

      default:
         break;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onTransferCallToVehicleError(
   const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< TransferCallToVehicleError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onTransferCallToVehicleResult(
   const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< TransferCallToVehicleResult >& /*result*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _bluetoothSettingsProxy)
   {
      // register for property to receive acknowledgment via onBluetoothOnOffStatus
      // for sendSwitchBluetoothOnOffStart
      _bluetoothSettingsProxy->sendBluetoothOnOffUpReg(*this);
      _bluetoothSettingsProxy->sendAutoConnectOnOffUpReg(*this);
      _bluetoothSettingsProxy->sendAutoConnectionStatusUpReg(*this);
      _bluetoothSettingsProxy->sendDeviceListExtendedUpReg(*this);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _bluetoothSettingsProxy)
   {
      _bluetoothSettingsProxy->sendBluetoothOnOffRelUpRegAll();
      _bluetoothSettingsProxy->sendAutoConnectOnOffRelUpRegAll();
      _bluetoothSettingsProxy->sendAutoConnectionStatusRelUpRegAll();
      _bluetoothSettingsProxy->sendDeviceListExtendedRelUpRegAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onBluetoothOnOffError(
   const ::boost::shared_ptr< MOST_BTSet_FI::MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_BTSet_FI::BluetoothOnOffError >& /*error*/)
{
}


/**************************************************************************//**
 *Ideally if Auto-ConnectionSettings is off, then we can post MR immediately.
 * But currently irrespective of the Settings, AutoConnection is happening
 * hence commented the _autoConnectSettings condition check
 * Also if user is switching on the BT "ON" then we need to delay the MR
 * In case of No devices, we need wait AutoConnectionStatus to come to IDLE
 * we can post the MR immediately
 ******************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onBluetoothOnOffStatus(
   const ::boost::shared_ptr< MOST_BTSet_FI::MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_BTSet_FI::BluetoothOnOffStatus >& status)
{
   if (status->getBBTOnOff() == _btOnOffState)
   {
      if (status->getBBTOnOff() == true)
      {
         if (_noPairedDevices == 0 /*|| _autoConnectSettings == false*/)
         {
            postMethodResult();
         }
      }
      else if (status->getBBTOnOff() == false)
      {
         postMethodResult();
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onAutoConnectionStatusError(
   const ::boost::shared_ptr< MOST_BTSet_FI::MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_BTSet_FI::AutoConnectionStatusError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onAutoConnectionStatusStatus(
   const ::boost::shared_ptr< MOST_BTSet_FI::MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_BTSet_FI::AutoConnectionStatusStatus >& status)
{
   if (status->getE8AutoConnectionStatus() ==  T_e8_BTAutoconnectionStatus__e8IDLE)
   {
      postMethodResult();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onAutoConnectOnOffError(
   const ::boost::shared_ptr< MOST_BTSet_FI::MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_BTSet_FI::AutoConnectOnOffError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onAutoConnectOnOffStatus(
   const ::boost::shared_ptr< MOST_BTSet_FI::MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_BTSet_FI::AutoConnectOnOffStatus >& status)
{
   _autoConnectSettings = status->getBAutoConnectOnOff();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onDeviceListExtendedError(
   const ::boost::shared_ptr< MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DeviceListExtendedError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onDeviceListExtendedStatus(
   const ::boost::shared_ptr< MOST_BTSet_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DeviceListExtendedStatus >& status)
{
   _noPairedDevices = status->getU8NumPairedDevices();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::onExpired(asf::core::Timer& /*timer*/, boost::shared_ptr<asf::core::TimerPayload> /*data*/)
{
   postMethodResult();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_PhoneSetPhoneSetting::postMethodResult()
{
   if (_methodResponseWaitTimer.isActive())
   {
      _methodResponseWaitTimer.stop();
   }
   vSendMethodResult();
}
