/**************************************************************************//**
 * \file       clSDS_Method_PhoneDialNumber.cpp
 *
 * clSDS_Method_PhoneDialNumber method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Method_PhoneDialNumber.h"
#include "application/clSDS_Iconizer.h"
#include "application/clSDS_ListScreen.h"
#include "application/PhonebookContacts.h"
#include "application/AudioSourceHandler.h"
#include "application/GuiService.h"
#include "SdsAdapter_Trace.h"
#include "application/PhoneUtils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_PhoneDialNumber.cpp.trc.h"
#endif


using namespace MOST_Tel_FI;
using namespace most_Tel_fi_types;
using namespace MOST_PhonBk_FI;
using namespace most_PhonBk_fi_types;
using namespace sds_gui_fi::SdsGuiService;


/**************************************************************************//**
* Destructor
******************************************************************************/
clSDS_Method_PhoneDialNumber::~clSDS_Method_PhoneDialNumber()
{
   _pListScreen = NULL;
   _pPhonebookContacts = NULL;
   _pAudioSourceHandler = NULL;
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_Method_PhoneDialNumber::clSDS_Method_PhoneDialNumber(
   ahl_tclBaseOneThreadService* pServer,
   clSDS_ListScreen* pListScreen,
   PhonebookContacts* pPhonebookContacts,
   ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy > telephoneProxy,
   ::boost::shared_ptr< MOST_PhonBk_FIProxy > phonebookProxy,
   AudioSourceHandler* pAudioSourceHandler,
   GuiService& guiService)

   : clServerMethod(SDS2HMI_SDSFI_C_U16_PHONEDIALNUMBER, pServer)
   , _telephoneProxy(telephoneProxy)
   , _phonebookProxy(phonebookProxy)
   , _pPhonebookContacts(pPhonebookContacts)
   , _pListScreen(pListScreen)
   , _pAudioSourceHandler(pAudioSourceHandler)
   , _guiService(guiService)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneDialNumber::vMethodStart(amt_tclServiceData* pInMessage)
{
   sds2hmi_sdsfi_tclMsgPhoneDialNumberMethodStart oMessage;
   vGetDataFromAmt(pInMessage, oMessage);
   ETG_TRACE_COMP((
                     "PhoneDialNumber.vMethodStart(type=%d index=%d number='%s')",
                     oMessage.nSelectionType.enType,
                     oMessage.nIDValue,
                     oMessage.sPhoneNumber.szValue));
   vInvokeDialNumber(oMessage);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneDialNumber::vInvokeDialNumber(const sds2hmi_sdsfi_tclMsgPhoneDialNumberMethodStart& oMessage)
{
   switch (oMessage.nSelectionType.enType)
   {
      case sds2hmi_fi_tcl_e8_GEN_SelectionType::FI_EN_BYVAL:
      case sds2hmi_fi_tcl_e8_GEN_SelectionType::FI_EN_UNKNOWN:
         vDialNumberByValue(oMessage.sPhoneNumber.szValue);
         break;

      case sds2hmi_fi_tcl_e8_GEN_SelectionType::FI_EN_BYLISTNUMBER:
         vDialByIndex(oMessage.nIDValue);
         break;

      case sds2hmi_fi_tcl_e8_GEN_SelectionType::FI_EN_BYUSW:
         vDialNumberByUWID(oMessage.nIDValue);
         break;

      default:
         vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
         break;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneDialNumber::vDialNumberByValue(std::string strPhoneNumber)
{
   if (strPhoneNumber != "" && _pAudioSourceHandler)
   {
      strPhoneNumber = clSDS_Iconizer::oRemoveIconPrefix(strPhoneNumber);
      _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PHONE_CALL_SIRI_VA);
      _telephoneProxy->sendDialStart(*this, strPhoneNumber, T_e8_TelEchoCancellationNoiseReductionSetting__e8ECNR_NOCHANGE);
      _pAudioSourceHandler->onOutgoingCall();
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneDialNumber::vDialByIndex(tU32 u32Index)
{
   ETG_TRACE_USR4(("clSDS_Method_PhoneDialNumber::vDialByIndex index = %d", u32Index));

   std::string phoneNumber = "";

   if (u32Index == 0)
   {
      phoneNumber = _pPhonebookContacts->getContactNumber(u32Index);
   }
   else
   {
      phoneNumber = _pListScreen->oGetSelectedItem(u32Index);
   }

   ETG_TRACE_USR4(("clSDS_Method_PhoneDialNumber::vDialByIndex phoneNumber = %s", phoneNumber.c_str()));

   vDialNumberByValue(phoneNumber);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneDialNumber::vDialNumberByUWID(tU32 u32UwId)
{
   if (_phonebookProxy->isAvailable())
   {
      _phonebookProxy->sendGetContactDetailsExtendedStart(*this, u32UwId);
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneDialNumber::onDialError(
   const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DialError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_PhoneDialNumber::onDialResult(
   const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DialResult >& /*result*/)
{
   vSendMethodResult();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_PhoneDialNumber::onGetContactDetailsExtendedError(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::GetContactDetailsExtendedError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_PhoneDialNumber::onGetContactDetailsExtendedResult(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::GetContactDetailsExtendedResult >& result)
{
   std::string phoneNumber = PhoneUtils::getPhoneNumber(result->getOContactDetailsExtended());

   vDialNumberByValue(phoneNumber);
}
