/**************************************************************************//**
 * \file       clSDS_Method_NaviValidateDestinationEntries.cpp
 *
 * clSDS_Method_NaviValidateDestinationEntries method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/

#include "Sds2HmiServer/functions/clSDS_Method_NaviValidateDestinationEntries.h"

#include "SdsAdapter_Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_NaviValidateDestinationEntries.cpp.trc.h"
#endif


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_Method_NaviValidateDestinationEntries::~clSDS_Method_NaviValidateDestinationEntries()
{
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_Method_NaviValidateDestinationEntries::clSDS_Method_NaviValidateDestinationEntries(
   ahl_tclBaseOneThreadService* pService,
   ::boost::shared_ptr<NavigationServiceProxy> naviProxy,
   NdsCountryStateRequestor* pNdsCountryStateRequestor,
   clSDS_LanguageMediator* pLanguageMediator)

   :  clServerMethod(SDS2HMI_SDSFI_C_U16_NAVIVALIDATEDESTINATIONENTRIES, pService)
   , _navigationProxy(naviProxy)
   , _pNdsCountryStateRequestor(pNdsCountryStateRequestor)
   , _pLanguageMediator(pLanguageMediator)
   , _destinationItemId(0)
   , _countryStateInfoAvailable(false)
   , _countryStateInfoRequested(false)

{
   _pNdsCountryStateRequestor->addCountryStateObserver(this);
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _navigationProxy)
   {
      _navigationProxy->sendSdsAddressWithOptionsDeregisterAll();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::onAvailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _navigationProxy)
   {
      _navigationProxy->sendSdsAddressWithOptionsRegister(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::onSdsCheckAddressError(const ::boost::shared_ptr<NavigationServiceProxy>& /*proxy*/,
      const ::boost::shared_ptr< SdsCheckAddressError >& /*Error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::onSdsAddressWithOptionsError(const ::boost::shared_ptr<NavigationServiceProxy>& /*proxy*/,
      const ::boost::shared_ptr< SdsAddressWithOptionsError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_UNSPECIFIED);
}


/***********************************************************************//**
 *
 ***************************************************************************/

void clSDS_Method_NaviValidateDestinationEntries::vMethodStart(amt_tclServiceData* pInMessage)
{
   sds2hmi_sdsfi_tclMsgNaviValidateDestinationEntriesMethodStart oMethodStart;
   vGetDataFromAmt(pInMessage, oMethodStart);

   _destinationItemList.clear();
   _destinationItemId = 0;
   _validatedList.clear();

   _destinationItemList = oMethodStart.VDEList;

   if (!_pNdsCountryStateRequestor->isCountryStateInfoAvailable() && (clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::USA))
   {
      _pNdsCountryStateRequestor->getNdsCountryStateIds();
      _countryStateInfoRequested = true;
   }
   else
   {
      sendValidationRequestToNavi();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::sendValidationRequestToNavi()
{
   if (_destinationItemId < _destinationItemList.size())
   {
      setRequestAddressMap(_destinationItemList[_destinationItemId]);
      setSDSCheckAddressFromAddressMap();
      sendSdsCheckAddressRequest();
   }
   else if (_destinationItemId == _destinationItemList.size())
   {
      sendMethodResult();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::setRequestAddressMap(sds2hmi_fi_tcl_DestinationItem address)
{
   _requestedAddressMap.clear();

   for (tU16 i = 0; i < address.DestinationItem.size(); i++)
   {
      std::string addressString = address.DestinationItem[i].DestinationItem.szGet(sds2hmi_fi_tclString::FI_EN_UTF8);
      if (!addressString.empty())
      {
         _requestedAddressMap[address.DestinationItem[i].DestinationItemType.enType] = addressString;
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::setSDSCheckAddressFromAddressMap()
{
   ETG_TRACE_USR1(("NaviValidateDestinationEntries ::setSDSCheckAddressFromDestinationItem()"));

   _oRequestingsdsAddress.clear();

   bool replacedStateName = false;

   if (_requestedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_STATE] != "" && (clSDS_KDSConfiguration::USA == clSDS_KDSConfiguration::getMarketRegionType()))
   {
      std::string correctStateName = _pNdsCountryStateRequestor->getNdsStateName(_requestedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_STATE].c_str(), convertISOLanguageCodetoString((_pLanguageMediator->getLanguageOfActiveSpeaker()).ISO639_3_SDSLanguageCode));
      setSDSAddressElement(_pNdsCountryStateRequestor->getNdsCountryName(), SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY);
      setSDSAddressElement(correctStateName, SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_STATE);
      replacedStateName = true;
   }
   else
   {
      //For regions where only city is available we can directly use the country name from SDS.
      setSDSAddressElement(_requestedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_COUNTRY], SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY);
   }

   if (!replacedStateName)
   {
      setSDSAddressElement(_requestedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_STATE], SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_STATE);
   }

   setSDSAddressElement(_requestedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_CITY], SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_PLACE);
   setSDSAddressElement(_requestedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_OAZA], SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CITYDESTRICT);
   setSDSAddressElement(_requestedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_AZA], SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROAD);

   setSDSAddressElement(_requestedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_STREET], SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROAD);
   setSDSAddressElement(_requestedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_HOUSENO], SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_HOUSENUMBER);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::setSDSAddressElement(const std::string& searchContent, SDSAddressElementType searchElement)
{
   if (!searchContent.empty())
   {
      ETG_TRACE_USR1(("setSDSAddressElement= %s", searchContent.c_str()));
      SDSAddressElement sdsAddressElement;
      sdsAddressElement.setAddressElementType(searchElement);
      SDSElementOptions sdsElementOptions;
      sdsElementOptions.setIsAmbigious(false);
      sdsAddressElement.setElementOptions(sdsElementOptions);
      sdsAddressElement.setData(searchContent);
      _oRequestingsdsAddress.push_back(sdsAddressElement);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::sendSdsCheckAddressRequest()
{
   ETG_TRACE_USR1(("sendSdsCheckAddressRequest"));
   _navigationProxy->sendSdsCheckAddressRequest(*this, _oRequestingsdsAddress);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::onSdsAddressWithOptionsUpdate(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsAddressWithOptionsUpdate >& /*oSdsAddressWithOptionsUpdate*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::onSdsCheckAddressResponse(const ::boost::shared_ptr<NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsCheckAddressResponse >& /*osdsCheckAddressResponse*/)
{
   ETG_TRACE_USR1(("clSDS_Method_NaviValidateDestinationEntries::onSdsCheckAddressResponse()- Function entered"));

   getAddressValidationResultFromNavi();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::getAddressValidationResultFromNavi()
{
   std::vector< SDSAddressElement> verifiedSdsAddress;
   SDSAddressOptions sdsAddressOptions;

   if (_navigationProxy->hasSdsAddressWithOptions())
   {
      verifiedSdsAddress = _navigationProxy->getSdsAddressWithOptions().getAddress();
      sdsAddressOptions = _navigationProxy->getSdsAddressWithOptions().getAddressOptions();
   }

   setVerifiedAddressMap(verifiedSdsAddress);
   setValidationResult(sdsAddressOptions);
}


/***********************************************************************//**
 *
 ***************************************************************************/


void clSDS_Method_NaviValidateDestinationEntries::setVerifiedAddressMap(::std::vector< SDSAddressElement >& verifiedAddress)
{
   ETG_TRACE_USR1(("setVerifiedAddressMap- Function entered"));

   ::std::vector<SDSAddressElement>::iterator it;
   _verifiedAddressMap.clear();

   for (it = verifiedAddress.begin(); it != verifiedAddress.end(); ++it)
   {
      if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_HOUSENUMBER)
      {
         _verifiedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_HOUSENO] = it->getData();
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROAD)
      {
         if (clSDS_KDSConfiguration::JPN == clSDS_KDSConfiguration::getMarketRegionType())
         {
            _verifiedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_AZA] = it->getData();
         }
         else
         {
            _verifiedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_STREET] = it->getData();
         }
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CITYDESTRICT)
      {
         if (clSDS_KDSConfiguration::JPN == clSDS_KDSConfiguration::getMarketRegionType())
         {
            _verifiedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_OAZA] = it->getData();
         }
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_PLACE)
      {
         _verifiedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_CITY] = it->getData();
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_STATE)
      {
         _verifiedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_STATE] = it->getData();
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY)
      {
         _verifiedAddressMap[sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_COUNTRY] = it->getData();
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::setValidationResult(const SDSAddressOptions&  sdsAddressOptions)
{
   sds2hmi_fi_tcl_DestinationElement validatedResult;
   if (!sdsAddressOptions.getRemovedInvalidEntry())
   {
      validatedResult.DestinationItemType.enType = sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_RESERVED;
      _validatedList.push_back(validatedResult);
   }
   else
   {
      std::map<sds2hmi_fi_tcl_e8_SDSDestinationType::tenType, std::string>::iterator it = _requestedAddressMap.begin();

      for (; it != _requestedAddressMap.end(); it++)
      {
         if (!(it->second).empty() && _verifiedAddressMap.find(it->first) == _verifiedAddressMap.end())
         {
            validatedResult.DestinationItemType.enType = it->first;
            validatedResult.DestinationItem.bSet(it->second.c_str(), sds2hmi_fi_tclString::FI_EN_UTF8) ;
            break;
         }
      }
      if (clSDS_KDSConfiguration::JPN == clSDS_KDSConfiguration::getMarketRegionType())
      {
         if (validatedResult.DestinationItemType.enType == sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_HOUSENO)
         {
            if ((_requestedAddressMap.find(sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_OAZA) != _requestedAddressMap.end()) &&
                  _verifiedAddressMap.find(sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_OAZA) == _verifiedAddressMap.end())
            {
               validatedResult.DestinationItemType.enType =  sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_OAZA;
               validatedResult.DestinationItem.bSet((_requestedAddressMap.find(sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_OAZA)->second).c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            }
            else if ((_requestedAddressMap.find(sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_AZA) != _requestedAddressMap.end()) &&
                     _verifiedAddressMap.find(sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_AZA) == _verifiedAddressMap.end())
            {
               validatedResult.DestinationItemType.enType =  sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_AZA;
               validatedResult.DestinationItem.bSet((_requestedAddressMap.find(sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_AZA)->second).c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            }
         }
      }

      _validatedList.push_back(validatedResult);
   }
   _destinationItemId ++;
   _requestedAddressMap.clear();
   _verifiedAddressMap.clear();
   sendValidationRequestToNavi();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::sendMethodResult()
{
   ETG_TRACE_USR1(("clSDS_Method_NaviValidateDestinationEntries::sendMethodResult"));

   sds2hmi_sdsfi_tclMsgNaviValidateDestinationEntriesMethodResult result;
   if (_validatedList.size() == _destinationItemList.size())
   {
      result.InvalidList = _validatedList;
      vSendMethodResult(result);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviValidateDestinationEntries::NdsCountryStateUpdated()
{
   if (_pNdsCountryStateRequestor->isCountryStateInfoAvailable())
   {
      _countryStateInfoAvailable = true;
      if (_countryStateInfoRequested)
      {
         sendValidationRequestToNavi();
         _countryStateInfoRequested = false;
      }
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/

std::string clSDS_Method_NaviValidateDestinationEntries::convertISOLanguageCodetoString(sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode languageCode)
{
   if (languageCode.enType == sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_ENG)
   {
      return "eng";
   }
   else if (languageCode.enType == sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_FRA)
   {
      return "fra";
   }
   else if (languageCode.enType == sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_SPA)
   {
      return "spa";
   }
   else
   {
      return "";
   }
}
