/**************************************************************************//**
 * \file       clSDS_Method_NavDataGetStringAndPhoneme.cpp
 *
 * clSDS_Method_NavDataGetStringAndPhoneme method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Method_NavDataGetStringAndPhoneme.h"
#include "SdsAdapter_Trace.h"
#include "application/NdsStringRequestor.h"
#include "application/NdsPhonemeRequestor.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_NavDataGetStringAndPhoneme.cpp.trc.h"
#endif


using namespace org::bosch::cm::navigation::NavigationSDSService;


/**************************************************************************//**
* Destructor
******************************************************************************/
clSDS_Method_NavDataGetStringAndPhoneme::~clSDS_Method_NavDataGetStringAndPhoneme()
{
}


/**************************************************************************//**
* Constructor
******************************************************************************/
clSDS_Method_NavDataGetStringAndPhoneme::clSDS_Method_NavDataGetStringAndPhoneme(
   ahl_tclBaseOneThreadService* pService,
   NdsStringRequestor* pNdsStringRequestor,
   NdsPhonemeRequestor* pNdsPhonemeRequestor)
   : clServerMethod(SDS2HMI_SDSFI_C_U16_NAVDATAGETSTRINGANDPHONEME, pService)
   , _pNdsStringRequestor(pNdsStringRequestor)
   , _pNdsPhonemeRequestor(pNdsPhonemeRequestor)
{
   _pNdsStringRequestor->addNdsStringObserver(this);
   _pNdsPhonemeRequestor->addNdsPhonemeObserver(this);
   _isStringRequested = false;
   _isPhonemeRequested = false;
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NavDataGetStringAndPhoneme::vMethodStart(amt_tclServiceData* pInMessage)
{
   sds2hmi_sdsfi_tclMsgNavDataGetStringAndPhonemeMethodStart oMessage;
   vGetDataFromAmt(pInMessage, oMessage);

   _ndsLanguageId = oMessage.NDSLanguageIDList;
   _requestedNDSIds = oMessage.IdList;

   _isStringRequested = oMessage.GetString;
   _isPhonemeRequested = oMessage.GetPhoneme;

   if (_isStringRequested)
   {
      _pNdsStringRequestor->getNdsFeautreType(oMessage.IdList, oMessage.NDSLanguageIDList);
   }
   else if (_isPhonemeRequested)
   {
      _pNdsPhonemeRequestor->getNdsFeautreTypeForPhoneme(oMessage.IdList, oMessage.NDSLanguageIDList);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NavDataGetStringAndPhoneme::ndsNamesUpdated()
{
   if (_isPhonemeRequested)
   {
      _pNdsPhonemeRequestor->getNdsFeautreTypeForPhoneme(_requestedNDSIds, _ndsLanguageId);
   }
   else
   {
      getStringList();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NavDataGetStringAndPhoneme::ndsPhonemesUpdated()
{
   getPhonemeList();
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Method_NavDataGetStringAndPhoneme::getStringList()
{
   sds2hmi_sdsfi_tclMsgNavDataGetStringAndPhonemeMethodResult result;

   std::vector< NamedObjectIdStrings > namedObjectList;
   std::vector< NamedObjectIdStrings > poiNameList;
   std::vector< NamedObjectIdStrings > poiCategoryList;
   std::vector< NamedObjectIdStrings > poiAttributeList;
   sds2hmi_fi_tcl_e8_NDSFeatureType ndsType;

   _pNdsStringRequestor->getNdsNamedObjectList(namedObjectList);
   ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT;
   setStringOrPhonemeListForNdsId(namedObjectList, result.StringPhonemeList, ndsType, _isStringRequested);

   _pNdsStringRequestor->getNdsPOINameList(poiNameList);
   ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POINAME;
   setStringOrPhonemeListForNdsId(poiNameList, result.StringPhonemeList, ndsType, _isStringRequested);

   _pNdsStringRequestor->getNdsPOICategoryList(poiCategoryList);
   ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_CATEGORY;
   setStringOrPhonemeListForNdsId(poiCategoryList, result.StringPhonemeList, ndsType, _isStringRequested);

   _pNdsStringRequestor->getNdsPOIAttributeList(poiAttributeList);
   ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_ATTRIBUTE;
   setStringOrPhonemeListForNdsId(poiAttributeList, result.StringPhonemeList, ndsType, _isStringRequested);

   vSendMethodResult(result);

   _pNdsStringRequestor->clearAllListData();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetStringAndPhoneme::setStringOrPhonemeListForNdsId(
   std::vector< NamedObjectIdStrings >& ndsStringData,
   std::vector<sds2hmi_fi_tcl_StringAndPhoneme>& sdsStringData,
   sds2hmi_fi_tcl_e8_NDSFeatureType ndsFeatureType,
   bool isStringRequested)
{
   std::vector< NamedObjectIdStrings >::iterator it = ndsStringData.begin();

   while (it != ndsStringData.end())
   {
      sds2hmi_fi_tcl_StringAndPhoneme stringObject;

      stringObject.NDSId.u16UpdateRegionId = it->getGlobalNamedObjectId().getGlobalUpdateRegionId().getUpdateRegionId();
      stringObject.NDSId.u32ProductId = it->getGlobalNamedObjectId().getGlobalUpdateRegionId().getProductId();
      stringObject.NDSId.u8SupplierId = it->getGlobalNamedObjectId().getGlobalUpdateRegionId().getSupplierId();
      stringObject.NDSId.u32NDSId = it->getGlobalNamedObjectId().getNamedObjectId();
      stringObject.NDSId.eNDSFeatureType.enType = ndsFeatureType.enType;

      std::vector< ::std::string >::iterator it1;
      std::vector< ::std::string > stringList;

      stringList = it->getSdsNamedStringsList();

      for (it1 = stringList.begin(); it1 != stringList.end(); ++it1)
      {
         if (isStringRequested)
         {
            sds2hmi_fi_tclString sdsString;
            sdsString.bSet(it1->c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            stringObject.OrthographyList.push_back(sdsString);
         }
         else
         {
            sds2hmi_fi_tcl_PhonemeForLanguage phonemeLanguageEntry;
            phonemeLanguageEntry.Language.ISO_CountryCode.bSet(it->getLanguageEntry().getISOCountryCode().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            phonemeLanguageEntry.Language.ISO_LanguageCode.bSet(it->getLanguageEntry().getISOLanguageCode().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            phonemeLanguageEntry.Language.ScriptCode.bSet(it->getLanguageEntry().getTranscriptCode().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            phonemeLanguageEntry.Phoneme.bSet(it1->c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            stringObject.PhonemeAndLanguageList.push_back(phonemeLanguageEntry);
         }
      }

      stringObject.DestItemtype = setDestinationType(it->getAddressElementType());
      sdsStringData.push_back(stringObject);

      ++it;
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Method_NavDataGetStringAndPhoneme::getPhonemeList()
{
   sds2hmi_sdsfi_tclMsgNavDataGetStringAndPhonemeMethodResult result;
   sds2hmi_fi_tcl_e8_NDSFeatureType ndsType;

   std::vector< NamedObjectIdStrings > phonemeObjectList;
   _pNdsPhonemeRequestor->getNdsPhonemeObjectList(phonemeObjectList);

   std::vector< NamedObjectIdStrings > phonemePOINameList;
   _pNdsPhonemeRequestor->getNdsPhonemePOINameList(phonemePOINameList);

   std::vector< NamedObjectIdStrings > phonemePOICategoryList;
   _pNdsPhonemeRequestor->getNdsPhonemePOICategoryList(phonemePOICategoryList);

   std::vector< NamedObjectIdStrings > phonemePOIAttributeList;
   _pNdsPhonemeRequestor->getNdsPhonemePOIAttributeList(phonemePOIAttributeList);

   if (!_isStringRequested)
   {
      ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT;
      setStringOrPhonemeListForNdsId(phonemeObjectList, result.StringPhonemeList, ndsType, _isStringRequested);
   }
   else
   {
      std::vector< NamedObjectIdStrings > namedObjectList;
      _pNdsStringRequestor->getNdsNamedObjectList(namedObjectList);
      ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT;
      setCombinedListForNdsId(namedObjectList, phonemeObjectList, result.StringPhonemeList, ndsType);
   }

   if (!_isStringRequested)
   {
      ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POINAME;
      setStringOrPhonemeListForNdsId(phonemePOINameList, result.StringPhonemeList, ndsType, _isStringRequested);
   }
   else
   {
      std::vector< NamedObjectIdStrings > poiNameList;
      _pNdsStringRequestor->getNdsPOINameList(poiNameList);
      ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POINAME;
      setCombinedListForNdsId(poiNameList, phonemePOINameList, result.StringPhonemeList, ndsType);
   }

   if (!_isStringRequested)
   {
      ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_CATEGORY;
      setStringOrPhonemeListForNdsId(phonemePOICategoryList, result.StringPhonemeList, ndsType, _isStringRequested);
   }
   else
   {
      std::vector< NamedObjectIdStrings > poiCategoryList;
      _pNdsStringRequestor->getNdsPOICategoryList(poiCategoryList);
      ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_CATEGORY;
      setCombinedListForNdsId(poiCategoryList, phonemePOICategoryList, result.StringPhonemeList, ndsType);
   }

   if (!_isStringRequested)
   {
      ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_ATTRIBUTE;
      setStringOrPhonemeListForNdsId(phonemePOIAttributeList, result.StringPhonemeList, ndsType, _isStringRequested);
   }
   else
   {
      std::vector< NamedObjectIdStrings > poiAttributeList;
      _pNdsStringRequestor->getNdsPOIAttributeList(poiAttributeList);
      ndsType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_ATTRIBUTE;
      setCombinedListForNdsId(poiAttributeList, phonemePOIAttributeList, result.StringPhonemeList, ndsType);
   }

   vSendMethodResult(result);

   _pNdsStringRequestor->clearAllListData();
   _pNdsPhonemeRequestor->clearAllPhonemeListData();
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Method_NavDataGetStringAndPhoneme::setCombinedListForNdsId(
   std::vector< NamedObjectIdStrings >& ndsStringList,
   std::vector< NamedObjectIdStrings >& ndsPhonemeList,
   std::vector<sds2hmi_fi_tcl_StringAndPhoneme>& sdsStringPhonemeData,
   sds2hmi_fi_tcl_e8_NDSFeatureType ndsFeatureType)
{
   std::vector< NamedObjectIdStrings >::iterator it = ndsStringList.begin();

   while (it != ndsStringList.end())
   {
      sds2hmi_fi_tcl_StringAndPhoneme stringObject;

      stringObject.NDSId.u16UpdateRegionId = it->getGlobalNamedObjectId().getGlobalUpdateRegionId().getUpdateRegionId();
      stringObject.NDSId.u32ProductId = it->getGlobalNamedObjectId().getGlobalUpdateRegionId().getProductId();
      stringObject.NDSId.u8SupplierId = it->getGlobalNamedObjectId().getGlobalUpdateRegionId().getSupplierId();
      stringObject.NDSId.u32NDSId = it->getGlobalNamedObjectId().getNamedObjectId();
      stringObject.NDSId.eNDSFeatureType.enType = ndsFeatureType.enType;

      std::vector< ::std::string > ::iterator it1;
      std::vector< ::std::string > stringList;

      stringList = it->getSdsNamedStringsList();

      for (it1 = stringList.begin(); it1 != stringList.end(); it1++)
      {
         sds2hmi_fi_tclString sdsString;
         sdsString.bSet(it1->c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
         stringObject.OrthographyList.push_back(sdsString);

         setPhonemeListForNdsId(stringObject, ndsPhonemeList);
      }

      stringObject.DestItemtype = setDestinationType(it->getAddressElementType());
      sdsStringPhonemeData.push_back(stringObject);

      ++it;
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Method_NavDataGetStringAndPhoneme::setPhonemeListForNdsId(
   sds2hmi_fi_tcl_StringAndPhoneme& phonemeObject,
   std::vector< NamedObjectIdStrings > ndsPhonemeList)
{
   std::vector< NamedObjectIdStrings >::iterator it = ndsPhonemeList.begin();

   while (it != ndsPhonemeList.end())
   {
      if (phonemeObject.NDSId.u32NDSId == it->getGlobalNamedObjectId().getNamedObjectId())
      {
         std::vector< ::std::string > ::iterator it1;
         std::vector< ::std::string > stringList;

         stringList = it->getSdsNamedStringsList();

         for (it1 = stringList.begin(); it1 != stringList.end(); it1++)
         {
            sds2hmi_fi_tcl_PhonemeForLanguage phonemeLanguageEntry;
            phonemeLanguageEntry.Language.ISO_CountryCode.bSet(it->getLanguageEntry().getISOCountryCode().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            phonemeLanguageEntry.Language.ISO_LanguageCode.bSet(it->getLanguageEntry().getISOLanguageCode().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            phonemeLanguageEntry.Language.ScriptCode.bSet(it->getLanguageEntry().getTranscriptCode().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            phonemeLanguageEntry.Phoneme.bSet(it1->c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            phonemeObject.PhonemeAndLanguageList.push_back(phonemeLanguageEntry);
         }
      }

      ++it;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
sds2hmi_fi_tcl_e8_SDSDestinationType clSDS_Method_NavDataGetStringAndPhoneme::setDestinationType
(AddressElementType addressElementType)
{
   sds2hmi_fi_tcl_e8_SDSDestinationType destinationType;

   switch (addressElementType)
   {
      case AddressElementType__COUNTRY:
      {
         destinationType.enType = sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_COUNTRY;
      }
      break;
      case AddressElementType__STATE:
      {
         destinationType.enType = sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_STATE;
      }
      break;
      case AddressElementType__PLACE:
      case AddressElementType__CITYDESTRICT:
      {
         destinationType.enType = sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_CITY;
      }
      break;
      case AddressElementType__ROAD:
      {
         destinationType.enType = sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_STREET;
      }
      break;
      case AddressElementType__HOUSENUMBER:
      {
         destinationType.enType = sds2hmi_fi_tcl_e8_SDSDestinationType::FI_EN_HOUSENO;
      }
      break;
      default:
      {
      }
      break;
   }
   return destinationType;
}
