/**************************************************************************//**
 * \file       clSDS_Method_MediaPlay.cpp
 *
 * clSDS_Method_MediaPlay method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Method_MediaPlay.h"
#include "Sds2HmiServer/functions/clSDS_Method_MediaGetDeviceInfo.h"
#include "application/GuiService.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_MediaPlay.cpp.trc.h"
#endif


#define DEVICETAG_INVALID 0


using namespace mplay_MediaPlayer_FI;
using namespace MPlay_fi_types;
using namespace sds_gui_fi::SdsGuiService;


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_Method_MediaPlay::~clSDS_Method_MediaPlay()
{
   _pMediaGetDeviceInfo = NULL;
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_Method_MediaPlay::clSDS_Method_MediaPlay(
   ahl_tclBaseOneThreadService* pService,
   ::boost::shared_ptr< Mplay_MediaPlayer_FIProxy > mediaPlayerProxy,
   GuiService& guiService,
   clSDS_Method_MediaGetDeviceInfo* pMediaGetDeviceInfo)

   : clServerMethod(SDS2HMI_SDSFI_C_U16_MEDIAPLAY, pService)
   , _mediaPlayerProxy(mediaPlayerProxy)
   , _guiService(guiService)
   , _pMediaGetDeviceInfo(pMediaGetDeviceInfo)
   , _oDeviceList()
   , _u32MediaId(0)
   , _e8DeviceType(T_e8_MPlayDeviceType__e8DTY_UNKNOWN)
   , _u8DeviceTag(DEVICETAG_INVALID)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_MediaPlay::vMethodStart(amt_tclServiceData* pInMessage)
{
   ETG_TRACE_USR1(("vMethodStart"));
   sds2hmi_sdsfi_tclMsgMediaPlayMethodStart oMessage;
   vGetDataFromAmt(pInMessage, oMessage);
   vProcessMessage(oMessage);
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_MediaPlay::vProcessMessage(const sds2hmi_sdsfi_tclMsgMediaPlayMethodStart& oMessage)
{
   ETG_TRACE_USR1(("vProcessMessage type :%d", oMessage.SelectionType.enType));
   switch (oMessage.SelectionType.enType)
   {
      case sds2hmi_fi_tcl_e8_GEN_SelectionType::FI_EN_BYVAL:
         vProcessMediaRequests(oMessage);
         break;

      case sds2hmi_fi_tcl_e8_GEN_SelectionType::FI_EN_BYTRACK:
         vCdPlayByTrack(oMessage);
         break;

      case sds2hmi_fi_tcl_e8_GEN_SelectionType::FI_EN_BYSOURCE:
         vPlaySource(oMessage.DriveID);
         break;

      case sds2hmi_fi_tcl_e8_GEN_SelectionType::FI_EN_BYPRESET:
         playAllSources(oMessage);
         break;

      default:
         vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
         break;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_MediaPlay::playAllSources(const sds2hmi_sdsfi_tclMsgMediaPlayMethodStart& oMessage)
{
   ETG_TRACE_USR1(("playAllSources %d", oMessage.Action.enType));
   // TODO jnd2hi: no result will be sent in case of
   // oMessage.MediaListType.enType != sds2hmi_fi_tcl_e8_MPL_ListType::FI_EN_LIST_TYPE_TITLES
   switch (oMessage.Action.enType)
   {
      case sds2hmi_fi_tcl_e8_PlayAction::FI_EN_SET:
         if ((_mediaPlayerProxy->hasActiveMediaDevice()) &&
               (_mediaPlayerProxy->getActiveMediaDevice().getBDeviceActiveSource() == true) &&
               (_mediaPlayerProxy->getActiveMediaDevice().getU8DeviceTag() != 0))
         {
            vSendMethodResult();
         }
         else
         {
            vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
         }
         break;

      case sds2hmi_fi_tcl_e8_PlayAction::FI_EN_PLAY:

         for (tU32 u32Index = 0; u32Index < _oDeviceList.size(); u32Index++)
         {
            if ((((_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB) && (_oDeviceList[u32Index].uiNoOfAudioFiles > 0))
                  || (_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD)
                  || (_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPHONE))
                  && (_oDeviceList[u32Index].bIsConnected == TRUE))
            {
               sendPlayMediaRequest(_oDeviceList[u32Index].uiDeviceTag, _oDeviceList[u32Index].uiDeviceType, _oDeviceList[u32Index].syspath);
               _mediaPlayerProxy->sendCreateMediaPlayerQuickPlayListStart(
                  *this,
                  ::MPlay_fi_types::T_e8_MPlayListType__e8LTY_SONG, 0, 0, 0, 0,
                  true, _oDeviceList[u32Index].uiDeviceTag);
               break;
            }
         }
         break;

      default:
         vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
         break;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_MediaPlay::vProcessMediaRequests(const sds2hmi_sdsfi_tclMsgMediaPlayMethodStart& oMessage)
{
   ETG_TRACE_USR1(("vProcessMediaRequests %d", oMessage.Action.enType));
   switch (oMessage.Action.enType)
   {
      case sds2hmi_fi_tcl_e8_PlayAction::FI_EN_SET:
         vSendMethodResult();
         break;

      case sds2hmi_fi_tcl_e8_PlayAction::FI_EN_PLAY:
         if (!oMessage.ValueIDs.empty())
         {
            ETG_TRACE_USR1(("vProcessMediaRequests !oMessage.ValueIDs.empty()"));
            _u32MediaId = oMessage.ValueIDs[0];
            _mediaPlayerProxy->sendGetMediaObjectStart(*this, _u32MediaId , ::MPlay_fi_types::T_e8_MPlayCategoryType__e8CTY_NONE);
         }
         else
         {
            mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsStatus mediastatus = _mediaPlayerProxy->getMediaPlayerDeviceConnections();
            ::MPlay_fi_types::T_MPlayDeviceInfo oDeviceInfo = mediastatus.getODeviceInfo();
            for (tU32 u32Index = 0; u32Index < oDeviceInfo.size(); u32Index++)
            {
               if (_oDeviceList[u32Index].uiDeviceTag == oMessage.DriveID)
               {
                  ETG_TRACE_USR1(("vProcessMediaRequests oMessage.DriveID : %d", oMessage.DriveID));
                  sendPlayMediaRequest(_oDeviceList[u32Index].uiDeviceTag, _oDeviceList[u32Index].uiDeviceType, _oDeviceList[u32Index].syspath);
                  vSendMethodResult();
               }
               else
               {
                  vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
               }
            }
            break;
         }
         ETG_TRACE_USR1(("vProcessMediaRequests _u32MediaId %d", _u32MediaId));
         _mediaPlayerProxy->sendPlayMediaPlayerObjectStart(*this, _u32MediaId, true);
         break;

      default:
         vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
         break;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_MediaPlay::vCdPlayByTrack(const sds2hmi_sdsfi_tclMsgMediaPlayMethodStart& oMessage)
{
   // first do a source change to CD
   enGetHmiSourceType(sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_CDA);
   tU32 u32TrackId = 0;
   if (oMessage.ValueIDs.size() > 0)
   {
      u32TrackId = oMessage.ValueIDs[0];
   }
   // play the track number
   if (_pMediaGetDeviceInfo != NULL && u32TrackId > 0)
   {
      unsigned int listHandle = _pMediaGetDeviceInfo->getListHandle();
      _mediaPlayerProxy->sendPlayItemFromListStart(*this, listHandle, (u32TrackId - 1), 0);
      vSendMethodResult();
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_MediaPlay::vPlaySource(tU32 u32SdsSourceId)
{
   ETG_TRACE_USR1(("vPlaySource %d", u32SdsSourceId));
   sds2hmi_fi_tcl_e8_MPL_SourceType::tenType sourcetype =  enGetHmiSourceType(u32SdsSourceId);

   if (sourcetype != sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_UNKNOWN)
   {
      vSendMethodResult();
   }
   else
   {
      ETG_TRACE_USR1(("vPlaySource vSendErrorMessage "));
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_Method_MediaPlay::bIsSupportedDVDDisc(tU32 u32Index) const
{
   if ((_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_DVD_DRIVE)
         && ((_oDeviceList[u32Index].discType == T_e8_MDiscType__e8AUDIO_CD_DISC)
             || (_oDeviceList[u32Index].discType == T_e8_MDiscType__e8DVD_AUDIO_DISC)
             || (_oDeviceList[u32Index].discType == T_e8_MDiscType__e8DVD_VIDEO_DISC)))
   {
      return true;
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
sds2hmi_fi_tcl_e8_MPL_SourceType::tenType clSDS_Method_MediaPlay::enGetHmiSourceType(tU32 u32SdsSourceId)
{
   ETG_TRACE_USR1(("enGetHmiSourceType "));
   sds2hmi_fi_tcl_e8_MPL_SourceType::tenType enSourceId = static_cast<sds2hmi_fi_tcl_e8_MPL_SourceType::tenType>(u32SdsSourceId);
   mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsStatus mediastatus = _mediaPlayerProxy->getMediaPlayerDeviceConnections();
   ::MPlay_fi_types::T_MPlayDeviceInfo oDeviceInfo = mediastatus.getODeviceInfo();
   ETG_TRACE_USR1(("enGetHmiSourceType %d", enSourceId));
   switch (enSourceId)
   {
      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_IPOD:
         for (tU32 u32Index = 0; u32Index < oDeviceInfo.size(); u32Index++)
         {
            if (((_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD) ||
                  (_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPHONE)) && (_oDeviceList[u32Index].bIsConnected == TRUE))
            {
               sendPlayMediaRequest(_oDeviceList[u32Index].uiDeviceTag, _oDeviceList[u32Index].uiDeviceType, _oDeviceList[u32Index].syspath);
               break;
            }
         }
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_IPOD;

      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_USB:
         for (tU32 u32Index = 0; u32Index < oDeviceInfo.size(); u32Index++)
         {
            if (((_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB) ||
                  (_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_MTP)) && (_oDeviceList[u32Index].bIsConnected == TRUE) && (_oDeviceList[u32Index].uiNoOfAudioFiles > 0))
            {
               sendPlayMediaRequest(_oDeviceList[u32Index].uiDeviceTag, _oDeviceList[u32Index].uiDeviceType, _oDeviceList[u32Index].syspath);
               break;
            }
         }
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_USB;

      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_CDA:
         for (tU32 u32Index = 0; u32Index < oDeviceInfo.size(); u32Index++)
         {
            if (((_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_CDDA)
                  || (_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_CDROM)
                  || (bIsSupportedDVDDisc(u32Index)))
                  && (_oDeviceList[u32Index].bIsConnected == TRUE))
            {
               sendPlayMediaRequest(_oDeviceList[u32Index].uiDeviceTag, _oDeviceList[u32Index].uiDeviceType, _oDeviceList[u32Index].syspath);
               break;
            }
         }
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_CDA;

      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_DVD:
         for (tU32 u32Index = 0; u32Index < oDeviceInfo.size(); u32Index++)
         {
            if ((bIsSupportedDVDDisc(u32Index)) && (_oDeviceList[u32Index].bIsConnected == TRUE))
            {
               sendPlayMediaRequest(_oDeviceList[u32Index].uiDeviceTag, _oDeviceList[u32Index].uiDeviceType, _oDeviceList[u32Index].syspath);
               break;
            }
         }
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_DVD;

      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_LINE_IN:
         //AUX source is always available, hence availability is not checked
         _guiService.sendActivateAudioSourceSignal(AudioSource__AUX);
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_LINE_IN;

      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_MUSIC_BOX:
         //MusicBox is always available for PIVI and valid for JAPAN region only
         for (tU32 u32Index = 0; u32Index < oDeviceInfo.size(); u32Index++)
         {
            if ((_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_MUSICBOX) && (_oDeviceList[u32Index].bIsConnected == TRUE))
            {
               sendPlayMediaRequest(_oDeviceList[u32Index].uiDeviceTag, _oDeviceList[u32Index].uiDeviceType, _oDeviceList[u32Index].syspath);
            }
         }
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_MUSIC_BOX;

      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_BTAUDIO:
         for (tU32 u32Index = 0; u32Index < oDeviceInfo.size(); u32Index++)
         {
            if ((_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH)
                  && (_oDeviceList[u32Index].bIsConnected == TRUE))
            {
               _mediaPlayerProxy->sendClearLastModePauseStart(*this, _oDeviceList[u32Index].uiDeviceTag);
               _guiService.sendActivateAudioSourceSignal(AudioSource__BT_AUDIO);
               break;
            }
         }
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_BTAUDIO;

      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_SDCARD:
         for (tU32 u32Index = 0; u32Index < oDeviceInfo.size(); u32Index++)
         {
            if ((_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_SD) && (_oDeviceList[u32Index].bIsConnected == TRUE))
            {
               sendPlayMediaRequest(_oDeviceList[u32Index].uiDeviceTag, _oDeviceList[u32Index].uiDeviceType, _oDeviceList[u32Index].syspath);
               break;
            }
         }
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_SDCARD;

      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_CARPLAY_AUD:
         _guiService.sendEventSignal(sds_gui_fi::SdsGuiService::Event__SPEECH_DIALOG_PLAY_CARPLAY_AUDIO);
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_CARPLAY_AUD;

      case sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_ANDROID_AUD:
         _guiService.sendEventSignal(sds_gui_fi::SdsGuiService::Event__SPEECH_DIALOG_PLAY_ANDROID_AUTO_AUDIO);
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_ANDROID_AUD;

      default:
         return sds2hmi_fi_tcl_e8_MPL_SourceType::FI_EN_SOURCE_TYPE_UNKNOWN;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_MediaPlay::vMediaPlayStatus()
{
   ETG_TRACE_USR1(("vMediaPlayStatus"));
   const bool result = enGetMediaAudioSource();
   if (result)
   {
      vSendMethodResult();
   }
   else
   {
      ETG_TRACE_USR1(("vMediaPlayStatus eror"));
      // TBD Error for Media Id not available should be defined by SDS
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_UNKNOWNERROR);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_MediaPlay::vMediaObjectValidationResponse(tBool bIsMediaObjectValid)
{
   if (bIsMediaObjectValid == TRUE)
   {
      vSendMethodResult();
   }
   else
   {
      ETG_TRACE_USR1(("vMediaObjectValidationResponse error "));
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_MEDIAOBJNOTAVAILABLE);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_Method_MediaPlay::enGetMediaAudioSource()
{
   ETG_TRACE_USR1(("enGetMediaAudioSource %d", _e8DeviceType));
   switch (_e8DeviceType)
   {
      case T_e8_MPlayDeviceType__e8DTY_USB:
      case T_e8_MPlayDeviceType__e8DTY_MTP:
         for (tU32 u32Index = 0; u32Index < _oDeviceList.size(); u32Index++)
         {
            if (((_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB) ||
                  (_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_MTP)) &&
                  (_oDeviceList[u32Index].uiDeviceTag == _u8DeviceTag) &&
                  (_oDeviceList[u32Index].bIsConnected == TRUE))
            {
               ETG_TRACE_USR1(("enGetMediaAudioSource sendActivateAudioDeviceTagSignal :%d", (int)_oDeviceList[u32Index].uiDeviceTag));
               if (isUSB1connected(_oDeviceList[u32Index].syspath))
               {
                  _guiService.sendActivateAudioDeviceTagSignal((int)_oDeviceList[u32Index].uiDeviceTag, AudioSource__USB1);
               }
               else
               {
                  _guiService.sendActivateAudioDeviceTagSignal((int)_oDeviceList[u32Index].uiDeviceTag, AudioSource__USB2);
               }

               break;
            }
         }
         break;

      case T_e8_MPlayDeviceType__e8DTY_IPOD:
      case T_e8_MPlayDeviceType__e8DTY_IPHONE:
         for (tU32 u32Index = 0; u32Index < _oDeviceList.size(); u32Index++)
         {
            if (((_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD) ||
                  (_oDeviceList[u32Index].uiDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPHONE)) &&
                  (_oDeviceList[u32Index].bIsConnected == TRUE))
            {
               ETG_TRACE_USR1(("enGetMediaAudioSource sendActivateAudioDeviceTagSignal :%d", (int)_oDeviceList[u32Index].uiDeviceTag));
               _guiService.sendActivateAudioDeviceTagSignal((int)_oDeviceList[u32Index].uiDeviceTag, AudioSource__IPOD);
               break;
            }
         }
         break;

      default:
         return false;
   }

   return true;
}


tBool clSDS_Method_MediaPlay::isUSB1connected(std::string& sysPath) const
{
   ETG_TRACE_USR4(("isUSB1connected Mount Point is %s", sysPath.c_str()));
   tBool status = TRUE;
   ::std::size_t found = ::std::string::npos;
   if (!sysPath.empty())
   {
      found = sysPath.find_last_of(".");
   }
   if (found != ::std::string::npos)
   {
      ETG_TRACE_USR4(("isUSB1connected PortNumber Substring is %s", sysPath.substr(found + 1).c_str()));
      int portNumber = atoi(sysPath.substr(found + 1).c_str());
      ETG_TRACE_USR4(("isUSB1connected PortNumber is %d", portNumber));
      if (portNumber <= 0) //If device connected Internally(Direct Connection), No External Hub Connection
      {
         status = TRUE;
      }
      else
      {
         if (portNumber % 2)
         {
            status = FALSE; //For Reno only- Device Connected in EVEN Port Number (2nd USB Port number is returned as 3)
         }
         else
         {
            status = TRUE; //For Reno only- Device Connected in ODD Port Number (1st USB Port number is returned as 2)
         }
      }
   }
   return status;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_MediaPlay::sendPlayMediaRequest(unsigned int deviceTag, unsigned int deviceType, std::string syspath)
{
   ETG_TRACE_USR1(("sendPlayMediaRequest sendActivateAudioDeviceTagSignal :%d", (int)deviceTag));
   _mediaPlayerProxy->sendClearLastModePauseStart(*this, deviceTag);
   switch (deviceType)
   {
      case T_e8_MPlayDeviceType__e8DTY_USB:
      case T_e8_MPlayDeviceType__e8DTY_MTP:
      {
         if (isUSB1connected(syspath))
         {
            _guiService.sendActivateAudioDeviceTagSignal((int)deviceTag, AudioSource__USB1);
         }
         else
         {
            _guiService.sendActivateAudioDeviceTagSignal((int)deviceTag, AudioSource__USB2);
         }
         break;
      }

      case T_e8_MPlayDeviceType__e8DTY_IPOD:
      case T_e8_MPlayDeviceType__e8DTY_IPHONE:
         _guiService.sendActivateAudioDeviceTagSignal((int)deviceTag, AudioSource__IPOD);
         break;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_MediaPlay::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("onAvailable"));
   if (proxy == _mediaPlayerProxy)
   {
      _mediaPlayerProxy->sendMediaPlayerDeviceConnectionsUpReg(*this);
      _mediaPlayerProxy->sendActiveMediaDeviceUpReg(*this);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_MediaPlay::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("onUnavailable"));
   if (proxy == _mediaPlayerProxy)
   {
      _mediaPlayerProxy->sendMediaPlayerDeviceConnectionsRelUpRegAll();
      _mediaPlayerProxy->sendActiveMediaDeviceRelUpRegAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/

void clSDS_Method_MediaPlay::onActiveMediaDeviceError(
   const ::boost::shared_ptr< Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< ActiveMediaDeviceError >& /*error*/)
{
   ETG_TRACE_USR1(("onActiveMediaDeviceError"));
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_MediaPlay:: onActiveMediaDeviceStatus(
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::ActiveMediaDeviceStatus >& /*status*/)
{
   ETG_TRACE_USR1(("onActiveMediaDeviceStatus"));
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_MediaPlay:: onMediaPlayerDeviceConnectionsError(
   const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const boost::shared_ptr< ::mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsError >& /*error*/)
{
   ETG_TRACE_USR1(("onMediaPlayerDeviceConnectionsError"));
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_MediaPlay::onMediaPlayerDeviceConnectionsStatus(
   const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const boost::shared_ptr< ::mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsStatus >& status)
{
   _oDeviceList.clear();
   for (size_t i = 0 ; i < status->getODeviceInfo().size(); i++)
   {
      const MPlay_fi_types::T_MPlayDeviceInfoItem& item = status->getODeviceInfo()[i];
      trDevice deviceItem;
      deviceItem.bIsConnected = item.getBDeviceConnected();
      deviceItem.uiDeviceTag = item.getU8DeviceTag();
      deviceItem.uiDeviceType = item.getE8DeviceType();
      deviceItem.uiNoOfAudioFiles = item.getU32NumberOfFiles() - (item.getU32NumberOfImageFiles() + item.getU32NumberOfVideoFiles());
      deviceItem.discType = item.getE8DiscType();
      deviceItem.syspath =  item.getSDeviceSysPath();
      _oDeviceList.push_back(deviceItem);
      ETG_TRACE_USR1(("bIsConnected  %d uiDeviceTag %d  uiDeviceType %d uiNoOfAudioFiles %d",
                      _oDeviceList[i].bIsConnected, _oDeviceList[i].uiDeviceTag, _oDeviceList[i].uiDeviceType, _oDeviceList[i].uiNoOfAudioFiles));
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_MediaPlay::onPlayMediaPlayerObjectError(
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::PlayMediaPlayerObjectError >& /*error*/)
{
   ETG_TRACE_USR1(("onPlayMediaPlayerObjectError"));
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_MEDIAOBJNOTAVAILABLE);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_MediaPlay::onPlayMediaPlayerObjectResult(
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::PlayMediaPlayerObjectResult >& /*result*/)
{
   ETG_TRACE_USR1(("onPlayMediaPlayerObjectResult"));
   //vMediaPlayStatus();
   vSendMethodResult();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_MediaPlay::onPlayItemFromListError(
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::PlayItemFromListError >& /*error*/)
{
   ETG_TRACE_USR1(("onPlayItemFromListError"));
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_MediaPlay::onPlayItemFromListResult(
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::PlayItemFromListResult >& /*result*/)
{
   ETG_TRACE_USR1(("onPlayItemFromListResult"));
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_MediaPlay::onGetMediaObjectError(const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< mplay_MediaPlayer_FI::GetMediaObjectError >& /*error*/)
{
   ETG_TRACE_USR1(("onGetMediaObjectError"));
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_MediaPlay::onGetMediaObjectResult(const ::boost::shared_ptr<mplay_MediaPlayer_FI:: Mplay_MediaPlayer_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< mplay_MediaPlayer_FI::GetMediaObjectResult >& result)
{
   const T_MPlayMediaObject& oMediaObject = result->getOMediaObject();
   _e8DeviceType = oMediaObject.getE8DeviceType();
   _u8DeviceTag = oMediaObject.getU8DeviceTag();
   vMediaPlayStatus();
   ETG_TRACE_USR1(("onGetMediaObjectResult _e8DeviceType :%d", _e8DeviceType));
   ETG_TRACE_USR1(("onGetMediaObjectResult _u8DeviceTag :%d", _u8DeviceTag));
}


/**************************************************************************//**
 *
 ******************************************************************************/

void clSDS_Method_MediaPlay::onCreateMediaPlayerQuickPlayListError(
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::CreateMediaPlayerQuickPlayListError >& /*error*/)
{
   ETG_TRACE_USR1(("onCreateMediaPlayerQuickPlayListError"));
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


/**************************************************************************//**
 *
 ******************************************************************************/

void clSDS_Method_MediaPlay::onCreateMediaPlayerQuickPlayListResult(
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::CreateMediaPlayerQuickPlayListResult >& /*result*/)
{
   ETG_TRACE_USR1(("onCreateMediaPlayerQuickPlayListResult"));
   vSendMethodResult();
}


/**************************************************************************//**
 *
 ******************************************************************************/

void clSDS_Method_MediaPlay::onClearLastModePauseError(
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::ClearLastModePauseError >& /*error*/)
{
   ETG_TRACE_USR1(("onClearLastModePauseError"));
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


/**************************************************************************//**
 *
 ******************************************************************************/

void clSDS_Method_MediaPlay::onClearLastModePauseResult(
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::ClearLastModePauseResult >& /*result*/)
{
   ETG_TRACE_USR1(("onClearLastModePauseResult"));
   vSendMethodResult();
}
