/**************************************************************************//**
 * \file       clSDS_Method_ContactsGetAmbiguityList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/

#include "Sds2HmiServer/functions/clSDS_Method_ContactsGetAmbiguityList.h"
#include "application/clSDS_AmbigContactList.h"
#include "application/clSDS_AmbigNumberList.h"
#include "application/clSDS_PhoneNumberFormatter.h"
#include "application/clSDS_StringVarList.h"
#include "application/StringUtils.h"
#include "SdsAdapter_Trace.h"
#include "application/PhoneUtils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_ContactsGetAmbiguityList.cpp.trc.h"
#endif

using namespace MOST_PhonBk_FI;
using namespace most_PhonBk_fi_types;
using namespace most_PhonBk_fi_types_Extended;


clSDS_Method_ContactsGetAmbiguityList::clSDS_Method_ContactsGetAmbiguityList(ahl_tclBaseOneThreadService* pService
      , ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy> phoneBookProxy
      , clSDS_AmbigContactList* ambigContactList
      , clSDS_AmbigNumberList* ambigNumberList)
   : clServerMethod(SDS2HMI_SDSFI_C_U16_CONTACTSGETAMBIGUITYLIST, pService)
   , _phoneBookProxy(phoneBookProxy)
   , _pAmbigContactList(ambigContactList)
   , _pAmbigNumberList(ambigNumberList)
{
   _locationType.enType = sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_UNKNOWN;
   _requireAllLocationIfNoExactMatch = false;
}


clSDS_Method_ContactsGetAmbiguityList::~clSDS_Method_ContactsGetAmbiguityList()
{
   _pAmbigContactList = NULL;
   _pAmbigNumberList = NULL;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_ContactsGetAmbiguityList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _phoneBookProxy)
   {
      _phoneBookProxy->sendPreferredPhoneBookSortOrderUpReg(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_ContactsGetAmbiguityList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _phoneBookProxy)
   {
      _phoneBookProxy->sendPreferredPhoneBookSortOrderRelUpRegAll();
   }
}


void clSDS_Method_ContactsGetAmbiguityList::vMethodStart(amt_tclServiceData* pInMsg)
{
   ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::vMethodStart"));
   sds2hmi_sdsfi_tclMsgContactsGetAmbiguityListMethodStart oMessage;
   vGetDataFromAmt(pInMsg, oMessage);

   _contactIDList.clear();
   _contactDetailList.clear();
   ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::type %d ", oMessage.EntryType.enType));

   if (sds2hmi_fi_tcl_e8_CON_EntryType::FI_EN_PHONE_NUMBER == oMessage.EntryType.enType)
   {
      _contactIDList = oMessage.ContactIDs;
      _locationType = oMessage.LocationType;
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::location type  %d ", _locationType.enType));

      _requireAllLocationIfNoExactMatch = (oMessage.RequireAllLocationIfNoExactMatch == TRUE);

      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::vMethodStart - _contactIDList.size() = %d", _contactIDList.size()));

      if (_contactIDList.size() > 0)
      {
         // Name disambiguation (_contactIDList.size() > 1) -> AMBIG_CONTACT
         // Number disambiguation (_contactIDList.size() == 1)-> AMBIG_ALL_ENTRIES / SINGLE_ENTRY / NO_ENTRY

         resolveAmbiguity();
      }
      else // contacts.size() == 0
      {
         vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
      }
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_ENTRYTYPENOTSUPPORTED);
   }
}


void clSDS_Method_ContactsGetAmbiguityList::resolveAmbiguity()
{
   if (_phoneBookProxy->isAvailable())
   {
      for (tU32 i = 0; i < _contactIDList.size(); i++) // only one iteration for number disambiguation
      {
         ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::resolveAmbiguity - ContactID = %d", _contactIDList[i]));

         _phoneBookProxy->sendGetContactDetailsExtendedStart(*this, _contactIDList[i]);
      }
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


void clSDS_Method_ContactsGetAmbiguityList::onGetContactDetailsExtendedError(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetContactDetailsExtendedError >& /*error*/)
{
   sds2hmi_sdsfi_tclMsgContactsGetAmbiguityListMethodResult methodResult;
   methodResult.ResultType.enType = sds2hmi_fi_tcl_e8_CON_AmbiguityListType::FI_EN_CONTACT_UNAVAILABLE;
   vSendMethodResult(methodResult);
}


void clSDS_Method_ContactsGetAmbiguityList::onGetContactDetailsExtendedResult(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetContactDetailsExtendedResult >& result)
{
   T_PhonBkContactDetailsExtended oContactDetails = result->getOContactDetailsExtended();
   _contactDetailList.push_back(oContactDetails);

   if (_contactIDList.size() == 1) // case of NUMBER disambiguation -> send method result with all phone numbers
   {
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::onGetContactDetailsResult - fullName = %s", getContactName(oContactDetails).c_str()));

      sds2hmi_sdsfi_tclMsgContactsGetAmbiguityListMethodResult methodResult;
      setResultEntriesAndType(_contactIDList[0], oContactDetails, methodResult);
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::onGetContactDetailsResult - results count = %d", methodResult.ResultEntries.size()));

      if (!getContactName(oContactDetails).empty())
      {
         clSDS_StringVarList::vSetVariable("$(ContactName)", getContactName(oContactDetails));
      }
      else
      {
         clSDS_StringVarList::vSetVariable("$(ContactName)", PhoneUtils::getPhoneNumber(_contactDetailList[0]));
      }
      clSDS_StringVarList::vSetVariable("$(Number)", PhoneUtils::getPhoneNumber(_contactDetailList[0]));

      vSendMethodResult(methodResult);
   }
   else // case of NAME disambiguation
   {
      if (_contactIDList.size() == _contactDetailList.size()) // we have received details for all contact ids
      {
         sds2hmi_sdsfi_tclMsgContactsGetAmbiguityListMethodResult methodResult;
         methodResult.ResultType.enType = sds2hmi_fi_tcl_e8_CON_AmbiguityListType::FI_EN_AMBIG_CONTACT;
         ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::onGetContactDetailsResult - _contactIDList.size() = %d", _contactIDList.size()));

         for (tU32 i = 0; i < _contactIDList.size(); i++)
         {
            sds2hmi_fi_tcl_CON_AmbiguityResultEntry resultEntry;
            resultEntry.LocationType = _locationType;
            resultEntry.ContactID = _contactIDList[i];
            methodResult.ResultEntries.push_back(resultEntry);
         }

         setAmbigContacts(_contactDetailList);

         vSendMethodResult(methodResult);
      }
      else
      {
         // still case of name disambiguation, but we have not received details for all contact ids
         // so, we have to wait for the details from the other remaining contacts
         // and only after that send the response with method result
      }
   }
}


void clSDS_Method_ContactsGetAmbiguityList::onPreferredPhoneBookSortOrderError(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::PreferredPhoneBookSortOrderError >& /*error*/)
{
}


void clSDS_Method_ContactsGetAmbiguityList::onPreferredPhoneBookSortOrderStatus(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::PreferredPhoneBookSortOrderStatus >& /*status*/)
{
}


void clSDS_Method_ContactsGetAmbiguityList::setResultEntriesAndType(tU32 contactID, const T_PhonBkContactDetailsExtended& contactDetails, sds2hmi_sdsfi_tclMsgContactsGetAmbiguityListMethodResult& methodResult)
{
   ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::setResultEntriesAndType - locationType = %d", _locationType.enType));

   switch (_locationType.enType)
   {
      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_HOME1:
         extractResultEntryByPhoneNumberType(contactID, contactDetails, T_e8_PhonBkNumberType__HOME, methodResult);
         break;

      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_OFFICE1:
         extractResultEntryByPhoneNumberType(contactID, contactDetails, T_e8_PhonBkNumberType__WORK, methodResult);
         break;

      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_MOBILE1:
         extractResultEntryByPhoneNumberType(contactID, contactDetails, T_e8_PhonBkNumberType__CELL, methodResult);
         break;

      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_OTHER:
         extractResultEntryByPhoneNumberType(contactID, contactDetails, T_e8_PhonBkNumberType__OTHER, methodResult);
         break;

      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_MAIN:
         extractResultEntryByPhoneNumberType(contactID, contactDetails, T_e8_PhonBkNumberType__PREFERRED, methodResult);
         break;

      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_UNKNOWN:
         extractResultEntriesByAllPhoneNumberType(contactID, contactDetails, methodResult);
         break;

      default:
         break;
   }

   setResultType(methodResult);

   if (methodResult.ResultEntries.size() == 0  && _requireAllLocationIfNoExactMatch)
   {
      extractResultEntriesByAllPhoneNumberType(contactID, contactDetails, methodResult);

      if (methodResult.ResultEntries.size() > 0)
      {
         methodResult.ResultType.enType = sds2hmi_fi_tcl_e8_CON_AmbiguityListType::FI_EN_NO_MATCHING_LOCATION;

         setAmbigNumbers(methodResult.ResultEntries);
      }
   }
}


void clSDS_Method_ContactsGetAmbiguityList::setResultType(sds2hmi_sdsfi_tclMsgContactsGetAmbiguityListMethodResult& methodResult)
{
   ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::setResultType - methodResult.ResultEntries.size() = %d", methodResult.ResultEntries.size()));
   if (methodResult.ResultEntries.size() == 1)
   {
      methodResult.ResultType.enType = sds2hmi_fi_tcl_e8_CON_AmbiguityListType::FI_EN_SINGLE_ENTRY;
   }
   else if (methodResult.ResultEntries.size() > 1)
   {
      switch (_locationType.enType)
      {
         case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_HOME1:
         case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_OFFICE1:
         case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_MOBILE1:
         case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_OTHER:
         case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_MAIN:
            methodResult.ResultType.enType = sds2hmi_fi_tcl_e8_CON_AmbiguityListType::FI_EN_AMBIG_ENTRY;
            break;
         case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_UNKNOWN:
            methodResult.ResultType.enType = sds2hmi_fi_tcl_e8_CON_AmbiguityListType::FI_EN_AMBIG_LOCATION;
            break;
         default:
            break;
      }
   }
   else //methodResult.ResultEntries.size() == 0
   {
      methodResult.ResultType.enType = sds2hmi_fi_tcl_e8_CON_AmbiguityListType::FI_EN_NO_ENTRY;
   }
   setAmbigNumbers(methodResult.ResultEntries);
}


void clSDS_Method_ContactsGetAmbiguityList::setAmbigContacts(const std::vector<T_PhonBkContactDetailsExtended >& contactDetailList)
{
   ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::setAmbigContacts"));
   std::vector<std::string> contactInfo;

   for (tU32 i = 0; i < contactDetailList.size(); i++)
   {
      std::string contactName = getContactName(contactDetailList[i]);
      if (!contactName.empty())
      {
         contactInfo.push_back(contactName);
      }
      else
      {
         contactInfo.push_back(PhoneUtils::getPhoneNumber(contactDetailList[i]));
      }
   }

   _pAmbigContactList->setAmbigContactList(contactInfo);
}


void clSDS_Method_ContactsGetAmbiguityList::setAmbigNumbers(const std::vector<sds2hmi_fi_tcl_CON_AmbiguityResultEntry>& resultEntries)
{
   ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::setAmbigNumbers"));
   std::vector<std::pair<std::string, unsigned int> > contactNumbers;
   std::pair<std::string, unsigned int> contactNumber;
   for (tU32 i = 0; i < resultEntries.size(); i++)
   {
      contactNumber.first = clSDS_PhoneNumberFormatter::oFormatNumber(resultEntries[i].DisplayString.szValue);
      contactNumber.second = resultEntries[i].LocationType.enType;
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::setAmbigNumbers - contactNumber.first = %s", contactNumber.first));
      addNewContact(contactNumbers, contactNumber);
   }
   _pAmbigNumberList->setAmbigNumberList(contactNumbers);
}


std::string clSDS_Method_ContactsGetAmbiguityList::getContactName(const T_PhonBkContactDetailsExtended& contactDetails) const
{
   ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::getContactName"));
   std::string fullName;

   if (_phoneBookProxy->hasPreferredPhoneBookSortOrder()
         && _phoneBookProxy->getPreferredPhoneBookSortOrder().getE8PreferredPhoneBookSortOrder() == T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_LASTNAME)
   {
      fullName = contactDetails.getSLastName() + " " + contactDetails.getSFirstName();
   }
   else
   {
      fullName = contactDetails.getSFirstName() + " " + contactDetails.getSLastName();
   }

   return StringUtils::trim(fullName);
}


static void addToResultEntries(const std::string& number, tU32 contactID, sds2hmi_fi_tcl_e8_PHN_NumberType::tenType type, sds2hmi_sdsfi_tclMsgContactsGetAmbiguityListMethodResult& result)
{
   ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::addToResultEntries"));
   if (!number.empty())
   {
      sds2hmi_fi_tcl_CON_AmbiguityResultEntry resultEntry;
      resultEntry.LocationType.enType = type;
      resultEntry.DisplayString = number.c_str();
      resultEntry.ContactID = contactID;
      result.ResultEntries.push_back(resultEntry);
   }
}


void clSDS_Method_ContactsGetAmbiguityList::extractResultEntriesByAllPhoneNumberType(tU32 contactID, const T_PhonBkContactDetailsExtended& contactDetails, sds2hmi_sdsfi_tclMsgContactsGetAmbiguityListMethodResult& result) const
{
   extractAvailablePhoneNumberTypes(contactID, contactDetails, T_e8_PhonBkNumberType__HOME, result, "Home <number>");
   extractAvailablePhoneNumberTypes(contactID, contactDetails, T_e8_PhonBkNumberType__WORK, result, "Work <number>");
   extractAvailablePhoneNumberTypes(contactID, contactDetails, T_e8_PhonBkNumberType__CELL, result, "Mobile <number>");
   extractAvailablePhoneNumberTypes(contactID, contactDetails, T_e8_PhonBkNumberType__PREFERRED, result, "Other <number>");
   extractAvailablePhoneNumberTypes(contactID, contactDetails, T_e8_PhonBkNumberType__OTHER,  result, "Other <number>");
   extractAvailablePhoneNumberTypes(contactID, contactDetails, T_e8_PhonBkNumberType__NONE, result, "Other <number>");
}


void clSDS_Method_ContactsGetAmbiguityList::extractAvailablePhoneNumberTypes(tU32 contactID,
      const T_PhonBkContactDetailsExtended& contactDetails,
      T_e8_PhonBkNumberType phoneBkFiType,
      sds2hmi_sdsfi_tclMsgContactsGetAmbiguityListMethodResult& result,
      const std::string displayString) const
{
   ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::extractAvailablePhoneNumberTypes"));
   sds2hmi_fi_tcl_e8_PHN_NumberType::tenType locationType = PhoneUtils::mapToSds2hmiType(phoneBkFiType);
   if ((!contactDetails.getOPhoneNumber1().getSNumber().empty() && (contactDetails.getOPhoneNumber1().getE8Type() == phoneBkFiType)) ||
         (!contactDetails.getOPhoneNumber2().getSNumber().empty() && (contactDetails.getOPhoneNumber2().getE8Type() == phoneBkFiType)) ||
         (!contactDetails.getOPhoneNumber3().getSNumber().empty() && (contactDetails.getOPhoneNumber3().getE8Type() == phoneBkFiType)) ||
         (!contactDetails.getOPhoneNumber4().getSNumber().empty() && (contactDetails.getOPhoneNumber4().getE8Type() == phoneBkFiType)) ||
         (!contactDetails.getOPhoneNumber5().getSNumber().empty() && (contactDetails.getOPhoneNumber5().getE8Type() == phoneBkFiType)) ||
         (!contactDetails.getOPhoneNumber6().getSNumber().empty() && (contactDetails.getOPhoneNumber6().getE8Type() == phoneBkFiType)) ||
         (!contactDetails.getOPhoneNumber7().getSNumber().empty() && (contactDetails.getOPhoneNumber7().getE8Type() == phoneBkFiType)) ||
         (!contactDetails.getOPhoneNumber8().getSNumber().empty() && (contactDetails.getOPhoneNumber8().getE8Type() == phoneBkFiType)))
   {
      addToResultEntries(displayString, contactID, locationType, result);
   }
}


void clSDS_Method_ContactsGetAmbiguityList::extractResultEntryByPhoneNumberType(tU32 contactID,
      const T_PhonBkContactDetailsExtended& contactDetails,
      T_e8_PhonBkNumberType phoneBkFiType,
      sds2hmi_sdsfi_tclMsgContactsGetAmbiguityListMethodResult& result) const
{
   ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::extractResultEntryByPhoneNumberType"));
   sds2hmi_fi_tcl_e8_PHN_NumberType::tenType locationType = PhoneUtils::mapToSds2hmiType(phoneBkFiType);
   if (!contactDetails.getOPhoneNumber1().getSNumber().empty() && (contactDetails.getOPhoneNumber1().getE8Type() == phoneBkFiType))
   {
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::extractResultEntryByPhoneNumberType - locationType = %d, phoneBkFiType = %d", locationType, phoneBkFiType));
      addToResultEntries(contactDetails.getOPhoneNumber1().getSNumber(), contactID, locationType, result);
   }
   if (!contactDetails.getOPhoneNumber2().getSNumber().empty() && (contactDetails.getOPhoneNumber2().getE8Type() == phoneBkFiType))
   {
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::extractResultEntryByPhoneNumberType - locationType = %d, phoneBkFiType = %d", locationType, phoneBkFiType));
      addToResultEntries(contactDetails.getOPhoneNumber2().getSNumber(), contactID, locationType, result);
   }
   if (!contactDetails.getOPhoneNumber3().getSNumber().empty() && (contactDetails.getOPhoneNumber3().getE8Type() == phoneBkFiType))
   {
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::extractResultEntryByPhoneNumberType - locationType = %d, phoneBkFiType = %d", locationType, phoneBkFiType));
      addToResultEntries(contactDetails.getOPhoneNumber3().getSNumber(), contactID, locationType, result);
   }
   if (!contactDetails.getOPhoneNumber4().getSNumber().empty() && (contactDetails.getOPhoneNumber4().getE8Type() == phoneBkFiType))
   {
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::extractResultEntryByPhoneNumberType - locationType = %d, phoneBkFiType = %d", locationType, phoneBkFiType));
      addToResultEntries(contactDetails.getOPhoneNumber4().getSNumber(), contactID, locationType, result);
   }
   if (!contactDetails.getOPhoneNumber5().getSNumber().empty() && (contactDetails.getOPhoneNumber5().getE8Type() == phoneBkFiType))
   {
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::extractResultEntryByPhoneNumberType - locationType = %d, phoneBkFiType = %d", locationType, phoneBkFiType));
      addToResultEntries(contactDetails.getOPhoneNumber5().getSNumber(), contactID, locationType, result);
   }
   if (!contactDetails.getOPhoneNumber6().getSNumber().empty() && (contactDetails.getOPhoneNumber6().getE8Type() == phoneBkFiType))
   {
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::extractResultEntryByPhoneNumberType - locationType = %d, phoneBkFiType = %d", locationType, phoneBkFiType));
      addToResultEntries(contactDetails.getOPhoneNumber6().getSNumber(), contactID, locationType, result);
   }
   if (!contactDetails.getOPhoneNumber7().getSNumber().empty() && (contactDetails.getOPhoneNumber7().getE8Type() == phoneBkFiType))
   {
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::extractResultEntryByPhoneNumberType - locationType = %d, phoneBkFiType = %d", locationType, phoneBkFiType));
      addToResultEntries(contactDetails.getOPhoneNumber7().getSNumber(), contactID, locationType, result);
   }
   if (!contactDetails.getOPhoneNumber8().getSNumber().empty() && (contactDetails.getOPhoneNumber8().getE8Type() == phoneBkFiType))
   {
      ETG_TRACE_USR4(("clSDS_Method_ContactsGetAmbiguityList::extractResultEntryByPhoneNumberType - locationType = %d, phoneBkFiType = %d", locationType, phoneBkFiType));
      addToResultEntries(contactDetails.getOPhoneNumber8().getSNumber(), contactID, locationType, result);
   }
}


void clSDS_Method_ContactsGetAmbiguityList::addNewContact(std::vector<std::pair <std::string, unsigned int> >& contactNumberDetails, std::pair <std::string, unsigned int>& contactNumber)
{
   bool duplicateExist = false;
   for (size_t i = 0; i < contactNumberDetails.size(); i++)
   {
      if ((contactNumberDetails[i].second == sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_UNKNOWN) && (contactNumberDetails[i].first == contactNumber.first))
      {
         duplicateExist = true;
         break;
      }
   }
   if (!duplicateExist)
   {
      contactNumberDetails.push_back(contactNumber);
   }
}
