/**************************************************************************//**
 * \file       clSDS_Method_CommonShowDialog.cpp
 *
 * clSDS_Method_CommonShowDialog method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Method_CommonShowDialog.h"
#include "application/clSDS_MenuManager.h"
#include "application/clSDS_ListScreen.h"
#include "application/clSDS_ScreenData.h"
#include "application/clSDS_View.h"
#include "application/GuiService.h"
#include "application/SdsPhoneService.h"
#include "application/clSDS_ConfigurationFlags.h"
#include "application/clSDS_ActiveScreenObserver.h"
#include "Sds2HmiServer/functions/clSDS_Property_PhoneStatus.h"
#include "SdsAdapter_Trace.h"
#include "application/StringUtils.h"
#include "view_db/Sds_ViewDB.h"
#include "application/VariantLayoutMap.h"

#define ETG_ENABLED
#include "trace_interface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX      "SdsAdapter_"
#define ETG_I_TRACE_CHANNEL         TR_TTFIS_SAAL
#define ETG_DEFAULT_TRACE_CLASS     TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_CommonShowDialog.cpp.trc.h"
#endif


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_Method_CommonShowDialog::~clSDS_Method_CommonShowDialog()
{
   _pListScreen = NULL;
   _pMenuManager = NULL;
   _pGuiService = NULL;
   _pSdsPhoneService = NULL;
   _pProperty_PhoneStatus = NULL;
   _pActiveScreenObserver = NULL;
   _pDisplay = NULL;
   _pVariantLayoutMap = NULL;
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_Method_CommonShowDialog::clSDS_Method_CommonShowDialog(
   ahl_tclBaseOneThreadService* pService,
   clSDS_MenuManager* pMenuManager,
   clSDS_ListScreen* pListScreen,
   clSDS_Property_PhoneStatus* pProperty_PhoneStatus,
   GuiService* pGuiService,
   SdsPhoneService* pSdsPhoneService,
   clSDS_Display* pDisplay,
   VariantLayoutMap* pVariantLayoutMap)

   : clServerMethod(SDS2HMI_SDSFI_C_U16_COMMONSHOWDIALOG, pService)
   , _pMenuManager(pMenuManager)
   , _pGuiService(pGuiService)
   , _pSdsPhoneService(pSdsPhoneService)
   , _pListScreen(pListScreen)
   , _pProperty_PhoneStatus(pProperty_PhoneStatus)
   , _pActiveScreenObserver(NULL)
   , _pDisplay(pDisplay)
   , _pVariantLayoutMap(pVariantLayoutMap)
   , _commandPageNumber(0)
   , _maxItemsPerPage(1)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_CommonShowDialog::vMethodStart(amt_tclServiceData* pInMessage)
{
   sds2hmi_sdsfi_tclMsgCommonShowDialogMethodStart oMessage;
   vGetDataFromAmt(pInMessage, oMessage);
   vTraceXmlData(oMessage.sXMLStream.szValue);
   clSDS_ScreenData oScreenData(oMessage.sXMLStream.szValue);
   _viewId = oScreenData.readScreenId();
   if (_pVariantLayoutMap != NULL)
   {
      ETG_TRACE_USR4(("clSDS_Method_CommonShowDialog::vMethodStart _viewId =  %s ", _viewId.c_str()));
      ETG_TRACE_USR4(("clSDS_Method_CommonShowDialog::vMethodStart Sds_ViewDB_getLayout(%s)", (Sds_ViewDB_getLayout(_viewId)).c_str()));
      ETG_TRACE_USR4(("clSDS_Method_CommonShowDialog::vMethodStart _maxItemsPerPage %d ", _maxItemsPerPage));
      _maxItemsPerPage = _pVariantLayoutMap->getItemsPerPage(Sds_ViewDB_getLayout(_viewId));
   }
   if (_pDisplay  &&
         ((oMessage.nPopUp_UpDateType.enType == sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_ALL) ||
          (oMessage.nPopUp_UpDateType.enType == sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_RESTORE_PREVIOUS) ||
          (oMessage.nPopUp_UpDateType.enType == sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_TOLASTPAGE)))
   {
      if (oScreenData.oGetScreenVariableData().size() > 1)
      {
         _pDisplay->setScreenVariableData(oScreenData);
      }
      if (_pDisplay  &&
            ((oMessage.nPopUp_UpDateType.enType == sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_ALL) ||
             (oMessage.nPopUp_UpDateType.enType == sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_RESTORE_PREVIOUS)))
      {
         _pDisplay->setSpeakableList(oScreenData.getIsSpeakableList());
      }
   }

   if (_pActiveScreenObserver != NULL)
   {
      _pActiveScreenObserver->viewIdChanged(_viewId);
   }
   if (_pListScreen)
   {
      _pListScreen->vSetCurrentList(_viewId);
   }
   vUpdatePageNumber(oMessage.nPopUp_UpDateType);

   if (_pListScreen && _pListScreen->bIsListScreen(_viewId))
   {
      _pListScreen->setMaxItemsPerPage(_maxItemsPerPage);
      ETG_TRACE_USR4(("clSDS_Method_CommonShowDialog::vMethodStart _maxItemsPerPage %d ", _maxItemsPerPage));
      std::vector<clSDS_ListItems> oListContents = _pListScreen->vGetListScreenContents(oScreenData, oMessage.nPopUp_UpDateType);
      oScreenData.vStoreListContents(oListContents);
   }
   else if (_pDisplay)
   {
      switch (oMessage.nPopUp_UpDateType.enType)
      {
         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_ALL:
         {
            std::map<std::string, std::string> screenVariableData = _pDisplay->getCombinedScreenVariableData();
            _CommandList = Sds_ViewDB_oGetCommandList(screenVariableData, _viewId);
            _commandPageNumber = 0;
         }
         break;

         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_TOLASTPAGE:
         {
            std::map<std::string, std::string> screenVariableData = _pDisplay->getCombinedScreenVariableData();
            _CommandList = Sds_ViewDB_oGetCommandList(screenVariableData, _viewId);
            _commandPageNumber = getLastCommandPage();
         }
         break;

         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_RESTORE_PREVIOUS:
         {
            std::map<std::string, std::string> screenVariableData = _pDisplay->getCombinedScreenVariableData();
            _CommandList = Sds_ViewDB_oGetCommandList(screenVariableData, _viewId);
            _commandPageNumber = 0;
         }
         break;

         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_LIST_NEXT:
         {
            std::map<std::string, std::string> screenVariableData = _pDisplay->getCombinedScreenVariableData();
            _CommandList = Sds_ViewDB_oGetCommandList(screenVariableData, _viewId);
            if (_commandPageNumber < getLastCommandPage())
            {
               ++_commandPageNumber;
            }
         }
         break;

         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_LIST_PREV:
         {
            std::map<std::string, std::string> screenVariableData = _pDisplay->getCombinedScreenVariableData();
            _CommandList = Sds_ViewDB_oGetCommandList(screenVariableData, _viewId);
            if (_commandPageNumber > 0)
            {
               --_commandPageNumber;
            }
         }
         break;
         default:
            break;
      }
   }

   if (_pMenuManager)
   {
      _pMenuManager->vShowSDSView(oScreenData, getPageNumber(_viewId), getNumberOfPages(_viewId), getNextPreviousOptions(_viewId), _maxItemsPerPage, oMessage.nPopUp_UpDateType);
   }
   if (_viewId == "SR_PHO_DialNumber2" || _viewId == "SR_PHO_DialNumberSendText" ||
         _viewId == "SR_PHO_InCall" || _viewId == "SR_PHO_InCall2")
   {
      vUpdateSpokenDigitString(oScreenData, _viewId);
   }
   vSendResult(oScreenData);
}


/**************************************************************************//**
 *.
 ******************************************************************************/
tVoid clSDS_Method_CommonShowDialog::vUpdatePageNumber(sds2hmi_fi_tcl_e8_PopUp_UpdateType nPopUp_UpDateType)
{
   if (_pListScreen)
   {
      switch (nPopUp_UpDateType.enType)
      {
         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_ALL:
            _pListScreen->vResetPageNumber();
            break;

         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_LIST_NEXT:
            _pListScreen->vIncrementPageNumber();
            break;

         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_LIST_PREV:
            _pListScreen->vDecrementPageNumber();
            break;

         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_TOFIRSTPAGE:
            _pListScreen->vResetPageNumber();
            _commandPageNumber = 0;
            break;

         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_TOLASTPAGE:
            while (_pListScreen->bHasNextPage())
            {
               _pListScreen->vIncrementPageNumber();
            }
            _commandPageNumber = getLastCommandPage();
            break;

         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_RESTORE_PREVIOUS:
         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_HEADLINE:
            // do nothing
            break;

         default:
            NORMAL_M_ASSERT_ALWAYS();   //unexpected update type
            _pListScreen->vResetPageNumber();   //lint !e527 Unreachable code - jnd2hi: behind assertion
            break;
      }
   }
}


/**************************************************************************//**
 *.
 ******************************************************************************/
tVoid clSDS_Method_CommonShowDialog::vSendResult(const clSDS_ScreenData& oScreenData)
{
   sds2hmi_sdsfi_tclMsgCommonShowDialogMethodResult oResult;
   std::string viewId = oScreenData.readScreenId();
   if (_pListScreen && _pListScreen->bIsListScreen(viewId))
   {
      oResult.nListCount  = _pListScreen->getNumOfElementsOnCurrentPage();
      oResult.nResultOptions.u8Value  = getNextPreviousListPageOptions();
      oResult.nPageNumber = (tU16)_pListScreen->u32GetPageNumber() + 1;
   }
   else
   {
      // TODO jnd2hi: do this for any command menu; move to new CommandList class
      oResult.nListCount  = (tU8) getNumberOfCommandsOnPage();
      oResult.nResultOptions.u8Value  = getNextPreviousCommandPageOptions();
      oResult.nPageNumber = _commandPageNumber;
   }
   oResult.PrevDomain.enType = getPreviousDomain(viewId);
   oResult.NextDomain.enType = getNextDomain(viewId);
   vSendMethodResult(oResult);
}


/**************************************************************************//**
 *.
 ******************************************************************************/
unsigned int clSDS_Method_CommonShowDialog::getNumberOfCommandsOnPage()
{
   if (_commandPageNumber < getLastCommandPage())
   {
      return _maxItemsPerPage;
   }
   else
   {
      return _CommandList.size() % _maxItemsPerPage;
   }
}


/**************************************************************************//**
 *.
 ******************************************************************************/
sds2hmi_fi_tcl_e8_MainDomains::tenType clSDS_Method_CommonShowDialog::getPreviousDomain(const std::string& viewId) const
{
   if ((_pListScreen && _pListScreen->bHasPreviousPage()) || (_commandPageNumber != 0))
   {
      return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NONE;
   }
   else if (!clSDS_ConfigurationFlags::getSwitchThroughDomainsKey())
   {
      return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NONE;
   }
   else
   {
      if (viewId == "SR_PHO_Main")
      {
         if (isVehicleInfoAvailable())
         {
            return  sds2hmi_fi_tcl_e8_MainDomains ::FI_EN_DOMAIN_VEH_INFO;
         }
         if (isEveAvailable())
         {
            return  sds2hmi_fi_tcl_e8_MainDomains ::FI_EN_DOMAIN_EVE;
         }
         if (isInfoAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_INFORMATION;
         }
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_AUDIO;
      }
      else if (viewId == "SR_NAV_Main")
      {
         if (isPhoneAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_PHONE;
         }
         if (isVehicleInfoAvailable())
         {
            return  sds2hmi_fi_tcl_e8_MainDomains ::FI_EN_DOMAIN_VEH_INFO;
         }
         if (isEveAvailable())
         {
            return  sds2hmi_fi_tcl_e8_MainDomains ::FI_EN_DOMAIN_EVE;
         }
         if (isInfoAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_INFORMATION;
         }
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_AUDIO;
      }
      else if (viewId == "SR_AUD_Main")
      {
         if (isNaviAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NAVIGATION;
         }
         if (isPhoneAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_PHONE;
         }
         if (isVehicleInfoAvailable())
         {
            return  sds2hmi_fi_tcl_e8_MainDomains ::FI_EN_DOMAIN_VEH_INFO;
         }
         if (isEveAvailable())
         {
            return  sds2hmi_fi_tcl_e8_MainDomains ::FI_EN_DOMAIN_EVE;
         }
         if (isInfoAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_INFORMATION;
         }
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_AUDIO;
      }
      else if (viewId == "SR_INF_Main")
      {
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_AUDIO;
      }
      else if (viewId == "SR_EVE_Main" || viewId == "SR_VINF_Main")
      {
         if (isInfoAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_INFORMATION;
         }
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_AUDIO;
      }
      else
      {
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NONE;
      }
   }
}


/**************************************************************************//**
 *.
 ******************************************************************************/
sds2hmi_fi_tcl_e8_MainDomains::tenType clSDS_Method_CommonShowDialog::getNextDomain(const std::string& viewId) const
{
   const bool onLastCommandPage = ((_commandPageNumber + 1) == getNumberOfCommandPages());

   if ((_pListScreen && _pListScreen->bHasNextPage()) || (!onLastCommandPage))
   {
      return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NONE;
   }
   else if (!clSDS_ConfigurationFlags::getSwitchThroughDomainsKey())
   {
      return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NONE;
   }

   else
   {
      if (viewId == "SR_PHO_Main")
      {
         if (isNaviAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NAVIGATION;
         }
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_AUDIO;
      }
      else if (viewId == "SR_NAV_Main")
      {
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_AUDIO;
      }
      else if (viewId == "SR_AUD_Main")
      {
         if (isInfoAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_INFORMATION;
         }
         if (isEveAvailable())
         {
            return  sds2hmi_fi_tcl_e8_MainDomains ::FI_EN_DOMAIN_EVE;
         }
         if (isVehicleInfoAvailable())
         {
            return  sds2hmi_fi_tcl_e8_MainDomains ::FI_EN_DOMAIN_VEH_INFO;
         }
         if (isPhoneAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_PHONE;
         }
         if (isNaviAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NAVIGATION;
         }
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_AUDIO;
      }
      else if (viewId == "SR_INF_Main")
      {
         if (isEveAvailable())
         {
            return  sds2hmi_fi_tcl_e8_MainDomains ::FI_EN_DOMAIN_EVE;
         }
         if (isVehicleInfoAvailable())
         {
            return  sds2hmi_fi_tcl_e8_MainDomains ::FI_EN_DOMAIN_VEH_INFO;
         }
         if (isPhoneAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_PHONE;
         }
         if (isNaviAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NAVIGATION;
         }
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_AUDIO;
      }
      else if (viewId == "SR_EVE_Main" || viewId == "SR_VINF_Main")
      {
         if (isPhoneAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_PHONE;
         }
         if (isNaviAvailable())
         {
            return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NAVIGATION;
         }
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_AUDIO;
      }
      else
      {
         return sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NONE;
      }
   }
}


/**************************************************************************//**
 *.
 ******************************************************************************/
tU8 clSDS_Method_CommonShowDialog::getNextPreviousListPageOptions() const
{
   if (_pListScreen)
   {
      if (_pListScreen->bHasPreviousPage() && _pListScreen->bHasNextPage())
      {
         return NEXT_PREVIOUS_PAGE;
      }
      if (_pListScreen->bHasNextPage())
      {
         return NEXT_PAGE;
      }
      if (_pListScreen->bHasPreviousPage())
      {
         return PREVIOUS_PAGE;
      }
   }
   return SINGLE_PAGE;
}


/**************************************************************************//**
 *.
 ******************************************************************************/
tU8 clSDS_Method_CommonShowDialog::getNextPreviousCommandPageOptions() const
{
   const bool nextAvail = (_commandPageNumber < getLastCommandPage());
   const bool prevAvail = (_commandPageNumber > 0);

   if (nextAvail && prevAvail)
   {
      return NEXT_PREVIOUS_PAGE;
   }
   if (nextAvail)
   {
      return NEXT_PAGE;
   }
   if (prevAvail)
   {
      return PREVIOUS_PAGE;
   }
   return SINGLE_PAGE;
}


/**************************************************************************//**
 *.
 ******************************************************************************/
tVoid clSDS_Method_CommonShowDialog::vTraceXmlData(const std::string& oXmlData) const
{
   const int MAX_TRACE_SENTENCE_WORDS = 200;
   size_t sentence_no = (oXmlData.size() / MAX_TRACE_SENTENCE_WORDS);
   sentence_no++;
   size_t pos = 0;
   while (sentence_no > 0)
   {
      std::string xmlSubStr = oXmlData.substr(pos, MAX_TRACE_SENTENCE_WORDS);
      if (!(xmlSubStr.empty()))
      {
         ETG_TRACE_USR4(("XML Stream-%d = %s", pos, xmlSubStr.c_str()));
      }
      sentence_no--;
      pos += MAX_TRACE_SENTENCE_WORDS;
   }
}


/**************************************************************************//**
 *.
 ******************************************************************************/
tVoid clSDS_Method_CommonShowDialog::vUpdateSpokenDigitString(const clSDS_ScreenData& oScreenData, const std::string& viewId)
{
   ETG_TRACE_USR1(("clSDS_Method_CommonShowDialog::vUpdateSpokenDigitString"));
   // TODO jnd2hi: get rid of hard-coded view id names below
   if (viewId == "SR_PHO_DialNumber2" || viewId == "SR_PHO_DialNumberSendText" ||
         viewId == "SR_PHO_InCall" || viewId == "SR_PHO_InCall2")
   {
      ETG_TRACE_USR1(("%s", oScreenData.getPhNumDigit().c_str()));
      std::string _SpokenDigits = oScreenData.getPhNumDigit();
      if (_SpokenDigits != "" && _pSdsPhoneService && _pGuiService)
      {
         ETG_TRACE_USR1(("sendSpokenDigitsSignal"));
         _pSdsPhoneService->sendSpokenDigitsSignal(_SpokenDigits);
         _pGuiService->sendSpokenDigitsSignal(_SpokenDigits);
      }
   }
}


/**************************************************************************//**
 * TODO jnd2hi remove this function; clSDS_Method_CommonShowDialog should not
 * expose the view id; better move to clSDS_View.
 ******************************************************************************/
std::string clSDS_Method_CommonShowDialog::getViewId() const
{
   return _viewId;
}


/**************************************************************************//**
 *.
 ******************************************************************************/
bool clSDS_Method_CommonShowDialog::isPhoneAvailable() const
{
   if (_pProperty_PhoneStatus)
   {
      switch (_pProperty_PhoneStatus->getPhoneStatus())
      {
         case sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_BUSY:
         case sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_IDLE:
         case sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_DIALLING:
         case sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_INCALL:
         case sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_INCALL_HOLD:
         case sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_RINGING:
            return true;

         case sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_NOT_AVAILABLE:
         case sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_PAIRING:
         case sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_PHONE_NOT_READY:
         case sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_UNKNOWN:
         default:
            return false;
      }
   }
   return false;
}


/**************************************************************************//**
 *.
 ******************************************************************************/
bool clSDS_Method_CommonShowDialog::isNaviAvailable() const
{
   return clSDS_ConfigurationFlags::getNavKey();
}


/**************************************************************************//**
 *.
 ******************************************************************************/
bool clSDS_Method_CommonShowDialog::isInfoAvailable()const
{
   return clSDS_ConfigurationFlags::getInfKey();
}


bool clSDS_Method_CommonShowDialog::isEveAvailable()const
{
   return clSDS_ConfigurationFlags::getEveKey();
}


bool clSDS_Method_CommonShowDialog::isVehicleInfoAvailable() const
{
   return clSDS_ConfigurationFlags::getVehicleInfoKey();
}


/**************************************************************************//**
 *.
 ******************************************************************************/
uint32 clSDS_Method_CommonShowDialog::getPageNumber(const std::string& viewId)
{
   if (_pListScreen && _pListScreen->bIsListScreen(viewId))
   {
      return _pListScreen->u32GetPageNumber();
   }
   else
   {
      return _commandPageNumber;
   }
}


/**************************************************************************//**
 *.
 ******************************************************************************/
uint32 clSDS_Method_CommonShowDialog::getNumberOfPages(const std::string& viewId)
{
   if (_pListScreen && _pListScreen->bIsListScreen(viewId))
   {
      return (_pListScreen->getListItemsSize() + _maxItemsPerPage - 1) / _maxItemsPerPage;
   }
   else
   {
      return getNumberOfCommandPages();
   }
}


/**************************************************************************//**
 *.
 ******************************************************************************/
uint32 clSDS_Method_CommonShowDialog::getNumberOfCommandPages() const
{
   return (_CommandList.size() + _maxItemsPerPage - 1) / _maxItemsPerPage;
}


/**************************************************************************//**
 *.
 ******************************************************************************/
uint32 clSDS_Method_CommonShowDialog::getLastCommandPage() const
{
   if (getNumberOfCommandPages())
   {
      return getNumberOfCommandPages() - 1;
   }
   return  0;
}


/**************************************************************************//**
 *.Next/Previous Button Handling for SDS list screens
 *.   .Next/previous buttons should always be enabled in Domain screens.
 *.   .The list screens like N-Best, ambiguity list should have buttons but in
 *.    greyed out of state if there is no previous page or next page.
 *.   .The screens like recent calls , movie theatres , fuel prices
 *.    should not have previous and next button at all
 ******************************************************************************/
tU8 clSDS_Method_CommonShowDialog::getNextPreviousOptions(const std::string& viewId) const
{
   if (_pListScreen && _pListScreen->bIsListScreen(viewId))
   {
      return getNextPreviousListPageOptions();
   }
   else if ((getNextPreviousCommandPageOptions() == SINGLE_PAGE) &&
            (getNextDomain(viewId) == sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NONE) &&
            (getPreviousDomain(viewId) == sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NONE))
   {
      return SINGLE_PAGE;
   }

   else if (!clSDS_ConfigurationFlags::getSwitchThroughDomainsKey())
   {
      return getNextPreviousCommandPageOptions();
   }
   else if (((getNextPreviousCommandPageOptions() == NEXT_PAGE) || getNextPreviousCommandPageOptions() == PREVIOUS_PAGE) &&
            (getNextDomain(viewId) == sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NONE) &&
            (getPreviousDomain(viewId) == sds2hmi_fi_tcl_e8_MainDomains::FI_EN_DOMAIN_NONE))
   {
      return getNextPreviousCommandPageOptions();
   }
   return NEXT_PREVIOUS_PAGE;
}


/**************************************************************************//**
 *. TODO jnd2hi: suspicious! it should not be necessary to know about concrete
 *.              screen ids
 ******************************************************************************/
void clSDS_Method_CommonShowDialog::setActiveScreenObserver(clSDS_ActiveScreenObserver* obs)
{
   if (obs != NULL)
   {
      _pActiveScreenObserver =  obs;
   }
}


void clSDS_Method_CommonShowDialog::vTTFisMethodStart(std::string paramString)
{
   sds2hmi_sdsfi_tclMsgCommonShowDialogMethodStart oMessage;
   oMessage.nPopUp_UpDateType.enType = sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_ALL;
   oMessage.sXMLStream.szValue = new char[500];
   const char* data = NULL;
   if (paramString == "SR_GLO_Main")
   {
      data = "<SDS_DIALOG><TYPE>screen</TYPE><SCREENID>SR_GLO_Main</SCREENID><PARAMETERS><PARAMETER><NAME>INFORMATION_AVAILABLE</NAME><VALUE>TRUE</VALUE></PARAMETER><PARAMETER><NAME>TRAFFIC_AVAILABLE</NAME><VALUE>TRUE</VALUE></PARAMETER><PARAMETER><NAME>CALL_NAME_ENABLED</NAME><VALUE>TRUE</VALUE></PARAMETER></PARAMETERS></SDS_DIALOG>";
   }
   else if (paramString == "SR_PHO_Main")
   {
      data = "<SDS_DIALOG><TYPE>screen</TYPE><SCREENID>SR_PHO_Main</SCREENID><PARAMETERS><PARAMETER><NAME>CALL_NAME_ENABLED</NAME><VALUE>TRUE</VALUE></PARAMETER></PARAMETERS></SDS_DIALOG>";
   }
   else if (paramString == "SR_PHO_NbestList")
   {
      data = "<SDS_DIALOG><TYPE>sdslist_screen</TYPE><SCREENID>SR_PHO_NbestList</SCREENID><HEADER><TEMPLATE><TID>DEFAULT</TID></TEMPLATE></HEADER><ELEMENT><![CDATA[dad ]]></ELEMENT><ELEMENT><![CDATA[karri dad ]]></ELEMENT><ELEMENT><![CDATA[deepu's dad ]]></ELEMENT></SDS_DIALOG>";
   }
   else if (paramString == "SR_PHO_Calling")
   {
      data = "<SDS_DIALOG><TYPE>screen</TYPE><SCREENID>SR_PHO_Calling</SCREENID><HEADER><TEMPLATE><TID>Number_ContactName_Type</TID><VARIABLE><TAG>Number</TAG><VALUE><![CDATA[1]]></VALUE></VARIABLE><VARIABLE><TAG>ContactName</TAG><VALUE><![CDATA[dad]]></VALUE></VARIABLE><VARIABLE><TAG>Type</TAG><VALUE><LOCALID>102</LOCALID></VALUE></VARIABLE></TEMPLATE></HEADER></SDS_DIALOG>";
   }
   else if (paramString == "SR_PHO_EnterNumber")
   {
      data = "<SDS_DIALOG><TYPE>screen</TYPE><SCREENID>SR_PHO_EnterNumber</SCREENID><HEADER><![CDATA[81848689]]></HEADER><PARAMETERS><PARAMETER><NAME>PHONENUMBER_AVAILABLE</NAME><VALUE>TRUE</VALUE></PARAMETER></PARAMETERS><HEADER><TEMPLATE><TID>PhoneNumberDigits</TID><VARIABLE><TAG>PhNumberDigits</TAG><VALUE><![CDATA[81848689]]></VALUE></VARIABLE></TEMPLATE></HEADER></SDS_DIALOG>";
   }
   else if (paramString == "SR_PHO_SelectPhoneNumber")
   {
      data = "<SDS_DIALOG><TYPE>hmilist_screen</TYPE><LISTBINDING>PhoneNumber_Disamb</LISTBINDING><SCREENID>SR_PHO_SelectPhoneNumber</SCREENID><HEADER><![CDATA[karri dad]]></HEADER><HEADER><TEMPLATE><TID>Contact_Name</TID><VARIABLE><TAG>ContactName</TAG><VALUE><![CDATA[karri dad]]></VALUE></VARIABLE></TEMPLATE></HEADER></SDS_DIALOG>";
   }
   else if (paramString == "SR_AUD_Main")
   {
      data = "<SDS_DIALOG><TYPE>screen</TYPE><SCREENID>SR_AUD_Main</SCREENID><PARAMETERS><PARAMETER><NAME>SONG_AVAILABLE</NAME><VALUE>TRUE</VALUE></PARAMETER><PARAMETER><NAME>ARTIST_AVAILABLE</NAME><VALUE>TRUE</VALUE></PARAMETER><PARAMETER><NAME>ALBUM_AVAILABLE</NAME><VALUE>TRUE</VALUE></PARAMETER><PARAMETER><NAME>GENRE_AVAILABLE</NAME><VALUE>TRUE</VALUE></PARAMETER></PARAMETERS></SDS_DIALOG>";
   }
   else if (paramString == "SR_AUD_Play_Song")
   {
      data = "<SDS_DIALOG><TYPE>screen</TYPE><SCREENID>SR_AUD_Play_Song</SCREENID><HEADER><TEMPLATE><TID>PlaySong</TID></TEMPLATE></HEADER></SDS_DIALOG>";
   }
   else if (paramString == "SR_NAV_Main")
   {
      data = "<SDS_DIALOG><TYPE>screen</TYPE><SCREENID>SR_NAV_Main</SCREENID><PARAMETERS><PARAMETER><NAME>ROUTE_IN_PROGRESS</NAME><VALUE>FALSE</VALUE></PARAMETER><PARAMETER><NAME>STATE_AVAILABLE</NAME><VALUE>FALSE</VALUE></PARAMETER></PARAMETERS><HEADER><TEMPLATE><TID>Navigation</TID></TEMPLATE></HEADER></SDS_DIALOG>";
   }
   else if (paramString == "SR_AUD_EnterSong")
   {
      data = "<SDS_DIALOG><TYPE>screen</TYPE><SCREENID>SR_AUD_EnterSong</SCREENID></SDS_DIALOG>";
   }
   else if (paramString == "SR_AUD_Media_Nbest")
   {
      data = "<SDS_DIALOG><TYPE>sdslist_screen</TYPE><SCREENID>SR_AUD_Media_Nbest</SCREENID><HEADER><TEMPLATE><TID>EnterAlbum</TID></TEMPLATE></HEADER><ELEMENT><![CDATA[Kutty ]]></ELEMENT></SDS_DIALOG>";
   }
   else
   {
      return;
   }
   strcpy(oMessage.sXMLStream.szValue, data);
   vTraceXmlData(oMessage.sXMLStream.szValue);
   clSDS_ScreenData oScreenData(oMessage.sXMLStream.szValue);
   _viewId = oScreenData.readScreenId();
   ETG_TRACE_USR4(("clSDS_Method_CommonShowDialog::vTTFisMethodStart Sds_ViewDB_getLayout(_viewId) %s", Sds_ViewDB_getLayout(_viewId).c_str()));
   ETG_TRACE_USR4(("clSDS_Method_CommonShowDialog::vTTFisMethodStart viewId %s ", _viewId.c_str()));
   if (_pVariantLayoutMap != NULL)
   {
      ETG_TRACE_USR4(("clSDS_Method_CommonShowDialog::vMethodStart _viewId =  %s ", _viewId.c_str()));
      ETG_TRACE_USR4(("clSDS_Method_CommonShowDialog::vMethodStart Sds_ViewDB_getLayout(%s)", (Sds_ViewDB_getLayout(_viewId)).c_str()));
      ETG_TRACE_USR4(("clSDS_Method_CommonShowDialog::vMethodStart _maxItemsPerPage %d ", _maxItemsPerPage));
      _maxItemsPerPage = _pVariantLayoutMap->getItemsPerPage(Sds_ViewDB_getLayout(_viewId));
   }
   if (_pDisplay  &&
         ((oMessage.nPopUp_UpDateType.enType == sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_ALL) ||
          (oMessage.nPopUp_UpDateType.enType == sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_RESTORE_PREVIOUS) ||
          (oMessage.nPopUp_UpDateType.enType == sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_TOLASTPAGE)))
   {
      if (oScreenData.oGetScreenVariableData().size() > 1)
      {
         _pDisplay->setScreenVariableData(oScreenData);
      }
   }
   if (_pActiveScreenObserver != NULL)
   {
      _pActiveScreenObserver->viewIdChanged(_viewId);
   }
   if (_pListScreen)
   {
      _pListScreen->vSetCurrentList(_viewId);
   }
   vUpdatePageNumber(oMessage.nPopUp_UpDateType);
   if (_pListScreen && _pListScreen->bIsListScreen(_viewId))
   {
      _pListScreen->setMaxItemsPerPage(_maxItemsPerPage);
      ETG_TRACE_USR4(("clSDS_Method_CommonShowDialog::vMethodStart _maxItemsPerPage %d ", _maxItemsPerPage));
      std::vector<clSDS_ListItems> oListContents = _pListScreen->vGetListScreenContents(oScreenData, oMessage.nPopUp_UpDateType);
      oScreenData.vStoreListContents(oListContents);
   }
   else if (_pDisplay)
   {
      switch (oMessage.nPopUp_UpDateType.enType)
      {
         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_ALL:
         {
            std::map<std::string, std::string> screenVariableData = _pDisplay->getCombinedScreenVariableData();
            _CommandList = Sds_ViewDB_oGetCommandList(screenVariableData, _viewId);
            _commandPageNumber = 0;
         }
         break;
         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_TOLASTPAGE:
         {
            std::map<std::string, std::string> screenVariableData = _pDisplay->getCombinedScreenVariableData();
            _CommandList = Sds_ViewDB_oGetCommandList(screenVariableData, _viewId);
            _commandPageNumber = getLastCommandPage();
         }
         break;
         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_RESTORE_PREVIOUS:
         {
            std::map<std::string, std::string> screenVariableData = _pDisplay->getCombinedScreenVariableData();
            _CommandList = Sds_ViewDB_oGetCommandList(screenVariableData, _viewId);
            _commandPageNumber = 0;
         }
         break;
         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_LIST_NEXT:
         {
            std::map<std::string, std::string> screenVariableData = _pDisplay->getCombinedScreenVariableData();
            _CommandList = Sds_ViewDB_oGetCommandList(screenVariableData, _viewId);
            if (_commandPageNumber < getLastCommandPage())
            {
               ++_commandPageNumber;
            }
         }
         break;
         case sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_LIST_PREV:
         {
            std::map<std::string, std::string> screenVariableData = _pDisplay->getCombinedScreenVariableData();
            _CommandList = Sds_ViewDB_oGetCommandList(screenVariableData, _viewId);
            if (_commandPageNumber > 0)
            {
               --_commandPageNumber;
            }
         }
         break;
         default:
            break;
      }
   }
   if (_pMenuManager)
   {
      _pMenuManager->vShowSDSView(oScreenData, getPageNumber(_viewId), getNumberOfPages(_viewId), getNextPreviousOptions(_viewId), _maxItemsPerPage, oMessage.nPopUp_UpDateType);
   }
   if (_viewId == "SR_PHO_DialNumber2" || _viewId == "SR_PHO_DialNumberSendText" ||
         _viewId == "SR_PHO_InCall" || _viewId == "SR_PHO_InCall2")
   {
      vUpdateSpokenDigitString(oScreenData, _viewId);
   }
   vSendResult(oScreenData);
}
