/**************************************************************************//**
 * \file       clServerMethod.cpp
 *
 * Base class for SDS server method implementations.
 *
 * This class decodes and delegates an incoming MethodStart requests to
 * the concrete implementation of the method.
 *
 * Multiple requests for the same method will be buffered until the
 * previous MethodStart has been answered with a MethodResult or with
 * an error message. Buffered requests will be submitted one by one
 * after the previous one was answered. Thus, it is guaranteed that
 * only one request at a time must be handled by the method implementation.
 *
 * In case processing of the method request takes longer than
 * 'processingTimeoutMs', the request will be cancelled and answered
 * with an error message. The default timeout can be overridden by the
 * method implementation.
 *
 * \copyright  (C) 2016-2018 Robert Bosch GmbH
 *             (C) 2016-2018 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/framework/clServerMethod.h"
#include "Sds2HmiServer/framework/clServerMethodTimerHelper.h"
#include "SdsAdapter_Trace.h"
#include <string.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCA_FW
#include "trcGenProj/Header/clServerMethod.cpp.trc.h"
#endif


clServerMethod::clServerMethod(tU16 u16FunctionID, ahl_tclBaseOneThreadService* pService, size_t processingTimeoutMs)
   : clFunction(u16FunctionID, pService)
   , _pTimerHelper(new clServerMethodTimerHelper(*this, processingTimeoutMs))
   , _requestPending(false)
   , _u16DestAppID(0)
   , _u16RegisterID(0)
   , _u16CmdCtr(0)
{
}


clServerMethod::~clServerMethod()
{
   delete _pTimerHelper;
   _pTimerHelper = NULL;
}


/**
* Performs opcode check for incoming message. Delegates message handling
* to derived class.
*/
tVoid clServerMethod::vHandleMessage(amt_tclServiceData* pInMsg)
{
   ETG_TRACE_USR1(("rx appid=%04x function=%04x regid=%d opcode=%d cmdctr=%d ", pInMsg->u16GetSourceAppID(), pInMsg->u16GetFunctionID(), pInMsg->u16GetRegisterID(), pInMsg->u8GetOpCode(), pInMsg->u16GetCmdCounter()));
   switch (pInMsg->u8GetOpCode())
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
         if (_requestPending || _messageQueue.hasMessages())
         {
            _messageQueue.queueRequest(pInMsg);
         }
         else
         {
            submitRequest(pInMsg);
         }
         break;

      default:
         break;
   }
}


/**
* Called by method implementation to send MethodResult without parameters.
*/
tVoid clServerMethod::vSendMethodResult()
{
   if (_requestPending)
   {
      vSendEmptyMessage(
         _u16DestAppID,
         _u16CmdCtr,
         _u16RegisterID,
         u16GetFunctionID(),
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
      requestWasAnswered();
   }
   else
   {
      ETG_TRACE_ERR(("vSendMethodResult() was called although there is no pending request"));
   }
}


/**
* Called by method implementation to send MethodResult with parameters.
*/
tVoid clServerMethod::vSendMethodResult(const fi_tclTypeBase& oOutData)
{
   if (_requestPending)
   {
      vSendMessage(
         _u16DestAppID,
         _u16CmdCtr,
         _u16RegisterID,
         oOutData,
         u16GetFunctionID(),
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
      requestWasAnswered();
   }
   else
   {
      ETG_TRACE_ERR(("vSendMethodResult() was called although there is no pending request"));
   }
}


/**
* Called by method implementation to close a request with an error.
*/
tVoid clServerMethod::vSendErrorMessage(tU16 u16ErrorCode)
{
   if (_requestPending)
   {
      clFunction::vSendErrorMessage(
         _u16DestAppID,
         _u16CmdCtr,
         _u16RegisterID,
         u16ErrorCode);
      requestWasAnswered();
   }
   else
   {
      ETG_TRACE_ERR(("vSendErrorMessage() was called although there is no pending request"));
   }
}


/**
* To be called after sending the MethodResult or method error.
* Continue processing of queued requests, if any.
*/
void clServerMethod::requestWasAnswered()
{
   if (_pTimerHelper)
   {
      _pTimerHelper->stopProccessingTimer();
   }
   _requestPending = false;
   if (_pTimerHelper && _messageQueue.hasMessages())
   {
      _pTimerHelper->scheduleQueuedRequest();
   }
}


/**
* Called on expiration of the processing timer.
* Notifies the method implementation, i.e. the derived class, about the
* timeout via onRequestTimeout().
* Defaults to canceling the pending request and answering it with an
* 'InternalError' message.
*/
void clServerMethod::requestTimeout()
{
   ETG_TRACE_ERR(("Method processing timed out for appid=%d, function=%04x regid=%d cmdctr=%d",
                  _u16DestAppID,
                  u16GetFunctionID(),
                  _u16RegisterID,
                  _u16CmdCtr));

   onRequestTimeout();  // notify derived class

   if (_requestPending)
   {
      clFunction::vSendErrorMessage(
         _u16DestAppID,
         _u16CmdCtr,
         _u16RegisterID,
         CCA_C_U16_ERROR_INTERNAL_FAILURE);

      _requestPending = false;
   }
}


/**
* Can be overridden by derived class to cancel any pending requests
* or perform other cleanup.
*/
void clServerMethod::onRequestTimeout()
{
}


/**
* Get the next queued request (if any) from the queue and submit it to
* the method implementation. Delete the queue, if the last element was
* removed so that memory resources are freed up.
*/
void clServerMethod::submitQueuedRequest()
{
   amt_tclServiceData* pMsg = _messageQueue.popQueuedRequest();
   if (pMsg)
   {
      ETG_TRACE_USR1(("submitting queued MethodStart for function=0x%04x cmdctr=%d",
                      pMsg->u16GetFunctionID(),
                      pMsg->u16GetCmdCounter()));
      submitRequest(pMsg);
      MessageQueue::releaseMessageWithContent(&pMsg);
   }
}


/**
* Submit 'pInMsg' to the method implementation via virtual call
* vMethodStart(). Save response parameters for the result message
* since the message object is deleted when this function returns.
*/
void clServerMethod::submitRequest(amt_tclServiceData* pInMsg)
{
   _requestPending = true;
   if (_pTimerHelper)
   {
      _pTimerHelper->startProccessingTimer();
   }
   saveResponseParameters(pInMsg);
   vMethodStart(pInMsg);
}


/**
* The saved response parameters are required for the MethodResult
* or error message.
*/
void clServerMethod::saveResponseParameters(const amt_tclServiceData* pInMessage)
{
   _u16DestAppID = pInMessage->u16GetSourceAppID();
   _u16RegisterID = pInMessage->u16GetRegisterID();
   _u16CmdCtr = pInMessage->u16GetCmdCounter();
}
