/**************************************************************************//**uint32
 * \file       MessageQueue.h
 *
 * This file is part of the ASF-CCA-Server Framework.
 *
 * \copyright  (C) 2016-2018 Robert Bosch GmbH
 *             (C) 2016-2018 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#ifndef MessageQueue_h
#define MessageQueue_h


#include <list>


class amt_tclServiceData;


/**
 * Implements a message queue for service data messages.
 * The queue will only allocate memory for the queue object
 * itself, in case a message is queued.
 * Thus, the queue instance is cheap on memory as long as
 * there is no queuing situation.
 */
class MessageQueue
{
   public:
      MessageQueue();
      virtual ~MessageQueue();

      /**
      * Put a message into the message queue.
      */
      void queueRequest(amt_tclServiceData* pInMessage);

      /**
       * Retrieve the next request message from the queue.
       * Returns a pointer to the message object or NULL in case
       * there are no queued messages.
       * NOTE: When the message processing is done, the caller must
       * release the message's resources with releaseMessageWithContent().
       */
      amt_tclServiceData* popQueuedRequest();

      /**
       * Returns true if there are any messages in the queue.
       * Otherwise returns false.
       */
      bool hasMessages() const;

      /**
       * Delete an AMT message including its shared memory buffer.
       * Takes the address of the message pointer and sets the
       * pointer to NULL to avoid misuse.
       */
      static void releaseMessageWithContent(amt_tclServiceData** ppMessage);

   private:
      amt_tclServiceData* copyMessage(amt_tclServiceData* pMsg);
      void releaseAllQueuedMessages();

      std::list<amt_tclServiceData*>* _queuedRequests;
};


#endif
