/**************************************************************************//**
 * \file       MessageQueue.cpp
 *
 * This file is part of the ASF-CCA-Server Framework.
 *
 * \copyright  (C) 2016-2018 Robert Bosch GmbH
 *             (C) 2016-2018 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/framework/MessageQueue.h"
#include "SdsAdapter_Trace.h"
#include <string.h>

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCA_FW
#include "trcGenProj/Header/MessageQueue.cpp.trc.h"
#endif


MessageQueue::MessageQueue()
   : _queuedRequests(NULL)
{
}


MessageQueue::~MessageQueue()
{
   releaseAllQueuedMessages();
}


/**
 * The queue object will only be allocated if required.
 */
void MessageQueue::queueRequest(amt_tclServiceData* pInMsg)
{
   if (!_queuedRequests)
   {
      _queuedRequests = new std::list<amt_tclServiceData*>();
   }
   if (_queuedRequests)
   {
      amt_tclServiceData* pCopy = copyMessage(pInMsg);
      if (pCopy)
      {
         ETG_TRACE_USR1(("MethodStart is queued"));
         _queuedRequests->push_back(pCopy);
      }
      else
      {
         ETG_TRACE_ERR(("failed to buffer message"));
      }
   }
}


/**
 * When the last message is retrieved, the queue object is deleted.
 */
amt_tclServiceData* MessageQueue::popQueuedRequest()
{
   amt_tclServiceData* pMsg = NULL;
   if (hasMessages())
   {
      pMsg = _queuedRequests->front();
      _queuedRequests->pop_front();
   }
   if (!hasMessages())
   {
      delete _queuedRequests;
      _queuedRequests = NULL;
   }
   return pMsg;
}


bool MessageQueue::hasMessages() const
{
   return _queuedRequests && _queuedRequests->size();
}


void MessageQueue::releaseMessageWithContent(amt_tclServiceData** ppMessage)
{
   if (ppMessage && *ppMessage)
   {
      (void)(*ppMessage)->bDelete();
      delete(*ppMessage);
      *ppMessage = NULL;
   }
}


/**
* Create a copy of a service data message for storing in the queue.
* The explicit clone function takes care of creating a copy of the
* message payload. Returns a pointer to the copy or NULL on error.
*/
amt_tclServiceData* MessageQueue::copyMessage(amt_tclServiceData* pMsg)
{
   amt_tclServiceData* pCopy = new amt_tclServiceData();
   if (pCopy != NULL)
   {
      if (!pCopy->bCloneMessageContent(pMsg) || !pCopy->bIsValid())
      {
         ETG_TRACE_ERR(("failed to copy message"));
         (void) pCopy->bDelete();
         delete pCopy;
         pCopy = NULL;
      }
   }
   return pCopy;
}


void MessageQueue::releaseAllQueuedMessages()
{
   while (_queuedRequests)
   {
      amt_tclServiceData* pMessage = popQueuedRequest();
      releaseMessageWithContent(&pMessage);
   }
}
