/**************************************************************************//**
 * \file       TraceCommandAdapter.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/TraceCommandAdapter.h"
#include "application/EarlyStartupPlayer.h"
#include "application/GuiService.h"
#include "application/PopUpService.h"
#include "application/StringUtils.h"
#include "Sds2HmiServer/functions/clSDS_Method_NaviSetDestinationItem.h"
#include "Sds2HmiServer/functions/clSDS_Property_NaviCurrentCountryState.h"
#include "Sds2HmiServer/functions/clSDS_Method_CommonShowDialog.h"
#include "SdsAdapter_Trace.h"
#include <fstream>

#define ETG_ENABLED
#include "trace_interface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX      "SdsAdapter_"
#define ETG_I_TRACE_CHANNEL         TR_TTFIS_SAAL
#define ETG_DEFAULT_TRACE_CLASS     TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/TraceCommandAdapter.cpp.trc.h"
#endif


typedef std::map<std::string, std::string> AddressRequestMap;


static AddressRequestMap readAddressRequestFile();
static AddressRequestMap readPositionInfoFromFile();

GuiService* TraceCommandAdapter::_pGuiService = NULL;
PopUpService* TraceCommandAdapter::_pPopUpService = NULL;
EarlyStartupPlayer* TraceCommandAdapter::_pEarlyStartupPlayer = NULL;
clSDS_Method_NaviSetDestinationItem* TraceCommandAdapter::_pMethod_NaviSetDestinationItem = NULL;
clSDS_Property_NaviCurrentCountryState* TraceCommandAdapter::_pPropertyNaviCurrentCountryState = NULL;
clSDS_Method_CommonShowDialog* TraceCommandAdapter::_pMethodCommonShowDialog = NULL;
clSDS_Display* TraceCommandAdapter::_pDisplay = NULL;

TraceCommandAdapter::TraceCommandAdapter(
   GuiService* guiService,
   PopUpService* popUpService,
   EarlyStartupPlayer* earlyStartupPlayer,
   clSDS_Method_NaviSetDestinationItem* pMethod_NaviSetDestinationItem,
   clSDS_Property_NaviCurrentCountryState* pPropertyNaviCurrentCountryState,
   clSDS_Method_CommonShowDialog* pMethodCommonShowDialog,
   clSDS_Display *pDisplay)
{
   vInitPlatformEtg();
   ETG_I_REGISTER_FILE();
   ETG_I_REGISTER_CHN(commandNotProcessed);
   _pGuiService = guiService;
   _pPopUpService = popUpService;
   _pEarlyStartupPlayer = earlyStartupPlayer;
   _pMethod_NaviSetDestinationItem = pMethod_NaviSetDestinationItem;
   _pPropertyNaviCurrentCountryState = pPropertyNaviCurrentCountryState;
   _pMethodCommonShowDialog = pMethodCommonShowDialog;
   _pDisplay = pDisplay;
}


TraceCommandAdapter::~TraceCommandAdapter()
{
}


ETG_I_CMD_DEFINE((TraceCommandAdapter::pttShortPress, "pttShortPress"))

void TraceCommandAdapter::pttShortPress()
{
   if (_pGuiService)
   {
      _pGuiService->vPttShortPressed();
   }
}


ETG_I_CMD_DEFINE((TraceCommandAdapter::testPttShortPress, "testPttShortPress %s", ETG_I_STRING))
void TraceCommandAdapter::testPttShortPress(std::string paramString)
{
   _pMethodCommonShowDialog->vTTFisMethodStart(paramString);
}


ETG_I_CMD_DEFINE((TraceCommandAdapter::closeView, "closeView"))
void TraceCommandAdapter::closeView()
{
	if (_pDisplay)
    {
      _pDisplay->vCloseView();
   }
}


ETG_I_CMD_DEFINE((TraceCommandAdapter::startAudio, "startAudio"))

void TraceCommandAdapter::startAudio()
{
   if (_pEarlyStartupPlayer)
   {
      _pEarlyStartupPlayer->requestStartPlayback();
   }
}


ETG_I_CMD_DEFINE((TraceCommandAdapter::stopAudio, "stopAudio"))

void TraceCommandAdapter::stopAudio()
{
   if (_pEarlyStartupPlayer)
   {
      _pEarlyStartupPlayer->requestStopPlayback();
   }
}


ETG_I_CMD_DEFINE((TraceCommandAdapter::setMicrophoneLevel, "setMicrophoneLevel %d", unsigned int))

void TraceCommandAdapter::setMicrophoneLevel(unsigned int level)
{
   if (_pPopUpService)
   {
      _pPopUpService->setMicrophoneLevel((uint32)level);
   }
}


ETG_I_CMD_DEFINE((TraceCommandAdapter::naviRequest, "naviRequest %s", ETG_I_STRING))

void TraceCommandAdapter::naviRequest(std::string paramString)
{
   if (_pMethod_NaviSetDestinationItem)
   {
      AddressRequestMap addressRequestMap = readAddressRequestFile();
      const std::string& request = addressRequestMap[paramString];
      if (request.empty())
      {
         ETG_TRACE_FATAL(("no address request found for '%s'", paramString.c_str()));
      }
      else
      {
         ETG_TRACE_FATAL(("create address request for '%s'", paramString.c_str()));
         ETG_TRACE_FATAL(("with '%s'", request.c_str()));
      }
   }
}


ETG_I_CMD_DEFINE((TraceCommandAdapter::positionInfo, "positionInfo"))

void TraceCommandAdapter::positionInfo()
{
   if (_pPropertyNaviCurrentCountryState)
   {
      AddressRequestMap addressRequestMap = readPositionInfoFromFile();

      _pPropertyNaviCurrentCountryState->sendCountryStateStatusUsingTTFis(addressRequestMap);
   }
}


/**
 * sink for not processed ttfis input messages
 */
void TraceCommandAdapter::commandNotProcessed(const unsigned char* data)
{
   if (data)
   {
      ETG_TRACE_FATAL(("trace command was not processed: 0x%*x", ETG_LIST_LEN(data[0]), ETG_LIST_PTR_T8(data)));
   }
}


/*
 * Helper Functions
 */

static std::string stripLeadingWhitespace(const std::string& s)
{
   const size_t pos = s.find_first_not_of(" \t\n");
   if (std::string::npos != pos)
   {
      return s.substr(pos);
   }
   else
   {
      return "";  // contains only whitespace or is empty
   }
}


static std::string stripTrailingWhitespace(const std::string& s)
{
   const size_t pos = s.find_last_not_of(" \t\n");
   if (std::string::npos != pos)
   {
      return s.substr(0, pos + 1);
   }
   else
   {
      return "";  // contains only whitespace or is empty
   }
}


static std::string stripWhitespace(const std::string& s)
{
   return stripLeadingWhitespace(stripTrailingWhitespace(s));
}


static std::string stripComment(const std::string& line)
{
   std::string s = line;
   size_t pos = line.find('#');
   if (pos != std::string::npos)
   {
      s.erase(pos);
   }
   return stripWhitespace(s);
}


static AddressRequestMap readAddressRequestFile()
{
   const char* filename = "/opt/bosch/sds/address_requests.txt";
   std::ifstream input(filename);
   std::string line;
   std::string section;
   size_t lineNum = 0;
   AddressRequestMap addressRequestMap;

   while (std::getline(input, line))
   {
      ++lineNum;  // increment here so we start at line 1 and 'continue' can be used
      line = stripComment(line);
      if (line.length() == 0)
      {
         continue;   // ignore empty or comment line
      }

      if ((line.length() > 2) && (line[0] == '[') && (line[line.length() - 1] == ']'))
      {
         section = line.substr(1, line.length() - 2);
         section = stripWhitespace(section);
         continue;   // new section found
      }

      if (section.length() == 0)
      {
         ETG_TRACE_FATAL(("content outside a section or empty section name in line %u of file %s", lineNum, filename));
         continue;   // section not set or section name was empty - ignore this line
      }

      if (addressRequestMap[section].length() == 0)
      {
         addressRequestMap[section] = line;
      }
      else
      {
         ETG_TRACE_FATAL(("ignoring duplicate definition in line %u for section '%s'", lineNum, section.c_str()));
      }
   }

   return addressRequestMap;
}


static AddressRequestMap readPositionInfoFromFile()
{
   AddressRequestMap positionInfoMap;
   const char* filename = "/opt/bosch/sds/positionInfo.txt";
   std::ifstream input(filename);
   std::string line;

   while (std::getline(input, line))
   {
      size_t colon = line.find(":");
      if (colon != std::string::npos)
      {
         std::string key = line.substr(0, colon);
         StringUtils::trim(key);
         std::string value = line.substr(colon + 1);
         StringUtils::trim(value);
         positionInfoMap[key] = value;
         ETG_TRACE_FATAL(("position info: %s", (key + " : " + value).c_str()));
      }
   }

   return positionInfoMap;
}
