/**************************************************************************//**
* \file     tunerSource.cpp
*
*           tunerSource method class implementation
*
* \remark   Copyright: 2010 Robert Bosch GmbH, Hildesheim
******************************************************************************/

#include "sessionManagement/audio/config/tunerSessionManagement.h"


using namespace tuner::sessionManagement::audio;
using namespace tuner::common;


static stTunerSessionManagement aTunerSources[] =
{
   #define TUNER_SESSIONMANAGEMENT(sourceId, band, sourceType, tunerClientAvailability)                  { sourceId, band, sourceType, tunerClientAvailability},
   #include "tunerSessionManagement.dat"
   #undef TUNER_SESSIONMANAGEMENT
};


static const stSrcActivityTunerStateMap aSrcActivityTunerStateMap[] =
{
   #define TUNER_SRCACTIVITY_TUNERSTATE(srcActivity, tunerState)                   { srcActivity, tunerState},
   #include "tunerSessionManagement.dat"
   #undef TUNER_SRCACTIVITY_TUNERSTATE
};


static const stSrcActivityResponseTimeMap aArlSrcActivtyResponseTimeMap[] =
{
   #define TUNER_SRCACTIVITY_MAXWAITTIME(tunerSrcActivity, timeInMs)                  { tunerSrcActivity, timeInMs},
   #include "tunerSessionManagement.dat"
   #undef TUNER_SRCACTIVITY_MAXWAITTIME
};


/**************************************************************************//**
*
******************************************************************************/
bool bIsSourceAvailable(enSourceId sourceId)
{
   const int numberofSources = sizeof aTunerSources / sizeof(stTunerSessionManagement);
   for (int u32Index = 0; u32Index < numberofSources; u32Index++)
   {
      if (aTunerSources[u32Index].oSourceId == sourceId)
      {
         return true;
      }
   }
   return false;
}


/**************************************************************************//**
*
******************************************************************************/
enBand oGetBand(enSourceId sourceId)
{
   const int numberofSources = sizeof aTunerSources / sizeof(stTunerSessionManagement);
   for (int u32Index = 0; u32Index < numberofSources; u32Index++)
   {
      if (aTunerSources[u32Index].oSourceId == sourceId)
      {
         return aTunerSources[u32Index].oBand;
      }
   }
   return BAND_LIMIT;
}


/**************************************************************************//**
*
******************************************************************************/
stTunerSessionManagement oGetSource(enSourceId sourceId)
{
   const int numberofSources = sizeof aTunerSources / sizeof(stTunerSessionManagement);
   for (int u32Index = 0; u32Index < numberofSources; u32Index++)
   {
      if (aTunerSources[u32Index].oSourceId == sourceId)
      {
         return aTunerSources[u32Index];
      }
   }
   stTunerSessionManagement oEmptySource;
   oEmptySource.oBand = BAND_LIMIT;
   oEmptySource.oSourceId = SOURCEID_LIMIT;
   oEmptySource.oSourceType = SOURCETYPE_LIMIT;
   oEmptySource.oTunerClient = TUNERCLIENT_LIMIT;
   return oEmptySource;
}


/**************************************************************************//**
*
******************************************************************************/
enTunerState oGetTunerState(enSrcActivity srcActivity)
{
   const int mapSize = sizeof aSrcActivityTunerStateMap / sizeof(stSrcActivityTunerStateMap);
   for (int u32Index = 0; u32Index < mapSize; u32Index++)
   {
      if (aSrcActivityTunerStateMap[u32Index].srcActivity == srcActivity)
      {
         return aSrcActivityTunerStateMap[u32Index].tunerState;
      }
   }
   return TUNERSTATE_UNKNOWN;
}


/**************************************************************************//**
*
******************************************************************************/
bool bIsStaticSource(enSourceId sourceId)
{
   const int numberofSources = sizeof aTunerSources / sizeof(stTunerSessionManagement);
   for (int u32Index = 0; u32Index < numberofSources; u32Index++)
   {
      if (aTunerSources[u32Index].oSourceId == sourceId)
      {
         if (aTunerSources[u32Index].oSourceType == tuner::common::STATIC)
		 {
			return true;
		 }
      }
   }
   return false;
}


/**************************************************************************//**
*
******************************************************************************/
bool bIsDynamicSource(enSourceId sourceId)
{
   const int numberofSources = sizeof aTunerSources / sizeof(stTunerSessionManagement);
   for (int u32Index = 0; u32Index < numberofSources; u32Index++)
   {
      if (aTunerSources[u32Index].oSourceId == sourceId)
      {
         if (aTunerSources[u32Index].oSourceType == tuner::common::DYNAMIC)
		 {
			return true;
		 }
      }
   }
   return false;
}


/**************************************************************************//**
*
******************************************************************************/
bool bIsTemporarySource(enSourceId sourceId)
{
   const int numberofSources = sizeof aTunerSources / sizeof(stTunerSessionManagement);
   for (int u32Index = 0; u32Index < numberofSources; u32Index++)
   {
      if (aTunerSources[u32Index].oSourceId == sourceId)
      {
         if (aTunerSources[u32Index].oSourceType == tuner::common::TEMPORARY)
		 {
			return true;
		 }
      }
   }
   return false;
}


/**************************************************************************//**
*
******************************************************************************/
int getMaxWaitTime(enSrcActivity tunerSrcActivity)
{
   const int mapSize = sizeof aArlSrcActivtyResponseTimeMap / sizeof(stSrcActivityResponseTimeMap);
   for (int u32Index = 0; u32Index < mapSize; u32Index++)
   {
      if (aArlSrcActivtyResponseTimeMap[u32Index].tunerSrcActivity == tunerSrcActivity)
      {
    	  return aArlSrcActivtyResponseTimeMap[u32Index].timeInMs;
      }
   }
   return 0;
}
/***********************************************************************
 *
 *****************************************************/
tuner::common::enTunerClient oGetTunerClient(tuner::common::enSourceId sourceId)
{
	const int mapSize = sizeof aTunerSources / sizeof(stTunerSessionManagement);
	for (int u32Index = 0; u32Index < mapSize; u32Index++)
	{
		if (aTunerSources[u32Index].oSourceId == sourceId)
		{
			return aTunerSources[u32Index].oTunerClient;
		}
	}
	return TUNERCLIENT_LIMIT;
}
tuner::common::enSourceId oGetTunerSourceId(tuner::common::enBand band)
{
	const int numberofSources = sizeof aTunerSources / sizeof(stTunerSessionManagement);
	for (int u32Index = 0; u32Index < numberofSources; u32Index++)
	{
		if (aTunerSources[u32Index].oBand == band)
		{
			return aTunerSources[u32Index].oSourceId;
		}
	}
	return SOURCEID_LIMIT;
}
/**************************************************************************//**
*
******************************************************************************/
void vSetMidwTunerBand(unsigned int u32CombiBand, unsigned int u32AvailableBand)
{
	if(u32CombiBand & READ_DRM_MODULATION_BIT)
	{
		vTunerSetCB1forDRMandBandForAM(u32CombiBand,u32AvailableBand);
	}
	else
	{
		vTunerSetNoneforDRMandBandForAM(u32CombiBand,u32AvailableBand);
	}
}
/**************************************************************************//**
*
******************************************************************************/
void vTunerSetCB1forDRMandBandForAM(unsigned int /*u32CombiBand*/, unsigned int u32AvailableBand)
{
   const int numberofSources = sizeof aTunerSources / sizeof(stTunerSessionManagement);
   for (int u32Index = 0; u32Index < numberofSources; u32Index++)
   {
		switch(aTunerSources[u32Index].oSourceId)
		{
			case DRM:
			aTunerSources[u32Index].oBand = BANDCB1;
			break;

			case AM:
			if(u32AvailableBand & LW_BIT_SET)
			{
				aTunerSources[u32Index].oBand = BANDLW;
			}
			else if(u32AvailableBand & MW_BIT_SET)
			{
				aTunerSources[u32Index].oBand = BANDMW;
			}
			else
			{
				
			}
			break;

			default: break;
		}
	}
}
/**************************************************************************//**
*
******************************************************************************/
void vTunerSetNoneforDRMandBandForAM(unsigned int u32CombiBand, unsigned int u32AvailableBand)
{
   const int numberofSources = sizeof aTunerSources / sizeof(stTunerSessionManagement);
   for (int u32Index = 0; u32Index < numberofSources; u32Index++)
   {
		switch(aTunerSources[u32Index].oSourceId)
		{
			case AM:
			if(u32CombiBand & LW_MW_BITS_SET)
			{
				aTunerSources[u32Index].oBand = BANDCB1;
			}
			else
			{
				if(u32AvailableBand & LW_BIT_SET)
				{
					aTunerSources[u32Index].oBand = BANDLW;
				}
				else if(u32AvailableBand & MW_BIT_SET)
				{
					aTunerSources[u32Index].oBand = BANDMW;
				}
				else
				{

				}
			}
			break;

			case DRM:
			aTunerSources[u32Index].oBand = BANDNONE;
			break;

			default: break;
		}
	}
}
