/**************************************************************************//**
* \file     clAudioSession.cpp
*
*           clAudioSession method class implementation
*
* \remark   Copyright: 2010 Robert Bosch GmbH, Hildesheim
******************************************************************************/

#include "sessionManagement/audio/clAudioSession.h"
#include "sessionManagement/audio/config/tunerSessionManagement.h"
#include "sessionManagement/audio/clAudioControl.h"
#include "sessionManagement/audio/clTunerControl.h"
#include "common/tuner_trace.h"


using namespace tuner::sessionManagement::audio;
using namespace tuner::common;


clAudioSession* clAudioSession::m_poSession = 0;
unsigned char clAudioSession::timerExpiryCount=0;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUNER_TRACE_CLASS_SESSIONMANAGEMENT
#include "trcGenProj/Header/clAudioSession.cpp.trc.h"
#endif


/**************************************************************************//**
* Constructor
******************************************************************************/
clAudioSession::clAudioSession()
: tunerControl(0)
{
   vInit();
   bFGBGLearnForNonRadioSource = false;
   DABPreviousState = tuner::common::TUNERSTATE_UNKNOWN;
   timer = 0;
   tS32 s32Ret = OSAL_s32TimerCreate( (OSAL_tpfCallback)vOnTimerExpiry,
               0, &timer );
	ETG_TRACE_USR4(("SESSION_MGT: clAudioSession OSAL_s32TimerCreate Ret = %d\n", s32Ret ));
}


/**************************************************************************//**
* Destructor
******************************************************************************/
clAudioSession::~clAudioSession()
{
   vDeInit();
   bFGBGLearnForNonRadioSource = false;
   tS32 s32OsalError = OSAL_s32TimerDelete( timer);
   (void)s32OsalError;
   timer = 0;
}


/**************************************************************************//**
*
******************************************************************************/
tuner::sessionManagement::audio::clAudioSession* clAudioSession::instance()
{
   if (0 == clAudioSession::m_poSession)
   {
      clAudioSession::m_poSession = new tuner::sessionManagement::audio::clAudioSession();
   }
   return clAudioSession::m_poSession;
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::deleteInstance()
{
   if (0 != clAudioSession::m_poSession)
   {
      delete clAudioSession::m_poSession;
      clAudioSession::m_poSession = 0;
   }
}


/**************************************************************************//**
* 
******************************************************************************/
void clAudioSession::vInit()
{
   tunerControl = clTunerControl::instance();
   activeSession.vReset();
   requestedSession.vReset();
   vPrintSessionInformation();
}


/**************************************************************************//**
* 
******************************************************************************/
void clAudioSession::vDeInit()
{
   activeSession.vReset();
   requestedSession.vReset();
   /*On streamer router service unavailabel send TA off info(TUNMSTR_G_ANNOUNCEMENT_STATUS) to HMI */
   tunerControl->vClearAnnouncementStates(FM1);
   tunerControl->vClearAnnouncementStates(DAB);
   instance()->tunerClientServiceState.vClearPreviousTunerState();
   instance()->tunerClientServiceState.vClearStoredRequest();
   instance()->tunerClientServiceState.vClearTunerState(AMFMTUNER);
   instance()->tunerClientServiceState.vClearTunerState(DABTUNER);
   instance()->tunerClientServiceState.vClearStoredAllocateRequest();
   instance()->timerExpiryCount=0;
   //activeSession.state = tuner::common::SRC_CHANGE_NOT_ALLOWED;
   vPrintSessionInformation();
}


/**************************************************************************//**
*
******************************************************************************/
bool clAudioSession::bOnAllocate(tuner::common::enSourceId tunerSourceId, tunerString inputDevice)
{
   ETG_TRACE_USR4(("SESSION_MGT: bOnAllocate 	      source         = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
   ETG_TRACE_USR4(("SESSION_MGT: bOnAllocate 	      inputDevice    = %s\n", inputDevice.c_str()));
   tunerClientServiceState.vClearStoredAllocateRequest();
   if( tunerClientServiceState.bIsServiceUnAvailable(tunerSourceId) == true || (tunerClientServiceState.bIsServiceUnKnown(tunerSourceId) == true))
   {
	   vSynchronizeOnServiceStateUnAvailable(tunerSourceId, inputDevice);
   }
   else if( tunerClientServiceState.bIsServiceAvailable(tunerSourceId) == true )
   {
	   vSynchronizeOnServiceStateAvailable(tunerSourceId, inputDevice);
   }
   return true;
}


/**************************************************************************//**
* 
******************************************************************************/
bool clAudioSession::bOnDeAllocate(tuner::common::enSourceId tunerSourceId)
{
   ETG_TRACE_USR4(("SESSION_MGT: bOnDeAllocate        source         = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
   vSynchronize();
   return true;
}


/**************************************************************************//**
* 
******************************************************************************/
bool clAudioSession::bOnSrcActivity(tuner::common::enSourceId tunerSourceId,
                     tuner::common::enSrcActivity tunerSrcActivity)
{
   ETG_TRACE_USR1(("CCR.STARTING.SOURCE_ACTIVE:bOnSrcActivity"));
   ETG_TRACE_USR4(("SESSION_MGT: bOnSrcActivity       source         = %d\n",  ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
   ETG_TRACE_USR4(("SESSION_MGT: bOnSrcActivity       srcactivity    = %d\n",  ETG_ENUM(TUNER_SRCACTIVITY, (tU8)tunerSrcActivity)));
   vHandleClearStoredRequest(tunerSourceId, tunerSrcActivity);
   if( (tunerClientServiceState.bIsServiceUnAvailable(tunerSourceId) == true) || (tunerClientServiceState.bIsServiceUnKnown(tunerSourceId) == true) )
   {
	   vSynchronizeOnServiceStateUnAvailable(tunerSourceId, tunerSrcActivity);
   }
   else if( tunerClientServiceState.bIsServiceAvailable(tunerSourceId) == true )
   {
	   vSynchronizeOnServiceStateAvailable(tunerSourceId, tunerSrcActivity);
   }
   return true;
}

/******************
 *
 */
void clAudioSession::vHandleClearStoredRequest(tuner::common::enSourceId tunerSourceId, tuner::common::enSrcActivity tunerSrcActivity)
{
	tuner::common::enSourceId storedSourceId = tunerClientServiceState.getStoredSource();
	tuner::common::enSrcActivity storedSourceActivity = tunerClientServiceState.getStoredSourceActivity();
	if((storedSourceId != SOURCEID_LIMIT) &&  (storedSourceActivity != SRCACTIVITY_UNKNOWN))
	{
		if((bIsTemporarySourceActive() == false) && (bIsTemporarySourceActivityOffRequest(tunerSourceId, tunerSrcActivity) == false))
		{
			tunerClientServiceState.vClearStoredRequest();
			ETG_TRACE_USR4(("SESSION_MGT: bOnSrcActivity     cleared stored request\n"));
		}
	}
}

/******************
 *
 */
void clAudioSession::vSynchronizeOnServiceStateUnAvailable(tuner::common::enSourceId tunerSourceId, tunerString inputDevice)
{
	ETG_TRACE_USR4(("SESSION_MGT: vSynchronizeOnServiceStateUnAvailable for Source       = %d\n",  ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
	tunerClientServiceState.vStoreRequest(tunerSourceId, inputDevice);
}

bool clAudioSession::bIsTemporarySourceActivityOffRequest(tuner::common::enSourceId tunerSourceId, tuner::common::enSrcActivity tunerSrcActivity)
{
	if( bIsTemporarySource(tunerSourceId) && (tunerSrcActivity == tuner::common::SRCOFF) )
	{
		return true;
	}
	return false;
}

/******************
 *
 */
void clAudioSession::vSynchronizeOnServiceStateAvailable(tuner::common::enSourceId tunerSourceId, tunerString inputDevice)
{
	tunerControl->vOnAllocate(tunerSourceId, inputDevice);
	if(bIsSrcActivityCompleted(tunerSourceId, requestedSession.tunerSrcActivity) == true)
	{
		ETG_TRACE_USR4(("SESSION_MGT: bOnAllocate - vSynchronizeOnServiceStateAvailable for Source       = %d\n",  ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
		return;
	}
	requestedSession.vReset();
	requestedSession.sourceId = tunerSourceId;
	requestedSession.state = tuner::common::SRC_ALLOCATED;
	requestedSession.inputDevice = inputDevice;
	vPrintSessionInformation();
	vSynchronize();
}

/******************
 *
 */
void clAudioSession::vSynchronizeOnServiceStateUnAvailable(tuner::common::enSourceId tunerSourceId,
        tuner::common::enSrcActivity tunerSrcActivity)
{
	ETG_TRACE_USR4(("SESSION_MGT: vSynchronizeOnServiceStateUnAvailable for Source       = %d\n",  ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
	if(bIsTemporarySource(tunerSourceId) == false)
	{
		if(tunerSrcActivity != tuner::common::SRCOFF)
		{
			tunerClientServiceState.vStoreRequest(tunerSourceId, tunerSrcActivity);
		}
		if(tunerSrcActivity != tuner::common::SRCON)
		{
			audioControl.vSendSrcChangeResult(tunerSourceId, tunerSrcActivity);
		}
	}
	else
	{
		if(bIsTemporarySource(tunerSourceId) && tunerSrcActivity == tuner::common::SRCOFF)
		{
			tunerControl->vSetTunerState(oGetTunerState(tunerSrcActivity), tunerSourceId, false);
			tunerControl->vClearAnnouncementStates(tunerSourceId);
		}
		audioControl.vSendSrcChangeResult(tunerSourceId, tunerSrcActivity);
	}
	vPrintSessionInformation();
	vStopTimer();
}

/********************
 *
 */
void clAudioSession::vSynchronizeOnServiceStateAvailable(tuner::common::enSourceId tunerSourceId,
        tuner::common::enSrcActivity tunerSrcActivity)
{
	ETG_TRACE_USR4(("SESSION_MGT: vSynchronizeOnServiceStateAvailable for source         = %d\n",  ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
	if( (tunerSrcActivity == tuner::common::SRCOFF && requestedSession.tunerSrcActivity == tuner::common::SRCACTIVITY_UNKNOWN)
		|| (bIsSrcActivityCompleted(tunerSourceId, tunerSrcActivity) == true))
	{
		audioControl.vSendSrcChangeResult(tunerSourceId, tunerSrcActivity);
		return ;
	}
	if((bIsTemporarySourceActive() == true)  && (bIsTemporarySourceActivityOffRequest(tunerSourceId, tunerSrcActivity) == false))
	{
		if(bIsTemporarySource(tunerSourceId) == false)
		{
			ETG_TRACE_USR4(("SESSION_MGT: bOnSrcActivity     stored request due to temporary source is active\n"));
			tunerClientServiceState.vStoreRequest(tunerSourceId, tunerSrcActivity);
		}
		audioControl.vSendSrcChangeResult(tunerSourceId, tunerSrcActivity);
		return ;
	}
	else if((bIsTemporarySourceActive() == false)  && (bIsTemporarySourceActivityOffRequest(tunerSourceId, tunerSrcActivity) == true))
	{
		/* Fix for - PSARCC30-4892*/
		vResetActiveSession();
		ETG_TRACE_USR4(("SESSION_MGT: bOnSrcActivity() Send an acknowledgement for Announcement-Off request, which was not put through\n"));
		audioControl.vSendSrcChangeResult(tunerSourceId, tunerSrcActivity);
		return ;
	}
	requestedSession.sourceId = tunerSourceId;
	requestedSession.state = tuner::common::SRC_CHANGE_REQUESTED;
	requestedSession.tunerSrcActivity = tunerSrcActivity;
	requestedSession.tunerBand = oGetBand(tunerSourceId);
	vStorePreviousSessionStateBeforeAnnoucement();
	vPrintSessionInformation();
	vSynchronize();
}
void clAudioSession::vStorePreviousSessionStateBeforeAnnoucement()
{
	if((bIsTemporarySource(requestedSession.sourceId)) && (requestedSession.tunerSrcActivity != tuner::common::SRCOFF))
	{
		if(!bIsTemporarySource(activeSession.sourceId))
		{
			previousSessionAfterAnnouncement.sourceId = activeSession.sourceId;
			previousSessionAfterAnnouncement.tunerSrcActivity= activeSession.tunerSrcActivity;
			previousSessionAfterAnnouncement.state = activeSession.state;
			previousSessionAfterAnnouncement.tunerBand=activeSession.tunerBand;
		}

	}
	else if (!bIsTemporarySource(requestedSession.sourceId))
	{
		previousSessionAfterAnnouncement.sourceId = requestedSession.sourceId;
		previousSessionAfterAnnouncement.tunerSrcActivity= requestedSession.tunerSrcActivity;
		previousSessionAfterAnnouncement.state = requestedSession.state;
		previousSessionAfterAnnouncement.tunerBand=requestedSession.tunerBand;
	}
	ETG_TRACE_USR4(("SESSION_MGT: vStorePreviousSessionStateBeforeAnnoucement source=%d,tunerSrcActivity=%d",
			previousSessionAfterAnnouncement.sourceId,previousSessionAfterAnnouncement.tunerSrcActivity));
}
/********************
 *
 */
bool clAudioSession::bIsSrcActivityCompleted(tuner::common::enSourceId tunerSourceId,
        tuner::common::enSrcActivity tunerSrcActivity)
{
	if(tunerSourceId == activeSession.sourceId && tunerSrcActivity == activeSession.tunerSrcActivity && activeSession.bIsSrcChangeCompleted())
	{
		return true;
	}
	return false;
}
/**************************************************************************//**
* 
******************************************************************************/
void clAudioSession::vSynchronize()
{
   if (activeSession.bIsSrcChangeAllowed())
   {
	   vSynchronizeOnSrcChangeAllowed();
   }
   vPrintSessionInformation();
}

/****************************************************************************
*
******************************************************************************/
void clAudioSession::vResetAllSessionOnServiceStateUnAvailable()
{
	ETG_TRACE_USR4(("SESSION_MGT: vResetAllSessionOnServiceStateUnAvailable  source          = %d\n",  ETG_ENUM(TUNER_SOURCE, (tU8)requestedSession.sourceId)));
	ETG_TRACE_USR4(("SESSION_MGT: vResetAllSessionOnServiceStateUnAvailable  available       = %d\n",  ETG_ENUM(TUNER_CLIENT_AVAILABLITY, 
		(tU8)(tunerClientServiceState.bIsServiceAvailable(requestedSession.sourceId) == true) ? 0: 1)));
	ETG_TRACE_USR4(("SESSION_MGT: vResetAllSessionOnServiceStateUnAvailable  source activity = %d\n",  ETG_ENUM(TUNER_SRCACTIVITY, (tU8)requestedSession.tunerSrcActivity)));
	tunerClientServiceState.vClearTunerState(oGetTunerClient(activeSession.sourceId));
	activeSession.vReset();
	requestedSession.vReset();
	muteRequest.vReset();
}

/****************************************************************************
*
******************************************************************************/

void clAudioSession::vSynchronizeOnSrcChangeAllowed()
{
	   if(bIsMuteRequestPending())
	   {
		   vSynchronizeOnMuteRequestPending();
	   }
	   else if (requestedSession.bIsSrcChangeCompleted() && !bIsDirectFrequencyRequestToBeProcessed() && !bIsBandChangeResultPending())
	   {
		   /*If received tuner state is not expected, do not send source change result */
		   if(activeSession.bIsSrcChangeFailed() == false)
		   {
			   vOnSrcChangeCompleted();
		   }
		   else
		   {
			   activeSession.state = tuner::common::SRC_CHANGE_COMPLETED;
			   vStopTimer();
		   }
	   }
	   else if (!activeSession.bIsSrcChangeInProgress() && (!muteRequest.bIsRequestInProgress() || !muteRequest.bBlockSourceChange))
	   {
		  vSynchronizeOnNoActiveSource();
	   }
	   else if (directFrequencyRequest.bIsRequestPending())
	   {
		   if (bProcessDirectFrequencyChange())
		   {
			   vSetDirectFrequency();
		   }
	   }
}

bool clAudioSession::bIsDirectFrequencyRequestToBeProcessed()
{
	if( directFrequencyRequest.bIsRequestPending() &&
			(directFrequencyRequest.sourceId != requestedSession.sourceId) && requestedSession.bSrcOff()
		)
	{
		return false;
	}
	else if(directFrequencyRequest.bIsRequestCompleted())
	{
		return false;
	}
	return true;
}

void clAudioSession::vSynchronizeOnMuteRequestPending()
{
	/*PSARCCB-6359 - If Mute tuning actions are in progress, forward demute request to ARL during source change.
	  PSARCCB-5529 - Forward mute request to ARL during source change
	*/
	  if (activeSession.bIsSrcChangeInProgress() && (bSendResponseDuringSrcChange() ||  muteRequest.bDeMute()))
	  {
		  /*NCG3D-15850 - If announcement source activity is in process, get the source id from band */
		  bIsTemporarySource(activeSession.sourceId)? vRequestMute(oGetTunerSourceId(activeSession.tunerBand)) : vRequestMute(activeSession.sourceId);
	  }
	  /*If source is in PAUSE state before temporary source putting through, forward mute request received after temporary source OFF */
	  else if(bIsTemporarySource(activeSession.sourceId) && activeSession.bSrcOff())
	  {
		  vRequestMute(oGetTunerSourceId(activeSession.tunerBand));
	  }
	  else
	  {
		  bIsTemporarySource(activeSession.sourceId)? vRequestMute(oGetTunerSourceId(activeSession.tunerBand)) : vRequestMute(activeSession.sourceId);
	  }
}

void clAudioSession::vSynchronizeOnNoActiveSource()
{
	  if (activeSession.bIsSrcChangeFailed())
	  {
		  vResetActiveSession();
		  vSynchronize();
	  }
	  else  if (bIsNewSrcRequested())
	  {
		 if (bProcessSrcChange())
		 {
			vStartSrcChange();
		 }
		 else if (bIsErrorState())
		 {
			vOnError();
		 }
	  }
	  else if (bIsStateChangeRequired())
	  {
		 vStartStateChange();
	  }

}

/**************************************************************************//**
* 
******************************************************************************/
bool clAudioSession::bProcessSrcChange()
{
   if (activeSession.state == tuner::common::SRC_CHANGE_COMPLETED)
   {
      return true;
   }
   return false;
}


/**************************************************************************//**
* 
******************************************************************************/
bool clAudioSession::bIsNewSrcRequested()
{
   if (requestedSession.sourceId == activeSession.sourceId)
   {
      return false;
   }
   return true;
}


/**************************************************************************//**
* 
******************************************************************************/
void clAudioSession::vStartSrcChange()
{
   activeSession.sourceId = requestedSession.sourceId;
   activeSession.tunerSrcActivity = tuner::common::SRCACTIVITY_UNKNOWN;
   if(requestedSession.bSrcOff() && !bIsTemporarySource(requestedSession.sourceId))
   {
	   activeSession.tunerBand = requestedSession.tunerBand;
   }
   vPrintSessionInformation();
   vSynchronize();
}


/**************************************************************************//**
*
******************************************************************************/
bool clAudioSession::bIsStateChangeRequired()
{
   if (requestedSession.state != activeSession.state)
   {
      return true;
   }
   return false;
}


/**************************************************************************//**
* 
******************************************************************************/
bool clAudioSession::bIsErrorState()
{
   if (activeSession.sourceId != requestedSession.sourceId)
   {
      if (activeSession.bIsSrcAllocate())
      {
         return true;
      }
   }
   return false;
}


/**************************************************************************//**
* 
******************************************************************************/
void clAudioSession::vStartStateChange()
{
   if (requestedSession.bIsSrcAllocate() &&  ( activeSession.bIsSrcChangeCompleted() ) )
   {
      vOnAllocate();
   }
   else if (requestedSession.bIsSrcChangeRequested() && (activeSession.bIsSrcAllocate() || activeSession.bIsSrcChangeCompleted() || activeSession.bIsSrcChangeFailed()))
   {
      vOnNewStateRequested();
   }
}


/**************************************************************************//**
* 
******************************************************************************/
void clAudioSession::vOnAllocate()
{
   activeSession.state = tuner::common::SRC_ALLOCATED;
   activeSession.inputDevice = requestedSession.inputDevice;
   //tunerControl->vOnAllocate(activeSession.sourceId, activeSession.inputDevice);
}


/**************************************************************************//**
* 
******************************************************************************/
void clAudioSession::vOnNewTunerBand(tuner::common::enBand tunerBand)
{
   ETG_TRACE_USR4(("SESSION_MGT: vOnNewTunerBand      band           = %d\n", ETG_ENUM(TUNER_BAND, (tU8)tunerBand)));
   ETG_TRACE_USR4(("SESSION_MGT: vOnNewTunerBand      active band    = %d\n", ETG_ENUM(TUNER_BAND, (tU8)activeSession.tunerBand)));
   ETG_TRACE_USR4(("SESSION_MGT: vOnNewTunerBand      requested band = %d\n", ETG_ENUM(TUNER_BAND, (tU8)oGetBand(requestedSession.sourceId))));
   if (activeSession.tunerBand != oGetBand(requestedSession.sourceId))
   {
	   if(!bIsTemporarySource(requestedSession.sourceId))
	   {
		   activeSession.tunerBand = oGetBand(requestedSession.sourceId);
		   /*set state as completed only if tuner is in FG (PSARCCB-8943)*/
		   /*gds5kor, mhv6kor: NCG3D-45655: fast band issue: make source cahnge complaeted only for ON/PAUSE activity.
		                       for OFF activity, band change method result is not considered */
		   if(getActiveTunerState(oGetTunerClient(requestedSession.sourceId)) == tuner::common::TUNERTOFG && (requestedSession.bSrcOn() || requestedSession.bSrcPause()))
		   {
			   requestedSession.state= tuner::common::SRC_CHANGE_COMPLETED;
		   }
		   else if(requestedSession.bSrcOff())
		   {
			   activeSession.state = tuner::common::SRC_CHANGE_COMPLETED;
			   requestedSession.state = tuner::common::SRC_CHANGE_REQUESTED;
		   }
	   }
	   /* For announcment sources, it not required to check band change method result.
	      Near future, it will removed completely for all sources as band change method removed */
	   vPrintSessionInformation();
	   vSynchronize();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vOnNewStateRequested()
{
   if (activeSession.tunerSrcActivity == requestedSession.tunerSrcActivity)
   {
      requestedSession.state = tuner::common::SRC_CHANGE_COMPLETED;
   }
   else if((activeSession.bSrcOn() && requestedSession.bSrcPause()) || (activeSession.bSrcPause() && requestedSession.bSrcOn()))
   {
	   // It is required for DRM explicitely to delete thread (logical frame data)
	   tunerControl->vSetModulationAndSrcActivity(requestedSession.sourceId, requestedSession.tunerSrcActivity);
	   requestedSession.state = tuner::common::SRC_CHANGE_COMPLETED;
   }
   else if((requestedSession.bSrcPause() || requestedSession.bSrcOn()) &&
		   (tunerClientServiceState.oGetCurrentTunerState(oGetTunerClient(requestedSession.sourceId)) == tuner::common::TUNERTOFG)
		   && !bIsTemporarySource(requestedSession.sourceId))
   {
	   requestedSession.state = tuner::common::SRC_CHANGE_COMPLETED;
	   vActivateTunerSource();
   }
   else
   {
      activeSession.state = tuner::common::SRC_CHANGE_IN_PROGRESS;
      vOnNewSrcactivity();
      vStartTimer();
   }
   vPrintSessionInformation();
   vSynchronize();
   
}


/**************************************************************************//**
* 
******************************************************************************/
void clAudioSession::vOnError()
{
   activeSession.state = tuner::common::SRC_CHANGE_FAILED;
   vPrintSessionInformation();
   vSynchronize();
}


/**************************************************************************//**
* 
******************************************************************************/
void clAudioSession::vOnNewSrcactivity()
{	
	if(bIsTemporarySourceActivityOffRequest())
	{
		if((tunerClientServiceState.oGetPreviousTunerState(tuner::common::AMFMTUNER) == tuner::common::TUNERSTATE_UNKNOWN)
				&& (activeSession.bIsSrcChangeInProgress() == true))
		{
			ETG_TRACE_USR4(("SESSION_MGT: vOnNewSrcactivity      UpdatetunerState  to      = %d\n",
					ETG_ENUM(TUNER_STATE, (tU8)tunerClientServiceState.oGetCurrentTunerState(tuner::common::AMFMTUNER)) ));
			tunerControl->vSetTunerState(tunerClientServiceState.oGetCurrentTunerState(tuner::common::AMFMTUNER), requestedSession.sourceId, true);
		}
		else
		{
			ETG_TRACE_USR4(("SESSION_MGT: vOnNewSrcactivity      UpdatetunerState  to      = %d\n",
					ETG_ENUM(TUNER_STATE, (tU8)oUpdateTunerStatus()) ));
			tunerControl->vSetTunerState(oUpdateTunerStatus(), requestedSession.sourceId, true);
		}
	}
	else
	{
		tunerControl->vSetTunerState(oGetTunerState(requestedSession.tunerSrcActivity), requestedSession.sourceId, false);
	}
    vActivateTunerSource();
    tunerControl->vSetModulationAndSrcActivity(requestedSession.sourceId, requestedSession.tunerSrcActivity);
}
/****************************************************************************
*
******************************************************************************/
tuner::common::enTunerState clAudioSession::oUpdateTunerStatus()
{
	if(oGetTunerClient(requestedSession.sourceId) == tuner::common::AMFMTUNER)
	{
		return tunerClientServiceState.oGetPreviousTunerState(tuner::common::AMFMTUNER);
	}
	return oGetTunerState(requestedSession.tunerSrcActivity);
}

/****
 *
 */
bool clAudioSession::bIsTemporarySourceActivityOffRequest()
{
	if(bIsTemporarySource(requestedSession.sourceId) && requestedSession.tunerSrcActivity == tuner::common::SRCOFF)
	{
		return true;
	}
	return false;
}
/**************************************************************************//**
* 
******************************************************************************/
void clAudioSession::vActivateTunerSource()
{
	if(m_bIsSBRVariant == false)
	{
		if(oGetTunerState(requestedSession.tunerSrcActivity) == tuner::common::TUNERTOFG &&
				activeSession.tunerBand != requestedSession.tunerBand)
		{
			if(requestedSession.tunerBand !=tuner::common::BANDDAB)
			{
				//tunerControl->vChangeBand(requestedSession.sourceId, oGetBand(requestedSession.sourceId)); band change requested along with fg_bg request
			}
			/*
			 * For DAB source, we don't have band change , Hence if DAB tuner is in FG,
			 * updated the tuner band as DAB*/
			else
			{
				activeSession.tunerBand=tuner::common::BANDDAB;
			}
		}
	}
}


/**************************************************************************//**
* 
******************************************************************************/
void clAudioSession::vInformSourceAvailability(tuner::common::enSourceId tunerSourceId,
		tuner::common::enAudioSrcAvailability AudioSrcAvailability ,tuner::common::enAudioSourceAvailabiltyReason AudioSourceAvailabiltyReason)
{
	ETG_TRACE_USR4(("SESSION_MGT:vInformSourceAvailability tunerSourceId=%d AudioSrcAvailability=%d",tunerSourceId,AudioSrcAvailability));
	if((tunerSourceId == tuner::common::DAB) && (AudioSrcAvailability ==tuner::common::AUDIO_SRC_PLAYABLE))
	{
		vBGorFGLearnForever(activeSession.sourceId, activeSession.tunerSrcActivity);
	}
	audioControl.vInformSourceAvailability(tunerSourceId,AudioSrcAvailability,AudioSourceAvailabiltyReason);
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vOnSrcChangeCompleted()
{
   if(! (activeSession.state == tuner::common::SRC_CHANGE_COMPLETED &&
         activeSession.tunerSrcActivity == requestedSession.tunerSrcActivity))
   {
         activeSession.state = tuner::common::SRC_CHANGE_COMPLETED;
         if(!bIsTemporarySource(requestedSession.sourceId))
         {
        	 activeSession.tunerBand = oGetBand(requestedSession.sourceId);
         }
         activeSession.tunerSrcActivity = requestedSession.tunerSrcActivity;
		 /*Required for PSA to update band info as none if no source are in FG (PSARCCB-9100) */
         ETG_TRACE_USR4(("SESSION_MGT: vOnSrcChangeCompleted tunerSrcActivity=%d",ETG_ENUM(TUNER_SRCACTIVITY, (tU8)activeSession.tunerSrcActivity)));
		 (activeSession.tunerSrcActivity != tuner::common::SRCOFF) ? tunerControl->vOnSrcChangeCompleted(activeSession.sourceId, false) : tunerControl->vOnSrcChangeCompleted(activeSession.sourceId, true);
         
         if(muteRequest.bIsToBeProcessStoredMuteRequest(activeSession.sourceId, activeSession.tunerSrcActivity, activeSession.tunerBand))
         {
        	 vProcessStoredMuteRequest(muteRequest.oGetStoredSourceId(), muteRequest.oGetStoredMuteInterrupt());
         }
		 vSendDemuteRequest();
		 audioControl.vSendSrcChangeResult(activeSession.sourceId, activeSession.tunerSrcActivity);
		 if(m_bIsSBRVariant == false)
		 {
			 vBGorFGLearnForever(activeSession.sourceId, activeSession.tunerSrcActivity);
		 }
         timerExpiryCount=0;
         ETG_TRACE_USR1(("CCR.ENDING.SOURCE_ACTIVE:vOnSrcChangeCompleted"));
   }
   vStopTimer();
   tuner::common::enSourceId storedSourceId = tunerClientServiceState.getStoredSource();
   tuner::common::enSrcActivity storedSourceActivity = tunerClientServiceState.getStoredSourceActivity();
   if(bIsTemporarySource(activeSession.sourceId) && activeSession.bIsSrcChangeCompleted() && activeSession.bSrcOff()
		   && (storedSourceId != SOURCEID_LIMIT &&  storedSourceActivity != SRCACTIVITY_UNKNOWN))
   {
	   bOnSrcActivity(storedSourceId, storedSourceActivity);
   }
}

/**************************************************************************//**
* send demute request to ARL explicitely for SRCOFF activity.
******************************************************************************/
void clAudioSession::vSendDemuteRequest()
{
	if(activeSession.bIsSrcChangeCompleted() && activeSession.bSrcOff() && !bIsTemporarySource(activeSession.sourceId))
	{
		muteRequest.muteState = tuner::common::DEMUTE;
		audioControl.vRequestMute(activeSession.sourceId, tuner::common::DEMUTE);
	}
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vStartTimer()
{
   OSAL_s32TimerSetTime( timer, getMaxWaitTime(requestedSession.tunerSrcActivity), 0 );
   timerExpiryCount = (unsigned char) (timerExpiryCount+1);
	ETG_TRACE_USR4(("SESSION_MGT: **********vStartTimer********** timerExpiryCount=%d",timerExpiryCount));
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vStopTimer()
{
   OSAL_s32TimerSetTime( timer, (tU32) 0, 0 );
	ETG_TRACE_USR4(("SESSION_MGT: **********vStopTimer********** timerExpiryCount=%d",timerExpiryCount));
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vOnTimerExpiry()
{
   ETG_TRACE_USR4(("SESSION_MGT: **********vOnTimerExpiry********** u8TimerExpireyCount=%d",instance()->timerExpiryCount));
   if((int)instance()->timerExpiryCount >= (int)MAX_SESSIONMANAGEMENT_TIMER_EXPIRY_COUNT)
   {
	   instance()->timerExpiryCount=0;
	   instance()->vPrintSessionInformation();
	   instance()->tunerClientServiceState.vClearTunerState(AMFMTUNER);
	   instance()->tunerClientServiceState.vClearTunerState(DABTUNER);
	   instance()->tunerControl->vClearAnnouncementStates(instance()->activeSession.sourceId);
	   instance()->activeSession.vReset();
	   instance()->requestedSession.vReset();
	   instance()->tunerClientServiceState.vClearStoredRequest();
	   return;
   }
   instance()->requestedSession.state = tuner::common::SRC_CHANGE_REQUESTED;
   instance()->activeSession.state = tuner::common::SRC_CHANGE_FAILED;
   instance()->vPrintSessionInformation();
   instance()->vSynchronize();
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vResetActiveSession()
{
   tuner::common::enBand previousTunerBandBeforeAnnouncment = activeSession.tunerBand;
   activeSession.vReset();
   /*If source change failed, do not reset band value. This will affect especially during transition from anno src to normal src*/
   ETG_TRACE_USR4(("SESSION_MGT: vResetActiveSession retained activeTuner band - %d\n", previousTunerBandBeforeAnnouncment));
   activeSession.tunerBand = previousTunerBandBeforeAnnouncment;
   vStopTimer();
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vOnNewTunerState(tuner::common::enTunerState tunerState, tuner::common::enTunerClient tunerClient, unsigned char audioSink)
{
   ETG_TRACE_USR4(("SESSION_MGT: vOnNewTunerState     tunerState     = %d tunerClient = %d, audioSink = %d\n", 
	   ETG_ENUM(TUNER_STATE, (tU8)tunerState), ETG_ENUM(TUNER_CLIENT, (tU8)tunerClient), audioSink));

   if(requestedSession.sourceId == tuner::common::SOURCEID_LIMIT || activeSession.sourceId == tuner::common::SOURCEID_LIMIT)
   {
	   /* If there are no source activity request, ignore tuner status. */
	   ETG_TRACE_USR4(("SESSION_MGT: vOnNewTunerState     ignore tuner status requested source = %d, active source     = %d",
	   	   ETG_ENUM(TUNER_SOURCE, (tU8)requestedSession.sourceId), ETG_ENUM(TUNER_SOURCE, (tU8)activeSession.sourceId)));
	   return;
   }
   tuner::common::enTunerState previousTunerState = tunerClientServiceState.oGetPreviousTunerState(tunerClient);
   vUpdateTunerState(tunerState, tunerClient, audioSink);
   if ((requestedSession.sourceId == activeSession.sourceId) && (bIsProperTunerStateReceived(tunerClientServiceState.oGetCurrentTunerState(tunerClient), previousTunerState, audioSink) == true))
   {
      if(bIsStaticSource(requestedSession.sourceId) || bIsDynamicSource(requestedSession.sourceId))
      {
         vOnNewTunerStateForNormalSources();
      }
      else
      {
         vOnNewTunerStateForTemporarySources();
      }

   }
   else
   {
	   activeSession.state = tuner::common::SRC_CHANGE_FAILED;
	   vPrintSessionInformation();
	   vSynchronize();

   }
}

/**************************************************************************//**
*
******************************************************************************/
bool clAudioSession::bIsProperTunerStateReceived(tuner::common::enTunerState tunerState, tuner::common::enTunerState previousTunerState, unsigned char audioSink)
{
    /*Check whether proper tuner state received, if not set source change as failed and resend FG/BG */
	if(bIsTemporarySource(requestedSession.sourceId) == false)
	{
		if((tunerState == tuner::common::TUNERTOFG) && (requestedSession.bSrcOn() || requestedSession.bSrcPause()))
		{
			return true;
		}
		else if((tunerState == tuner::common::TUNERTOBG) && (requestedSession.bSrcOff()))
		{
			return true;
		}
	}
	else if(oGetTunerClient(requestedSession.sourceId) == tuner::common::AMFMTUNER)
	{
		if((tunerState == tuner::common::TAFG) && (requestedSession.bSrcOn() || requestedSession.bSrcPause()))
		{
			return true;
		}
		else if((tunerState == previousTunerState) && (requestedSession.bSrcOff()))
		{
			return true;
		}
		/* NCG3D-15850, NCG3D-17144, PSARCCB-8547 - if TA OFF activity is received during TA ON activity is in process (TAFG methosStart didn't send yet)
		 * do not check for previous tunerState. */
		else if((requestedSession.bSrcOff()) && (previousTunerState == tuner::common::TUNERSTATE_UNKNOWN) &&
				(activeSession.tunerSrcActivity == tuner::common::SRCACTIVITY_UNKNOWN) && (audioSink == USR1_SWITCHED_TOFG))
		{
			return true;
		}
	}
	else if(oGetTunerClient(requestedSession.sourceId) == tuner::common::DABTUNER)
	{
		return true;
	}
	return false;
}

/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vUpdateTunerState(tuner::common::enTunerState tunerState, tuner::common::enTunerClient tunerClient, unsigned char audioSink)
{
	/*Store tuner state for temporary source request and clear upon receiving temporary source Off request*/
	if(!(audioSink == USR1_SWITCHED_TOFG) && (bIsTemporarySourceActivityOffRequest() == false))
	{
		if(tunerClient == tuner::common::AMFMTUNER)
		{
			tunerState = tuner::common::TAFG;
		}
		/*if Current tuner state is TAFG, do not store */
		if(tunerClientServiceState.oGetCurrentTunerState(tunerClient) != tuner::common::TAFG)
		{
			tunerClientServiceState.vStorePreviousTunerState();
		}
	}
	else if(bIsTemporarySourceActivityOffRequest())
	{
		tunerClientServiceState.vClearPreviousTunerState();
	}

	/*for DAB announcement FG/BG is not required */
	if(getRequestedSessionSrc() != tuner::common::DABANNOUNCEMENTS)
	{
		tunerClientServiceState.vOnNewTunerState(tunerClient, tunerState);
	}
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vPrintSessionInformation()
{
   ETG_TRACE_USR4(("SESSION_MGT: ACTIVESESSION   source              = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)activeSession.sourceId)));
   ETG_TRACE_USR4(("SESSION_MGT: ACTIVESESSION   sessionState        = %d\n", ETG_ENUM(TUNER_SESSION_STATE, (tU8)activeSession.state)));
   ETG_TRACE_USR4(("SESSION_MGT: ACTIVESESSION   srcActivity         = %d\n", ETG_ENUM(TUNER_SRCACTIVITY, (tU8)activeSession.tunerSrcActivity)));
   ETG_TRACE_USR4(("SESSION_MGT: ACTIVESESSION   tunerband         = %d\n", ETG_ENUM(TUNER_BAND, (tU8)activeSession.tunerBand)));
   ETG_TRACE_USR4(("SESSION_MGT: REQSESSION      source              = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)requestedSession.sourceId)));
   ETG_TRACE_USR4(("SESSION_MGT: REQSESSION      sessionState        = %d\n", ETG_ENUM(TUNER_SESSION_STATE, (tU8)requestedSession.state)));
   ETG_TRACE_USR4(("SESSION_MGT: REQSESSION      srcActivity         = %d\n", ETG_ENUM(TUNER_SRCACTIVITY, (tU8)requestedSession.tunerSrcActivity)));
   ETG_TRACE_USR4(("SESSION_MGT: REQSESSION   tunerband         = %d\n", ETG_ENUM(TUNER_BAND, (tU8)requestedSession.tunerBand)));
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vOnNewTunerStateForNormalSources()
{
   /* TODO: Find it correct state can be provided by tuner, else all tuner state changes to be considered valid */
	if(requestedSession.tunerBand == activeSession.tunerBand)
	{
		requestedSession.state = tuner::common::SRC_CHANGE_COMPLETED;
	}
   vPrintSessionInformation();
   vSynchronize();
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vOnNewTunerStateForTemporarySources()
{
   requestedSession.state = tuner::common::SRC_CHANGE_COMPLETED;
   vPrintSessionInformation();
   vSynchronize();
}


/**************************************************************************//**
*
******************************************************************************/
bool clAudioSession::bIsBandChangeResultPending()
{
	if(!bIsTemporarySource(requestedSession.sourceId))
	{
	   if( (requestedSession.bSrcOn() || requestedSession.bSrcPause()) &&
			 (oGetBand(requestedSession.sourceId) != activeSession.tunerBand ))
	   {
		  return true;
	   }
		if((requestedSession.sourceId == tuner::common::DRM) &&
			(requestedSession.state != tuner::common::SRC_CHANGE_COMPLETED))
		{
			return true;
		}
	}
   return false;
}

/*******
 *
 */
void clAudioSession::vProcessStoredMuteRequest(tuner::common::enSourceId tunerSourceId, bool bInterrupt)
{
	muteRequest.vClearStoredMuteRequest();
	vRequestMute(tuner::common::MUTE, bInterrupt, tunerSourceId);
}

/**************************************************************************//**
*
******************************************************************************/
bool clAudioSession::bOnMuteState(tuner::common::enSourceId tunerSourceId, tuner::common::enMuteState tunerMuteState)
{
   ETG_TRACE_USR4(("SESSION_MGT: bOnMutestate    muteState           = %d\n", ETG_ENUM(TUNER_MUTESTATES, (tU8)tunerMuteState)));
   ETG_TRACE_USR4(("SESSION_MGT: bOnMutestate    source              = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
   /* Is any temporary source is playing, simulate cancel message to respected tuner and then forward mute result */
   
	   if (muteRequest.muteState == tunerMuteState)
	   {
		  muteRequest.state = clMuteRequest::REQUEST_COMPLETED;
		  muteRequest.muteSourceId = tunerSourceId;
		  tunerControl->vOnMuteState(tunerSourceId, tunerMuteState);
	   }
	   vPrintSessionInformation();
	   vSynchronize();
   return true;
}

bool clAudioSession::bIsTemporarySourceActive()
{
    ETG_TRACE_USR4(("SESSION_MGT: bIsTemporarySourceActive  source              = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)activeSession.sourceId)));
	if(bIsTemporarySource(activeSession.sourceId) && (activeSession.bSrcOn() || activeSession.bSrcPause()) && activeSession.bIsSrcChangeCompleted())
	{
		return true;
	}
	/* PSARCCB-8547 - If temporary source ON/PAUSE activity is in process,
	    * simulate cancel message to respected tuner and store mute request and forward to ARL when requested SRC become active*/
	else if(bIsTemporarySource(requestedSession.sourceId) && (requestedSession.bSrcOn() || requestedSession.bSrcPause())
			&& activeSession.bIsSrcChangeInProgress())
	{
		return true;
	}
	return false;
}

bool clAudioSession::bIsPreviousTunerStateForeGround(tuner::common::enSourceId tunerSourceId)
{
	if(tunerClientServiceState.oGetPreviousTunerState(oGetTunerClient(tunerSourceId)) == tuner::common::TUNERTOFG)
	{
		return true;
	}
	return false;
}
/****************************************************************************
*
******************************************************************************/
void clAudioSession::vRequestMute(tuner::common::enMuteState muteStateRequested, bool bInterrupt, tuner::common::enSourceId tunerSourceId)
{
   ETG_TRACE_USR4(("SESSION_MGT: vRequestMute    muteStateRequested  = %d\n", ETG_ENUM(TUNER_MUTESTATES, (tU8)muteStateRequested)));
   ETG_TRACE_USR4(("SESSION_MGT: vRequestMute    bInterrupt          = %d\n", bInterrupt));
   ETG_TRACE_USR4(("SESSION_MGT: vRequestMute    tunerSourceId       = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
   if(tunerSourceId == tuner::common::SOURCEID_LIMIT)
   {
	   return;
   }
   /*Do not send MUTE request again if already reuqested to ARL. */
   if(((muteStateRequested == muteRequest.muteState) && (muteRequest.muteState== tuner::common::MUTE)) && (tunerSourceId == muteRequest.muteSourceId) &&
	   (muteRequest.bIsRequestInProgress() || muteRequest.bIsRequestCompleted()))
   {
	   ETG_TRACE_USR4(("SESSION_MGT: vRequestMute(), repeated mute request = %d, source = %d, muteState = %d\n", ETG_ENUM(TUNER_MUTESTATES, (tU8)muteStateRequested), ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId), muteRequest.muteState));
	   return;
   }
   vPrintSessionInformation();
   /* Is any temporary source is playing, simulate cancel message to respected tuner and store mute request and forward to ARL when requested SRC become active */
   if(bIsTemporarySourceActive() == true)
   {
	   /*Check previous tuner state for AMFMTuner and Current Tuner state for DABTuner.
	    * Unlike AMFMTuner, for DABTuner Announcments there is no FG/BG activity also DAB tuner state is in FG only
		*/
	   if ( (muteStateRequested == tuner::common::MUTE) &&
			   ( (bIsPreviousTunerStateForeGround(tunerSourceId) && oGetTunerClient(tunerSourceId) == tuner::common::AMFMTUNER) ||
				 (oGetTunerClient(tunerSourceId) == tuner::common::DABTUNER && tunerClientServiceState.oGetCurrentTunerState(tuner::common::DABTUNER) == tuner::common::TUNERTOFG)
			) )
	   {
		   tunerControl->vCancelAnnouncement(activeSession.sourceId);
  		   muteRequest.vStoreMuteRequest(tunerSourceId, bInterrupt);
  	   }
	   /*PSARCC30-2210  process demute always */
	   else if (muteStateRequested == tuner::common::DEMUTE)
	   {
		   muteRequest.muteState = muteStateRequested;
		   muteRequest.state = tuner::sessionManagement::audio::clMuteRequest::REQUEST_PENDING;
		   muteRequest.bBlockSourceChange = bInterrupt;
		   vSynchronize();
	   }
   }
   else
   {
	   if(bIsTemporarySourceActivityOffRequest() && (muteStateRequested == tuner::common::MUTE)
			   && (!activeSession.bIsSrcChangeCompleted() || !requestedSession.bIsSrcChangeCompleted()))
	   {
		   muteRequest.vStoreMuteRequest(tunerSourceId, bInterrupt);
	   }
	   else
	   {
		   muteRequest.muteState = muteStateRequested;
		   muteRequest.state = tuner::sessionManagement::audio::clMuteRequest::REQUEST_PENDING;
		   muteRequest.bBlockSourceChange = bInterrupt;
	   }
	   vSynchronize();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vOnDirectFrequency(tuner::common::enSourceId tunerSourceId, unsigned int directFrequency)
{
   ETG_TRACE_USR4(("SESSION_MGT: vOnDirectFrequency()      source              = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
   ETG_TRACE_USR4(("SESSION_MGT: vOnDirectFrequency()      directFrequency     = %d\n", directFrequency));

   if(tunerSourceId == tuner::common::SOURCEID_LIMIT)
   {
	   ETG_TRACE_USR4(("SESSION_MGT: vOnDirectFrequency()     Invalid request\n"));
	   return;
   }
   directFrequencyRequest.vReset();
   if (activeSession.sourceId == tunerSourceId && activeSession.bIsSrcChangeCompleted())
   {
	   tunerControl->vSetFrequency(tunerSourceId, oGetBand(tunerSourceId), directFrequency);
   }
   else
   {
	   directFrequencyRequest.frequency = directFrequency;
	   directFrequencyRequest.sourceId = tunerSourceId;
	   directFrequencyRequest.state = clDirectFrequencyRequest::REQUEST_PENDING;
   }
}

/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vSetDirectFrequency()
{
	tunerControl->vSetFrequency(directFrequencyRequest.sourceId, oGetBand(directFrequencyRequest.sourceId), directFrequencyRequest.frequency);
	directFrequencyRequest.state = clDirectFrequencyRequest::REQUEST_IN_PROGRESS;
}


/**************************************************************************//**
*
******************************************************************************/
bool clAudioSession::bProcessDirectFrequencyChange()
{
	if( (requestedSession.bSrcOn() || requestedSession.bSrcPause()) && activeSession.sourceId == directFrequencyRequest.sourceId )
	{
		return true;
	}
	return false;
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vOnDirectFrequencyResult()
{
	directFrequencyRequest.state = clDirectFrequencyRequest::REQUEST_COMPLETED;
	vPrintSessionInformation();
	vSynchronize();
}


/**************************************************************************//**
*
******************************************************************************/
bool clAudioSession::bIsMuteRequestPending()
{
	if (muteRequest.bIsRequestPending())
	{
		return true;
	}
	return false;
}


/**************************************************************************//**
*
******************************************************************************/
bool clAudioSession::bSendResponseDuringSrcChange()
{
	if (muteRequest.bBlockSourceChange)
	{
		return true;
	}
	return false;
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vRequestMute(tuner::common::enSourceId sourceId)
{
	if(sourceId != SOURCEID_LIMIT)
	{
		muteRequest.state = clMuteRequest::REQUEST_IN_PROGRESS;
		muteRequest.muteSourceId = sourceId;
		ETG_TRACE_USR4(("SESSION_MGT: vRequestMute    muteStateRequested  = %d", ETG_ENUM(TUNER_MUTESTATES, (tU8)muteRequest.muteState)));
		ETG_TRACE_USR4(("SESSION_MGT: vRequestMute    sourceId          = %d", ETG_ENUM(TUNER_SOURCE, (tU8)sourceId)));
		audioControl.vRequestMute(sourceId, muteRequest.muteState);
	}
}


/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vSendMuteResult()
{
    muteRequest.state = clMuteRequest::REQUEST_COMPLETED;
	tuner::common::enSourceId sourceId=activeSession.sourceId;
    if(bIsTemporarySource(sourceId))
    {
    	sourceId=oGetTunerSourceId(activeSession.tunerBand);
    }
    tunerControl->vOnMuteState(sourceId, muteRequest.muteState);
}
/**************************************************************************//**
*
******************************************************************************/
bool clAudioSession::bIsPauseActivity()
{
	if(activeSession.bSrcPause() && !bIsTemporarySource(activeSession.sourceId))
	{
		return true;
	}
	return false;
}
/**************************************************************************//**
*
******************************************************************************/
bool clAudioSession::bIsOnOrPauseActivityInProgress()
{
	if(requestedSession.bSrcPause() && (requestedSession.bIsSrcChangeInProgress() || requestedSession.bIsSrcChangeRequested())
			&& !bIsTemporarySource(activeSession.sourceId))
	{
		return true;
	}
	return false;
}
/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vHandleClearStoredRequest(bool bIsClearStoredData)
{
	if(bIsClearStoredData == true)
	{
		ETG_TRACE_USR4(("SESSION_MGT: vOnServiceState UnAvailable: Cleared Stored request"));
		tunerClientServiceState.vClearStoredRequest();
	}
}
/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vHandleStoreRequest()
{
	if(bIsPauseActivity() == true)
	{
		tunerClientServiceState.vStoreRequest(activeSession.sourceId, activeSession.tunerSrcActivity);
	}
	else if(bIsOnOrPauseActivityInProgress() == true)
	{
		tunerClientServiceState.vStoreRequest(requestedSession.sourceId, requestedSession.tunerSrcActivity);
	}
}
/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vHandleAcknowledgeToArl()
{
	if(requestedSession.bIsSrcChangeInProgress() || requestedSession.bIsSrcChangeRequested())
	{
		audioControl.vSendSrcChangeResult(requestedSession.sourceId, requestedSession.tunerSrcActivity);
	}
}
/**************************************************************************//**
*
******************************************************************************/
void clAudioSession::vOnServiceState(tuner::common::enTunerClient tunerClient, tuner::common::enTunerClientAvailability tunerClientAvailable)
{
	/*required to address race condition for service availability and bOnSrcActivity during startup */
	bool bIsClearStoredData = ((tunerClientAvailable == tuner::common::UNAVAILABLE) && (tunerClientServiceState.bIsServiceUnKnown(tunerClientServiceState.getStoredSource()) == false))? true: false;

	tunerClientServiceState.vOnServiceState(tunerClient, tunerClientAvailable);
	if(tunerClientAvailable == tuner::common::UNAVAILABLE)
	{
		ETG_TRACE_USR4(( "SESSION_MGT: vOnServiceState Availability = %d for Source = %d\n",
				ETG_ENUM(TUNER_CLIENT_AVAILABLITY, (tU8)(tunerClientServiceState.oGetCurrentTunerState(oGetTunerClient(activeSession.sourceId)))),
				ETG_ENUM(TUNER_SOURCE, (tU8)activeSession.sourceId) ));
		/* switch off sources with respective client service unavailable */
		if(tunerClientServiceState.bIsServiceUnAvailable(activeSession.sourceId) == true)
		{
			/* store if current Source Activity is PAUSE,
			 * since it is required to put tuner to FG after service becomes available */
			vHandleClearStoredRequest(bIsClearStoredData);
			vHandleStoreRequest();
			vHandleAcknowledgeToArl();
			instance()->tunerControl->vClearAnnouncementStates(instance()->activeSession.sourceId);
			vResetAllSessionOnServiceStateUnAvailable();
			vStopTimer();
		}
	}
	else if(tunerClientAvailable == tuner::common::AVAILABLE)
	{
		ETG_TRACE_USR4(("SESSION_MGT: vOnServiceState Availability = %d for Source = %d\n",
				ETG_ENUM(TUNER_CLIENT_AVAILABLITY, (tU8)(tunerClientServiceState.oGetCurrentTunerState(oGetTunerClient(activeSession.sourceId)))),
				ETG_ENUM(TUNER_SOURCE, (tU8)activeSession.sourceId)));
		/* handle stored requests with respective client service become available */		
		tuner::common::enSourceId allocateStoredSourceId = tunerClientServiceState.getAllocateStoredSourceId();
		/*Store tuner state as BG as soon service become available - NCG3D-39183*/
		tunerClientServiceState.vOnNewTunerState(tunerClient, tuner::common::TUNERTOBG);
		ETG_TRACE_USR4(("SESSION_MGT: vOnServiceState    allocate sourceId    = %d", ETG_ENUM(TUNER_SOURCE, (tU8)allocateStoredSourceId)));
		if(allocateStoredSourceId != SOURCEID_LIMIT && tunerClientServiceState.bIsServiceAvailable(allocateStoredSourceId) == true)
		{
			bOnAllocate(allocateStoredSourceId, tunerClientServiceState.getAllocateStoredInputDevice());
		}
		tuner::common::enSourceId storedSourceId = tunerClientServiceState.getStoredSource();
		tuner::common::enSrcActivity storedSourceActivity = tunerClientServiceState.getStoredSourceActivity();
		if(storedSourceId != SOURCEID_LIMIT &&  storedSourceActivity != SRCACTIVITY_UNKNOWN
				&& tunerClientServiceState.bIsServiceAvailable(storedSourceId) == true)
		{
			bOnSrcActivity(storedSourceId, storedSourceActivity);
		}
	}
}
/****************************************************************************
*
******************************************************************************/
bool clAudioSession::bIsServiceUnAvailable(tuner::common::enSourceId tunerSourceId)
{
	return tunerClientServiceState.bIsServiceUnAvailable(tunerSourceId);
}
/****************************************************************************
*
******************************************************************************/
tuner::common::enSourceId clAudioSession::getRequestedSessionSrc()
{
	ETG_TRACE_USR4(("SESSION_MGT: vOnServiceState getRequestedSessionSrc = %d, getRequestedSessionSrcAct = %d\n",
		ETG_ENUM(TUNER_SOURCE, (tU8)requestedSession.sourceId), ETG_ENUM(TUNER_SRCACTIVITY, (tU8)requestedSession.tunerSrcActivity)));
	return requestedSession.sourceId;
}
/****************************************************************************
*
******************************************************************************/
tuner::common::enSrcActivity clAudioSession::getRequestedSessionSrcActivity()
{
	return requestedSession.tunerSrcActivity;
}

tuner::common::enSessionState clAudioSession::getRequestedSessionState()
{
	return requestedSession.state;
}
tuner::common::enSourceId clAudioSession::getStoredPreviousSessionStateBeforeAnnoucementSrc()
{
	return previousSessionAfterAnnouncement.sourceId;
}
tuner::common::enSrcActivity clAudioSession::getStoredPreviousSessionStateBeforeAnnoucementSrcActivity()
{
	return previousSessionAfterAnnouncement.tunerSrcActivity;
}
/****************************************************************************
*
******************************************************************************/
tuner::common::enTunerState clAudioSession::getActiveTunerState(tuner::common::enTunerClient tunerClient)
{
	return tunerClientServiceState.oGetCurrentTunerState(tunerClient);
}
/****************************************************************************
*
******************************************************************************/
void clAudioSession::vOnNewDabTunerState(tuner::common::enTunerState tunerState)
{
	tunerClientServiceState.vOnNewTunerState(tuner::common::DABTUNER, tunerState);
}
/****************************************************************************
*
******************************************************************************/
tuner::common::enBand clAudioSession::oGetActiveTunerBand()
{
	ETG_TRACE_USR4(("SESSION_MGT: oGetActiveTunerBand band = %d\n", ETG_ENUM(TUNER_BAND, (tU8)activeSession.tunerBand)));
	/*if band change in progress, tunerSourceId identified by requested tuner band
	  required for mixed preset recall, forwarding mute requests during band change properly*/
	if(activeSession.bIsSrcChangeInProgress() && (requestedSession.bSrcOn() || requestedSession.bSrcPause())
			&& !bIsTemporarySource(requestedSession.sourceId))
	{
		ETG_TRACE_USR4(("SESSION_MGT: Requested Tuner Band = %d\n", ETG_ENUM(TUNER_BAND, (tU8)requestedSession.tunerBand)));
		return requestedSession.tunerBand;
	}
	return activeSession.tunerBand;
}
/****************************************************************************

 * FUNCTION:     clAudioSession::vBGorFGLearnForever
 * DESCRIPTION:  This method is to decide if BG_LEARN_FOREVER or FG__LEARN_FOREVER is sent.
                  Decision is based on sourcetype and source states
 * PARAMETER:    Type of radio source
 *	 	 	 	 Source Activity (ON ,PAUSE, OFF)
 * RETURNVALUE:  void

 Handling for radio sources ?
 When the source becomes ON or PAUSE only

 Handling when Non Radio source?
 FM Off, DAB off and AM off)

 When to update the DAB tuner state?
  ->Updated when Src change result to ARL

******************************************************************************/
void clAudioSession::vBGorFGLearnForever(tuner::common::enSourceId tunerSourceId, tuner::common::enSrcActivity tunerSrcActivity)
{

#ifndef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
	tuner::common::enTunerState dabtunerState = getActiveTunerState(DABTUNER);
	tuner::common::enTunerState dabtunerStatetoSet = getActiveTunerState(DABTUNER);
	if((dabtunerState == TUNERTOBG) || (dabtunerState == TUNERTOFG))
	{
		DABPreviousState = tuner::common::TUNERSTATE_UNKNOWN;
	}
	ETG_TRACE_USR4(("SESSION_MGT: vBGorFGLearnForever()tunerSourceId=%d tunerSrcActivity=%d",tunerSourceId,tunerSrcActivity));
	if(m_bIsSBRVariant && (tunerSrcActivity != tuner::common::SRCOFF) && (tunerSourceId == tuner::common::DAB))
	{
		/*Do not send FG/BG Learn when DAB in FG */
		ETG_TRACE_USR4(("SESSION_MGT: vBGorFGLearnForever() No action as DAB in FG"));
		return;
	}
	switch(tunerSourceId)
	{
		case FM1:
		case AM:
		case DAB:
			if (tuner::common::SRCPAUSE == tunerSrcActivity || tuner::common::SRCON == tunerSrcActivity )
			{	/*Radio sources*/
				if(bIsBGorFGLearnForever(tunerSourceId))
				{
					dabtunerStatetoSet= LEARN_FOREVER;
				}
				else
				{
					dabtunerStatetoSet = BG_LEARN_FOREVER;
				}
			}
			else if(tuner::common::SRCOFF == tunerSrcActivity)
			{
				/*Non Radio sources*/
				if(bFGBGLearnForNonRadioSource )
				{
					dabtunerStatetoSet= LEARN_FOREVER;
				}
				else
				{
					dabtunerStatetoSet = BG_LEARN_FOREVER;
				}
			}
			ETG_TRACE_USR4(("SESSION_MGT: vBGorFGLearnForever()-> currentdabstate= %d dabtunerStatetoSet= %d DABPreviousState=%d"
					,dabtunerState,dabtunerStatetoSet,DABPreviousState));
			if((dabtunerStatetoSet != dabtunerState) && (DABPreviousState !=dabtunerStatetoSet ))
			{
				DABPreviousState = dabtunerStatetoSet;
				tunerControl->vSetTunerState(dabtunerStatetoSet, DAB);
				/*As there is no method result for FG/BG learn forever, set DABTuner state accordingly.*/
				//vOnNewDabTunerState(dabtunerStatetoSet);
			}
		break;
		case SOURCEID_LIMIT:
		{
			/*Non Radio sources*/
			if(bFGBGLearnForNonRadioSource )
			{
				dabtunerStatetoSet= LEARN_FOREVER;
			}
			else
			{
				dabtunerStatetoSet = BG_LEARN_FOREVER;
			}
			DABPreviousState = dabtunerStatetoSet;
			tunerControl->vSetTunerState(dabtunerStatetoSet, DAB);
			/*As there is no method result for FG/BG learn forever, set DABTuner state accordingly.*/
			vOnNewDabTunerState(dabtunerStatetoSet);
			break;
		}
		default: break;
    }
#endif
}

/****************************************************************************
 * FUNCTION:     clAudioSession::vFGBGLearnForever
 * DESCRIPTION:  This method is used to updated based on KDS parameter
 * PARAMETER:     bool to indicate FG or BG LearnForever
 * RETURNVALUE:  void
 *
******************************************************************************/
void clAudioSession::vFGBGLearnForever(bool bBGorFGLearnForever)
{
	bFGBGLearnForNonRadioSource = bBGorFGLearnForever;
}


/****************************************************************************
 * FUNCTION:     clAudioSession::bIsBGorFGLearnForever
 * DESCRIPTION:  Type stored in Local map to maintained for storing FG or BG learnforever updated based on KDS values
 * PARAMETER:    src ID
 * RETURNVALUE:  bool type of learning
  *
******************************************************************************/

bool clAudioSession::bIsBGorFGLearnForever(tuner::common::enSourceId tunerSourceId)
{
	if (FGorBGLearForeverStatus.find(tunerSourceId) != FGorBGLearForeverStatus.end())
	{
		return FGorBGLearForeverStatus[tunerSourceId];
	}
	return false;
}

/****************************************************************************
 * FUNCTION:     clAudioSession::vTunerSetFGBGLearnForever
 * DESCRIPTION:  Local map to maintained for storing FG or BG learnforever updated based on KDS values
 * PARAMETER:    src ID and type
 * RETURNVALUE:  void
  *
******************************************************************************/

void clAudioSession::vTunerSetFGBGLearnForever(tuner::common::enSourceId tunerSourceId, bool bSupport)
{
	ETG_TRACE_USR4(("clAudioSession: src = %d FGorBGLearForeverStatus            = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId),bSupport));
	FGorBGLearForeverStatus[tunerSourceId] = bSupport;
}

/***
 * FUNCTION:     clAudioSession::vOnNewTunerState
 * DESCRIPTION:  Validate received tuner state for requested source. This function applies only for Gen4 onwards with SBR
 * PARAMETER:    tuner State and Source Id
 * RETURNVALUE:  void
  *
 */
void clAudioSession::vOnNewTunerState(tuner::common::enTunerState tunerState, tuner::common::enSourceId tunerSourceId)
{
	if(requestedSession.sourceId == tuner::common::SOURCEID_LIMIT || activeSession.sourceId == tuner::common::SOURCEID_LIMIT)
	{
		   /* If there are no source activity request, ignore tuner status. */
		   ETG_TRACE_USR4(("SESSION_MGT: vOnNewTunerState     ignore tuner status requested source = %d, active source     = %d",
		   	   ETG_ENUM(TUNER_SOURCE, (tU8)requestedSession.sourceId), ETG_ENUM(TUNER_SOURCE, (tU8)activeSession.sourceId)));
		   return;
	}
	ETG_TRACE_USR4(("clAudioSession: tunerState = %d, tunerSource = %d \n", ETG_ENUM(TUNER_STATE, (tU8)tunerState), ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
	if(tunerState == tuner::common::TUNERTOBG)
	{
		if(requestedSession.bSrcOff())
		{
			requestedSession.state = tuner::common::SRC_CHANGE_COMPLETED;
			vPrintSessionInformation();
			vSynchronize();
			if(!bIsTemporarySource(requestedSession.sourceId))
			{
				tunerClientServiceState.vOnNewTunerState(oGetTunerClient(requestedSession.sourceId), tunerState);
			}
		}
	}
	else
	{
		if(m_bIsSBRVariant && bIsTemporarySourceActivityOffRequest()) {
			ETG_TRACE_USR4(("SESSION_MGT: vOnNewTunerState SBR temporary SRCOFF request"));
			vOnNewTunerStateForTemporarySources();
		}
		if((requestedSession.bSrcOn() || requestedSession.bSrcPause()) && tunerState == tuner::common::TUNERTOFG && requestedSession.sourceId == tunerSourceId)
		{
			bIsTemporarySource(tunerSourceId) ? vOnNewTunerStateForTemporarySources() : vOnNewTunerStateForNormalSources();
			bIsTemporarySource(tunerSourceId) ? tunerClientServiceState.vOnNewTunerState(oGetTunerClient(requestedSession.sourceId), tuner::common::TAFG) : tunerClientServiceState.vOnNewTunerState(oGetTunerClient(requestedSession.sourceId), tunerState);
		}
		else if(tunerState == tuner::common::TUNERTOFG && bIsTemporarySourceActivityOffRequest() && (previousSessionAfterAnnouncement.sourceId == tunerSourceId)
				&& (previousSessionAfterAnnouncement.bSrcOn() || previousSessionAfterAnnouncement.bSrcPause()))
		{
			vOnNewTunerStateForNormalSources();
			tunerClientServiceState.vOnNewTunerState(oGetTunerClient(previousSessionAfterAnnouncement.sourceId), tunerState);
		}
		else
		{
			activeSession.state = tuner::common::SRC_CHANGE_FAILED;
			vPrintSessionInformation();
			vSynchronize();
		}
	}
}

void clAudioSession::vSetSBRVariant(bool bSBRVariant)
{
	ETG_TRACE_USR4(("SESSION_MGT:vSetSBRVariant - %d", bSBRVariant));
	m_bIsSBRVariant = bSBRVariant;
}
