/**************************************************************************//**
 * \file     clDRMInput.cpp
 *
 *           clDRMInput method class implementation
 *
 * \remark   Copyright: 2010 Robert Bosch GmbH, Hildesheim
 ******************************************************************************/

#include "drmPlayer/clDRMInput.h"
#include "drmPlayer/drmPlayerInterface.h"

#include "DRMTrace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUNER_TRACE_CLASS_DRMPLAYER 
#include "trcGenProj/Header/clDRMInput.cpp.trc.h"
#endif


using namespace tuner::drm;

clDRMInput* clDRMInput::m_poInput = NULL;


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clDRMInput::clDRMInput()
: fp(0)
, m_bIsEndofFile(0)
, m_bQuitThread(0)
{
	pthread_mutex_init (&mutex, NULL);
	sem_init (&semaphore,0,0);
	m_poInput = this;
	thread=(pthread_t)NULL;
	m_poFrame=0;
	referenceTime.tv_sec =0;
	referenceTime.tv_usec = 0;
	drmFrameTime.tv_sec = 0;
	drmFrameTime.tv_usec =0 ;
}


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clDRMInput::~clDRMInput()
{
	m_bQuitThread = 1;
	if(thread)
	{
		pthread_join (thread, NULL);
	}
	pthread_mutex_destroy(&mutex);
	sem_destroy (&semaphore);
	fp = 0;
	m_poFrame=0;
}
/**************************************************************************//**
 *
 ******************************************************************************/
void clDRMInput::vFeedData(tunerString oFileName )
{
	m_oFileName = oFileName;
	timerclear (&referenceTime);
	pthread_create(&thread, NULL, vFileReaderThread, NULL);
	sem_wait (&semaphore);

}


/**************************************************************************//**
 *
 ******************************************************************************/
void* clDRMInput::vFileReaderThread( void *)
{
	if (m_poInput)
	{
		m_poInput->vOnFileReaderThread();
	}
	return 0;

}

/**************************************************************************//**
 *
 ******************************************************************************/
void clDRMInput::vOnFileReaderThread()
{
	fp = fopen(m_oFileName.c_str(), "rb");
	if (fp == 0)
	{
		ETG_TRACE_USR4(("READ_THREAD: clDRMInput no file present: %s", m_oFileName.c_str()));
		return ;
	}
	ETG_TRACE_USR4(("READ_THREAD: clDRMInput file present: %s", m_oFileName.c_str()));
	drmFrameTime.tv_sec = 0;
	drmFrameTime.tv_usec = 400000;
	drmPlayerInterface* poInterface = drmPlayerInterface::getInstance();
	while (!m_bIsEndofFile)
	{
		struct timeval currentTime;
		gettimeofday (&currentTime, NULL);

		if (timercmp(&currentTime, &referenceTime, > ))
		{
			pthread_mutex_lock (&mutex);
			ETG_TRACE_USR4(("READ_THREAD: clDRMInput reading frame count"));

			if (poInterface->getFrameCount() == 0)
			{

			      try{
			    	  vReadNextFrame();
			      }
			      catch(...){

			    	  ETG_TRACE_USR4(("READ_THREAD: vReadNextFrame "));
			      }

				 if (bIsLastFrame() || m_bQuitThread)
				 {
					vSetAsLastFrame();
					ETG_TRACE_USR4(("READ_THREAD: clDRMInput reached EOF"));
					m_bIsEndofFile = true;
				 }

			      try{
			    	  vSendFrame();
			      }
			      catch(...){

			    	  ETG_TRACE_USR4(("READ_THREAD: vSendFrame "));
			      }

				sem_post (&semaphore);
				if (!timerisset(&referenceTime))
				{
					gettimeofday(&referenceTime, NULL);
				}
				timeradd (&referenceTime, &drmFrameTime, &referenceTime);
			}
			else if (m_bQuitThread)
			{
				m_bIsEndofFile = true;
				ETG_TRACE_USR4(("READ_THREAD: clDRMInput waiting for read of previous frame"));
			}
			pthread_mutex_unlock (&mutex);

		}
		usleep (20000);
	}
	fclose(fp);
	ETG_TRACE_USR4(("READ_THREAD: clDRMInput exiting loop"));
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clDRMInput::vReadNextFrame()
{
   m_poFrame = new clCMDRMAudioLogicalFrameStatus();
   vAddHeaderInformation();
   vAddPayloadInformation();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clDRMInput::vSendFrame()
{
	drmPlayerInterface* poInterface = drmPlayerInterface::getInstance();
	ETG_TRACE_USR4(("READ_THREAD: Sent new frame"));
	poInterface->vOnNewData(m_poFrame);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clDRMInput::vSetAsLastFrame()
{
   if(m_poFrame!=NULL)
   {
      m_poFrame->Action.enType = clCMEnum_Action::END_OF_STREAM;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clDRMInput::vAddHeaderInformation()
{
	gint  readBytes;
	static int frameCounter = 0;
	guint interfaceFrameLen;
	if((NULL==fp)||(NULL==m_poFrame))
	{
		return;
	}
	readBytes = fread(actInterfaceHeader, sizeof(guchar), INTERFACE_HEADER_SIZE, fp);
	if (frameCounter == 0)
	{
        m_poFrame->Action.enType = clCMEnum_Action::START_OF_STREAM;
	}
	interfaceFrameLen = ntos(actInterfaceHeader[0]);
    //m_poFrame->config.audioSamplingRate = ntos(actInterfaceHeader[2]);
    //m_poFrame->config.coderField = ntos(actInterfaceHeader[3]) & 0x1F;
    //m_poFrame->config.audioCoding = (ntos(actInterfaceHeader[4]) % 10) & 0x03;
    //m_poFrame->config.audioMode = ntos(actInterfaceHeader[5]) & 0x03;
    //m_poFrame->config.sbrFlag = ntos(actInterfaceHeader[6]) & 0x01;
    m_poFrame->LenFrameData =  ntos(actInterfaceHeader[7]);
    //m_poFrame->config.lengthOfAudioSuperFrame = ntos(actInterfaceHeader[7]);
    m_poFrame->LenHigherProtFrameData = ntos(actInterfaceHeader[8]);
    //m_poFrame->config.drmPlus = ((ntos(actInterfaceHeader[4]) % 10) & 0x04) >> 2;
	m_poFrame->FrameCntr = frameCounter++;
    m_poFrame->LenAudioInfoEntity = interfaceFrameLen - m_poFrame->LenFrameData - INTERFACE_HEADER_SIZE;
    readBytes = fread(  &(m_poFrame->AudioInfoEntity[0]), sizeof(guchar), m_poFrame->LenAudioInfoEntity, fp );

}


/**************************************************************************//**
 *
 ******************************************************************************/
void clDRMInput::vAddPayloadInformation()
{
	if((NULL!=fp) &&(NULL!=m_poFrame))
	{
        m_poFrame->FrameData.resize(m_poFrame->LenFrameData);
        size_t noofBytes= fread(  &(m_poFrame->FrameData[0]), sizeof(guchar), m_poFrame->LenFrameData, fp );
        ETG_TRACE_USR4(("vAddPayloadInformation: %d", noofBytes));
	}
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clDRMInput::bIsLastFrame()
{
	if(NULL!=fp)
	{
		if (feof(fp))
		{
			return true;
		}
	return false;
	}
	return false;
}

/**************************************************************************//**
 *
 ******************************************************************************/
int clDRMInput::ntos(int value) {
		   int ret =0;
	ret |= (value & 0x000000FF) << 24;
	ret |= (value & 0x0000FF00) <<  8;
	ret |= (value & 0x00FF0000) >>  8;
	ret |= (value & 0xFF000000) >> 24;
	return ret;
}

