/************************************************************************
* FILE:         clIFiPropRegHelper.cpp
* PROJECT:      Ford HSRNS
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION: helper-interface to manage all registered/unregistered
*              properties of HSI handler
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
*************************************************************************/

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include <algorithm>
#include <vector>




#include "CcaApp/clFiPropRegHelper.h"

/* interface for HSI-handler */
#include "CcaApp/clIFiPropRegHelper.h"


using namespace tuner::CcaApp;


/*************************************************************************
*
* FUNCTION:    clIFiPropRegHelper::clIFiPropRegHelper()
* 
* DESCRIPTION: default constructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*************************************************************************/
clIFiPropRegHelper::clIFiPropRegHelper()
{
   m_pclHelper = NULL;
   oFiPropRegHelper.reserve(MAX_SIZE_OF_PROPERTIES);
   oFiPropRegHelper.clear();
}

/*************************************************************************
*
* FUNCTION:    clIFiPropRegHelper::clIFiPropRegHelper(hmicca_tclApp* poGWMainApp)
* 
* DESCRIPTION: destructor, deletes all framework objects
*
* PARAMETER:   none 
*
* RETURNVALUE: none
*
*************************************************************************/
clIFiPropRegHelper::~clIFiPropRegHelper()
{
   tU32 u32Count                            = 0;
   tU32 u32Size                             = 0;

   u32Size = (tU32)oFiPropRegHelper.size();  
  
   for ( u32Count = 0; u32Count < u32Size; u32Size--)
   {
       std::vector < sFiPropRegHelper * >::iterator it = oFiPropRegHelper.begin();

       if (*it != NULL)
       {
          if ((*it)->poFi != NULL)
              OSAL_DELETE((*it)->poFi);

          OSAL_DELETE (*it);   
          oFiPropRegHelper.erase( it );
       }
   }

   oFiPropRegHelper.clear();

   if ( m_pclHelper != NULL )
   {
     m_pclHelper = NULL;
   }
}


/*************************************************************************
*
* FUNCTION:    clIFiPropRegHelper::bAddFunctionID()
* 
* DESCRIPTION:  this helpermethod is called from the HSIlhandler to add 
*               the function-ID which should be supported by the 
*               helper-class.
*
* PARAMETER:   tU32 u32FunctionID 
*
* RETURNVALUE: tBool
*
*************************************************************************/

tBool clIFiPropRegHelper::bAddFunctionID(tU16 u16FunctionID, tBool bUsePort)
{
   tBool bRetVal   = FALSE;
   tBool bIsInList = FALSE;

   // check if pointer is in list

    for (std::vector<sFiPropRegHelper*>::iterator it = oFiPropRegHelper.begin(); it < oFiPropRegHelper.end(); it++)
   {
       if (u16FunctionID == (*it)->poFi->u16GetFunctionID())
       {
           bIsInList = TRUE;
       }
   }

   if ( bIsInList )
   {
       // nothing to do
   }
   else
   {
       sFiPropRegHelper* poPropRegHelper = (sFiPropRegHelper*)pclCreatePropHelperElem();

       if ( poPropRegHelper != NULL )
       {
           poPropRegHelper->poFi->vSetFunctionID( u16FunctionID );
           poPropRegHelper->poFi->vSetRetVal( FALSE );
           poPropRegHelper->poFi->vSetRegisterDone( FALSE );

           poPropRegHelper->bUsePort = bUsePort;
            oFiPropRegHelper.push_back(poPropRegHelper);
            bRetVal   = TRUE;
       }
       // 4LINT
       poPropRegHelper = NULL;
   }

   return bRetVal;
}


/*************************************************************************
*
* FUNCTION:    clIFiPropRegHelper::vClearAllRegistration(tVoid)
* 
* DESCRIPTION:  this helpermethod resets all registrations.
*
* PARAMETER:   tU32 u32FunctionID 
*
* RETURNVALUE: tBool
*
*************************************************************************/

tVoid clIFiPropRegHelper::vClearAllRegistration(tVoid)
{
    for (std::vector<sFiPropRegHelper*>::iterator it = oFiPropRegHelper.begin(); it < oFiPropRegHelper.end(); it++)
   {
       (*it)->poFi->vSetRegisterDone( FALSE );
       (*it)->poFi->vSetRetVal( FALSE );
   }
}


/*************************************************************************
*
* FUNCTION:    clIFiPropRegHelper::u16GetFunctionIDToRegister(tVoid)
* 
* DESCRIPTION:  this helpermethod is called from the HSIlhandler to add 
*               the function-ID which should be supported by the 
*               helper-class.
*
* PARAMETER:   tU32 u32FunctionID 
*
* RETURNVALUE: tBool
*
*************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_LOOPED_FID_REGISTRATION
tU16 clIFiPropRegHelper::u16GetFunctionIDToRegister(tBool *pbUsePort)
{
   tU16 u16FunctionID = 0;
   
    for (std::vector<sFiPropRegHelper*>::iterator it = oFiPropRegHelper.begin(); it < oFiPropRegHelper.end(); it++)
   {
      if ( ( (*it)->poFi->bGetRetVal() == FALSE )/* && ( (*it)->poFi->bIsRegisterDone() == FALSE )*/ )
      {
         u16FunctionID = (*it)->poFi->u16GetFunctionID();
         if (pbUsePort != NULL)
             *pbUsePort = (*it)->bUsePort;

         break;
      }
   }

   return u16FunctionID;
}
#else
tU16 clIFiPropRegHelper::u16GetFunctionIDToRegister(tU32 u32Idx, tBool* pbUsePort)
{
   tU16 u16FunctionID = 0;

   if (u32Idx < oFiPropRegHelper.size())
   {
      if ( ( oFiPropRegHelper.at(u32Idx)->poFi->bGetRetVal() == FALSE ) )
      {
         u16FunctionID = oFiPropRegHelper.at(u32Idx)->poFi->u16GetFunctionID();
         if (pbUsePort != NULL)
             *pbUsePort = oFiPropRegHelper.at(u32Idx)->bUsePort;
      }
   }

   return u16FunctionID;
}
#endif
/*************************************************************************
*
* FUNCTION:     clIFiPropRegHelper::vSetRegisterID( )
* 
* DESCRIPTION:  set the return value of register for property
*
* PARAMETER:    tU32 u32FunctionID
*               tBool bRetVal
*
* RETURNVALUE:  tVoid
*
*************************************************************************/

tVoid clIFiPropRegHelper:: vSetRetValForRegister( tU16 u16FunctionID, tBool bRetVal )
{
   for (std::vector<sFiPropRegHelper*>::iterator it = oFiPropRegHelper.begin(); it < oFiPropRegHelper.end(); it++)
   {
       if ( (*it)->poFi->u16GetFunctionID() == u16FunctionID  )
       {
           (*it)->poFi->vSetRetVal(bRetVal);
           (*it)->poFi->vSetRegisterDone(bRetVal);
           break;
       }
   }
}


/*************************************************************************
*
* FUNCTION:    clIFiPropRegHelper::bAreAllPropertiesRegistered( )
* 
* DESCRIPTION: gets information if there are properties not registered
*
* PARAMETER:   none
*
* RETURNVALUE: tVoid*
*
*************************************************************************/

tBool clIFiPropRegHelper::bAreAllPropertiesRegistered(  )
{
    tBool bRetVal = TRUE;

    for (std::vector<sFiPropRegHelper*>::iterator it = oFiPropRegHelper.begin(); it < oFiPropRegHelper.end(); it++)
    {
        bRetVal = (tBool)(bRetVal & (*it)->poFi->bGetRetVal());
//        (*it)->vSetRegisterDone(FALSE);
    }

    return bRetVal;
}

/*************************************************************************
*
* FUNCTION:    clIFiPropRegHelper::vAddResponseForFctID()
* 
* DESCRIPTION:  this helpermethod is called from the HSI_handler to inform 
*               how the response from the upreg was.
*
* PARAMETER:   tU32  u32FunctionID  functionID 
*              tBool bResponse      Response succeeded
*
* RETURNVALUE: tVoid
*
*************************************************************************/

tVoid clIFiPropRegHelper::vAddResponseForFctID(tU16 u16FunctionID, tBool bResponse)
{
   for (std::vector<sFiPropRegHelper*>::iterator it = oFiPropRegHelper.begin(); it < oFiPropRegHelper.end(); it++)
   {
       if ( (*it)->poFi->u16GetFunctionID() == u16FunctionID  )
       {
           (*it)->poFi->vSetUpRegResponse(bResponse);
           break;
       }
   }
}


/*************************************************************************
*
* FUNCTION:    clIFiPropRegHelper::bAreAllRegsFinished( )
* 
* DESCRIPTION: gets information if all responses of registered properties 
*              were succeeded
*
* PARAMETER:   none
*
* RETURNVALUE: tVoid*
*
*************************************************************************/

tBool clIFiPropRegHelper::bAreAllRegsFinished(  )
{
    tBool bRetVal = true;
    for ( std::vector<sFiPropRegHelper*>::iterator it = oFiPropRegHelper.begin() ; it < oFiPropRegHelper.end() ; it ++ )
    {
        bRetVal = ((*it)->poFi->bGetRetVal()) && ((*it)->poFi->bIsUpRegValid()) && bRetVal;
    }

    return bRetVal;
}

/*************************************************************************
*
* FUNCTION:    pclCreateServiceDataObject()
* 
* DESCRIPTION: this method creates a Prop-Reg-helper object
*
* PARAMETER:   tVoid
*
* RETURNVALUE: tVoid*
*
*************************************************************************/
tVoid* clIFiPropRegHelper::pclCreatePropHelperElem(tVoid)
{
   m_pclHelper = OSAL_NEW sFiPropRegHelper();
   if (m_pclHelper)
      m_pclHelper->poFi = OSAL_NEW clFiPropRegHelper();
   return (tVoid*) m_pclHelper;
}

