/**************************************************************************//**
* \file     clAnnouncements.cpp
*
*           clAnnouncements method class implementation
*
* \remark   Copyright: 2010 Robert Bosch GmbH, Hildesheim
******************************************************************************/

#include "announcements/clAnnouncements.h"
#include "announcements/clTunerControl.h"
#include "announcements/clAnnouncementSettings.h"


#include "datatypes/tunerDatatypes.h"
#include "common/tuner_trace.h"


using namespace tuner::announcements;
using namespace tuner::common;

clAnnouncements* clAnnouncements::m_poAnnouncements = 0;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUNER_TRACE_CLASS_ANNOUNCEMENTS
#include "trcGenProj/Header/clAnnouncements.cpp.trc.h"
#endif


/**************************************************************************//**
* Constructor
******************************************************************************/
clAnnouncements::clAnnouncements()
: tunerControl(0)
,PICode(0)
{
   vInit();
}


/**************************************************************************//**
* Destructor
******************************************************************************/
clAnnouncements::~clAnnouncements()
{
   vDeInit();
}


/**************************************************************************//**
*
******************************************************************************/
tuner::announcements::clAnnouncements* clAnnouncements::instance()
{
   if (0 == clAnnouncements::m_poAnnouncements)
   {
      clAnnouncements::m_poAnnouncements = new tuner::announcements::clAnnouncements();
   }
   return clAnnouncements::m_poAnnouncements;
}


/**************************************************************************//**
*
******************************************************************************/
void clAnnouncements::deleteInstance()
{
   if (0 != clAnnouncements::m_poAnnouncements)
   {
      delete clAnnouncements::m_poAnnouncements;
      clAnnouncements::m_poAnnouncements = 0;
   }
}


/**************************************************************************//**
* 
******************************************************************************/
void clAnnouncements::vInit()
{
   requestedAnnouncement.vReset();
   activeAnnouncement.vReset();
   tunerControl = clTunerControl::instance();
   activeAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_COMPLETED;
   requestedAnnouncementTemp.vReset();
   activesessionTemp.vReset();
   vPrintAnnouncementInformation();
}


/**************************************************************************//**
* 
******************************************************************************/
void clAnnouncements::vDeInit()
{
   requestedAnnouncement.vReset();
   activeAnnouncement.vReset();
   activeAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_COMPLETED;
   requestedAnnouncementTemp.vReset();
   activesessionTemp.vReset();
   vPrintAnnouncementInformation();
}


/**************************************************************************//**
*
******************************************************************************/
void clAnnouncements::vPrintAnnouncementInformation()
{
   ETG_TRACE_USR4(("ANNOUNCEMENTS: REQUESTED       announcementId      = %d\n", ETG_ENUM(TUNER_ANNOUNCEMENT, (tU8)requestedAnnouncement.announcementId)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: REQUESTED       State               = %d\n", ETG_ENUM(TUNER_ANNOUNCEMENT_STATE, (tU8)requestedAnnouncement.state)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: REQUESTED       priority            = %d\n", (tU8)requestedAnnouncement.priority));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: REQUESTED       sourceId            = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)requestedAnnouncement.tunerSource)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: REQUESTED       srcActivity         = %d\n", ETG_ENUM(TUNER_SRCACTIVITY, (tU8)requestedAnnouncement.tunerSrcActivity)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: ACTIVE          announcementId      = %d\n", ETG_ENUM(TUNER_ANNOUNCEMENT, (tU8)activeAnnouncement.announcementId)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: ACTIVE          State               = %d\n", ETG_ENUM(TUNER_ANNOUNCEMENT_STATE, (tU8)activeAnnouncement.state)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: ACTIVE          priority            = %d\n", (tU8)activeAnnouncement.priority));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: ACTIVE          sourceId            = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)activeAnnouncement.tunerSource)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: ACTIVE          srcActivity         = %d\n", ETG_ENUM(TUNER_SRCACTIVITY, (tU8)activeAnnouncement.tunerSrcActivity)));

}


/**************************************************************************//**
*
******************************************************************************/
void clAnnouncements::vOnNewAnnouncementRequest(tuner::common::enAnnouncement announcementId, tuner::common::enSrcActivity tunerSrcActivity,  tuner::common::enSourceId tunerSourceId,
		tuner::common::enSourceId activetunerSourceId,tuner::common::enSrcActivity activetunerSrcActivity)
{
   /*For midw_fi_tcl_e8_Anno_Message::FI_EN_TA_PTY_WEATHERALERT_OFF,
   getAnnouncement will return ANNOUNCEMENT_LIMIT according to midwTunerMap.dat defined in tuner master*/
   if((announcementId == tuner::common::ANNOUNCEMENT_LIMIT) && (activeAnnouncement.announcementId == tuner::common::WEATHERALERT))
   {
      ETG_TRACE_USR4(("ANNOUNCEMENTS: vOnNewAnnouncementRequest"));
      tunerSourceId = tuner::common::WEATHERANNOUNCEMENTS;
      announcementId = tuner::common::WEATHERALERT;
   }
   ETG_TRACE_USR4(("ANNOUNCEMENTS: Request         announcementId      = %d\n", ETG_ENUM(TUNER_ANNOUNCEMENT, (tU8)announcementId)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: Request         tunerSrcActivity    = %d\n", ETG_ENUM(TUNER_SRCACTIVITY, (tU8)tunerSrcActivity)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: Request         tunerSourceId       = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)tunerSourceId)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: Request         activetunerSrcActivity    = %d\n", ETG_ENUM(TUNER_SRCACTIVITY, (tU8)activetunerSrcActivity)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: Request         activetunerSourceId       = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)activetunerSourceId)));
   if(bIsRequestedStateCompleted(announcementId, tunerSrcActivity, tunerSourceId) == true)
   {
	   return;
   }
   requestedAnnouncementTemp = requestedAnnouncement;
   activesessionTemp = activesession;
   requestedAnnouncement.vReset();
   requestedAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_REQUESTED;
   requestedAnnouncement.announcementId = oGetAnnouncement(announcementId, tunerSourceId, tunerSrcActivity);
   requestedAnnouncement.tunerSource = tunerSourceId;
   requestedAnnouncement.tunerSrcActivity = tunerSrcActivity;
   requestedAnnouncement.priority = (unsigned char)clAnnouncementSettings::instance()->u16GetAnnouncementPriority(requestedAnnouncement.announcementId);
   activesession.tunerSource =activetunerSourceId;
   activesession.tunerSrcActivity =activetunerSrcActivity;
   vPrintAnnouncementInformation();
   vSynchronize();
}

/**************************************************************************//**
*
******************************************************************************/
bool clAnnouncements::bIsRequestedStateCompleted(tuner::common::enAnnouncement announcementId, tuner::common::enSrcActivity tunerSrcActivity,  tuner::common::enSourceId tunerSourceId)
{
	if((announcementId == activeAnnouncement.announcementId || announcementId == tuner::common::ANNOUNCEMENT_LIMIT)
			&& (tunerSrcActivity == activeAnnouncement.tunerSrcActivity) && (tunerSourceId == activeAnnouncement.tunerSource)
			&& (activeAnnouncement.bIsAnnouncementRequestCompleted() == true) )
	{
		ETG_TRACE_USR4(("ANNOUNCEMENTS: bIsRequestedStateCompleted: Requested State Completed."));
		return true;
	}
	return false;
}

/**************************************************************************//**
*
******************************************************************************/
void clAnnouncements::vOnNewAnnouncementList(tunerVector<tuner::announcements::stTunerAnnouncementInformation>  announcementList,
		tuner::common::enSourceId activetunerSourceId,tuner::common::enSrcActivity activetunerSrcActivity)
{
   ETG_TRACE_USR4(("ANNOUNCEMENTS: RequestList     announcementListSize= %d\n", announcementList.size()));

   if (announcementList.size())
   {
	   for(unsigned int u32Index = 0; u32Index < announcementList.size(); u32Index++)
	   {
		   stTunerAnnouncementInformation announcementInfo = announcementList[u32Index];
		   
		   if( bIsRequestedStateCompleted(announcementInfo.tunerAnnouncement, announcementInfo.tunerSrcActivity, announcementInfo.tunerSourceId) || 
			   bIsRequestedStateIsSwitched(announcementInfo.tunerSrcActivity) ||
			   bIsRequestedAnnouncementPriorityIsLessThanOREqualActiveAnnounementPriority(announcementInfo.tunerAnnouncement) || (false == bIsValidAnnouncementType(announcementInfo.tunerAnnouncement) ))
		   {
			   continue;
		   }
		   else if( bIsActiveAnnounementSrcOffRequest(announcementList) )
		   {
			   /* Received request for DABTA off (which is currently being played) and other DAB announcements are to be put through */
			   requestedAnnouncement.tunerSrcActivity = activeAnnouncement.tunerSrcActivity = tuner::common::SRCOFF;
			   tunerControl->vOnAnnouncementCompleted(activeAnnouncement.tunerSource, activeAnnouncement.announcementId, activeAnnouncement.tunerSrcActivity);
		   }
		   else if ( bIsLowestPriorityAnnouncementSrcOffRequest(announcementInfo) )
		   {
			   tunerControl->vOnAnnouncementCompleted(announcementInfo.tunerSourceId, announcementInfo.tunerAnnouncement, announcementInfo.tunerSrcActivity);
		   }
		   else
		   {
			   requestedAnnouncementTemp = requestedAnnouncement;
			   activesessionTemp = activesession;
			   requestedAnnouncement.vReset();
			   requestedAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_REQUESTED;
			   requestedAnnouncement.announcementId = announcementInfo.tunerAnnouncement;
			   requestedAnnouncement.tunerSource = announcementInfo.tunerSourceId;
			   requestedAnnouncement.tunerSrcActivity = announcementInfo.tunerSrcActivity;
			   requestedAnnouncement.priority = (unsigned char)clAnnouncementSettings::instance()->u16GetAnnouncementPriority(requestedAnnouncement.announcementId);
			   activesession.tunerSource =activetunerSourceId;
			   activesession.tunerSrcActivity =activetunerSrcActivity;
			   vPrintAnnouncementInformation();
			   vSynchronize();
		   }
	   }
   }
}

bool clAnnouncements::bIsActiveAnnounementSrcOffRequest(tunerVector<tuner::announcements::stTunerAnnouncementInformation>  announcementList)
{
	/* minimum list elements shouls be 2 and announcement source activity should be ON */
	/* handling of "if end of active announcment and start of another announcment is received in one list" */
	if(announcementList.size() > 1 && activeAnnouncement.bIsActivateRequest())
	{
		for(unsigned int u32Index = 0; u32Index < announcementList.size(); u32Index++)
		{
			if (activeAnnouncement.announcementId == announcementList[u32Index].tunerAnnouncement)
			{
				if(announcementList[u32Index].tunerSrcActivity == tuner::common::SRCOFF)
				{
					ETG_TRACE_USR4(("ANNOUNCEMENTS: bIsActiveAnnounementSrcOffRequest: return true "));
					return true;
				}
			}
		}
	}
	return false;
}

bool clAnnouncements::bIsLowestPriorityAnnouncementSrcOffRequest(tuner::announcements::stTunerAnnouncementInformation  announcementInfo)
{
	/* During high priority announcment is active and before to this any lowest priority anouncement was active but not ended and received lower priority announcent src off request, then update status to HMI*/
	if( (announcementInfo.tunerAnnouncement != activeAnnouncement.announcementId) &&
		(announcementInfo.tunerSrcActivity == tuner::common::SRCOFF) &&
		(tU8)clAnnouncementSettings::instance()->u16GetAnnouncementPriority(announcementInfo.tunerAnnouncement) > activeAnnouncement.priority )
	{
		ETG_TRACE_USR4(("ANNOUNCEMENTS: bIsLowestPriorityAnnouncementSrcOffRequest: Requested anno = %d, priority = %d, Active Anno  = %d, priority = %d",
			ETG_ENUM(TUNER_ANNOUNCEMENT, (tU8)announcementInfo.tunerAnnouncement), ETG_ENUM(TUNER_ANNOUNCEMENT, (tU8)activeAnnouncement.announcementId),
			(tU8)clAnnouncementSettings::instance()->u16GetAnnouncementPriority(announcementInfo.tunerAnnouncement), activeAnnouncement.priority));
		return true;
	}
	return false;
}

bool clAnnouncements::bIsRequestedStateIsSwitched(tuner::common::enSrcActivity tunerSrcActivity)
{
	/* Ignore if active DAB anno state as "SWITCHED" */
	if(tunerSrcActivity == tuner::common::SRCACTIVITY_UNKNOWN)
	{
		ETG_TRACE_USR4(("ANNOUNCEMENTS: bIsRequestedStateIsSwitched: Requested State Source Activity is SWITCHED "));
		return true;
	}
	return false;
}

bool clAnnouncements::bIsRequestedAnnouncementPriorityIsLessThanOREqualActiveAnnounementPriority(tuner::common::enAnnouncement tunerAnnouncement)
{
	/*Ignore least or equal priority announcements requests than active announcment */
	unsigned char priority = (tU8)clAnnouncementSettings::instance()->u16GetAnnouncementPriority(tunerAnnouncement);
	if(priority >= activeAnnouncement.priority && (activeAnnouncement.tunerSrcActivity == tuner::common::SRCON) && 
		(tunerAnnouncement != activeAnnouncement.announcementId) )
	{
		ETG_TRACE_USR4(("ANNOUNCEMENTS: bIsRequestedAnnouncementPriorityIsLessThanOrEqualActiveAnnounementPriority: Requested announcement (%d) priority (%d) is less than or equal to active announcment (%d) priority (%d)", 
				   ETG_ENUM(TUNER_ANNOUNCEMENT, (tU8)tunerAnnouncement), priority, ETG_ENUM(TUNER_ANNOUNCEMENT, (tU8)activeAnnouncement.announcementId), (tU8)activeAnnouncement.priority ));
		return true;
	}
	return false;
}


/**************************************************************************//**
*
******************************************************************************/
void clAnnouncements::vSynchronize()
{
	if(requestedAnnouncement.bIsAnnouncementFailed())
	{
		activeAnnouncement.vReset();
		activeAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_FAILED;
	}
	else if(requestedAnnouncement.bIsAnnouncementRequestCompleted() && activeAnnouncement.bIsAnnouncementRequestInProgress()
			&& (requestedAnnouncement.announcementId == activeAnnouncement.announcementId)
			&& (requestedAnnouncement.tunerSrcActivity == activeAnnouncement.tunerSrcActivity))
	{
		/*Upon receiving proper tuner state (FG/BG method result) send annuoncement_status to clients */
		activeAnnouncement.state = requestedAnnouncement.state;
		tunerControl->vOnAnnouncementCompleted(activeAnnouncement.tunerSource, activeAnnouncement.announcementId, activeAnnouncement.tunerSrcActivity);
	}
	else
	{
		vSynchronizeOnNewRequest();
	}
	vPrintAnnouncementInformation();
}

void clAnnouncements::vClearAnnouncementStates()
{
	/*if service is unavailable, and if SRCOFF activity is in process, make it complete and acknowledge to clients and reset variables
	* if SRCON then request ARL for SRCOFF and send announcement status to HMI and clear states
	*/
	ETG_TRACE_USR4(("ANNOUNCEMENTS::vClearAnnouncementStates src = %d, announcementId = %d, tunerSrcActivity = %d", requestedAnnouncement.tunerSource, requestedAnnouncement.announcementId, requestedAnnouncement.tunerSrcActivity));

		/*coverity  77228 - this function should be used only during service become when tuner unavailable to clear states*/
		audioControl.vRequestSourceChange(requestedAnnouncement.tunerSource, tuner::common::SRCOFF);
		tunerControl->vOnAnnouncementCompleted(requestedAnnouncement.tunerSource, requestedAnnouncement.announcementId, tuner::common::SRCOFF);

	vInit();
}

bool clAnnouncements::bIsRequestedAnnouncementFailed()
{
	if( activeAnnouncement.bIsAnnouncementFailed() && activeAnnouncement.announcementId == ANNOUNCEMENT_LIMIT
			&& activeAnnouncement.tunerSource == SOURCEID_LIMIT && activeAnnouncement.tunerSrcActivity == SRCACTIVITY_UNKNOWN
			&& requestedAnnouncement.bIsDeactivateRequest() == false )
	{
		return true;
	}
	return false;
}

bool clAnnouncements::bIsActiveStateFailed()
{
	if( activeAnnouncement.bIsAnnouncementFailed() && activeAnnouncement.announcementId == ANNOUNCEMENT_LIMIT
			&& activeAnnouncement.tunerSource == SOURCEID_LIMIT && activeAnnouncement.tunerSrcActivity == SRCACTIVITY_UNKNOWN )
	{
		return true;
	}
	return false;
}

void clAnnouncements::vSynchronizeOnNewRequest()
{
	if(requestedAnnouncement.bIsAnnouncementRequestInProgress() || bIsRequestedAnnouncementFailed())
	{
		/*previous anno request is failed due to improper tuner state,
		 * and received announcemet message either previous announcement Id or other proceed with respective anno Id FG method start */
		vSetActiveAnnouncement();
	}
	else if (requestedAnnouncement.bIsActivateRequest() && !activeAnnouncement.bIsAnnouncementRequestInProgress())
	{
		vSynchronizeConditionToActivateAnnouncement();
	}
	else if  (requestedAnnouncement.bIsActivateRequest() && activeAnnouncement.bIsAnnouncementRequestInProgress())
	{
		vSynchronizeRequestedAnnouncementOfHigherPriority();
	}
	else if (requestedAnnouncement.bIsDeactivateRequest())
	{
		/*previous announcement request is failed due to improper tuner state,
		 * and if announcement message '0' is received, request ARL for source change to deactivate TA */
		
		/* PSARCC30-3741 - if active state is in requested state i.e., previous ON-request was sent to ARL, but not received bOnSrcActivity
		 * then forward OFF-request to ARL if TA end message is received. Also block repetitive request to ARL */

		if ( bIsRequestedAnnouncementActive() || bIsActiveStateFailed() || 
			(activeAnnouncement.bIsAnnouncementRequested() && requestedAnnouncement.tunerSrcActivity != audioControl.getRequestedSourceActivity()) )
		{
			vRequestSourceChange();
		}
		else
		{
			/*if condition for requested announcement is failed, update stored variables back*/
			vRestoreStoredVariables();
		}
	}
}

void clAnnouncements::vSynchronizeConditionToActivateAnnouncement()
{
	ETG_TRACE_USR4(("AnnoucementValidation:: Requested Source =%d Activity=%d",
	ETG_ENUM(TUNER_SOURCE,oGetMainSourceId(requestedAnnouncement.tunerSource)),ETG_ENUM(TUNER_SRCACTIVITY,requestedAnnouncement.tunerSrcActivity)));
	ETG_TRACE_USR4(("AnnoucementValidation:: Active Source =%d Activity=%d",
	ETG_ENUM(TUNER_SOURCE,activesession.tunerSource),ETG_ENUM(TUNER_SRCACTIVITY,activesession.tunerSrcActivity)));
	if (bIsConditionToActivateAnnouncement())
	{
		if (bIsSourceChangeRequired())
		{
			vRequestSourceChange();
		}
		else
		{
			vSetActiveAnnouncement();
		}
	}
	else
	{
	   /*if condition for requested announcement is failed, update stored variables back*/
		vRestoreStoredVariables();
	}
}

void clAnnouncements::vSynchronizeRequestedAnnouncementOfHigherPriority()
{
	if (!bIsRequestedAnnouncementOfHigherPriority())
	{
		/*anno OFF is in progress, received same or less priority anno ON request, request ARL for source change */
		if(activeAnnouncement.bIsDeactivateRequest())
		{
			/*If condition not satisfied to put through requested announcment, do not request ARL for source change and clear tentatively stored request and replace previous announcement state back */
			if(bIsConditionToActivateAnnouncement())
			{
				vRequestSourceChange();
			}
			else
			{
				/*if condition for requested announcement is failed, update stored variables back*/
				vRestoreStoredVariables();
			}
		}
		if (requestedAnnouncement.tunerSource == oGetSourceId(activeAnnouncement.announcementId))
		{
			requestedAnnouncement.announcementId = activeAnnouncement.announcementId;
		}
	}
}

void clAnnouncements::vRestoreStoredVariables()
{
	/*PSARCC30-2595: if condition for requested announcement is failed, update stored variables back*/
	requestedAnnouncement = requestedAnnouncementTemp;
	activesession = activesessionTemp;
	requestedAnnouncementTemp.vReset();
	activesessionTemp.vReset();
	vPrintAnnouncementInformation();
}

/**************************************************************************//**
*
******************************************************************************/
bool clAnnouncements::bIsConditionToActivateAnnouncement()
{
	if(bIsAnnouncementOfMainSource())
	{
		if (requestedAnnouncement.bIgnoreAnnouncement())
		{
			ETG_TRACE_USR4(("bIsConditionToActivateAnnouncement:bIgnoreAnnouncement() return false"));
			return false;
		}
		if (bIsRequestedAnnouncementOfHigherPriority())
		{
			return true;
		}
	}
	else if (bPlayOtherAnnouncements())
	{

#ifndef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
		if (bIsServiceFollowingActive())
		{
			return true;
		}
		if (bIsRequestedAnnouncementOfHigherPriority())
		{
			return true;
		}
#endif
	}
	return false;
}
/**************************************************************************//**
*
******************************************************************************/
bool clAnnouncements::bIsSourceChangeRequired()
{
   if (requestedAnnouncement.tunerSource == activeAnnouncement.tunerSource)
   {
	   if(activeAnnouncement.bIsActivateRequest())
	   {
		   return false;
	   }
	   else if(bIsAnnouncementSource(audioControl.getActiveSource()) && audioControl.getActiveSourceActivity() == tuner::common::SRCON)
	   {
		   ETG_TRACE_USR4(("bIsSourceChangeRequired() returns false"));
		   return false;
	   } 
   }
   ETG_TRACE_USR4(("bIsSourceChangeRequired() returns true")); 
   return true;
}


/**************************************************************************//**
*
******************************************************************************/
bool clAnnouncements::bIsRequestedAnnouncementOfHigherPriority()
{
	ETG_TRACE_USR4(("bIsRequestedAnnouncementOfHigherPriority() activesession.tunerSource=%d, audioControl.getActiveSource() = %d",
			ETG_ENUM(TUNER_SOURCE,activesession.tunerSource), ETG_ENUM(TUNER_SOURCE,audioControl.getActiveSource()) ));
	if (!bIsAnnouncementSource(activesession.tunerSource) && !bIsAnnouncementSource(audioControl.getActiveSource()))
	{
		return true;
	}
	else if (requestedAnnouncement.priority < activeAnnouncement.priority)
   {
		/*Coverity fix for 22797 removed always true check*/
	   ETG_TRACE_USR4(("bIsRequestedAnnouncementOfHigherPriority() returns true, activesession.tunerSource=%d", ETG_ENUM(TUNER_SOURCE,activesession.tunerSource)));
	   return true;
   }
	else if((requestedAnnouncement.priority >= activeAnnouncement.priority) &&
			   (activeAnnouncement.tunerSrcActivity == tuner::common::SRCOFF ||activeAnnouncement.tunerSrcActivity == tuner::common::SRCACTIVITY_UNKNOWN )) /*Adding check to fix PSARCC30-3105 */
	{
		ETG_TRACE_USR4(("bIsRequestedAnnouncementOfHigherPriority() Similar or Lower prio activesession.tunerSource"));
	    return true;
	}
   /*For FM PTY and FM TA priority check is not required, handled in dat files*/
   else if ((audioControl.getActiveSource() == tuner::common::FMANNOUNCEMENTS) && (requestedAnnouncement.tunerSource == tuner::common::FMANNOUNCEMENTS))
   {
	   return true;
   }
   ETG_TRACE_USR4(("bIsRequestedAnnouncementOfHigherPriority() returns false"));
   return false;
}


/**************************************************************************//**
*
******************************************************************************/
void clAnnouncements::vRequestSourceChange()
{
	activeAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_REQUESTED;
	activeAnnouncement.announcementId = requestedAnnouncement.announcementId;
	activeAnnouncement.priority = (unsigned char)clAnnouncementSettings::instance()->u16GetAnnouncementPriority(activeAnnouncement.announcementId);
	audioControl.vRequestSourceChange(requestedAnnouncement.tunerSource, requestedAnnouncement.tunerSrcActivity);
}


/**************************************************************************//**
*
******************************************************************************/
void clAnnouncements::vSetActiveAnnouncement()
{
	activeAnnouncement.tunerSrcActivity = requestedAnnouncement.tunerSrcActivity;
	activeAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_REQUEST_IN_PROGRESS;
	activeAnnouncement.tunerSource = requestedAnnouncement.tunerSource;
	activeAnnouncement.announcementId =  requestedAnnouncement.announcementId;
	activeAnnouncement.tunerState = requestedAnnouncement.tunerState;
	vSendAnnouncementInformation();
}


/**************************************************************************//**
*
******************************************************************************/
void clAnnouncements::vSendAnnouncementInformation()
{
	/*For ON activity tuner state is decided based upon audiosink, MIDWSubCommand and Tuner State (Read from DAT file based on requested source activity).
	 * For OFF activity "activeAnnouncement.tunerState" value is considered for sending FG/BG metod start which is received from session lib */
	ETG_TRACE_USR4(("ANNOUNCEMENTS: vSendAnnouncementInformation  tunerState = %d\n", activeAnnouncement.tunerState));
	tunerControl->vOnAnnouncementState(activeAnnouncement.tunerSource, activeAnnouncement.announcementId, activeAnnouncement.tunerSrcActivity, activeAnnouncement.tunerState);
}


/**************************************************************************//**
*
******************************************************************************/
void clAnnouncements::vOnSourceChangeResponse(tuner::common::enSourceId sourceId, tuner::common::enSrcActivity tunerSrcActivity, tuner::common::enTunerState tunerState)
{
   ETG_TRACE_USR4(("ANNOUNCEMENTS: vOnSourceChangeResponse   = %d\n", ETG_ENUM(TUNER_SOURCE, (tU8)sourceId)));
   ETG_TRACE_USR4(("ANNOUNCEMENTS: vOnSourceChangeResponse   = %d\n", ETG_ENUM(TUNER_SRCACTIVITY, (tU8)tunerSrcActivity)));

   if (!activeAnnouncement.bIsAnnouncementRequestCompleted())
   {
	if (activeAnnouncement.tunerSource != sourceId || activeAnnouncement.tunerSrcActivity != tunerSrcActivity)
	{
		activeAnnouncement.tunerSource = sourceId;
		activeAnnouncement.tunerSrcActivity = tunerSrcActivity;
		if (sourceId == requestedAnnouncement.tunerSource &&
				tunerSrcActivity == requestedAnnouncement.tunerSrcActivity)
		{
			requestedAnnouncement.tunerState = tunerState;
			requestedAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_REQUEST_IN_PROGRESS;
		}
		vSynchronize();
	}
   }
   else
   {
	   if (activeAnnouncement.tunerSource == sourceId && tunerSrcActivity == tuner::common::SRCOFF)
	   {
		   requestedAnnouncement.tunerSource = sourceId;
		   requestedAnnouncement.tunerSrcActivity = tunerSrcActivity;
		   requestedAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_REQUEST_IN_PROGRESS;
		   requestedAnnouncement.tunerState = tunerState;
		   ETG_TRACE_USR4(("ANNOUNCEMENTS: vOnSourceChangeResponse  tunerState = %d\n", requestedAnnouncement.tunerState));
		   vSynchronize();
	   }
   }
}


/**************************************************************************//**
*
******************************************************************************/
bool clAnnouncements::bIsRequestedAnnouncementActive()
{
	if (requestedAnnouncement.announcementId == activeAnnouncement.announcementId)
	{
		if (activeAnnouncement.bIsActivateRequest())
		{
			return true;
		}
	}
	return false;
}


/**************************************************************************//**
*
******************************************************************************/
tuner::common::enAnnouncement clAnnouncements::oGetAnnouncement(tuner::common::enAnnouncement announcementId, tuner::common::enSourceId tunerSourceId, tuner::common::enSrcActivity tunerSrcActivity)
{
	if(announcementId == ANNOUNCEMENT_LIMIT)
	{
		if (tunerSrcActivity == tuner::common::SRCOFF && activeAnnouncement.tunerSource == tunerSourceId)
		{
			return activeAnnouncement.announcementId;
		}
		return clAnnouncementSettings::instance()->oGetHighestPriorityAnnouncement(tunerSourceId);
	}
	return announcementId;
}


/**************************************************************************//**
*
******************************************************************************/
stTunerAnnouncementInformation clAnnouncements::oGetHighestPriorityAnnouncement(tunerVector<tuner::announcements::stTunerAnnouncementInformation>  announcementList)
{
	return clAnnouncementSettings::instance()->oGetHighestPriorityAnnouncement(announcementList);
}


/**************************************************************************//**
*
******************************************************************************/
bool clAnnouncements::bIsAnnouncementOfMainSource()
{

	//In FM- FMTA, DAB-DAB TA should return true
	if((oGetMainSourceId(requestedAnnouncement.tunerSource) == activesession.tunerSource) && (activesession.tunerSrcActivity !=tuner::common::SRCOFF && activesession.tunerSrcActivity !=tuner::common::SRCACTIVITY_UNKNOWN ))
	{
		ETG_TRACE_USR4(("AnnoucementValidation:: Announcement is from Main source"));
		return true;
	}
	else
	{
		ETG_TRACE_USR4(("AnnoucementValidation::Announcement is not from Main source"));
		return false;
	}
}
/**************************************************************************//**
*
******************************************************************************/
bool clAnnouncements::bIsTunerInBackGround()
{
	if (audioControl.getActiveSourceActivity() == tuner::common::SRCON)
	{
		return false;
	}
	ETG_TRACE_USR4(("bIsTunerInBackGround() returns true"));
	return true;
}


/**************************************************************************//**
* FM-DAB TA, AM-FMTA/DABTA , DAB-FMTA, AUX- FMTA/DABTA
******************************************************************************/
bool clAnnouncements::bPlayAnnouncementInNonRadio()
{
	bool isAnnouFromPreferredSrc = clAnnouncementSettings::instance()->bPlayOtherAnnouncements(SOURCEID_LIMIT, requestedAnnouncement.tunerSource);
	//if announcement is from the preferred source
	if (isAnnouFromPreferredSrc)
	{
		return true;
	}
	//if the announcement is from non preferred source,check the preferred source TP status
	else
	{
		tuner::common::enSourceId requestedAnnounSourceId =oGetMainSourceId(requestedAnnouncement.tunerSource);
		if(requestedAnnounSourceId==tuner::common::DAB)
		{
			if (clAnnouncementSettings::instance()->bIsTpSupportedInBand(tuner::common::FM1))
			{
				ETG_TRACE_USR4(("AnnoucementValidation:: Announcement from DAB Source is ignored, bec preferred FM source is tuned to TP station"));
				return false;
			}
		}
		else
		{
			if (clAnnouncementSettings::instance()->bIsTpSupportedInBand(tuner::common::DAB))
			{
				ETG_TRACE_USR4(("AnnoucementValidation:: Announcement from FM Source is ignored, bec preferred DAB source is tuned to TP station"));
				return false;
			}
		}
		return true;
	}
}
bool clAnnouncements::bPlayOtherAnnouncements()
{
	bool blret=false;
	//In non radio source, announcement is active
	if(activesession.tunerSrcActivity ==tuner::common::SRCOFF ||activesession.tunerSrcActivity ==tuner::common::SRCACTIVITY_UNKNOWN )
	{
		ETG_TRACE_USR4(("AnnoucementValidation:: Announcement in Non-Radio"));
		return bPlayAnnouncementInNonRadio();
	}


	//In FM-DAB TA, DAB-FMTA, AM-DABTA/FMTA
	else if(oGetMainSourceId(requestedAnnouncement.tunerSource) != activesession.tunerSource)
	{
		//Special case to Handle DAB-TA in AM source, since we don't have any KDS parameter
		if(activesession.tunerSource==tuner::common::AM)
		{
			ETG_TRACE_USR4(("AnnoucementValidation:: Announcement in AM source"));
			//If FM-TA shd put through in AM
			if(clAnnouncementSettings::instance()->bPlayOtherAnnouncements(activesession.tunerSource,tuner::common::FMANNOUNCEMENTS))
			{
				if((clAnnouncementSettings::instance()->bIsTpSupportedInBand(tuner::common::FM1)) && (oGetMainSourceId(requestedAnnouncement.tunerSource) ==tuner::common::DAB ))
				{
					//return false;
				}
				else
				{
					blret=true;
				}
			}
			//FM-TA not supported in AM
			else
			{
				/*Weather alert from weather band should be supported in all source. No KDS parameter available */
			   if((requestedAnnouncement.tunerSource ==tuner::common::DAB) || (requestedAnnouncement.tunerSource ==tuner::common::WEATHERANNOUNCEMENTS))
				{
					blret=true;
				}
				//else return false;
			}
		}


		//In FM-DAB TA, DAB-FMTA
		else if(clAnnouncementSettings::instance()->bPlayOtherAnnouncements(activesession.tunerSource,(tuner::common::enSourceId)u8getSourceFromAnnouncement(requestedAnnouncement.announcementId)))
		{
			ETG_TRACE_USR4(("AnnoucementValidation:: Announcement from Other source "));
			//If the TP is supported by the current source
			if(clAnnouncementSettings::instance()->bIsTpSupportedInBand(activesession.tunerSource))
			{
				//return false;
			}
			//If the TP is not supported by the current source
			else
			{
				blret= true;
			}
			//Weather alert support  only in NAR, in NAR pty31 announcement active, Tp state will be set to true.
			if(requestedAnnouncement.tunerSource == tuner::common::WEATHERANNOUNCEMENTS)
			{
			   blret= true;
			}
		}
	}
	ETG_TRACE_USR4(("AnnoucementValidation:: bPlayOtherAnnouncements() returns =%d",blret));
	return blret;
}
/**************************************************************************//**
*
******************************************************************************/
bool clAnnouncements::isPreferredAnnouncementSource(bool isDABPreferred, tuner::common::enSourceId tunerSourceId )
{
	if(isDABPreferred)
	{
		if(tunerSourceId == tuner::common::DAB)
		{
			return true;
		}
	}
	else
	{
		if(tunerSourceId == tuner::common::FM1)
		{
			return true;
		}
	}
	ETG_TRACE_USR4(("isPreferredAnnouncementSource(): returns False "));
	return false;
}
/**************************************************************************//**
*
******************************************************************************/
tU8 clAnnouncements::u8getSourceFromAnnouncement(tuner::common::enAnnouncement announcementId)
{
	switch(announcementId)
	{
	case tuner::common::FMTA:
	case tuner::common::PTY31FM:
		return (tU8)tuner::common::FMANNOUNCEMENTS;
	case tuner::common::DABALARM:
	case tuner::common::DABTA		  	 :
	case tuner::common::DABTRANSPORT 	 :
	case tuner::common::DABWARNING 	 :
	case tuner::common::DABNEWS   	:
	case tuner::common::DABWEATHER     :
	case tuner::common::DABEVENT 		 :
	case tuner::common::DABSPECIALEVENT:
	case tuner::common::DABPROGRAMME	 :
	case tuner::common::DABSPORTS		 :
	case tuner::common::DABFINANCE     :
		return (tU8) tuner::common::DABANNOUNCEMENTS;
	case tuner::common::WEATHERALERT:
	   return (tU8) tuner::common::WEATHERANNOUNCEMENTS;
	default:
		return (tU8)tuner::common::ANNOUNCEMENT_LIMIT;

	}
}



/**************************************************************************//**
*
******************************************************************************/
bool clAnnouncements::bIsServiceFollowingActive()
{
     ETG_TRACE_USR4(("bIsServiceFollowingActive() returns =%d",
	 clAnnouncementSettings::instance()->bIsServiceFollowingActive(audioControl.getActiveSource())));
	return clAnnouncementSettings::instance()->bIsServiceFollowingActive(audioControl.getActiveSource());
}


/**************************************************************************//**
*
******************************************************************************/
void clAnnouncements::vCancelAnnouncement()
{
	if (activeAnnouncement.bIsAnnouncementRequestCompleted() && activeAnnouncement.bIsActivateRequest())
	{
	   /*ADR3 API doesn't support canceling the weather alert by CM_AnnouncementEscape.
	    Hence FID_TUN_S_ANNOUNCEMENT_ESCAPE is not extend.
	    On FID_TUNMSTR_S_STOP_ANNOUNCEMENT, Weather alert off request will be send to ARL*/
	   if((activeAnnouncement.announcementId == tuner::common::WEATHERALERT) &&
	            (activeAnnouncement.tunerSource == tuner::common::WEATHERANNOUNCEMENTS))
	   {
	      audioControl.vRequestSourceChange(tuner::common::WEATHERANNOUNCEMENTS, tuner::common::SRCOFF);
	   }
	   else
         tunerControl->vCancelAnnouncement(activeAnnouncement.tunerSource, activeAnnouncement.announcementId);
	}
}
/**************************************************************************//**
*
******************************************************************************/
tuner::common::enAnnouncement clAnnouncements::oGetActiveAnnouncementId()
{
	return activeAnnouncement.announcementId;
}
/**************************************************************************//**
*
******************************************************************************/
tuner::common::enSrcActivity clAnnouncements::oGetActiveAnnouncementActivity()
{
	return activeAnnouncement.tunerSrcActivity;
}
/**************************************************************************//**
*
* FUNCTION:     clAnnouncements::oGetActiveSessionActivity()
 *
 * DESCRIPTION:  This function will update the src activity (Of Main source)
 *
 * PARAMETER:    void
 *
 * RETURNVALUE: tuner::common::enSrcActivity  
Added for NCG3D-39180 fix 
******************************************************************************/
tuner::common::enSrcActivity clAnnouncements::oGetActiveSessionActivity()
{
	return activesession.tunerSrcActivity;
}
/*****
 *
 */
tuner::announcements::clAnnouncementState::enState clAnnouncements::oGetActiveAnnouncementState()
{
	return activeAnnouncement.state;
}
/*****
 *
 */
void clAnnouncements::vSetPICode(unsigned short piCode)
{
	PICode = piCode;
	ETG_TRACE_USR4(("vSetPICode piCode=%d",piCode));
}
/******
 *
 */
unsigned short clAnnouncements::oGetPICode()
{
	return PICode;
}
/******
 *
 */
void clAnnouncements::vOnNewTunerState(tuner::common::enTunerState tunerState)
{
	ETG_TRACE_USR4(("ANNOUNCEMENTS: vOnNewTunerState received tunerState=%d",tunerState));
	/*if expected tuner state is received, make announcement state as completed and send announcement status to clients (handled in vOnAnnouncementCompleted)
	 * else make announcement state as failed and reset active session*/
	if((tunerState == tuner::common::TAFG) &&
			((requestedAnnouncement.tunerSrcActivity == tuner::common::SRCON) || (requestedAnnouncement.tunerSrcActivity == tuner::common::SRCPAUSE)))
	{
		requestedAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_COMPLETED;
	}
	else if((tunerState != tuner::common::TAFG) && (requestedAnnouncement.tunerSrcActivity == tuner::common::SRCOFF))
	{
		/*Work around for receiving 2 response for MTCAtAudioSource from SBR */
		if(activeAnnouncement.state == clAnnouncementState::ANNOUNCEMENT_COMPLETED) {
			ETG_TRACE_USR4(("ANNOUNCEMENTS: vOnNewTunerState No action. already SRCOFF completed"));
			return;
		}
		requestedAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_COMPLETED;
	}
	else
	{
		requestedAnnouncement.state = clAnnouncementState::ANNOUNCEMENT_FAILED;
	}
	vPrintAnnouncementInformation();
	vSynchronize();
}

/**************************************************************************//**
*
* FUNCTION:     clAnnouncements::bIsValidAnnouncementType()
 *
 * DESCRIPTION:  This is added to validate DAB Announcement type
 *
 * PARAMETER:    void
 *
 * RETURNVALUE: bool valid TRUE or invalid type FALSE
 *
 * Info : Fix for Bug Bug 255233

******************************************************************************/
bool clAnnouncements::bIsValidAnnouncementType(tuner::common::enAnnouncement announcementId )
{
	if( tuner::common::ANNOUNCEMENT_LIMIT != u8getSourceFromAnnouncement(announcementId) )
	{
		return true;
	}
	else
	{
		ETG_TRACE_USR4(("bIsValidAnnouncementType: Invalid announcementId=%d",announcementId));
		return false;
	}
}
