/************************************************************************
 * FILE:           tunmstr_SDSHandler.cpp
 * PROJECT:        G3g
 * SW-COMPONENT:   FC_TunerMaster.
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:    
 *
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date			 | Author						| Modification
 * 	 			 | auj1kor 						| initial version
 *************************************************************************/

#include "fc_tunermaster_featureControl.h"

#include "tunmstr_SDSHandler.h"
#include "tunmstr_MsgFromHMI.h"
#include "tunmstr_SrcChgManager.h"
#include "tunmstr_TunerClientHandler.h"
#include "tunmstr_DABClient_Handler.h"
#include "tunmstr_SrcChg_Defines.h"
#include "tunmstr_Preset_Handler.h"
#include "kdsconfig/clConfigInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_SXMCLIENT
#include "tunermaster_SXMClient.h"
#endif

#include "tunmstr_SrcChg_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_GENERICMSGS_MOCK
#include "ahl_mock/ahl_if.h"
#include "amt_mock/amt_if.h"
#else
#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_SRCCHANGE_LIB
#define TUN_CONFIG_DEFAULT_TRF_LOW_FREQ  1620
#define TUN_CONFIG_DEFAULT_TRF_HIGH_FREQ  1629
#define TUN_CONFIG_DEFAULT_COMBI_BANDS 0x10000006
#define TUN_MIDW_COMBIBAND_ENABLE 0x0006
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUNMSTR_TRACE_CLASS_SRCCHG
#include "trcGenProj/Header/tunmstr_SDSHandler.cpp.trc.h"
#endif

/*************************************************************************
 * FUNCTION:     tunmstr_SDSHandler
 *
 * DESCRIPTION:  Initialize all the member variable to default values
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_SRCCHANGE_LIB
tunmstr_SDSHandler::tunmstr_SDSHandler(tuner::sessionManagement::audio::clArlSourceImpl* poArlImpl)
#else
tunmstr_SDSHandler::tunmstr_SDSHandler()
#endif
{
	m_potunmstr_MsgFromHMI = NULL;
	m_potunmstr_SrcChgManager = NULL;
	m_potunmstr_TunerClientHandler=NULL;
	m_potunmstr_DABClient_Handler=NULL;
	m_potunmstr_MsgToHMI = NULL;
	m_potunmstr_Preset_Handler = NULL;
	m_poclConfigInterface = NULL;
#ifdef VARIANT_S_FTR_ENABLE_SXMCLIENT
	m_potunermaster_SXMClient = NULL;
#endif

#ifdef VARIANT_S_FTR_ENABLE_SRCCHANGE_LIB
	m_poArlImpl = poArlImpl;
	vClearSDSRequestParams();
#endif
	m_u16RegID = AMT_C_U16_REGID_INVALID;
	m_u16CmdCntr = 0;
	m_u16SourceAppID = 0;
	m_u8BandReqFromHMI = (tU8)midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_NONE;
}

/*************************************************************************
 * FUNCTION:     ~tunmstr_SDSHandler
 *
 * DESCRIPTION:  Destructor
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *************************************************************************/
tunmstr_SDSHandler::~tunmstr_SDSHandler()
{
	m_potunmstr_MsgFromHMI = NULL;
	m_potunmstr_SrcChgManager = NULL;
	m_potunmstr_TunerClientHandler=NULL;
	m_potunmstr_DABClient_Handler=NULL;
	m_potunmstr_MsgToHMI = NULL;
	m_potunmstr_Preset_Handler = NULL;
	m_poclConfigInterface = NULL;
#ifdef VARIANT_S_FTR_ENABLE_SXMCLIENT
	m_potunermaster_SXMClient = NULL;
#endif

#ifdef VARIANT_S_FTR_ENABLE_SRCCHANGE_LIB
	m_poArlImpl = NULL;
	vClearSDSRequestParams();
#endif
	m_u16RegID = AMT_C_U16_REGID_INVALID;
	m_u16CmdCntr = 0;
	m_u16SourceAppID = 0;
	m_u8BandReqFromHMI = (tU8)midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_NONE;
}


/*************************************************************************
 * FUNCTION:    vSet_tunmstr_MsgFromHMI_Ptr
 *
 * DESCRIPTION:  Update the tunmstr_MsgFromHMI pointer member variable m_potunmstr_MsgFromHMI
 *
 * PARAMETER:    tunmstr_MsgFromHMI* potunmstr_MsgFromHMI
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_SDSHandler::vSet_tunmstr_MsgFromHMI_Ptr( tunmstr_MsgFromHMI* potunmstr_MsgFromHMI )
{
	if( potunmstr_MsgFromHMI != NULL )
	{
		m_potunmstr_MsgFromHMI = ( tunmstr_MsgFromHMI* )potunmstr_MsgFromHMI;
	}
	else
	{
		m_potunmstr_MsgFromHMI = NULL;
		NORMAL_M_ASSERT ( m_potunmstr_MsgFromHMI != NULL );
	}
}

/*************************************************************************
 * FUNCTION:    vSet_tunmstr_SrcChgManager_Ptr
 *
 * DESCRIPTION:  Update the tunmstr_SrcChgManager pointer member variable m_potunmstr_SrcChgManager
 *
 * PARAMETER:    tunmstr_SrcChgManager* potunmstr_SrcChgManager
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_SDSHandler::vSet_tunmstr_SrcChgManager_Ptr( tunmstr_SrcChgManager* potunmstr_SrcChgManager )
{
	if( potunmstr_SrcChgManager != NULL )
	{
		m_potunmstr_SrcChgManager = ( tunmstr_SrcChgManager* )potunmstr_SrcChgManager;
	}
	else
	{
		m_potunmstr_SrcChgManager = NULL;
		NORMAL_M_ASSERT ( m_potunmstr_SrcChgManager != NULL );
	}
}

/*************************************************************************
 * FUNCTION:     vSet_tunmstr_TunerClientHandler_Ptr
 *
 * DESCRIPTION:  Update the tunmstr_TunerClientHandler pointer member variable m_potunmstr_TunerClientHandler
 *
 * PARAMETER:    tunmstr_TunerClientHandler* potunmstr_TunerClientHandler
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_SDSHandler::vSet_tunmstr_TunerClientHandler_Ptr( tunmstr_TunerClientHandler* potunmstr_TunerClientHandler )
{
	if( potunmstr_TunerClientHandler != NULL )
	{
		m_potunmstr_TunerClientHandler = ( tunmstr_TunerClientHandler* )potunmstr_TunerClientHandler;
	}
	else
	{
		m_potunmstr_TunerClientHandler = NULL;
		NORMAL_M_ASSERT ( m_potunmstr_TunerClientHandler != NULL );
	}
}

/*************************************************************************
 * FUNCTION:     vSet_tunmstr_DABClient_Handler_Ptr
 *
 * DESCRIPTION:  Update the tunmstr_DABClient_Handler pointer member variable m_potunmstr_DABClient_Handler
 *
 * PARAMETER:    tunmstr_DABClient_Handler* potunmstr_DABClient_Handler
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_SDSHandler::vSet_tunmstr_DABClient_Handler_Ptr(tunmstr_DABClient_Handler* potunmstr_DABClient_Handler)
{
	if( potunmstr_DABClient_Handler != NULL )
	{
		m_potunmstr_DABClient_Handler = ( tunmstr_DABClient_Handler* )potunmstr_DABClient_Handler;
	}
	else
	{
		m_potunmstr_DABClient_Handler = NULL;
		NORMAL_M_ASSERT ( m_potunmstr_DABClient_Handler != NULL );
	}
}

/*************************************************************************
 * FUNCTION:     vSet_tunmstr_Preset_Handler_Ptr
 *
 * DESCRIPTION:  Update the m_potunmstr_Preset_Handler pointer member variable
 *
 * PARAMETER:    tunmstr_Preset_Handler* potunmstr_Preset_Handler
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_SDSHandler::vSet_tunmstr_Preset_Handler_Ptr(tunmstr_Preset_Handler* potunmstr_Preset_Handler)
{
	if( potunmstr_Preset_Handler != NULL )
	{
		m_potunmstr_Preset_Handler = ( tunmstr_Preset_Handler* )potunmstr_Preset_Handler;
	}
	else
	{
		m_potunmstr_Preset_Handler = NULL;
		NORMAL_M_ASSERT ( m_potunmstr_Preset_Handler != NULL );
	}
}


/*************************************************************************
* FUNCTION:    vSet_tunmstr_ConfigInterface_Ptr
*
* DESCRIPTION:
*
* PARAMETER:   clConfigInterface* poclConfigInterface
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_SDSHandler::vSet_tunmstr_ConfigInterface_Ptr(clConfigInterface* poclConfigInterface)
{
	if( poclConfigInterface != NULL )
	{
		m_poclConfigInterface = poclConfigInterface;
	}
	else
	{
		m_poclConfigInterface = NULL;
		NORMAL_M_ASSERT ( m_poclConfigInterface != NULL );
	}

}

#ifdef VARIANT_S_FTR_ENABLE_SXMCLIENT
/*************************************************************************
* FUNCTION:    vSet_tunmstr_SXMClient_Handler_Ptr
*
* DESCRIPTION:
*
* PARAMETER:   tunermaster_SXMClient* potunermaster_SXMClient
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_SDSHandler::vSet_tunmstr_SXMClient_Handler_Ptr(tunermaster_SXMClient* potunermaster_SXMClient)
{
	if( potunermaster_SXMClient != NULL )
	{
		m_potunermaster_SXMClient = (tunermaster_SXMClient*)potunermaster_SXMClient;
	}
	else
	{
		m_potunermaster_SXMClient = NULL;
		NORMAL_M_ASSERT ( m_potunermaster_SXMClient != NULL );
	}

}
#endif


/*************************************************************************
* FUNCTION:    vSet_tunmstr_MsgToHMI_Ptr
*
* DESCRIPTION:
*
* PARAMETER:   tunmstr_MsgToHMI* potunmstr_MsgToHMI
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_SDSHandler::vSet_tunmstr_MsgToHMI_Ptr( tunmstr_MsgToHMI* potunmstr_MsgToHMI )
{
	if( potunmstr_MsgToHMI != NULL )
	{
		m_potunmstr_MsgToHMI = potunmstr_MsgToHMI;
	}
	else
	{
		m_potunmstr_MsgToHMI = NULL;
		NORMAL_M_ASSERT ( m_potunmstr_MsgToHMI != NULL );
	}

}

/*************************************************************************
*
* FUNCTION:     tun_Config:: bIsFrequencyWithinTheRange( )
*
* DESCRIPTION:  Returns true if the Frequency is within the Range
*
* PARAMETER:    tU32 u32Frequncy
*
* RETURNVALUE:  tBool
*
*************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_SRCCHANGE_LIB
tBool tunmstr_SDSHandler::bIsFrequencyWithinTheRange(tU8 u8Band, tU32 u32Frequency)const
{
	tBool flag = false;

	if (m_poclConfigInterface == NULL)
	{
		NORMAL_M_ASSERT(m_poclConfigInterface != NULL);
		return false;
	}

	switch (u8Band)
	{
	case midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_FM:
		if (u32Frequency >= (unsigned int)m_poclConfigInterface->u32GetConfigData("LowFM") && u32Frequency <= (unsigned int)m_poclConfigInterface->u32GetConfigData("HighFM"))
		{
			flag = true;
		}
		break;

	case midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_LW:
	case midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_MW:
		if ((u32Frequency >= (unsigned int)m_poclConfigInterface->u32GetConfigData("LowLW") && u32Frequency <= (unsigned int)m_poclConfigInterface->u32GetConfigData("HighLW")) ||
			(u32Frequency >= (unsigned int)m_poclConfigInterface->u32GetConfigData("LowMW") && u32Frequency <= (unsigned int)m_poclConfigInterface->u32GetConfigData("HighMW")))
		{
			flag = true;
		}
		break;
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	case midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_CB1:
	{
		if (m_poclConfigInterface->u32GetConfigData("CombiBands") == TUN_CONFIG_DEFAULT_COMBI_BANDS)
		{
			if ((u32Frequency >= (unsigned int)m_poclConfigInterface->u32GetConfigData("LowLW") && u32Frequency <= (unsigned int)m_poclConfigInterface->u32GetConfigData("HighLW")) ||
				(u32Frequency >= (unsigned int)m_poclConfigInterface->u32GetConfigData("LowMW") && u32Frequency <= (unsigned int)m_poclConfigInterface->u32GetConfigData("HighMW")))
			{
				flag = true;
			}
		}
		else
		{
			if (u32Frequency >= (unsigned int)m_poclConfigInterface->u32GetConfigData("LowMW") && u32Frequency <= (unsigned int)m_poclConfigInterface->u32GetConfigData("HighMW"))
			{
				flag = true;
			}
		}
	}
	break;
#endif
	case midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_TRF:
	{
		if ((u32Frequency >= (unsigned int)TUN_CONFIG_DEFAULT_TRF_LOW_FREQ) && (u32Frequency <= (unsigned int)TUN_CONFIG_DEFAULT_TRF_HIGH_FREQ))
		{
			flag = true;
		}
		break;
	}
	case midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_WB:
	{
		if (u32Frequency >= (unsigned int)m_poclConfigInterface->u32GetConfigData("LowWB") && u32Frequency <= (unsigned int)m_poclConfigInterface->u32GetConfigData("HighWB"))
		{
			flag = true;
		}
		break;
	}
	default:
		flag = false;
		break;
	}

	return(flag);

}

/*******************************************************************************
 *FUNCTION:    tunmstr_SDSHandler::vHandleStationSelection
 *
 *DESCRIPTION: This function will extarct CCA messeage for station selection requests from SDS
 *
 *PARAMETER:    poMessage
 *
 *AUTHOR:
 *RETURNVALUE: None
 ********************************************************************************/
tVoid tunmstr_SDSHandler::vHandleStationSelection(amt_tclServiceData* poMessage)
{
	// Check message pointer.
	if( poMessage == NULL || m_potunmstr_MsgFromHMI == NULL || m_poclConfigInterface == NULL)
	{
		NORMAL_M_ASSERT( poMessage != NULL );
		NORMAL_M_ASSERT( m_potunmstr_MsgFromHMI != NULL );
		NORMAL_M_ASSERT( m_poclConfigInterface != NULL );
		return;
	}

	midw_tunermasterfi_tclMsgFID_TUNMSTR_S_STATION_SELECTIONMethodStart  oMessageStart;
	m_potunmstr_MsgFromHMI->vGetDatafromAMT(poMessage,oMessageStart);
	vClearSDSRequestParams();
	tU8 u8Option = (tU8) oMessageStart.Option.enType;

	m_rSDSTunerRequest.u32Frequency = oMessageStart.Frequency;
	m_rSDSTunerRequest.u8ElemID = oMessageStart.u8Element_ID;
	m_rSDSTunerRequest.u8ListID = (tU8)oMessageStart.List.enType;
	m_rSDSTunerRequest.u32PI = oMessageStart.PI;
	m_rSDSTunerRequest.u16SecondaryPI = oMessageStart.SecondaryPI;
	m_rSDSTunerRequest.u8HDProgramNumber = oMessageStart.HDProgramNumber;
	if (((m_poclConfigInterface->u32GetConfigData("CombiBands") & TUN_MIDW_COMBIBAND_ENABLE) == TUN_MIDW_COMBIBAND_ENABLE) &&
		(oMessageStart.Band.enType == midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_MW || oMessageStart.Band.enType == midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_LW))
	{
		m_u8BandReqFromHMI = vConvertSessionBand((tU8)oMessageStart.Band.enType);
		ETG_TRACE_USR4((" vHandleStationSelection CommbiBands LW, MW , m_u8BandReqFromHMI = %d ", m_u8BandReqFromHMI));
		oMessageStart.Band.enType = midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_CB1;
	}

	m_rSDSTunerRequest.u8Band = vConvertSessionBand((tU8)oMessageStart.Band.enType);

	if(m_rSDSTunerRequest.u8Band == (tU8)tuner::common::BANDNONE)
	{
		if(m_poclConfigInterface->u32GetConfigData("PresetListType"))
		{
			m_rSDSTunerRequest.u8Band = vConvertSessionBand(m_potunmstr_Preset_Handler->e8GetPresetBand(m_rSDSTunerRequest.u8ElemID).enType);
		}
		else
		{
			if(((tU8)m_poArlImpl->getActiveSourceActivity() == (tU8)tuner::common::SRCOFF) || ((tU8)m_poArlImpl->getActiveSourceActivity() == (tU8)tuner::common::SRCACTIVITY_UNKNOWN))
			{
				m_rSDSTunerRequest.u8Band = (tU8)tuner::common::BANDSXM;
			}
			else
			{
				m_rSDSTunerRequest.u8Band = (tU8)(clSessionControl::getInstance()->getActiveTunerBand());
			}
		}
	}
	m_u16RegID = poMessage->u16GetRegisterID();
	m_u16CmdCntr = poMessage->u16GetCmdCounter();
	m_u16SourceAppID = poMessage->u16GetSourceAppID();

	ETG_TRACE_USR4((" vHandleStationSelection(), option : %d", u8Option));

	switch(oMessageStart.Option.enType)
	{
		case midw_fi_tcl_e8_StationSelection_Option::FI_EN_FREQUENCY_INPUT :
		{
			if(bIsFrequencyWithinTheRange(m_rSDSTunerRequest.u8Band, m_rSDSTunerRequest.u32Frequency))
				vHandleDirectFrequency();
			else
				vSendMethodResult(m_rSDSTunerRequest.u8Band, (tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE, (tU8)midw_fi_tcl_e8_TunMstr_ResponseStatus::FI_EN_TUN_MSTR_RES_REQ_FAIL);
			break;
		}
		case midw_fi_tcl_e8_StationSelection_Option::FI_EN_PRESET_INPUT :
		{
			vHandleStationSelctionByPresetNumber();
			break;
		}
		case midw_fi_tcl_e8_StationSelection_Option::FI_EN_STATIONNAME_INPUT :
		{
			vHandleStationSelctionByStationName((tU8)oMessageStart.ModulationType.enType);
			break;
		}
		default :
		{
			ETG_TRACE_USR4((" vHandleStationSelection(), invalid option"));
			vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE,(tU8)midw_fi_tcl_e8_TunMstr_ResponseStatus::FI_EN_TUN_MSTR_RES_REQ_FAIL);
			break;
		}
	}

}
/*******************************************************************************
 *FUNCTION:    tunmstr_SDSHandler::vHandleDirectFrequency
 *
 *DESCRIPTION: This function will handle SDS direct frequency input
 *
 *PARAMETER:
 *
 *AUTHOR:
 *RETURNVALUE: None
 ********************************************************************************/
tVoid tunmstr_SDSHandler::vHandleDirectFrequency()
{
	if( m_potunmstr_TunerClientHandler == NULL)
	{
		NORMAL_M_ASSERT( m_potunmstr_TunerClientHandler != NULL );
		return;
	}

	if(bCheckReqChangeLSM())
	{
		m_rSDSTunerRequest.u8Option = 0x00;
		m_potunmstr_TunerClientHandler->vSend_s_ChangeLSM();
		vSendMethodResult(m_rSDSTunerRequest.u8Band,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_PLAY);
	}
	else
	{
		m_potunmstr_TunerClientHandler->vSend_s_SetPIFreq_Input(m_rSDSTunerRequest.u32Frequency,(tU16)m_rSDSTunerRequest.u32PI);
		vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE);
	}
}


/*************************************************************************
 * FUNCTION:		vHandleStationSelctionByPresetNumber()
 *
 **DESCRIPTION: 	This function will handle SDS preset input
 *
 * PARAMETER:		amt_tclServiceData* poMessage
 *
 * RETURNVALUE:		tVoid
 *************************************************************************/
tVoid tunmstr_SDSHandler:: vHandleStationSelctionByPresetNumber()
{

	switch(m_rSDSTunerRequest.u8Band)
	{
		case (tU8)tuner::common::BANDFM:
		case (tU8)tuner::common::BANDMW:
		case (tU8)tuner::common::BANDLW:
		case (tU8)tuner::common::BANDCB1:
		{
			vHandleTunerPresetSelect();
			break;
		}

		case (tU8)tuner::common::BANDDAB:
		{
			vHandleDABPresetSelect();
			break;
		}

		case (tU8)tuner::common::BANDSXM:
		{
#ifdef VARIANT_S_FTR_ENABLE_SXMCLIENT
			vHandleSXMPresetSelect();
#endif
#ifdef VARIANT_S_FTR_ENABLE_SXM360LCLIENT
			vHandleSXM360LPresetSelect();
#endif
			break;
		}
		default:
		{
			ETG_TRACE_USR4((" vHandleStationSelection(), invalid band"));
			vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE,(tU8)midw_fi_tcl_e8_TunMstr_ResponseStatus::FI_EN_TUN_MSTR_RES_REQ_FAIL);
			break;
		}
	}

}


/*************************************************************************
 * FUNCTION:		vHandleDABPresetSelect()
 *
 **DESCRIPTION:		handles DAB preset request
 *
 * PARAMETER:
 *
 * RETURNVALUE:		tVoid
 *************************************************************************/
tVoid tunmstr_SDSHandler:: vHandleDABPresetSelect()
{
	if( m_potunmstr_DABClient_Handler == NULL )
	{
		NORMAL_M_ASSERT( m_potunmstr_DABClient_Handler != NULL );
		return;
	}

	ETG_TRACE_USR4((" vHandleDABPresetSelect(), presetID : %d",m_rSDSTunerRequest.u8ElemID));

	midw_ext_dabtunerfi_tclMsgFID_DAB_S_ACTIVATE_PRESET_LIST_ELEMMethodStart oMethodStart;
	oMethodStart.u8ActPresetElemID=m_rSDSTunerRequest.u8ElemID;
	m_potunmstr_DABClient_Handler->vSendMessage( CCA_C_U16_APP_FC_DABTUNER,
			oMethodStart,           // parameter&
			MIDW_EXT_DABTUNERFI_C_U16_FID_DAB_S_ACTIVATE_PRESET_LIST_ELEM,
			AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
			0 ,//CmdCntr
			m_potunmstr_DABClient_Handler->u16GetDABServiceRegID());
	if(bCheckReqChangeLSM())
	{
		vSendMethodResult((tU8)tuner::common::BANDDAB,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_PLAY);
	}
	else
	{
		vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE);
	}
}


/*************************************************************************
 * FUNCTION:		vHandleTunerPresetSelect()
 *
 **DESCRIPTION:		handles AM FM preset request
 *
 * PARAMETER:
 *
 * RETURNVALUE:		tVoid
 *************************************************************************/
tVoid tunmstr_SDSHandler:: vHandleTunerPresetSelect()
{
	if( m_potunmstr_TunerClientHandler == NULL )
	{
		NORMAL_M_ASSERT( m_potunmstr_TunerClientHandler != NULL );
		return;
	}

	ETG_TRACE_USR4((" vHandleTunerPresetSelect(), presetID : %d",m_rSDSTunerRequest.u8ElemID));

	if(bCheckReqChangeLSM())
	{
		tU8 u8Band = m_rSDSTunerRequest.u8Band;
		m_rSDSTunerRequest.u8Option = 0x03;
		m_potunmstr_TunerClientHandler->vSend_s_ChangeLSM();
		if(m_rSDSTunerRequest.u8Band == (tU8)tuner::common::BANDLW || m_rSDSTunerRequest.u8Band == (tU8)tuner::common::BANDCB1)
		{
			u8Band = (tU8)tuner::common::BANDMW;
		}
		vSendMethodResult(u8Band,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_PLAY);
	}
	else
	{
		midw_tunerfi_tclMsgFID_TUN_S_PRESET_HANDLINGMethodStart oMethodStart;
		oMethodStart.e8Action.enType	= 	midw_fi_tcl_e8_Tun_PresetAction::FI_EN_RECALL_ELEMENT;
		oMethodStart.u8ListElementID	=	m_rSDSTunerRequest.u8ElemID;
		oMethodStart.e8ListID.enType	=	(midw_fi_tcl_e8_Tun_ListID::tenType)m_rSDSTunerRequest.u8ListID;

		m_potunmstr_TunerClientHandler->vSendMessage( CCA_C_U16_APP_TUNER,// Destination APP ID
				oMethodStart,           // parameter&
				MIDW_TUNERFI_C_U16_FID_TUN_S_PRESET_HANDLING, //FID
				AMT_C_U8_CCAMSG_OPCODE_METHODSTART,  // OpCode                           /* amt_tclServiceData* */
				0 );

		vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE);

	}
}


#ifdef VARIANT_S_FTR_ENABLE_SXMCLIENT
/*************************************************************************
 * FUNCTION:		vHandleSXMPresetSelect()
 *
 **DESCRIPTION:		handles SXM preset request
 *
 * PARAMETER:
 *
 * RETURNVALUE:		tVoid
 *************************************************************************/
tVoid tunmstr_SDSHandler:: vHandleSXMPresetSelect()
{
	if( m_potunermaster_SXMClient == NULL )
	{
		NORMAL_M_ASSERT( m_potunermaster_SXMClient != NULL );
		return;
	}

	ETG_TRACE_USR4((" vHandleSXMPresetSelect(), presetID : %d",m_rSDSTunerRequest.u8ElemID));

    midw_ext_sxm_audiofi_tclMsgRecallPresetMethodStart oMethodStart;
	oMethodStart.PresetElement = m_rSDSTunerRequest.u8ElemID;
	m_potunermaster_SXMClient->vSendMessage( CCA_C_U16_APP_FC_SXM,// Destination APP ID
		oMethodStart,           // parameter&
		MIDW_EXT_SXM_AUDIOFI_C_U16_RECALLPRESET, //FID
		AMT_C_U8_CCAMSG_OPCODE_METHODSTART,  // OpCode
		0);

	if(bCheckReqChangeLSM())
	{
		vSendMethodResult((tU8)tuner::common::BANDSXM,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_PLAY);
	}
	else
	{
		vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE);
	}
}
#endif


#ifdef VARIANT_S_FTR_ENABLE_SXM360LCLIENT
/*************************************************************************
 * FUNCTION:		vHandleSXM360LPresetSelect()
 *
 **DESCRIPTION:		handles SXM preset request for 360L
 *
 * PARAMETER:
 *
 * RETURNVALUE:		tVoid
 *************************************************************************/
tVoid tunmstr_SDSHandler:: vHandleSXM360LPresetSelect()
{
	if( m_potunmstr_MsgToHMI == NULL )
	{
		NORMAL_M_ASSERT( m_potunmstr_MsgToHMI != NULL );
		return;
	}

	ETG_TRACE_USR4((" vHandleSXM360LPresetSelect(), presetID : %d",m_rSDSTunerRequest.u8ElemID));
	midw_tunermasterfi_tclMsgFID_TUNMSTR_G_SXM_PRESET_HANDLINGStatus oStatus;
	oStatus.Preset_Action.enType = midw_fi_tcl_e8_Preset_Action::FI_EN_TUNMSTR_PRESETRECALL;
	oStatus.PresetIndex = m_rSDSTunerRequest.u8ElemID;
	m_potunmstr_MsgToHMI->vHandleSXM360LPresetRequest(oStatus);

	if(bCheckReqChangeLSM())
	{
		vSendMethodResult((tU8)tuner::common::BANDSXM,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_PLAY);
	}
	else
	{
		vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE);
	}
}
#endif


/*************************************************************************
 * FUNCTION:		vHandleStationSelctionByStationName()
 *
 **DESCRIPTION:
 *
 * PARAMETER:		tU8 u8ModulationType
 *
 * RETURNVALUE:		tVoid
 *************************************************************************/
tVoid tunmstr_SDSHandler:: vHandleStationSelctionByStationName(tU8 u8ModulationType)
{
	if( m_potunmstr_DABClient_Handler == NULL )
	{
		NORMAL_M_ASSERT( m_potunmstr_DABClient_Handler != NULL );
		return;
	}

	if(u8ModulationType == (tU8)midw_fi_tcl_e8_ModulationType::FI_EN_TUN_MSTR_MODULATION_HD)
	{
		vHandleHDStationSelection();
	}
	else if(u8ModulationType == (tU8)midw_fi_tcl_e8_ModulationType::FI_EN_TUN_MSTR_MODULATION_DRM)
	{
		vHandleDRMStationSelection();
	}
	else
	{
		switch(m_rSDSTunerRequest.u8Band)
		{
			case (tU8)tuner::common::BANDFM:
			case (tU8)tuner::common::BANDMW:
			case (tU8)tuner::common::BANDLW:
			case (tU8)tuner::common::BANDCB1:
			{
				if(bIsFrequencyWithinTheRange(m_rSDSTunerRequest.u8Band, m_rSDSTunerRequest.u32Frequency))
				vHandleDirectFrequency();
				else
				vSendMethodResult(m_rSDSTunerRequest.u8Band, (tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE, (tU8)midw_fi_tcl_e8_TunMstr_ResponseStatus::FI_EN_TUN_MSTR_RES_REQ_FAIL);
				
				break;
			}

			case (tU8)tuner::common::BANDDAB:
			{
				vHandleDABStationNameSelect();
				break;
			}

			case (tU8)tuner::common::BANDSXM:
			{
#ifdef VARIANT_S_FTR_ENABLE_SXMCLIENT
				vHandleSXMStationNameSelect();
#endif
				break;
			}
			default:
			{
				ETG_TRACE_USR4((" vHandleStationSelctionByStationName(), invalid band"));
				vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE,(tU8)midw_fi_tcl_e8_TunMstr_ResponseStatus::FI_EN_TUN_MSTR_RES_REQ_FAIL);
				break;
			}
		}
	}

}


/*************************************************************************
 * FUNCTION:		vHandleDABStationNameSelect()
 *
 **DESCRIPTION:		handles DAB station selection by PI
 *
 * PARAMETER:
 *
 * RETURNVALUE:		tVoid
 *************************************************************************/
tVoid tunmstr_SDSHandler:: vHandleDABStationNameSelect()
{
	if( m_potunmstr_DABClient_Handler == NULL )
	{
		NORMAL_M_ASSERT( m_potunmstr_DABClient_Handler != NULL );
		return;
	}

	ETG_TRACE_USR4((" vHandleDABStationNameSelect() "));
	midw_ext_dabtunerfi_tclMsgFID_DAB_S_SRV_SELECTMethodStart oMethodStart;
	oMethodStart.SelectMode.enType = (midw_ext_fi_tcl_e8_DAB_ChnSelectMode::tenType) 0x00;
	oMethodStart.u32Value = m_rSDSTunerRequest.u32PI;
	oMethodStart.u16Scids = m_rSDSTunerRequest.u16SecondaryPI;
	m_potunmstr_DABClient_Handler->vSendMessage( CCA_C_U16_APP_FC_DABTUNER,// Destination APP ID
			oMethodStart,           // parameter&
			MIDW_EXT_DABTUNERFI_C_U16_FID_DAB_S_SRV_SELECT, //FID
			AMT_C_U8_CCAMSG_OPCODE_METHODSTART,  // OpCode                           /* amt_tclServiceData* */
			0 ,//CmdCntr
			m_potunmstr_DABClient_Handler->u16GetDABServiceRegID());

	if(bCheckReqChangeLSM())
	{
		vSendMethodResult((tU8)tuner::common::BANDDAB,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_PLAY);
	}
	else
	{
		vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE);
	}
}


#ifdef VARIANT_S_FTR_ENABLE_SXMCLIENT
/*************************************************************************
 * FUNCTION:		vHandleSXMStationNameSelect()
 *
 **DESCRIPTION:     Handles station selection by CID
 *
 * PARAMETER:
 *
 * RETURNVALUE:		tVoid
 *************************************************************************/
tVoid tunmstr_SDSHandler:: vHandleSXMStationNameSelect()
{
	if( m_potunermaster_SXMClient == NULL )
	{
		NORMAL_M_ASSERT( m_potunermaster_SXMClient != NULL );
		return;
	}

	ETG_TRACE_USR4((" vHandleSXMStationNameSelect() "));
	midw_ext_sxm_audiofi_tclMsgSelectChannelMethodStart oMethodStart;
	oMethodStart.SelectionMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_CID;
	oMethodStart.CidOrSid = m_rSDSTunerRequest.u16SecondaryPI;
	oMethodStart.CategoryType = 0;
	oMethodStart.Steps = 0;
	m_potunermaster_SXMClient->vSendMessage( CCA_C_U16_APP_FC_SXM,// Destination APP ID
		oMethodStart,           // parameter&
		MIDW_EXT_SXM_AUDIOFI_C_U16_SELECTCHANNEL, //FID
		AMT_C_U8_CCAMSG_OPCODE_METHODSTART,  // OpCode
		0);

	if(bCheckReqChangeLSM())
	{
		vSendMethodResult((tU8)tuner::common::BANDSXM,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_PLAY);
	}
	else
	{
		vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE);
	}
}
#endif


/*************************************************************************
 * FUNCTION:		vHandleHDStationSelection()
 *
 **DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:		tVoid
 *************************************************************************/
tVoid tunmstr_SDSHandler:: vHandleHDStationSelection()
{
	if( m_potunmstr_TunerClientHandler == NULL )
	{
		NORMAL_M_ASSERT( m_potunmstr_TunerClientHandler != NULL );
		return;
	}

	if(bCheckReqChangeLSM())
	{
		m_rSDSTunerRequest.u8Option = 0x01;
		m_potunmstr_TunerClientHandler->vSend_s_ChangeLSM();
		vSendMethodResult(m_rSDSTunerRequest.u8Band,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_PLAY);
	}
	else
	{
		if(bIsFrequencyWithinTheRange(m_rSDSTunerRequest.u8Band, m_rSDSTunerRequest.u32Frequency))
		{
			midw_tunerfi_tclMsgFID_TUN_S_SEL_HD_AUDIOPRGMDIRECTMethodStart oMethodStart;
			oMethodStart.u32Frequency = m_rSDSTunerRequest.u32Frequency;
			oMethodStart.e8HDAudioPrg.enType = (midw_fi_tcl_e8_Tun_HDAudioProgram::tenType)m_rSDSTunerRequest.u8HDProgramNumber;
			m_potunmstr_TunerClientHandler->vSendMessage( CCA_C_U16_APP_TUNER,// Destination APP ID
				oMethodStart,           // parameter&
				MIDW_TUNERFI_C_U16_FID_TUN_S_SEL_HD_AUDIOPRGMDIRECT, //FID
				AMT_C_U8_CCAMSG_OPCODE_METHODSTART,  // OpCode                           /* amt_tclServiceData* */
				0 );
			vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE);
		}
		else
		{
			vSendMethodResult(m_rSDSTunerRequest.u8Band, (tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE, (tU8)midw_fi_tcl_e8_TunMstr_ResponseStatus::FI_EN_TUN_MSTR_RES_REQ_FAIL);
		}
	}
}


/*************************************************************************
 * FUNCTION:		vHandleDRMStationSelection()
 *
 **DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:		tVoid
 *************************************************************************/
tVoid tunmstr_SDSHandler:: vHandleDRMStationSelection()
{
	if( m_potunmstr_TunerClientHandler == NULL )
	{
		NORMAL_M_ASSERT( m_potunmstr_TunerClientHandler != NULL );
		return;
	}

	if(bCheckReqChangeLSM())
	{
		m_rSDSTunerRequest.u8Option = 0x02;
		m_potunmstr_TunerClientHandler->vSend_s_ChangeLSM();
		vSendMethodResult(m_rSDSTunerRequest.u8Band,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_PLAY);
	}
	else
	{
		if(bIsFrequencyWithinTheRange(m_rSDSTunerRequest.u8Band, m_rSDSTunerRequest.u32Frequency))
		{
			midw_tunerfi_tclMsgFID_TUN_S_SELECT_DRM_AUDIOPRGMMethodStart oMethodStart;
			oMethodStart.u32Frequency = m_rSDSTunerRequest.u32Frequency;
			oMethodStart.u32DRMServiceId = (midw_fi_tcl_e8_Tun_HDAudioProgram::tenType)m_rSDSTunerRequest.u8HDProgramNumber;
			m_potunmstr_TunerClientHandler->vSendMessage( CCA_C_U16_APP_TUNER,// Destination APP ID
				oMethodStart,           // parameter&
				MIDW_TUNERFI_C_U16_FID_TUN_S_SELECT_DRM_AUDIOPRGM, //FID
				AMT_C_U8_CCAMSG_OPCODE_METHODSTART,  // OpCode                           /* amt_tclServiceData* */
				0 );

			vSendMethodResult((tU8)midw_fi_tcl_e8_Source::FI_EN_TUN_MSTR_SOURCE_NONE,(tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE);
		}
		
		else
		{
			vSendMethodResult(m_rSDSTunerRequest.u8Band, (tU8)midw_fi_tcl_e8_Action::FI_EN_TUN_MSTR_NONE, (tU8)midw_fi_tcl_e8_TunMstr_ResponseStatus::FI_EN_TUN_MSTR_RES_REQ_FAIL);
		}
	}

}


/*******************************************************************************
 *FUNCTION:    tunmstr_SDSHandler::bCheckReqChangeLSM
 *
 *DESCRIPTION: This function will check if requested source and active source are same
 *
 *PARAMETER:
 *
 *AUTHOR:
 *RETURNVALUE: bool
 ********************************************************************************/
tBool tunmstr_SDSHandler::bCheckReqChangeLSM()
{
	tBool bCheckReqChangeLSM = FALSE;
#ifdef VARIANT_S_FTR_ENABLE_SRCCHANGE_LIB
	if( m_poArlImpl == NULL )
	{
		NORMAL_M_ASSERT( m_poArlImpl != NULL );
		return FALSE;
	}
	tU8 u8ActiveTunerBand = (tU8)(clSessionControl::getInstance()->getActiveTunerBand());
	
    if((u8ActiveTunerBand == (tU8)tuner::common::BANDCB1) &&
           ((m_rSDSTunerRequest.u8Band == (tU8)tuner::common::BANDMW || (m_rSDSTunerRequest.u8Band == (tU8)tuner::common::BANDLW))))
	{
                m_rSDSTunerRequest.u8Band = (tU8)tuner::common::BANDCB1;
	}

	tU8 u8SrcActivity  = (tU8)m_poArlImpl->getActiveSourceActivity();
#else
	if( m_potunmstr_SrcChgManager == NULL )
	{
		NORMAL_M_ASSERT( m_potunmstr_SrcChgManager != NULL );
		return FALSE;
	}
	tU8 u8ActiveTunerBand = m_potunmstr_SrcChgManager->u8ConvertSrcToBand(m_potunmstr_SrcChgManager->u8GetCurrentSource());
	tU8 u8SrcActivity = m_potunmstr_SrcChgManager->u8GetActionFromenSourcendAction();
#endif

	ETG_TRACE_USR4((" bCheckReqChangeLSM(), ActiveBand : %d", u8ActiveTunerBand));
	ETG_TRACE_USR4((" bCheckReqChangeLSM(), RequestedBand : %d", (tU8)m_rSDSTunerRequest.u8Band));

#ifdef VARIANT_S_FTR_ENABLE_SRCCHANGE_LIB
	if((u8ActiveTunerBand != m_rSDSTunerRequest.u8Band) || (u8SrcActivity == (tU8)tuner::common::SRCOFF) || (u8SrcActivity == (tU8)tuner::common::SRCACTIVITY_UNKNOWN))
#else
	if((u8ActiveTunerBand != m_rSDSTunerRequest.u8Band) || (u8SrcActivity == (tU8)TUNMSTR_SRCCHG_ACTION_OFF) || (u8SrcActivity == (tU8)TUNMSTR_SRCCHG_INVALID))
#endif
	{
		bCheckReqChangeLSM = TRUE;
	}
	else
	{
		bCheckReqChangeLSM = FALSE;
	}

	if((m_rSDSTunerRequest.u8Band == (tU8)tuner::common::BANDSXM) &&
			((u8SrcActivity == (tU8)tuner::common::SRCON) || (u8SrcActivity == (tU8)tuner::common::SRCPAUSE)))
	{
		bCheckReqChangeLSM = TRUE;
	}
	return bCheckReqChangeLSM;
}


/*******************************************************************************
 *FUNCTION:    tunmstr_SDSHandler::vConvertSessionBand
 *
 *DESCRIPTION: This function will convert session band to TUNMSTR band value
 *
 *PARAMETER:
 *
 *AUTHOR:
 *RETURNVALUE: None
 ********************************************************************************/
tU8 tunmstr_SDSHandler::vConvertSessionBand(tU8 u8Band)
{
	tU8 u8RetBand = 0xFF;

	switch(u8Band)
	{
		case (tU8)midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_FM:
		{
				u8RetBand = (tU8)tuner::common::BANDFM;
				break;
		}
		case (tU8)midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_MW:
		{
				u8RetBand = (tU8)tuner::common::BANDMW;
				break;
		}
		case (tU8)midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_LW:
		{
				u8RetBand = (tU8)tuner::common::BANDLW;
				break;
		}
		case (tU8)midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_DAB:
		{
				u8RetBand = (tU8)tuner::common::BANDDAB;
				break;
		}
		case (tU8)midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_SXM:
		{
				u8RetBand = (tU8)tuner::common::BANDSXM;
				break;
		}
		case (tU8)midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_CB1:
		{
				u8RetBand = (tU8)tuner::common::BANDCB1;
				break;
		}
		case (tU8)midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_NONE:
		{
				u8RetBand = (tU8)tuner::common::BANDNONE;
				break;
		}
		default : break;
	}

	return u8RetBand;
}

/*******************************************************************************
 *FUNCTION:    tunmstr_SDSHandler::vClearSDSRequestParams
 *
 *DESCRIPTION: This function will clear the previous SDS requests
 *
 *PARAMETER:
 *
 *AUTHOR:
 *RETURNVALUE: None
 ********************************************************************************/
tVoid tunmstr_SDSHandler::vClearSDSRequestParams()
{
	m_rSDSTunerRequest.u32PI = 0;
	m_rSDSTunerRequest.u32Frequency = 0;
	m_rSDSTunerRequest.u16SecondaryPI = 0;
	m_rSDSTunerRequest.u8Band = 0xFF;
	m_rSDSTunerRequest.u8ElemID = 0xFF;
	m_rSDSTunerRequest.u8HDProgramNumber = 0;
	m_rSDSTunerRequest.u8ListID = 0xFF;
	m_rSDSTunerRequest.u8Option = 0;
}

/*******************************************************************************
 *FUNCTION:    tunmstr_SDSHandler::vSendMethodResult
 *
 *DESCRIPTION: This function handle method result
 *
 *PARAMETER: tU8 u8Source , tU8 u8Action , tU8 u8ResultStatus
 *
 *AUTHOR:
 *RETURNVALUE: None
 ********************************************************************************/
tVoid tunmstr_SDSHandler::vSendMethodResult(tU8 u8Source , tU8 u8Action , tU8 u8ResultStatus)
{
	if( m_potunmstr_MsgFromHMI == NULL )
	{
		NORMAL_M_ASSERT( m_potunmstr_MsgFromHMI != NULL );
		return;
	}

	midw_tunermasterfi_tclMsgFID_TUNMSTR_S_STATION_SELECTIONMethodResult oResponce;

	oResponce.Action.enType = (midw_fi_tcl_e8_Action::tenType)u8Action;
	if (u8Source == (tU8)tuner::common::BANDCB1)
	{
		u8Source = m_u8BandReqFromHMI;
		ETG_TRACE_USR4((" vSendMethodResult ,u8Source = %d ", u8Source));
	}
	oResponce.Source.enType = (midw_fi_tcl_e8_Source::tenType)u8Source;

	oResponce.TunerMaster_StatusResponse.u8TunermasterStatus.enType = \
		(midw_fi_tcl_e8_TunMstr_ResponseStatus::tenType)u8ResultStatus;
	stSendCCAMsg ostSendCCAMsg;
	ostSendCCAMsg.u16DestAppID=m_u16SourceAppID;
	ostSendCCAMsg.u16Fid=MIDW_TUNERMASTERFI_C_U16_FID_TUNMSTR_S_STATION_SELECTION;
	ostSendCCAMsg.u8OpCode=AMT_C_U8_CCAMSG_OPCODE_METHODRESULT;
	ostSendCCAMsg.u16CmdCtr=m_u16CmdCntr;
	ostSendCCAMsg.u16RegID=m_u16RegID;
	m_potunmstr_MsgFromHMI->vSendMessage( ostSendCCAMsg,oResponce);
}

#endif

