/************************************************************************
 * FILE:         tunermaster_SPMClient.cpp
 * PROJECT:      G3g
 * SW-COMPONENT: TunerMaster server
 *----------------------------------------------------------------------
 *
 * DESCRIPTION: This file handles all the CCA messages where audio acts
 *              as a client to SPM. Currently, it acts as a client to SPM
 *              for copro state.
 *
*----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------

 * HISTORY:
 * Date     	 | Author                     | Modification
 * 13.09.10 	 | RBEI/ECV4 Pallavi HN       | initial version
* 25.04.13  	 | NGP1KOR   				  | First version of the G3g after porting from NISSAN LCN2Kai
				
 *************************************************************************/

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------

#ifdef VARIANT_S_FTR_ENABLE_GENERICMSGS_MOCK
//AMT AHL included below
#else
// Include generic messages
#ifndef GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" 
#endif
#endif

#ifdef VARIANT_S_FTR_ENABLE_AMT_MOCK
#include "amt_mock/amt_if.h"
#else
// use AIL template with MessageMaps
#ifndef AIL_S_IMPORT_INTERFACE_GENERIC
#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"
#endif
#endif

#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
	#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
	#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_TYPES
	#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
	#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_SERVICEINFO
	#include "spm_fi_if.h"
#else
	#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_STDVISITORS
	#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_FUNCTIONIDS
	#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_SERVICEINFO
	#include "cfc_fi_if.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_AHL_MOCK
#include "ahl_mock/ahl_if.h"
#else
// use Application Help Library
#ifndef AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"
#endif
#endif

// Use OSAL defines
#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#endif

// Access to raw (unorganised) trace
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#ifndef TUNERMASTER_DEFINES_H
#include "tunermaster_defines.h"
#endif
#ifndef TUNMSTR_SRCCHG_SRCCHG_DEFINES_H
#include "tunmstr_SrcChg_Defines.h"
#endif

//------------------------
// Local headers 
//------------------------
#include "fc_tunermaster_featureControl.h"
// Self header
#include "tunermaster_SPM_Client.h"

// Include parent
#include "tunermaster_main.h"

// trace defines.
#include "tunermaster_trace.h"

#include "tunermaster_defines.h"
#include "tunermaster_PrjConfig.h"
#include "tunermaster_Config.h"
#include "tunmstr_SrcChgManager.h"
#ifndef VARIANT_S_FTR_ENABLE_FEATURE_GMMY16
#include "tunmstr_TunerClientHandler.h"
#endif
#include "tunmstr_antdiag_Manager.h"
#include "tunmstr_DiagLogClientHandler.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUNMSTR_TRACE_CLASS_SPM_CLIENT
#include "trcGenProj/Header/tunermaster_SPM_Client.cpp.trc.h"
#endif
#include "tunmstr_MsgToHMI.h"
/* ---
MESSAGE MAP:
enter the function IDs (FID) and the corresponding functions here.
the function will be called when a message with the corresponding FID arrives
--- */
BEGIN_MSG_MAP(tunmaster_tclSPMClient, ahl_tclBaseWork)
//ON_MESSAGE(CFC_SPMFI_C_U16_SYSTEMSTATE,  vHandleSysState)
#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
	ON_MESSAGE( SPM_COREFI_C_U16_CVMEVENT,     vHandleCVM)
#else
	ON_MESSAGE( CFC_SPMFI_C_U16_CVMEVENT,     vHandleCVM)
#endif
END_MSG_MAP()

/*************************************************************************
 *
 * FUNCTION:     tunmaster_tclSPMClient::tunmaster_tclSPMClient
 *
 * DESCRIPTION:  constructor, creates object tunmaster_tclSPMClient
 *
 * PARAMETER:    poGWMainApp: main - object of this application
 *               poMainSrv  : pointer to virtual block
 *
 * RETURNVALUE:  None
 *
 *12.04.06 aca2kor
 *         Initial version
 *
 *************************************************************************/
tunmaster_tclSPMClient::tunmaster_tclSPMClient(tunmaster_tclApp* poMainApp): 
m_u16SrvRegID(AMT_C_U16_REGID_INVALID),
m_fFidReg(FALSE),
m_fFidSysStateReg(FALSE),
m_fFidCVMReg(FALSE)
{
	// Store the reference to main (parent) object
	poGWMain      = poMainApp;

	m_ptunmaster_Config = NULL;
	m_potunmstr_SrcChgManager =NULL;
	m_potunmstr_TunClnt_Handlr = NULL;
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr_MsgToHMI=NULL;
	m_potunmstr__DiagLog=NULL;
	m_u32CVMEvent=0xFF;

	// Variable created for future, under voltage handling.
	m_u32VoltageEvent = 0;
    m_u8CriticalStatus =  (TUNMSTR_CRITICAL_STATUS_COPRO_BIT |TUNMSTR_CRITICAL_STATUS_COPRO_STANDBY_BIT);
	// Variable created for future, under voltage handling.
	m_fIsHighVoltageOnStartup = TRUE;
	m_u32SPMAppstate=AMT_C_U32_STATE_INVALID;

}

/*************************************************************************
 *
 * FUNCTION:     tunmaster_tclSPMClient::~tunmaster_tclSPMClient
 *
 * DESCRIPTION:  destructor, destroys the object
 *
 * PARAMETER:    None
 *
 * RETURNVALUE:  None
 *
 *14.04.06 aca2kor
 *         Initial version
 *
 *************************************************************************/
tunmaster_tclSPMClient::~tunmaster_tclSPMClient()
{
	poGWMain     = NULL;
	m_ptunmaster_Config = NULL;
	m_potunmstr_SrcChgManager =NULL;
	m_potunmstr_TunClnt_Handlr =NULL;
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr_MsgToHMI=NULL;
	m_potunmstr__DiagLog=NULL;
	m_u32CVMEvent=0XFF;
	m_u32SPMAppstate=AMT_C_U32_STATE_INVALID;
}


/*************************************************************************
 *
 * FUNCTION:     tunmaster_tclSPMClient::vSet_tunmaster_Config_Ptr( )
 *
 * DESCRIPTION:   With this config data can be accessed
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *
 *************************************************************************/
tVoid tunmaster_tclSPMClient::vSet_tunmaster_Config_Ptr( tunmaster_Config* ptunmaster_Config )
{
	if( ptunmaster_Config != NULL )
	{
		m_ptunmaster_Config = ( tunmaster_Config* )ptunmaster_Config;
	}
}
/*************************************************************************
 *
 * FUNCTION:     tunmaster_tclSPMClient::tunmaster_tclSPMClient::vSet_tunmaster_SrcChgManager_Ptr
 *
 * DESCRIPTION:  Set pointer to pointer tunmstr_SrcChgManager
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *
 *************************************************************************/
tVoid tunmaster_tclSPMClient::vSet_tunmaster_SrcChgManager_Ptr( tunmstr_SrcChgManager* ptunmaster_SrcChgMan )
{
	if( ptunmaster_SrcChgMan != NULL )
	{
		m_potunmstr_SrcChgManager = ptunmaster_SrcChgMan;
	}
}

/*************************************************************************
 *
 * FUNCTION:     tunmaster_tclSPMClient::tunmaster_tclSPMClient::vSet_tunmaster_ClientHandler_Ptr
 *
 * DESCRIPTION:  Set pointer to pointer tunmstr_TunerClientHandler
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *
 *************************************************************************/
tVoid tunmaster_tclSPMClient::vSet_tunmaster_ClientHandler_Ptr( tunmstr_TunerClientHandler* ptunmaster_ClientHandler )
{
	if( ptunmaster_ClientHandler != NULL )
	{
		m_potunmstr_TunClnt_Handlr = ptunmaster_ClientHandler;
	}
}
/*************************************************************************
 * FUNCTION:    tunmaster_tclSPMClient::vSet_tunmstr_antdiag_Manager(tunmstr_antdiag_Manager* potunmstr_antdiag_Manager)
 *
 * DESCRIPTION: Set pointer to pointer tunmstr_antdiag_Manager
 *
 * PARAMETER:   tunmstr_antdiag_Manager* potunmstr_antdiag_Manager
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmaster_tclSPMClient::vSet_tunmstr_antdiag_Manager(tunmstr_antdiag_Manager* potunmstr_antdiag_Manager)
{
	if( potunmstr_antdiag_Manager != NULL )
	{
		m_potunmstr_antdiag_Manager = ( tunmstr_antdiag_Manager* )potunmstr_antdiag_Manager;
	}
}
/*************************************************************************
* FUNCTION:    vSet_tunmstr_MsgToHMI_Ptr
*
* DESCRIPTION: update tunmstr_MsgToHMI pointer variable
*
* PARAMETER:   tunmstr_MsgToHMI* potunmstr_MsgToHMI
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmaster_tclSPMClient::vSet_tunmstr_MsgToHMI_Ptr(tunmstr_MsgToHMI* potunmstr_MsgToHMI)
{
	if(potunmstr_MsgToHMI!=NULL)
	{
		m_potunmstr_MsgToHMI =(tunmstr_MsgToHMI*) potunmstr_MsgToHMI;
	}
}
/*************************************************************************
* FUNCTION:    vSet_tunmstr_antdiag_Measure_Ptr( tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler )
*
* DESCRIPTION: update tunmstr_DiagLogClientHandler pointer variable
*
* PARAMETER:   tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmaster_tclSPMClient::vSet_tunmstr_DiagLogClientHandler_Ptr(tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler)
{
	if( potunmstr_DiagLogClientHandler != NULL )
	{
		m_potunmstr__DiagLog = ( tunmstr_DiagLogClientHandler* )potunmstr_DiagLogClientHandler;
	}
}
/*************************************************************************
 *
 * FUNCTION:     tunmaster_tclSPMClient::vOnUnknownMessage
 *
 * DESCRIPTION:  Frame work calls this function when it receives a message with
 *              an unknown FID. Used for error handling.
 *
 * PARAMETER:    poMessage : Pointer to CCA data
 *
 * RETURNVALUE:  None
 *
 * HISTORY:
 * 12.04.06 aca2kor
 *          Initial version
 *
 *************************************************************************/
tVoid tunmaster_tclSPMClient::vOnUnknownMessage(amt_tclBaseMessage* /*poMessage*/)
{
	// Unknown message: send an error message
	/*
  vSendError (poMessage->u16GetSourceAppID(),
              poMessage->u16GetTargetAppID(),
              poMessage->u16GetRegisterID(),
              poMessage->u16GetCmdCounter(),
              poMessage->u16GetServiceID(),
              poMessage->u16GetFunctionID(),
              (tU8) AUDIO_UNKNOWN_MSG);
	 */
	// TODO : Trace this condition
}


/*************************************************************************
 *
 * FUNCTION:     tunmaster_tclSPMClient::vOnNewAppState
 *
 * DESCRIPTION:  Main calls this function whenever there is a change in
 *              power state to handle CCA service
 *
 * PARAMETER:    u32OldAppState : Last set application state
 *               u32AppState    : Current requested app state
 *
 * RETURNVALUE:  None
 *
 * HISTORY:
 * 14.09.06 dip2kor
 *          Initial version
 *
 *************************************************************************/
tVoid tunmaster_tclSPMClient::vOnNewAppState ( tU32 u32OldAppState, 
		tU32 u32AppState )
{

	if(poGWMain == NULL)
	{
		ETG_TRACE_USR4((" tunmaster_tclSPMClient::vOnNewAppState, Null pointer"));
				NORMAL_M_ASSERT (NULL != poGWMain);
				return;
	}
	m_u32SPMAppstate=u32AppState;
	// Decide to register/unregister based on the app state
	switch(u32AppState)
	{
	case AMT_C_U32_STATE_NORMAL:
	{
		// Server has already entered this state, hence ignore
		if(u32OldAppState == AMT_C_U32_STATE_NORMAL)
		{
			break; // Switching to the same state
		}

		// Register for service
		fRegisterForService();
	}
	break;

	case AMT_C_U32_STATE_DIAGNOSIS:
	{
		// Server has already entered this state, hence ignore
		if(u32OldAppState == AMT_C_U32_STATE_DIAGNOSIS)
		{
			break; // Switching to the same state
		}

		// The application can enter this state from every other state
		// (e.g. NORMAL or OFF) so register for service should have the same
		// behavior everytime
		fRegisterForService();
	}
	break;

	case AMT_C_U32_STATE_PAUSE:
	{
		// Don't do anything in this state, audio don't do anything in this
		// state

		// Indicate the component has not reached stability
		vSetCompStabilityState (FALSE);
	}
	break;

	case AMT_C_U32_STATE_OFF:
	{
		// Server has already entered this state, hence ignore
		if(u32OldAppState == AMT_C_U32_STATE_OFF)
		{
			break; // Switching to the same state
		}

		// Indicate the component has not reached stability
		vSetCompStabilityState (FALSE);

		// Unregister for service
		fUnregisterForService();
	}
	break;

	default:
	{
		// Ideally should  not come here
		ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, ET_EN_T16 _ \
				0xFFFF _ ET_EN_DONE);
		break;
	}
	} // switch(u32AppState)

	ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, ET_EN_T16 _ \
			TUNMSTR_TRACE_SYSTEM_STATE_CHANGE _\
			ET_EN_T32 _ u32AppState _\
			ET_EN_DONE);
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tunmaster_tclSPMClient::vHandleCVM
 *
 *DESCRIPTION:  React on messages with FID = CFC_SPMFI_C_U16_COPROSTATE
 *             
 *PARAMETER:    poMessage : Pointer to CCA data
 *
 *RETURNVALUE:  None
 *
 *HISTORY:
 *10.05.06 mor2hi
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/

tVoid tunmaster_tclSPMClient::vHandleCVM
(
		amt_tclServiceData* poMessage
)
{
	if((NULL==poMessage)||(NULL==m_potunmstr_SrcChgManager)\
		||(NULL==m_potunmstr_antdiag_Manager)||(NULL==m_potunmstr_MsgToHMI)||(NULL==m_potunmstr__DiagLog) || (NULL == m_potunmstr_TunClnt_Handlr))
	{
		return;
	}
	switch (poMessage->u8GetOpCode())
	{
		case AMT_C_U8_CCAMSG_OPCODE_STATUS:
		{
			spm_corefi_tclMsgCvmEventStatus oMsg;
			m_potunmstr_TunClnt_Handlr->vGetDatafromAMT(poMessage,oMsg);
			m_u32CVMEvent = (tU32)oMsg.CvmEvent.enType;
			ETG_TRACE_USR4(("vHandleCVM()  m_u32CVMEvent=%d",m_u32CVMEvent));
			if((tU32)CVM_CRITICAL_LOW_VOLTAGE_START==m_u32CVMEvent)
			{
				ETG_TRACE_USR4(("vHandleCVM()  m_u32CVMEvent=%d",m_u32CVMEvent));
#ifndef VARIANT_S_FTR_ENABLE_FEATURE_GMMY16
				m_potunmstr_SrcChgManager->vClearRadioStates();
				m_potunmstr_SrcChgManager->vClearDABStates();
#endif
				m_potunmstr_antdiag_Manager->vPostEvent(TUNMSTR_ANTDIAG_U32_EVENT_MASK_SPM_LOWVG_START);
			}
			else
			{
			   m_potunmstr_antdiag_Manager->vPostEvent(TUNMSTR_ANTDIAG_U32_EVENT_MASK_SPM_LOWVG_END);
			}
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_RNAIVI
			if((tU32)CVM_HIGH_VOLTAGE_START==m_u32CVMEvent)
#else
			if((tU32)CVM_CRITICAL_HIGH_VOLTAGE_START==m_u32CVMEvent)
#endif
			{
			   m_potunmstr_antdiag_Manager->vPostEvent(TUNMSTR_ANTDIAG_U32_EVENT_MASK_SPM_HIGHVG_START);
			}
			else
			{
			   m_potunmstr_antdiag_Manager->vPostEvent(TUNMSTR_ANTDIAG_U32_EVENT_MASK_SPM_HIGHVG_END);
			}
			break;
		}
		default:
		{
			break;
		}
	}

}
/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tunmaster_tclSPMClient::vHandleSysState
 *
 *DESCRIPTION:  React on messages with FID = FID_SPM_SYSTEM_STATE
 *             
 *PARAMETER:    poMessage : Pointer to CCA data
 *
 *RETURNVALUE:  None
 *
 *HISTORY:
 *10.05.06 mor2hi
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/

tVoid tunmaster_tclSPMClient::vHandleSysState
(
		amt_tclServiceData* poMessage
)const
{
	if (NULL == poMessage)
	{
		// CCA framework problem - Notify through trace
		NORMAL_M_ASSERT(NULL != poMessage);
		return;
	}


	switch (poMessage->u8GetOpCode())
	{
	case AMT_C_U8_CCAMSG_OPCODE_STATUS:
	{
		spm_corefi_tclMsgSystemStateStatus oMsg;
		if(m_potunmstr_TunClnt_Handlr !=NULL)
			m_potunmstr_TunClnt_Handlr->vGetDatafromAMT(poMessage,oMsg);
		tU32 u32SysState = (tU32) oMsg.SystemState.enType;
#ifndef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
		ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, \
				ET_EN_T16 _ TUNMSTR_TRACE_SPM_SYSTEM_STATE _ \
				ET_EN_T32 _ u32SysState _ ET_EN_DONE);
#endif
		// Update the voltage and critical state
		/* Update of landscape memory is not required for the NAR target as there is no concept of FM station list.
          Hence it is commented out using the dynamic switch which checks the PRODUCT ID of the device being used and
          if it is a NAR device then this part of the code is not executed */
#ifndef VARIANT_S_FTR_ENABLE_FEATURE_GMMY16
		if(( NULL != m_ptunmaster_Config)  && (m_potunmstr_SrcChgManager !=NULL)&& (m_potunmstr_TunClnt_Handlr !=NULL))
		{
			tBool bLandscapeSeek = m_ptunmaster_Config->fGetLandscapeFlag();
			ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, \
					ET_EN_T16 _ TUNMSTR_TRACE_KDS_LANDSCAPE_ENABLE _ \
					ET_EN_T8 _ bLandscapeSeek _ ET_EN_DONE);


			if ( bLandscapeSeek == TRUE )
			{
				if( (u32SysState == SPM_SYSTEM_STATE_STANDBY))
				{
					tBool bTunestate = m_potunmstr_SrcChgManager->u8GetTunerState();
					ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, \
							ET_EN_T16 _ TUNMSTR_TRACE_TUNERSTATE _ \
							ET_EN_T8 _ bTunestate _ ET_EN_DONE);
					if (bTunestate == TUNMSTR_FOREGROUND)
					{
						// Send the FG tuner to BG to update the Landscape Memory!
						m_potunmstr_TunClnt_Handlr->vSend_s_Audio_FG_BG_New(
								(tU8)TUNMSTR_FOREGROUND,(tU8)TUNMSTR_SRCCHG_CHG_TO_BG,(tU8)TUNMSTR_SRCCHG_USR1_CHG_FGBG, (tU8)TUNMSTR_FOREGROUND, 0xff);
						m_potunmstr_SrcChgManager->vSetTunerState(TUNMSTR_BACKGROUND);
						ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, ET_EN_T16 _ \
								TUNMSTR_TRACE_LANDSCAPE_FGBG _\
								ET_EN_DONE);
					}
				}
			}
		}
#endif
	}

	break;

	default:
	{
		// Ignore any other message from SPM
	}
	break;

	} // switch (poMessage->u8GetOpCode())

}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tunmaster_tclSPMClient::fRegisterForService
 *
 *DESCRIPTION:  Sends a register message to SPM.
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  TRUE  : No Error in registeration
 *              FALSE : Error in registeration
 *
 *HISTORY:
 *12.04.06 dip2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tBool tunmaster_tclSPMClient::fRegisterForService()
{
	// Return value
	tBool fIsServiceRegistered = FALSE;

	// Is registration invalid ?
	if ( AMT_C_U16_REGID_INVALID == m_u16SrvRegID )
	{
		// Signal that service is registered
		m_fFidReg = FALSE;

		m_fFidSysStateReg = FALSE;

		m_fFidCVMReg = FALSE;

		// Try to register
		// Try to register

#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
		m_u16SrvRegID = poGWMain->bRegisterAsync(CCA_C_U16_SRV_SPM,
				SPM_COREFI_C_U16_SERVICE_MAJORVERSION,                      /* Major version */
				SPM_COREFI_C_U16_SERVICE_MINORVERSION,                      /* Minor version */
				0x0
				);

		fIsServiceRegistered = ( m_u16SrvRegID != AMT_C_U16_REGID_INVALID );
#else
#ifndef VARIANT_S_FTR_ENABLE_FEAT_VDFMAM_AUD_CCACOMM
		m_u16SrvRegID = poGWMain->u16RegisterService(CCA_C_U16_SRV_SPM,
				CFC_SPMFI_C_U16_SERVICE_MAJORVERSION,                      /* Major version */
				CFC_SPMFI_C_U16_SERVICE_MINORVERSION,                      /* Minor version */
				0x0,
				CCA_C_U16_APP_SPM);

		fIsServiceRegistered = ( m_u16SrvRegID != AMT_C_U16_REGID_INVALID );
#endif
#endif
	}


	// Determine whether the service is registered / registeration was successful
	// fIsServiceRegistered = ( AMT_C_U16_REGID_INVALID != m_u16SrvRegID);

	if (FALSE == fIsServiceRegistered)
	{
		ET_TRACE_ERROR_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT,  \
				ET_EN_T16 _ TUNMSTR_TRACE_REGISTER_FOR_SERVICE_FAIL _\
				ET_EN_DONE );

	}
	else
	{
		ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, \
				ET_EN_T16 _ TUNMSTR_TRACE_REGISTER_FOR_SERVICE_SUCC _ \
				ET_EN_DONE);

	}

	// return whether the service is registered
	return fIsServiceRegistered;
}


/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tunmaster_tclSPMClient::fUnregisterForService
 *
 *DESCRIPTION:  Sends a unregister message to SPM
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  TRUE  : No Error in unregisteration
 *              FALSE : Error in unregisteration
 *
 *HISTORY:
 *12.04.06 dip2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tBool tunmaster_tclSPMClient::fUnregisterForService()
{
	// Return value
	tBool fIsServiceUnregistered = TRUE;

	// Unregister if registration is valid?
	if ( AMT_C_U16_REGID_INVALID != m_u16SrvRegID )
	{

		// Send an unregister for service message
		poGWMain->vUnregisterService( (tU16) CCA_C_U16_SRV_SPM );

		// Make register-ID invalid:
		m_u16SrvRegID = AMT_C_U16_REGID_INVALID;

		ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, \
				ET_EN_T16 _ TUNMSTR_TRACE_UNREGISTER_FOR_SERVICE _ \
				ET_EN_T16 _ m_u16SrvRegID _ \
				ET_EN_DONE);
	}


	// return whether the service is unregistered
	return fIsServiceUnregistered;
}


/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tunmaster_tclSPMClient::fRegisterForFID
 *
 *DESCRIPTION:  Registers for a function of a service
 *              !!! before this, you must register for the service !!!
 *             
 *PARAMETER:    u16FID : FID of the function to be registered (upreg)
 *
 *RETURNVALUE:  TRUE  : No Error in FID registeration
 *              FALSE : Error in FID registeration
 *
 *HISTORY:
 *12.04.06 dip2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tBool tunmaster_tclSPMClient::fRegisterForFID(tU16 u16FID)
{
	// Return value
	tBool fIsServiceRegistered = FALSE;

	ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, \
			ET_EN_T16 _ TUNMSTR_TRACE_REGISTER_FOR_FID _ \
			ET_EN_DONE);


	// Is registration invalid?
	if ( AMT_C_U16_REGID_INVALID == m_u16SrvRegID )
	{
		// Ideally should  not come here
		ET_TRACE_ERROR_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, ET_EN_T16 _ \
				TUNMSTR_TRACE_REGISTER_FOR_FID_INVALID_SRVID _  ET_EN_T8 _ \
				0xFF _ ET_EN_DONE);
	}

	// Create message with upreg - request:
	gm_tclEmptyMessage oUpRegMessage
	(
			CCA_C_U16_APP_TUNERMASTER,          // AppID of this application
			CCA_C_U16_APP_SPM,            // AppID of the other Server
			m_u16SrvRegID,                    // RegId for the service
			0,                            // always 0
			CCA_C_U16_SRV_SPM,            // the SID of the service
			u16FID,                       // the FID to register for
			AMT_C_U8_CCAMSG_OPCODE_UPREG  // this is an Upreg-message!
	);

	if( AIL_EN_N_NO_ERROR != poGWMain->enPostMessage(&oUpRegMessage) )
	{
		// Error in sending the message
		if(!oUpRegMessage.bDelete())
		{
			// Error in deleting the message
			// TODO : Send an error trace here
		}
	}
	else
	{
		fIsServiceRegistered = TRUE;
	}

	// return whether the service is registered
	return fIsServiceRegistered;
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tunmaster_tclSPMClient::fUnregisterForFID
 *
 *DESCRIPTION:  Unregisters for a function of a service
 *             
 *PARAMETER:    u16FID : FID of the function to be registered (upreg)
 *
 *RETURNVALUE:  TRUE  : No Error in FID unregisteration
 *              FALSE : Error in FID unregisteration
 *
 *HISTORY:
 *14.09.06 dip2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tBool tunmaster_tclSPMClient::fUnregisterForFID(tU16 u16FID)
{  
	// Return value
	tBool fIsServiceUnregistered = FALSE;

	ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, \
			ET_EN_T16 _ TUNMSTR_TRACE_UNREGISTER_FOR_FID _ \
			ET_EN_DONE);

	// Is register id valid?
	if ( AMT_C_U16_REGID_INVALID != m_u16SrvRegID )
	{
		// Create message with relupreg - request:

		gm_tclEmptyMessage oRelupRegMessage
		(
				CCA_C_U16_APP_TUNERMASTER,             // AppID of this application
				CCA_C_U16_APP_SPM,               // AppID of the other Server
				m_u16SrvRegID,                       // RegId for the service
				0,                               // always 0
				CCA_C_U16_SRV_SPM,               // the SID of the service
				u16FID,                          // the FID to register for
				AMT_C_U8_CCAMSG_OPCODE_RELUPREG  // this is an Upreg-message!
		);



		if( AIL_EN_N_NO_ERROR != poGWMain->enPostMessage(&oRelupRegMessage))
		{
			// Error in sending the message
			if(!oRelupRegMessage.bDelete())
			{
				// Error in deleting the message
				// TODO : Send an error trace here
			}
		}
		else
		{
			fIsServiceUnregistered = TRUE;
		}

	} // if ( AMT_C_U16_REGID_INVALID != m_u16SrvRegID )

	// return whether the service is unregistered
	return fIsServiceUnregistered;
}
/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tunmaster_tclSPMClient::vSetVoltageState
 *
 *DESCRIPTION:  Sets the voltage status sent by SPM
 *             
 *PARAMETER:    u32Event : Voltage event
 *
 *RETURNVALUE:  None
 *
 *HISTORY:
 *14.04.06 aca2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tVoid tunmaster_tclSPMClient::vSetVoltageState (tU32 u32Event){  
	m_u32CVMEvent = u32Event;

	// Trace this event value
	ET_TRACE_INFO_BIN(TUNMSTR_TRACE_CLASS_SPM_CLIENT, ET_EN_T16 _ \
			TUNMSTR_TRACE_LOW_VOLTAGE_EVENT _  ET_EN_T32 _ \
			u32Event _ ET_EN_DONE );

}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tunmaster_tclSPMClient::vSetCompStabilityState
 *
 *DESCRIPTION:  Sets the componet stability status after sending soft on
 *             
 *PARAMETER:    fStatus : TRUE  : Comp is stable
 *                        FALSE : Comp is unstable
 *
 *RETURNVALUE:  None
 *
 *HISTORY:
 *18.07.06 aca2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tVoid tunmaster_tclSPMClient::vSetCompStabilityState (tBool fStatus)const
{


	if ( TRUE == fStatus )
	{
		// To do..
	}


}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tunmaster_tclSPMClient::vOnServiceState
 *
 *DESCRIPTION:  Handles the state changes of the service we use
 *             
 *PARAMETER:    u16ServiceId  : Service id whose service state is sent
 *              u16ServerId   : Server id who sends this service state
 *              u16RegisterId : Register id for service whose service state
 *                              is sent
 *              u8ServiceState: Service state of the registered service
 *              u16SubId      : SubId whose service state is sent
 *
 *RETURNVALUE: None
 *
 *HISTORY:
 *12.04.06 aca2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tVoid tunmaster_tclSPMClient::vOnServiceState( tU16 u16ServiceId,tU16 u16ServerId,
											  tU16 u16RegisterId,tU8  u8ServiceState,tU16 u16SubId)
{
	// We are interested only in SPM service
	if( CCA_C_U16_SRV_SPM != u16ServiceId )
	{
		// Trace which information has come to us
		ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT,ET_EN_T16 _\
				TUNMSTR_TRACE_ON_SERVICE_STATE _\
				ET_EN_T16 _ u16ServiceId _ \
				ET_EN_T16 _ u16ServerId _ ET_EN_T16 _ u16RegisterId _ \
				ET_EN_T8 _ u8ServiceState _ ET_EN_T16 _ u16SubId _ \
				ET_EN_DONE );

		return;
	}



	// Get new service ID
	m_u16SrvRegID = u16RegisterId;

	if (m_u16SrvRegID == AMT_C_U16_REGID_INVALID)
	{
		ET_TRACE_ERROR_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT,  \
				ET_EN_T16 _ TUNMSTR_TRACE_REGISTER_FOR_SERVICE_FAIL _\
				ET_EN_T16 _ m_u16SrvRegID _ \
				ET_EN_DONE );

	}
	// Analyze the state of the server
	switch(u8ServiceState)
	{
	case AMT_C_U8_SVCSTATE_AVAILABLE:
		{
			ET_TRACE_INFO_BIN( TUNMSTR_TRACE_CLASS_SPM_CLIENT, ET_EN_T16 _ \
				TUNMSTR_TRACE_ON_SERVICE_STATE_AVAILABLE _ ET_EN_DONE );
			// Service is available, implement your reaction here
#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
			if( !m_fFidReg) // Register only once
			{
				m_fFidReg = fRegisterForFID( SPM_COREFI_C_U16_COPROSTATE );
			} // if( !m_fFidReg)
			// Service is available, implement your reaction here
			if( !m_fFidSysStateReg) // Register only once
			{
				m_fFidSysStateReg = fRegisterForFID( SPM_COREFI_C_U16_TUNERSTATE );
			} // if( !m_fFidSysStateReg)
			if ( !m_fFidCVMReg)
			{
				m_fFidCVMReg = fRegisterForFID( SPM_COREFI_C_U16_CVMEVENT );
			}
#else
			if( !m_fFidReg) // Register only once
			{
				m_fFidReg = fRegisterForFID( CFC_SPMFI_C_U16_COPROSTATE );
			} // if( !m_fFidReg)
			// Service is available, implement your reaction here
			if( !m_fFidSysStateReg) // Register only once
			{
				m_fFidSysStateReg = fRegisterForFID( CFC_SPMFI_C_U16_SYSTEMSTATE );
			} // if( !m_fFidSysStateReg)
			if ( !m_fFidCVMReg)
			{
				m_fFidCVMReg = fRegisterForFID( CFC_SPMFI_C_U16_CVMEVENT );
			}
#endif
			break;
		}
	case AMT_C_U8_SVCSTATE_REG_INVALID:
		{
			// Server has lost our registration.
			// Try to register again
			fRegisterForService();
			break;
		}
	default:
		{
			break;
		}
	} // switch(u8ServiceState)
}
/******************************************************************+FUNCHEADER**
 *FUNCTION:     tunmaster_tclSPMClient::u32GetCVMEvent
 *
 *DESCRIPTION:  Reads the CVM Event
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  m_u32CVMEvent
 ******************************************************************-FUNCHEADER*/
tU32 tunmaster_tclSPMClient::u32GetCVMEvent ()const
{
	return m_u32CVMEvent;
}
/*************************************************************************
 *
 * FUNCTION:     tunmaster_tclSPMClient::vHandleTTFiS_CMD( )
 *
 * DESCRIPTION:  Handler to handle TTFIs Commands
 *
 * PARAMETER:    tU8DataLength, const tU8*u8DataSpecial
 *
 * RETURNVALUE:  void
 *
 *************************************************************************/
tVoid tunmaster_tclSPMClient::vHandleTTFiS_CMD(const tU8* u8DataSpecial)
{
	tU8 u8FunctionID = u8DataSpecial[0];
	if ( NULL == poGWMain )
	{ 	   
		ETG_TRACE_USR4((" tunmaster_tclSPMClient::vHandleTTFiS_CMD, Null pointer"));
		NORMAL_M_ASSERT (NULL != poGWMain);
		return;
	}
	tU16 u16RegID = poGWMain->u16GetRegID();
	switch(u8FunctionID)
	{
#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
	case SPM_COREFI_C_U16_CVMEVENT:
	{   
		spm_corefi_tclMsgCvmEventStatus cvmstatus;
		cvmstatus.CvmEvent.enType =(spm_fi_tcl_SPM_e32_CVM_EVENT::tenType)u8DataSpecial[1];
		vSendMessage( CCA_C_U16_APP_TUNERMASTER,// Destination APP ID
				cvmstatus,           // parameter&
				SPM_COREFI_C_U16_CVMEVENT, //FID
				AMT_C_U8_CCAMSG_OPCODE_STATUS,  // OpCode                           /* amt_tclServiceData* */
				0,//CmdCounetr
				u16RegID //Registration ID
		);
		ETG_TRACE_USR4((" tunmaster_tclSPMClient::vHandleTTFiS_CMD, SPM_COREFI_C_U16_CVMEVENT sent"));
	}
#else
	case CFC_SPMFI_C_U16_CVMEVENT:
	{   
		cfc_spmfi_tclMsgCvmEventStatus cvmstatus;
		cvmstatus.CvmEvent.enType =(cfc_fi_tcl_SPM_e32_CVM_EVENT::tenType)u8DataSpecial[1];
		vSendMessage( CCA_C_U16_APP_TUNERMASTER,// Destination APP ID
				cvmstatus,           // parameter&
				CFC_SPMFI_C_U16_CVMEVENT, //FID
				AMT_C_U8_CCAMSG_OPCODE_STATUS,  // OpCode                           /* amt_tclServiceData* */
				0,//CmdCounetr
				u16RegID //Registration ID
		);
		ETG_TRACE_USR4((" tunmaster_tclSPMClient::vHandleTTFiS_CMD, CFC_SPMFI_C_U16_CVMEVENT sent"));
	}
#endif
	break;
	default:
	{
		break;
	}
	}
}
/*************************************************************************************
 *FUNCTION:    tunmaster_tclSPMClient::vSendMessage
 *
 *DESCRIPTION: This function converts Amt message to visitor class and
 *             get the data from visitor to the FI class. Refer FI Manual.Doc.
 *
 *PARAMETER:    tU16,fi_tclTypeBase& ,tU16,tU8,tU16,tU16
 *
 *RETURNVALUE: None
 ************************************************************************************/
tVoid tunmaster_tclSPMClient::vSendMessage( tU16             u16DestAppID,
		const fi_tclTypeBase&  oOutData,
		tU16             u16Fid,
		tU8              u8OpCode,
		tU16             u16CmdCtr,
		tU16             u16RegId)
{
	if((poGWMain==NULL) || (u16RegId == AMT_C_U16_REGID_INVALID))
	{
		NORMAL_M_ASSERT (NULL != poGWMain);
		return;
	}
	// Create the Visitor Message
	fi_tclVisitorMessage oOutVisitorMsg(oOutData,1);
	// Set the CCA message information
	oOutVisitorMsg.vInitServiceData(
			CCA_C_U16_APP_TUNERMASTER,                      /* Source app-ID    */
			u16DestAppID,                             /* Dest. app-ID     */
			AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,        /* stream type      */
			0,                                        /* stream counter   */
			u16RegId,                                 /* Registry ID      */
			u16CmdCtr,                                /* Command counter  */
			CCA_C_U16_SRV_SPM,               /* Service-ID       */
			u16Fid,                                   /* Function-ID      */
			u8OpCode,                                 /* OpCode           */
			0,                                        /* ACT              */
			0,                                        /* Source sub-ID    */
			0 );                                      /* Dest. sub-ID     */
	/* ---
   send it to the client 
   --- */
	if(poGWMain->enPostMessage( &oOutVisitorMsg, TRUE ) != AIL_EN_N_NO_ERROR)
	{
		// Message not sent successfully.
		ETG_TRACE_USR4(("tunmaster_tclSPMClient::vSendMessage->Message not sent successfully"));
	}
	else
	{
		// Message  sent successfully.
		ETG_TRACE_USR4(("tunmaster_tclSPMClient::vSendMessage->Message sent successfully"));
	}
}
/*************************************************************************
 *
 * FUNCTION:     tunmaster_tclSPMClient::u32GetSPMAppstate( )
 *
 * DESCRIPTION:  Return the current SPM APP state
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  m_u32SPMAppstate
 *
 *************************************************************************/
tU32 tunmaster_tclSPMClient::u32GetSPMAppstate() const
{
	return m_u32SPMAppstate;
}
