/************************************************************************
* FILE:           tunmstr_DiagLogClientHandler.cpp
* PROJECT:        G3g
* SW-COMPONENT:   FC_TunerMaster.
*----------------------------------------------------------------------
*
* DESCRIPTION:    Implementation of the Class  tunmstr_DiagLogClientHandler
*
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:
* Date            | Author                       		 | Modification
*25/05/2012		  | ECV4 ( RBEI ) Priya Prabhuswamy      | Initial version.
* 25.04.13  	  | NGP1KOR   							 | First version of the G3g after porting from NISSAN LCN2Kai

*************************************************************************/
#include "tunmstr_DiagLogClientHandler.h"
#ifndef tunmstr_antdiag_Manager_H
#include "tunmstr_antdiag_Manager.h"
#endif
#ifndef tunmstr_antdiag_Debounce_H
#include "tunmstr_antdiag_Debounce.h"
#endif
#ifndef tunmstr_antdiag_Measure_H
#include "tunmstr_antdiag_Measure.h"
#endif
#ifndef tunermaster_cfcVehicleClientHandler_H
#include "tunermaster_cfcVehicleClientHandler.h"
#endif
#ifndef tunmstr_MsgToHMI_H
#include "tunmstr_MsgToHMI.h"
#endif

#ifndef tunmaster_Config_H
#include "tunermaster_Config.h"
#endif

#ifndef tunmstr_antdiag_i2cHandler_H
#include "tunmstr_antdiag_i2cHandler.h"
#endif

#include "kdsconfig/clConfigInterface.h"
#include "tunermaster_SPM_Client.h"

#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
	#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
	#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_TYPES
	#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
	#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_SERVICEINFO
	#include "spm_fi_if.h"
#else
	#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_STDVISITORS
	#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_FUNCTIONIDS
	#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_SERVICEINFO
	#include "cfc_fi_if.h"
#endif

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUNMSTR_TRACE_CLASS_ANTDIAG 
#include "trcGenProj/Header/tunmstr_DiagLogClientHandler.cpp.trc.h"
#endif



//MESSAGE MAP:
BEGIN_MSG_MAP(tunmstr_DiagLogClientHandler, ahl_tclBaseWork)
	//Properties
	ON_MESSAGE( MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT,         vHandle_g_Send_Next_Test_Result)
	/*Added to resolve double deletion */
	ON_MESSAGE( MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT, vHandle_s_UnUsedMessages)
END_MSG_MAP()

/* Initialize Semaphore */
OSAL_tSemHandle tunmstr_DiagLogClientHandler::m_hNotTable_SemHandle  = OSAL_C_INVALID_HANDLE;
/*************************************************************************
* FUNCTION:     tunmstr_DiagLogClientHandler(tunmaster_tclApp* poMainApp )
*
* DESCRIPTION:  Constructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*************************************************************************/
tunmstr_DiagLogClientHandler::tunmstr_DiagLogClientHandler(tunmaster_tclApp* poMainApp )
{
	poGWMain = poMainApp;
	m_u16RegID=AMT_C_U16_REGID_INVALID;
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr_antdiag_Debounce=NULL;
	m_pocfcVehicleClientHand=NULL;
	m_potunmstr_MsgToHMI=NULL;
	m_potunmstr_antdiag_Measure=NULL;
	m_potunmstr_antdiag_i2cHandler=NULL;
	m_poclConfigInterface = NULL;
	m_potunermaster_SPM_Client= NULL;
	m_blPreviousIgnState=FALSE;
	m_u8IgnCounter=0;
	for(tU8 u8Index=(tU8)enFM1;u8Index<enClientNone;u8Index++)
	{
		m_u8AntennaStatus[u8Index]=(tU8)enAntennState_Unknown;
		for(tU8 u8AntStateIndex=(tU8)enOpenInd;u8AntStateIndex<=(tU8)enShortInd;u8AntStateIndex++)
		{
			m_blConsider_Vg[u8Index][u8AntStateIndex]=FALSE;
			m_blConsider_Ign[u8Index][u8AntStateIndex]=FALSE;
			f_32BatterVgState[u8Index][u8AntStateIndex]=0xff;
		}
	}
	u8IgnTime_InSec=0;
}
/*************************************************************************
* FUNCTION:    ~tunmstr_DiagLogClientHandler( )
*
* DESCRIPTION:  Destructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*************************************************************************/
tunmstr_DiagLogClientHandler::~tunmstr_DiagLogClientHandler( )
{
	poGWMain = NULL;
	m_u16RegID=AMT_C_U16_REGID_INVALID;
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr_antdiag_Debounce=NULL;
	m_pocfcVehicleClientHand=NULL;
	m_potunmstr_MsgToHMI=NULL;
	m_potunmstr_antdiag_Measure=NULL;
	m_potunmstr_antdiag_i2cHandler=NULL;
	m_poclConfigInterface = NULL;
	m_potunermaster_SPM_Client =NULL;
	for(tU8 u8Index=(tU8)enFM1;u8Index<(tU8)enClientNone;u8Index++)
	{
		m_u8AntennaStatus[u8Index]=(tU8)enAntennState_Unknown;
		for(tU8 u8AntStateIndex=(tU8)enOpenInd;u8AntStateIndex<=(tU8)enShortInd;u8AntStateIndex++)
		{
			m_blConsider_Vg[u8Index][u8AntStateIndex]=FALSE;
			m_blConsider_Ign[u8Index][u8AntStateIndex]=FALSE;
			f_32BatterVgState[u8Index][u8AntStateIndex]=0xff;
		}
	}
	u8IgnTime_InSec=0;
}
/*************************************************************************
* FUNCTION:    vSet_tunmstr_antdiag_Manager( tunmstr_antdiag_Manager* potunmstr_antdiag_Manager )
*
* DESCRIPTION: update tunmstr_antdiag_Manager pointer variable
*
* PARAMETER:   tunmstr_antdiag_Manager* potunmstr_antdiag_Manager
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vSet_tunmstr_antdiag_Manager(tunmstr_antdiag_Manager* potunmstr_antdiag_Manager)
{
	if( potunmstr_antdiag_Manager != NULL )
	{
		m_potunmstr_antdiag_Manager = ( tunmstr_antdiag_Manager* )potunmstr_antdiag_Manager;
	}
}
/*************************************************************************
* FUNCTION:    vSet_tunmstr_cfcVehicleClientHand_Ptr()
*
* DESCRIPTION: update tunermaster_cfcVehicleClientHandler pointer variable
*
* PARAMETER:   tunermaster_cfcVehicleClientHandler* pocfcVehicleClientHand
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vSet_tunmstr_cfcVehicleClientHand_Ptr(tunermaster_cfcVehicleClientHandler* pocfcVehicleClientHand)
{
	if( pocfcVehicleClientHand != NULL )
	{
		m_pocfcVehicleClientHand = ( tunermaster_cfcVehicleClientHandler* )pocfcVehicleClientHand;
	}
}
/*************************************************************************
* FUNCTION:    vSet_tunmstr_MsgToHMI_Ptr()
*
* DESCRIPTION: update tunmstr_MsgToHMI pointer variable
*
* PARAMETER:   tunmstr_MsgToHMI* potunmstr_MsgToHMI
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vSet_tunmstr_MsgToHMI_Ptr(tunmstr_MsgToHMI* potunmstr_MsgToHMI)
{
	if(potunmstr_MsgToHMI!=NULL)
	{
		m_potunmstr_MsgToHMI =(tunmstr_MsgToHMI*) potunmstr_MsgToHMI;
	}
}
/*************************************************************************
* FUNCTION:    vSet_tunmstr_antdiag_Debounce()
*
* DESCRIPTION: update tunmstr_antdiag_Debounce pointer variable
*
* PARAMETER:   tunmstr_antdiag_Debounce* potunmstr_antdiag_Debounce
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vSet_tunmstr_antdiag_Debounce(tunmstr_antdiag_Debounce* potunmstr_antdiag_Debounce)
{
	if(potunmstr_antdiag_Debounce!=NULL)
	{
		m_potunmstr_antdiag_Debounce =(tunmstr_antdiag_Debounce*) potunmstr_antdiag_Debounce;
	}
}
/*************************************************************************
* FUNCTION:    vSet_tunmstr_antdiag_Measure_Ptr( tunmstr_antdiag_Measure* potunmstr_antdiag_Measure )
*
* DESCRIPTION: update tunmstr_antdiag_Measure pointer variable
*
* PARAMETER:   tunmstr_antdiag_Measure* potunmstr_antdiag_Measure
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vSet_tunmstr_antdiag_Measure_Ptr(tunmstr_antdiag_Measure* potunmstr_antdiag_Measure)
{
	if( potunmstr_antdiag_Measure != NULL )
	{
		m_potunmstr_antdiag_Measure = ( tunmstr_antdiag_Measure* )potunmstr_antdiag_Measure;
	}
}
/*************************************************************************
* FUNCTION:    vSet_tunmstr_antdiag_i2cHandler
*
* DESCRIPTION: update m_potunmstr_antdiag_i2cHandler pointer variable
*
* PARAMETER:   tunmstr_antdiag_i2cHandler* Ptunmstr_antdiag_i2cHandler
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vSet_tunmstr_antdiag_i2cHandler
(tunmstr_antdiag_i2cHandler* Ptunmstr_antdiag_i2cHandler)
{
	m_potunmstr_antdiag_i2cHandler=Ptunmstr_antdiag_i2cHandler;
}

/*************************************************************************
* FUNCTION:    vSet_tunmstr_ConfigInterface_Ptr
*
* DESCRIPTION: 
*
* PARAMETER:   clConfigInterface* poclConfigInterface
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vSet_tunmstr_ConfigInterface_Ptr
(clConfigInterface* poclConfigInterface)
{
	if( poclConfigInterface != NULL )
	{
		m_poclConfigInterface = poclConfigInterface;
	}

}
/*
 *
 */
tVoid tunmstr_DiagLogClientHandler::vSet_tunermaster_SPM_Client_Ptr(tunmaster_tclSPMClient* potunermaster_SPM_Client)
{
	if( potunermaster_SPM_Client != NULL )
	{
		m_potunermaster_SPM_Client = potunermaster_SPM_Client;
	}

}


/*************************************************************************************
*FUNCTION:     vOnServiceState
*
*DESCRIPTION:  update tunmaster_Config pointer variable
*
*PARAMETER:    tU16,tU16,tU16,tU8,tU16,
*
*RETURNVALUE: None
************************************************************************************/
tVoid tunmstr_DiagLogClientHandler :: vOnServiceState(   tU16  u16ServiceId ,
													  tU16  u16ServerId ,
													  tU16  u16RegisterId,
													  tU8   u8ServiceState,
													  tU16  u16SubId      )
{
	if(u16ServiceId  !=  CCA_C_U16_SRV_DIAGLOG)
	{
		return;
	}
	ETG_TRACE_USR1(("vOnServiceState()->"
		"u16ServiceId=%d u16ServerId=%d u16RegisterId=%d u8ServiceState=%d u16SubId=%d",
		u16ServiceId,u16ServerId,u16RegisterId,u8ServiceState,u16SubId));
	m_u16RegID = u16RegisterId;
	switch(u8ServiceState)
	{
	case AMT_C_U8_SVCSTATE_AVAILABLE:
		{
			tU16 u16FID;
			u16FID = MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT;
			fregisterForFID(u16FID); // register for MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT
			vUpdateAntennaITC();
			break;
		}
	case AMT_C_U8_SVCSTATE_NOT_AVAILABLE:
		{
			// Service is not available anymore
			// Don't do anything..wait for availability
			break;
		}
	case AMT_C_U8_SVCSTATE_REG_INVALID:
		{
			// Server has lost our registration,Try to register again
			vRegisterForService();
			break;
		}
	default:
		{
			break;
		}
	} // switch(u8ServiceState)
}
/*************************************************************************************
*
*FUNCTION:     fregisterForFID
*
*DESCRIPTION:  Handles the state changes of the service we use
*
*PARAMETER:    tU16,tU16,tU16,tU8,tU16,
*
*RETURNVALUE: None
************************************************************************************/
tBool tunmstr_DiagLogClientHandler :: fregisterForFID(tU16 u16FID) 
{
	tBool bRetVal = FALSE;
	if ( NULL == poGWMain )
	{
		NORMAL_M_ASSERT (NULL != poGWMain);
		return FALSE;
	}
	gm_tclEmptyMessage oUpRegMessage( CCA_C_U16_APP_TUNERMASTER,  // AppID of this application
		CCA_C_U16_APP_DIAGLOG,      // AppID of the Server
		m_u16RegID,                // RegId for the service (see bRegisterForService )
		0,                        // always 0
		CCA_C_U16_SRV_DIAGLOG,  // SID of the service
		u16FID,                   // FID to register for
		AMT_C_U8_CCAMSG_OPCODE_UPREG);
	if (poGWMain->enPostMessage(&oUpRegMessage, TRUE) == AIL_EN_N_NO_ERROR)
	{
		// message send success - set the return value to TRUE
		bRetVal = TRUE;
		ETG_TRACE_USR1(("fregisterForFID()->REG Success"));
	}
	else
	{
		ETG_TRACE_ERR(("fregisterForFID()->REG NOK"));
	}
	return bRetVal;
}
/*************************************************************************************
*FUNCTION:     vRegisterForService
*
*DESCRIPTION:  Sends a register message to DIAGLOG
*
*PARAMETER:    None
*
*RETURNVALUE:  tVoid
************************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vRegisterForService()
{
	tBool fIsServiceRegistered = FALSE;
	if ( NULL == poGWMain )
	{
		NORMAL_M_ASSERT (NULL != poGWMain);
		return;
	}
	// check valid reqisteration
	if ( AMT_C_U16_REGID_INVALID == m_u16RegID )
	{
		fIsServiceRegistered = poGWMain->bRegisterAsync (CCA_C_U16_SRV_DIAGLOG,
			MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION                     /* Major version */,
			0                      /* Minor version */,
			0x0,
			CCA_C_U16_APP_DIAGLOG
			);
	}
	if(fIsServiceRegistered == TRUE)
	{
		ETG_TRACE_USR1(("vRegisterForService()->, REG for Service OK"));
	}
	else
	{
		ETG_TRACE_ERR(("vRegisterForService()->REG for Service NOK"));
	}
}
/*******************************************************************************
*FUNCTION:    vGetDataFromAmt
*
*DESCRIPTION: This function converts Amt message to visitor class and
*             get the data from visitor to the Fi class. Refer FI Manual.
*             Doc.
*PARAMETER:
*           poFiMsg   - Amt Message
*           oFIData   - Base class of Fi class
*
*RETURNVALUE: None
********************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vGetDataFromAmt(amt_tclServiceData* poFIMsg,
													fi_tclTypeBase&     oFIData )const
{
	// Create Visitor message object for incoming message
	fi_tclVisitorMessage oInVisitorMsg(poFIMsg);

	/* Get the data in FI class from Vistor class*/
	if(OSAL_ERROR == oInVisitorMsg.s32GetData(oFIData,1))
	{
		// assert here
		NORMAL_M_ASSERT_ALWAYS ();
	}
}
/****************************************************************************
*FUNCTION:    vSendMessage
*
*DESCRIPTION: This function converts Amt message to visitor class and
*             get the data from visitor to the Fi class. Refer FI Manual.Doc.
*
*PARAMETER:
*             u16DestAppID      - Destination ID of Out Msg
*             oOutData          - Base class of out going Fi object
*             u16Fid            - Funtion id of the out going message
*             u8OpCode          - Opcode of the message
*
*RETURNVALUE: None
******************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vSendMessage( tU16             u16DestAppID,
												 const fi_tclTypeBase&  oOutData,
												 tU16             u16Fid,
												 tU8              u8OpCode,
												 tU16             u16CmdCtr,
												 tU16             u16RegId)
{
	if(AMT_C_U16_REGID_INVALID==u16RegId)
	{
		return;
	}
	// Create the Visitor Message
	fi_tclVisitorMessage oOutVisitorMsg(oOutData,1);
	// Set the CCA message information
	oOutVisitorMsg.vInitServiceData(
		CCA_C_U16_APP_TUNERMASTER,                      /* Source app-ID    */
		u16DestAppID,                             /* Dest. app-ID     */
		AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,        /* stream type      */
		0,                                        /* stream counter   */
		u16RegId,                                 /* Registry ID      */
		u16CmdCtr,                                /* Command counter  */
		CCA_C_U16_SRV_DIAGLOG,               /* Service-ID       */
		u16Fid,                                   /* Function-ID      */
		u8OpCode,                                 /* OpCode           */
		0,                                        /* ACT              */
		0,                                        /* Source sub-ID    */
		0 );                                      /* Dest. sub-ID     */
	/* ---
	send it to the client 
	--- */
	if(poGWMain->enPostMessage( &oOutVisitorMsg, TRUE ) != AIL_EN_N_NO_ERROR)
	{
		// Message sent successfully.
		ETG_TRACE_ERR(("vSendMessage()->Message not sent successfully"));
	}
	else
	{
		// Message not sent successfully.
		ETG_TRACE_USR4(("vSendMessage()->Message sent successfully"));
	}
}
/*************************************************************************************
*FUNCTION:     vNewAppState
*
*DESCRIPTION:  Main calls this function whenever there is a change in
*               power state to handle CCA service
*
*PARAMETER:    tU16,tU16,tU16,tU8,tU16,
*
*RETURNVALUE: None
************************************************************************************/
tVoid tunmstr_DiagLogClientHandler :: vOnNewAppState(  tU32 u32OldAppState,  tU32 u32AppState  )
{
	// Decide to reg/unreg based on the state of APP
	ETG_TRACE_USR4(("entered into vOnNewAppState"));
	if(u32AppState==u32OldAppState)
	{
		return;
	}
	switch(u32AppState)
	{
	case AMT_C_U32_STATE_NORMAL:
		{
			vRegisterForService();
			break;
		}
	case AMT_C_U32_STATE_DIAGNOSIS:
		{
			vRegisterForService();
			break;
		}
	default:
		{
			ETG_TRACE_USR4(("vOnNewAppState()->Invalid APPID"));
			break;
		}
	} // end of Switch
}// end of NewAppState
/*************************************************************************************
*FUNCTION:     vHandle_g_Send_Next_Test_Result
*
*DESCRIPTION: Request from diaglog to send next test result to DiagLog after DTC w has been cleared in order to
			 update the DTC with the latest antenna DTC
************************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vHandle_g_Send_Next_Test_Result( amt_tclServiceData* poMessage )
{
	if(( NULL == poMessage) || (NULL==m_potunermaster_SPM_Client) || (nullptr == m_potunmstr_antdiag_Manager))
	{
		NORMAL_M_ASSERT (NULL != poMessage);
		return;
	}
	else
	{
		switch (poMessage->u8GetOpCode())
		{
		case AMT_C_U8_CCAMSG_OPCODE_STATUS :
		{
			midw_diaglogfi_tclMsgSendNextTestResultStatus oStatus;
			vGetDataFromAmt ( poMessage,oStatus );
			ETG_TRACE_USR4(("vHandle_g_Send_Next_Test_Result()-> Status=%d",oStatus.Status));
			#ifndef GTEST_x86LINUX_BUILD
			if(oStatus.Status == TRUE)
			#endif
			{
#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
				if(m_potunermaster_SPM_Client->u32GetCVMEvent()==spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_CRITICAL_HIGH_VOLTAGE_START)
#else
				if(m_potunermaster_SPM_Client->u32GetCVMEvent()==cfc_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_CRITICAL_HIGH_VOLTAGE_START)
#endif
				{
					vUpdateITCAsNoResult();
				}
				else
				{
				   m_potunmstr_antdiag_Manager->vPostEvent(TUNMSTR_ANTDIAG_U32_EVENT_MASK_ITC_UPDATE_REQ);
				}
			}
			break;
		}
		default:
		{
			break;
		}
		}
	}
	//poMessage->bDelete();
}
/*************************************************************************************
*FUNCTION:     vHandleErrorLog
*
*DESCRIPTION:  Handle the Error Log request messages
*
*PARAMETER:    tenAntDiagClients enAntDiagClient,tU8 u8AntennaStatus,tFloat fCurrent
*
*RETURNVALUE:  tVoid
************************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vHandleErrorLog(tenAntDiagClients enAntDiagClient,tU8 u8AntennaStatus,tFloat fCurrent)
{
	if((NULL==m_potunmstr_MsgToHMI)||(NULL==m_potunmstr_antdiag_Manager)||(NULL==m_potunmstr_antdiag_Measure))
	{
		return;
	}
	ETG_TRACE_USR4(("vHandleErrorLog()-> enAntDiagClient=%02x u8AntennaStatus=%02x",
		ETG_CENUM(tenAntDiagClients,enAntDiagClient),
		ETG_CENUM(tenRfAntennaStatus,u8AntennaStatus)));
	tBool blRes=blCheckBoundaryCond(u8AntennaStatus,enAntDiagClient);
	if(!blRes)
	{
		if((tU8)enAntennState_Deactivated_HighVg!=u8AntennaStatus)
		{
			u8AntennaStatus=(tU8)enAntennState_Unknown;
		}
	}
	#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
	tenCMExtPowerSupplyType enCMExtPowerSupplyType=m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType(enAntDiagClient);
	if(ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))
	{
		if((blRes==true) && (u8AntennaStatus ==enAntennState_Open))
		{
			trI2CRead otrI2CRead = m_potunmstr_antdiag_i2cHandler->trReadI2CReg(enSUP_VoltageReg,enCMExtPowerSupplyType);
			if(!otrI2CRead.blReadSucess)
			{
				return;
			}
			tFloat fSupvoltage = otrI2CRead.u8Value;
			fSupvoltage=TUNMSTR_ANTDIAG_ELMOS_SUPPLY_VG_MUL_FAC*fSupvoltage;
			if(fSupvoltage<=TUNMSTR_ANTDIAG_ELMOS_SUPPLY_VG)
			{
				m_potunmstr_antdiag_Debounce->vUpdateAntennaStatus(enAntDiagClient,enAntennState_ELMOSOpen_Noresult);
				u8AntennaStatus = enAntennState_ELMOSOpen_Noresult;
				ETG_TRACE_USR4(("Supply Voltage of ELMOS is %f <= %f V, no diagnosis must be done",TUNMSTR_ANTDIAG_ELMOS_SUPPLY_VG,fSupvoltage));
			}
		}
	}
	#endif
	if(blUpdateAntennaStatus(enAntDiagClient,u8AntennaStatus))
	{
		if(m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType(enAntDiagClient)==enMAXIM_16946)
		{
			fCurrent=m_potunmstr_antdiag_Measure->fGetMaxIC_CalculatedCur();
		}
		m_potunmstr_MsgToHMI->vSend_g_RFAntennaTest((tU8)enAntDiagClient,u8AntennaStatus,fCurrent);
		tU16 u16ITCOpen=u16GetITCCodeForOpen(enAntDiagClient);
		tU16 u16ITCShort=u16GetITCCodeForShort(enAntDiagClient);
		tU16 u16ITCOpen2Amp=u16GetITCCodeForOpen2Amp(enAntDiagClient);
		vUpdateErrorLog(u16ITCOpen,u16ITCShort,u16ITCOpen2Amp,u8AntennaStatus);
	}
	if(u8AntennaStatus==enAntennState_Short)
    {
        vHandleAntennaStateShort(enAntDiagClient);
	}
}
/*************************************************************************************
*FUNCTION:     vHandleAntennaStateShort
*
*DESCRIPTION:  Handle the Antenna state short
*
*PARAMETER:     tenAntDiagClients enAntDiagClient
*
*RETURNVALUE:  tVoid
************************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vHandleAntennaStateShort(tenAntDiagClients enAntDiagClient)
{
	ETG_TRACE_USR4(("vHandleAntennaStateShort()-> enAntDiagClient=%02x",
		ETG_CENUM(tenAntDiagClients,enAntDiagClient)));
	if((NULL!=m_potunmstr_antdiag_Debounce)&& (NULL!=m_potunmstr_antdiag_Manager)&& (NULL!=m_potunmstr_antdiag_i2cHandler))
	{
	//If AntennaShortCircuitWaitTime =0, Don't turn off the antenna
	if(FALSE==m_potunmstr_antdiag_Debounce->u8GetShortCkt_WaitOverTime(enAntDiagClient))
	{
		ETG_TRACE_USR4(("vHandleAntennaStateShort()-> no short circuit wait over time"));	
		return;
	}
	tenCMExtPowerSupplyType enCMExtPowerSupplyType=m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType(enAntDiagClient);
	if((ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType) )|| (ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType)))
    {
    
        ETG_TRACE_USR4(("vHandleAntennaStateShort()-> enCMExtPowerSupplyType=%02x write status register",
			ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
        m_potunmstr_antdiag_i2cHandler->vSetOutputEnabled(enCMExtPowerSupplyType,FALSE);
        m_potunmstr_antdiag_i2cHandler->blEnableOutput(enCMExtPowerSupplyType);
    }
	else
	{
		ETG_TRACE_USR4(("vHandleAntennaStateShort()-> enCMExtPowerSupplyType=%02x turn off",
			ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
		//Turn off the GPIO pin
		tBool blRes=m_potunmstr_antdiag_Manager->blTurnOffGPIOPin(m_potunmstr_antdiag_Manager->enGetOSALGPIOPinName(enCMExtPowerSupplyType));
		m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enCMExtPowerSupplyType,!(blRes));
	}
	}
}
/*************************************************************************************
*FUNCTION:     blUpdateAntennaStatus
*
*DESCRIPTION:  Function to decide whether the Antenna status has to be sent to all the registered clients or not
			    Based on Previous Antenna status and current Antenna status.
*
*PARAMETER:     tenAntDiagClients enAntDiagClient,tU8 u8AntennaStatus
*
*RETURNVALUE:  tBool
************************************************************************************/
tBool tunmstr_DiagLogClientHandler::blUpdateAntennaStatus
(tenAntDiagClients enAntDiagClient,tU8 u8AntennaStatus)
{
	tBool blRes=FALSE;
	if(enAntDiagClient>=enClientNone)
	{
		return blRes;
	}
	else
	{
		ETG_TRACE_USR4(("blUpdateAntennaStatus()-> enAntDiagClient=%02x u8AntennaStatus=%02x m_u8AntennaStatus=%02x",
		ETG_CENUM(tenAntDiagClients,enAntDiagClient),
		ETG_CENUM(tenRfAntennaStatus,u8AntennaStatus),
		ETG_CENUM(tenRfAntennaStatus,m_u8AntennaStatus[enAntDiagClient])));
		if(m_u8AntennaStatus[enAntDiagClient]!=u8AntennaStatus)
		{
			m_u8AntennaStatus[enAntDiagClient]=u8AntennaStatus;
			blRes=TRUE;
		}
	}
	return blRes;
}
/*************************************************************************************
*FUNCTION:     blCheckBoundaryCond
*
*DESCRIPTION:  Function to the boundary conditions for the Antenna State update based on Ignition and Battery Voltage configured in KDS.
*
*PARAMETER:     tU8 u8AntennaStatus,tenAntDiagClients enAntDiagClient
*
*RETURNVALUE:  tBool
************************************************************************************/
tBool tunmstr_DiagLogClientHandler::blCheckBoundaryCond
(tU8 u8AntennaStatus,tenAntDiagClients enAntDiagClient)
{
	if((m_potunmstr_antdiag_Manager==NULL)||(m_potunmstr_antdiag_i2cHandler==NULL)||(m_potunmstr_antdiag_Debounce==NULL))
	{
		return false;
	}
	tBool blRes=TRUE;
	tU8 u8Index=u8GetIndex(u8AntennaStatus);
	if (u8Index==(tU8)enOkInd)
	{
		blRes=TRUE;
	}
	else if(u8Index<(tU8)enOkInd)
	{
		if((!m_blConsider_Vg[enAntDiagClient][u8Index])&&(!m_blConsider_Ign[enAntDiagClient][u8Index]))
		{
			blRes=TRUE;
		}
		else if((!m_blConsider_Vg[enAntDiagClient][u8Index])&&(m_blConsider_Ign[enAntDiagClient][u8Index]))
		{
			blRes=blGetIgnitationState((tU8)enAntDiagClient,u8Index);
		}
		else if((m_blConsider_Vg[enAntDiagClient][u8Index])&&(!m_blConsider_Ign[enAntDiagClient][u8Index]))
		{
			blRes= blBatterVgState((tU8)enAntDiagClient,u8Index);
		}
		else
		{
			blRes=(blBatterVgState((tU8)enAntDiagClient,u8Index)&& blGetIgnitationState((tU8)enAntDiagClient,u8Index));
		}
	}
	else
	{
		//Return False
	}
	ETG_TRACE_USR4(("blCheckBoundaryCond()-> enAntDiagClient=%02x blRes=%d",
	ETG_CENUM(tenAntDiagClients,enAntDiagClient),blRes));
	return blRes;
}
/*************************************************************************************
*FUNCTION:     u8GetIndex
*
*DESCRIPTION:  Function to find out the index of array based on Antenna status .
*
*PARAMETER:     tU8 u8AntennaStatus
*
*RETURNVALUE:  tU8 u8index
************************************************************************************/
tU8 tunmstr_DiagLogClientHandler::u8GetIndex(tU8 u8AntennaStatus)const
{
	tU8 u8Index=(tU8)enAntennaStateIndexMaxVal;
	switch(u8AntennaStatus)
	{
	case enAntennState_Ok:
		{
			u8Index=(tU8)enOkInd;
			break;
		}
	case enAntennState_Open:
		{
			u8Index=(tU8)enOpenInd;
			break;
		}
	case enAntennState_OpentoAmplifier:
		{
			u8Index=(tU8)enOpen2AmpInd;
			break;
		}
	case enAntennState_Short:
		{
			u8Index=(tU8)enShortInd;
			break;
		}
	default:
		{
			break;
		}
	}
	ETG_TRACE_USR4(("u8GetIndex()-> u8Index=%02x",
	ETG_CENUM(ten_AntennaStateIndex,u8Index)));
	return u8Index;
}
/*************************************************************************************
*FUNCTION:     blGetIgnitationState
*
*DESCRIPTION:  Function to get the ignitation state
*
*PARAMETER:    tU8 u8Client tU8 u8AntennaStatus
*
*RETURNVALUE:  tBool
************************************************************************************/
tBool tunmstr_DiagLogClientHandler::blGetIgnitationState(tU8 u8Client, tU8 u8AntStateInd)const
{
	if(NULL==m_pocfcVehicleClientHand)
	{
		NORMAL_M_ASSERT (m_pocfcVehicleClientHand != NULL);
		return FALSE;
	}
	ETG_TRACE_USR4(("blGetIgnitationState()->Ignitionstate=%d u8Client=%d u8AntStateInd=%d"
		,m_pocfcVehicleClientHand->blGetIgnitionstate(),u8Client,u8AntStateInd));
	return (m_pocfcVehicleClientHand->blGetIgnitionstate());
}
/*************************************************************************************
*FUNCTION:     blBatterVgState
*
*DESCRIPTION:  Function to get the battery volatage condition 
*
*PARAMETER:    tU8 u8Client tU8 u8AntennaStatus
*
*RETURNVALUE:  tBool
************************************************************************************/
tBool tunmstr_DiagLogClientHandler::blBatterVgState(tU8 u8Client, tU8 u8AntStateInd)const
{
	tBool blRet=FALSE;
	if(NULL==m_pocfcVehicleClientHand)
	{
		NORMAL_M_ASSERT (m_pocfcVehicleClientHand != NULL);
		return blRet;
	}
	if(m_pocfcVehicleClientHand->f32GetBatteryVg()>=(f_32BatterVgState[u8Client][u8AntStateInd]))
	{
		blRet=TRUE;
	}
	else
	{
		blRet= FALSE;
	}
	ETG_TRACE_USR4(("blBatterVgState()->BatterVgState=%d",blRet));
	return blRet;
}
/*************************************************************************************
*FUNCTION:     u16GetITCCodeForOpen
*
*DESCRIPTION:  Returns the ITC code for request client and antenna.
*
*PARAMETER:     tU8 u8WhichClient,tU8 u8WhichAntenna
*
*RETURNVALUE:  tU16 u16ITCOpen
************************************************************************************/
tU16 tunmstr_DiagLogClientHandler::u16GetITCCodeForOpen(tenAntDiagClients enAntDiagClient)const
{
	tU16 u16ITCOpen=0;
	switch(enAntDiagClient)
	{
	case (tU8)enDAB1:
		{
			u16ITCOpen=(tU16)ITC_DAB_TUNER_ANT1_OPEN;
			break;
		}
	case (tU8)enFM1:
		{
			u16ITCOpen=(tU16)ITC_AMFM_TUNER_ANT1_OPEN;
			break;
		}
	case (tU8)enFM2:
		{
			u16ITCOpen=(tU16)ITC_AMFM_TUNER_ANT2_OPEN;
			break;
		}
	case (tU8) enGPS:
		{
			u16ITCOpen=(tU16)ITC_SENSOR_GPS_ANTENNA_OPEN;
			break;
		}
	case enDTV1:
	{
		u16ITCOpen=ITC_DTV_ANT1_OPEN;
		break;
	}
	case enDTV2:
	{
		u16ITCOpen=ITC_DTV_ANT2_OPEN;
		break;
	}
	case enDTV3:
	{
		u16ITCOpen=ITC_DTV_ANT3_OPEN;
		break;
	}
	case enDTV4:
	{
		u16ITCOpen=ITC_DTV_ANT4_OPEN;
		break;
	}
	default:
		{
			//only two clients i.e DAB and FM
			break;
		}
	}
	return u16ITCOpen;
}
/*************************************************************************************
*FUNCTION:     u16GetITCCodeForShort
*
*DESCRIPTION:  Returns the ITC code for request client and antenna.
*
*PARAMETER:     tU8 u8WhichClient,tU8 u8WhichAntenna
*
*RETURNVALUE:  tU16 u16ITCShort
************************************************************************************/
tU16 tunmstr_DiagLogClientHandler::u16GetITCCodeForShort(tenAntDiagClients enAntDiagClient)const
{
	tU16 u16ITCShort=0;
	switch(enAntDiagClient)
	{
	case (tU8)enDAB1:
		{
			u16ITCShort=(tU16)ITC_DAB_TUNER_ANT1_SHORT;
			break;
		}
	case (tU8)enFM1:
		{
			u16ITCShort=(tU16)ITC_AMFM_TUNER_ANT1_SHORT;
			break;
		}
	case (tU8)enFM2:
		{
			u16ITCShort=(tU16)ITC_AMFM_TUNER_ANT2_SHORT;
			break;
		}
	case (tU8)enGPS:
		{
			u16ITCShort=(tU16)ITC_SENSOR_GPS_ANTENNA_SHORT;
			break;
		}
	case enDTV1:
	{
		u16ITCShort=ITC_DTV_ANT1_SHORT_GND;
		break;
	}
	case enDTV2:
	{
		u16ITCShort=ITC_DTV_ANT2_SHORT_GND;
		break;
	}
	case enDTV3:
	{
		u16ITCShort=ITC_DTV_ANT3_SHORT_GND;
		break;
	}
	case enDTV4:
	{
		u16ITCShort=ITC_DTV_ANT4_SHORT_GND;
		break;
	}
	default:
		{
			break;
		}
	}
	return u16ITCShort;
}
/*************************************************************************************
*FUNCTION:     u16GetITCCodeForOpen2Amp
*
*DESCRIPTION:  Returns the ITC code for request client and antenna.
*
*PARAMETER:     tU8 u8WhichClient,tU8 u8WhichAntenna
*
*RETURNVALUE:  tU16 u16ITCOpen2Amp
************************************************************************************/
tU16 tunmstr_DiagLogClientHandler::u16GetITCCodeForOpen2Amp(tenAntDiagClients enAntDiagClient)const
{
	tU16 u16ITCOpen2Amp=0;
	switch(enAntDiagClient)
	{
	case (tU8)enDAB1:
		{
			u16ITCOpen2Amp=(tU16)ITC_DAB_TUNER_ANT1_OPEN_TO_AMPLIFIER;
			break;
		}
	case (tU8)enFM1:
		{
			u16ITCOpen2Amp=(tU16)ITC_AMFM_TUNER_ANT1_OPEN_TO_AMPLIFIER;
			break;
		}
	case (tU8)enFM2:
		{
			u16ITCOpen2Amp=(tU16)ITC_AMFM_TUNER_ANT2_OPEN_TO_AMPLIFIER;
			break;
		}
	case (tU8)enGPS:
		{
			u16ITCOpen2Amp=(tU16)ITC_SENSOR_GPS_ANTENNA_OPEN_TO_AMPLIFIER;
			break;
		}
	default:
		{
			break;
		}
	}
	return u16ITCOpen2Amp;
}
/*************************************************************************************
*FUNCTION:     vUpdateErrorLog
*
*DESCRIPTION:  Updates the error error log with antenna status
*
*PARAMETER:     tU16 u16ITCOpen,tU16 u16ITCShort,tU8 u8AntennaStatus
*
*RETURNVALUE:  tVoid
************************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vUpdateErrorLog(tU16 u16ITCOpen,tU16 u16ITCShort,tU16 u16ITCOpen2Amp,tU8 u8AntennaStatus)
{
	ETG_TRACE_USR1(("vUpdateErrorLog()"));
	//Creating an object of MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT
	midw_diaglogfi_tclMsgSaveTestResultMethodStart oMsg;

	// Copy the test results - Trouble code and status
	midw_fi_tcl_TestResult oTestResult;
	tBool blSend=TRUE;
	switch(u8AntennaStatus)
	{
	case (tU8)enAntennState_Ok:
		{
			oTestResult.TroubleCode = u16ITCOpen;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCShort;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCOpen2Amp;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			ETG_TRACE_USR2(("vUpdateErrorLog()->Error log updated by OK for ITCOpen =%x ITCShort=%x ITCOpen2Amp=%x"
					,u16ITCOpen,u16ITCShort,u16ITCOpen2Amp));
			break;
		}
	case (tU8)enAntennState_Short:
		{
			oTestResult.TroubleCode = u16ITCOpen;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCShort;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_FAILED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCOpen2Amp;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			ETG_TRACE_USR2(("vUpdateErrorLog()->Error log updated by SHORT for ITCOpen =%x ITCShort=%x ITCOpen2Amp=%x"
					,u16ITCOpen,u16ITCShort,u16ITCOpen2Amp));
			break;
		}
	case (tU8)enAntennState_Open:
		{
			oTestResult.TroubleCode = u16ITCOpen;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_FAILED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCShort;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCOpen2Amp;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);
			ETG_TRACE_USR2(("vUpdateErrorLog()->Error log updated by OPEN for ITCOpen =%x ITCShort=%x ITCOpen2Amp=%x"
					,u16ITCOpen,u16ITCShort,u16ITCOpen2Amp));
			break;
		}
	case enAntennState_OpentoAmplifier:
		{
			oTestResult.TroubleCode = u16ITCOpen;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCShort;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCOpen2Amp;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_FAILED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);
			ETG_TRACE_USR2(("vUpdateErrorLog()->Error log updated by OPEN2AMP for ITCOpen =%x ITCShort=%x ITCOpen2Amp=%x"
					,u16ITCOpen,u16ITCShort,u16ITCOpen2Amp));
			break;
		}
	case enAntennState_Unknown:
		{
			oTestResult.TroubleCode = u16ITCOpen;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_NORESULT;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCShort;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_NORESULT;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCOpen2Amp;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_NORESULT;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);
			ETG_TRACE_USR2(("vUpdateErrorLog()->Error log updated by NORESULT Unknow state ITCOpen =%x ITCShort=%x ITCOpen2Amp=%x"
					,u16ITCOpen,u16ITCShort,u16ITCOpen2Amp));
			break;
		}
	#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
		case enAntennState_ELMOSOpen_Noresult:
		{
			oTestResult.TroubleCode = u16ITCOpen;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_NORESULT;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCShort;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);

			oTestResult.TroubleCode = u16ITCOpen2Amp;
			oTestResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oMsg.TestResultList.TestResultList.push_back (oTestResult);
			ETG_TRACE_USR2(("vUpdateErrorLog()->Error log updated by NORESULT for ITCOpen =%x ITCShort=%x ITCOpen2Amp=%x"
					,u16ITCOpen,u16ITCShort,u16ITCOpen2Amp));
			break;
		}
#endif
	default:
		{
			blSend=FALSE;
			ETG_TRACE_USR1(("vUpdateErrorLog()->Default Case"));
			break;
		}
	}
	// Send the message to VD_DIAGLOG
	if(blSend)
	{
	vSendMessage ( CCA_C_U16_APP_DIAGLOG,
		oMsg,
		MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT,
		AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
		0,
		m_u16RegID );
	}
}
/*************************************************************************
*
* FUNCTION:     vHandleTTFiS_CMD( )
*
* DESCRIPTION:  Handler to handle TTFIs Commands
*
* PARAMETER:    tU8DataLength, const tU8*u8DataSpecial
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vHandleTTFiS_CMD(const tU8* u8DataSpecial)
{
	tU8 u8FunctionID = u8DataSpecial[0];
	//CCA message Function ID
	if ( NULL == poGWMain )
	{
		ETG_TRACE_USR4(("vHandleTTFiS_CMD()->Null pointer"));
		NORMAL_M_ASSERT (NULL != poGWMain);
		return;
	}
	tU16 u16RegID = poGWMain->u16GetRegID();
	ETG_TRACE_USR4((" vHandleTTFiS_CMD()->u8FunctionID=%d u16RegID=%d ",
		u8FunctionID,
		u16RegID));
	switch(u8FunctionID)
	{
	case MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT:
		{
			midw_diaglogfi_tclMsgSendNextTestResultStatus oStatus;
			oStatus.Status =(tBool)u8DataSpecial[1];
			vSendMessage( CCA_C_U16_APP_TUNERMASTER,// Destination APP ID
				oStatus,           // parameter&
				MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT, //FID
				AMT_C_U8_CCAMSG_OPCODE_STATUS,  // OpCode                           /* amt_tclServiceData* */
				0,//CmdCounetr
				0 //Registration ID
				);
			ETG_TRACE_USR4(("vHandleTTFiS_CMD()->MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT sent"));
			break;
		}
	case TUNMSTR_FM1ConsiderBatVg:
		{
			m_blConsider_Vg[enFM1][enOpenInd]=u8DataSpecial[1];
			m_blConsider_Vg[enFM1][enOpen2AmpInd]=u8DataSpecial[2];
			m_blConsider_Vg[enFM1][enShortInd]=u8DataSpecial[3];
			break;
		}
	case TUNMSTR_DAB1ConsiderBatVg:
		{
			m_blConsider_Vg[enDAB1][enOpenInd]=u8DataSpecial[1];
			m_blConsider_Vg[enDAB1][enOpen2AmpInd]=u8DataSpecial[2];
			m_blConsider_Vg[enDAB1][enShortInd]=u8DataSpecial[3];
			break;
		}
	case TUNMSTR_FM1ConsiderIgn:
		{
			m_blConsider_Ign[enFM1][enOpenInd]=u8DataSpecial[1];
			m_blConsider_Ign[enFM1][enOpen2AmpInd]=u8DataSpecial[2];
			m_blConsider_Ign[enFM1][enShortInd]=u8DataSpecial[3];
			break;
		}
	case TUNMSTR_DAB1ConsiderIgn:
		{
			m_blConsider_Ign[enDAB1][enOpenInd]=u8DataSpecial[1];
			m_blConsider_Ign[enDAB1][enOpen2AmpInd]=u8DataSpecial[2];
			m_blConsider_Ign[enDAB1][enShortInd]=u8DataSpecial[3];
			break;
		}
	default:
		{
			//nothing
			break;
		}
	}
}
/*************************************************************************
* FUNCTION:     vHandleSendNextTestResultReq
*
* DESCRIPTION:  Function to handle the request from Diglog to turnon/off the antenna

* PARAMETER:   void
*
* RETURNVALUE:  tVoid
*************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vHandleSendNextTestResultReq()
{
	if((NULL!=m_potunmstr_antdiag_Manager) && (NULL!=m_potunmstr_antdiag_Debounce))
	{
		ETG_TRACE_USR4(("ANTDIAG::REQ from diagnosis to update Antenna Status after DTC has been cleared"));
		for(tU8 u8Index=enFM1;u8Index<enClientNone;u8Index++)
		{
			tenCMExtPowerSupplyType enCMExtPowerSupplyType=m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType((tenAntDiagClients)u8Index);
			if(m_potunmstr_antdiag_Manager->blIsExtSup_Supports(enCMExtPowerSupplyType))
			{
				//If antenna power supply is turned off
				if(!(m_potunmstr_antdiag_Manager->blGetExtPwrSupEN_PinState(enCMExtPowerSupplyType)))
				{
					//Turn on
					ETG_TRACE_USR4(("vHandleSendNextTestResultReq()-> %02x Antenna is OFF. Turn on the Antenna",
					ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
					m_potunmstr_antdiag_Manager->vTurnOnExtPowerSupplyType(enCMExtPowerSupplyType);
				}
				ETG_TRACE_USR4(("vHandleSendNextTestResultReq()-> Clear the counters and Antenna status for %02x client",
					ETG_CENUM(tenAntDiagClients, u8Index)));
				m_potunmstr_antdiag_Debounce->vClearShortWaitOverCounter(m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType((tenAntDiagClients)u8Index));
				m_potunmstr_antdiag_Debounce->vClearCounter((tenAntDiagClients)u8Index);
				blUpdateAntennaStatus((tenAntDiagClients)u8Index,enAntennState_Unknown);
			}
		}
	}
}
tVoid tunmstr_DiagLogClientHandler::vUpdateITCAsNoResult()
{
	if(NULL!=m_potunmstr_antdiag_Manager)
	{
		for(tU8 u8Index=enFM1;u8Index<enClientNone;u8Index++)
		{
			tenCMExtPowerSupplyType enCMExtPowerSupplyType=m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType((tenAntDiagClients)u8Index);
			if(m_potunmstr_antdiag_Manager->blIsExtSup_Supports(enCMExtPowerSupplyType))
			{
				tU16 u16ITCOpen=u16GetITCCodeForOpen((tenAntDiagClients)u8Index);
				tU16 u16ITCShort=u16GetITCCodeForShort((tenAntDiagClients)u8Index);
				tU16 u16ITCOpen2Amp=u16GetITCCodeForOpen2Amp((tenAntDiagClients)u8Index);
				vUpdateErrorLog(u16ITCOpen,u16ITCShort,u16ITCOpen2Amp,enAntennState_Unknown);
			}
		}
	}
}
/*************************************************************************
* FUNCTION:     vClearAllAntennaStatus
*
* DESCRIPTION:  Function to clear all antenna status

* PARAMETER:    None
*
* RETURNVALUE:  tVoid
*************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vClearAllAntennaStatus()
{
	m_u8AntennaStatus[enFM1]=(tU8)enAntennState_Unknown;
	m_u8AntennaStatus[enFM2]=(tU8)enAntennState_Unknown;
	m_u8AntennaStatus[enDAB1]=(tU8)enAntennState_Unknown;
	m_u8AntennaStatus[enGPS]=(tU8)enAntennState_Unknown;
	m_u8AntennaStatus[enDTV1]=(tU8)enAntennState_Unknown;
	m_u8AntennaStatus[enDTV2]=(tU8)enAntennState_Unknown;
	m_u8AntennaStatus[enDTV3]=(tU8)enAntennState_Unknown;
	m_u8AntennaStatus[enDTV4]=(tU8)enAntennState_Unknown;
	m_u8AntennaStatus[enDAB2]=(tU8)enAntennState_Unknown;
}
/*************************************************************************
* FUNCTION:     u8GetAntenanstate
*
* DESCRIPTION:  Function to get the request antenna state

* PARAMETER:    tenAntDiagClients enAntDiagClients
*
* RETURNVALUE:  tU8
*************************************************************************/
tU8 tunmstr_DiagLogClientHandler::u8GetAntenanstate(tenAntDiagClients enAntDiagClients)
{
	if(enAntDiagClients>=(tU8)enClientNone)
	{
		return (tU8)enAntennState_Unknown;
	}
	return m_u8AntennaStatus[enAntDiagClients];
}
/*************************************************************************
 * FUNCTION:    vUpdateAntennaParam_FromKDS()
 *
 * DESCRIPTION: function to update the member variables by reading the Antenna Parameter set of KDS
 *
 * PARAMETER:   void
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vUpdateAntennaParam_FromKDS(tunerVector<tU8> &AntennaPar)
{
	ETG_TRACE_USR4(("vUpdateAntennaParam_FromKDS()"));
	if(NULL == m_potunmstr_antdiag_Manager)
	{
		return;
	}

	tU8 u8AntStateIndex=0; //OPEN,OPEN2AMP,SHORT,ok
	tU16 u16KDSindex=enAntennaParameter_idx_FM1ConsiderVg_Open;
	for(tU8 i= enOpenInd;i<=enShortInd;i++)
	{

		for(tU8 u8Client=(tU8)enkdsFM1;u8Client<enClientNone;u8Client++)
		{
			tU8 u8AntDiagClientInx= m_potunmstr_antdiag_Manager->enGetAntDaigClientInx((tenAntDiagKDS)u8Client);
			if((tU16)(AntennaPar.size()) > (u16KDSindex +1))
			{
				if(AntennaPar.at(u16KDSindex)==0xff)
				{
					m_blConsider_Vg[u8AntDiagClientInx][u8AntStateIndex]=FALSE;
				}
				else
				{
					m_blConsider_Vg[u8AntDiagClientInx][u8AntStateIndex]=TRUE;
					f_32BatterVgState[u8AntDiagClientInx][u8AntStateIndex]=(tFloat)AntennaPar.at(u16KDSindex);
					f_32BatterVgState[u8AntDiagClientInx][u8AntStateIndex]=(f_32BatterVgState[u8AntDiagClientInx][u8AntStateIndex])/10;
				}
				if(AntennaPar.at(u16KDSindex+1)==0xff)
				{
					m_blConsider_Ign[u8AntDiagClientInx][u8AntStateIndex]=FALSE;
				}
				else
				{
					m_blConsider_Ign[u8AntDiagClientInx][u8AntStateIndex]=TRUE;
					u8IgnTime_InSec=AntennaPar.at(u16KDSindex+1);
				}
				u16KDSindex=(tU16)(u16KDSindex+TUNMSTR_ANTDIAG_DIFFERENCE_BTW_ANT_CLIENT_INDEX);
			}
		}
		u8AntStateIndex= (tU8)(u8AntStateIndex+1);
		u16KDSindex=(tU16)(u16KDSindex-TUNMSTR_ANTDIAG_DIFFERENCE_BTW_ANT_START_ND_END_INDEX);
		if(u8AntStateIndex==enShortInd)
		{
			u16KDSindex=(tU16)(u16KDSindex-1);
		}
	}
	vTraceKDSParam();
}
/*************************************************************************
 * FUNCTION:    vTraceKDSParam()
 *
 * DESCRIPTION: function to trace the member variables by reading the Antenna Parameter set of KDS
 *
 * PARAMETER:   void
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vTraceKDSParam()
{
	for(tU8 u8Client=(tU8)enFM1;u8Client<(tU8)enClientNone;u8Client++)
	{
		ETG_TRACE_USR4(("***********************vUpdateAntennaParam_FromKDS() for u8Client=%02x***********************",ETG_CENUM(tenAntDiagClients,u8Client)));
		for(tU8 u8AntIndex=(tU8)enOpenInd;u8AntIndex<=(tU8)enShortInd;u8AntIndex++)
		{
			ETG_TRACE_USR4(("vUpdateAntennaParam_FromKDS() u8AntStateIndex=%02x m_blConsider_Vg=%d,m_blConsider_Ign=%d",
			ETG_CENUM(ten_AntennaStateIndex,u8AntIndex),m_blConsider_Vg[u8Client][u8AntIndex],m_blConsider_Ign[u8Client][u8AntIndex]));
		}
	}
}
/*************************************************************************************
*FUNCTION:     vValidateIgnitionState
*
*DESCRIPTION:  Function to Validate the Ignition state based on KDS configuartion
*
*PARAMETER:     tVoid
*
*RETURNVALUE:  tVoid
************************************************************************************/
tVoid tunmstr_DiagLogClientHandler::vValidateIgnitionState()
{
	if(NULL==m_pocfcVehicleClientHand)
	{
		return;
	}
	tBool blCurIgnState	=	m_pocfcVehicleClientHand->blGetIgnitionstate();
	if((TRUE==blCurIgnState)&&(TRUE==m_blPreviousIgnState))
	{
		m_u8IgnCounter++;
	}
	else
	{
		m_u8IgnCounter=0;
	}
	m_blPreviousIgnState=blCurIgnState;
	ETG_TRACE_USR4(("vValidateIgnitionState() blCurIgnState=%d,Ignition Counter=%d",
		blCurIgnState,m_u8IgnCounter));
}
tVoid tunmstr_DiagLogClientHandler::vUpdateAntennaITC()
{
	for (tU8 u8AntDiagClient=enFM1;u8AntDiagClient<enClientNone;u8AntDiagClient++)
	{
		tU16 u16ITCOpen=u16GetITCCodeForOpen((tenAntDiagClients)u8AntDiagClient);
		tU16 u16ITCShort=u16GetITCCodeForShort((tenAntDiagClients)u8AntDiagClient);
		tU16 u16ITCOpen2Amp=u16GetITCCodeForOpen2Amp((tenAntDiagClients)u8AntDiagClient);
		vUpdateErrorLog(u16ITCOpen,u16ITCShort,u16ITCOpen2Amp,u8GetAntenanstate((tenAntDiagClients)u8AntDiagClient));
	}

}
tVoid tunmstr_DiagLogClientHandler::vClearAllInternalState()
{
	for(tU8 u8Index=(tU8)enFM1;u8Index<(tU8)enClientNone;u8Index++)
	{
		m_u8AntennaStatus[u8Index]=(tU8)enAntennState_Unknown;
		for(tU8 u8AntStateIndex=(tU8)enOpenInd;u8AntStateIndex<=(tU8)enShortInd;u8AntStateIndex++)
		{
			m_blConsider_Vg[u8Index][u8AntStateIndex]=FALSE;
			m_blConsider_Ign[u8Index][u8AntStateIndex]=FALSE;
			f_32BatterVgState[u8Index][u8AntStateIndex]=0xff;
		}
	}
	u8IgnTime_InSec=0;
	m_blPreviousIgnState=FALSE;
	m_u8IgnCounter=0;
}

tVoid tunmstr_DiagLogClientHandler::vHandle_s_UnUsedMessages()
{
	// do nothing
}
