/************************************************************************
 * FILE:        HandleTTFISCommands.h
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of HandleTTFISCommands.h
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
    Mar 14, 2016                     pmn3kor
				
 *************************************************************************/


#include "tuner_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_TUNAPP
#include "trcGenProj/Header/HandleTTFISCommands.cpp.trc.h"
#endif

#ifndef VARIANT_S_FTR_ENABLE_GENERICMSGS_MOCK
#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"
#endif

#ifndef VARIANT_S_FTR_ENABLE_AIL_MOCK
#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"         // use AIL template with MessageMaps
#endif

#ifndef VARIANT_S_FTR_ENABLE_AHL_MOCK
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library
#else
#include "ahl_mock/ahl_if.h"         // use Application Help Library
#endif

#ifndef VARIANT_S_FTR_ENABLE_AMT_MOCK
#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"
#else
#include "amt_mock/amt_if.h"
#endif

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_SERVICEINFO

#include "CcaApp/clServerMethod.h"
#include "CcaApp/clServerProperty.h"
#include "HandleTTFISCommands.h"
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "midw_fi_if.h"
#include "tun_MsgDefines.h"
#include "tun_defines.h"
#include "tun_MsgFromHMI.h"
#include "tun_MsgToADR.h"
#include "tun_Config.h"
#include "tun_DrvAdrIf.hpp"
#include "tun_ActualData.h"
#include "tun_main.h"
#include "tun_MsgToHMI.h"
#include "TunerApp/clTmcTuner_stationLogo_map.h"
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
#include "Aars/clAars_CM_Commit.h"
#endif
#include "AMFMTuner/clTmcTuner_Method_S_AUDIO_FG_BG_NEW.h"
HandleTTFISCommands* HandleTTFISCommands::_poTTFISHandler = NULL;

/*****************************************************************************
* Constructor
******************************************************************************/
HandleTTFISCommands::HandleTTFISCommands()
{
	_fdTrace = 0;
   _poTTFISHandler = this;
   m_potun_MsgFromHMI = NULL;
   m_potun_MsgToADR = NULL;
   m_potun_Config = NULL;
   m_potun_ActualData = NULL;
   m_potun_main = NULL;
   m_poclTmcTuner_stationLogo_map = NULL;
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
   m_poclAars_CM_Commit = NULL;
#endif
}

/*****************************************************************************
* Destructor
******************************************************************************/
HandleTTFISCommands::~HandleTTFISCommands()
{
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
   m_poclAars_CM_Commit = NULL;
#endif
   _poTTFISHandler = NULL;
   m_potun_MsgFromHMI = NULL;
   m_potun_MsgToADR = NULL;
   m_potun_Config = NULL;
   m_potun_ActualData = NULL;
   m_potun_main = NULL;
   m_poclTmcTuner_stationLogo_map = NULL;
}

/*****************************************************************************
* vSet_tun_AarsCommit
******************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
void HandleTTFISCommands::vSet_tun_AarsCommit(clAars_CM_Commit* poclAars_CM_Commit)
{
   m_poclAars_CM_Commit = poclAars_CM_Commit;
}
#endif

/*****************************************************************************
* vSet_tun_MsgFromHMI_Ptr
******************************************************************************/
void HandleTTFISCommands::vSet_tun_MsgFromHMI_Ptr(tun_MsgFromHMI* potun_MsgFromHMI)
{
   m_potun_MsgFromHMI = potun_MsgFromHMI;
}

/*****************************************************************************
* vSet_tun_MsgToADR_Ptr
******************************************************************************/
void HandleTTFISCommands::vSet_tun_MsgToADR_Ptr(tun_MsgToADR* potun_MsgToADR)
{
   m_potun_MsgToADR = potun_MsgToADR;
}

/*****************************************************************************
* vSet_tun_Config_Ptr
******************************************************************************/
void HandleTTFISCommands::vSet_tun_Config_Ptr(tun_Config* potun_Config)
{
   m_potun_Config = potun_Config;
}

/*****************************************************************************
* vSet_tun_ActualData_Ptr
******************************************************************************/
void HandleTTFISCommands::vSet_tun_ActualData_Ptr(tun_ActualData* potun_ActualData)
{
   m_potun_ActualData = potun_ActualData;
}

/*****************************************************************************
* vSet_tun_Main_Ptr
******************************************************************************/
void HandleTTFISCommands::vSet_tun_Main_Ptr(tuner_tclApp* potun_main)
{
   m_potun_main = potun_main;
}
/*****************************************************************************
* vSet_StationLogoMapPtr
******************************************************************************/
void HandleTTFISCommands::vSet_StationLogoMapPtr(clTmcTuner_stationLogo_map* poclTmcTuner_stationLogo_map)
{
	m_poclTmcTuner_stationLogo_map = poclTmcTuner_stationLogo_map;
}
/*****************************************************************************
* vRegisterInTraceCmdService
******************************************************************************/
void HandleTTFISCommands::vRegisterInTraceCmdService()
{

   tS32 s32Error;
   OSAL_trIOCtrlLaunchChannel  oTraceChannel;

   ETG_TRACE_USR1(( " tuner_tclApp::vRegisterInTraceCmdService() -> "
         "Register trace interface." ));

   _fdTrace = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE);

   oTraceChannel.enTraceChannel = TR_TTFIS_TUNER;
   oTraceChannel.pCallback = (OSAL_tpfCallback)vTraceTunerChannelMgr;
   s32Error = OSAL_s32IOControl (_fdTrace, OSAL_C_S32_IOCTRL_CALLBACK_REG, (intptr_t) &oTraceChannel);

   if (OSAL_OK != s32Error)
   {
      ETG_TRACE_ERR(( " HandleTTFISCommands::vRegisterInTraceCmdService() -> ERROR: "
            "Registration of call back function failed, s32Error=%x."
                      ,s32Error ));
      return;
   }
   else
   {
      ETG_TRACE_USR1(( " HandleTTFISCommands::vRegisterInTraceCmdService() -> "
            "Registration of call back function is successfull." ));
   }
}

/*****************************************************************************
* vUnregisterInTraceCmdService( )
******************************************************************************/
void HandleTTFISCommands::vUnregisterInTraceCmdService() throw()
{
   tS32 s32Error;
   OSAL_trIOCtrlLaunchChannel oTraceChannel;
   oTraceChannel.enTraceChannel = TR_TTFIS_TUNER;
   oTraceChannel.pCallback = (OSAL_tpfCallback)vTraceTunerChannelMgr;

   s32Error = OSAL_s32IOControl (_fdTrace, OSAL_C_S32_IOCTRL_CALLBACK_UNREG, (intptr_t) &oTraceChannel);

   if (OSAL_OK != s32Error)
   {
      ETG_TRACE_ERR(( " HandleTTFISCommands::vUnregisterInTraceCmdService() -> "
            "ERROR: UnRegistration of call back function failed, s32Error=%x."
                            ,s32Error ));
      return;
   }
   else
   {
      ETG_TRACE_USR1(( " HandleTTFISCommands::vUnregisterInTraceCmdService() -> "
            "UnRegister trace interface." ));
   }
   OSAL_s32IOClose (_fdTrace);
}

/*****************************************************************************
* vTraceTunerChannelMgr
******************************************************************************/
void HandleTTFISCommands::vTraceTunerChannelMgr(unsigned char* puchData)
{
   if((( puchData == NULL ) || (_poTTFISHandler == NULL)) ||
         (_poTTFISHandler->m_potun_Config == NULL) ||
         (_poTTFISHandler->m_potun_MsgToADR==NULL)||
         (_poTTFISHandler->m_potun_ActualData==NULL))
   {
      NORMAL_M_ASSERT( puchData != NULL );
      NORMAL_M_ASSERT( _poTTFISHandler != NULL );
      if(_poTTFISHandler != NULL)
      {
          NORMAL_M_ASSERT( _poTTFISHandler->m_potun_Config != NULL );
          NORMAL_M_ASSERT( _poTTFISHandler->m_potun_MsgToADR != NULL );
          NORMAL_M_ASSERT( _poTTFISHandler->m_potun_ActualData != NULL );
      }
      return;
   }
   /*puchData[0] = gives length of the array*/
   unsigned char u8Messagelen = puchData[0];

   if(u8Messagelen >= 3 && puchData[1] == TUN_NEW_POST_SELF_HEADER)
   {
       unsigned short fiID = puchData [3];
       fiID = (unsigned short) (fiID |( ((unsigned short) puchData [2]) << 8));

       ETG_TRACE_USR4((" tuner_tclApp::vTraceTunerChannelMgr() : fiID = %d, puchData [ 1 ] = %d, puchData [ 2 ] = %d, puchData [ 3 ] = %d", fiID, puchData [1], puchData [2], puchData [3]));
       tuner::CcaApp::clFunction* poFunction = _poTTFISHandler->m_potun_MsgFromHMI->poGetFunction(fiID);
       if(poFunction != NULL)
       {
           ETG_TRACE_USR4((" tuner_tclApp::vTraceTunerChannelMgr() : fiID[%d] found", fiID));
           tuner::CcaApp::clServerMethod *poServerMethod = dynamic_cast<tuner::CcaApp::clServerMethod *>(poFunction);
           if(poServerMethod != NULL)
           {
               poServerMethod->vConstructSelfPostMessage(puchData);
               return;
           }
           else
           {
               tuner::CcaApp::clServerProperty *poServerProperty = dynamic_cast<tuner::CcaApp::clServerProperty *>(poFunction);
               if(poServerProperty != NULL)
               {
                   poServerProperty->vConstructSelfPostMessage(puchData);
                   return;
               }
           }
       }
   }
   /*puchData+1 = TRACEID*/
   switch(puchData [ 1 ])
   {

   case TUN_TRACE_SET_RAW_MESSAGE_TO_ADR: //command in TTFIs is TUN_MSG_TO_ADR
   {
      _poTTFISHandler->vHandle_TUN_MSG_TO_ADR_Trace(puchData);
      break;
   }

   case TUN_TRACE_HELP: //command in TTFIs is fc_tmctuner_Help
   {
      /* prints a help text */
      ET_TRACE_BIN( TUN_TRACE_CLASS_TUNAPP, TR_LEVEL_FATAL, ET_EN_T8LIST _ u8Messagelen _ &puchData[ 1] _ ET_EN_DONE);
      break;
   }

   case TUN_TRACE_ACT_DATA: //command in TTFIs is TUN_GET_ACT_DATA
   {
      _poTTFISHandler->m_potun_ActualData->TraceGetActTunerData( puchData[ 2 ] );
      break;
   }

   case TUN_TRACE_TUN_DRVADRIF:  // commands for TTFIs are TUN_DRVADRIF_GET_INFO and TUN_DRVADRIF_RESET_ADR
   {
      if( u8Messagelen >= 2)
      {
         ETG_TRACE_FATAL(( " HandleTTFISCommands::vTraceTunerChannelMgr-> TTFis command: TUN_TRACE_TUN_DRVADRIF."
            ));
         tun_DrvAdrIf::instance()->vTraceCommand(
            (tPU8)&puchData[ 2],    /* tPU8 pu8Msg */
            u8Messagelen - 1);      /* tU32 u32Len */
      }
      break;
   }

   case TUN_TRACE_TUN_MSGTOADR: // command for TTFIs is TUN_MSGTOADR_SETCONFIGURATION
   {
      if( u8Messagelen >= 2)
      {
         tun_MsgToADR::instance()->vTraceCommand(
         (tPU8)&puchData[ 2],    /* tPU8 pu8Msg */
         u8Messagelen - 1);      /* tU32 u32Len */
      }
      break;
   }

   case TUN_TRACE_TUN_CONFIG: //command for TTFIs is TUN_CONFIG
   {
      if( u8Messagelen >= 2)
      {
         _poTTFISHandler->m_potun_Config->vTraceCommand(
         (tPU8)&puchData[ 2],    /* tPU8 pu8Msg */
         u8Messagelen - 1);      /* tU32 u32Len */
      }
      break;
   }

   case TUN_CHECK_CONTAINER:
   {
      _poTTFISHandler->vHandleContainerCheck(puchData);
      break;
   }
   case TUN_POST_SELF_MSG_DDA:
   {
	   _poTTFISHandler->vHandleDDAGet();
	   break;
   }
   case TUN_POST_SELF_MSG_AF:
   {
	   _poTTFISHandler->vHandleAFGet();
	   break;
   }
   case TUN_POST_SELF_MSG_SEEK:
   {
      _poTTFISHandler->vHandleSeekMethod(puchData+1);
      break;
   }
   case TUN_POST_SELF_MSG_PRESET_HANDLING:
   {
      _poTTFISHandler->vHandlePresetHandlingMethod(puchData + 1);
      break;
   }
   case TUN_POST_SELF_MSG_ATFREQUENCY: 
   {
      _poTTFISHandler->vHandleAtfrequencyMethod(puchData + 1);
      break;
   }
   case TUN_POST_SELF_MSG_CHANGE_LSM: 
   {
      _poTTFISHandler->vHandleChangeLSMMethod(puchData + 1);
      break;
   }
   case TUN_POST_SELF_MSG_GET_STATIC_LIST: 
   {
      _poTTFISHandler->vHandleGetStaticListMethod(puchData + 1);
      break;
   }
   	case TUN_POST_SELF_MSG_ANNOUNCEMENT_ESCAPE:
   {
      _poTTFISHandler->vHandleAnnouncementEscapeMethod(puchData + 1);
      break;
   }
   case TUN_POST_SELF_MSG_ANNOUNCEMENT_SWITCH:
   {
      _poTTFISHandler->vHandleAnnouncementSwitchMethod(puchData + 1);
      break;
   }
   case TUN_POST_SELF_MSG_SENSOR_POSITION:
   {
	   _poTTFISHandler->vHandleSensorPositionProperty(puchData+1);
	   break;
   }

   case TUN_ADR_RESET:
   {
      tun_DrvAdrIf::instance()->vResetAdr();
      break;
   }

   case TUN_SDS_DIRECT_FREQUENCY:
   {
      unsigned int u32Frequency = 87500;
      unsigned  char u8Band = 0x00;
      //unsigned char u8PhysicalTuner = 0xFF;
      //_poExemplar ->m_ptun_MsgToADR->vSend_TUNFgBg(0,0,0,1,u8PhysicalTuner);
      _poTTFISHandler ->m_potun_MsgToADR->vSendLSMSetup(u8Band,u32Frequency,0x00,0x00,0x00);
      break;
   }

   case TUN_SEND_COMMIT:
   {
       ETG_TRACE_USR1(("CCR.STARTING.LASTMODE_DATA:TUN_SEND_COMMIT"));
      //commit is implemented in new type. so changes made. (clAars_CM_Commit::vSendCommitMsg)
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20
      _poTTFISHandler ->m_potun_MsgToADR->vSendCommit();
#else
      _poTTFISHandler->vHandleCommitMethod();
#endif
      break;
   }

   case TUN_CMD_FEATURE_HIGHCUT:
   {
      _poTTFISHandler->vHandleHicutMethod(puchData+1);
      break;
   }

   case TUN_CMD_FEATURE_SHARX:
   {
      _poTTFISHandler->vHandleSharxMethod(puchData+1);
      break;
   }

   case TUN_POST_CM_EVALUATIONPARAMETER:
   {
      _poTTFISHandler->m_potun_MsgToADR->vSendSetEvaluationParameter(puchData+1);
      break;
   }

   case TUN_GET_CURRENTSTATE_UPDATES:
   {
      tun_MsgToHMI* m_potun_MsgToHMI  = tun_MsgToHMI::instance();
      m_potun_MsgToHMI->vGetCurrentStateUpdates();
      break;
   }
   case TUN_POST_SELF_MSG_ANNOSWITCH:
   {
      tun_MsgToHMI* m_potun_MsgToHMI  = tun_MsgToHMI::instance();
      tU8 u8Message[20];
      u8Message[enAdrMsgOffset_DATA] = puchData[ 2 ];
      m_potun_MsgToHMI->vHandle_r_CM_AnnouncementSwitch(u8Message);
      break;
   }
   case TUN_POST_SELF_MSG_ANNOINFO:
   {
      tun_MsgToHMI* m_potun_MsgToHMI  = tun_MsgToHMI::instance();
      tU8 u8Message[40];
	  memset(u8Message, 0, 40);
      m_potun_MsgToHMI->vHandle_r_CM_AnnouncementInfo(u8Message);
      break;
   }
   case TUN_POST_SELF_MSG_ANNOMSG:
   {
      tun_MsgToHMI* m_potun_MsgToHMI  = tun_MsgToHMI::instance();
      tU8 u8Message[40];
      u8Message[enAdrMsgOffset_DATA] = puchData[ 2 ];
      m_potun_MsgToHMI->vHandle_r_CM_AnnouncementMessage(u8Message);
      break;
   }
   case TUN_POST_SELF_MSG_AUDIOSOURCE:
   {
      tun_MsgToHMI* m_potun_MsgToHMI  = tun_MsgToHMI::instance();
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
      m_potun_MsgToHMI->getTun_AudioFgBg_Ptr()->vOnNewTunerFGBGStatus(puchData[ 2 ]);
#endif
      break;
   }
   case TUN_POST_SELF_MSG_SEL_LIST_ELEM:
   {
      /*self post message for FID_TUN_S_SEL_LIST_ELEM*/
      _poTTFISHandler->vHandleSelecetListElement(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_SET_ENGG_PARAM:
   {
      /*self post message for FID_TUN_S_SET_ENGG_PARAMETER*/
      _poTTFISHandler->vHandleSetEnggParameterMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_GET_ENGG_PARAM:
   {
      /*self post message for FID_TUN_G_SET_ENGG_PARAMETER*/
      _poTTFISHandler->vHandleSetEnggParameterGet(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_SET_ACTIVE_PRSEST_LIST:
   {
      /*self post message for FID_TUN_S_SET_ACTIVE_PRESETLIST*/
      _poTTFISHandler->vHandleSetActivePresetMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_SET_SPEED_INFO:
   {
      /*self post message for FID_TUN_S_SPEED_INFO*/
      _poTTFISHandler->vHandleSpeedInfoMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_GET_AVAIL_BAND_RANGE:
   {
      /*self post message for FID_TUN_G_AVAILABLE_BAND_RANGE*/
      _poTTFISHandler->vHandleAvailBandRangeGet();
      break;
   }

   case TUN_POST_SELF_MSG_UPDATE_CONFIGLIST:
   {
      /*self post message for FID_TUN_S_CONFIGLIST_UPDATE*/
      _poTTFISHandler->vHandleUpdateConfigList(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_DEFAULT_SETTINGS:
   {
      /*self post message for FID_TUN_S_DEFAULT_SETTING*/
      _poTTFISHandler->vHandleDefaultSettingsMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_SET_DEMODULATION_SWITCH:
   {
      /*self post message for FID_TUN_S_SET_DEMODULATION_SWITCH*/
      _poTTFISHandler->vHandleDemodulationSwitchMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_SELECT_HD_AUDIOPRGM:
   {
      /*self post message for FID_TUN_S_SELECT_HD_AUDIOPRGM*/
      _poTTFISHandler->vHandleHDAudioPrgmMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_SET_DRM_AUDIOSERVICEDIRECT:
   {
      /*self post message for FID_TUN_S_SELECT_DRM_AUDIOPRGM*/
      _poTTFISHandler->vHandleDRMAudioServiceDirectMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_SET_ENHANCED_TESTMODE:
   {
      /*self post message for FID_TUN_S_SET_TESTMODE*/
      _poTTFISHandler->vHandleEnhancedTestmodeMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_SEL_HD_AUDIOPRGMDIRECT:
   {
      /*self post message for FID_TUN_S_SEL_HD_AUDIOPRGMDIRECT*/
      _poTTFISHandler->vHandleSelHDAudioPrgmDirectMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_SET_DIRECT_FREQUENCY:
   {
      /*self post message for FID_TUN_S_DIRECT_FREQUENCY*/
      _poTTFISHandler->vHandleDirectFrequencyMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_SET_DRM_TESTMODE:
   {
      /*invoke FID_TUN_S_DRM_TESTMODE trace handler*/
      _poTTFISHandler->vDRMTestModeMethodHandler(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_SET_HD_SECTION_PARAMETER:
   {
      /*self post message for FID_TUN_S_SET_AUDIO_SPLIT_MODE*/
      _poTTFISHandler->vHandleSetHDSectionParameterMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_HD_RESET_TRANSITIONCOUNT:
   {
      /*self post message for FID_TUN_S_HD_RESET_TRANSITIONCOUNT*/
      _poTTFISHandler->vHandleHDResetTransitionCount();
      break;
   }

   case TUN_POST_SELF_MSG_GET_CONFIG_LIST:
   {
      /*self post message for FID_TUN_S_GET_CONFIG_LIST*/
      _poTTFISHandler->vHandleGetConfigListMethod(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_ADR3COMMICATIONERR:
   {
      _poTTFISHandler->vHandleADRCommunicationErr(puchData);
      break;
   }

   case TUN_POST_SELF_MSG_JAPAN_SET_AREA:
   {
      /*self post message for FID_TUN_S_JAPAN_SET_AREA*/
      _poTTFISHandler->vHandleJapanSetArea(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_JAPAN_CHANGE_STATION_NAME:
   {
      _poTTFISHandler->vHandleJapanStationNameChange(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_USER_PROFILE_UPDATE:
   {
      _poTTFISHandler->vHandleUserProfileUpdate(puchData+1);
      break;
   }

   case TUN_POST_SELF_MSG_LANGUAGE_SETTING:
   {
	   _poTTFISHandler->vHandleLanguageSetting(puchData+1);
	   break;
   }

   case TUN_POST_SELF_MSG_RESET_MUTECOUNTER:
   {
	   _poTTFISHandler->vHandleDRMResetMuteCounter(puchData+1);
	   break;
   }

   case TUN_POST_SELF_MSG_PRESET_STORAGE_PS_FREEZE:
	{
		_poTTFISHandler->vHandlePresetStoragePSFreezeMethod(puchData+1);
		break;
	}
   case TUN_POST_SELF_MSG_JAPAN_SELECT_STATION_NAME:
   {
   		_poTTFISHandler->vHandleJapanSelectStationName(puchData+1);
		break;
   }
   default:
   {
      ETG_TRACE_USR1(("does not match any case in vTraceTunerChannelMgr TRACEID = 0x%x", puchData [ 1 ]));
      break;
   }

   }/* switch(puchData [ 1 ]) */
}

/*****************************************************************************
* vDRMTestModeMethodHandler
******************************************************************************/
void HandleTTFISCommands::vDRMTestModeMethodHandler(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL || puchData==NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      NORMAL_M_ASSERT(puchData != NULL);
      return;
   }

   midw_tunerfi_tclMsgFID_TUN_S_DRM_TESTMODEMethodStart oMsg;
   oMsg.e8DRMTestModeActivity.enType = (midw_fi_tcl_e8_Tun_DRMTestModeActivity::tenType)puchData[1]; /*puchData+1 = DRMTestMode ON/OFF*/

   ETG_TRACE_USR4((" HandleTTFISCommands::vDRMTestModeMethodHandler() : "
            "TTFis MSG -> TUN_POST_SELF_MSG_SET_DRM_TESTMODE : "
            "DRMTestMode = %d",
            puchData[1]));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
                     oMsg,                            /** parameter */
                     MIDW_TUNERFI_C_U16_FID_TUN_S_DRM_TESTMODE, /** FID */
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
                     0,                               /** Command Counter */
                     0                                /** Reg ID*/
                     );
}

/*****************************************************************************
* vHandle_TUN_MSG_TO_ADR_Trace
******************************************************************************/
void HandleTTFISCommands::vHandle_TUN_MSG_TO_ADR_Trace(unsigned char* puchData)
{
   if(m_potun_MsgToADR == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgToADR != NULL);
      return;
   }
   /*puchData[0] = gives length of the array*/
   unsigned char u8Messagelen = puchData[0];

   if( u8Messagelen >= (tU8)enAdrMsgOffset_DATA_LEN)
   {
      //ETG_TRACE_USR1(( " HandleTTFISCommands::vHandle_TUN_MSG_TO_ADR_Trace() -> TTFis command: TUN_MSG_TO_ADR." ));
      m_potun_MsgToADR->vMsgToADRFromTrace( (tPU8)&puchData[ 2] ,u8Messagelen - 1);
   }
   else
   {
      ETG_TRACE_USR1(( " HandleTTFISCommands::vHandle_TUN_MSG_TO_ADR_Trace() -> Error: Message is to short Len=%d bytes."
         ,u8Messagelen - 1));
   }
}

/*****************************************************************************
* vHandleContainerCheck
******************************************************************************/
void HandleTTFISCommands::vHandleContainerCheck(unsigned char* puchData)
{
   if(m_potun_MsgToADR == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgToADR != NULL);
      return;
   }
   unsigned char u8Messagelen = puchData[0];
   /** Vnd4kor: Changed index to u8index to avoid lint warnings */
   if( u8Messagelen >= 1)
   {
      //ETG_TRACE_USR1(( " HandleTTFISCommands::vHandleContainerCheck() -> "
      //      "TTFis command: TUN_ATM_CONTAINER. length:%d" , (u8Messagelen-2)   ));
      unsigned char u8DataLength = ( unsigned char)( u8Messagelen-2);
      unsigned char u8Data[ TUN_DRVADRIF_CONF_SSI_AMFM_TUNER_TX_BUFFER_SIZE] = {0};
      for(unsigned char u8index=0;u8index<u8DataLength;u8index++)
      {
         u8Data[u8index] = puchData[u8index+3];
      }
      m_potun_MsgToADR->vSendSetATMContainer( u8DataLength,u8Data );
   }
}

/*****************************************************************************
* vHandleSeekMethod
******************************************************************************/
void HandleTTFISCommands::vHandleSeekMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }

   midw_tunerfi_tclMsgFID_TUN_S_ATSEEK_WITH_TUNERHEADERMethodStart oMsg;
   oMsg.u8TunerSeek.enType = (midw_fi_tcl_e8_Tun_TunerSeek::tenType)puchData[1]; /*puchData+1 = Seek Type*/
   oMsg.u8TunerHeader.enType = (midw_fi_tcl_e8_Tun_TunerHeader::tenType)puchData[2];

   unsigned int u32Frequency;
            u32Frequency  = ( tU32 ) ( ( (tU32)  puchData[3]  ) << 24 ) ;
            u32Frequency |= ( tU32 ) ( ( (tU32)  puchData[4]  ) << 16 ) ;
            u32Frequency |= ( tU32 ) ( ( (tU32)  puchData[5]  ) << 8  ) ;
            u32Frequency |= ( tU32 ) (   (tU32)  puchData[6]  );

   oMsg.u32Frequency = u32Frequency;
//
//   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleSeekMethod() : "
//            "TTFis MSG -> TUN_POST_SELF_MSG_SEEK : u8TunerSeek = %d, u32Frequency=%d",
//            oMsg.u8TunerSeek.enType, u32Frequency));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
                     oMsg,                            /** parameter */
                     MIDW_TUNERFI_C_U16_FID_TUN_S_ATSEEK_WITH_TUNERHEADER, /** FID */
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
                     0,                               /** Command Counter */
                     0                                /** Reg ID*/
                     );
}

/*****************************************************************************
* vHandle_s_Preset_Handling
******************************************************************************/
void HandleTTFISCommands::vHandlePresetHandlingMethod(unsigned char* puchData)
{
   if (m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }

   midw_tunerfi_tclMsgFID_TUN_S_PRESET_HANDLINGMethodStart oMsg;
   oMsg.e8Action.enType = (midw_fi_tcl_e8_Tun_PresetAction::tenType)puchData[1];             // e8Action

   oMsg.u8ListElementID = puchData[2];                                                       // u8ListElementID
   oMsg.e8ListID.enType = (midw_fi_tcl_e8_Tun_ListID::tenType)puchData[3];                   // e8ListID
   unsigned int u32Frequency;
   u32Frequency = (tU32)(((tU32)puchData[4]) << 24);
   u32Frequency |= (tU32)(((tU32)puchData[5]) << 16);
   u32Frequency |= (tU32)(((tU32)puchData[6]) << 8);
   u32Frequency |= (tU32)((tU32)puchData[7]);

   oMsg.u32Frequency = u32Frequency;                                                         // u32Frequency

   unsigned short u16PIValue = 0;
   u16PIValue |= (tU16)((puchData[8]) << 8);
   u16PIValue = (unsigned char)((u16PIValue) | ((tU16)(puchData[9])));
   oMsg.u16PI = u16PIValue;                                                                  // u16PI

   unsigned short u16SecondaryPIValue = 0;
   u16SecondaryPIValue |= (tU16)((puchData[10]) << 8);
   u16SecondaryPIValue = (unsigned char)((u16SecondaryPIValue) | ((tU16)(puchData[11])));
   oMsg.u16SecondaryPI = u16SecondaryPIValue;                                                // u16SecondaryPI

      ETG_TRACE_USR4((" HandleTTFISCommands::vHandlePresetHandlingMethod() : "
               "TTFis MSG -> FID_TUN_S_PRESET_HANDLING : e8Action = %d, u8ListElementID = %d, e8ListID = %d, u32Frequency=%d",
               oMsg.e8Action.enType, oMsg.u8ListElementID, oMsg.e8ListID.enType,  u32Frequency));

   m_potun_MsgFromHMI->vSendMessage(CCA_C_U16_APP_TUNER,  /** Destination APP ID */
      oMsg,                            /** parameter */
      MIDW_TUNERFI_C_U16_FID_TUN_S_PRESET_HANDLING, /** FID */
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
      0,                               /** Command Counter */
      0                                /** Reg ID*/
   );
}
/*****************************************************************************
* vHandle_s_Atfrequency
******************************************************************************/
void HandleTTFISCommands::vHandleAtfrequencyMethod(unsigned char* puchData)
{
   if (m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }

   midw_tunerfi_tclMsgFID_TUN_S_ATFREQUENCYMethodStart oMsg;

   oMsg.TunerHeader.enType = (midw_fi_tcl_e8_Tun_TunerHeader::tenType)puchData[1];                                         // TunerHeader
   oMsg.TunerManualSingleStep.ManualFreqAdjustment.enType = (midw_fi_tcl_e8_Tun_ManualFreqAdjustment::tenType)puchData[2]; // ManualFreqAdjustment
   oMsg.TunerManualSingleStep.u8NumberOfTicks = puchData[3];                                                               // NumberOfTicks
   oMsg.u8AcceptAF.enType = (midw_fi_tcl_e8_Tun_AcceptAF::tenType)puchData[4];                                             // AcceptAF

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleAtfrequencyMethod() : "
      "TTFis MSG -> FID_TUN_S_ATFREQUENCY : TunerHeader = %d, ManualFreqAdjustment = %d, NumberOfTicks = %d, AcceptAF = %d",
      oMsg.TunerHeader.enType , oMsg.TunerManualSingleStep.ManualFreqAdjustment.enType
      , oMsg.TunerManualSingleStep.u8NumberOfTicks ,oMsg.u8AcceptAF.enType));

   m_potun_MsgFromHMI->vSendMessage(CCA_C_U16_APP_TUNER,  /** Destination APP ID */
      oMsg,                            /** parameter */
      MIDW_TUNERFI_C_U16_FID_TUN_S_ATFREQUENCY, /** FID */
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
      0,                               /** Command Counter */
      0                                /** Reg ID*/
   );
}

/*****************************************************************************
* vHandle_s_ChangeLSM
******************************************************************************/
void HandleTTFISCommands::vHandleChangeLSMMethod(unsigned char* puchData)
{
   if (m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_CHANGE_LSMMethodStart oMsg;
   oMsg.Option.enType = (midw_fi_tcl_e8_StationSelectionOption::tenType)puchData[1];
   oMsg.Band.enType = (midw_fi_tcl_e8_Tun_TunerBand::tenType)puchData[2];

   unsigned int u32Frequency;
   u32Frequency = (tU32)(((tU32)puchData[3]) << 24);
   u32Frequency |= (tU32)(((tU32)puchData[4]) << 16);
   u32Frequency |= (tU32)(((tU32)puchData[5]) << 8);
   u32Frequency |= (tU32)((tU32)puchData[6]);

   oMsg.Frequency = u32Frequency;                                                         // u32Frequency
   oMsg.u8Element_ID = (tU8)puchData[7];
   oMsg.e8ListID.enType = (midw_fi_tcl_e8_Tun_ListID::tenType)puchData[8];
   unsigned short u8PI = 0;
   u8PI |= (tU16)((puchData[9]) << 8);
   u8PI = (unsigned char)((u8PI) | ((tU16)(puchData[10])));
   oMsg.PI = u8PI;
   oMsg.HDProgramNumber = (tU8)puchData[11];

      ETG_TRACE_USR4((" HandleTTFISCommands::vHandleChangeLSMMethod() : "
         "TTFis MSG -> FID_TUN_S_CHANGE_LSM : Option = %d, Band = %d, Frequency = %d, u8Element_ID = %d\
         , e8ListID = %d, PI = %d, HDProgramNumber = %d"
         , oMsg.Option.enType, oMsg.Band.enType, oMsg.Frequency, oMsg.u8Element_ID, oMsg.e8ListID.enType, oMsg.PI, oMsg.HDProgramNumber));

   m_potun_MsgFromHMI->vSendMessage(CCA_C_U16_APP_TUNER,  /** Destination APP ID */
      oMsg,                            /** parameter */
      MIDW_TUNERFI_C_U16_FID_TUN_S_CHANGE_LSM, /** FID */
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
      0,                               /** Command Counter */
      0                                /** Reg ID*/
   );
}

/*****************************************************************************
* vHandle_s_Get_Static_List
******************************************************************************/
void HandleTTFISCommands::vHandleGetStaticListMethod(unsigned char* puchData)
{
   if (m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }

   midw_tunerfi_tclMsgFID_TUN_S_GET_STATIC_LISTMethodStart oMsg;

   oMsg.e8ListID.enType = (midw_fi_tcl_e8_Tun_ListID::tenType)puchData[1];

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleGetStaticListMethod() : "
      "TTFis MSG -> FID_TUN_S_GET_STATIC_LIST : e8ListID = %d ", oMsg.e8ListID.enType));

   m_potun_MsgFromHMI->vSendMessage(CCA_C_U16_APP_TUNER,  /** Destination APP ID */
      oMsg,                            /** parameter */
      MIDW_TUNERFI_C_U16_FID_TUN_S_GET_STATIC_LIST, /** FID */
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
      0,                               /** Command Counter */
      0                                /** Reg ID*/
   );
}
/*****************************************************************************
* vHandle_s_Announcement_Escape
******************************************************************************/
void HandleTTFISCommands::vHandleAnnouncementEscapeMethod(unsigned char* puchData)
{
   if (m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }

   midw_tunerfi_tclMsgFID_TUN_S_ANNOUNCEMENT_ESCAPEMethodStart oMsg;
   oMsg.e8AnnoType.enType = (midw_fi_tcl_e8_Tun_AnnoTypes::tenType)puchData[1];             // e8AnnoType

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleAnnouncementEscapeMethod() : "
      "TTFis MSG -> FID_TUN_S_ANNOUNCEMENT_ESCAPE : e8AnnoType = %d",
      oMsg.e8AnnoType.enType));

   m_potun_MsgFromHMI->vSendMessage(CCA_C_U16_APP_TUNER,  /** Destination APP ID */
      oMsg,                            /** parameter */
      MIDW_TUNERFI_C_U16_FID_TUN_S_ANNOUNCEMENT_ESCAPE, /** FID */
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
      0,                               /** Command Counter */
      0                                /** Reg ID*/
   );
}

/*****************************************************************************
* vHandle_s_Announcement_Switch
******************************************************************************/
void HandleTTFISCommands::vHandleAnnouncementSwitchMethod(unsigned char* puchData)
{
   if (m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }

   midw_tunerfi_tclMsgFID_TUN_S_ANNOUNCEMENT_SWITCHMethodStart oMsg;
   oMsg.b8AnnoType.u8Value = (tU8)puchData[1];                                    // b8AnnoType

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleAnnouncementSwitchMethod() : "
      "TTFis MSG -> FID_TUN_S_ANNOUNCEMENT_SWITCH : b8AnnoType = %d",
      oMsg.b8AnnoType.u8Value));

   m_potun_MsgFromHMI->vSendMessage(CCA_C_U16_APP_TUNER,  /** Destination APP ID */
      oMsg,                            /** parameter */
      MIDW_TUNERFI_C_U16_FID_TUN_S_ANNOUNCEMENT_SWITCH, /** FID */
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
      0,                               /** Command Counter */
      0                                /** Reg ID*/
   );
}

/*****************************************************************************
* vHandleSensorPositionProperty
******************************************************************************/

void HandleTTFISCommands::vHandleSensorPositionProperty(unsigned char* puchData)
{
	if(m_poclTmcTuner_stationLogo_map == NULL)
   {
      NORMAL_M_ASSERT(m_poclTmcTuner_stationLogo_map != NULL);
      return;
   }
   unsigned int u32Latitude;
   unsigned int u32Longitude;
   
   u32Latitude  = ( tU32 ) ( ( (tU32)  puchData[1]  ) << 24 ) ;
   u32Latitude |= ( tU32 ) ( ( (tU32)  puchData[2]  ) << 16 ) ;
   u32Latitude |= ( tU32 ) ( ( (tU32)  puchData[3]  ) << 8  ) ;
   u32Latitude |= ( tU32 ) (   (tU32)  puchData[4]  );
   
   u32Longitude  = ( tU32 ) ( ( (tU32)  puchData[5]  ) << 24 ) ;
   u32Longitude |= ( tU32 ) ( ( (tU32)  puchData[6]  ) << 16 ) ;
   u32Longitude |= ( tU32 ) ( ( (tU32)  puchData[7]  ) << 8  ) ;
   u32Longitude |= ( tU32 ) (   (tU32)  puchData[8]  );
   
   double dLatitude = (double)(u32Latitude) / 1000000.0;
   double dLongitude = (double)(u32Longitude) / 1000000.0;
   
   m_poclTmcTuner_stationLogo_map->vOnSensorStatus(dLatitude, dLongitude);
}

/*****************************************************************************
* vHandleHicutMethod
******************************************************************************/
void HandleTTFISCommands::vHandleHicutMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_SET_HICUTMethodStart oMsg;
   oMsg.u8AbsVal = puchData[1];

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleHicutMethod() : "
   "TTFis MSG -> TUN_CMD_FEATURE_HIGHCUT : "));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,   /** Destination APP ID */
          oMsg,                            /** parameter */
          MIDW_TUNERFI_C_U16_FID_TUN_S_SET_HICUT, /** FID */
          AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
          0,                               /** Command Counter */
          0                                /** Reg ID*/
    );
}

/*****************************************************************************
* vHandleSharxMethod
******************************************************************************/
void HandleTTFISCommands::vHandleSharxMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_SET_SHARXLEVELMethodStart oMsg;
   oMsg.u8SharxStep = puchData[1];

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleSharxMethod() : "
   "TTFis MSG -> TUN_CMD_FEATURE_HIGHCUT : "));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,   /** Destination APP ID */
          oMsg,                            /** parameter */
          MIDW_TUNERFI_C_U16_FID_TUN_S_SET_SHARXLEVEL, /** FID */
          AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
          0,                               /** Command Counter */
          0                                /** Reg ID*/
    );
}

/*****************************************************************************
* vHandleSelecetListElement
******************************************************************************/
void HandleTTFISCommands::vHandleSelecetListElement(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }

   midw_tunerfi_tclMsgFID_TUN_S_SEL_LIST_ELEMMethodStart oMsg;
   oMsg.Sel_List_Elem.eConfigListID.enType = (midw_fi_tcl_e8_Tun_ConfigListID::tenType)puchData[1];

   unsigned int u32FrequencyValue = 0;
   u32FrequencyValue  = ( (  puchData[2]  ) << 24 ) ;
   u32FrequencyValue |= ( (  puchData[3]  ) << 16 ) ;
   u32FrequencyValue |= ( (  puchData[4]  ) << 8  ) ;
   u32FrequencyValue |= (    puchData[5]  );
   oMsg.Sel_List_Elem.u32Frequency = u32FrequencyValue;

   unsigned short u16PICodeValue = 0;
   u16PICodeValue |= ( tU16 ) ( (  puchData[6]  ) << 8  ) ;
   u16PICodeValue = ( unsigned char)((u16PICodeValue) |(( tU16 ) ( puchData[7]  )));
   oMsg.Sel_List_Elem.u16PICode = u16PICodeValue;

   oMsg.Sel_List_Elem.u8ListElementID = puchData[8];

   unsigned int u32HDStnIDDRMSrvID = 0;
   u32HDStnIDDRMSrvID  = ( (  puchData[9]  ) << 24 ) ;
   u32HDStnIDDRMSrvID |= ( (  puchData[10]  ) << 16 ) ;
   u32HDStnIDDRMSrvID |= ( (  puchData[11]  ) << 8  ) ;
   u32HDStnIDDRMSrvID |= (    puchData[12]  );
   oMsg.Sel_List_Elem.u32HDStnID_DRMSrvID = u32HDStnIDDRMSrvID;

   oMsg.Sel_List_Elem.e8ModulationType.enType = (midw_fi_tcl_e8_Tun_ModulationType::tenType)puchData[13];

//   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleSelecetListElement() : "
//   "TTFis MSG -> TUN_POST_SELF_MSG_SEL_LIST_ELEM : "));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,   /** Destination APP ID */
          oMsg,                            /** parameter */
          MIDW_TUNERFI_C_U16_FID_TUN_S_SEL_LIST_ELEM, /** FID */
          AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
          0,                               /** Command Counter */
          0                                /** Reg ID*/
    );
}

/*****************************************************************************
* vHandleSetEnggParameterMethod
******************************************************************************/
void HandleTTFISCommands::vHandleSetEnggParameterMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }

   /** Self msg to SET_ENGINEERING_PARAMETER */
    midw_tunerfi_tclMsgFID_TUN_S_SET_ENGG_PARAMETERMethodStart oMsg;
   oMsg.e8ParameterID.enType = (midw_fi_tcl_e8_Tun_Engg_Param_ID::tenType)puchData[1];

    unsigned int u32ParameterValue;
    u32ParameterValue  = ( tU32 ) ( ( (tU32)  puchData[2]  ) << 24 ) ;
    u32ParameterValue |= ( tU32 ) ( ( (tU32)  puchData[3]  ) << 16 ) ;
    u32ParameterValue |= ( tU32 ) ( ( (tU32)  puchData[4]  ) << 8  ) ;
    u32ParameterValue |= ( tU32 ) (   (tU32)  puchData[5]  );
    oMsg.u32ParameterValue = u32ParameterValue;

//   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleSetEnggParameterMethod() : "
//   "TTFis MSG -> TUN_POST_SELF_MSG_SET_ENGG_PARAM : "
//   "e8ParameterID = %d,u32ParameterValue = %d ",  oMsg.e8ParameterID.enType, oMsg.u32ParameterValue));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
            oMsg,                            /** parameter */
            MIDW_TUNERFI_C_U16_FID_TUN_S_SET_ENGG_PARAMETER, /** FID */
            AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
            0,                               /** Command Counter */
            0                                /** Reg ID*/
      );
}

/*****************************************************************************
* vHandleSetEnggParameterMethod
******************************************************************************/
void HandleTTFISCommands::vHandleSetEnggParameterGet(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   /** Self msg to GET_ENGINEERING_PARAMETER */
   midw_tunerfi_tclMsgFID_TUN_G_SET_ENGG_PARAMETERGet oMsg;
   oMsg.e8ParameterID.enType = (midw_fi_tcl_e8_Tun_Engg_Param_ID::tenType)puchData[1];

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleSetEnggParameterGet() : "
   "TTFis MSG -> TUN_POST_SELF_MSG_GET_ENGG_PARAM : "
   "e8ParameterID = %d",puchData[1]));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
            oMsg,                            /** parameter */
            MIDW_TUNERFI_C_U16_FID_TUN_G_SET_ENGG_PARAMETER, /** FID */
            AMT_C_U8_CCAMSG_OPCODE_GET,         /** OpCode */
            0,                               /** Command Counter */
            0                                /** Reg ID*/
      );
}

/*****************************************************************************
* vHandleSetActivePresetMethod
******************************************************************************/
void HandleTTFISCommands::vHandleSetActivePresetMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   /** Self msg to CM_ATPresetListSetActive */
   midw_tunerfi_tclMsgFID_TUN_S_SET_ACTIVE_PRESETLISTMethodStart oMsg;
   oMsg.e8ListID.enType = (midw_fi_tcl_e8_Tun_ListID::tenType)puchData[1];

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleSetActivePresetMethod() : "
   "TTFis MSG -> TUN_POST_SELF_MSG_SET_ACTIVE_PRSEST_LIST : "
   "e8ListID = %d",
   puchData[1]));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,                 /** Destination APP ID */
        oMsg,                            /** parameter */
        MIDW_TUNERFI_C_U16_FID_TUN_S_SET_ACTIVE_PRESETLIST, /** FID */
        AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
        0,                               /** Command Counter */
        0                                /** Reg ID*/
         );
}

/*****************************************************************************
* vHandleSpeedInfoMethod
******************************************************************************/
void HandleTTFISCommands::vHandleSpeedInfoMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   /** Self msg to Set CM_SpeedInformation */
   midw_tunerfi_tclMsgFID_TUN_S_SPEED_INFOMethodStart oMsg;
   oMsg.u8Speed = puchData[1];

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleSpeedInfoMethod() : "
   "TTFis MSG -> TUN_POST_SELF_MSG_SET_SPEED_INFO : u8Speed = %d",
   puchData[1]));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,                 /** Destination APP ID */
            oMsg,                            /** parameter */
            MIDW_TUNERFI_C_U16_FID_TUN_S_SPEED_INFO, /** FID */
            AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
            0,                               /** Command Counter */
            0                                /** Reg ID*/
            );
}

/*****************************************************************************
* vHandleAvailBandRangeGet
******************************************************************************/
void HandleTTFISCommands::vHandleAvailBandRangeGet()
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   /** Self msg to Get Available Band Range*/
   midw_tunerfi_tclMsgFID_TUN_G_AVAILABLE_BAND_RANGEGet oMsg;

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleAvailBandRangeGet() : "
   "TTFis MSG -> TUN_POST_SELF_MSG_GET_AVAIL_BAND_RANGE : "));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,                 /** Destination APP ID */
            oMsg,                            /** parameter */
            MIDW_TUNERFI_C_U16_FID_TUN_G_AVAILABLE_BAND_RANGE, /** FID */
            AMT_C_U8_CCAMSG_OPCODE_GET,         /** OpCode */
            0,                               /** Command Counter */
            0                                /** Reg ID*/
      );
}

/*************************************************************************************************
*
* FUNCTION :	HandleTTFISCommands::vHandleDDAGet
*
* DESCRIPTION : Get the status of the DDA switch in TTFIS using TUN_POST_SELF_MSG_DDA command.
*
* PARAMETER :	Void
*
* RETURNVALUE : Void
*
*************************************************************************************************/
void HandleTTFISCommands::vHandleDDAGet()
{
	if (m_potun_MsgFromHMI == NULL)
	{
		NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
		return;
	}
	/** Self msg to Get status response to HMI regarding the DDA*/
	midw_tunerfi_tclMsgFID_TUN_G_SET_DDAGet oMsg;
	ETG_TRACE_USR4((" HandleTTFISCommands::vHandleDDAGet() : "
		"TTFis MSG -> TUN_POST_SELF_MSG_DDA : "));

	m_potun_MsgFromHMI->vSendMessage(CCA_C_U16_APP_TUNER,                 /** Destination APP ID */
		oMsg,                            /** parameter */
		MIDW_TUNERFI_C_U16_FID_TUN_G_SET_DDA, /** FID */
		AMT_C_U8_CCAMSG_OPCODE_GET,         /** OpCode */
		0,                               /** Command Counter */
		0                                /** Reg ID*/
	);
}

/*************************************************************************************************
*
* FUNCTION :	HandleTTFISCommands::vHandleAFGet
*
* DESCRIPTION : Get the status of the AF switch in TTFIS using TUN_POST_SELF_MSG_AF command.
*
* PARAMETER :	Void
*
* RETURNVALUE : Void
*
*************************************************************************************************/
void HandleTTFISCommands::vHandleAFGet()
{
	if (m_potun_MsgFromHMI == NULL)
	{
		NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
		return;
	}
	/** Self msg to Get status response to HMI regarding the DDA*/
	midw_tunerfi_tclMsgFID_TUN_G_SET_AFGet oMsg;

	ETG_TRACE_USR4((" HandleTTFISCommands::vHandleAFGet() : "
		"TTFis MSG -> TUN_POST_SELF_MSG_AF : "));

	m_potun_MsgFromHMI->vSendMessage(CCA_C_U16_APP_TUNER,                 /** Destination APP ID */
		oMsg,                            /** parameter */
		MIDW_TUNERFI_C_U16_FID_TUN_G_SET_AF, /** FID */
		AMT_C_U8_CCAMSG_OPCODE_GET,         /** OpCode */
		0,                               /** Command Counter */
		0                                /** Reg ID*/
	);
}

/*****************************************************************************
* vHandleUpdateConfigList
******************************************************************************/
void HandleTTFISCommands::vHandleUpdateConfigList(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   /** Self msg to Get Available Band Range*/
   midw_tunerfi_tclMsgFID_TUN_S_CONFIGLIST_UPDATEMethodStart oMsg;
   oMsg.e8ConfigListID.enType = (midw_fi_tcl_e8_Tun_ConfigListID::tenType)puchData[1];
   oMsg.bStatus = puchData[2];

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleUpdateConfigList() : "
   "TTFis MSG -> TUN_POST_SELF_MSG_UPDATE_CONFIGLIST : configlistID=%d Status=%d",
   oMsg.e8ConfigListID.enType, oMsg.bStatus));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,                 /** Destination APP ID */
            oMsg,                            /** parameter */
            MIDW_TUNERFI_C_U16_FID_TUN_S_CONFIGLIST_UPDATE, /** FID */
            AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
            0,                               /** Command Counter */
            0                                /** Reg ID*/
      );
}

/*****************************************************************************
* vHandleDefaultSettingsMethod
******************************************************************************/
void HandleTTFISCommands::vHandleDefaultSettingsMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   /** Self msg to Get Available Band Range*/
   midw_tunerfi_tclMsgFID_TUN_S_DEFAULT_SETTINGMethodStart oMsg;
   oMsg.e8DefaultSettingOption.enType = (midw_fi_tcl_e8_Tun_DefaultSettingOption::tenType)puchData[1];
   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleDefaultSettingsMethod() : "
   "TTFis MSG -> TUN_POST_SELF_MSG_DEFAULT_SETTINGS : "));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,                 /** Destination APP ID */
            oMsg,                            /** parameter */
            MIDW_TUNERFI_C_U16_FID_TUN_S_DEFAULT_SETTING, /** FID */
            AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
            0,                               /** Command Counter */
            0                                /** Reg ID*/
      );
}

/*****************************************************************************
* vHandleDemodulationSwitchMethod
******************************************************************************/
void HandleTTFISCommands::vHandleDemodulationSwitchMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_SET_DEMODULATION_SWITCHMethodStart oMsg;
   oMsg.b8DemodulationSwitchMask.u8Value = puchData[1];
   oMsg.b8DemodulationSwitchValue.u8Value = puchData[2];

   
   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleDemodulationSwitchMethod() : "
   "TTFis MSG -> TUN_POST_SELF_MSG_SET_DEMODULATION_SWITCH : "
   "b8DemodulationSwitchMask = %d , b8DemodulationSwitchValue = %d",
   puchData[1],puchData[2]));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
            oMsg,                            /** parameter */
            MIDW_TUNERFI_C_U16_FID_TUN_S_SET_DEMODULATION_SWITCH, /** FID */
            AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
            0,                               /** Command Counter */
            0                                /** Reg ID*/
      );
}

/*****************************************************************************
* vHandleHDAudioPrgmMethod
******************************************************************************/
void HandleTTFISCommands::vHandleHDAudioPrgmMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_SELECT_HD_AUDIOPRGMMethodStart oMsg;
   oMsg.SelectMode.enType= (midw_fi_tcl_e8_Tun_SelectMode::tenType)puchData[1];
   oMsg.e8HDAudioPrg.enType= (midw_fi_tcl_e8_Tun_HDAudioProgram::tenType)puchData[2];
   oMsg.e8Direction.enType = (midw_fi_tcl_e8_Tun_RelativeDirection::tenType)puchData[3];


   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleHDAudioPrgmMethod() : "
   "TTFis MSG -> TUN_POST_SELF_MSG_SELECT_HD_AUDIOPRGM : "
   "SelectMode = %d,e8HDAudioPrg = %d,e8Direction = %d",
   puchData[1],puchData[2], puchData[3]));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
           oMsg,                            /** parameter */
           MIDW_TUNERFI_C_U16_FID_TUN_S_SELECT_HD_AUDIOPRGM, /** FID */
           AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
           0,                               /** Command Counter */
           0                                /** Reg ID*/
     );
}

/*****************************************************************************
* vHandleDRMAudioServiceDirectMethod
******************************************************************************/
void HandleTTFISCommands::vHandleDRMAudioServiceDirectMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   unsigned int u32Frequency;
   u32Frequency  = ( unsigned int ) ( ( puchData[1]  ) << 24 ) ;
   u32Frequency |= ( unsigned int ) ( ( puchData[2]  ) << 16 ) ;
   u32Frequency |= ( unsigned int ) ( ( puchData[3]  ) << 8  ) ;
   u32Frequency |= ( unsigned int) (  puchData[4]  );

   unsigned int u32DRMServiceId;
   u32DRMServiceId  = ( unsigned int ) ( (puchData[5]  ) << 24 ) ;
   u32DRMServiceId |= ( unsigned int ) ( (  puchData[6]  ) << 16 ) ;
   u32DRMServiceId |= ( unsigned int ) ( (  puchData[7]  ) << 8  ) ;
   u32DRMServiceId |= ( unsigned int ) (  puchData[8]  );

   midw_tunerfi_tclMsgFID_TUN_S_SELECT_DRM_AUDIOPRGMMethodStart oMsg;
   oMsg.u32Frequency=u32Frequency;
   oMsg.u32DRMServiceId=u32DRMServiceId;

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
            oMsg,                            /** parameter */
            MIDW_TUNERFI_C_U16_FID_TUN_S_SELECT_DRM_AUDIOPRGM, /** FID */
            AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
            0,                               /** Command Counter */
            0                                /** Reg ID*/
      );
}

/*****************************************************************************
* vHandleEnhancedTestmodeMethod
******************************************************************************/
void HandleTTFISCommands::vHandleEnhancedTestmodeMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_SET_TESTMODEMethodStart oMsg;
   oMsg.e8TestModeSource.enType = (midw_fi_tcl_e8_Tun_TestModeSource::tenType)puchData[1];
   oMsg.e8TestModeActivity.enType = (midw_fi_tcl_e8_Tun_TestModeActivity::tenType)puchData[2];

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleEnhancedTestmodeMethod() : "
            "TTFis MSG -> TUN_POST_SELF_MSG_SET_ENHANCED_TESTMODE : "
            "e8TestModeSource = %d"
            "TestModeActivity = %d", puchData[1],puchData[2]));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
                     oMsg,                            /** parameter */
                     MIDW_TUNERFI_C_U16_FID_TUN_S_SET_TESTMODE, /** FID */
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
                     0,                               /** Command Counter */
                     0                                /** Reg ID*/
               );
}

/*****************************************************************************
* vHandleSelHDAudioPrgmDirectMethod
******************************************************************************/
void HandleTTFISCommands::vHandleSelHDAudioPrgmDirectMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_SEL_HD_AUDIOPRGMDIRECTMethodStart oMsg;
   unsigned int u32Frequency;
   unsigned char u8AudioProgram = puchData[5];

   oMsg.e8HDAudioPrg.enType = (midw_fi_tcl_e8_Tun_HDAudioProgram::tenType)puchData[5];

   u32Frequency  = ( unsigned int ) ( ( puchData[1]  ) << 24 ) ;
   u32Frequency |= ( unsigned int ) ( ( puchData[2]  ) << 16 ) ;
   u32Frequency |= ( unsigned int ) ( ( puchData[3]  ) << 8  ) ;
   u32Frequency |= ( unsigned int ) ( puchData[4]  );

   oMsg.u32Frequency = u32Frequency;

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleSelHDAudioPrgmDirectMethod() : "
           "TTFis MSG -> TUN_POST_SELF_MSG_SEL_HD_AUDIOPRGMDIRECT : "
           "u32Frequency = %d, u8HDAudioProgram = %d",
           oMsg.u32Frequency, u8AudioProgram));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
                    oMsg,                            /** parameter */
                    MIDW_TUNERFI_C_U16_FID_TUN_S_SEL_HD_AUDIOPRGMDIRECT, /** FID */
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
                    0,                               /** Command Counter */
                    0                                /** Reg ID*/
              );
}

/*****************************************************************************
* vHandleDirectFrequencyMethod
******************************************************************************/
void HandleTTFISCommands::vHandleDirectFrequencyMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_DIRECT_FREQUENCYMethodStart oMsg;
   oMsg.u8Band.enType       = (midw_fi_tcl_e8_Tun_TunerBand::tenType)puchData[1];

   unsigned int u32Frequency;
   u32Frequency  = ( unsigned int  ) ( ( puchData[2]  ) << 24 ) ;
   u32Frequency |= ( unsigned int  ) ( ( puchData[3]  ) << 16 ) ;
   u32Frequency |= ( unsigned int  ) ( ( puchData[4]  ) << 8  ) ;
   u32Frequency |= ( unsigned int  ) (   puchData[5]  );
   oMsg.u32Frequency = u32Frequency;

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleDirectFrequencyMethod() : "
                        "TTFis MSG -> TUN_POST_SELF_MSG_SET_DIRECT_FREQUENCY : "
                        "u8Band = %d, u32Frequency = %d", oMsg.u32Frequency, oMsg.u8Band.enType));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
                 oMsg,                            /** parameter */
                 MIDW_TUNERFI_C_U16_FID_TUN_S_DIRECT_FREQUENCY, /** FID */
                 AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
                 0,                               /** Command Counter */
                 0                                /** Reg ID*/
           );
}

/*****************************************************************************
* vHandleSetHDSectionParameterMethod
******************************************************************************/
void HandleTTFISCommands::vHandleSetHDSectionParameterMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   /** Self msg to SET_HD_SECTION_PARAMETER */
   midw_tunerfi_tclMsgFID_TUN_S_SET_HD_SECTION_PARAMETERMethodStart oMsg;

   unsigned short u16Parameter_ID = 0;
   u16Parameter_ID |= ( tU16 ) ( (  puchData[1]  ) << 8  ) ;
   u16Parameter_ID = (unsigned short)(u16Parameter_ID |( tU16 ) (    puchData[2]  ));
   oMsg.e16Parameter_ID.enType = (midw_fi_tcl_e16_Tun_ParameterID::tenType)u16Parameter_ID;

   oMsg.s8ParameterValue = puchData[3];

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleSetHDSectionParameterMethod() : "
   "TTFis MSG -> TUN_POST_SELF_MSG_SET_HD_SECTION_PARAMETER : "
   "Parameter ID = %u, es8ParameterValue = %c ",u16Parameter_ID,
   puchData[3]));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
         oMsg,                            /** parameter */
         MIDW_TUNERFI_C_U16_FID_TUN_S_SET_HD_SECTION_PARAMETER, /** FID */
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
         0,                               /** Command Counter */
         0                                /** Reg ID*/
   );

}

/*****************************************************************************
* vHandleHDResetTransitionCount
******************************************************************************/
void HandleTTFISCommands::vHandleHDResetTransitionCount()
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   /** Self msg to Reset HD Transition count in Testmode. */
   midw_tunerfi_tclMsgFID_TUN_S_HD_RESET_TRANSITIONCOUNTMethodStart oMsg;

//   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleHDResetTransitionCount() : "
//   "TTFis MSG -> TUN_POST_SELF_MSG_HD_RESET_TRANSITIONCOUNT "
//   ));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
            oMsg,                            /** parameter */
            MIDW_TUNERFI_C_U16_FID_TUN_S_HD_RESET_TRANSITIONCOUNT, /** FID */
            AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
            0,                               /** Command Counter */
            0                                /** Reg ID*/
      );

}

/*****************************************************************************
* vHandleUserProfileUpdate
******************************************************************************/
void HandleTTFISCommands::vHandleUserProfileUpdate(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   /** Self msg to Set user */
   midw_tunerfi_tclMsgFID_TUN_S_USER_PROFILE_UPDATEMethodStart oMsg;

//   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleUserProfileUpdate() : "
//   "TTFis MSG -> TUN_POST_SELF_MSG_USER_PROFILE_UPDATE "
//   ));

	oMsg.Action.enType = (midw_fi_tcl_Tun_UserProfileAction::tenType)puchData[1];
	oMsg.UserID = puchData[2];

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
            oMsg,                            /** parameter */
            MIDW_TUNERFI_C_U16_FID_TUN_S_USER_PROFILE_UPDATE, /** FID */
            AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
            0,                               /** Command Counter */
            0                                /** Reg ID*/
      );

}

/*****************************************************************************
* vHandleGetConfigListMethod
******************************************************************************/
void HandleTTFISCommands::vHandleGetConfigListMethod(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
      NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
      return;
   }
   /** Self msg to configure station list. */
   midw_tunerfi_tclMsgFID_TUN_S_GET_CONFIG_LISTMethodStart oMsg;

   ETG_TRACE_USR1(("HandleTTFISCommands::vHandleGetConfigListMethod"
         "TUN_POST_SELF_MSG_GET_CONFIG_LIST"));

   oMsg.TunMakeConfigList.eConfigListID.enType = (midw_fi_tcl_e8_Tun_ConfigListID::tenType)(puchData[1]);
   unsigned int u32Pty= 0;
   u32Pty  = ( unsigned int  ) ( ( puchData[2]  ) << 24 ) ;
   u32Pty |= ( unsigned int  ) ( ( puchData[3]  ) << 16 ) ;
   u32Pty |= ( unsigned int  ) ( ( puchData[4]  ) << 8  ) ;
   u32Pty |= ( unsigned int  ) (   puchData[5]  );
   oMsg.TunMakeConfigList.u32PTY = u32Pty;
   oMsg.TunMakeConfigList.eSortingCriteria.enType = (midw_fi_tcl_e8_Tun_SortingCriteria::tenType)(puchData[6]);
   oMsg.TunMakeConfigList.u8DigitalFilters.u8Value = (puchData[7]);
   oMsg.TunMakeConfigList.bFrozenList = puchData[8];

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
            oMsg,                            /** parameter */
            MIDW_TUNERFI_C_U16_FID_TUN_S_GET_CONFIG_LIST, /** FID */
            AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
            0,                               /** Command Counter */
            0                                /** Reg ID*/
      );
}

/*****************************************************************************
* vHandleADRCommunicationErr
******************************************************************************/
void HandleTTFISCommands::vHandleADRCommunicationErr(unsigned char* puchData)
{
   tU32 u32Req= puchData [ 2 ];
   u32Req=u32Req*100;
   //ETG_TRACE_USR4(("u32Req=%d",u32Req));
   if(u32Req!=100)
   {
#ifdef VARIANT_S_FTR_ENABLE_INC_ADAPTATION_TO_NISSANLCN2KAI
      if(_poTTFISHandler->m_potun_main->m_potun_tclINCComm == NULL)
        {
           NORMAL_M_ASSERT(_poTTFISHandler->m_potun_main->m_potun_tclINCComm != NULL);
           return;
        }
      _poTTFISHandler->m_potun_main->m_potun_tclINCComm->vCloseAdr3Communication();
#endif
   }
   _poTTFISHandler->m_potun_main->vSetRestartTimeValue(u32Req);
}

/*****************************************************************************
* vHandleCommitMethod
******************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
void HandleTTFISCommands::vHandleCommitMethod()
{
   if(_poTTFISHandler->m_poclAars_CM_Commit == NULL)
   {
     NORMAL_M_ASSERT(m_poclAars_CM_Commit != NULL);
     return;
   }
   _poTTFISHandler->m_poclAars_CM_Commit->vSendCommitMsg();

}
#endif


/*****************************************************************************
* vHandleJapanSetArea
******************************************************************************/
void HandleTTFISCommands::vHandleJapanSetArea(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
     NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
     return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_JAPAN_SET_AREAMethodStart oMsg;
   oMsg.e8SelectedRegion.enType = (midw_fi_tcl_e8_Tun_Japan_Region::tenType)puchData[1];
   oMsg.bAreaSelectionMode = tBool (puchData[2]);

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleJapanSetArea() : "
            "TTFis MSG -> TUN_POST_SELF_MSG_JAPAN_SET_AREA : "
            "e8SelectedRegion = %d   bAreaSelectionMode = %d",
            oMsg.e8SelectedRegion.enType, oMsg.bAreaSelectionMode));
   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
                     oMsg,                            /** parameter */
                     MIDW_TUNERFI_C_U16_FID_TUN_S_JAPAN_SET_AREA, /** FID */
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
                     0,                               /** Command Counter */
                     0                                /** Reg ID*/
               );
}
/*****************************************************************************
* vHandleJapanStationNameChange
******************************************************************************/
void HandleTTFISCommands::vHandleJapanStationNameChange(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
     NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
     return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_JAPAN_CHANGE_STATION_NAMEMethodStart oMsg;
   oMsg.e8ChangeStationMode.enType = (midw_fi_tcl_e8_Tun_ChangeStationMode::tenType)puchData[1];

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleJapanStationNameChange() : "
         "TTFis MSG -> TUN_POST_SELF_MSG_JAPAN_CHANGE_STATION_NAME "
         "e8ChangeStationMode = %d", oMsg.e8ChangeStationMode.enType));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
                     oMsg,                            /** parameter */
                     MIDW_TUNERFI_C_U16_FID_TUN_S_JAPAN_CHANGE_STATION_NAME, /** FID */
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
                     0,                               /** Command Counter */
                     0                                /** Reg ID*/
               );
}

/*****************************************************************************
* vHandleLanguageSetting
******************************************************************************/
void HandleTTFISCommands::vHandleLanguageSetting(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
     NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
     return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_LANGUAGE_SETTINGMethodStart oMsg;
   oMsg.e8Language.enType = (midw_fi_tcl_e8_Tun_Language::tenType)puchData[1];

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleLanguageSetting() : "
         "TTFis MSG -> TUN_POST_SELF_MSG_LANGUAGE_SETTING "
         "e8Language = %d", oMsg.e8Language.enType));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
                     oMsg,                            /** parameter */
                     MIDW_TUNERFI_C_U16_FID_TUN_S_LANGUAGE_SETTING, /** FID */
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
                     0,                               /** Command Counter */
                     0                                /** Reg ID*/
               );
}

/*****************************************************************************
* vHandleDRMResetMuteCounter
******************************************************************************/
void HandleTTFISCommands::vHandleDRMResetMuteCounter(unsigned char* /*puchData*/)
{
   if(m_potun_MsgFromHMI == NULL)
   {
     NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
     return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_DRM_RESET_MUTECOUNTERMethodStart oMsg;

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleDRMResetMuteCounter() : "
         "TTFis MSG -> TUN_POST_SELF_MSG_RESET_MUTECOUNTER "));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
                     oMsg,                            /** parameter */
                     MIDW_TUNERFI_C_U16_FID_TUN_S_DRM_RESET_MUTECOUNTER, /** FID */
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
                     0,                               /** Command Counter */
                     0                                /** Reg ID*/
               );
}
/*****************************************************************************
* vHandlePresetStoragePSFreezeMethod
******************************************************************************/
void HandleTTFISCommands::vHandlePresetStoragePSFreezeMethod(unsigned char* /*puchData*/)
{
   if(m_potun_MsgFromHMI == NULL)
   {
     NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
     return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_PRESET_STORAGE_PS_FREEZEMethodStart oMsg;

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandlePresetStoragePSFreezeMethod() : "
         "TTFis MSG -> TUN_POST_SELF_MSG_PRESET_STORAGE_PS_FREEZE "));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
                     oMsg,                            /** parameter */
                     MIDW_TUNERFI_C_U16_FID_TUN_S_PRESET_STORAGE_PS_FREEZE, /** FID */
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
                     0,                               /** Command Counter */
                     0                                /** Reg ID*/
               );
}

/*****************************************************************************
* vHandleJapanSelectStationName
******************************************************************************/
void HandleTTFISCommands::vHandleJapanSelectStationName(unsigned char* puchData)
{
   if(m_potun_MsgFromHMI == NULL)
   {
     NORMAL_M_ASSERT(m_potun_MsgFromHMI != NULL);
     return;
   }
   midw_tunerfi_tclMsgFID_TUN_S_JAPAN_SELECT_STATION_NAMEMethodStart oMsg;
   oMsg.e8SelectMode.enType = (midw_fi_tcl_e8_Tun_SelectMode::tenType)puchData[1];
   oMsg.e8Direction.enType = (midw_fi_tcl_e8_Tun_RelativeDirection::tenType)puchData[2];
	unsigned short u16ElemID = puchData [4];
	u16ElemID = (unsigned short) (u16ElemID |( ((unsigned short) puchData [3]) << 8));
	oMsg.u16ElementID = u16ElemID;

   ETG_TRACE_USR4((" HandleTTFISCommands::vHandleJapanSelectStationName() : "
         "TTFis MSG -> TUN_POST_SELF_MSG_JAPAN_SELECT_STATION_NAME "
         "e8SelectMode = %d e8Direction=%d u16ElementID=%d", oMsg.e8SelectMode.enType, oMsg.e8Direction.enType, oMsg.u16ElementID));

   m_potun_MsgFromHMI->vSendMessage( CCA_C_U16_APP_TUNER,  /** Destination APP ID */
                     oMsg,                            /** parameter */
                     MIDW_TUNERFI_C_U16_FID_TUN_S_JAPAN_SELECT_STATION_NAME, /** FID */
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART,       /** OpCode */
                     0,                               /** Command Counter */
                     0                                /** Reg ID*/
               );
}
