/************************************************************************
 * FILE:        clJapanStationName.cpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of  clJapanStationName.cpp
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   Mar 22, 2016       pmn3kor
				
 *************************************************************************/



#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#include "tun_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_MSGTOHMI
#include "trcGenProj/Header/clJapanStationName.cpp.trc.h"
#endif

#include "datatypes/tMap.h"

#include <iostream>
#include <fstream>
#include <errno.h>
#include <sys/wait.h>
#include <unistd.h>
#include "clJapanStationName.h"
#include "dbus/dbus.h"

#define DYNAMIC_STATION_DB_PATH "/var/opt/bosch/dynamic/tuner/Japan_Station_Name.db"
#define STATIC_STATION_DB_PATH "/var/opt/bosch/static/tuner/Japan_Station_Name.db"
#define RAM_STATION_DB_PATH "/tmp/Japan_Station_Name.db"
#define ENABLE 1
#define DISABLE 0
#define CHANGEMODE_TUN_UP 1
#define CHANGEMODE_TUN_DOWN 0
#define DATABASE_VERSION 9 //increment the number whenever there is change is database entries. New database will get copied only after database version comparison.
#define JAPAN_LANG 1

namespace localdefines
{
   enum
   {
      NO_OF_BYTES_TO_BE_READ = -1,
      INDEX_ELEMENT = 0,
      FIRST_ELEMENT = 1,
      SECOND_ELEMENT,
      THIRD_ELEMENT,
      FOURTH_ELEMENT,
	  FIFTH_ELEMENT
   };
}

typedef struct
{
   unsigned char areacode;
   unsigned int frequency;
   tunerString stationname_japanese;
   tunerString stationname_english;
   bool userselect;
}strJapanStationName;

/*map is needed coz key is unique for each data */
typedef tunerMap<unsigned int, strJapanStationName> mapJapanStationName;
mapJapanStationName mapJpnStnName;
typedef mapJapanStationName::iterator itrJapanStationName;
itrJapanStationName itrJpnStnName;

/******************************************************************************
*
******************************************************************************/
clJapanStationName::clJapanStationName()
{
   ErrMsg = 0;
   u32SQLiteCommandReturnValue = 0;
   m_databaseobj = NULL;
   vSetSelectedLanguage(0); // default lang is english
   sTempStationNameString = "";
   sTemp2StationNameString = "";
   u8JapanStationCount = 0;
}

/******************************************************************************
*
******************************************************************************/
clJapanStationName::~clJapanStationName()
{
   ErrMsg = 0;
   u32SQLiteCommandReturnValue = 0;
   m_databaseobj = NULL;
   vSetSelectedLanguage(0); // default lang is english
   sTempStationNameString = "";
   sTemp2StationNameString = "";
   u8JapanStationCount = 0;
}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vCloseDatabase()
{
   ETG_TRACE_USR4(("vCloseDatabase  "));
   //during shutdown RAM database will be copied back to dynamic folder database. User changes will be saved to only during proper shutdown.
   vDifferentialCopyDatabaseFromRAMToFlash();

   u32SQLiteCommandReturnValue = sqlite3_close(m_databaseobj);
   if(u32SQLiteCommandReturnValue==SQLITE_OK)
   {
	   m_databaseobj = NULL;
   }
   else
   {
	      ETG_TRACE_USR4(("Can't close database: %s", sqlite3_errmsg(m_databaseobj)));
   }
}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vOpenDatabase()
{
	//compare the dynamic folder database version with static folder database version.
	vCompareDatabaseVersion();

	ETG_TRACE_USR4(("vOpenDatabase "));
	u32SQLiteCommandReturnValue = sqlite3_open(RAM_STATION_DB_PATH, &m_databaseobj);

	if (u32SQLiteCommandReturnValue==SQLITE_OK )
	{
		//copy flash memory dynamic folder database to RAM memory (/tmp). All operations will be performed on RAM database copy.
	   vCopyDatabase(DYNAMIC_STATION_DB_PATH, RAM_STATION_DB_PATH);
	}
	else
	{
	   ETG_TRACE_ERR(("Can't open database: %s", sqlite3_errmsg(m_databaseobj)));
	}
}

/******************************************************************************
*
******************************************************************************/
signed int clJapanStationName::s32GetDatabaseVersion(sqlite3* tempobj)
{
	sqlite3_stmt *poStatement = NULL;
	char query1[] = "PRAGMA user_version;";
	//default value :: -1. Coz demo database has no version and by default returns 0. Now database version is mentioned in DATABASE_VERSION define
	signed int s32DynamicDatabaseVersion = -1;

	if( (sqlite3_prepare_v2(tempobj, query1, localdefines::NO_OF_BYTES_TO_BE_READ,  &poStatement,  NULL )) == SQLITE_OK)
	{
		//! Step through the sqlite query. SQLITE_ROW is returned if a row matches the query
		if (SQLITE_ROW == sqlite3_step(poStatement))
		{
			//fetch the version
			s32DynamicDatabaseVersion = sqlite3_column_int(poStatement, localdefines::INDEX_ELEMENT);
			ETG_TRACE_USR4((" s32GetDatabaseVersion() DynamicDatabaseVersion = %d", s32DynamicDatabaseVersion));
		}
	}
	else
	{
		ETG_TRACE_USR4((" s32GetDatabaseVersion() could not prepare statement %s", sqlite3_errmsg(m_databaseobj)));
	}
	//destroy the prepared statement. otherwise will lead to memory leak
    sqlite3_finalize(poStatement);
	return s32DynamicDatabaseVersion;
}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vSetDatabaseVersion(sqlite3* tempobj)
{
	//construct query to set the user version for database
	char *poQuery = sqlite3_mprintf("PRAGMA user_version=%d;", DATABASE_VERSION);

	//Send the query request
	if(NULL != poQuery)
	{
		ETG_TRACE_USR4((" vSetDatabaseVersion %d ", DATABASE_VERSION));
		vExecuteQuery(poQuery, tempobj);
		sqlite3_free(poQuery);
	}

}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vCompareDatabaseVersion()
{
	sqlite3* tempobj;
	/*open the dynamic database and read version data*/
    if (sqlite3_open(DYNAMIC_STATION_DB_PATH, &tempobj)==SQLITE_OK )
    {
    	signed int s32DynamicDatabaseVersion = s32GetDatabaseVersion(tempobj);

        if (DATABASE_VERSION == s32DynamicDatabaseVersion)
        {
        	ETG_TRACE_USR4(("\n**********************************************************************************************\n"
        	            " Existing version on target u32DynamicDatabaseVersion = %d | Release version of database DATABASE_VERSION = %d\n"
        	            "*************************************************************************************************************\n"
        	            "No change in database version.. No need to copy from static to dynamic folder\n"
        	            "************************************************************************************************************",
						s32DynamicDatabaseVersion, DATABASE_VERSION));
        }
        else
        {
        	ETG_TRACE_USR4(("\n**********************************************************************************************\n"
    					" Existing version on target u32DynamicDatabaseVersion = %d | Release version of database DATABASE_VERSION = %d\n"
    					"*************************************************************************************************************\n"
    					"database version changed.. Copy new version from static folder\n"
    					"************************************************************************************************************",
						s32DynamicDatabaseVersion, DATABASE_VERSION));

        	//close dynamic database and rsync the file with new version of database available in static folder.
			if(sqlite3_close(tempobj)!=SQLITE_OK)
			{
				   ETG_TRACE_USR4(("vCompareDatabaseVersion Can't close database: %s", sqlite3_errmsg(m_databaseobj)));
			}
			tempobj = NULL;

        	//rsync the dynamic database copy. Copy default database from static memory which is available from SW update
        	vHandleDynamicDatabase();
        }
    }
	else
	{
	   ETG_TRACE_ERR(("vCompareDatabaseVersion Can't open database: %s", sqlite3_errmsg(tempobj)));
	}
}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vHandleDynamicDatabase()
{
	if(bFileExists(STATIC_STATION_DB_PATH))
	{
		if(bDifferentialCopyDatabaseFromStaticToDynamic()) // no error
		{
			sqlite3* tempobj;
			ETG_TRACE_USR4(("vHandleDynamicDatabase File rsync successful"));
			//open newly copied database and set version
			if (sqlite3_open(DYNAMIC_STATION_DB_PATH, &tempobj) == SQLITE_OK )
			{
				//after rsync the new version of database to from static to dynamic, set the database version for the same
				vSetDatabaseVersion(tempobj);
				//close dynamic database and delete the file
				if(sqlite3_close(tempobj) != SQLITE_OK)
				{
					   ETG_TRACE_USR4(("vHandleDynamicDatabase Can't close database: %s", sqlite3_errmsg(tempobj)));
				}
			}
			else
			{
				ETG_TRACE_ERR(("vHandleDynamicDatabase Can't open database: %s", sqlite3_errmsg(tempobj)));
			}
		}
		else
		{
			ETG_TRACE_ERR(("vHandleDynamicDatabase Error during file rsync copy. "));
		}
	}


}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vExecuteQuery(tunerString sQuery, sqlite3* tempobj)
{
	if(tempobj != NULL)
	{
		   u32SQLiteCommandReturnValue = sqlite3_exec(tempobj, sQuery.c_str(), NULL, NULL, &ErrMsg);
		   if( (u32SQLiteCommandReturnValue!=SQLITE_OK) && (ErrMsg != NULL))
		   {
			   ETG_TRACE_USR4(("vExecuteQuery::SQL error %s", ErrMsg));
		   }
	}

}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vExecuteQueryWithCallbackFunction(tunerString sQuery)
{
   //execute command with callback function to print retrived data for debugging
   u32SQLiteCommandReturnValue =  sqlite3_exec(m_databaseobj, sQuery.c_str(), Callback_SQLiteQuery, NULL, &ErrMsg);
}

/******************************************************************************
*
******************************************************************************/
int clJapanStationName::Callback_SQLiteQuery(void */*data*/, int argc, char **argv, char **/*column*/)
{
   ETG_TRACE_USR4(("vCallback_SQLiteQuery: argc=%d", argc));
   for(int i=0; i<argc; i++)
   {
      //print the sqlite command result
      ETG_TRACE_USR4(("%s",  (argv[i] ? argv[i] : "NULL")));
   }
   return 0;
}

/******************************************************************************
*
******************************************************************************/
unsigned char clJapanStationName::GetJapanStationCount(unsigned char u8AreaCode, unsigned int u32Freq)
{
   std::lock_guard<std::mutex> lock(m_JapanMapLock);
   sqlite3_stmt *poStatement = NULL;
   char query1[] = "SELECT * FROM JapanStationName WHERE frequency=? AND area_code=?;";

   if( (sqlite3_prepare_v2(m_databaseobj, query1, localdefines::NO_OF_BYTES_TO_BE_READ,  &poStatement,  NULL )) == SQLITE_OK)
   {
      mapJpnStnName.clear();
      ETG_TRACE_USR4(("the prepared statement has %d wildcards", sqlite3_bind_parameter_count(poStatement)));
      sqlite3_bind_int(poStatement, localdefines::FIRST_ELEMENT, u32Freq);
      sqlite3_bind_int(poStatement, localdefines::SECOND_ELEMENT, u8AreaCode);

      /*If the SQL statement being executed returns any data, then SQLITE_ROW is returned each time
       * a new row of data is ready for processing by the caller. After finishing the execution SQLITE_DONE will be returned.*/
      for(int index=0; SQLITE_ROW == sqlite3_step(poStatement); index++)
      {
         strJapanStationName rJpnStnName;
         rJpnStnName.areacode = (unsigned char)sqlite3_column_int(poStatement, localdefines::FIRST_ELEMENT);
         rJpnStnName.frequency = sqlite3_column_int(poStatement, localdefines::SECOND_ELEMENT);
         const char* tempString = (const char*)sqlite3_column_blob(poStatement, localdefines::THIRD_ELEMENT);
         //pack frequency as station names for the entries which doesn't have station name in excelsheet
         rJpnStnName.stationname_japanese = (tempString == NULL)? (std::to_string(rJpnStnName.frequency)): tempString;
         tempString = NULL;
         tempString = (const char*)sqlite3_column_blob(poStatement, localdefines::FOURTH_ELEMENT);
         rJpnStnName.stationname_english = (tempString == NULL)? (std::to_string(rJpnStnName.frequency)): tempString;
         rJpnStnName.userselect = sqlite3_column_int(poStatement, localdefines::FIFTH_ELEMENT);

         mapJpnStnName.insert(std::pair<unsigned int, strJapanStationName>(sqlite3_column_int(poStatement, localdefines::INDEX_ELEMENT), rJpnStnName));
      }
      ETG_TRACE_USR4(("GetJapanStationCount:: contains %d elements", (int)mapJpnStnName.size()));
      vPrintJapanStationsForGivenFrequency();
   }
   else
   {
      ETG_TRACE_USR4(("could not prepare the statement = %s", sqlite3_errmsg(m_databaseobj)));
   }
   //destroy the prepared statement. otherwise will lead to memory leak
   sqlite3_finalize(poStatement);
   u8JapanStationCount = (unsigned char)mapJpnStnName.size();
   return u8JapanStationCount;
}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vPrintJapanStationsForGivenFrequency()
{
   //do not add lock here. map lock is called in other functions where this sub func is called
   if(!mapJpnStnName.empty())
	{
		for(itrJpnStnName = mapJpnStnName.begin(); itrJpnStnName!=mapJpnStnName.end(); ++itrJpnStnName)
		{
			ETG_TRACE_USR4((" station_index= %d | area_code = %d | user_select = %d | freq = %d | stationname_japanese = %s ",
					itrJpnStnName->first, itrJpnStnName->second.areacode, itrJpnStnName->second.userselect,
					itrJpnStnName->second.frequency, itrJpnStnName->second.stationname_japanese.c_str()));
			ETG_TRACE_USR4(("stationname_english = %s", itrJpnStnName->second.stationname_english.c_str()));
		}
	}
   else
   {
      ETG_TRACE_USR4(("No station available for current frequency and selected area code"));
   }
}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vSendStationName(unsigned char u8ChangeStationMode)
{
	ETG_TRACE_USR4(("vSendStationName u8ChangeStationMode = %d", u8ChangeStationMode));

	std::lock_guard<std::mutex> lock(m_JapanMapLock);
	for(itrJpnStnName = mapJpnStnName.begin(); itrJpnStnName!=mapJpnStnName.end(); ++itrJpnStnName)
   {
      if(itrJpnStnName->second.userselect)
      {
         itrJpnStnName->second.userselect = DISABLE;
         //update database with new user select value for current index
         vUpdateMapDataToDatabase(itrJpnStnName->first, itrJpnStnName->second.userselect);

         if(bIsChangeStationNameUpward(u8ChangeStationMode))
         {
         	 ++itrJpnStnName;
             if(itrJpnStnName==mapJpnStnName.end())
             {
                itrJpnStnName =  mapJpnStnName.begin();
             }
         }
         else if(bIsChangeStationNameDownward(u8ChangeStationMode))
         {
        	 if(itrJpnStnName == mapJpnStnName.begin())
        	 {
        		 //access the last element of the map
        		 itrJpnStnName=(--mapJpnStnName.end());
        	 }
        	 else
        	 {
        		 --itrJpnStnName;
        	 }
         }
         else
         {
         	ETG_TRACE_USR4(("vSendStationName:: Not valid ChangeStationMode"));
         }

         itrJpnStnName->second.userselect = ENABLE;
         //update database with new user select value for current index
         vUpdateMapDataToDatabase(itrJpnStnName->first, itrJpnStnName->second.userselect);
         break;
      }
   }
   vPrintJapanStationsForGivenFrequency();
}

/******************************************************************************
*
******************************************************************************/
bool clJapanStationName::bIsStationNamePresent(unsigned short u16Position)
{
   ETG_TRACE_USR4(("bIsStationNamePresent u16Position = %d", u16Position));
   std::lock_guard<std::mutex> lock(m_JapanMapLock);
   //search if the requested element exists in map or not
   itrJpnStnName = mapJpnStnName.find(u16Position);
   if(itrJpnStnName == mapJpnStnName.end()) // element not available in MAP
   {
   	ETG_TRACE_USR4(("Requested position not available in station name list"));
   	return false;
   }
   return true;
}

/******************************************************************************
*
******************************************************************************/
bool clJapanStationName::bSelectStationName(unsigned short u16Position)
{
   ETG_TRACE_USR4(("bSelectStationName u16Position = %d", u16Position));

   std::lock_guard<std::mutex> lock(m_JapanMapLock);
   for(itrJpnStnName = mapJpnStnName.begin(); itrJpnStnName!=mapJpnStnName.end(); ++itrJpnStnName)
   {
   	//do not change the evaluation sequence.
      if((itrJpnStnName->second.userselect) && (u16Position == itrJpnStnName->first))
      {
         //current selected and newly requested position are same. no action needed
      	ETG_TRACE_USR4(("Requested and current position both are same"));
      	return false;
      }
      //update user select value of the current selected station name to 0. (no more selected)
      if(itrJpnStnName->second.userselect)
      {
      	itrJpnStnName->second.userselect = DISABLE;
      	//update database with new user select value for current index
      	vUpdateMapDataToDatabase(itrJpnStnName->first, itrJpnStnName->second.userselect);
      }
      //update user select value of requested station name to 1. (newly selected)
      if(itrJpnStnName->first == u16Position)
		{
			itrJpnStnName->second.userselect = ENABLE;
			//update database with new user select value for current index
			vUpdateMapDataToDatabase(itrJpnStnName->first, itrJpnStnName->second.userselect);
		}
   }
   vPrintJapanStationsForGivenFrequency();
   return true;
}


/******************************************************************************
*
******************************************************************************/
const char* clJapanStationName::sGetStationNameFromDB()
{
   std::lock_guard<std::mutex> lock(m_JapanMapLock);
   for(itrJpnStnName = mapJpnStnName.begin(); itrJpnStnName!=mapJpnStnName.end(); ++itrJpnStnName)
   {
      //return the station name whose user select value is 1
      if(itrJpnStnName->second.userselect )
      {
          //re-initialize temporary string to NULL
          sTempStationNameString.erase();
    	  // check if language is Japanese (FID_TUN_S_LANGUAGE_SETTING from HMI)
    	  if(bCheckIfJapaneseLanguage())
    	  {
    	     sTempStationNameString = itrJpnStnName->second.stationname_japanese;
    	     if(bCheckIfValidUTF8(sTempStationNameString.c_str()))
        	  {
    	        return (sTempStationNameString.c_str());
        	  }
    	  }
    	  else //extract english name otherwise
    	  {
    	     sTempStationNameString = itrJpnStnName->second.stationname_english;
    	     if(bCheckIfValidUTF8(sTempStationNameString.c_str()))
        	  {
        	     return (sTempStationNameString.c_str());
        	  }
    	  }
      }
   }
   return NULL;
}


/******************************************************************************
*
******************************************************************************/
const char* clJapanStationName::sGetStationNameForUpdList(unsigned int u32Freq, unsigned char u8AreaCode)
{
   sqlite3_stmt *poStatement = NULL;
   sTemp2StationNameString.erase();

   char query1[] = "SELECT * FROM JapanStationName WHERE frequency=? AND area_code=? and user_select=?;";
   if( (sqlite3_prepare_v2(m_databaseobj, query1, localdefines::NO_OF_BYTES_TO_BE_READ,  &poStatement,  NULL )) == SQLITE_OK)
   {
      ETG_TRACE_USR4(("the prepared statement has %d wildcards", sqlite3_bind_parameter_count(poStatement)));
		sqlite3_bind_int(poStatement, localdefines::FIRST_ELEMENT, u32Freq);
		sqlite3_bind_int(poStatement, localdefines::SECOND_ELEMENT, u8AreaCode);
		sqlite3_bind_int(poStatement, localdefines::THIRD_ELEMENT, 1); // user select value is enabled.
		int rc = sqlite3_step(poStatement);// return code
		if(SQLITE_ROW == rc)
		{
		   if(bCheckIfJapaneseLanguage())
		   {
		      //if Lang set is Japanese, return japanese name from database
		      sTemp2StationNameString = (const char*)sqlite3_column_blob(poStatement, localdefines::THIRD_ELEMENT);
		   }
		   else
		   {
		      //otherwise return english name from database
		      sTemp2StationNameString = (const char*)sqlite3_column_blob(poStatement, localdefines::FOURTH_ELEMENT);
		   }
		   if(! (bCheckIfValidUTF8(sTemp2StationNameString.c_str())))
		   {
		      ETG_TRACE_USR4(("invalid station name for %d freq and %d area code", u32Freq, u8AreaCode));
		      //set station name to NULL due to invalid station name
		      sTemp2StationNameString.erase();
		   }
		}
		else if(rc == SQLITE_DONE)
		{
			ETG_TRACE_USR4(("No station name available for current freq=%d areacode=%d ", u32Freq, u8AreaCode ));
		}
		else
		{
			ETG_TRACE_USR4(("error while retrieving data %s", sqlite3_errmsg(m_databaseobj)));
		}
   }
   else
   {
	   ETG_TRACE_USR4(("could not prepare the statement = %s", sqlite3_errmsg(m_databaseobj)));
   }
   //destroy the prepared statement. otherwise will lead to memory leak
   sqlite3_finalize(poStatement);
   return sTemp2StationNameString.c_str();

}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vUpdateMapDataToDatabase(unsigned int u8Index, bool bUserSelect)
{
   char query1[] = "UPDATE JapanStationName SET user_select=? where station_index=?;";
   sqlite3_stmt *poStatement = NULL;

   //prepare the statement so that we can bind the required data in later stage
   if( (sqlite3_prepare_v2(m_databaseobj, query1, localdefines::NO_OF_BYTES_TO_BE_READ,  &poStatement,  NULL )) == SQLITE_OK)
   {
      ETG_TRACE_USR4(("vUpdateMapDataToDatabase: database update  for station_index:%d user_select=%d",
            u8Index, bUserSelect));

      //the index value starts from 1 and not 0. Bind can return  SQLITE_BUSY, SQLITE_DONE, SQLITE_ROW, SQLITE_ERROR, or SQLITE_MISUSE as result.
      sqlite3_bind_int(poStatement, localdefines::FIRST_ELEMENT, bUserSelect);
      sqlite3_bind_int(poStatement, localdefines::SECOND_ELEMENT, u8Index);

      //if sqlite3_step has finished the execution then we get result as SQLITE_DONE. (there is no return data)
      if(SQLITE_DONE != sqlite3_step(poStatement))
      {
         ETG_TRACE_USR4(("vUpdateMapDataToDatabase: database update was not successful for error:%s", sqlite3_errmsg(m_databaseobj)));
      }
   }
   else
   {
      ETG_TRACE_USR4(("vUpdateMapDataToDatabase : could not prepare the statement = %s", sqlite3_errmsg(m_databaseobj)));
   }
   //destroy the prepared statement. otherwise will lead to memory leak
   sqlite3_finalize(poStatement);
}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vCopyDatabase(const char* poSrcFileName, const char* poDestFileName)
{
   if(bFileExists(poSrcFileName))
   {
      //open source memory database (as input) to copy it to destination folder.
      std::ifstream in;
       in.open(poSrcFileName, std::ios::in);

       //Destination memory path for copying the database.
       std::ofstream out;
       out.open(poDestFileName, std::ios::out);

       char ch;
       while(in.get(ch))
       {
          //copy contents of Source database to Destination database.
          out.put(ch);
       }
       in.close();
       out.close();
   }
   else
   {
	   ETG_TRACE_ERR((  "vCopyDatabase :: File not copied" ));
   }

   ETG_TRACE_USR4((  "vCopyDatabase ::  Source file = %s", poSrcFileName));
   ETG_TRACE_USR4((  "vCopyDatabase ::  Destination file = %s", poDestFileName));
}

/******************************************************************************
*
******************************************************************************/
bool clJapanStationName::bFileExists(const char* path)
{
   FILE * fp;
   fp = fopen(path, "r");

   if(fp == NULL)
   {
      ETG_TRACE_USR4((  "bFileExists :: %s doesnt exist ",path));
      return false;
   }
   else
   {
      fclose(fp);
      return true;
   }
}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vDifferentialCopyDatabaseFromRAMToFlash()
{
   ETG_TRACE_USR4(("vDifferentialCopyDatabaseFromRAMToFlash : start the copy"));
   
   	char  *copy_argv[10];
	copy_argv[0] = (char *)"rsync";
	copy_argv[1] = (char *)"--inplace";
	copy_argv[2] = (char *)"--no-whole-file";
	copy_argv[3] = (char *)"-c";
	copy_argv[4] = (char *)"-B=4096";
	copy_argv[5] = (char *)"/tmp/Japan_Station_Name.db";
	copy_argv[6] = (char *)"/var/opt/bosch/dynamic/tuner/Japan_Station_Name.db";
	copy_argv[7] = NULL;
	
	pid_t  pid;
	int   status;
	
	int u8Result = 0;
	pid = fork();
	
	if (pid < 0) 
	{     /* fork a child process */
		  ETG_TRACE_ERR(("vDifferentialCopyDatabaseFromRAMToFlash ERROR: forking child process failed"));
	}
	else if (pid == 0) 
	{ /* child process execution*/
	  u8Result = execvp(*copy_argv, copy_argv);
	  if (u8Result < 0) 
	  {   
		  ETG_TRACE_ERR(("execvp failed"));
		  exit(1);
	  }
	}
	else
	{
		while (wait(&status) != pid) // parent process wait for child process to complete
			;
	}
	ETG_TRACE_USR4(("vDifferentialCopyDatabaseFromRAMToFlash Result : %d",u8Result));
}

/******************************************************************************
*
******************************************************************************/
bool clJapanStationName::bDifferentialCopyDatabaseFromStaticToDynamic()
{
   ETG_TRACE_USR4(("bDifferentialCopyDatabaseFromStaticToDynamic : start the copy"));

	char  *copy_argv[10];
	copy_argv[0] = (char *)"rsync";
	copy_argv[1] = (char *)"--inplace";
	copy_argv[2] = (char *)"--no-whole-file";
	copy_argv[3] = (char *)"-c";
	copy_argv[4] = (char *)"-B=4096";
	copy_argv[5] = (char *)"/var/opt/bosch/static/tuner/Japan_Station_Name.db";
	copy_argv[6] = (char *)"/var/opt/bosch/dynamic/tuner/Japan_Station_Name.db";
	copy_argv[7] = NULL;

	pid_t  pid;
	int   status;

	int u8Result = 0;
	pid = fork();

	if (pid < 0)
	{     /* fork a child process */
		  ETG_TRACE_ERR(("bDifferentialCopyDatabaseFromStaticToDynamic ERROR: forking child process failed"));
	}
	else if (pid == 0)
	{ /* child process execution*/
	  u8Result = execvp(*copy_argv, copy_argv);
	  if (u8Result < 0)
	  {
		  ETG_TRACE_ERR(("bDifferentialCopyDatabaseFromStaticToDynamic execvp failed"));
		  exit(1);
	  }
	}
	else
	{
		while (wait(&status) != pid) // parent process wait for child process to complete
			;
	}
	ETG_TRACE_USR4(("bDifferentialCopyDatabaseFromStaticToDynamic Result : %d",u8Result));

	if(u8Result == 0)
		   return true ; //copy is successful
	   else
		   return false;
}

/******************************************************************************
*
******************************************************************************/
bool clJapanStationName::bDifferentialCopyDatabaseFromDynamicToRAM()
{
   ETG_TRACE_USR4(("bDifferentialCopyDatabaseFromDynamicToRAM : start the copy"));

   	char  *copy_argv[10];
	copy_argv[0] = (char *)"rsync";
	copy_argv[1] = (char *)"--inplace";
	copy_argv[2] = (char *)"--no-whole-file";
	copy_argv[3] = (char *)"-c";
	copy_argv[4] = (char *)"-B=4096";
	copy_argv[5] = (char *)"/var/opt/bosch/dynamic/tuner/Japan_Station_Name.db";
	copy_argv[6] = (char *)"/tmp/Japan_Station_Name.db";
	copy_argv[7] = NULL;

	pid_t  pid;
	int   status;

	int u8Result = 0;
	pid = fork();

	if (pid < 0)
	{     /* fork a child process */
		  ETG_TRACE_ERR(("bDifferentialCopyDatabaseFromDynamicToRAM ERROR: forking child process failed"));
	}
	else if (pid == 0)
	{ /* child process execution*/
	  u8Result = execvp(*copy_argv, copy_argv);
	  if (u8Result < 0)
	  {
		  ETG_TRACE_ERR(("execvp failed"));
		  exit(1);
	  }
	}
	else
	{
		while (wait(&status) != pid) // parent process wait for child process to complete
			;
	}
	ETG_TRACE_USR4(("bDifferentialCopyDatabaseFromDynamicToRAM Result : %d",u8Result));

	if(u8Result == 0)
	   return true ; //copy is successful
   else
	   return false;
}


/******************************************************************************
*
******************************************************************************/
unsigned char clJapanStationName::u8GetStationCount()
{
   return u8JapanStationCount;
}


/******************************************************************************
*
******************************************************************************/
bool clJapanStationName::bIsChangeStationNameUpward(unsigned char u8NewVal)
{
	return ((u8NewVal == CHANGEMODE_TUN_UP)? true: false);
}

/******************************************************************************
*
******************************************************************************/
bool clJapanStationName::bIsChangeStationNameDownward(unsigned char u8NewVal)
{
	return ((u8NewVal == CHANGEMODE_TUN_DOWN )?true: false);
}

/******************************************************************************
*
******************************************************************************/
bool clJapanStationName::bCheckIfValidUTF8(const char* utf8String)
{
	if(utf8String == NULL)
	{
		ETG_TRACE_USR4(("bCheckIfValidUTF8 string is empty"));
		return false;
	}
	if(dbus_validate_utf8(utf8String,0))
	{
		//ETG_TRACE_USR4(("bCheckIfValidUTF8 valid utf8 string"));
		return true;
	}
	else
	{
		ETG_TRACE_USR4(("bCheckIfValidUTF8 not a valid utf8 string"));
		return false;
	}

}
/******************************************************************************
*
******************************************************************************/
bool clJapanStationName::bCheckIfJapaneseLanguage()
{
	if( m_LanguageSetting == JAPAN_LANG )
		return true;
	else
		return false;
}
/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vSetSelectedLanguage(unsigned char u8NewLang)
{
	m_LanguageSetting = u8NewLang;
}

/******************************************************************************
*
******************************************************************************/
void clJapanStationName::vChangeToDefaultJapanDatabase()
{
	// Default setting changes should be available in dynamic folder eventhough cold restart is made. 
	// So changes are made in both the copies /dynamic and /tmp during factory reset.
	ETG_TRACE_USR4(("vChangeToDefaultJapanDatabase"));
	if(bFileExists(STATIC_STATION_DB_PATH))
	{
		vHandleDynamicDatabase();
		if(sqlite3_close(m_databaseobj) != SQLITE_OK)
		{
			   ETG_TRACE_USR4(("vChangeToDefaultJapanDatabase Can't close database: %s", sqlite3_errmsg(m_databaseobj)));
			   return;
		}
		m_databaseobj = NULL; // /tmp (RAM) database is closed. assign object to NULL.
		if( bDifferentialCopyDatabaseFromDynamicToRAM() )
		{
			u32SQLiteCommandReturnValue = sqlite3_open(RAM_STATION_DB_PATH, &m_databaseobj);
			if (u32SQLiteCommandReturnValue!=SQLITE_OK )
			{
				ETG_TRACE_ERR(("Can't open database: %s", sqlite3_errmsg(m_databaseobj)));
				return;
			}
		}
	}
}

/******************************************************************************
*
******************************************************************************/
/*be careful while using this function.
itrJpnStnName value is global and value is considered from the calling function*/
const char* clJapanStationName::pchUpdateStationNameFromDB()
{
	//do not introduce lock in pchUpdateStationNameFromDB. map is already locked in u16GetJpnStnNameListData
    if((! itrJpnStnName->second.stationname_japanese.empty()) && (! itrJpnStnName->second.stationname_english.empty()))
	{
	   // re-initialze temporary string to NULL before assigning actual value
	   sTempStationNameString.erase();
	   // check if language is Japanese (FID_TUN_S_LANGUAGE_SETTING from HMI)
		if(bCheckIfJapaneseLanguage())
		{
		    sTempStationNameString = itrJpnStnName->second.stationname_japanese;
			if(bCheckIfValidUTF8(sTempStationNameString.c_str()))
			{
			   return (sTempStationNameString.c_str());
			}
		}
		else //extract english name otherwise
		{
		    sTempStationNameString = itrJpnStnName->second.stationname_english;
			if(bCheckIfValidUTF8(sTempStationNameString.c_str()))
			{
			   return (sTempStationNameString.c_str());
			}
		}
	}
	ETG_TRACE_USR4(("pchUpdateStationNameFromDB:: english name :%s", itrJpnStnName->second.stationname_english.c_str()));
	ETG_TRACE_USR4(("pchUpdateStationNameFromDB:: japanese name :%s", itrJpnStnName->second.stationname_japanese.c_str()));
	return NULL;
}

/******************************************************************************
*
******************************************************************************/
unsigned short clJapanStationName::u16GetJpnStnNameListData(tunerVector<TunerJapanStationName> &oJpnStnNameList)
{
	ETG_TRACE_USR4(("clJapanStationName::u16GetJpnStnNameListData"));
	unsigned short u16SelectedIndex = 0;
	std::lock_guard<std::mutex> lock(m_JapanMapLock);
	for(itrJpnStnName = mapJpnStnName.begin(); itrJpnStnName!=mapJpnStnName.end(); ++itrJpnStnName)
	{
		if(itrJpnStnName->second.userselect)
		{
			u16SelectedIndex = (unsigned short)itrJpnStnName->first;
		}
		TunerJapanStationName oJpnStnName;
		oJpnStnName.u16IndexID = (unsigned short)itrJpnStnName->first;
		//itrJpnStnName is globally available for this file. So no need to pass it. Directly value can be used in pchUpdateStationNameFromDB
		//do not introduce lock in pchUpdateStationNameFromDB. map is already locked in u16GetJpnStnNameListData
		const char* tempString= pchUpdateStationNameFromDB();
		oJpnStnName.sJpnStnName = (tempString == NULL)? "" : tempString ;
		oJpnStnNameList.push_back(oJpnStnName);
	}

	return u16SelectedIndex;
}
