/************************************************************************
 * FILE:          clPresetListConfiguration.cpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of clPresetListConfiguration.cpp
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   8-Aug-2015			vnd4kor
				
 *************************************************************************/

#include "TunerApp/ADRConfiguration/clPresetListConfiguration.h"

#include "tun_Config.h"
#include "kdsconfig/clConfigInterface.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "tun_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_MSGTOADR
#include "trcGenProj/Header/clPresetListConfiguration.cpp.trc.h"
#endif

#define TUN_PRESETLIST_SIZE 61
#define FM_AST_AVAILABLE 0x08000000

clPresetListConfiguration::clPresetListConfiguration() :clBaseConfigSection(enPresetListConfiguration)
,m_u8NumberOfFMPresetBank(0)
,m_u8NumberOfFMPresetsPerBank(0)
,m_u8NumberOfPresetsPerFMASTBank(0)
,m_u8NumberOfMWPresetBank(0)
,m_u8NumberOfMWPresetPerBank(0)
,m_u8NumberOfLWPresetBank(0)
,m_u8NumberOfLWPresetPerBank(0)
,m_u8NumberOfMixedPresetBank(0)
,m_u8NumberOfMixedPresetPerBank(0)
,m_u8NumberOfDRMPresetBank(0)
,m_u8NumberOfDRMPresetPerBank(0)
,m_potun_Config(NULL)
,m_poclConfigInterface(NULL)
{
}

clPresetListConfiguration::~clPresetListConfiguration()
{
	m_potun_Config = 0;
	m_poclConfigInterface = 0;
}

void clPresetListConfiguration::vAddPresetListToPayload(unsigned char u8Band,unsigned char u8NumberOfPresetsPerBank, unsigned char u8PresetIndex)
{
	if(m_poclConfigInterface == NULL)
	{
		NORMAL_M_ASSERT(m_poclConfigInterface != NULL);
		return;
	}
    m_payload.push_back(u8Band);
	m_payload.push_back(u8NumberOfPresetsPerBank);
		
	for(unsigned char u8PresetCount = 0; u8PresetCount < u8NumberOfPresetsPerBank; u8PresetCount++, u8PresetIndex++)
	{
		vAddInt(m_payload,m_poclConfigInterface->u32GetPreset( (unsigned int)u8PresetIndex));
	}
}

void clPresetListConfiguration::vSerialize()
{
	m_payload.clear();
	bool isPresetListOutOfRange = false;
	if(((m_poclConfigInterface->u32GetConfigData("PresetListMapping") & FM_AST_AVAILABLE) == FM_AST_AVAILABLE) && (m_u8NumberOfPresetsPerFMASTBank != 0))
	{
		if(
			((((m_u8NumberOfFMPresetBank-1) * m_u8NumberOfFMPresetsPerBank) 		+
			 (m_u8NumberOfPresetsPerFMASTBank)+
			 (m_u8NumberOfMWPresetBank * m_u8NumberOfMWPresetPerBank)  		+
			 (m_u8NumberOfLWPresetBank * m_u8NumberOfLWPresetPerBank)  		+
			 (m_u8NumberOfMixedPresetBank * m_u8NumberOfMixedPresetPerBank)	+
			 (m_u8NumberOfDRMPresetBank * m_u8NumberOfDRMPresetPerBank)) > TUN_PRESETLIST_SIZE)
		  )
		{
			isPresetListOutOfRange =  true;
		}
	}
	else
	{
		if(
			(((m_u8NumberOfFMPresetBank * m_u8NumberOfFMPresetsPerBank) 		+
			 (m_u8NumberOfMWPresetBank * m_u8NumberOfMWPresetPerBank)  		+
			 (m_u8NumberOfLWPresetBank * m_u8NumberOfLWPresetPerBank)  		+
			 (m_u8NumberOfMixedPresetBank * m_u8NumberOfMixedPresetPerBank)	+
			 (m_u8NumberOfDRMPresetBank * m_u8NumberOfDRMPresetPerBank)) > TUN_PRESETLIST_SIZE)
		  )
		{
			isPresetListOutOfRange =  true;
		}
	}
	
	if(isPresetListOutOfRange)
	{
			// send invalid configuration to ADR since preset list is out of range
			m_payload.push_back(0xFF);
			m_payload.push_back(0xFF);

			m_u16NumberOfBytesInSection = (tU16) m_payload.size();

			vAddNumberOfBytesInSectionToPayload(m_payload);
			vAddSectionIdToPayload(m_payload);
			return;		
	}
	
	unsigned char u8PresetIndex = 0;
    /*==========================================================================
     * FM preset lists
     *------------------------------------------------------------------------*/
    for( unsigned char u8BankCount = 0; u8BankCount < m_u8NumberOfFMPresetBank; u8BankCount++)
    {
    	if(((m_poclConfigInterface->u32GetConfigData("PresetListMapping") & FM_AST_AVAILABLE) == FM_AST_AVAILABLE) && (m_u8NumberOfPresetsPerFMASTBank != 0) && (u8BankCount == (m_u8NumberOfFMPresetBank-1)))
    	{
    		vAddPresetListToPayload((unsigned char)enAvailBand_FM, m_u8NumberOfPresetsPerFMASTBank,u8PresetIndex);
    		u8PresetIndex = (unsigned char) (u8PresetIndex + m_u8NumberOfPresetsPerFMASTBank);
    		continue;
    	}
		vAddPresetListToPayload((unsigned char)enAvailBand_FM, m_u8NumberOfFMPresetsPerBank,u8PresetIndex);
		u8PresetIndex = (unsigned char) (u8PresetIndex + m_u8NumberOfFMPresetsPerBank);
    }
    /*==========================================================================
     * MW preset lists
     *------------------------------------------------------------------------*/
    for( unsigned char u8BankCount = 0; u8BankCount < m_u8NumberOfMWPresetBank; u8BankCount++)
    {
		vAddPresetListToPayload((unsigned char)enAvailBand_MW, m_u8NumberOfMWPresetPerBank,u8PresetIndex);
		u8PresetIndex = (unsigned char)(u8PresetIndex + m_u8NumberOfMWPresetPerBank);
    }    
	/*==========================================================================
     * LW preset lists
     *------------------------------------------------------------------------*/
    for( unsigned char u8BankCount = 0; u8BankCount < m_u8NumberOfLWPresetBank; u8BankCount++)
    {
		vAddPresetListToPayload((unsigned char)enAvailBand_LW, m_u8NumberOfLWPresetPerBank,u8PresetIndex);
		u8PresetIndex = (unsigned char)(u8PresetIndex + m_u8NumberOfLWPresetPerBank);
    }   
	/*==========================================================================
     * MIXED Preset lists
     *------------------------------------------------------------------------*/
    for( unsigned char u8BankCount = 0; u8BankCount < m_u8NumberOfMixedPresetBank; u8BankCount++)
    {
		vAddPresetListToPayload((unsigned char)enAvailBand_GetCurrent, m_u8NumberOfMixedPresetPerBank,u8PresetIndex);
		u8PresetIndex = (unsigned char)(u8PresetIndex + m_u8NumberOfMixedPresetPerBank);
    }  
	/*==========================================================================
     * DRM Preset lists
     *------------------------------------------------------------------------*/
    for( unsigned char u8BankCount = 0; u8BankCount < m_u8NumberOfDRMPresetBank; u8BankCount++)
    {
		vAddPresetListToPayload((unsigned char)enAvailBand_CB1, m_u8NumberOfDRMPresetPerBank,u8PresetIndex);
		u8PresetIndex = (unsigned char)(u8PresetIndex + m_u8NumberOfDRMPresetPerBank);
    }
    /*==========================================================================
     * End of List
     *------------------------------------------------------------------------*/
	m_payload.push_back(0);
	m_payload.push_back(0);
	/*========================================================================*/
	
	m_u16NumberOfBytesInSection = (tU16) m_payload.size();

	vAddNumberOfBytesInSectionToPayload(m_payload);	
	vAddSectionIdToPayload(m_payload);
}

void clPresetListConfiguration::vSetParameters(tun_Config* potun_Config , clConfigInterface* poclConfigInterface)
{
	if(potun_Config == NULL)
	{
		return;
	}

	if(poclConfigInterface == NULL)
	{
		return;
	}

	m_potun_Config = potun_Config;

	m_poclConfigInterface = poclConfigInterface;

	m_u8NumberOfFMPresetBank = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetListFM");
	m_u8NumberOfFMPresetsPerBank  = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankFM");
	m_u8NumberOfPresetsPerFMASTBank = (tU8)m_poclConfigInterface->u32GetConfigData("UserAutostoreNoOfStations");
	m_u8NumberOfMWPresetBank = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetListMW");
	m_u8NumberOfMWPresetPerBank = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankMW");
	m_u8NumberOfLWPresetBank = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetListLW");
	m_u8NumberOfLWPresetPerBank = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankLW");
	m_u8NumberOfMixedPresetBank = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetListForMixedAllBands");
	m_u8NumberOfMixedPresetPerBank = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankForMixedAllBands");
	m_u8NumberOfDRMPresetBank = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetListDRM");
	m_u8NumberOfDRMPresetPerBank = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankDRM");
	
}

tunerVector<unsigned char> clPresetListConfiguration::oGetPayload()
{
	return m_payload;
}

short clPresetListConfiguration::u16GetSize()
{
	return m_u16NumberOfBytesInSection;
}

void clPresetListConfiguration::vConstruct(tunerVector<unsigned char> /*bitStream*/)
{

}

void clPresetListConfiguration::vTrace()
{

	if(((m_poclConfigInterface->u32GetConfigData("PresetListMapping") & FM_AST_AVAILABLE) == FM_AST_AVAILABLE) && (m_u8NumberOfPresetsPerFMASTBank != 0))
	{
		if(
			((((m_u8NumberOfFMPresetBank-1) * m_u8NumberOfFMPresetsPerBank) 		+
			 (m_u8NumberOfPresetsPerFMASTBank)+
			 (m_u8NumberOfMWPresetBank * m_u8NumberOfMWPresetPerBank)  		+
			 (m_u8NumberOfLWPresetBank * m_u8NumberOfLWPresetPerBank)  		+
			 (m_u8NumberOfMixedPresetBank * m_u8NumberOfMixedPresetPerBank)	+
			 (m_u8NumberOfDRMPresetBank * m_u8NumberOfDRMPresetPerBank)) > TUN_PRESETLIST_SIZE)
		  )
		{
			ETG_TRACE_USR2(( "clPresetListConfiguration::vTrace(), preset list not packed in ADR configuration message since number of presets out of range"));
			return;
		}
	}
	else
	{
		if(
			(((m_u8NumberOfFMPresetBank * m_u8NumberOfFMPresetsPerBank) 		+
			 (m_u8NumberOfMWPresetBank * m_u8NumberOfMWPresetPerBank)  		+
			 (m_u8NumberOfLWPresetBank * m_u8NumberOfLWPresetPerBank)  		+
			 (m_u8NumberOfMixedPresetBank * m_u8NumberOfMixedPresetPerBank)	+
			 (m_u8NumberOfDRMPresetBank * m_u8NumberOfDRMPresetPerBank)) > TUN_PRESETLIST_SIZE)
		  )
		{
			ETG_TRACE_USR2(( "clPresetListConfiguration::vTrace(), preset list not packed in ADR configuration message since number of presets out of range"));
			return;
		}
	}

	ETG_TRACE_USR2(( "clPresetListConfiguration::vTrace() part 1 -> \n************************************\n* Section Id \t\t\t\t\t= %d *\n"\
					 "* m_u16NumberOfBytesInSection = %2d *\n************************************",m_enSectionID,m_u16NumberOfBytesInSection ));

	unsigned char u8PresetIndex = 0;
	 /*==========================================================================
     * FM preset lists
     *------------------------------------------------------------------------*/
	ETG_TRACE_USR2(( "clPresetListConfiguration::vTrace() part 2-> FM Preset Lists : m_u8NumberOfFMPresetBank = %d",m_u8NumberOfFMPresetBank ));
    for( unsigned char u8BankCount = 0; u8BankCount < m_u8NumberOfFMPresetBank; u8BankCount++)
    {
    	if(((m_poclConfigInterface->u32GetConfigData("PresetListMapping") & FM_AST_AVAILABLE) == FM_AST_AVAILABLE) && (m_u8NumberOfPresetsPerFMASTBank != 0) && (u8BankCount == (m_u8NumberOfFMPresetBank-1)))
    	{
    		ETG_TRACE_USR2(( "clPresetListConfiguration::FM AST Preset Lists : m_u8NumberOfPresetsPerFMASTBank = %d",m_u8NumberOfPresetsPerFMASTBank ));
    		vTracePresetList((unsigned char)enAvailBand_FM, m_u8NumberOfPresetsPerFMASTBank,u8PresetIndex);
    		u8PresetIndex = (unsigned char) (u8PresetIndex + m_u8NumberOfPresetsPerFMASTBank);
    		continue;
    	}
    	vTracePresetList((unsigned char)enAvailBand_FM,m_u8NumberOfFMPresetsPerBank,u8PresetIndex);
		u8PresetIndex = (unsigned char) (u8PresetIndex + m_u8NumberOfFMPresetsPerBank);
    }

    /*==========================================================================
     * MW preset lists
     *------------------------------------------------------------------------*/
	ETG_TRACE_USR2(( "clPresetListConfiguration::vTrace() part 2-> MW Preset Lists : m_u8NumberOfMWPresetBank = %d",m_u8NumberOfMWPresetBank ));
    for( unsigned char u8BankCount = 0; u8BankCount < m_u8NumberOfMWPresetBank; u8BankCount++)
    {
		vTracePresetList((unsigned char)enAvailBand_MW, m_u8NumberOfMWPresetPerBank,u8PresetIndex);
		u8PresetIndex  = (unsigned char)(u8PresetIndex + m_u8NumberOfMWPresetPerBank);
    }    
	/*==========================================================================
     * LW preset lists
     *------------------------------------------------------------------------*/
	ETG_TRACE_USR2(( "clPresetListConfiguration::vTrace() part 2-> LW Preset Lists : m_u8NumberOfLWPresetBank = %d",m_u8NumberOfLWPresetBank ));
    for( unsigned char u8BankCount = 0; u8BankCount < m_u8NumberOfLWPresetBank; u8BankCount++)
    {
		vTracePresetList((unsigned char)enAvailBand_LW, m_u8NumberOfLWPresetPerBank,u8PresetIndex);
		u8PresetIndex  = (unsigned char)(u8PresetIndex + m_u8NumberOfLWPresetPerBank);
    }   
	/*==========================================================================
     * MIXED Preset lists
     *------------------------------------------------------------------------*/
	ETG_TRACE_USR2(( "clPresetListConfiguration::vTrace() part 2-> MIXED Preset Lists : m_u8NumberOfMIXEDPresetBank = %d",m_u8NumberOfMixedPresetBank ));
    for( unsigned char u8BankCount = 0; u8BankCount < m_u8NumberOfMixedPresetBank; u8BankCount++)
    {
		vTracePresetList((unsigned char)enAvailBand_GetCurrent, m_u8NumberOfMixedPresetPerBank,u8PresetIndex);
		u8PresetIndex  = (unsigned char)(u8PresetIndex +  m_u8NumberOfMixedPresetPerBank);
    }  
	/*==========================================================================
     * DRM Preset lists
     *------------------------------------------------------------------------*/
	ETG_TRACE_USR2(( "clPresetListConfiguration::vTrace() part 2-> DRM Preset Lists : m_u8NumberOfDRMPresetBank = %d",m_u8NumberOfDRMPresetBank ));
    for( unsigned char u8BankCount = 0; u8BankCount < m_u8NumberOfDRMPresetBank; u8BankCount++)
    {
		vTracePresetList((unsigned char)enAvailBand_CB1, m_u8NumberOfDRMPresetPerBank,u8PresetIndex);
		u8PresetIndex  = (unsigned char)(u8PresetIndex +  m_u8NumberOfDRMPresetPerBank);
    }
}

void clPresetListConfiguration::vTracePresetList(unsigned char u8Band,unsigned char u8NumberOfPresetsPerBank, unsigned char u8PresetIndex)
{
	if(m_poclConfigInterface == NULL)
	{
		NORMAL_M_ASSERT(m_poclConfigInterface != NULL);
		return;
	}
	ETG_TRACE_USR2(( "clPresetListConfiguration::vTracePresetList() \n u8Band : 0x%02X \n u8NumberOfPresetsPerBank = %d \n ",u8Band,u8NumberOfPresetsPerBank));
	for(unsigned char u8PresetCount = 0; u8PresetCount < u8NumberOfPresetsPerBank; u8PresetCount++, u8PresetIndex++)
	{
		ETG_TRACE_USR2(( "clPresetListConfiguration::vTracePresetList() -> FM_PRESET [%d] = %8d kHz ",u8PresetCount + 1,m_poclConfigInterface->u32GetPreset( (unsigned int)u8PresetIndex) ));
	}
}