/************************************************************************
* FILE:         tun_SensorClient.cpp
* PROJECT:      IVI
* SW-COMPONENT: Tuner server
*----------------------------------------------------------------------
*
* DESCRIPTION: This file handles all the CCA messages received from VD_CLOCK component.
*
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:
* Date      | Author                     | Modification
* 11.11.16  | RBEI/ECO11 gds5kor         | initial version
*************************************************************************/

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
// Include generic messages
#ifndef VARIANT_S_FTR_ENABLE_GENERICMSGS_MOCK
#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"
#endif

#ifndef VARIANT_S_FTR_ENABLE_AIL_MOCK
#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"         // use AIL template with MessageMaps
#endif

#ifndef VARIANT_S_FTR_ENABLE_AHL_MOCK
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library
#else
#include "ahl_mock/ahl_if.h"         // use Application Help Library
#endif

// Use OSAL defines
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"


// Access to raw (unorganised) trace
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_TUNAPP
#include "trcGenProj/Header/tun_SensorClient.cpp.trc.h"
#endif

#include "fi_gen_if.h"

#define FI_S_IMPORT_INTERFACE_BASE_TYPES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

//------------------------
// Local headers
//------------------------

// Self header
#include "tun_SensorClient.h"

// Include parent
#include "tun_main.h"

// trace defines.
#include "tun_trace.h"

#include "tun_Singleton.h"
#include "tun_Utility.h"
#include "kdsconfig/clConfigInterface.h"
#include "../AMFMTuner/clTmcTuner_Property_G_GET_STATIC_LIST.h"
#include "../interface/clTmcTuner_TMCTunerControl.h"
#include "tun_MsgFromHMI.h"
/* ---
MESSAGE MAP:
enter the function IDs (FID) and the corresponding functions here.
the function will be called when a message with the corresponding FID arrives
--- */
//#ifdef VARIANT_S_FTR_ENABLE_VDCLOCK

BEGIN_MSG_MAP(tun_tclSensorClient, ahl_tclBaseWork)
  ON_MESSAGE( SENSOR_PVFI_C_U16_POSITION,  vHandleSensorStatus)
END_MSG_MAP()

//#endif
/*************************************************************************
*
* FUNCTION:     tun_tclSensorClient::tun_tclSensorClient
*
* DESCRIPTION:  constructor, creates object tun_tclSensorClient
*
* PARAMETER:    poGWMainApp: main - object of this application
*               poMainSrv  : pointer to virtual block
*
* RETURNVALUE:  None
*
*12.04.06 aca2kor
*         Initial version
*
*************************************************************************/
tun_tclSensorClient::tun_tclSensorClient(tuner_tclApp* poMainApp,  clTmcTuner_stationLogo_map *poclTmcTuner_stationLogo_map)
{
   // Store the reference to main (parent) object
   poGWMain      = poMainApp;
   m_poclConfigInterface = NULL;
   m_poclTmcTuner_stationLogo_map = poclTmcTuner_stationLogo_map;
   m_poclTmcTuner_Property_G_GET_STATIC_LIST = NULL;
   m_u16SrvRegID = AMT_C_U16_REGID_INVALID;
   m_fFidPositionReg = FALSE;
   blFirstValidPositionStatusRx = false;
   m_potun_MsgFromHMI = NULL;

}

/*************************************************************************
*
* FUNCTION:     tun_tclSensorClient::~tun_tclSensorClient
*
* DESCRIPTION:  destructor, destroys the object
*
* PARAMETER:    None
*
* RETURNVALUE:  None
*
*14.04.06 aca2kor
*         Initial version
*
*************************************************************************/
tun_tclSensorClient::~tun_tclSensorClient()
{
   poGWMain     = NULL;
   m_poclConfigInterface = NULL;
   m_poclTmcTuner_stationLogo_map = NULL;
   m_poclTmcTuner_Property_G_GET_STATIC_LIST = NULL;
   m_u16SrvRegID = AMT_C_U16_REGID_INVALID;
   m_fFidPositionReg = FALSE;
   blFirstValidPositionStatusRx = false;
   m_potun_MsgFromHMI = NULL;
}

/*************************************************************************
*
* FUNCTION:     tun_tclSensorClient::vSet_tun_ConfigInterface_Ptr( )
*
* DESCRIPTION:   With this config data can be accessed
*
* PARAMETER:    clConfigInterface* poclConfigInterface
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_tclSensorClient::vSet_tun_ConfigInterface_Ptr( clConfigInterface* poclConfigInterface )
{
  if( poclConfigInterface != NULL )
  {
	  m_poclConfigInterface = poclConfigInterface;
  }

}
tVoid tun_tclSensorClient::vSet_tun_MsgFromHMI_ptr(tun_MsgFromHMI* potun_MsgFromHMI)
{
	if(potun_MsgFromHMI != NULL)
	{
		m_potun_MsgFromHMI = potun_MsgFromHMI;
	}
}

/*************************************************************************
*
* FUNCTION:     tun_tclSensorClient::vOnUnknownMessage
*
* DESCRIPTION:  Frame work calls this function when it receives a message with
 *              an unknown FID. Used for error handling.
*
* PARAMETER:    poMessage : Pointer to CCA data
*
* RETURNVALUE:  None
*
* HISTORY:
* 12.04.06 aca2kor
*          Initial version
*
*************************************************************************/
tVoid tun_tclSensorClient::vOnUnknownMessage(amt_tclBaseMessage* /*poMessage*/)
{
   // Unknown message: send an error message
  /*
  vSendError (poMessage->u16GetSourceAppID(),
              poMessage->u16GetTargetAppID(),
              poMessage->u16GetRegisterID(),
              poMessage->u16GetCmdCounter(),
              poMessage->u16GetServiceID(),
              poMessage->u16GetFunctionID(),
              (tU8) AUDIO_UNKNOWN_MSG);
  */

}


/*************************************************************************
*
* FUNCTION:     tun_tclSensorClient::vOnNewAppState
*
* DESCRIPTION:  Main calls this function whenever there is a change in
 *              power state to handle CCA service
*
* PARAMETER:    u32OldAppState : Last set application state
*               u32AppState    : Current requested app state
*
* RETURNVALUE:  None
*
* HISTORY:
* 14.09.06 dip2kor
*          Initial version
*
*************************************************************************/
tVoid tun_tclSensorClient::vOnNewAppState ( tU32 u32OldAppState,
                                         tU32 u32NewAppState )
{

  if(u32OldAppState==u32NewAppState)
  {
	  return;
  }
  switch(u32NewAppState)
  {
    case AMT_C_U32_STATE_NORMAL:
      {
		  fRegisterForService();
		  break;
	  }
    case AMT_C_U32_STATE_OFF:
      {
        fUnregisterForService();
		 break;
      }
    default:
      {
		  break;
	  }
  }
}
/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSensorClient::vHandleClock
 *
 *DESCRIPTION:  Handle SENSOR_PVFI_C_U16_POSITION property
 *
 *PARAMETER:    poMessage : Pointer to CCA data
 *
 *RETURNVALUE:  None
 *
 *HISTORY:
 *11.11.2016 gds5kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/

tVoid tun_tclSensorClient::vHandleSensorStatus(amt_tclServiceData* poMessage)
{
	if (NULL == poMessage)
	{
		NORMAL_M_ASSERT(NULL != poMessage);
		return;
	}
	switch (poMessage->u8GetOpCode())
	{
		case AMT_C_U8_CCAMSG_OPCODE_STATUS:
		{
			sensor_pvfi_tclMsgpositionStatus oMsg;
			vGetDatafromAMT(poMessage, oMsg);
			if(oMsg.posSource.enType != sensor_fi_tcl_e8_position_source::FI_EN_SRC_POS_INVALID)
			{
				ETG_TRACE_USR4(( "tun_tclSensorClient() Status: Latitude : %f, Longitude : %f",(tFloat) oMsg.posLatitude,(tFloat) oMsg.posLongitude));
				m_poclTmcTuner_stationLogo_map->vOnSensorStatus(oMsg.posLatitude, oMsg.posLongitude);
				if(!blFirstValidPositionStatusRx)
				{
					blFirstValidPositionStatusRx = true;
					if(m_poclTmcTuner_Property_G_GET_STATIC_LIST != NULL)
					{
						m_poclTmcTuner_Property_G_GET_STATIC_LIST->vGetPresetList();
					    clTmcTuner_TMCTunerControl oTunerControl;
					    oTunerControl.vOnCM_ChangeOfSurveilledStationsStatus();
					}
					if(NULL != m_potun_MsgFromHMI)
						m_potun_MsgFromHMI->vSendStationInfoStatusMessage();

				}
			}
			else
			{
				ETG_TRACE_USR4(( "tun_tclSensorClient() Invalid status" ));
			}
		}
		break;
		default:
		break;
	}
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSensorClient::fRegisterForService
 *
 *DESCRIPTION:  Sends a register message to Clock.
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  TRUE  : No Error in registeration
 *              FALSE : Error in registeration
 *
 *HISTORY:
 *11.11.2016 gds5kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tVoid tun_tclSensorClient::fRegisterForService()
{
  tBool fIsServiceRegistered = FALSE;

  // Is registration invalid ?
  if ( AMT_C_U16_REGID_INVALID == m_u16SrvRegID )
  {
     m_fFidPositionReg = FALSE;
     fIsServiceRegistered = poGWMain->bRegisterAsync(CCA_C_U16_SRV_PV,
    		 SENSOR_PVFI_C_U16_SERVICE_MAJORVERSION                      /* Major version */,
    		 SENSOR_PVFI_C_U16_SERVICE_MINORVERSION                      /* Minor version */,
                                           0x0 );	/*do not set application id, since we do not know */
  }

  ETG_TRACE_USR1(( "tun_tclSensorClient() Sensor Service Registration status = %d",  fIsServiceRegistered));

}


/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSensorClient::fUnregisterForService
 *
 *DESCRIPTION:  Sends a unregister message to Clock
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  TRUE  : No Error in unregisteration
 *              FALSE : Error in unregisteration
 *
 *HISTORY:
 *11.11.2016 gds5kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tVoid tun_tclSensorClient::fUnregisterForService()
{

  // Unregister if registration is valid?
//#ifdef VARIANT_S_FTR_ENABLE_VDCLOCK
  if ( AMT_C_U16_REGID_INVALID != m_u16SrvRegID )
  {

    // Send an unregister for service message
    poGWMain->vUnregisterService( (tU16) CCA_C_U16_SRV_PV );

    // Make register-ID invalid:
    m_u16SrvRegID = AMT_C_U16_REGID_INVALID;

    ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_TUNAPP, \
                       ET_EN_T16 _ TUN_TRACE_UNREGISTER_FOR_SERVICE _ \
                       ET_EN_T16 _ m_u16SrvRegID _ \
                       ET_EN_DONE);
  }
  ETG_TRACE_USR2(( "tun_tclSensorClient() Sensor Service UnRegistration status = %d",  m_u16SrvRegID));
//#endif
}
/*************************************************************************
 * FUNCTION:     vOnAsyncRegisterConf
 *
 * DESCRIPTION:  Sensor service registeration confirmation function
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tun_tclSensorClient::vOnAsyncRegisterConf(tU16 u16RegisterId,tU16 u16ServerAppId,
				tU16 u16ServiceId,
				tU16 u16SourceSubID)
{
	m_u16SrvRegID=u16RegisterId;
	ETG_TRACE_USR1(("vRegister For Sensor Service() m_u16SrvRegID=%d u16ServiceId=%d,u16SourceSubID=%d u16ServerAppId=%d"\
		,m_u16SrvRegID,u16ServiceId,u16SourceSubID,u16ServerAppId));
}


/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSensorClient::fRegisterForFID
 *
 *DESCRIPTION:  Registers for a function of a service
 *              !!! before this, you must register for the service !!!
 *
 *PARAMETER:    u16FID : FID of the function to be registered (upreg)
 *
 *RETURNVALUE:  TRUE  : No Error in FID registeration
 *              FALSE : Error in FID registeration
 *
 *HISTORY:
 *11.11.2016 gds5kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tBool tun_tclSensorClient::fRegisterForFID(tU16 u16FID)
{
  // Return value
  tBool fIsServiceRegistered = FALSE;

  ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_TUNAPP, \
                     ET_EN_T16 _ TUN_TRACE_REGISTER_FOR_FID _ \
                     ET_EN_DONE);


  // Is registration invalid?
  if ( AMT_C_U16_REGID_INVALID == m_u16SrvRegID )
  {
    // Ideally should  not come here
    ET_TRACE_ERROR_BIN( TUN_TRACE_CLASS_TUNAPP, ET_EN_T16 _ \
                        TUN_TRACE_REGISTER_FOR_FID_INVALID_SRVID _  ET_EN_T8 _ \
                        0xFF _ ET_EN_DONE);
  }

  // Create message with upreg - request:
  gm_tclEmptyMessage oUpRegMessage
                    (
                      CCA_C_U16_APP_TUNER,          // AppID of this application
                      AMT_C_U16_APPID_INVALID,      // Let server can decide
                      m_u16SrvRegID,                // RegId for the service
                      0,                            // always 0
                      CCA_C_U16_SRV_PV,             // the SID of the service
                      u16FID,                       // the FID to register for
                      AMT_C_U8_CCAMSG_OPCODE_UPREG  // this is an Upreg-message!
                    );

  if( AIL_EN_N_NO_ERROR != poGWMain->enPostMessage(&oUpRegMessage) )
  {
    // Error in sending the message
    if(!oUpRegMessage.bDelete())
    {
      // Error in deleting the message
      // TODO : Send an error trace here
    }
  }
  else
  {
    fIsServiceRegistered = TRUE;
  }

  return fIsServiceRegistered;
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSensorClient::fUnregisterForFID
 *
 *DESCRIPTION:  Unregisters for a function of a service
 *
 *PARAMETER:    u16FID : FID of the function to be registered (upreg)
 *
 *RETURNVALUE:  TRUE  : No Error in FID unregisteration
 *              FALSE : Error in FID unregisteration
 *
 *HISTORY:
 *11.11.2016 gds5kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tBool tun_tclSensorClient::fUnregisterForFID(tU16 u16FID)
{
  // Return value
  tBool fIsServiceUnregistered = FALSE;
//#ifdef VARIANT_S_FTR_ENABLE_VDCLOCK
  ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_TUNAPP, \
                     ET_EN_T16 _ TUN_TRACE_UNREGISTER_FOR_FID _ \
                     ET_EN_DONE);

  // Is register id valid?
  if ( AMT_C_U16_REGID_INVALID != m_u16SrvRegID )
  {
    // Create message with relupreg - request:

    gm_tclEmptyMessage oRelupRegMessage
                (
                  CCA_C_U16_APP_TUNER,             // AppID of this application
                  AMT_C_U16_APPID_INVALID,         // Let Server can decide
                  m_u16SrvRegID,                   // RegId for the service
                  0,                               // always 0
                  CCA_C_U16_SRV_PV,                // the SID of the service
                  u16FID,                          // the FID to register for
                  AMT_C_U8_CCAMSG_OPCODE_RELUPREG  // this is an Upreg-message!
                );



    if( AIL_EN_N_NO_ERROR != poGWMain->enPostMessage(&oRelupRegMessage))
    {
      // Error in sending the message
      if(!oRelupRegMessage.bDelete())
      {
        // Error in deleting the message
        // TODO : Send an error trace here
      }
    }
    else
    {
      fIsServiceUnregistered = TRUE;
    }

  } // if ( AMT_C_U16_REGID_INVALID != m_u16SrvRegID )
//#else
//	ETG_TRACE_USR4(( "tun_tclSensorClient() Sensor FID = %d UnRegistration not success", u16FID));
//#endif
  // return whether the service is unregistered
  return fIsServiceUnregistered;
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSensorClient::vOnServiceState
 *
 *DESCRIPTION:  Handles the state changes of the service we use
 *
 *PARAMETER:    u16ServiceId  : Service id whose service state is sent
 *              u16ServerId   : Server id who sends this service state
 *              u16RegisterId : Register id for service whose service state
 *                              is sent
 *              u8ServiceState: Service state of the registered service
 *              u16SubId      : SubId whose service state is sent
 *
 *RETURNVALUE: None
 *
 *HISTORY:
 *11.11.2016 gds5kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tVoid tun_tclSensorClient::vOnServiceState
(
 tU16 u16ServiceId,
 tU16 /*u16ServerId*/,
 tU16 u16RegisterId,
 tU8  u8ServiceState,
 tU16 /*u16SubId*/
)
{
	ETG_TRACE_USR1(( " tun_tclSensorClient::Register ID:=%d",u16ServiceId ));

	if( CCA_C_U16_SRV_PV != u16ServiceId )
	{
		ETG_TRACE_USR1(( " tun_tclSensorClient::vOnServiceState ServiceId:%d",u16ServiceId ));
		return;
	}
	m_u16SrvRegID = u16RegisterId;

	if (m_u16SrvRegID == AMT_C_U16_REGID_INVALID)
	{
		ETG_TRACE_USR1(( " tun_tclSensorClient::Register ID updated as Invalid "));
	}
	switch(u8ServiceState)
	{
		case AMT_C_U8_SVCSTATE_AVAILABLE:
		{
			ETG_TRACE_USR1(( "tun_tclSensorClient::Sensor Service is available" ));
			if( !m_fFidPositionReg) // Register only once
			{
				m_fFidPositionReg = fRegisterForFID( SENSOR_PVFI_C_U16_POSITION );
				ETG_TRACE_USR1(( "tun_tclSensorClient() Sensor FID SENSOR_PVFI_C_U16_POSITION status is =%d",m_fFidPositionReg ));
			}
		break;
      }
		case AMT_C_U8_SVCSTATE_NOT_AVAILABLE:
		{
			ETG_TRACE_USR1(( "tun_tclSensorClient() Sensor Service is Not available" ));
			break;
		}
		case AMT_C_U8_SVCSTATE_REG_INVALID:
		{
			break;
		}
	  default:
		  break;
	}
}

/*******************************************************************************
 *FUNCTION:    tun_tclSensorClient::vGetDatafromAMT
 *
 *DESCRIPTION: This function converts Amt message to visitor class and
 *             get the data from visitor to the Fi class. Refer FI Manual.
 *             Doc.
 *
 *PARAMETER:
 *           poFiMsg   - Amt Message
 *           oFIData   - Base class of Fi class
 *
 *RETURNVALUE: None
 ********************************************************************************/
//#ifdef VARIANT_S_FTR_ENABLE_VDCLOCK
tVoid tun_tclSensorClient::vGetDatafromAMT( amt_tclServiceData* poFIMsg, fi_tclTypeBase& oFIData)const
{
	// Create Visitor message object for incoming message
	fi_tclVisitorMessage oInVisitorMsg(poFIMsg);

	/* Get the data in FI class from Vistor class*/
	if(OSAL_ERROR == oInVisitorMsg.s32GetData(oFIData,SENSOR_PVFI_C_U16_SERVICE_MAJORVERSION))
	{
		// assert here
		NORMAL_M_ASSERT_ALWAYS ();
	}

}
void tun_tclSensorClient::vSet_clTmcTuner_Property_G_GET_STATIC_LIST_Ptr( clTmcTuner_Property_G_GET_STATIC_LIST* poclTmcTuner_Property_G_GET_STATIC_LIST )
{
	if(poclTmcTuner_Property_G_GET_STATIC_LIST != NULL)
	{
		m_poclTmcTuner_Property_G_GET_STATIC_LIST = poclTmcTuner_Property_G_GET_STATIC_LIST;
	}
}
//#endif
