/************************************************************************
* FILE:         tun_SPMClient.cpp
* PROJECT:      FORDHSRNS
* SW-COMPONENT: Tuner server
*----------------------------------------------------------------------
*
* DESCRIPTION: This file handles all the CCA messages where audio acts
*              as a client to SPM. Currently, it acts as a client to SPM
*              for copro state.
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:      
* Date      | Author                     | Modification
* 12.09.06  | RBIN/EDI1 Dinesh Punja     | initial version
* 25.04.13  | NGP1KOR    | First version of the G3g after porting 
				from NISSAN LCN2Kai
*************************************************************************/

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
// Include generic messages
#ifndef VARIANT_S_FTR_ENABLE_GENERICMSGS_MOCK
#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"
#endif

#ifndef VARIANT_S_FTR_ENABLE_AIL_MOCK
#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"         // use AIL template with MessageMaps
#endif

#ifndef VARIANT_S_FTR_ENABLE_AHL_MOCK
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library
#else
#include "ahl_mock/ahl_if.h"         // use Application Help Library
#endif

// Use OSAL defines
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"




// Access to raw (unorganised) trace
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*****************************************************************************
 * ETG trace
 * 0x2618:    TUN_TRACE_CLASS_SPM_CLIENT
 ****************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_SPM_CLIENT
#include "trcGenProj/Header/tun_SPM_Client.cpp.trc.h"
#endif

//------------------------
// Local headers 
//------------------------

// Self header
#include "tun_SPM_Client.h"

// Include parent
#include "tun_main.h"

// trace defines.
#include "tun_trace.h"
#include "Aars/clAars_CM_UpdateLandscape.h"
#include "tun_MsgToADR.h"
#include "tun_ActualData.h"
#include "tun_defines.h"
#include "tun_Manager.h"
#include "tun_Config.h"
#include "tun_MsgToHMI.h"
#include "tun_HMIManager.h"
#include "tun_DrvAdrIf.hpp"
#include "tun_Singleton.h"
#include "tun_Utility.h"
#include "kdsconfig/clConfigInterface.h"

#ifdef GTEST_x86LINUX_BUILD
#define SPM_COREFI_C_U16_SERVICE_MAJORVERSION 0x01
#define SPM_COREFI_C_U16_SERVICE_MINORVERSION 0x03
#define CFC_SPMFI_C_U16_SERVICE_MAJORVERSION 0x01
#define CFC_SPMFI_C_U16_SERVICE_MINORVERSION 0x00
#endif

/* ---
MESSAGE MAP:
enter the function IDs (FID) and the corresponding functions here.
the function will be called when a message with the corresponding FID arrives
--- */
BEGIN_MSG_MAP(tun_tclSPMClient, ahl_tclBaseWork)
#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
	ON_MESSAGE( SPM_COREFI_C_U16_TUNERSTATE, vHandleSysState)
	ON_MESSAGE( SPM_COREFI_C_U16_CVMEVENT,     vHandleCVM)
#else
	ON_MESSAGE( CFC_SPMFI_C_U16_SYSTEMSTATE,  vHandleSysState)
	ON_MESSAGE( CFC_SPMFI_C_U16_CVMEVENT,     vHandleCVM)
#endif
END_MSG_MAP()

/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient::tun_tclSPMClient
* 
* DESCRIPTION:  constructor, creates object tun_tclSPMClient
*
* PARAMETER:    poGWMainApp: main - object of this application
*               poMainSrv  : pointer to virtual block  
*
* RETURNVALUE:  None
*
*12.04.06 aca2kor
*         Initial version
*
*************************************************************************/
tun_tclSPMClient::tun_tclSPMClient(tuner_tclApp* poMainApp): 
    m_u16SrvRegID(AMT_C_U16_REGID_INVALID), 
    m_fFidSysStateReg(FALSE),
    m_fFidCVMReg(FALSE)
{
   // Store the reference to main (parent) object
   poGWMain      = poMainApp;
   
   m_potu_Manager = NULL;

   m_potun_ActualData = NULL;
   m_potun_MsgToADR = NULL;
   m_potun_MsgToHMI = NULL;
   m_potun_HMIManager = NULL;
   m_potun_Config = NULL;
   m_poclConfigInterface = NULL;
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
   m_poclAars_CM_UpdateLandscape = NULL;
#endif

   // Initialise data
#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
#ifndef GTEST_x86LINUX_BUILD
	m_u32CoproState   = (tU32)spm_fi_tcl_SPM_e32_PROCESSOR_STATE::FI_EN_SPM_U32_PROC_STATE_DOWN;
#endif
#else
	m_u32CoproState   = (tU32)cfc_fi_tcl_SPM_e32_PROCESSOR_STATE::FI_EN_SPM_U32_PROC_STATE_DOWN;
#endif
   // Variable created for future, under voltage handling.
   m_u32VoltageEvent = 0;
   
   m_u8CriticalStatus =  (TUN_CRITICAL_STATUS_COPRO_BIT |TUN_CRITICAL_STATUS_COPRO_STANDBY_BIT);


   // Variable created for future, under voltage handling.
   m_fIsHighVoltageOnStartup = TRUE;
   
   m_LowvoltState = FALSE;

}

/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient::~tun_tclSPMClient
* 
* DESCRIPTION:  destructor, destroys the object
*
* PARAMETER:    None
*
* RETURNVALUE:  None
*
*14.04.06 aca2kor
*         Initial version
*
*************************************************************************/
tun_tclSPMClient::~tun_tclSPMClient()
{
   poGWMain     = NULL;   
   m_potun_ActualData = NULL;
   m_potun_MsgToADR = NULL;
   m_potu_Manager = NULL;
   m_potun_MsgToHMI = NULL;
   m_potun_HMIManager = NULL;
   m_potun_Config = NULL;
   m_poclConfigInterface = NULL;
   m_LowvoltState = FALSE;
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
   m_poclAars_CM_UpdateLandscape = NULL;
#endif
}


/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient::vSet_tun_Config_Ptr( )
* 
* DESCRIPTION:   With this config data can be accessed
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_tclSPMClient::vSet_tun_Config_Ptr( tun_Config* potun_Config )
{
  if( potun_Config != NULL )
  {
    m_potun_Config = ( tun_Config* )potun_Config;
  }

}

/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient::vSet_tun_ConfigInterface_Ptr( )
*
* DESCRIPTION:   With this config data can be accessed
*
* PARAMETER:    clConfigInterface* poclConfigInterface
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_tclSPMClient::vSet_tun_ConfigInterface_Ptr( clConfigInterface* poclConfigInterface )
{
  if( poclConfigInterface != NULL )
  {
	  m_poclConfigInterface = poclConfigInterface;
  }

}

/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient::vSet_tu_Manager_Ptr( )
* 
* DESCRIPTION:  
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_tclSPMClient::vSet_tu_Manager_Ptr( tu_Manager* potu_Manager )
{
  if( potu_Manager != NULL )
  {
    m_potu_Manager = potu_Manager;
  }
}


/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient::vSet_tun_ActualData_Ptr( )
* 
* DESCRIPTION:  
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_tclSPMClient::vSet_tun_ActualData_Ptr( tun_ActualData* potun_ActualData )
{
  if( potun_ActualData != NULL )
  {
    m_potun_ActualData = ( tun_ActualData* )potun_ActualData;
  }
  else
  {
    NORMAL_M_ASSERT( potun_ActualData != NULL );
  }
}


/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient ::vSet_tun_MsgToADR_Ptr( )
* 
* DESCRIPTION:  
*
* PARAMETER:    tun_MsgToADR*
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_tclSPMClient ::vSet_tun_MsgToADR_Ptr( tun_MsgToADR* potun_MsgToADR)
{
  if( potun_MsgToADR != NULL )
  {  
    m_potun_MsgToADR = potun_MsgToADR;
  }
  else
  {
    NORMAL_M_ASSERT ( potun_MsgToADR != NULL );

  }
}

/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient ::vSet_tun_MsgToHMI_Ptr()
* 
* DESCRIPTION:  Set the internal pointer to tun_MsgToHMI class to allow 
*				accessing it.
*
* PARAMETER:    tun_MsgToHMI*
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_tclSPMClient ::vSet_tun_MsgToHMI_Ptr( tun_MsgToHMI* potun_MsgToHMI ){
	if( potun_MsgToHMI != NULL ){  
		m_potun_MsgToHMI = potun_MsgToHMI;
	}
	else{
		NORMAL_M_ASSERT ( potun_MsgToHMI != NULL );
	}
}
/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient::vSet_HMIManager_Ptr( )
* 
* DESCRIPTION:  
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_tclSPMClient::vSet_tun_HMIManager_Ptr( tun_HMIManager* potun_HMIManager )
{
  if( potun_HMIManager != NULL )
  {
    m_potun_HMIManager = potun_HMIManager;
  }
}
/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient::vSet_UpdateLandscape_Ptr( )
* 
* DESCRIPTION:   With this update landscape data can be accessed
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
tVoid tun_tclSPMClient::vSet_UpdateLandscape_Ptr( clAars_CM_UpdateLandscape* poclAars_CM_UpdateLandscape )
{
  if( poclAars_CM_UpdateLandscape != NULL )
  {
    m_poclAars_CM_UpdateLandscape = poclAars_CM_UpdateLandscape;
  }
}
#endif
/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient::vOnUnknownMessage
* 
* DESCRIPTION:  Frame work calls this function when it receives a message with
 *              an unknown FID. Used for error handling.
*
* PARAMETER:    poMessage : Pointer to CCA data
*
* RETURNVALUE:  None
*
* HISTORY:
* 12.04.06 aca2kor
*          Initial version
*
*************************************************************************/
tVoid tun_tclSPMClient::vOnUnknownMessage(amt_tclBaseMessage* /*poMessage*/)
{
   // Unknown message: send an error message
  /*
  vSendError (poMessage->u16GetSourceAppID(),
              poMessage->u16GetTargetAppID(),
              poMessage->u16GetRegisterID(),
              poMessage->u16GetCmdCounter(),
              poMessage->u16GetServiceID(),
              poMessage->u16GetFunctionID(),
              (tU8) AUDIO_UNKNOWN_MSG);
  */
  
}


/*************************************************************************
*
* FUNCTION:     tun_tclSPMClient::vOnNewAppState
* 
* DESCRIPTION:  Main calls this function whenever there is a change in 
 *              power state to handle CCA service
*
* PARAMETER:    u32OldAppState : Last set application state 
*               u32AppState    : Current requested app state
*
* RETURNVALUE:  None
*
* HISTORY:
* 14.09.06 dip2kor
*          Initial version
*
*************************************************************************/
tVoid tun_tclSPMClient::vOnNewAppState ( tU32 u32OldAppState, 
                                         tU32 u32NewAppState )
{
  
  if(u32OldAppState==u32NewAppState)
  {
	  return;
  }
  switch(u32NewAppState)
  {
    case AMT_C_U32_STATE_NORMAL:
      {
		  fRegisterForService();        
		  break;
	  }
    case AMT_C_U32_STATE_OFF:
      {       
        fUnregisterForService();   
		 break;
      }
    default:
      {
		  break;
	  }
  }
}
/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSPMClient::vHandleCVM
 *
 *DESCRIPTION:  React on messages with FID = FID_SPM_COPRO_STATE
 *             
 *PARAMETER:    poMessage : Pointer to CCA data
 *
 *RETURNVALUE:  None
 *
 *HISTORY:
 *10.05.06 mor2hi
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/

tVoid tun_tclSPMClient::vHandleCVM(amt_tclServiceData* poMessage) 
{
	if (NULL == poMessage)
	{
		NORMAL_M_ASSERT(NULL != poMessage);
		return;
	}
#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
#ifndef GTEST_x86LINUX_BUILD
	switch (poMessage->u8GetOpCode())
	{
		case AMT_C_U8_CCAMSG_OPCODE_STATUS: 
		{
			if(NULL != m_potun_MsgToHMI)
			{

				spm_corefi_tclMsgCvmEventStatus oMsg;
				m_potun_MsgToHMI->vGetDatafromAMT(poMessage,oMsg);
				vSetVoltageState((tU32)oMsg.CvmEvent.enType);
			}
		}
		break;
		default:
		break;
	}
#endif
#endif
}
/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSPMClient::vHandleSysState
 *
 *DESCRIPTION:  React on messages with FID = FID_SPM_SYSTEM_STATE
 *             
 *PARAMETER:    poMessage : Pointer to CCA data
 *
 *RETURNVALUE:  None
 *
 *HISTORY:
 *10.05.06 mor2hi
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/

tVoid tun_tclSPMClient::vHandleSysState
(
 amt_tclServiceData* poMessage
)
{
	if ((NULL == poMessage) || ( NULL == m_potun_MsgToADR ) || ( NULL == m_potun_ActualData ) ||
	  (NULL== m_potun_HMIManager) || (m_potun_MsgToHMI == NULL) || (m_potun_Config == NULL) || (m_poclConfigInterface == NULL)
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	  || (NULL == m_poclAars_CM_UpdateLandscape)
#endif
	  )
	  {
	    // CCA framework problem - Notify through trace
	    NORMAL_M_ASSERT(NULL != poMessage);
		NORMAL_M_ASSERT(NULL != m_potun_MsgToADR);
		NORMAL_M_ASSERT(NULL != m_potun_ActualData);
		NORMAL_M_ASSERT(NULL != m_potun_HMIManager);
		NORMAL_M_ASSERT(NULL != m_potun_MsgToHMI);
		NORMAL_M_ASSERT(NULL != m_potun_Config);
		NORMAL_M_ASSERT(NULL != m_poclConfigInterface);
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
		NORMAL_M_ASSERT(NULL != m_poclAars_CM_UpdateLandscape);
#endif
	    return;
	  }




  switch (poMessage->u8GetOpCode())
  {
    case AMT_C_U8_CCAMSG_OPCODE_STATUS: 
      {
         gm_tclU32Message oSysStateMsg(poMessage);
         tU32 u32SysState = oSysStateMsg.u32GetDWord();
#ifndef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
         ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_SPM_CLIENT, \
                            ET_EN_T16 _ TUN_TRACE_SPM_SYSTEM_STATE _ \
                            ET_EN_T32 _ u32SysState _ ET_EN_DONE);	
#endif
         // Update the voltage and critical state
#ifndef GTEST_x86LINUX_BUILD
         m_potun_ActualData->m_otun_Data_MainStatus.vSetSystemState(u32SysState);
#endif

         /* Update of landscape memory only if configured*/

           //tU8 u8ProjectConfig = m_potun_Config->u8GetCurrentProjectInformation();
           tBool bLandscapeSeek = m_potun_Config->fGetLandscapeSeekNeeded();
           ETG_TRACE_USR1(( " tun_tclSPMClient::vHandleSPMState() Landscape update needed -> %d", bLandscapeSeek ));
           if(bLandscapeSeek == TRUE )
           {
        	   tBool fLandscapeStatus = m_potun_HMIManager->bGetLandscapeFlag();

        	   ETG_TRACE_USR1(( " tun_tclSPMClient::vHandleSPMState() -> Landscape update needed u32SysState = %d fLandscapeStatus = %d"
				   ,u32SysState ,fLandscapeStatus ));
			   if((u32SysState == SPM_SYSTEM_STATE_STANDBY) && /*(m_potun_MsgToHMI->u8GetTunFGBGStatus() == (tU8)enAudioSource_No_audio) &&*/ (fLandscapeStatus == FALSE )) //Tuner in BG
             {
               m_potun_HMIManager->vSetLandscapeFlag(TRUE);
			   
			   /*Send CM_LandscapeUpdate to ADR*/
               tU32 u32Band = m_poclConfigInterface->u32GetConfigData("AvailableBands");
			   u32Band = u32Band & TUN_KDS_AVAILABLEBANDS_FM_MW_LW_MASK;
				if (m_poclConfigInterface->u32GetConfigData("DRMSupported") && (m_poclConfigInterface->u32GetConfigData("CombiBands") & (KDS_BIT_31)))
				{
         		   u32Band = u32Band | TUN_UPDATELANDSCAPE_CB1_BIT;
				}
#ifndef GTEST_x86LINUX_BUILD
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
			   m_poclAars_CM_UpdateLandscape->vUpdateLandscape(u32Band);
#else
			   m_potun_MsgToADR->vUpdateLandscape(u32Band);
#endif
#endif
			   //ETG_TRACE_USR1(( " tun_tclSPMClient::vHandleSysState() -> Update Landscape in STANDBY state, Available Bands: %08x",m_poclConfigInterface->u32GetConfigData("AvailableBands") ));
             }
			   else if(( u32SysState == SPM_SYSTEM_STATE_ON) 
#ifndef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
				   || (cfc_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_STATE_PROFILE==u32SysState)
#endif				   
				   )
             {
				 if(m_potun_HMIManager->bGetLandscapeFlag())
				 {
#ifndef GTEST_x86LINUX_BUILD
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
			   m_poclAars_CM_UpdateLandscape->vUpdateLandscape(0);
#else
					 m_potun_MsgToADR->vUpdateLandscape(0);
#endif
#endif
			   	 	 m_potun_HMIManager->vSetLandscapeFlag(FALSE);
				 }
				/*If Seek is ongoing, stop the seek*/
				
				if(m_potun_MsgToHMI->u8GetSeekMode() == TUN_SEEKTYPE_OFF)
				{	
					ETG_TRACE_USR1(( " tun_tclSPMClient::vHandleSysState() -> no seek ongoing" ));
					//No seek ongoing
				}
				else
				{
					ETG_TRACE_USR1(( " tun_tclSPMClient::vHandleSysState() -> stop seek" ));
					/*Stop ongoing seek*/
					tU8 u8SeekMode = (tU8)enATSeekMode_Off;
#ifndef GTEST_x86LINUX_BUILD
					m_potun_MsgToADR->vSendSeekStart(u8SeekMode, 0x00);
#endif
				
				}
             }
           }
      }
      break;

    default:
      {
         // Ignore any other message from SPM
      }
      break;

  } // switch (poMessage->u8GetOpCode())

}


/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSPMClient::fRegisterForService
 *
 *DESCRIPTION:  Sends a register message to SPM.
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  TRUE  : No Error in registeration
 *              FALSE : Error in registeration
 *
 *HISTORY:
 *12.04.06 dip2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tBool tun_tclSPMClient::fRegisterForService()
{
  // Return value
  tBool fIsServiceRegistered = FALSE;

  // Is registration invalid ?  
  if ( AMT_C_U16_REGID_INVALID == m_u16SrvRegID )
  {
     m_fFidSysStateReg = FALSE;
     m_fFidCVMReg = FALSE;

#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
	m_u16SrvRegID = poGWMain->u16RegisterService(CCA_C_U16_SRV_SPM,
                                           SPM_COREFI_C_U16_SERVICE_MAJORVERSION,                      /* Major version */ 
                                           SPM_COREFI_C_U16_SERVICE_MINORVERSION,                      /* Minor version */
                                           0x0,
                                           CCA_C_U16_APP_SPM);

    fIsServiceRegistered = ( m_u16SrvRegID != AMT_C_U16_REGID_INVALID );
#else
#ifndef VARIANT_S_FTR_ENABLE_FEAT_VDFMAM_AUD_CCACOMM
	m_u16SrvRegID = poGWMain->u16RegisterService(CCA_C_U16_SRV_SPM,
                                           CFC_SPMFI_C_U16_SERVICE_MAJORVERSION,              /* Major version */ 
                                           CFC_SPMFI_C_U16_SERVICE_MINORVERSION,              /* Minor version */
                                           0x0,
                                           CCA_C_U16_APP_SPM);

    fIsServiceRegistered = ( m_u16SrvRegID != AMT_C_U16_REGID_INVALID );
#endif
#endif
  }
  if (FALSE == fIsServiceRegistered)
  {
  
    ETG_TRACE_USR1(( "SPM Service Registration  failed" ));
 
  }
  return fIsServiceRegistered;
}


/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSPMClient::fUnregisterForService
 *
 *DESCRIPTION:  Sends a unregister message to SPM
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  TRUE  : No Error in unregisteration
 *              FALSE : Error in unregisteration
 *
 *HISTORY:
 *12.04.06 dip2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tBool tun_tclSPMClient::fUnregisterForService()
{
  // Return value   
  tBool fIsServiceUnregistered = TRUE; 

  // Unregister if registration is valid?  
  if ( AMT_C_U16_REGID_INVALID != m_u16SrvRegID )
  {
  
    // Send an unregister for service message    
    poGWMain->vUnregisterService( (tU16) CCA_C_U16_SRV_SPM );   
  
    // Make register-ID invalid:
    m_u16SrvRegID = AMT_C_U16_REGID_INVALID;

    ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_SPM_CLIENT, \
                       ET_EN_T16 _ TUN_TRACE_UNREGISTER_FOR_SERVICE _ \
                       ET_EN_T16 _ m_u16SrvRegID _ \
                       ET_EN_DONE);
  }
  

  // return whether the service is unregistered
  return fIsServiceUnregistered;
}


/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSPMClient::fRegisterForFID
 *
 *DESCRIPTION:  Registers for a function of a service
 *              !!! before this, you must register for the service !!!
 *             
 *PARAMETER:    u16FID : FID of the function to be registered (upreg)
 *
 *RETURNVALUE:  TRUE  : No Error in FID registeration
 *              FALSE : Error in FID registeration
 *
 *HISTORY:
 *12.04.06 dip2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tBool tun_tclSPMClient::fRegisterForFID(tU16 u16FID)
{
  // Return value
  tBool fIsServiceRegistered = FALSE;

  ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_SPM_CLIENT, \
                     ET_EN_T16 _ TUN_TRACE_REGISTER_FOR_FID _ \
                     ET_EN_DONE);


  // Is registration invalid?  
  if ( AMT_C_U16_REGID_INVALID == m_u16SrvRegID )
  { 
    // Ideally should  not come here
    ET_TRACE_ERROR_BIN( TUN_TRACE_CLASS_SPM_CLIENT, ET_EN_T16 _ \
                        TUN_TRACE_REGISTER_FOR_FID_INVALID_SRVID _  ET_EN_T8 _ \
                        0xFF _ ET_EN_DONE);
  }

  // Create message with upreg - request:
  gm_tclEmptyMessage oUpRegMessage
                    ( 
                      CCA_C_U16_APP_TUNER,          // AppID of this application
                      CCA_C_U16_APP_SPM,            // AppID of the other Server
                      m_u16SrvRegID,                    // RegId for the service
                      0,                            // always 0
                      CCA_C_U16_SRV_SPM,            // the SID of the service
                      u16FID,                       // the FID to register for
                      AMT_C_U8_CCAMSG_OPCODE_UPREG  // this is an Upreg-message!
                    ); 

  if( AIL_EN_N_NO_ERROR != poGWMain->enPostMessage(&oUpRegMessage) )
  {
    // Error in sending the message
    if(!oUpRegMessage.bDelete())
    {
      // Error in deleting the message
      // TODO : Send an error trace here
    }    
  }  
  else
  {
    fIsServiceRegistered = TRUE;
  }

  // return whether the service is registered
  return fIsServiceRegistered;
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSPMClient::fUnregisterForFID
 *
 *DESCRIPTION:  Unregisters for a function of a service
 *             
 *PARAMETER:    u16FID : FID of the function to be registered (upreg)
 *
 *RETURNVALUE:  TRUE  : No Error in FID unregisteration
 *              FALSE : Error in FID unregisteration
 *
 *HISTORY:
 *14.09.06 dip2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tBool tun_tclSPMClient::fUnregisterForFID(tU16 u16FID)
{  
  // Return value   
  tBool fIsServiceUnregistered = FALSE; 

  ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_SPM_CLIENT, \
                     ET_EN_T16 _ TUN_TRACE_UNREGISTER_FOR_FID _ \
                     ET_EN_DONE);

  // Is register id valid?  
  if ( AMT_C_U16_REGID_INVALID != m_u16SrvRegID ) 
  {     
    // Create message with relupreg - request:
    
    gm_tclEmptyMessage oRelupRegMessage
                ( 
                  CCA_C_U16_APP_TUNER,             // AppID of this application
                  CCA_C_U16_APP_SPM,               // AppID of the other Server 
                  m_u16SrvRegID,                       // RegId for the service
                  0,                               // always 0 
                  CCA_C_U16_SRV_SPM,               // the SID of the service    
                  u16FID,                          // the FID to register for   
                  AMT_C_U8_CCAMSG_OPCODE_RELUPREG  // this is an Upreg-message!
                );

    

    if( AIL_EN_N_NO_ERROR != poGWMain->enPostMessage(&oRelupRegMessage))
    {
      // Error in sending the message
      if(!oRelupRegMessage.bDelete())
      {
        // Error in deleting the message
        // TODO : Send an error trace here
      }      
    }
    else
    {
      fIsServiceUnregistered = TRUE;
    }

  } // if ( AMT_C_U16_REGID_INVALID != m_u16SrvRegID ) 

  // return whether the service is unregistered
  return fIsServiceUnregistered;
}
/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSPMClient::vSetVoltageState
 *
 *DESCRIPTION:  Sets the voltage status sent by SPM
 *             
 *PARAMETER:    u32Event : Voltage event
 *
 *RETURNVALUE:  None
 *
 *HISTORY:
 *14.04.06 aca2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tVoid tun_tclSPMClient::vSetVoltageState (tU32 u32Event)
{  
 m_u32CoproState = u32Event;

 // Trace this event value
 ET_TRACE_INFO_BIN(TUN_TRACE_CLASS_SPM_CLIENT, ET_EN_T16 _ \
             TUN_TRACE_LOW_VOLTAGE_EVENT _  ET_EN_T32 _ \
             u32Event _ ET_EN_DONE );

 // Set/reset the critical status low voltage bit also
 switch(m_u32CoproState)
 {
   // Although we are not handling any high voltage => this trigger can 
   // can be useful at the startup.
   case CVM_HIGH_VOLTAGE_START:
     // intentional fallthrough
   case CVM_CRITICAL_HIGH_VOLTAGE_START:
     // intentional fallthrough
   case CVM_CRITICAL_HIGH_VOLTAGE_END:
     // intentional fallthrough
   case CVM_HIGH_VOLTAGE_END:
     break;
     // intentional fallthrough

   // Default value to indicate normal voltage ( same as low voltage end)
   case CVM_NO_EVENT:
     // intentional fallthrough
   // Normal voltage
   case CVM_LOW_VOLTAGE_END:
   {

     if((m_potun_ActualData != NULL) && (poGWMain != NULL))
     {
       m_potun_ActualData->m_fIsLowVoltageEnd = TRUE;
       m_potun_ActualData->m_fIsLowVoltageOnStartup = FALSE;         
         
     } 
		m_LowvoltState = FALSE;

      }
   break;

   // Low voltage
   case CVM_LOW_VOLTAGE_START:
   {

     if((m_potun_ActualData != NULL) && (poGWMain != NULL) && (m_potun_MsgToADR != NULL))
     {
	   
	   if(m_potun_MsgToADR->bGetTestModeActivity())
	   {
			m_potun_ActualData->m_bIsTmActiveBeforeLowVoltage = TRUE;
	   }
	   else
	   {
			m_potun_ActualData->m_bIsTmActiveBeforeLowVoltage = FALSE;
	   }

       m_potun_ActualData->m_fIsLowVoltageOnStartup = TRUE;
       m_potun_ActualData->m_fIsLowVoltageEnd = FALSE;

       if(!m_potun_ActualData->m_bIsTunerShutdown)
       {
         m_potun_ActualData->m_bIsTunerShutdown = TRUE;
       }
     }
	 
	 m_LowvoltState = TRUE;
	 
	 if( ( tun_DrvAdrIf::instance() )->enGetADRState() == enADRState_DEAD )
	 {
		  ETG_TRACE_USR1(( " tun_tclSPMClient::vSetVoltageState() , low voltage start and ADR is dead : Clear Ping Timer "));

		  tS32 s32RetVal = OSAL_s32TimerSetTime( tun_DrvAdrIf::instance()->m_hTimerPingTimer, 0, 0 );

		  ETG_TRACE_USR1((" Ping Timer  : %u", tun_DrvAdrIf::instance()->m_hTimerPingTimer));

		  		    		  
  	      TUN_ASSERT_RETURN( s32RetVal == OSAL_OK);
	 }
	 
	 if(m_potun_MsgToHMI != NULL)
	 {
		m_potun_MsgToHMI->m_u8PrevBand = 0xFF;
	 }

   }
   break;
   
   case CVM_CRITICAL_LOW_VOLTAGE_START:
   {
       //ETG_TRACE_USR1(( " tun_tclSPMClient::vSetVoltageState() , critical low voltage start"));
   }
   break;
   
   case CVM_CRITICAL_LOW_VOLTAGE_END:
   {
	   //ETG_TRACE_USR1(( " tun_tclSPMClient::vSetVoltageState() , critical low voltage end"));
   }
   break;
   
   // Other status
   default:
      {
          // Ignore this since we currently we don't require
          // critical low voltage info
      }
      break;
 } // switch(m_u32CoproState) 
  

 if ( NULL == poGWMain )
 {
   // Trace this corrupted reference
   NORMAL_M_ASSERT ( NULL != poGWMain );
 }
 else
 {
   // to do
 }
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSPMClient::u32GetActualCoproState
 *
 *DESCRIPTION:  Reads the copro state
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  Copro State
 *
 *HISTORY:
 *14.04.06 aca2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tU32 tun_tclSPMClient::u32GetActualCoproState ()const
{
  return(m_u32CoproState);
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSPMClient::u32GetVoltageState
 *
 *DESCRIPTION:  Reads the voltage state
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  Voltage State
 *
 *HISTORY:
 *14.04.06 aca2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tU32 tun_tclSPMClient::u32GetVoltageState ()const
{
  return(m_u32CoproState);
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSPMClient::u8GetCriticalStatus
 *
 *DESCRIPTION:  Read the critical status bits
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  Critical status bits
 *
 *HISTORY:
 *14.04.06 aca2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tU8 tun_tclSPMClient::u8GetCriticalStatus ()const
{
  return(m_u8CriticalStatus);
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_tclSPMClient::vOnServiceState
 *
 *DESCRIPTION:  Handles the state changes of the service we use
 *             
 *PARAMETER:    u16ServiceId  : Service id whose service state is sent
 *              u16ServerId   : Server id who sends this service state
 *              u16RegisterId : Register id for service whose service state
 *                              is sent
 *              u8ServiceState: Service state of the registered service
 *              u16SubId      : SubId whose service state is sent
 *
 *RETURNVALUE: None
 *
 *HISTORY:
 *12.04.06 aca2kor
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tVoid tun_tclSPMClient::vOnServiceState
( 
 tU16 u16ServiceId,
 tU16 /*u16ServerId*/,
 tU16 u16RegisterId,
 tU8  u8ServiceState,
 tU16 /*u16SubId*/
)
{
	// We are interested only in SPM service
	if( CCA_C_U16_SRV_SPM != u16ServiceId )
	{
		ETG_TRACE_USR1(( " vOnServiceState ServiceId:%d",u16ServiceId ));
		return;
	}  
	m_u16SrvRegID = u16RegisterId;
	ETG_TRACE_USR1(( " Register ID:=%d",m_u16SrvRegID ));
	if (m_u16SrvRegID == AMT_C_U16_REGID_INVALID)
	{
		ETG_TRACE_USR1(( " Register ID updated as Invalid "));
	}
	switch(u8ServiceState)
	{
		case AMT_C_U8_SVCSTATE_AVAILABLE:
		{
			//ETG_TRACE_USR1(( "SPM Service is available" ));
#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI			
			ETG_TRACE_USR1(( "SPM Service is available" ));
			if( !m_fFidSysStateReg) // Register only once
			{
				#ifndef GTEST_x86LINUX_BUILD
				m_fFidSysStateReg = fRegisterForFID( SPM_COREFI_C_U16_TUNERSTATE );
				#endif
				ETG_TRACE_USR1(( "SPM FID SPM_COREFI_C_U16_TUNERSTATE status is =%d",m_fFidSysStateReg ));
			}
#else
			if( !m_fFidSysStateReg) // Register only once
			{
				m_fFidSysStateReg = fRegisterForFID( CFC_SPMFI_C_U16_SYSTEMSTATE );
				ETG_TRACE_USR1(( "SPM FID CFC_SPMFI_C_U16_SYSTEMSTATE status is =%d",m_fFidSysStateReg ));
			}
#endif
		
#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_TMCTUNER_VAG
#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
			if( !m_fFidCVMReg)
			{
				#ifndef GTEST_x86LINUX_BUILD
				m_fFidCVMReg = fRegisterForFID( SPM_COREFI_C_U16_CVMEVENT );
				#endif
				ETG_TRACE_USR1(( "SPM FID SPM_COREFI_C_U16_CVMEVENT status is =%d",m_fFidCVMReg ));
			}
#else
			if( !m_fFidCVMReg)
			{
				m_fFidCVMReg = fRegisterForFID( CFC_SPMFI_C_U16_CVMEVENT );
				ETG_TRACE_USR1(( "SPM FID CFC_SPMFI_C_U16_CVMEVENT status is =%d",m_fFidCVMReg ));
			}
#endif
#endif
		break;
      }
		case AMT_C_U8_SVCSTATE_NOT_AVAILABLE:
		{
			ETG_TRACE_USR1(( "SPM Service is Not available" ));
			break;
		}
		case AMT_C_U8_SVCSTATE_REG_INVALID:
		{
			fRegisterForService();   
			break;
		}
	  default:
		  break;
	}
}
#ifndef GTEST_x86LINUX_BUILD
tVoid tun_tclSPMClient::vSendMessage( tU16             u16DestAppID,
                                     const fi_tclTypeBase&  oOutData,
                                     tU16             u16Fid,
                                     tU8              u8OpCode,
                                     tU16             u16CmdCtr,
                                     tU16             u16RegId)
{
   fi_tclVisitorMessage oOutVisitorMsg(oOutData,1);
   oOutVisitorMsg.vInitServiceData(
         CCA_C_U16_APP_TUNER,
         u16DestAppID,
         AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
         0,
         u16RegId,
         u16CmdCtr,
         CCA_C_U16_SRV_SPM,
         u16Fid,
         u8OpCode,
         0,
         0,
         0 );

  if( AIL_EN_N_NO_ERROR != poGWMain->enPostMessage( &oOutVisitorMsg) )
  {
     if (!oOutVisitorMsg.bDelete())
     {
       // Error in deleting the message
     }
  }

}
#endif
void tun_tclSPMClient::vRestartSBRProcess()
{
#ifdef VARIANT_S_FTR_ENABLE_NEW_SPM_CORE_FI
#ifndef GTEST_x86LINUX_BUILD
	spm_corefi_tclMsgRestartProcessMethodStart oMsg;
	tunerString sProcessName = "rbcm-ars-radio.service";
	oMsg.strLocation.bSet(sProcessName.c_str() ,spm_fi_tclString::FI_EN_UTF8);
	vSendMessage( CCA_C_U16_APP_SPM,
			oMsg,
			SPM_COREFI_C_U16_RESTARTPROCESS,
			AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
			0,
			m_u16SrvRegID);
#endif
#endif
}
