/************************************************************************
* FILE:           tun_MsgDispatcher.cpp
* PROJECT:        g3g
* SW-COMPONENT:
*----------------------------------------------------------------------
*
* DESCRIPTION:    Radio Unit messages are received has to be dispached to
the concerned classes. ( HMI, Audio, TMC ). These
classes will decide what action has to done.
*
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------

* HISTORY:
* Date      | Author                       | Modification
* 20.01.05  | CM-DI/ESA2 ( RBIN ) Dinesh   | Initial version.
*
*************************************************************************/



#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*****************************************************************************
 * ETG trace
 * 0x2610:    TUN_TRACE_CLASS_DISPATCHER
 ****************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_DISPATCHER
#include "trcGenProj/Header/tun_MsgDispatcher.cpp.trc.h"
#endif

#ifndef VARIANT_S_FTR_ENABLE_AMT_MOCK
#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"
#else
#include "amt_mock/amt_if.h"
#endif

//-------------------------- FI includes

/*  Interface for Visitor Message */
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE

#define FI_S_IMPORT_INTERFACE_TMCTUNERFI_STDVISITORS // not sure
#define FI_S_IMPORT_INTERFACE_TMCTUNERFI_ERRORCODES
#define FI_S_IMPORT_INTERFACE_TMCTUNERFI_TYPES
#define FI_S_IMPORT_INTERFACE_TMCTUNERFI_FUNCTIONIDS
#define FI_S_IMPORT_INTERFACE_TMCTUNERFI_SERVICEINFO

#include "fi_gen_if.h" // Include tmctuner_fi in VOB:di_fi using "fi_gen_if.h"

#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_ERRORCODES

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "midw_fi_if.h"

//------------------------------------


#include "tun_Utility.h"
#include "tun_trace.h"

#include "tun_defines.h"
#include "tun_ActualData.h"

#include "tun_MsgToADR.h"

#include "tun_Manager.h"
#include "tun_HMIManager.h"

#include "tun_MsgToHMI.h"

#include "tun_MsgDispatcher.h"
#include "tun_MsgFromHMI.h"
#include "tun_Config.h"
#include "tun_MsgtoTunMaster.h"

#include "tun_DiagManager.h"

#include "tun_DrvAdrIf.hpp"

#include "kdsconfig/clConfigInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
#include "Aars/clAars_ATPresetList_XX.h"
#include "Aars/clAars_DemodulationSwitch.h"
#include "AMFMTuner/clTmcTuner_Method_S_SET_ACTIVE_PRESETLIST.h"
#include "AMFMTuner/clTmcTuner_Method_S_USER_PROFILE_UPDATE.h"
#include "AMFMTuner/clTmcTuner_Method_S_JAPAN_SET_AREA.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_TUNAPP
#include "trcGenProj/Header/tun_MsgDispatcher.cpp.trc.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_AARSPROXY
#include "AarsDBusProxy/AarsDBusProxyControl.h"
#endif

#define WORKAROUND_FOR_STARTUP_MISING_FROM_ADR3 0

tBool tun_MsgDispatcher::m_bWorkaroundStartupADR3 = false;

/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::tun_MsgDispatcher( )
*
* DESCRIPTION:  Constructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*************************************************************************/
tun_MsgDispatcher::tun_MsgDispatcher(tuner_tclApp* potunerMainApp  ): m_potu_Manager( NULL ),\
m_potun_MsgToHMI( NULL ),\
m_potun_MsgFromHMI( NULL ),\
m_potun_ActualData( NULL ),\
m_potun_HMIManager ( NULL ),\
m_potun_DiagManager(NULL),\
m_potun_MsgToADR(NULL),\
m_potun_Config(NULL),\
m_poclConfigInterface(NULL),\
m_potun_MsgToTunMaster(NULL)

{
	   poGWMain         = potunerMainApp;
	   m_potun_tclDiagHandler=NULL;
#ifdef VARIANT_S_FTR_ENABLE_AARSPROXY
	   AarsDBusProxyControl::instance()->vSetAarsDBusProxyImpl(LUNID, this);
#endif

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	   m_poclAars_ATPresetList_XX=NULL;
	   m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST=NULL;
	   m_poclAars_DemodulationSwitch=NULL;
	   m_poclTmcTuner_Method_S_USER_PROFILE_UPDATE=NULL;
	   m_poclTmcTuner_Method_S_JAPAN_SET_AREA = NULL;
#endif
}



/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::~tun_MsgDispatcher( )
*
* DESCRIPTION:  Destructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_MsgDispatcher::~tun_MsgDispatcher( )
{
   m_potu_Manager = NULL;

   m_potun_MsgToHMI = NULL;

   m_potun_MsgFromHMI = NULL;

   m_potun_ActualData = NULL;

   m_potun_HMIManager = NULL;

   m_potun_DiagManager = NULL;

   m_potun_MsgToADR = NULL;

   m_potun_Config = NULL;

   m_potun_MsgToTunMaster = NULL;

   m_poclConfigInterface = NULL;

   poGWMain =  NULL;

   m_bWorkaroundStartupADR3 = false;
   m_potun_tclDiagHandler =NULL;
   #ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
   m_poclAars_ATPresetList_XX= NULL;
   m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST=NULL;
   m_poclTmcTuner_Method_S_USER_PROFILE_UPDATE=NULL;
   m_poclAars_DemodulationSwitch=NULL;
   m_poclTmcTuner_Method_S_JAPAN_SET_AREA = NULL;
   #endif

}

/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vSet_tun_Config_Ptr( )
*
* DESCRIPTION:   With this config data can be accessed
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgDispatcher::vSet_tun_MsgToTunMaster_Ptr( tun_MsgToTunMaster* potun_MsgToTunMaster )
{
   if( potun_MsgToTunMaster != NULL )
   {
      m_potun_MsgToTunMaster = ( tun_MsgToTunMaster* )potun_MsgToTunMaster;
   }

}

/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vSet_tun_Config_Ptr( )
*
* DESCRIPTION:   With this config data can be accessed
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgDispatcher::vSet_tun_Config_Ptr( tun_Config* potun_Config )
{
   if( potun_Config != NULL )
   {
      m_potun_Config = ( tun_Config* )potun_Config;
   }

}

/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vSet_tun_ConfigInterface_Ptr( )
*
* DESCRIPTION:   With this config data can be accessed
*
* PARAMETER:    clConfigInterface* poclConfigInterface
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgDispatcher::vSet_tun_ConfigInterface_Ptr( clConfigInterface* poclConfigInterface )
{
  if( poclConfigInterface != NULL )
  {
	  m_poclConfigInterface = poclConfigInterface;
  }

}

/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vSet_tu_Manager_Ptr( )
*
* DESCRIPTION:
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgDispatcher::vSet_tu_Manager_Ptr( tu_Manager* potu_Manager )
{
   if( potu_Manager != NULL )
   {
      m_potu_Manager = ( tu_Manager* )potu_Manager;
   }
   else
   {
      NORMAL_M_ASSERT( potu_Manager != NULL );
   }
}


/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher ::vSet_tun_MsgToADR_Ptr( )
*
* DESCRIPTION:
*
* PARAMETER:    tun_MsgToADR*
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgDispatcher ::vSet_tun_MsgToADR_Ptr( tun_MsgToADR* potun_MsgToADR)
{
   if( potun_MsgToADR != NULL )
   {
      m_potun_MsgToADR = potun_MsgToADR;
   }
   else
   {
      NORMAL_M_ASSERT ( potun_MsgToADR != NULL );

   }
}
/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vSet_tun_ActualData_Ptr( )
*
* DESCRIPTION:
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgDispatcher::vSet_tun_ActualData_Ptr( tun_ActualData* potun_ActualData )
{
   if( potun_ActualData != NULL )
   {
      m_potun_ActualData = ( tun_ActualData* )potun_ActualData;
   }
   else
   {
      NORMAL_M_ASSERT( potun_ActualData != NULL );
   }
}




/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vSet_tun_MsgToHMI_Ptr( )
*
* DESCRIPTION:
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgDispatcher::vSet_tun_MsgToHMI_Ptr( tun_MsgToHMI* potun_MsgToHMI )
{
   if( potun_MsgToHMI != NULL )
   {
      m_potun_MsgToHMI = ( tun_MsgToHMI* )potun_MsgToHMI;
   }
   else
   {
      NORMAL_M_ASSERT( potun_MsgToHMI != NULL );
   }
}

/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vSet_tun_MsgFromHMI_Ptr( )
*
* DESCRIPTION:
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgDispatcher::vSet_tun_MsgFromHMI_Ptr( tun_MsgFromHMI* potun_MsgFromHMI )
{
   if( potun_MsgFromHMI != NULL )
   {
      m_potun_MsgFromHMI = ( tun_MsgFromHMI* )potun_MsgFromHMI;
   }
   else
   {
      NORMAL_M_ASSERT( potun_MsgFromHMI != NULL );
   }
}
/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vSet_tun_HMIManager_Ptr( )
*
* DESCRIPTION:
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgDispatcher::vSet_tun_HMIManager_Ptr( tun_HMIManager* potun_HMIManager )
{
   if( potun_HMIManager != NULL )
   {
      m_potun_HMIManager = ( tun_HMIManager* )potun_HMIManager;
   }
   else
   {
      NORMAL_M_ASSERT( potun_HMIManager != NULL );
   }
}


/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vSet_tun_DiagManager_Ptr( )
*
* DESCRIPTION:  This is added to Access the Diagnostics function
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgDispatcher :: vSet_tun_DiagManager_Ptr(tun_DiagManager* potun_DiagManager )
{
   if( potun_DiagManager != NULL )
   {
      m_potun_DiagManager = ( tun_DiagManager* )potun_DiagManager;
   }
   else
   {
      NORMAL_M_ASSERT( potun_DiagManager != NULL );
   }
}

tVoid tun_MsgDispatcher :: vSet_tun_tclDiagHandler(tun_tclDiagHandler* potun_tclDiagHandler )
{
   if( potun_tclDiagHandler != NULL )
   {
      m_potun_tclDiagHandler = ( tun_tclDiagHandler* )potun_tclDiagHandler;
   }
   else
   {
      NORMAL_M_ASSERT( potun_tclDiagHandler != NULL );
   }
}
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
tVoid tun_MsgDispatcher::vSet_clAars_ATPresetList_XXPtr(clAars_ATPresetList_XX* poclAars_ATPresetList_XX)
  {
		if(NULL != poclAars_ATPresetList_XX)
	   {
		  m_poclAars_ATPresetList_XX = ( clAars_ATPresetList_XX* )poclAars_ATPresetList_XX;
	   }
  }
tVoid tun_MsgDispatcher::vSet_TmcTuner_Method_S_SET_ACTIVE_PRESETLISTPtr(clTmcTuner_Method_S_SET_ACTIVE_PRESETLIST* poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST)
{
	if(NULL != poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST)
	{
		m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST =(clTmcTuner_Method_S_SET_ACTIVE_PRESETLIST*)poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST;
	}

}
tVoid tun_MsgDispatcher::vSet_TmcTuner_Method_S_USER_PROFILE_UPDATEPtr(clTmcTuner_Method_S_USER_PROFILE_UPDATE* poclTmcTuner_Method_S_USER_PROFILE_UPDATE)
{
	if(NULL != poclTmcTuner_Method_S_USER_PROFILE_UPDATE)
	{
		m_poclTmcTuner_Method_S_USER_PROFILE_UPDATE =(clTmcTuner_Method_S_USER_PROFILE_UPDATE*)poclTmcTuner_Method_S_USER_PROFILE_UPDATE;
	}
}
tVoid tun_MsgDispatcher::vSet_clAars_DemodulationSwitch(clAars_DemodulationSwitch* poclAars_DemodulationSwitch)
{
	m_poclAars_DemodulationSwitch=poclAars_DemodulationSwitch;
}

tVoid tun_MsgDispatcher::vSet_TmcTuner_Method_S_JAPAN_SET_AREAPtr(clTmcTuner_Method_S_JAPAN_SET_AREA* poclTmcTuner_Method_S_JAPAN_SET_AREA)
{
	if(poclTmcTuner_Method_S_JAPAN_SET_AREA != NULL)
	{
		m_poclTmcTuner_Method_S_JAPAN_SET_AREA = poclTmcTuner_Method_S_JAPAN_SET_AREA;
	}
}
#endif
/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::bDispatchRUMessage( tU8* u8Message, tU32 u32Len )
*
* DESCRIPTION:  Dispatch the RU Msg to respective classes -> Audio, HMI, TMC, Diagnostics
*
* PARAMETER:    void
*
* RETURNVALUE:  tBool
*
*************************************************************************/
tVoid tun_MsgDispatcher::bDispatchRUMessage( tU8* u8Message, tU32 u32Len )
{
    if(( m_potun_Config == NULL ) || ( m_potun_MsgToTunMaster == NULL )||\
            ( m_potun_DiagManager == NULL) || (m_potun_MsgToHMI == NULL) ||\
            (m_potun_MsgToADR == NULL)  || (m_potun_ActualData == NULL)||\
            (m_potun_MsgFromHMI == NULL) ||(m_potun_tclDiagHandler==NULL) || (m_poclConfigInterface == NULL)
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
			|| (m_poclAars_ATPresetList_XX == NULL)
			|| (m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST == NULL)
#endif
			)
    {
        NORMAL_M_ASSERT( m_potun_Config != NULL );
        NORMAL_M_ASSERT( m_potun_MsgToADR != NULL );
        NORMAL_M_ASSERT( m_potun_ActualData != NULL );
        NORMAL_M_ASSERT( m_potun_MsgToTunMaster != NULL );
        NORMAL_M_ASSERT( m_potun_DiagManager != NULL );
        NORMAL_M_ASSERT( m_potun_MsgToHMI != NULL );
        NORMAL_M_ASSERT( poGWMain != NULL );
        NORMAL_M_ASSERT( m_potun_MsgFromHMI != NULL );
		NORMAL_M_ASSERT( m_poclConfigInterface != NULL );
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
		NORMAL_M_ASSERT( m_poclAars_ATPresetList_XX != NULL );
		NORMAL_M_ASSERT( m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST != NULL );
#endif
        return;
    }
    /*-----------------------------------------------*/
    /* Update the RU messages to Diagnostics manager */
    /*-----------------------------------------------*/
    m_potun_DiagManager ->vHandleMessagesfromRU( u8Message, u32Len );

    tun_SsiRxBuffer oRxBuffer( u8Message, u32Len);


    /*--------------------------------------------------------------------*/
    /* trace error */
    /**pmn3kor:
     * created new function to solve logiscope issues
     * Handler for error optype */

	if( oRxBuffer.enGetOpType() == enOpType_ERROR)
	{
		vHandleOpTypeError( u8Message, oRxBuffer);
		return;
	}


    /*--------------------------------------------------------------------*/
    /**pmn3kor:
     * created new function to solve logiscope issues
     * Handler for Updating TMC Info*/
	if( TUN_DRVADRIF_GET_U16( &u8Message[enAdrMsgOffset_FBLOCK_ID]) == (tU16)enFBlockId_TMCTuner)
	{
		vHandleTMCData(u8Message, u32Len, oRxBuffer);
		return;
	}


    if((m_potu_Manager != NULL) && (m_potun_HMIManager != NULL)  && (TUN_DRVADRIF_GET_U16(&u8Message[enAdrMsgOffset_FBLOCK_ID]) == (tU16)enFBlockId_AmFmTuner))
    {

        tU16 u16FktID = TUN_DRVADRIF_GET_U16( &u8Message[enAdrMsgOffset_FKT_ID]);

		ETG_TRACE_USR4(( " tun_MsgDispatcher::bDispatchRUMessage() -> FBlock AMFM Tuner : %04X",\
			ETG_CENUM( tun_tenAmFmTunerMsg,    u16FktID)));


        if(u16FktID >= (tU16)enMsg_CM_ATPresetList_00 && u16FktID <= (tU16)enMsg_CM_ATPresetList_31) //response for CM_AtPresetList
        {
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20
			m_potun_MsgToHMI ->vHandle_r_CM_AtPresetList( u8Message, u32Len );
#endif
			return;
        }


            switch( TUN_DRVADRIF_GET_U16( &u8Message[enAdrMsgOffset_FKT_ID]) )//Function ID
            {
            /*------------------------------------------------------------------*/
            /*Response for CM_Testmode*/
            case enMsg_CM_TestMode:
            {
                vProcess<trMsg_CM_TestMode_Status>( &oRxBuffer);       // distribute message to all notified clients within fc_tmctuner
                m_potun_MsgToHMI ->vHandle_r_Get_TestMode( u8Message);
            }
                break;
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20
                /** Response for enMsg_CM_EnhancedTestMode */
            case enMsg_CM_EnhancedTestMode:
            {
                m_potun_MsgToHMI ->vHandle_r_Get_EnhancedTestMode( u8Message );
            }
                break;
#endif
            case enMsg_CM_UpdateLandscape:
            {
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20
            	/*If Landscape was requested, on response check if SPM is not in Normal send Commit*/
                if((m_potu_Manager->u8GetTunerSystemState()) != AMT_C_U32_STATE_NORMAL)
                {
                    m_potun_MsgToADR->vSendCommit();

                }
				else
				{
					m_potun_MsgToHMI->vHandle_r_ConfigList_Update((tU8)FI_EN_TUN_RES_REQ_SUCCEED);
				}
                m_potun_MsgToADR->vSetLandscapeRequested(FALSE);
#endif
			}
                break;


            case enMsg_CM_ReceptionParameter:
            {
                vParseAndTrace<trMsg_CM_ReceptionParameter_Status>( &oRxBuffer);
                tU8 u8ReceptionParam = u8Message[10];
                if(u8ReceptionParam == (tU8)enCM_ReceptionParameterPosSharx) //Sharx
                {
                    m_potun_MsgToHMI ->vHandle_r_Sharx_Level( u8Message);
               }
                else if(u8ReceptionParam == (tU8)enCM_ReceptionParameterPosHighcut) //HiCut
                {
                    m_potun_MsgToHMI ->vHandle_r_Set_Highcut( u8Message);

                }
                else
                {
                    //Do nothing
                }
            }
                break;

                /*------------------------------------------------------------------*/

            case enMsg_AFBestSwitch:
            case enMsg_AFBestSwitchFeatures:
            {
                vParseAndTrace<trMsg_CM_AFBestSwitch_Status>( &oRxBuffer);
                m_potun_MsgToHMI ->vHandle_r_AFBestSwitch( u8Message);
            }
                break;

                /*------------------------------------------------------------------*/

            case enMsg_CM_DDASwitch:
            {
                vParseAndTrace<trMsg_CM_DDASwitch_Status>( &oRxBuffer);
                m_potun_MsgToHMI ->vHandle_r_DDASwitch( u8Message );
            }
                break;

                /*------------------------------------------------------------------*/

            case enMsg_RDSSwitch:
            {
                vParseAndTrace<trMsg_RDSSwitch_Status>( &oRxBuffer);
                m_potun_MsgToHMI ->vHandle_r_RDSSwitch( u8Message );
            }
                break;

                /*------------------------------------------------------------------*/

          /*------------------------------------------------------------------*/

                /*------------------------------------------------------------------*/

            case enMsg_CM_Autostore:
            {
                m_potun_MsgToHMI ->vHandle_r_Autostore( u8Message);

            }
                break;

                /*------------------------------------------------------------------*/
            case enMsg_REG:
            {
                vParseAndTrace<trMsg_CM_REG_Status>( &oRxBuffer);
                m_potun_MsgToHMI ->vHandle_r_REGMode( u8Message, u32Len );
            }
                break;

                /*------------------------------------------------------------------*/

            case enMsg_CM_CalibrationData:
            {
                vParseAndTrace<trMsg_CM_CalibrationData_Status>( &oRxBuffer);
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20
                m_potun_MsgToHMI ->vHandle_r_Get_CalibrationData( u8Message, u32Len );
#endif
            }
                break;

                /*------------------------------------------------------------------*/

            case enMsg_CM_CurrentRadiotext:
            {
                m_potun_MsgToHMI ->vHandle_r_Get_RadioText( u8Message);
            }
                break;


            case enMsg_CM_ATTunerState:
            {
                vParseAndTrace<trMsg_CM_ATTunerState_Status>( &oRxBuffer);
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20
                m_potun_MsgToHMI ->vHandle_r_TunerState( u8Message );
#endif
            }
                break;

            case enMsg_CM_SpecialDiagnosisMode:
            {
                //vParseAndTrace<trMsg_CM_ATTunerState_Status>( &oRxBuffer);
                m_potun_MsgToHMI ->vHandle_r_SpecialDiagnosis( u8Message);
            }
                break;

            case enMsg_CM_ATSeek:
            {
                vParseAndTrace<trMsg_CM_ATSeek_Status>( &oRxBuffer);
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20
                m_potun_MsgToHMI ->vHandle_r_ATSeek( u8Message, u32Len );
#endif
            }
                break;

            case enMsg_CM_AudioSource:
#ifdef VARIANT_S_FTR_ENABLE_SBR_API_A2S1
            case enMsg_CM_MTC_ATAudioSource:
#endif
            {
                vParseAndTrace<trMsg_CM_AudioSource_Status>( &oRxBuffer);
                tBool bLandscapeSeek = m_potun_Config->fGetLandscapeSeekNeeded();
                if(bLandscapeSeek)
                {
                    ETG_TRACE_USR4(( " tun_MsgDispatcher::bDispatchRUMessage() ->enMsg_CM_AudioSource->landscape update needed"));

                    tBool fLandscapeStatus = m_potun_HMIManager->bGetLandscapeFlag();

                    /* Update_Lanscape Seek is sent only when the System state is STANDBY*/
                    if(((m_potun_ActualData->m_otun_Data_MainStatus.u32GetSystemState())
                        == SPM_SYSTEM_STATE_STANDBY ) && (u8Message[enAdrMsgOffset_DATA] == (tU8)enAudioSource_No_audio) && (fLandscapeStatus ==  FALSE ))
                    {
                        ETG_TRACE_USR4(( " tun_MsgDispatcher::bDispatchRUMessage()->enMsg_CM_AudioSource-> send updatelandscape "));
                        m_potun_HMIManager->vSetLandscapeFlag(TRUE);

                        /*Send CM_LandscapeUpdate to ADR*/
                        m_potun_MsgToADR->vUpdateLandscape(TUN_CM_LANDSCAPEUPDATE_BAND_MASK);
                    }
                }

               // m_potun_MsgToHMI -> vHandle_r_Audio_Fg_Bg_new( u8Message, u32Len ); method result sent when tunerstate received idle.
            }
                break;
                /*------------------------------------------------------------------*/

            case enMsg_CM_Ping:
            {
                vProcess<trMsg_CM_Ping_Status>( &oRxBuffer);       // distribute message to all notified clients within fc_tmctuner
                tun_DrvAdrIf::instance()->vHandle_r_Ping( u8Message, u32Len );

                /**Added to sync with ADR3 as start up message is missing/lost from/in INC.*/
#ifdef WORKAROUND_FOR_STARTUP_MISING_FROM_ADR3

                if(m_bWorkaroundStartupADR3 == false)
                {
                    /**calling the same function when we receive the ADR3 startup message*/
                    m_potun_MsgToADR->vConfigureAdr();

                    /**It should enter only once, hence made TRUE.*/
                    m_bWorkaroundStartupADR3 = true;

                }

#endif
            }
                break;

                /*------------------------------------------------------------------*/

            case enMsg_CM_Startup:
            {

				/** to prevent sending  configuration twice*/
				m_bWorkaroundStartupADR3 = true;

                /*--------------------------------------*/
                /* Component Status ACTIVE / INACTIVE   */
                /*--------------------------------------*/

                m_potun_MsgToHMI->m_u8PrevBand = 0xFF;
//				 m_potun_MsgToADR->vSet_bSetConfigurationMessageSent(false);
                m_potun_MsgToADR->vConfigureAdr();
				vProcess<trMsg_CM_Startup_Status>( &oRxBuffer);   // distribute message to all notified clients within fc_tmctuner


                if( m_potun_ActualData->m_bIsTmActiveBeforeLowVoltage )
                {
                    ETG_TRACE_USR4(( " tun_MsgDispatcher::bDispatchRUMessage() -> enMsg_CM_Startup, TM on before low voltage"));
                    // Testmode was active before undervoltage. Turn it on again.
                    tU32 u32TestmodeFeature = 0x00001000;
#ifndef GTEST_x86LINUX_BUILD
                    m_potun_MsgFromHMI->vSetTestModeFeatureValue(u32TestmodeFeature);
#endif
                    m_potun_MsgToADR->vSendTestmode(u32TestmodeFeature,(tU16)enMsg_CM_EnhancedTestMode);
                }
            }

                break;

                /*------------------------------------------------------------------*/
            case enMsg_CM_ATBand:
            {
                m_potun_MsgToHMI ->vHandle_r_AtBand( u8Message, u32Len );
            }
                break;

            case enMsg_CM_SystemPresets:
            {
                m_potun_MsgToHMI ->vHandle_r_SystemPresets( u8Message, u32Len );
            }
                break;
            case enMsg_CM_SetConfiguration:
			case enMsg_CM_SetConfigSections:
            {
                m_potun_MsgToADR ->vHandle_r_SetConfiguration( u8Message, u32Len );
            }
                break;

                /*------------------------------------------------------------------*/

            /*case enMsg_CM_GetConfigurationInfo:
            {
                m_potun_MsgToADR ->vHandle_r_GetConfigurationInfo( u8Message, u32Len );
            }
                break;*/


                /*------------------------------------------------------------------*/

            case enMsg_CM_VersionInfo:
            {
                m_potun_MsgToADR ->vHandle_r_CM_VersionInfo( u8Message, u32Len );
#ifdef VARIANT_S_FTR_ENABLE_ACTIVEPRESETLIST_BANDCHG
                /**To get the active preset list of the ADR3*/
				//pmn3kor: reading from datapool
                //m_potun_MsgToADR ->vGetBandDetails( 255);
#endif
            }
                break;

            /*
            case enMsg_CM_AdditionalVersionInfo:
            {
                m_potun_MsgToADR ->vHandle_r_CM_AdditionalVersionInfo( u8Message );
            }
                break;
            */
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20

            case enMsg_CM_Commit:
            {
                /*After receiving the Result for CM_Commit, send Service Unavailability
                    to clients*/
                if((m_potu_Manager->u8GetTunerSystemState()) != AMT_C_U32_STATE_NORMAL) // Not Normal State
                {
                    ETG_TRACE_USR4(( " tun_MsgDispatcher::bDispatchRUMessage() -> enMsg_CM_Commit , sending tunershutdownprocess since system state not normal"));
                    /** NCG3D-10402 and NCG3D-8305 tuner shut down handled in main thread */
					//poGWMain->vTunerShutDownProcess();
                }
                else
                {
                    ETG_TRACE_USR4(( " tun_MsgDispatcher::bDispatchRUMessage() -> enMsg_CM_Commit , tunershutdownprocess not sent"));
                }
            }
                break;


#endif
            case enMsg_CM_ATStationInfo:
            {
                vProcess<trMsg_CM_ATStationInfo_Status>( &oRxBuffer);       // distribute message to all notified clients within fc_tmctuner
                m_potun_MsgToHMI ->vHandle_r_AtStationInfo( u8Message);

                if(m_potun_MsgFromHMI != NULL)
                {
                    tBool bRFSeqStatus =  FALSE;
                    bRFSeqStatus = m_potun_MsgFromHMI->bGetFSRequestStatus();

                    if(bRFSeqStatus)
                    {
                        m_potun_MsgFromHMI->vSetFSRequestStatus(FALSE);
                        m_potun_MsgToHMI->vHandleFSMethodResult(u8Message);
                    }
                }
            }
                break;

                /*------------------------------------------------------------------*/

            case enMsg_CM_ATEnhancedAfList:
            {

                m_potun_MsgToHMI ->vHandle_r_AfList( u8Message);
            }
                break;



                /*------------------------------------------------------------------*/


                /*------------------------------------------------------------------*/

            case enMsg_CM_DefaultSetting:
            {
				m_potun_MsgToADR->vSetDefaultSettingMsg(FALSE);
				m_potun_MsgToADR->vSendGetAnnouncementSwitch();
				m_potun_MsgToADR->vGetREGMode();
				m_potun_MsgToADR->vGetAFMode();
				#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
					if (m_poclAars_DemodulationSwitch != NULL)
					{
						m_poclAars_DemodulationSwitch->vGetDemodulationSwitch();
					}
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_RNAIVI
					if (m_poclTmcTuner_Method_S_USER_PROFILE_UPDATE != NULL)
					{
						m_poclTmcTuner_Method_S_USER_PROFILE_UPDATE->vOnDeleteProfileResult(true);
					}
					if (m_poclTmcTuner_Method_S_JAPAN_SET_AREA != NULL)
					{
						if(m_poclTmcTuner_Method_S_JAPAN_SET_AREA->bIsJapanRegion())
						{
							m_poclTmcTuner_Method_S_JAPAN_SET_AREA->vSetDefaultSettingRequested(true);
							m_poclTmcTuner_Method_S_JAPAN_SET_AREA->vReadKDSJapanBytes();
							/*reset the database user select values of /tmp and /dynamic/tuner/  to default as present in /static/tuner database*/
							m_poclTmcTuner_Method_S_JAPAN_SET_AREA->vGetDefaultJapanDatabase();
						    /*value read from KDS finished. set default setting flag to false*/
							m_poclTmcTuner_Method_S_JAPAN_SET_AREA->vSetDefaultSettingRequested(false);
						}
					}
#endif
				#endif
				m_potun_tclDiagHandler->vSendSystemSetResponse();
				if(m_potun_tclDiagHandler->blGetSystemSetFinished())
				{
					m_potun_MsgToADR->vGetDefaultPresetListAfterDefaultSetting();
				}
			}

                break;


                /*----------------------------------------------------------------------*/
            case enMsg_CM_StationListConfig:
            {
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20
                m_potun_MsgToHMI ->vHandle_r_GetConfigList( u8Message );
#endif


            }
                break;

                /*------------------------------------------------------------------*/

            case enMsg_CM_StationsList:
            {
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20
            	m_potun_MsgToHMI ->vHandle_r_CM_StationsList( u8Message );
#endif

            }
                break;
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
      case enMsg_CM_ATHD_PSDDataFrame:
         {
            m_potun_MsgToHMI ->vHandle_r_CM_ATHD_PSDDataFrame( u8Message);
         }
         break;
#endif

                /*------------------------------------------------------------------*/

            case enMsg_CM_ATPresetListSetActive:
            {
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20
				m_potun_MsgToHMI ->vHandle_r_CM_ATPresetListSetActive( );
#endif
            }
                break;

                /*------------------------------------------------------------------*/

            case enMsg_CM_SetEngineeringParameter:
            {
                m_potun_MsgToHMI ->vHandle_r_CM_EngineeringParameter( u8Message );
            }
                break;

                /*------------------------------------------------------------------*/

            case enMsg_CM_TuningActionConfiguration:
            {
                m_potun_MsgToHMI ->vHandle_r_CM_TuningActionConfiguration( u8Message );
            }
                break;

                /** ------------------------------------------------------------------- */
                /** -------------Announcement related switch cases ---------------------*/
                /** ------------------------------------------------------------------- */
            case enMsg_CM_AnnouncementSwitch:
            {
                vParseAndTrace<trMsg_CM_AnnouncementSwitch_Status>( &oRxBuffer);
                m_potun_MsgToHMI ->vHandle_r_CM_AnnouncementSwitch( u8Message );
            }
                break;
            case enMsg_CM_AnnouncementMessage:
            {
                m_potun_MsgToHMI ->vHandle_r_CM_AnnouncementMessage( u8Message );
            }
                break;
            case enMsg_CM_AnnouncementInfo:
            {
                m_potun_MsgToHMI ->vHandle_r_CM_AnnouncementInfo( u8Message );
            }
                break;
                /** ------------------------------------------------------------------ */
            case enMsg_CM_PiPsRelation:
            {
                m_potun_MsgToHMI ->vHandle_r_CM_PiPsRelation( u8Message );
            }
                break;
                /** ------------------------------------------------------------------ */
            case enMsg_CM_SpeedInformation:
            {
                ETG_TRACE_USR4(( " tun_MsgDispatcher::bDispatchRUMessage() -> \
                                 enMsg_CM_SpeedInformation Status : Speed = %d",\
                        u8Message[enAdrMsgOffset_DATA]));
            }
                break;
            case enMsg_CM_EvaluationParameter:
            {
                m_potun_MsgToHMI ->vHandle_r_CM_EvaluationParameter( u8Message );
            }
                break;
            case enMsg_CM_UpdateLandscapeProgress:
            {
#ifndef VARIANT_S_FTR_ENABLE_ADR_API_3_20
            	m_potun_MsgToHMI ->vHandle_r_CM_UpdateLandscapeProgress( u8Message );
#endif
            }
            	break;
            case enMsg_CM_SetUser:
            {
            	if((m_potun_MsgToADR->bGetPersonalisationReq()) && (m_potun_tclDiagHandler->blGetSystemSetFinished()))
            	{
            		 m_potun_MsgToADR->vGetDefaultPresetListAfterSetUser();
				}
				m_potun_tclDiagHandler->vSendSystemSetResponse();
            	m_potun_MsgToADR ->vHandle_CM_SetUser();

            }
            	break;
            case enMsg_CM_GetUser:
            {
            	m_potun_MsgToADR ->vHandle_CM_GetUser(u8Message);
            	break;
            }
			default:
            {
                ET_TRACE_ERROR_BIN( TUN_TRACE_CLASS_DISPATCHER, ET_EN_T8 _ TUN_TRACE_ADRIF_RX_MSG_NOT_SUPPORTED _ ET_EN_T8LIST _ u32Len _ u8Message _ ET_EN_DONE);
            }
                break;
            };


    }
	//moved because data should be updated in actualdata after handling CM_AtStationInfo in tun_MsgToHMI.
	//we are referring old data stored in actualdata class for validation for japanstation name in vUpdateStationInfo
    /*------------------------------*/
    /* Update the Actual tuner data */
    /*------------------------------*/
    m_potun_ActualData ->vUpdateDataFromRU( u8Message, u32Len );
}
/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vHandleTMCData
*
* DESCRIPTION:  Dispatch the TMC related info
*
* PARAMETER:    tU8 *u8Message, tU32 u32Len, tun_SsiRxBuffer oRxBuffer
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgDispatcher::vHandleTMCData(tU8 *u8Message, tU32 u32Len, tun_SsiRxBuffer oRxBuffer)
{
	  if(m_potun_MsgToHMI == NULL)
    {
          NORMAL_M_ASSERT( m_potun_MsgToHMI != NULL );

        return;
    }

	  tU16 u16FktID = TUN_DRVADRIF_GET_U16( &u8Message[enAdrMsgOffset_FKT_ID]);

        ETG_TRACE_USR4(( " tun_MsgDispatcher::bDispatchRUMessage() -> FBlock TMC Tuner: %04X",\
			ETG_CENUM( tun_tenAmFmTunerMsg,    u16FktID)));
        //Messages related to TMC
        switch( u16FktID )//Function ID
        {
        case enMsg_CM_TMCStations:
        {
            m_potun_MsgToHMI ->vHandle_r_TMCStationsCount( u8Message);
            m_potun_MsgToHMI ->vHandle_r_TMCStations( u8Message);
        }
            break;

		case enMsg_CM_TMCMonitoredStationsUpdate:
			{

				m_potun_MsgToHMI ->vHandle_r_TMC_MonitoredStations_Update( u8Message, u32Len );
			}
			break;

		case enMsg_CM_TMCTrafficInfos:
			{
				vParseAndTrace<trMsg_CM_TMCTrafficInfos_Status>( &oRxBuffer);
				m_potun_MsgToHMI ->vHandle_r_TMC_Buffered_TrafficInfos( u8Message );
			}
			break;

		case enMsg_CM_TMCSelectionFilter:
			{
				m_potun_MsgToHMI->vHandle_r_TMCSelctionFilter( u8Message);
			}
			break;
		default:
			{
				ET_TRACE_ERROR_BIN( TUN_TRACE_CLASS_DISPATCHER, ET_EN_T8 _ TUN_TRACE_ADRIF_RX_MSG_NOT_SUPPORTED _ ET_EN_T8LIST _ u32Len _ u8Message _ ET_EN_DONE);
			}
			break;
		};

}

/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vHandleOpTypeError
*
* DESCRIPTION:  Dispatch the error traces
*
* PARAMETER:    tU8 *u8Message, tun_SsiRxBuffer oRxBuffer
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgDispatcher::vHandleOpTypeError(tU8 *u8Message, tun_SsiRxBuffer oRxBuffer)
{
	  if(   (m_potun_MsgToADR == NULL)  ||\
            (m_potun_MsgFromHMI == NULL) ||
			(m_potun_MsgToHMI == NULL)
			)
    {
        NORMAL_M_ASSERT( m_potun_MsgToADR != NULL );
        NORMAL_M_ASSERT( m_potun_MsgFromHMI != NULL );
        NORMAL_M_ASSERT( m_potun_MsgToHMI != NULL );

        return;
    }

        trMsg_Generic_Error oErrMsg;
        (tVoid)oErrMsg.enParse( &oRxBuffer);
        oErrMsg.vTrace();

        /** WorkAround : If enMsg_CM_EnhancedTestMode not implemented in ADR
        then send enMsg_CM_TestMode*/
        /** vnd4kor: bugfix : only one byte was used for comparison instead of complete 2 bytes */
        if(TUN_DRVADRIF_GET_U16( &u8Message[enAdrMsgOffset_FKT_ID])==(tU16)enMsg_CM_EnhancedTestMode)
        {
            m_potun_MsgToADR->vSendTestmode(\
                        m_potun_MsgFromHMI->u32GetTestModeFeatureValue()\
                        ,(tU16)enMsg_CM_TestMode);
        }
		/** esponse for CM_AtPresetList */
		else if(TUN_DRVADRIF_GET_U16( &u8Message[enAdrMsgOffset_FKT_ID])>= (tU16)enMsg_CM_ATPresetList_00
			&&
			TUN_DRVADRIF_GET_U16( &u8Message[enAdrMsgOffset_FKT_ID])<= (tU16)enMsg_CM_ATPresetList_31 )
		{
			m_potun_MsgToHMI ->vSendPresetHandlingMethodResult(u8Message,0);
		}
		else if(TUN_DRVADRIF_GET_U16( &u8Message[enAdrMsgOffset_FKT_ID])== (tU16)enMsg_CM_UpdateLandscape)
		{
			m_potun_MsgToHMI ->vHandle_r_ConfigList_Update((tU8)FI_EN_TUN_RES_REQ_FAIL);
		}
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_RNAIVI
		else if(TUN_DRVADRIF_GET_U16( &u8Message[enAdrMsgOffset_FKT_ID])== (tU16)enMsg_CM_DefaultSetting)
		{ 	/*Added for Personalization delete feature */
			/*We need to send negative response when CM_DefaultSetting with option 3 fails */
			m_potun_MsgToADR->vSetDefaultSettingMsg(FALSE);
			m_poclTmcTuner_Method_S_USER_PROFILE_UPDATE->vOnDeleteProfileResult(false);
		}
#endif
        else
        {
            if(u8Message[enAdrMsgOffset_DATA]==0xf5)
            {

				ETG_TRACE_USR4(( " tun_MsgDispatcher::bDispatchRUMessage() -> 0xf5 response received"));
                /*Set to Diagnose mode, so that no further messages are sent including PING*/
                tun_DrvAdrIf::instance()->vSetADRCommnState(enComState_DiagnoseSession);
                return;
            }
            else if(u8Message[enAdrMsgOffset_DATA]!=0x41)
            {
                /*Error Response to messages from ADR3 is not processed,
                to be removed once ADR3 sends 0xf5 for PING also*/
                ETG_TRACE_USR4(( " tun_MsgDispatcher::bDispatchRUMessage() -> \
                                 ERROR msg Received with Error Code : %d",u8Message[enAdrMsgOffset_DATA]));
                                 return;
            }
            else
            {
                // Forward the messages for further processing
            }
        }


}



/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vProcess( )
*
* DESCRIPTION:  Parse the the message buffer, given by parameter,
*               do a trace and forward the message to all notified clients.
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
template< class MSG>
tVoid tun_MsgDispatcher::vProcess( tun_SsiRxBuffer* poRxBuffer)
{
  MSG oMsg;
  if( oMsg.enParse( poRxBuffer) == enErrorCode_No_Error)
  {
    /*======================================================================
     * send 'normal' messages to the notified clients
     *--------------------------------------------------------------------*/
    oMsg.vTrace();

    list< tun_CallBack*>* poSubscriberList = oMsg.poGetSubscriberList();

    for( list< tun_CallBack*>::iterator iter = poSubscriberList->begin(); iter != poSubscriberList->end(); ++iter)
    {
      if( (*iter)->_bActive == TRUE)
      {
        (*iter)->vCall( &oMsg);
      }
    }
    /*====================================================================*/
  }
  else
  {
    /*======================================================================
     * send 'error' messages to the notified clients
     *--------------------------------------------------------------------*/
    trMsg_Generic_Error oErrMsg;
    (tVoid)oErrMsg.enParse( poRxBuffer);
    oErrMsg.vTrace();

    list< tun_CallBack*>* poSubscriberList    = oMsg.poGetSubscriberList();
    list< tun_CallBack*>* poErrSubscriberList = oErrMsg.poGetSubscriberList();

    for( list< tun_CallBack*>::iterator iter = poSubscriberList->begin(); iter != poSubscriberList->end(); ++iter)
    {
      /* client has this message subscribed */
      for( list< tun_CallBack*>::iterator iterErr = poErrSubscriberList->begin(); iterErr != poErrSubscriberList->end(); ++iterErr)
      {
        if( (*iter)->_bActive == TRUE)
        {
          /* client has an error callback subscribed */
          (*iterErr)->vCall( &oErrMsg);
        }
      }
    }
    /*====================================================================*/
  }
}


/*************************************************************************
*
* FUNCTION:     tun_MsgDispatcher::vParseAndTrace( )
*
* DESCRIPTION:  Parse the the message buffer, given by parameter,
*               do a trace.
*
* PARAMETER:    tun_SsiRxBuffer* poRxBuffer
*
* RETURNVALUE:  void
*
*************************************************************************/
template< class MSG>
tVoid tun_MsgDispatcher::vParseAndTrace( tun_SsiRxBuffer* poRxBuffer)
{
  MSG oMsg;
  if( oMsg.enParse( poRxBuffer) == enErrorCode_No_Error)
  {
    oMsg.vTrace();
  }
  else
  {
    /*======================================================================
     * Error message trace
     *--------------------------------------------------------------------*/
    trMsg_Generic_Error oErrMsg;
    (tVoid)oErrMsg.enParse( poRxBuffer);
    oErrMsg.vTrace();
    /*====================================================================*/
  }
}

#ifdef VARIANT_S_FTR_ENABLE_AARSPROXY
void tun_MsgDispatcher::vDispatchRUMessage(unsigned char *pu8ByteArray, int actualLength)
{
   tun_DrvAdrIf::instance()->INCDataIndication((tU8*)pu8ByteArray,(tU32)actualLength);
}

void tun_MsgDispatcher::vHandleSBRState(unsigned char SBRState)
{
	ETG_TRACE_USR1(("tun_MsgDispatcher::vHandleSBRState received"));
	tun_DrvAdrIf::instance()->vCallBackADRState(SBRState);
}

#endif

/*** END OF FILE *************************************************************/
