/************************************************************************
* FILE:           tun_MessageQ.cpp
* PROJECT:        g3g
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION:    Tuner message Que.
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------

* HISTORY:      
* Date      | Author                       | Modification
* 25.01.05  | CM-DI/ESA2 ( RBIN ) Dinesh   | Initial version.
* 25.04.13  | NGP1KOR    | First version of the G3g after porting 
                from NISSAN LCN2Kai
* 19.08.14  | VND4KOR    | Changed static user defined queue to dynamic STL queue
*************************************************************************/
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "tun_trace.h"
#include "tun_defines.h"
#include "tun_MessageQ.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_MSGQUE
#include "trcGenProj/Header/tun_MessageQ.cpp.trc.h"
#endif

/*************************************************************************
*
* FUNCTION:     tun_Message::tun_Message( )
* 
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
* AUTHOR:LSH1HC 16.05.2016
*************************************************************************/
tun_Message::tun_Message( )
{
    m_pu8Message = NULL;
    m_u32MessageLen = 0;
}

/*************************************************************************
*
* FUNCTION:     tun_Message::tun_Message( )
*
* DESCRIPTION:  Constructor -> allocates memory to members
*
* PARAMETER:     tU8* pu8Message, tU32 u32MessageLen
*
* RETURNVALUE:  void
*
* AUTHOR: VND4KOR 19.08.14/LSH1HC 16.05.2016
*************************************************************************/
tun_Message::tun_Message( tU8* pu8Message, tU32 u32MessageLen)
{
    m_pu8Message = NULL;
    m_u32MessageLen = 0;
    ETG_TRACE_USR1(( " tun_Message::tun_Message u32MessageLen = %d", u32MessageLen));
    if(pu8Message != NULL && u32MessageLen > 0)
    {
        /** create new array with size = u32MessageLen */
        m_pu8Message = new tU8[u32MessageLen];
        if(m_pu8Message != NULL)
        {
            /** Copy the contents of the message */
            /** parameter: destination , source, length*/
             OSAL_pvMemoryCopy( m_pu8Message, pu8Message ,u32MessageLen );
        }
        /** Save the message length */
        m_u32MessageLen = u32MessageLen;
    }
}

/*************************************************************************
*
* FUNCTION:     tun_Message::~tun_Message( )
* 
* DESCRIPTION:  Destructor -> deallocates memory assigned to members
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
* AUTHOR: VND4KOR 19.08.14/LSH1HC 16.05.2016
*************************************************************************/
tun_Message::~tun_Message( )
{
    ETG_TRACE_USR1(( " tun_Message::~tun_Message "));
    if(m_pu8Message  != NULL )
    {
        /** delete the memory allocated to array */
        delete [] m_pu8Message;
        m_pu8Message = NULL;
    }
}
/*************************************************************************
*
* FUNCTION:     tun_Message::tun_Message( )
*
* DESCRIPTION:  Copy constructor
*
* PARAMETER:
*
* RETURNVALUE:
*
* AUTHOR: LSH1HC 16.05.2016
*************************************************************************/
tun_Message::tun_Message(const tun_Message &obj)
{
    ETG_TRACE_USR1(( "tun_Message:Copy Constructor m_u32MessageLen = %d", obj.m_u32MessageLen));
    m_pu8Message = NULL;
    m_u32MessageLen = 0;
    if(obj.m_pu8Message != NULL && obj.m_u32MessageLen > 0)
    {
        m_pu8Message = new tU8[obj.m_u32MessageLen];
        if(m_pu8Message != NULL)
        {
             OSAL_pvMemoryCopy( m_pu8Message, obj.m_pu8Message , obj.m_u32MessageLen );
             m_u32MessageLen = obj.m_u32MessageLen;
        }
    }
}

/*************************************************************************
*
* FUNCTION:     tun_Message::tun_Message( )
*
* DESCRIPTION:  assignment operator
*
* PARAMETER:
*
* RETURNVALUE:
*
* AUTHOR: LSH1HC 16.05.2016
*************************************************************************/
tun_Message& tun_Message::operator=(const tun_Message &obj)
{
	if(this == &obj)
	{
		return *this;
	}
	delete[] m_pu8Message;
    ETG_TRACE_USR1(( "tun_Message:assignment operator m_u32MessageLen = %d", obj.m_u32MessageLen));
    m_pu8Message = NULL;
    m_u32MessageLen = 0;
    if(obj.m_pu8Message != NULL && obj.m_u32MessageLen > 0)
    {
        m_pu8Message = new tU8[obj.m_u32MessageLen];
        if(m_pu8Message != NULL)
        {
             OSAL_pvMemoryCopy( m_pu8Message, obj.m_pu8Message , obj.m_u32MessageLen );
             m_u32MessageLen = obj.m_u32MessageLen;
        }
    }
    return *this;
}

/*--------------------------------------*/
/* class tun_MessageQ implementation    */
/*--------------------------------------*/

/*************************************************************************
*
* FUNCTION:     tun_MessageQ::tun_MessageQ( )
* 
* DESCRIPTION:  Constructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
* AUTHOR: VND4KOR 19.08.14/LSH1HC 16.05.2016
*************************************************************************/
tun_MessageQ::tun_MessageQ( ): m_fWaitingForConfirmation( FALSE )
{
}

/*************************************************************************
*
* FUNCTION:     tun_MessageQ::~tun_MessageQ( )
* 
* DESCRIPTION:  Destructor -> removes all elements from the queue and 
*               deallocates memory assigned to them.
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
* AUTHOR: VND4KOR 19.08.14/LSH1HC 16.05.2016
*************************************************************************/
tun_MessageQ::~tun_MessageQ( )
{
    ETG_TRACE_USR1(( " tun_MessageQ::~tun_MessageQ "));
    /** Remove all elements from q */
    while(!m_SenderQueue.empty())
    {
        vRemoveFrontMsgFromQueue();
    }
}

/*************************************************************************
*
* FUNCTION:     tun_MessageQ::fGetNextMessage( )
* 
* DESCRIPTION:  sends the next message from the Queue
*
* PARAMETER:    tU8* pu8Message, tU32* pu32MessageLen
*
* RETURNVALUE:  tBool
*
* AUTHOR: VND4KOR 19.08.14/LSH1HC 16.05.2016
*************************************************************************/
tBool tun_MessageQ::fGetNextMessage( tU8* pu8Message, tU32* pu32MessageLen  )
{
    tBool fRet = FALSE;
    if(m_SenderQueue.empty())
    {
        ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_MSGQUE, ET_EN_T16 _ TUN_TRACE_MESSAGEQUE_GETNEXTMSG_FAIL _ ET_EN_DONE );
    }
    else
    {
        /*----------------------------------------------*/
        /* Get the pointer to first message from Q      */
        /*----------------------------------------------*/
        tun_Message & pMessage = m_SenderQueue.front();

        /*--- Copy Message length --*/
        *pu32MessageLen = pMessage.u32GetMessageLength();

        /*--- Copy Message data --*/
        OSAL_pvMemoryCopy( pu8Message, pMessage.u8GetMessageData(), *pu32MessageLen );

        ETG_TRACE_USR1(( " vCopyFrontMessageFromQ() Entered : m_u32MessageLen = %d", *pu32MessageLen));
        ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_MSGQUE, ET_EN_T16 _ TUN_TRACE_MESSAGEQUE_GETNEXTMSG_OK _ ET_EN_DONE );

        fRet = TRUE;
    }
    return fRet;
}

/*************************************************************************
*
* FUNCTION:     tun_MessageQ::vAddNewMessageInQ( )
* 
* DESCRIPTION:  Adds new message to the Queue.
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
* AUTHOR: VND4KOR 19.08.14/LSH1HC 16.05.2016
*************************************************************************/
tVoid tun_MessageQ::vAddNewMessageInQ( tU8* pu8Message, tU32 u32MessageLen, tU8 /*u8Priority*/)
{
    ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_MSGQUE, ET_EN_T16 _ TUN_TRACE_MESSAGEQUE_ADDNEWMSG _ ET_EN_DONE );
    /** Check for validity of message */
    if((pu8Message != NULL) && (u32MessageLen != 0))
    {
      tun_Message po_tun_Message(pu8Message, u32MessageLen);
      m_SenderQueue.push(po_tun_Message);

      ETG_TRACE_USR1(( " vAddNewMessageInQ()-> Queue Size : %u", m_SenderQueue.size()));
      ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_MSGQUE, ET_EN_T16 _ TUN_TRACE_MESSAGEQUE_PUTINQUE_OK _ ET_EN_DONE );
    }
    else
    {
        ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_MSGQUE, ET_EN_T16 _ TUN_TRACE_MESSAGEQUE_PUTINQUE_FAIL _ ET_EN_DONE );
        ETG_TRACE_USR1(( " vAddNewMessageInQ()->Message not added , length is 0"));
    }
}  

/*************************************************************************
*
* FUNCTION:     tun_MessageQ::vMessageSuccessfullySent( )
* 
* DESCRIPTION:  Message is successfully sent over SPI.
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
* AUTHOR: VND4KOR 19.08.14/LSH1HC 16.05.2016
*************************************************************************/
tVoid tun_MessageQ::vMessageSuccessfullySent( )
{
    ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_MSGQUE, ET_EN_T16 _ TUN_TRACE_MESSAGEQUE_MSGSENTSUCCESS _ ET_EN_DONE );
    /** Check if message was sent previously */
    if( fCheckWaitingForConfirmation( ))
    {
        /* Message is sent successfully to RU, new message can be sent now.*/
        m_fWaitingForConfirmation = FALSE;
        /* Remove Front message from the Queue */
        vRemoveFrontMsgFromQueue( );
        ETG_TRACE_USR1(( "vMessageSuccessfullySent: Ready to handle next message"));
    }
    else
    {
      ETG_TRACE_USR1(( " vMessageSuccessfullySent: The previous message is still going on"));
    }
}

/*************************************************************************
*
* FUNCTION:     tun_MessageQ::vRemoveFrontMsgFromQueue( )
* 
* DESCRIPTION:  Front message in the queue will be deleted.
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
* AUTHOR: VND4KOR 19.08.14/LSH1HC 16.05.2016
*************************************************************************/
tVoid tun_MessageQ::vRemoveFrontMsgFromQueue( )
{
    /** Check if message queue is empty */
    if(!m_SenderQueue.empty())
    {
        m_SenderQueue.pop();
    }
    else
    {
        ETG_TRACE_USR1(( " vRemoveFrontMsgFromQueue() : Q empty, cannot remove message"));
    }
    /** printing Queue Size */
    ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_MSGQUE, ET_EN_T16 _ TUN_TRACE_MESSAGEQUE_INCRHEADER _ ET_EN_T8 _ m_SenderQueue.size() _ ET_EN_DONE );
}

/*************************************************************************
*
* FUNCTION:     tun_MessageQ::vClearQueue( )
*
* DESCRIPTION: clearing all message in queue
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
* AUTHOR: LSH1HC 07.06.2016
*************************************************************************/
tVoid tun_MessageQ::vClearQueue( )
{
    ETG_TRACE_USR1(( " tun_MessageQ::vClearQueue()"));
    while(!m_SenderQueue.empty())
    {
        m_SenderQueue.pop();
    }
    m_fWaitingForConfirmation = FALSE;
}

/*************************************************************************
*
* FUNCTION:  tun_MessageQ::fCheckWaitingForConfirmation( )
* 
* DESCRIPTION:
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tBool tun_MessageQ::fCheckWaitingForConfirmation(  )const
{
    if( m_fWaitingForConfirmation )
    {
        ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_MSGQUE, ET_EN_T16 _ TUN_TRACE_MESSAGEQUE_WAITINGFORCONF _ ET_EN_DONE );
    }
    else
    {
        ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_MSGQUE, ET_EN_T16 _ TUN_TRACE_MESSAGEQUE_NOT_WAITINGFORCONF _ ET_EN_DONE );
    }
    return m_fWaitingForConfirmation;
}

/*************************************************************************
*
* FUNCTION:  tun_MessageQ::vWaitingForConfirmation( )
* 
* DESCRIPTION:
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MessageQ::vWaitingForConfirmation()
{
    ETG_TRACE_USR1(( " vSetWaitingForConfirmationTrue()"));
    m_fWaitingForConfirmation = TRUE;
}

/*************************************************************************
*
* FUNCTION:  tun_MessageQ::vWaitingForConfirmation( )
*
* DESCRIPTION:
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MessageQ::vResetWaitingForConfirmation(  )
{
    ETG_TRACE_USR1(( " vResetWaitingForConfirmation()"));
    m_fWaitingForConfirmation = FALSE ;
}
