/* FILE:           tun_ActualData.cpp
* PROJECT:        Ford H/L RNS
* SW-COMPONENT:   Tuner
*----------------------------------------------------------------------
*
* DESCRIPTION:    Actual Data of Tuner: Actual frequency, band, feature
                  will be contained in this class.
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:      
* Date      | Author                       | Modification
* 23.02.05  | CM-DI/ESA2 ( RBIN ) Dinesh   | Initial version.
* 25.04.13  | NGP1KOR    | First version of the G3g after porting 
				from NISSAN LCN2Kai
*************************************************************************/





#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*****************************************************************************
 * ETG trace
 * 0x2609:    TUN_TRACE_CLASS_ACTDATA
 ****************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_ACTDATA
#include "trcGenProj/Header/tun_ActualData.cpp.trc.h"
#endif

#include "tun_Utility.h"

#include "tun_trace.h"


#include "tun_ActualData.h"

#include "tun_HMIManager.h"

#include "tun_MsgDefines.h"

#include "tun_defines.h"

#include "tun_Msg_types.h"

#include "tun_DrvAdrIf.hpp"

#include "tun_MsgToHMI.h"
#include "kdsconfig/clConfigInterface.h"
#include "AMFMTuner/clTmcTuner_Method_S_JAPAN_SET_AREA.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_ACTDATA
#include "trcGenProj/Header/tun_ActualData.cpp.trc.h"
#endif


/*************************************************************************
*
* FUNCTION:     tun_ActualData::tun_ActualData( )
* 
* DESCRIPTION:  Constructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_ActualData::tun_ActualData( ):m_potun_Utility( NULL ),\
m_potun_MsgToHMI(NULL),\
m_potun_MsgToADR(NULL)
{
  m_fIsLowVoltageOnStartup = FALSE;
  m_fIsLowVoltageEnd = FALSE;
  m_bIsVdamfmaudAvailable = FALSE;
  m_bIsTunerShutdown = FALSE;
  m_bIsTmActiveBeforeLowVoltage = FALSE;  
  m_potun_HMIManager = NULL;
  m_poclConfigInterface = NULL;
  m_poclTmcTuner_Method_S_JAPAN_SET_AREA = NULL;
}



/*************************************************************************
*
* FUNCTION:     tun_ActualData::~tun_ActualData( )
* 
* DESCRIPTION:  Constructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_ActualData::~tun_ActualData( )
{
  m_potun_Utility = NULL;
  m_potun_MsgToHMI = NULL;
  m_potun_MsgToADR = NULL;
  m_potun_HMIManager = NULL;
  m_poclConfigInterface = NULL;
}



/*************************************************************************
*
* FUNCTION:     tun_ActualData::vSet_tun_Utility_Ptr( )
* 
* DESCRIPTION:  
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_ActualData::vSet_tun_Utility_Ptr( tun_Utility* potun_Utility )
{
  if( potun_Utility != NULL )
  {  
    m_potun_Utility = potun_Utility;
  }
  else
  {
    /* Assert to do */
  }
}

/*************************************************************************
*
* FUNCTION:     tun_ActualData::vSet_tun_MsgToADR_Ptr( )
*
* DESCRIPTION:
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_ActualData::vSet_tun_MsgToADR_Ptr( tun_MsgToADR* potun_MsgToADR )
{
  if( potun_MsgToADR != NULL )
  {
    m_potun_MsgToADR = potun_MsgToADR;
  }
  else
  {
    /* Assert to do */
  }
}
    
/*************************************************************************
*
* FUNCTION:     tun_ActualData::vSet_tun_MsgToHMI_Ptr( )
*
* DESCRIPTION:
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_ActualData::vSet_tun_MsgToHMI_Ptr( tun_MsgToHMI* potun_MsgToHMI )
{
  if( potun_MsgToHMI != NULL )
  {
    m_potun_MsgToHMI = ( tun_MsgToHMI* )potun_MsgToHMI;
  }
  else
  {
    /* Assert to do */
  }
}

/*************************************************************************
*
* FUNCTION:     tun_ActualData::vSet_tun_HMIManager_Ptr( )
*
* DESCRIPTION:
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_ActualData::vSet_tun_HMIManager_Ptr(tun_HMIManager* potun_HMIManager)
{
  if( potun_HMIManager != NULL )
  {
     m_potun_HMIManager = ( tun_HMIManager* )potun_HMIManager;
  }
  else
  {
    /* Assert to do */
  }
}


/*************************************************************************
*
* FUNCTION:     tun_ActualData::vUpdateData( )
* 
* DESCRIPTION:  Update the actual tuner data.
*
* PARAMETER:    tU8* pu8NewData.
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_ActualData::vUpdateDataFromRU( tU8* pu8NewData, tU32 u32MessageLen )
{
  
  // Validate input parameters, pointers...
  
  if( m_potun_Utility == NULL )
  {
    NORMAL_M_ASSERT( m_potun_Utility != NULL ); 
	return;
  }

  if( m_potun_MsgToHMI == NULL )
  {
    NORMAL_M_ASSERT( m_potun_MsgToHMI != NULL );
	return;
  }


  if( pu8NewData == NULL )
  {
    NORMAL_M_ASSERT( pu8NewData != NULL );
	return;
  }
  if( u32MessageLen == 0 )
  {
    NORMAL_M_ASSERT( u32MessageLen != 0 );
	return;
  }

  /* check for the Background tuner responses   */
  /* If response is for the Background tuner    */
  /* the response wont be updated into Act data */

  if(true == bCheckforBackgroundTuner(pu8NewData ,u32MessageLen))
  {
    ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_ACTDATA, ET_EN_T8 _ 0xF0 _
                                                ET_EN_DONE );
    //Do not update the Act data
    return;
  }



    switch( TUN_DRVADRIF_GET_U16( &pu8NewData[ enAdrMsgOffset_FKT_ID]) )//Function ID

    {
      /*--------------------------------------------------------------------*/
     
      case enMsg_CM_Startup:
        {
          m_otun_Data_MainStatus.vSetStatus( TUN_COMPONENT_STATUS_ACTIVE);
        }
        break;

      /*--------------------------------------------------------------------*/

      case enMsg_CM_ATStationInfo:
        {
          if(((m_potun_MsgToHMI->bIsTunerFG()) || (m_potun_MsgToHMI->u8GetAnnoMessageType() & TUN_ANNOUNCEMENTMESSAGE_BIT_MSK_EON)
#if defined(VARIANT_S_FTR_ENABLE_FEATURE_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC)//gds5kor - NCG3D-38335
					|| ( m_potun_MsgToHMI->bIsFMAnnouFG())
#endif
					)
        		  && (m_potun_MsgToHMI->m_u8TunerState == 0))
          {
			  vUpdateDataFromRUExt(pu8NewData);
          }
        }
        break;
     default:
       {
       }
       break;
    };


  

  

}

tVoid tun_ActualData::vUpdateDataFromRUExt(tU8* pu8NewData)
{

   if(m_potun_Utility == NULL)
    {
       NORMAL_M_ASSERT( m_potun_Utility != NULL );
       return;
    }

   /*Tunerheader*/
   m_otun_Data_Station.vSetTunerHeader(pu8NewData[enAdrMsgOffset_INSTANCE_ID]);

   /*u32AvailFlags*/
   //data not available

   /*u32Frequency*/
   tU32 u32Frequency = TUN_DRVADRIF_GET_U32( &pu8NewData[ enCM_ATStationInfo_idx_Frequency] );
   m_otun_Data_Station.vSetFrequency( u32Frequency );

   /*u16PI*/
   m_otun_Data_Station.vSetPI( TUN_DRVADRIF_GET_U16( &pu8NewData[ enCM_ATStationInfo_idx_PI] ) );

   /*sPSName*/
   m_otun_Data_Station.vSetPSName( (tChar*)&pu8NewData[ enCM_ATStationInfo_idx_Stationname] );

   /*e8Band*/
   tU8 u8Band = pu8NewData[ enCM_ATStationInfo_idx_Band];
 #ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
            if(m_potun_MsgToADR == NULL)
            {
               NORMAL_M_ASSERT( m_potun_MsgToADR != NULL );
               return;
            }
            u8Band = m_potun_MsgToADR->u8ConvertCombibandToTunerband(u8Band, u32Frequency);
 #endif
   m_otun_Data_Station.vSetBand( u8Band );

   /*e8AtStationTPinfo*/
   m_otun_Data_Station.vSetTPInfo(pu8NewData[ enCM_ATStationInfo_idx_ATStationTpInfo]);

   /*u16PresetListID*/
   tU16 u16PresetListID = m_potun_Utility->u16Convert2SeparateBytesTo16Bit( &pu8NewData[ enCM_ATStationInfo_idx_ActivePresetList] );
   m_otun_Data_Station.vSetStaticListID( u16PresetListID );

   /*u8ActiveStationList*/
   m_otun_Data_Station.vSetConfigListID(pu8NewData[ enCM_ATStationInfo_idx_ActiveStationList]);

   /*u16StaticListElementID*/
   tU32 u16Preset = m_potun_Utility->u16Convert2SeparateBytesTo16Bit( &pu8NewData[ enCM_ATStationInfo_idx_PresetAutocompare] );
   m_otun_Data_Station.vSetStaticListElementID((tU16)(u16Preset));

   /*u16ConfigListElementID*/
   tU32 u16StationListIndex = m_potun_Utility->u16Convert2SeparateBytesTo16Bit( &pu8NewData[ enCM_ATStationInfo_idx_StationListAutocompare] );
   m_otun_Data_Station.vSetConfigListElementID((tU16)(u16StationListIndex));

   /*u8AtStationInfoBits*/
   m_otun_Data_Station.vSetFrequencyQualification(pu8NewData[ enCM_ATStationInfo_idx_ATStationInfo]);

   /*u8PTY*/
   m_otun_Data_Station.vSetPTY( pu8NewData[ enCM_ATStationInfo_idx_PTY]);

   m_otun_Data_Station.vSetBaseFrequency( TUN_DRVADRIF_GET_U32( &pu8NewData[ enCM_ATStationInfo_idx_Frequency] ));  // lsh2hi tbd.


#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
   /*u8FieldStrngth*/
   m_otun_Data_Station.vSetFieldStrngth(pu8NewData[enCM_ATStationInfo_idx_Fieldstrength]);

   /*u8Quality*/
   m_otun_Data_Station.vSetQuality(pu8NewData[enCM_ATStationInfo_idx_Quality]);

   vUpdateHDData(pu8NewData);

   vUpdateDRMData(pu8NewData);


#endif

}

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
tVoid tun_ActualData::vUpdateHDData(tU8* pu8NewData)
{

   /*b8HDStationInfoBits*/
   m_otun_Data_Station.vSetHDStationInfoBits(pu8NewData[ enCM_ATStationInfo_idx_HDStationInfoBits ]);

   /*u32StationInfoHDStationID*/
   m_otun_Data_Station.vSetStationInfoHDStationID(TUN_DRVADRIF_GET_U32(&pu8NewData[ enCM_ATStationInfo_idx_HDStationId ]));

   /*sHDCallSign*/
   m_otun_Data_Station.vSetHDCallSign( (tChar*)&pu8NewData[ enCM_ATStationInfo_idx_HDCallSign] );

   /*e8HDCallSignTextEncoding*/
   m_otun_Data_Station.vSetHDCallSignTextEncoding(pu8NewData[ enCM_ATStationInfo_idx_HDCallsignTextEncoding ]);

   /*u8HDStationGenre*/
   m_otun_Data_Station.vSetHDStationGenre(pu8NewData[ enCM_ATStationInfo_idx_HDStationGenre ]);

   /*b8HDAudioPrgmAvail*/
   m_otun_Data_Station.vSetHDAudioPrgmAvail(pu8NewData[ enCM_ATStationInfo_idx_HDNumberOfAudioPrograms ]);

   /*u8HDQuality*/
   m_otun_Data_Station.vSetHDQuality(pu8NewData[ enCM_ATStationInfo_idx_HDQuality ]);

   /*e8ActiveHDAudPrgm*/
   m_otun_Data_Station.vSetActiveHDAudPrgm(pu8NewData[ enCM_ATStationInfo_idx_HDActiveAudioProgram ]);

}


tVoid tun_ActualData::vUpdateDRMData(tU8* pu8NewData)
{

   /*b8DRMStationInfoBits*/
   m_otun_Data_Station.vSetDRMStationInfoBits(pu8NewData[ enCM_ATStationInfo_idx_DRMStationInfoBits ]);

   /*u32DRMServiceId*/
   m_otun_Data_Station.vSetDRMServiceId(TUN_DRVADRIF_GET_U32(&pu8NewData[ enCM_ATStationInfo_idx_DRMServiceId ]) );

   /*sDRMServiceLabel*/
   m_otun_Data_Station.vSetDRMServiceLabel( (tChar*)&pu8NewData[ enCM_ATStationInfo_idx_DRMServiceLabel] );

   /*u8DRMPTY*/
   m_otun_Data_Station.vSetDRMPTY(pu8NewData[ enCM_ATStationInfo_idx_DRMPTY ]);

   /*b16DRMAnnouBits*/
   m_otun_Data_Station.vSetDRMAnnouBits(TUN_DRVADRIF_GET_U16(&pu8NewData[ enCM_ATStationInfo_idx_DRMAnnoSupport]));

   /*b8DRMNumberOfAudioServices*/
   m_otun_Data_Station.vSetDRMNumberOfAudioServices(pu8NewData[ enCM_ATStationInfo_idx_DRMNumberOfAudioServices ]);

   /*u8DRMQuality*/
   m_otun_Data_Station.vSetDRMQuality(pu8NewData[ enCM_ATStationInfo_idx_DRMQuality ]);

   /*u8DRMServiceShortId*/
   m_otun_Data_Station.vSetDRMServiceShortId(pu8NewData[ enCM_ATStationInfo_idx_DRMServiceShortId ]);

   /*u8Drmlanguage*/
   m_otun_Data_Station.vSetDrmlanguage(pu8NewData[ enCM_ATStationInfo_idx_Drmlanguage ]);

   /*u32DrmlanguageCode*/
   m_otun_Data_Station.vSetDrmlanguageCode(TUN_DRVADRIF_GET_U32(&pu8NewData[ enCM_ATStationInfo_idx_DrmlanguageCode] ));

   /*u16DrmCountryCode*/
   m_otun_Data_Station.vSetDrmCountryCode(TUN_DRVADRIF_GET_U16(&pu8NewData[ enCM_ATStationInfo_idx_DrmCountryCode]));
   
   /*s8DrmFrequencyCorrection*/
   m_otun_Data_Station.vSetDrmFrequencyCorrection(pu8NewData[ enCM_ATStationInfo_idx_DrmFrequencyCorrection]);

}

#endif


tVoid tun_ActualData::vPackStationInfoData(midw_tunerfi_tclMsgFID_TUN_G_ATSTATIONINFOStatus &oStatusMsg )
{
   if((m_potun_HMIManager == NULL) || (m_potun_MsgToHMI == NULL) || (m_poclTmcTuner_Method_S_JAPAN_SET_AREA == NULL) )
   {
      NORMAL_M_ASSERT( m_potun_HMIManager != NULL );
      NORMAL_M_ASSERT( m_potun_MsgToHMI != NULL );
      NORMAL_M_ASSERT( m_poclTmcTuner_Method_S_JAPAN_SET_AREA != NULL);
      return;
   }
   /*u8TunerHeader*/
   oStatusMsg.Atstationinfo.u8TunerHeader.enType =
         (midw_fi_tcl_e8_Tun_TunerHeader::tenType)m_otun_Data_Station.u8GetTunerHeader();

   /*u32AvailFlags - Data not available*/

   /** Frequency */
   oStatusMsg.Atstationinfo.u32Frequency =  m_otun_Data_Station.u32GetFrequency();

   /** PI */
   oStatusMsg.Atstationinfo.u16PI = m_otun_Data_Station.u16GetPI();


   midw_fi_tclString sUtf8String;
   #ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20 
   if(m_poclTmcTuner_Method_S_JAPAN_SET_AREA->bIsJapanRegion())
   {
      oStatusMsg.Atstationinfo.sPSName = m_potun_MsgToHMI->sGetStationName();
   }
   else
   #endif
   {
     /*sPSName*/
     const tChar* po_tempPSName =  m_otun_Data_Station.pcocGetPSName();

     m_potun_HMIManager -> vConvertRDSPSNametotString((const tU8 *)po_tempPSName,&sUtf8String) ;
        oStatusMsg.Atstationinfo.sPSName.bSet( sUtf8String ,
        midw_fi_tclString::FI_EN_UTF8
        );
   }


	  if(oStatusMsg.Atstationinfo.u16PI != 0)
	  {
		  m_potun_MsgToHMI->vGetStationLogoPath(oStatusMsg.Atstationinfo.u16PI , sUtf8String);
		  oStatusMsg.Atstationinfo.StationLogo.bSet(  sUtf8String,
	  											 midw_fi_tclString::FI_EN_UTF8);
	  }


	ETG_TRACE_USR2((  "vPackStationInfoData(), StationLogo=%s",oStatusMsg.Atstationinfo.StationLogo.szGet(midw_fi_tclString::FI_EN_UTF8)));
   /** Band */
   oStatusMsg.Atstationinfo.e8Band.enType =
      (midw_fi_tcl_e8_Tun_TunerBand::tenType)m_otun_Data_Station.u8GetBand();

   /** TP info */
   oStatusMsg.Atstationinfo.e8AtStationTPinfo.enType =
         (midw_fi_tcl_e8_AtStationTPinfo::tenType)m_otun_Data_Station.u8GetTPInfo();

   /** Preset List Id */
   oStatusMsg.Atstationinfo.u16PresetListID =m_otun_Data_Station.u16GetStaticListID();

   /** u8ActiveStationList */
   oStatusMsg.Atstationinfo.u8ActiveStationList = m_otun_Data_Station.u8GetConfigListID();

   /** u16StaticListElementID */
   oStatusMsg.Atstationinfo.u16StaticListElementID = m_otun_Data_Station.u16GetStaticListElementID();

   /** u16ConfigListElementID */
   oStatusMsg.Atstationinfo.u16ConfigListElementID = m_otun_Data_Station.u16GetConfigListElementID();

   /*u8AtStationInfoBits*/
   oStatusMsg.Atstationinfo.u8AtStationInfoBits.u8Value =  m_otun_Data_Station.vGetFrequencyQualification();

   /*u8PTY*/
   oStatusMsg.Atstationinfo.u8PTY=  m_otun_Data_Station.u8GetPTY();

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
   /*u8FieldStrngth*/
   oStatusMsg.Atstationinfo.u8FieldStrngth=  m_otun_Data_Station.u8GetFieldStrength();

   /*u8Quality*/
   oStatusMsg.Atstationinfo.u8Quality=  m_otun_Data_Station.u8GetQuality();

   vUpdateHDStationData(oStatusMsg);

   vUpdateDRMStationData(oStatusMsg);
#endif
}

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
tVoid tun_ActualData::vUpdateHDStationData(midw_tunerfi_tclMsgFID_TUN_G_ATSTATIONINFOStatus &oStatusMsg)
{

   ETG_TRACE_USR4(( " tun_ActualData :: vUpdateHDStationData Data "));

   /*b8HDStationInfoBits*/
   oStatusMsg.Atstationinfo.b8HDStationInfoBits.u8Value= m_otun_Data_Station.u8GetHDStationInfoBits();

   /*u32StationInfoHDStationID*/
   oStatusMsg.Atstationinfo.u32StationInfoHDStationID= m_otun_Data_Station.u32GetStationInfoHDStationID();
   
   /*e8ActiveHDAudPrgm*/
   oStatusMsg.Atstationinfo.e8ActiveHDAudPrgm.enType = \
         (midw_fi_tcl_e8_Tun_HDAudioProgram::tenType)m_otun_Data_Station.u8GetActiveHDAudPrgm();
	/*HD station logo */
	if((m_poclConfigInterface != NULL) && (m_potun_MsgToHMI !=NULL))
	{
		  if(m_poclConfigInterface->u32GetConfigData("HDSupported") == true)
		  {
			  if(oStatusMsg.Atstationinfo.u32StationInfoHDStationID != 0x00 && (m_potun_MsgToHMI->bIsFMDemodulationON(oStatusMsg.Atstationinfo.e8Band.enType) || m_potun_MsgToHMI->bIsAMDemodulationON(oStatusMsg.Atstationinfo.e8Band.enType)) )
			  {
				  m_potun_MsgToHMI->vGetActiveHDStationLogoPath(oStatusMsg.Atstationinfo.StationLogo, oStatusMsg.Atstationinfo.u32StationInfoHDStationID, (tU8)oStatusMsg.Atstationinfo.e8ActiveHDAudPrgm.enType);
			  }
		  }
	}
	ETG_TRACE_USR2((  "vUpdateHDStationData(), StationLogo=%s",oStatusMsg.Atstationinfo.StationLogo.szGet(midw_fi_tclString::FI_EN_UTF8)));
   /*e8HDCallSignTextEncoding*/
   oStatusMsg.Atstationinfo.e8HDCallSignTextEncoding.enType=  \
   (midw_fi_tcl_e8_Tun_HD_TextEncodingType::tenType) m_otun_Data_Station.u8GetHDCallSignTextEncoding();

   /*u8HDStationGenre*/
   oStatusMsg.Atstationinfo.u8HDStationGenre = m_otun_Data_Station.u8GetHDStationGenre();

   /*b8HDAudioPrgmAvail*/
   oStatusMsg.Atstationinfo.b8HDAudioPrgmAvail.u8Value = m_otun_Data_Station.u8GetHDAudioPrgmAvail();

   /*u8HDQuality*/
   oStatusMsg.Atstationinfo.u8HDQuality = m_otun_Data_Station.u8GetHDQuality();
}

tVoid tun_ActualData::vUpdateDRMStationData(midw_tunerfi_tclMsgFID_TUN_G_ATSTATIONINFOStatus &oStatusMsg)
{
   if(m_potun_HMIManager == NULL)
   {
      NORMAL_M_ASSERT( m_potun_HMIManager != NULL );
      return;
   }
   ETG_TRACE_USR4(( " tun_ActualData :: vUpdateDRMStationData Data "));

   /*b8DRMStationInfoBits*/
   oStatusMsg.Atstationinfo.b8DRMStationInfoBits.u8Value =m_otun_Data_Station.u8GetDRMStationInfoBits();

   /*u32DRMServiceId*/
   oStatusMsg.Atstationinfo.u32DRMServiceId = m_otun_Data_Station.u32GetDRMServiceId();

   /*sDRMServiceLabel*/
   const tChar* po_tempServiceLabel =  m_otun_Data_Station.pcocGetDRMServiceLabel();
   midw_fi_tclString sUtf8StringLabel;
   m_potun_HMIManager -> vConvertRDSPSNametotString((const tU8 *)po_tempServiceLabel,&sUtf8StringLabel) ;
   oStatusMsg.Atstationinfo.sDRMServiceLabel.bSet( sUtf8StringLabel ,  midw_fi_tclString::FI_EN_UTF8 );

   /*u8DRMPTY*/
   oStatusMsg.Atstationinfo.u8DRMPTY = m_otun_Data_Station.u8GetDRMPTY();

   /*b16DRMAnnouBits*/
   oStatusMsg.Atstationinfo.b16DRMAnnouBits.u16Value =m_otun_Data_Station.u16GetDRMAnnouBits();

   /*b8DRMNumberOfAudioServices*/
   oStatusMsg.Atstationinfo.b8DRMNumberOfAudioServices.u8Value =m_otun_Data_Station.u8GetDRMNumberOfAudioServices();

   /*u8DRMQuality*/
   oStatusMsg.Atstationinfo.u8DRMQuality =m_otun_Data_Station.u8GetDRMQuality();

   /*u8DRMServiceShortId*/
   oStatusMsg.Atstationinfo.u8DRMServiceShortId = m_otun_Data_Station.u8GetDRMStationInfoBits();

   /*u8Drmlanguage*/
   oStatusMsg.Atstationinfo.u8Drmlanguage =m_otun_Data_Station.u8GetDrmlanguage();

   /*u32DrmlanguageCode*/
   oStatusMsg.Atstationinfo.u32DrmlanguageCode = m_otun_Data_Station.u32GetDrmlanguageCode();

   /*u16DrmCountryCode*/
   oStatusMsg.Atstationinfo.u16DrmCountryCode = m_otun_Data_Station.u16GetDrmCountryCode();
   
   /*s8DrmFrequencyCorrection*/
   oStatusMsg.Atstationinfo.s8DrmFrequencyCorrection = m_otun_Data_Station.s8GetDrmFrequencyCorrection();
}

#endif
/*----------------------------------------------*/
/*  class tun_Data_MainStatus implementation  */
/*----------------------------------------------*/



/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::tun_Data_MainStatus( )
* 
* DESCRIPTION:  Constructor .
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Data_MainStatus::tun_Data_MainStatus (  ): m_u8Status(0), \
                                               m_u8AudioSinkTuner(0),\
                                               m_u8BackgroundModeTuner(0),\
                                               m_ou8TunerHeader(0),\
                                               m_ou8Band(0xff),\
                                               m_ou8Autostore(0),\
                                               m_ou8Mode(0),\
                                               m_ou8State(0),\
                                               m_ou8ActType(0),\
                                               m_ou8Announcement(0),\
                                               m_ou8LastActionState(0),\
                                               m_ou32Features(0),\
                                               m_ou32AvailFlags(0),\
                                               m_ou8StaticListID(0xff),\
                                               m_ou8StaticListElementID(0),\
                                               m_ou8ConfigListID(0),\
                                               m_ou8ConfigListElementID(0),\
                                               m_u32SystemState(0)
                                                
{    
    
    
  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::~tun_Data_MainStatus (  )
* 
* DESCRIPTION:  Destructor .
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Data_MainStatus::~tun_Data_MainStatus (  )
{    
    
    
  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8AudioSinkTuner ( )
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetAudioSinkTuner () const 
{ 
    return m_u8AudioSinkTuner; 
}   


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetBackgroundModeTuner ( )
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetBackgroundModeTuner () const
{ 
    return m_u8BackgroundModeTuner; 
}  



/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetStatus ( )
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetStatus () const 
{ 
    return m_u8Status; 
}  

/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetTunerHeader ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetTunerHeader () const 
{ 
    return m_ou8TunerHeader; 
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetBand ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetBand () const 
{  
    return m_ou8Band;  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetAutostore ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetAutostore () const 
{  
    return m_ou8Autostore;  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetMode ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetMode () const 
{  
    return m_ou8Mode;  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetState ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetState () const 
{  
    return m_ou8State;
  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetActType ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetActType () const 
{  
    return m_ou8ActType;
  
}

/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetAnnouncement ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetAnnouncement () const 
{
  
    return m_ou8Announcement;
  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetLastActionState ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetLastActionState () const 
{  
    return m_ou8LastActionState;  
}



/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u32GetFeatures ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU32 tun_Data_MainStatus::u32GetFeatures () const 
{  
    return m_ou32Features;
  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u32GetAvailFlags ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU32 tun_Data_MainStatus::u32GetAvailFlags () const 
{  
    return m_ou32AvailFlags;
  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetStaticListID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetStaticListID () const 
{  
    return m_ou8StaticListID;  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetStaticListElementID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetStaticListElementID () const 
{  
    return m_ou8StaticListElementID;  
}




/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetConfigListID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetConfigListID () const 
{
  
   return m_ou8ConfigListID;
  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetConfigListElementID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_MainStatus::u8GetConfigListElementID () const 
{
 
   return m_ou8ConfigListElementID;  
  
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetTunerHeader ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetTunerHeader (tU8 u8TunerHeader)
{
    m_ou8TunerHeader = u8TunerHeader;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Monitor_Station::vSetClient ()
* 
* DESCRIPTION:  
*
* PARAMETER:    Client
*
* RETURNVALUE:  void 
*
*************************************************************************/
tVoid tun_Data_Monitor_Station::vSetClient (tU8 u8Client)
{
    m_ou8Client = u8Client;
	
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Monitor_Station::vSetMode ()
* 
* DESCRIPTION:  
*
* PARAMETER:    Mode
*
* RETURNVALUE:  void 
*
*************************************************************************/
tVoid tun_Data_Monitor_Station::vSetMode (tU8 u8Mode)
{
    m_ou8Mode = u8Mode;
	
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Monitor_Station::vSetQuality ()
* 
* DESCRIPTION:  
*
* PARAMETER:    Quality
*
* RETURNVALUE:  void 
*
*************************************************************************/
tVoid tun_Data_Monitor_Station::vSetQuality (tU8 u8Quality)
{
    m_ou8Quality = u8Quality;
	
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Monitor_Station::vSetPI (tU32 u32PI) 
* 
* DESCRIPTION:  
*
* PARAMETER:    PI
*
* RETURNVALUE:  void rrb1kor
*
*************************************************************************/
tVoid tun_Data_Monitor_Station::vSetPI (tU32 u32PI)
{
    m_ou32PI = u32PI;
	
}




/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetStatus ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetStatus (tU8 u8Status )
{
    m_u8Status = u8Status;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetAudioSinkTuner ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetAudioSinkTuner (tU8 u8AudioSinkTuner )
{
    m_u8AudioSinkTuner = u8AudioSinkTuner;
}



/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetBackgroundModeTuner ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetBackgroundModeTuner (tU8 u8BackgroundModeTuner )
{
    m_u8BackgroundModeTuner = u8BackgroundModeTuner;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetBand ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetBand (tU8 u8Band) 
{
    m_ou8Band = u8Band;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetAutostore ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetAutostore (tU8 u8Autostore) 
{
   m_ou8Autostore = u8Autostore;
}



/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetMode ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetMode (tU8 u8Mode) 
{
   m_ou8Mode = u8Mode;
}



/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetState  ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetState (tU8 u8State) 
{
    m_ou8State = u8State;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetActType  ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetActType (tU8 u8ActType) 
{
    m_ou8ActType = u8ActType;
}



/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetAnnouncement ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetAnnouncement (tU8 u8Announcement) 
{
    m_ou8Announcement = u8Announcement;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetLastActionState ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetLastActionState (tU8 u8LastActionState) 
{
  m_ou8LastActionState = u8LastActionState;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetFeatures ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetFeatures (tU32 u32Features) 
{
   m_ou32Features = u32Features;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetAvailFlags ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetAvailFlags (tU32 u32AvailFlags) 
{
   m_ou32AvailFlags = u32AvailFlags;
}




/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetStaticListID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetStaticListID (tU8 u8StaticListID) 
{
   m_ou8StaticListID = u8StaticListID;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetStaticListID ()
* 
* DESCRIPTION:  Overloaded funtion to store static list id during startup,depending 
				on u8BackGroundTuner parameter
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetStaticListID (tU8 u8StaticListID,tU8 u8BackGroundTuner) 
{
	if ( u8BackGroundTuner == TUN_TEMP_FM )
	{
		m_ou8StaticListID = 0xff;
	}
	else
	{
		m_ou8StaticListID = u8StaticListID;
	}

}
/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetStaticListElementID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetStaticListElementID (tU8 u8StaticListElementID) 
{
	
    m_ou8StaticListElementID = u8StaticListElementID;
	

}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetStaticListElementID ()
* 
* DESCRIPTION:  Overloaded funtion to store static list Element id during startup,depending 
				on u8BackGroundTuner parameter
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetStaticListElementID (tU8 u8StaticListElementID,tU8 u8BackGroundTuner) 
{
	if ( u8BackGroundTuner == TUN_TEMP_FM )
	{
		m_ou8StaticListElementID = 0xff;
	}
	else
	{
    m_ou8StaticListElementID = u8StaticListElementID;
	}

}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetConfigListID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetConfigListID (tU8 u8ConfigListID) 
{
   m_ou8ConfigListID = u8ConfigListID;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetConfigListElementID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_MainStatus::vSetConfigListElementID (tU8 u8ConfigListElementID) 
{
    m_ou8ConfigListElementID = u8ConfigListElementID;
}












/*--------------------------------------------*/
/*  class tun_Data_Station implementation     */
/*--------------------------------------------*/


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::tun_Data_Station( )
* 
* DESCRIPTION:  Constructor.
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Data_Station::tun_Data_Station ( ): m_ou8TunerHeader(0),\
		                                m_ou32AvailFlags(0),\
		                                m_ou32Frequency(0),\
		                                m_ou16PI(0),\
		                                m_ou8Band(0),\
		                                m_ou8TPInfo(0),\
		                                m_ou16StaticListID(0),\
		                                m_ou8ConfigListID(0),\
		                                m_ou16StaticListElementID(0),\
		                                m_ou16ConfigListElementID(0),\
		                                m_ou8FrequencyQualification(0),\
		                                m_ou8PTY(0)
{
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
   m_ou8FieldStrngth = 0;
   m_ou8Quality = 0;
   m_ou8HDStationInfoBits = 0;
   m_ou32StationInfoHDStationID = 0;
   m_ou8HDCallSignTextEncoding = 0;
   m_ou8HDStationGenre = 0;
   m_ou8HDAudioPrgmAvail = 0;
   m_ou8HDQuality = 0;
   m_ou8ActiveHDAudPrgm = 0;
   m_ou8DRMStationInfoBits = 0;
   m_ou32DRMServiceId = 0;
   m_ou8DRMPTY = 0;
   m_ou16DRMAnnouBits = 0;
   m_ou8DRMNumberOfAudioServices = 0;
   m_ou8DRMQuality = 0;
   m_ou8DRMServiceShortId = 0;
   m_ou8Drmlanguage = 0;
   m_ou32DrmlanguageCode = 0;
   m_ou16DrmCountryCode = 0;
   m_os8DrmFrequencyCorrection = 0;
#endif
   	m_u8OperationFlags = 0;
	m_ou16PackedInfo = 0;
    m_ou32BaseFrequency = 0;
    m_ou32BGTunerFrequency = 0;
	memset (m_acPSName,0,TUN_PS_NAME_MAX_SIZE );
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
   memset (m_acHDCallSign,0,TUN_HD_CALLSIGN_MAX_SIZE);
   memset (m_acDRMServiceLabel,0,TUN_DRM_SERVICE_LABEL_MAX_SIZE);
#endif
}




/*************************************************************************
*
* FUNCTION:     tun_Data_Station::~tun_Data_Station( )
* 
* DESCRIPTION:  Destructor.
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Data_Station::~tun_Data_Station()
{
   
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Monitor_Station::tun_Data_Monitor_Station( )      tun_Data_Monitor_Station (  );
* 
* DESCRIPTION:  Constructor.
*
* PARAMETER:    
*
* RETURNVALUE:  void rrrb1kor
*
*************************************************************************/
tun_Data_Monitor_Station::tun_Data_Monitor_Station ( ): m_ou8Client(0),\
                                        m_ou8Mode(0),\
                                        m_ou8Quality(0),\
                                        m_ou32PI(0)
{   
  
  
   
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Monitor_Station::~tun_Data_Monitor_Station( )
* 
* DESCRIPTION:  Destructor.
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Data_Monitor_Station::~tun_Data_Monitor_Station()
{
   
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetTunerHeader ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetTunerHeader(tU8 u8TunerHeader)
{
   m_ou8TunerHeader = u8TunerHeader;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetOperationFlags ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetOperationFlags(tU8 u8OperationFlags)
{
   m_u8OperationFlags = u8OperationFlags;
}





/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetPTY ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetPTY(tU8 u8PTY)
{
   m_ou8PTY = u8PTY;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetPackedInfo ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetPackedInfo(tU16 u16PackedInfo)
{
   m_ou16PackedInfo = u16PackedInfo;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetAvailFlags ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetAvailFlags(tU32 u32AvailFlags)
{
   m_ou32AvailFlags = u32AvailFlags;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetFrequency ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetFrequency(tU32 u32Frequency)
{
   m_ou32Frequency = u32Frequency;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetFrequency ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetFrequency(tU32 u32Frequency,tU8 u8BackGroundTuner)
{
	if (u8BackGroundTuner ==TUN_TEMP_FM)
	{
		m_ou32Frequency = 0x00000000;
	}
	else
	{
		m_ou32Frequency = u32Frequency;
	}
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetBaseFrequency ()
* 
* DESCRIPTION:  This function is used to set the base frequency. Function is used
*               only when the U8FrequencyQualification is set to ORIGINAL(0x00) in
*               GET_DATA_FRQ message
*
* PARAMETER:    u32Frequency
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetBaseFrequency(tU32 u32Frequency)
{
   m_ou32BaseFrequency = u32Frequency;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetBGFrequency ()
* 
* DESCRIPTION:  This function is used to set the base frequency. Function is used
*               only when the U8FrequencyQualification is set to ORIGINAL(0x00) in
*               GET_DATA_FRQ message
*
* PARAMETER:    u32Frequency
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetBGFrequency(tU32 u32Frequency)
{
   m_ou32BGTunerFrequency = u32Frequency;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetPI ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetPI(tU16 u16PI)
{
   m_ou16PI = u16PI;
}



/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetPSName ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetPSName(const tChar* pcocPSData)
{
   OSAL_pvMemoryCopy( m_acPSName,   /* Destination      */
                                        pcocPSData,   /* Source           */
                                        TUN_PS_NAME_MAX_SIZE );          /* Size of PS Name  */

	

}

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
tVoid tun_Data_Station::vSetHDCallSign(const tChar* pcocPSData)
{
   OSAL_pvMemoryCopy( m_acHDCallSign,   /* Destination      */
                                        pcocPSData,   /* Source           */
                                        TUN_HD_CALLSIGN_MAX_SIZE );          /* Size of PS Name  */

	

}
tVoid tun_Data_Station::vSetDRMServiceLabel(const tChar* pcocPSData)
{
   OSAL_pvMemoryCopy( m_acDRMServiceLabel,   /* Destination      */
                                        pcocPSData,   /* Source           */
                                        TUN_DRM_SERVICE_LABEL_MAX_SIZE );          /* Size of PS Name  */

	

}
#endif

/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetPSName ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetPSName(tChar* pcocPSData,tU8 u8BackGroundTuner)
{
	tU8 u8count =0;

	if (u8BackGroundTuner ==TUN_TEMP_FM)
	{
		while(u8count < 8)
		{
			pcocPSData[u8count] = 0x20;
			u8count++;
		}
	}
	
	OSAL_pvMemoryCopy( m_acPSName,   /* Destination      */
                                              pcocPSData,   /* Source           */
                                              8 );          /* Size of PS Name  */
	
}


/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetBand ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetBand (tU8 u8Band)
{
    m_ou8Band = u8Band;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetStaticListID ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetStaticListID (tU16 u16StaticListID)
{
   m_ou16StaticListID = u16StaticListID;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetConfigListID ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetConfigListID (tU8 u8ConfigListID)
{
   m_ou8ConfigListID = u8ConfigListID;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::vSetStaticListElementID ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetStaticListElementID (tU16 u16StaticListElementID)
{

    m_ou16StaticListElementID = u16StaticListElementID;


}

/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetConfigListElementID ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetConfigListElementID (tU16 u16ConfigListElementID)
{
    m_ou16ConfigListElementID = u16ConfigListElementID;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetFrequencyQualification ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetFrequencyQualification (tU8 u8FreqQualification)
{
    m_ou8FrequencyQualification = u8FreqQualification;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vSetTPInfo ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_Station::vSetTPInfo(tU8 u8TPInfo)
{
	m_ou8TPInfo = u8TPInfo;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Station::u8GetTunerHeader ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_Station::u8GetTunerHeader () const
{
    return m_ou8TunerHeader;
}



/*************************************************************************
*
* FUNCTION:     tun_Data_Station::u8GetOperationFlags ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_Station::u8GetOperationFlags () const
{
    return m_u8OperationFlags;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::u8GetPTY ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_Station::u8GetPTY () const
{
    return m_ou8PTY;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::u16GetPackedInfo ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU16 tun_Data_Station::u16GetPackedInfo () const
{
    return m_ou16PackedInfo;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::u32GetAvailFlags ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU32 tun_Data_Station::u32GetAvailFlags () const
{
    return m_ou32AvailFlags;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::u32GetFrequency ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU32 tun_Data_Station::u32GetFrequency () const
{
    return m_ou32Frequency;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Station::u32GetBaseFrequency ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU32 tun_Data_Station::u32GetBaseFrequency () const
{
    return m_ou32BaseFrequency;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::u32GetBGFrequency ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU32 tun_Data_Station::u32GetBGFrequency () const
{
    return m_ou32BGTunerFrequency;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Station::u32GetPI ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU16 tun_Data_Station::u16GetPI () const
{
    return m_ou16PI;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::pcocGetPSName ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
const tChar* tun_Data_Station::pcocGetPSName () const
{
    return m_acPSName;   
}
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
const tChar* tun_Data_Station::pcocGetHDCallSign () const
{
    return m_acHDCallSign;   
}
const tChar* tun_Data_Station::pcocGetDRMServiceLabel () const
{
    return m_acDRMServiceLabel;   
}


#endif

/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetBand ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_Station::u8GetBand () const
{
    return m_ou8Band;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u16GetStaticListID ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tU16 tun_Data_Station::u16GetStaticListID () const
{
    return m_ou16StaticListID;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetConfigListID ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_Station::u8GetConfigListID () const
{

   return m_ou8ConfigListID;

}

/*************************************************************************
*
* FUNCTION:     tun_Data_MainStatus::u8GetStaticListElementID ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tU16 tun_Data_Station::u16GetStaticListElementID () const
{
    return m_ou16StaticListElementID;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Station::u8GetConfigListElementID ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tU16 tun_Data_Station::u16GetConfigListElementID () const
{
   return m_ou16ConfigListElementID;

}

/*************************************************************************
*
* FUNCTION:     tun_Data_Station::vGetFrequencyQualification ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_Station::vGetFrequencyQualification() const
{
	return m_ou8FrequencyQualification;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Station::u8GetTPInfo ()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_Station::u8GetTPInfo() const
{
	return m_ou8TPInfo;
}
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetFieldStrength() const
{
   return m_ou8FieldStrngth;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetQuality() const
{
   return m_ou8Quality;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetHDStationInfoBits() const
{
   return m_ou8HDStationInfoBits;
}

/*************************************************************************
*************************************************************************/
tU32 tun_Data_Station::u32GetStationInfoHDStationID()const
{
   return m_ou32StationInfoHDStationID;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetHDCallSignTextEncoding() const
{
   return m_ou8HDCallSignTextEncoding;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetHDStationGenre() const
{
   return m_ou8HDStationGenre;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetHDAudioPrgmAvail() const
{
   return m_ou8HDAudioPrgmAvail;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetHDQuality() const
{
   return m_ou8HDQuality;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetActiveHDAudPrgm() const
{
   return m_ou8ActiveHDAudPrgm;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetDRMStationInfoBits() const
{
   return m_ou8DRMStationInfoBits;
}

/*************************************************************************
*************************************************************************/
tU32 tun_Data_Station::u32GetDRMServiceId() const
{
   return m_ou32DRMServiceId;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetDRMPTY() const
{
   return m_ou8DRMPTY;
}

/*************************************************************************
*************************************************************************/
tU16 tun_Data_Station::u16GetDRMAnnouBits() const
{
   return m_ou16DRMAnnouBits;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetDRMNumberOfAudioServices() const
{
   return (tU8)m_ou8DRMNumberOfAudioServices;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetDRMQuality() const
{
   return m_ou8DRMQuality;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetDRMServiceShortId() const
{
   return m_ou8DRMServiceShortId;
}

/*************************************************************************
*************************************************************************/
tU8 tun_Data_Station::u8GetDrmlanguage() const
{
   return m_ou8Drmlanguage;
}

/*************************************************************************
*************************************************************************/
tU32 tun_Data_Station::u32GetDrmlanguageCode() const
{
   return m_ou32DrmlanguageCode;
}

/*************************************************************************
*************************************************************************/
tU16 tun_Data_Station::u16GetDrmCountryCode() const
{
   return m_ou16DrmCountryCode;
}

/*************************************************************************
*************************************************************************/
tS8 tun_Data_Station::s8GetDrmFrequencyCorrection() const
{
   return m_os8DrmFrequencyCorrection;
}


/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetFieldStrngth(tU8 u8FieldStrngth)
{
   m_ou8FieldStrngth = u8FieldStrngth;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetQuality(tU8 u8Quality)
{
   m_ou8Quality = u8Quality;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetHDStationInfoBits(tU8 u8HDStationInfoBits)
{
   m_ou8HDStationInfoBits = u8HDStationInfoBits;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetStationInfoHDStationID(tU32 u32StationInfoHDStationID)
{
   m_ou32StationInfoHDStationID = u32StationInfoHDStationID;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetHDCallSignTextEncoding(tU8 u8HDCallSignTextEncoding)
{
   m_ou8HDCallSignTextEncoding = u8HDCallSignTextEncoding;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetHDStationGenre(tU8 u8HDStationGenre)
{
   m_ou8HDStationGenre = u8HDStationGenre;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetHDAudioPrgmAvail(tU8 u8HDAudioPrgmAvail)
{
   m_ou8HDAudioPrgmAvail = u8HDAudioPrgmAvail;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetHDQuality(tU8 u8HDQuality)
{
   m_ou8HDQuality = u8HDQuality;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetActiveHDAudPrgm(tU8 u8ActiveHDAudPrgm)
{
   m_ou8ActiveHDAudPrgm = u8ActiveHDAudPrgm;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetDRMStationInfoBits(tU8 u8DRMStationInfoBits)
{
   m_ou8DRMStationInfoBits = u8DRMStationInfoBits;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetDRMServiceId(tU32 u32DRMServiceId)
{
   m_ou32DRMServiceId = u32DRMServiceId;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetDRMPTY(tU8 u8DRMPTY)
{
   m_ou8DRMPTY = u8DRMPTY;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetDRMAnnouBits(tU16 u16DRMAnnouBits)
{
   m_ou16DRMAnnouBits = (tU8)u16DRMAnnouBits;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetDRMNumberOfAudioServices(tU8 u8DRMNumberOfAudioServices)
{
   m_ou8DRMNumberOfAudioServices = u8DRMNumberOfAudioServices;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetDRMQuality(tU8 u8DRMQuality)
{
   m_ou8DRMQuality = u8DRMQuality;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetDRMServiceShortId(tU8 u8DRMServiceShortId)
{
   m_ou8DRMServiceShortId = u8DRMServiceShortId;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetDrmlanguage(tU8 u8Drmlanguage)
{
   m_ou8Drmlanguage = u8Drmlanguage;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetDrmlanguageCode(tU32 u32DrmlanguageCode)
{
   m_ou32DrmlanguageCode = u32DrmlanguageCode;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetDrmCountryCode(tU16 u16DrmCountryCode)
{
   m_ou16DrmCountryCode = u16DrmCountryCode;
}

/*************************************************************************
*************************************************************************/
tVoid tun_Data_Station::vSetDrmFrequencyCorrection(tS8 s8DrmFrequencyCorrection)
{
	m_os8DrmFrequencyCorrection = s8DrmFrequencyCorrection;
}

/*************************************************************************
*************************************************************************/
#endif

/*--------------------------------------------*/
/* tun_Data_GetStaticListElem implementation */
/*--------------------------------------------*/

/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::tun_Data_GetStaticListElem ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Data_GetStaticListElem::tun_Data_GetStaticListElem (  ): m_ou8ListID(0),       m_ou8ElementID(0),		m_ou8Band(0),
                                                             m_ou32AvailFlags(0),  m_ou32Frequency(0)
															 /** Vnd4kor: Solving lint warnings*/
															 ,m_ou8ListElementInfo(0)


#ifdef VARIANT_S_FTR_ENABLE_EXT_PRESETLIST
															 ,m_ou32PI(0),	       m_ou16PackedInfo(0)	
#endif
                                                             
{
  /*Initialize the variable m_ocaPSName */
   tU8 u8Index =0;
   while( u8Index < TUN_PS_NAME_MAX_SIZE )
   {
     m_ocaPSName[u8Index]=0;
     u8Index++;
   }
  
}

/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::~tun_Data_GetStaticListElem( )
* 
* DESCRIPTION:  Destructor
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Data_GetStaticListElem::~tun_Data_GetStaticListElem (  )
                      
{    
    
  
}



/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u8GetListID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8  tun_Data_GetStaticListElem::u8GetListID( ) const
{
    return m_ou8ListID;    
}

/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u8GetElementID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8  tun_Data_GetStaticListElem::u8GetElementID( ) const
{
    return m_ou8ElementID;    
}


/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u8GetListElementInfo ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8  tun_Data_GetStaticListElem::u8GetListElementInfo( ) const
{
    return m_ou8ListElementInfo;    
}



/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u32GetAvailFlags ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU32 tun_Data_GetStaticListElem::u32GetAvailFlags( )const
{
    return m_ou32AvailFlags;    
}
   
/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u32GetFrequency ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU32 tun_Data_GetStaticListElem::u32GetFrequency( )const
{
    return m_ou32Frequency;    
}

   


/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u32GetBand ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_GetStaticListElem::u8GetBand( )const
{
    return m_ou8Band;    
}
 
/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::pcocGetPSName ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
 tChar*  tun_Data_GetStaticListElem::pcocGetPSName () 
{
    return m_ocaPSName;  

}
  
#ifdef VARIANT_S_FTR_ENABLE_EXT_PRESETLIST
/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u32GetPI ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  tU32
*
*************************************************************************/
tU32 tun_Data_GetStaticListElem :: u32GetPI( ) const
{
   return m_ou32PI;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u16GetPackedInfo ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  tU16
*
*************************************************************************/
tU16 tun_Data_GetStaticListElem :: u16GetPackedInfo( )const
{
   return m_ou16PackedInfo;
}
    
#endif
   
/*----------------------------------------*/
/* Set functions to be used by the server */
/*----------------------------------------*/

    

/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u8SetListID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_GetStaticListElem::u8SetListID (tU8 u8ListID ) 
{
   m_ou8ListID = u8ListID;
}



   

/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u8SetElementID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_GetStaticListElem::u8SetElementID (tU8 u8ElementID ) 
{
   m_ou8ElementID = u8ElementID;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u8SetBand ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_GetStaticListElem::vSetBand (tU8 u8Band ) 
{
   m_ou8Band = u8Band;
}
    
/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u8SetListElementInfo(tU8 u8ListElementInfo)
* 
* DESCRIPTION:  
*
* PARAMETER:    tU8 u8ListElementInfo
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_GetStaticListElem::u8SetListElementInfo (tU8 u8ListElementInfo ) 
{
   m_ou8ListElementInfo = u8ListElementInfo;
}


    
/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u32SetAvailFlags ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_GetStaticListElem::u32SetAvailFlags (tU32 u32AvailFlags ) 
{
   m_ou32AvailFlags = u32AvailFlags;
}

   

/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u32SetFrequency ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_GetStaticListElem::u32SetFrequency ( tU32 u32Frequency ) 
{
   m_ou32Frequency = u32Frequency;
}
   
 
/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::vSetPSName ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_GetStaticListElem::vSetPSName (const tChar* pcocPSName)
{
    OSAL_pvMemoryCopy( m_ocaPSName,   /* Destination      */
                                              pcocPSName,    /* Source           */
                                              8 );           /* Size of PS Name  */
	
}
#ifdef VARIANT_S_FTR_ENABLE_EXT_PRESETLIST

/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u32SetPI ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_GetStaticListElem :: u32SetPI( tU32 u32PI )
{
   m_ou32PI = u32PI;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_GetStaticListElem::u16SetPackedInfo ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_GetStaticListElem :: u16SetPackedInfo( tU16 u16PackedInfo )
{
   m_ou16PackedInfo= u16PackedInfo;
}

#endif
/*--------------------------------------------*/
/* m_oatun_Data_GetConfigListElem             */
/*--------------------------------------------*/

/*************************************************************************
*
* FUNCTION:     tun_Data_GetConfigListElem::tun_Data_GetConfigListElem ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Data_GetConfigListElem :: tun_Data_GetConfigListElem (  ): m_ou8ListID(0xff),\
                                                               m_ou8NumOfElementsInList(0)
															  
{

  
}

/*************************************************************************
*
* FUNCTION:     tun_Data_GetConfigListElem::~tun_Data_GetConfigListElem( )
* 
* DESCRIPTION:  Destructor
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Data_GetConfigListElem::~tun_Data_GetConfigListElem (  )
                      
{    
    
  
}

/*************************************************************************
*
* FUNCTION:     tun_Data_GetConfigListElem::u8GetListID ()
* 
* DESCRIPTION:  The List ID of the Config list will be sent through this 
*               function.
*
* PARAMETER:    None
*
* RETURNVALUE:  tU8
*
*************************************************************************/
tU8  tun_Data_GetConfigListElem::u8GetListID( ) const
{
    return m_ou8ListID;    
}

/*************************************************************************
*
* FUNCTION:     tun_Data_GetConfigListElem::u8GetNumOfElementsInList ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  tU8
*
*************************************************************************/
tU8 tun_Data_GetConfigListElem::u8GetNumOfElementsInList( ) const
{
    return m_ou8NumOfElementsInList;    
}
 
   
/*----------------------------------------*/
/* Set functions to be used by the server */
/*----------------------------------------*/

/*************************************************************************
*
* FUNCTION:     tun_Data_GetConfigListElem::vSetListID ()
* 
* DESCRIPTION:  
*
* PARAMETER:    tU8 u8ListID
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_GetConfigListElem :: vSetListID (tU8 u8ListID ) 
{
   m_ou8ListID = u8ListID;
}
   
/*************************************************************************
*
* FUNCTION:     tun_Data_GetConfigListElem :: vSetNumOfElementsInList ()
* 
* DESCRIPTION:  
*
* PARAMETER:    tU8 u8NumOfElementsInList
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_GetConfigListElem :: vSetNumOfElementsInList (tU8 u8NumOfElementsInList ) 
{
   m_ou8NumOfElementsInList = u8NumOfElementsInList;
}

/*--------------------------------------------*/
/*  class tun_Data_TMC implementation     */
/*--------------------------------------------*/


/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::tun_Data_TMC( )
* 
* DESCRIPTION:  Constructor.
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Data_TMC::tun_Data_TMC ( ): m_ou8TunerHeader(0),\
                                        m_ou16TMCData1(0),\
                                        m_ou16TMCData2(0),\
                                        m_ou16TMCData3(0),\
                                        m_ou16TMCData4(0)                                        

{   
     
}






/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::~tun_Data_TMC( )
* 
* DESCRIPTION:  Destructor.
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Data_TMC::~tun_Data_TMC()
{
   
}




/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::vSetTunerHeader ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_TMC::vSetTunerHeader(tU8 u8TunerHeader)
{
   m_ou8TunerHeader = u8TunerHeader;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::vSetu16TMCData1 ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_TMC::vSetu16TMCData1(tU16 u16TMCData1)
{
   m_ou16TMCData1 = u16TMCData1;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::vSetu16TMCData2 ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_TMC::vSetu16TMCData2(tU16 u16TMCData2)
{
   m_ou16TMCData2 = u16TMCData2;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::vSetu16TMCData3 ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_TMC::vSetu16TMCData3(tU16 u16TMCData3)
{
   m_ou16TMCData3 = u16TMCData3;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::vSetu16TMCData4 ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Data_TMC::vSetu16TMCData4(tU16 u16TMCData4)
{
   m_ou16TMCData4 = u16TMCData4;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::u8GetTunerHeader ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_TMC::u8GetTunerHeader () const
{
    return m_ou8TunerHeader;
}



/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::u16GetTMCData1 ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU16 tun_Data_TMC::u16GetTMCData1 () const
{
    return m_ou16TMCData1;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::u16GetTMCData2 ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU16 tun_Data_TMC::u16GetTMCData2 () const
{
    return m_ou16TMCData2;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::u16GetTMCData3 ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU16 tun_Data_TMC::u16GetTMCData3 () const
{
    return m_ou16TMCData3;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_TMC::u16GetTMCData1 ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU16 tun_Data_TMC::u16GetTMCData4 () const
{
    return m_ou16TMCData4;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Monitor_Station::u8GetClient ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_Monitor_Station::u8GetClient () const
{
    return m_ou8Client;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Monitor_Station::u8GetMode ()
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_Monitor_Station::u8GetMode () const
{
    return m_ou8Mode;
}


/*************************************************************************
*
* FUNCTION:     tun_Data_Monitor_Station::u8GetQuality ()
* 
* DESCRIPTION:  
*
* PARAMETER:    rrb1kor
*
* RETURNVALUE:  void
*
*************************************************************************/
tU8 tun_Data_Monitor_Station::u8GetQuality () const
{
   return m_ou8Quality;
}

/*************************************************************************
*
* FUNCTION:     tun_Data_Monitor_Station::u32GetPI ()
* 
* DESCRIPTION:  
*
* PARAMETER:    rrb1kor
*
* RETURNVALUE:  void
*
*************************************************************************/
tU32 tun_Data_Monitor_Station::u32GetPI () const
{
   return m_ou32PI;
}




/*************************************************************************
*
* FUNCTION:     tun_ActualData::bCheckforBackgroundTuner ()
* 
* DESCRIPTION:  Trace to see currect data in Tuner sever.
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tBool tun_ActualData :: bCheckforBackgroundTuner(const tU8* pu8NewData, tU32 u32MessageLen )const
{
 if(( pu8NewData == NULL ) ||( u32MessageLen == 0 ) )
 {if( pu8NewData == NULL )
  {
    NORMAL_M_ASSERT( pu8NewData != NULL );

  }
  if( u32MessageLen == 0 )
  {
    NORMAL_M_ASSERT( u32MessageLen != 0 );

  }
  return true;
 }

  bool flag = false; 

  if(pu8NewData[1] == TU_BACKGROUND)
  {
    switch( pu8NewData[0] )
    {
     case TUN_R_AUDIO_FG_BG:
     case TUN_R_SET_FRQ:
     case TUN_R_GET_DATA_FRQ:
       {
          ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_ACTDATA, ET_EN_T8 _ 0xFF _
                                                     ET_EN_T8 _ *pu8NewData _
                                                     ET_EN_T8 _ 0xFF _
                                               ET_EN_DONE );
          flag = true;
          break;
       }
     default:
       {
         ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_ACTDATA, ET_EN_T8 _ *pu8NewData _
                                               ET_EN_DONE );
      break;
       }
    }// end of Switch

  }
   return flag;

}

/*************************************************************************
*
* FUNCTION:     tun_ActualData::TraceGetActTunerData ()
* 
* DESCRIPTION:  Trace to see currect data in Tuner sever.
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
/** Solving Logiscope issue: FAIR maintainability */
tVoid tun_ActualData::TraceGetActTunerData( tU8 u8WhichClass) const
{
	if(TU_ACTDATA_STATION == u8WhichClass)
    {
	   ETG_TRACE_USR4(("u8TunerHeader = %d\n"
	         "u32AvailFlags = %d\n"
	         "u32Frequency = %d\n"
	         "u16PI=%d\n"
	         "sPSName = %s",
	         m_otun_Data_Station.u8GetTunerHeader()
	         , m_otun_Data_Station.u32GetAvailFlags()
	         , m_otun_Data_Station.u32GetFrequency()
	         , m_otun_Data_Station.u16GetPI()
	         , m_otun_Data_Station.pcocGetPSName()));

	   ETG_TRACE_USR4(("e8Band=%d\n"
	         "e8AtStationTPinfo = %d\n"
	         "u16PresetListID = %d\n"
	         "u8ActiveStationList = %d\n"
	         "u16StaticListElementID = %d\n"
	         "u16ConfigListElementID = %d\n"
	         "u8AtStationInfoBits = %d\n"
	         "u8PTY =%d"
            , m_otun_Data_Station.u8GetBand()
	         , m_otun_Data_Station.u8GetTPInfo()
	         , m_otun_Data_Station.u16GetStaticListID()
	         , m_otun_Data_Station.u8GetConfigListID()
	         , m_otun_Data_Station.u16GetStaticListElementID()
	         , m_otun_Data_Station.u16GetConfigListElementID()
	         , m_otun_Data_Station.vGetFrequencyQualification()
	         , m_otun_Data_Station.u8GetPTY()));
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20

	   ETG_TRACE_USR4(("u8FieldStrngth = %d\n"
	         "u8Quality = %d\n"
	         "b8HDStationInfoBits = %d\n"
	         "u32StationInfoHDStationID = 0x%08x\n"
	         "sHDCallSign = %s",
	         m_otun_Data_Station.u8GetFieldStrength()
	         , m_otun_Data_Station.u8GetQuality()
	         , m_otun_Data_Station.u8GetHDStationInfoBits()
	         , m_otun_Data_Station.u32GetStationInfoHDStationID()
	         , (const tChar*)m_otun_Data_Station.pcocGetHDCallSign()));

	   ETG_TRACE_USR4(( "e8HDCallSignTextEncoding = %d\n"
            "u8HDStationGenre = %d\n"
            "b8HDAudioPrgmAvail = %d\n"
            "u8DRMPTY = 0x%02x\n"
            "b16DRMAnnouBits = %d\n"
            "b8DRMNumberOfAudioServices = 0x%02x",
            m_otun_Data_Station.u8GetHDCallSignTextEncoding()
            , m_otun_Data_Station.u8GetHDStationGenre()
            , m_otun_Data_Station.u8GetHDAudioPrgmAvail()
            , m_otun_Data_Station.u8GetDRMPTY()
            , m_otun_Data_Station.u16GetDRMAnnouBits()
            , m_otun_Data_Station.u8GetDRMNumberOfAudioServices()));

	   ETG_TRACE_USR4(("u8HDQuality = %d\n"
	               "e8ActiveHDAudPrgm = %d\n"
	               "b8DRMStationInfoBits = 0x%02x\n"
	               "u32DRMServiceId = 0x%08x\n"
	               "sDRMServiceLabel = %s",
	               m_otun_Data_Station.u8GetHDQuality()
	               , m_otun_Data_Station.u8GetActiveHDAudPrgm()
	               , m_otun_Data_Station.u8GetDRMStationInfoBits()
	               , m_otun_Data_Station.u32GetDRMServiceId()
	               , (const tChar*)m_otun_Data_Station.pcocGetDRMServiceLabel()));

	   ETG_TRACE_USR4(("u8DRMQuality = %d\n"
	                     "u8DRMServiceShortId = %d\n"
	                     "u8Drmlanguage = %d\n"
	                     "u32DrmlanguageCode = %d\n"
	                     "u16DrmCountryCode = %d\n"
						 "s8DrmFrequencyCorrection = %d",
	                     m_otun_Data_Station.u8GetDRMQuality()
	                     , m_otun_Data_Station.u8GetDRMServiceShortId()
	                     , m_otun_Data_Station.u8GetDrmlanguage()
	                     , m_otun_Data_Station.u32GetDrmlanguageCode()
	                     , m_otun_Data_Station.u16GetDrmCountryCode()
						 , m_otun_Data_Station.s8GetDrmFrequencyCorrection()));

#endif
    
      }
	if(TU_ACTDATA_MAINSTATUS == u8WhichClass || u8WhichClass == TU_ACTDATA_GET_LIST_ELEM)
	{
	   ETG_TRACE_USR4(("Nothing to print for this class\n"));
	}
      

}



/*************************************************************************
*
* FUNCTION:     tun_ActualData::vSendBandbasedOnPreset ()
* 
* DESCRIPTION:  To update the Band during startup when Component status 
*               response comes from RU.
*
* PARAMETER:    
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_ActualData::vSendBandbasedOnPreset( )
{   

       switch(m_otun_Data_MainStatus.u8GetStaticListID())
          {
          case FI_EN_TUN_TUN_STATIC_LIST_FM1:
          case FI_EN_TUN_TUN_STATIC_LIST_FM2:
          case FI_EN_TUN_TUN_STATIC_LIST_FM3:
          case FI_EN_TUN_TUN_STATIC_LIST_AST:
            {
              m_otun_Data_MainStatus.vSetBand( (tU8)FI_EN_TUN_BAND_FM );
              break;
            }
            
          case FI_EN_TUN_TUN_STATIC_LIST_AM1:
            {
              
              m_otun_Data_MainStatus.vSetBand( (tU8)FI_EN_TUN_BAND_MW);
              break;
            }
            
            
          case FI_EN_TUN_TUN_STATIC_LIST_LW1:
            {
              
              m_otun_Data_MainStatus.vSetBand( (tU8)FI_EN_TUN_BAND_LW );
              break;
            }

		  case FI_EN_TUN_TUN_STATIC_LIST_INVALID:
			  {
				m_otun_Data_MainStatus.vSetBand( (tU8)FI_EN_TUN_BAND_MW);
			  }
			  break;
            
          default:
            {break;
            }
          }

}
/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_ActualData::vSetSystemState
 *
 *DESCRIPTION:  Stores the current system state when received by SPM through the property 
 *              SystemState.
 *             
 *PARAMETER:    tU32 SystemState
 *
 *RETURNVALUE:  None
 *
 *HISTORY:     18.07.06 vau2kor
 *             Initial version
 *
 ******************************************************************-FUNCHEADER*/
tVoid tun_Data_MainStatus::vSetSystemState (tU32 u32SystemState)
{
  m_u32SystemState = u32SystemState;
}
/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     tun_ActualData::u32GetSystemState
 *
 *DESCRIPTION:  Reads the System state
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  System State
 *
 *HISTORY:      vau2kor
 *              Initial version
 *
 ******************************************************************-FUNCHEADER*/
tU32 tun_Data_MainStatus::u32GetSystemState ()const
{
  return(m_u32SystemState);
}
tVoid tun_ActualData::vSet_tun_ConfigInterface_Ptr( clConfigInterface* poclConfigInterface )
{
  if( poclConfigInterface != NULL )
  {
	  m_poclConfigInterface = poclConfigInterface;
  }

}

tVoid tun_ActualData::vSet_TmcTuner_Method_S_JAPAN_SET_AREAPtr(clTmcTuner_Method_S_JAPAN_SET_AREA* poclTmcTuner_Method_S_JAPAN_SET_AREA)
{
  if( poclTmcTuner_Method_S_JAPAN_SET_AREA != NULL )
  {
     m_poclTmcTuner_Method_S_JAPAN_SET_AREA = poclTmcTuner_Method_S_JAPAN_SET_AREA;
  }

}
/*** END OF FILE *************************************************************/
