/*
//====================================================================
//
// MWT_WSIRLEToRAW.c:
//
// TEST MODULE
//
// Copyright � 2010 - 2013 Sirius XM.  All rights reserved.
// Author: Alexander Pylchagin
//
// Implementation of the MWT_WSIRLEToRAW module.
// See .h file for interface and description.
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/
#include <stdio.h>
#include <memory.h>
#include <malloc.h>
#include "MWT_WSIRLEToRAW.h"
#include "MW_ProductHeaders.h"
#include "MWT_ProductIDs.h"
#include <osal.h>

/*
//====================================================================
//====================================================================
// Internal methods
//====================================================================
//====================================================================
*/

/*
//====================================================================
// MWTWSIRLEToRAW_Internal_readRun:
//
// This method reads the WSI RLE run codes.
//
// 
// PARAMETERS:
//
//    p_sourceFile:
//        The file holding the RLE codes.  It should currently be
//        positioned at the start of the next RLE code.  Calls to this
//        method automatically update it to point at the start of the
//        next RLE code in the file.
//
//    p_inout_unshiftedPixelValue:
//        Pointer to the byte that holds the previous unshifted pixel
//        value.  It will be replaced by the one indicated for the
//        new run.  If this is the first time this method is being
//        called in a raster plane, the pixel value should be set
//        to 0 before calling.
//
//        Note this is the value without any shifting for data plane
//        merging having been applied.
//
//    p_out_runLength:
//        Pointer to a variable which receives the number of consequetive
//        pixels, in scanline order, that occur of the unshifted pixel
//        value.
//
// RETURNS:
//
//    MWRes_Success
//
//    MWRes_NullPointer
//
//    MWRes_CommunicationsError:
//        On file read error
//
//    MWRes_ProductContentsInvalid:
//        On end of file
//
//====================================================================
*/
static TMW_ResultCode MWTWSIRLEToRAW_Internal_readRun
(
    FILE* p_sourceFile,
    TByte_Unsigned* p_inout_unshiftedPixelValue,
    TFourByteInteger_Unsigned* p_out_runLength
)
{
    TByte_Unsigned pixelHighNibble;
    TByte_Unsigned codeBytes[4];
    TFourByteInteger_Unsigned numBytesRead;


    /*
    //##############################################################
    // Test parameters
    //##############################################################
    */
    if ((p_sourceFile == NULL) || (p_inout_unshiftedPixelValue == NULL) || (p_out_runLength == NULL))
    {
        return MWRes_NullPointer;
    }

    /*
    //##############################################################
    // What was last high nibble of pixel value?
    //##############################################################
    */
    pixelHighNibble = *p_inout_unshiftedPixelValue & 0xF0;

    /*
    //##############################################################
    // Consume any high nibble change codes
    //##############################################################
    */

    while (1)
    {

        /* Read a byte */
        numBytesRead = 0;
        numBytesRead = fread (&codeBytes[0], sizeof(TByte_Unsigned), 1, p_sourceFile);
        if (numBytesRead != 1)
        {
            if (feof(p_sourceFile))
            {
                /* Not supposed to end here */
                return MWRes_ProductContentsInvalid;
            }
            else
            {
                return MWRes_CommunicationsError;
            }
        }

        /* Is it a high nibble change */
        if ((codeBytes[0] & 0xF0) == 0xF0)
        {
            /*
            // Its a pixel value high nibble change code
            // Low nibble gives new high nibble of pixel
            */
            pixelHighNibble = (codeBytes[0] & 0x0F) << 4;
        }
        else
        {
            /* Done with high nibble changes */
            break;
        }
    }

    /*
    //##############################################################
    // Branch based on high nibble of first byte
    //##############################################################
    */

    if ((codeBytes[0] & 0xF0) == 0xF0)
    {
        /* Code error */
        return MWRes_CodeFault;
    }
    else if ((codeBytes[0] & 0xF0) == 0xE0)
    {
        /*
        // Pixel value is current pixel high nibble and low nibble of this
        // byte.
        */
        *p_inout_unshiftedPixelValue = pixelHighNibble | (codeBytes[0] & 0x0F);

        /* Run length is (unsigned byte following this one) + 1*/
        /* Read a byte */
        numBytesRead = 0;
        numBytesRead = fread (&(codeBytes[1]), sizeof(TByte_Unsigned), 1, p_sourceFile);
        if (numBytesRead != 1)
        {
            if (feof(p_sourceFile))
            {
                /* Not supposed to end here */
                return MWRes_ProductContentsInvalid;
            }
            else
            {
                return MWRes_CommunicationsError;
            }
        }

        /* Set run length */
        *p_out_runLength = ((TFourByteInteger_Unsigned) codeBytes[1]) +1;


    }
    else if ((codeBytes[0] & 0xF0) == 0xD0)
    {
        /*
        // Pixel value is current pixel high nibble and low nibble of this
        // byte.
        */
        *p_inout_unshiftedPixelValue = pixelHighNibble | (codeBytes[0] & 0x0F);

        /* Run length is (two byte lohi unsigned int following this byte) + 1*/
        /* Read a byte */
        numBytesRead = 0;
        numBytesRead = fread (&(codeBytes[1]), sizeof(TByte_Unsigned), 2, p_sourceFile);
        if (numBytesRead != 2)
        {
            if (feof(p_sourceFile))
            {
                /* Not supposed to end here */
                return MWRes_ProductContentsInvalid;
            }
            else
            {
                return MWRes_CommunicationsError;
            }
        }

        /* Set run length */
        *p_out_runLength = (TFourByteInteger_Unsigned) (codeBytes[1]);
        *p_out_runLength = (*p_out_runLength) +
            ( ((TFourByteInteger_Unsigned) codeBytes[2]) * 256);
        *p_out_runLength = (*p_out_runLength) + 1;
    }
    else
    {
        /*
        // Pixel value is current pixel high nibble and low nibble of this
        // byte.
        */
        *p_inout_unshiftedPixelValue = pixelHighNibble | (codeBytes[0] & 0x0F);

        /*
        // Run length is high nibble +1
        */
        *p_out_runLength = ((codeBytes[0] & 0xF0) >> 4) + 1;

    }

    /*
    //##############################################################
    // Success
    //##############################################################
    */
    return MWRes_Success;

}

/*
//====================================================================
// MWTWSIRLEToRAW_Internal_processRasterPlane_RLE:
//
// This method reads the RLE codes for the plane and merges
// the pixels with the data already in the raster plane.  
//
// The merge is achieved by leftshifting the run pixel value by
// the parameterized amount, and then bitwise orring the resulting 
// pixel value with the value already in the raster surface.
//
// The RLE codes are assumed to start at the upper left of the 
// image and encompass the entire raster body.
//
// PARAMETERS:
//
//    numColumns:
//        The number of columns in the raster
//
//    numRows:
//        The number of rows in the raster
//
//    numPitchBytes:
//        The number of bytes per row.  This must be
//        >= (numColumns * sizeof(TByte_Unsigned) )
//
//    p_rasterSurface:
//        The upper left pixel of the raster surface as an
//        array in scanline order.
//
//    p_RLEFile:
//        The source file from which the RLE codes will be read.
//        It should be positioned at the start of the first RLE code
//        of the plane before this method is called.
//
//    dataPlaneLeftShiftNumBits:
//        The number of bits to leftshift a run pixel value from
//        this plane before or-ing it with the merger value already
//        in the respective pixel of the merge raster surface.
//
// RETURNS:
//
//    MWRes_Success
//
//    MWRes_ProductContentsInvalid:
//        If reaches end of file mid run, or the runs do not line
//        up with the number of pixels in the plane.
//
//    MWRes_NullPointer:
//        If a pointer parameter is NULL
//
//    MWRes_BadParamValue:
//        If numPitchBytes <  (numColumns * sizeof(TByte_Unsigned) )
//
//====================================================================
*/
static TMW_ResultCode MWTWSIRLEToRAW_Internal_processRasterPlane_RLE
(
    TTwoByteInteger_Unsigned numColumns,
    TTwoByteInteger_Unsigned numRows,
    TFourByteInteger_Unsigned numPitchBytes,
    TByte_Unsigned* p_rasterSurface,
    FILE* p_RLEFile,
    TByte_Unsigned dataPlaneLeftShiftNumBits
)
{
    TMW_ResultCode MWRes;
    TFourByteInteger_Unsigned numPixelsInRaster;
    TFourByteInteger_Unsigned pixelWriteCount;
    TFourByteInteger_Unsigned runLengthRemaining;
    TByte_Unsigned unshiftedPixelValue;
    TByte_Unsigned shiftedPixelValue;
    TFourByteInteger_Unsigned extraPitchBytesPerRow;

    TByte_Unsigned *p_pixelCursor;
    TTwoByteInteger_Unsigned columnIndex;
    TTwoByteInteger_Unsigned rowIndex;

    /*
    //##############################################################
    // Test parameters
    //##############################################################
    */
    if (numPitchBytes < (numColumns * sizeof(TByte_Unsigned)) )
    {
        return MWRes_BadParamValue;
    }
    else if ((p_rasterSurface == NULL) || (p_RLEFile == NULL))
    {
        return MWRes_NullPointer;
    }

    /*
    //##############################################################
    // Calculate extra pitch bytes per row
    //##############################################################
    */
    extraPitchBytesPerRow = numPitchBytes - (numColumns * sizeof(TByte_Unsigned));

    /*
    //##############################################################
    // Iterate destination until it is full
    //##############################################################
    */
    numPixelsInRaster = numColumns * numRows;
    p_pixelCursor = p_rasterSurface;
    runLengthRemaining = 0;
    unshiftedPixelValue = 0;
    shiftedPixelValue = 0;

    pixelWriteCount = 0;
    for (rowIndex = 0; rowIndex < numRows; rowIndex ++)
    {
        for (columnIndex = 0; columnIndex < numColumns; columnIndex ++)
        {
            /*
            //##############################################################
            // If we need more run data, get some
            //##############################################################
            */
            if (runLengthRemaining == 0)
            {
                MWRes = MWTWSIRLEToRAW_Internal_readRun
                (
                    p_RLEFile,
                    &unshiftedPixelValue,
                    &runLengthRemaining
                );
                if (MWRes != MWRes_Success)
                {
                    return MWRes;
                }

                /* Check that run is not longer than end of raster */
                /* If it is, we have a problem */
                if (runLengthRemaining > (numPixelsInRaster - pixelWriteCount))
                {
                    return MWRes_ProductContentsInvalid;
                }

                /* Shift pixel value*/
                shiftedPixelValue = unshiftedPixelValue << dataPlaneLeftShiftNumBits;
            }

            /*
            //##############################################################
            // Encode run into the raster
            //
            //##############################################################
            */

            *p_pixelCursor = (*p_pixelCursor) | shiftedPixelValue;

            runLengthRemaining --;
            pixelWriteCount ++;

            /*
            //##############################################################
            // Bump up pixel cursor
            //##############################################################
            */
            p_pixelCursor += 1;


        } /* Next column */

        /* Bump up pixel cursor by extra pitch at end of row */
        p_pixelCursor += extraPitchBytesPerRow;


    } /* Next row */

    /*
    //##############################################################
    // Test for extra run length
    //##############################################################
    */
    if (runLengthRemaining != 0)
    {

        return MWRes_ProductContentsInvalid;
    }

    /*
    //##############################################################
    // Success
    //##############################################################
    */
    return MWRes_Success;

}


/*
//====================================================================
// MWTWSIRLEToRAW_Internal_processRasterPlane_Raw:
//
// This method reads the raw pixel codes for the plane and merges
// the pixels with the data already in the raster plane.  
//
// The merge is achieved by leftshifting the run pixel value by
// the parameterized amount, and then bitwise orring the resulting 
// pixel value with the value already in the raster surface.
//
// The pixels are assumed to start at the upper left of the 
// image and encompass the entire raster body.
//
// PARAMETERS:
//
//    numColumns:
//        The number of columns in the raster
//
//    numRows:
//        The number of rows in the raster
//
//    numPitchBytes:
//        The number of bytes per row.  This must be
//        >= (numColumns * sizeof(TByte_Unsigned) )
//
//    p_rasterSurface:
//        The upper left pixel of the raster surface as an
//        array in scanline order.
//
//    p_sourceFile:
//        The source file from which the pixel codes will be read.
//        It should be positioned at the start of the first pixel code
//        of the plane before this method is called.
//
//    dataPlaneLeftShiftNumBits:
//        The number of bits to leftshift a run pixel value from
//        this plane before or-ing it with the merger value already
//        in the respective pixel of the merge raster surface.
//
// RETURNS:
//
//    MWRes_Success
//
//    MWRes_ProductContentsInvalid:
//        If reaches end of file mid run, or the runs do not line
//        up with the number of pixels in the plane.
//
//    MWRes_NullPointer:
//        If a pointer parameter is NULL
//
//    MWRes_BadParamValue:
//        If numPitchBytes <  (numColumns * sizeof(TByte_Unsigned) )
//
//====================================================================
*/
TMW_ResultCode MWTWSIRLEToRAW_Internal_processRasterPlane_Raw
(
    TTwoByteInteger_Unsigned numColumns,
    TTwoByteInteger_Unsigned numRows,
    TFourByteInteger_Unsigned numPitchBytes,
    TByte_Unsigned* p_rasterSurface,
    FILE* p_sourceFile,
    TByte_Unsigned dataPlaneLeftShiftNumBits
)
{
    //TMW_ResultCode MWRes;
    TFourByteInteger_Unsigned numPixelsInRaster;
    TFourByteInteger_Unsigned pixelWriteCount;
    TFourByteInteger_Unsigned runLengthRemaining;
    TByte_Unsigned unshiftedPixelValue;
    TByte_Unsigned shiftedPixelValue;
    TFourByteInteger_Unsigned extraPitchBytesPerRow;

    TByte_Unsigned *p_pixelCursor;
    TTwoByteInteger_Unsigned columnIndex;
    TTwoByteInteger_Unsigned rowIndex;
    TFourByteInteger_Signed ReadRes;

#define BYTE_BUFFER_SIZE 1024
    TByte_Unsigned byteBuffer[BYTE_BUFFER_SIZE];
    TFourByteInteger_Unsigned numBytesInByteBuffer = 0;
    TFourByteInteger_Unsigned numBytesInByteBufferRead = 0;

    /*
    //##############################################################
    // Test parameters
    //##############################################################
    */
    if (numPitchBytes < (numColumns * sizeof(TByte_Unsigned)) )
    {
        return MWRes_BadParamValue;
    }
    else if ((p_rasterSurface == NULL) || (p_sourceFile == NULL))
    {
        return MWRes_NullPointer;
    }

    /*
    //##############################################################
    // Calculate extra pitch bytes per row
    //##############################################################
    */
    extraPitchBytesPerRow = numPitchBytes - (numColumns * sizeof(TByte_Unsigned));

    /*
    //##############################################################
    // Iterate destination until it is full
    //##############################################################
    */
    numPixelsInRaster = numColumns * numRows;
    p_pixelCursor = p_rasterSurface;
    runLengthRemaining = 0;
    unshiftedPixelValue = 0;
    shiftedPixelValue = 0;

    pixelWriteCount = 0;
    for (rowIndex = 0; rowIndex < numRows; rowIndex ++)
    {
        for (columnIndex = 0; columnIndex < numColumns; columnIndex ++)
        {
            /*
            //##############################################################
            // If we need more run data, get some
            //##############################################################
            */
            if (runLengthRemaining == 0)
            {
                if (numBytesInByteBufferRead >= numBytesInByteBuffer)
                {
                    // Read next pixels
                    ReadRes = fread
                    (
                        byteBuffer,
                        1,
                        BYTE_BUFFER_SIZE,
                        p_sourceFile
                    );
                    if (ReadRes < 1)
                    {
                        return MWRes_ProductContentsInvalid;
                    }
                    numBytesInByteBuffer = ReadRes;
                    numBytesInByteBufferRead = 0;
                }

                // Pull byte from byte buffer
                unshiftedPixelValue = byteBuffer[numBytesInByteBufferRead];
                numBytesInByteBufferRead ++;

                // We read in 1 pixel
                runLengthRemaining = 1;

                /* Check that run is not longer than end of raster */
                /* If it is, we have a problem */
                if (runLengthRemaining > (numPixelsInRaster - pixelWriteCount))
                {
                    return MWRes_ProductContentsInvalid;
                }

                /* Shift pixel value*/
                shiftedPixelValue = unshiftedPixelValue << dataPlaneLeftShiftNumBits;
            }

            /*
            //##############################################################
            // Encode run into the raster
            //
            //##############################################################
            */

            *p_pixelCursor = (*p_pixelCursor) | shiftedPixelValue;

            runLengthRemaining --;
            pixelWriteCount ++;

            /*
            //##############################################################
            // Bump up pixel cursor
            //##############################################################
            */
            p_pixelCursor += 1;


        } /* Next column */

        /* Bump up pixel cursor by extra pitch at end of row */
        p_pixelCursor += extraPitchBytesPerRow;


    } /* Next row */

    /*
    //##############################################################
    // Test for extra run length
    //##############################################################
    */
    if (runLengthRemaining != 0)
    {

        return MWRes_ProductContentsInvalid;
    }

    /*
    //##############################################################
    // Success
    //##############################################################
    */
    return MWRes_Success;
}

/*
//====================================================================
// MWTWSIRLEToRAW_Internal_write8BPPRAWFile:
//
// Given a raster in surface memory, writes out an 8BPP RAW
// file using a palette determined from the given ProductID.
//
// PARAMETERS:
//
//    numColumns:
//        The number of columns in the raster
//
//    numRows:
//        The number of rows in the raster
//
//    numSourcePitchBytes:
//        The number of bytes per row.  This must be
//        >= (numColumns * sizeof(TByte_Unsigned) )
//
//    p_rasterSurface:
//        The upper left pixel of the raster surface as an
//        array in scanline order.
//
//    pstr_rawFilePath:
//        The path of the file to be created/replaced with the
//        resulting bitmap image file.
//
//    productID:
//        The product ID from the product header. The palette used
//        in the image is dependent on this.
//
//
//====================================================================
*/
static TMW_ResultCode MWTWSIRLEToRAW_Internal_write8BPPRAWFile
(
    TTwoByteInteger_Unsigned numColumns,
    TTwoByteInteger_Unsigned numRows,
    TFourByteInteger_Unsigned numSourcePitchBytes,
    TByte_Unsigned* p_rasterSurface,
    const TASCIIChar* pstr_rawFilePath,
    TTwoByteInteger_Unsigned productID
)
{
    FILE *p_outFile;
    TMW_ResultCode resultCode = MWRes_OperationFailed;
    size_t tWrittenSize;

    /*
    //##############################################################
    //  Test parameters
    //##############################################################
    */
    if ((pstr_rawFilePath == NULL) || (p_rasterSurface == NULL))
    {
        return MWRes_NullPointer;
    }

    /*
    //##############################################################
    // Get palette for product ID
    //##############################################################
    */

    if ((productID == CompositePrecipNEXRAD_MWProductID) ||
        (productID == Canadian_RADAR_MWProductID) ||
        (productID == SeaSurfaceTemperatures_MWProductID) ||
        ((productID >= Min_WaveGrids_MWProductID) &&
         (productID <= Max_WaveGrids_MWProductID)) ||
         ((productID >= Min_SurfaceWindGrids_MWProductID) &&
         (productID <= Max_SurfaceWindGrids_MWProductID)))
    {
    }
    else
    {
        /* Not recognized */
        return MWRes_OperationFailed;
    }

    do
    {
        /*
        //##############################################################
        // Open the output file
        //##############################################################
        */
        p_outFile = fopen (pstr_rawFilePath, "wb");
        if (p_outFile == NULL)
        {
            resultCode = MWRes_CommunicationsError;
            break;
        }

        //#################################################################
        // Write out the run raw image bytes
        //#################################################################
        tWrittenSize = fwrite(p_rasterSurface, 1, numColumns * numRows, p_outFile);
        if (tWrittenSize != numColumns * numRows)
        {
            resultCode = MWRes_OperationFailed;
            break;
        }

        //##############################################################
        // Success
        //##############################################################
        resultCode = MWRes_Success;
    } while (0);

    if (p_outFile != NULL)
    {
        fclose(p_outFile);
        p_outFile = NULL;
    }

    if (resultCode != MWRes_Success)
    {
        // Remove file if the compression has failed
        remove(pstr_rawFilePath);
    }

    return resultCode;
}

/*
//====================================================================
//====================================================================
// Public interface
//====================================================================
//====================================================================
*/
TMW_ResultCode MWTWSIRLEToRAW_convertRasterProductFile
(
    const TASCIIChar* pstr_rasterProductPath,
    const TASCIIChar* pstr_outputRAWFilePath,
    TMW_CompressionType sourceCompressionType,
    TByte_Unsigned planeNumber
)
{
    TMW_ResultCode MWRes;

    /* The number of data planes expected for this ProductID */
    TByte_Unsigned numDataPlanesExpected;

    /* The plane parsing loop*/
    TByte_Unsigned planeIndex;

    /* The amount to leftshift each data value for plane merge */
    TByte_Unsigned dataPlaneLeftShiftNumBits[4];

    /* The source file*/
    FILE* p_RLEFile;

    /* The expected header in the source file*/
    TMW_GenericRasterProductHeader_Output rasterHeader;
    TMW_GenericRasterProductHeader_FixedScaleSpecifier fixedScaleSpecifiers[MAX_PLANES_IN_RASTER];

    /* The number of bytes read */
    TFourByteInteger_Unsigned numBytesRead;

    /* The raster surface memory */
    TByte_Unsigned* p_rasterSurface;
    TFourByteInteger_Unsigned numRasterSurfaceBytes;


    /*
    //##############################################################
    // Test parameters
    //##############################################################
    */
    if ((pstr_rasterProductPath == NULL) || (pstr_outputRAWFilePath == NULL))
    {
        return MWRes_NullPointer;
    }

    /*
    //##############################################################
    // Open the source file and read in the raster header
    //##############################################################
    */

    p_RLEFile = fopen (pstr_rasterProductPath, "rb");
    if (p_RLEFile == NULL)
    {
        return MWRes_BadParamValue;
    }

    /* Read the header */
    numBytesRead = 0;
    memset (&rasterHeader, 0, sizeof(rasterHeader));

    numBytesRead = fread
    (
        &rasterHeader,
        1,
        sizeof(rasterHeader),
        p_RLEFile
    );
    if (numBytesRead != sizeof(rasterHeader))
    {
        if (ferror(p_RLEFile))
        {
            fclose (p_RLEFile);
            return MWRes_CommunicationsError;
        }
        else
        {
            fclose (p_RLEFile);
            return MWRes_OperationFailed;
        }
    }

    /* Make sure it is a raster header */
    if (rasterHeader.headerType != GenericRaster_ProductHeaderType)
    {
        fclose (p_RLEFile);
        return MWRes_OperationFailed;
    }

    /* Ensure that the bpp is <= 8 */
    if (rasterHeader.bitsPerPixel > 8)
    {
        fclose (p_RLEFile);
        return MWRes_OperationFailed;
    }

    /*
    //##############################################################
    // Read in the fixed scale specifier for each plane
    //##############################################################
    */
    if (rasterHeader.numPlanes > MAX_PLANES_IN_RASTER)
    {
        fclose (p_RLEFile);
        return MWRes_ProductHeaderFieldInvalid;
    }

    for (planeIndex = 0; planeIndex < rasterHeader.numPlanes; planeIndex ++)
    {
        /* Clear struct */
        memset
        (
            &fixedScaleSpecifiers[planeIndex],
            0,
            sizeof(TMW_GenericRasterProductHeader_FixedScaleSpecifier)
        );

        /* Read struct */
        numBytesRead = fread
        (
            &fixedScaleSpecifiers[planeIndex],
            1,
            sizeof(TMW_GenericRasterProductHeader_FixedScaleSpecifier),
            p_RLEFile
        );
        if (numBytesRead != sizeof(TMW_GenericRasterProductHeader_FixedScaleSpecifier))
        {
            if (ferror(p_RLEFile))
            {
                fclose (p_RLEFile);
                return MWRes_CommunicationsError;
            }
            else
            {
                fclose (p_RLEFile);
                return MWRes_OperationFailed;
            }
        }

    } /* Next plane fixed-scale specifier */

    /*
    //##############################################################
    // Based on the product ID, determine the expected number of
    // data planes, and the masks and shifts for merging each data
    // plane.
    //##############################################################
    */

    if
    (
        (rasterHeader.productID == CompositePrecipNEXRAD_MWProductID)
    )
    {
        /*
        //##############################################################
        // There are expected to be 2 planes
        //##############################################################
        */
        if (rasterHeader.numPlanes == 2)
        {
            numDataPlanesExpected = 2;

            /* First plane is precip intensity and does not shift */
            dataPlaneLeftShiftNumBits[0] = 0;

            /* Second plane is intensity and shifts to the left by 4 bytes */
            dataPlaneLeftShiftNumBits[1] = 4;
        }
        else if (rasterHeader.numPlanes == 1)
        {
            numDataPlanesExpected = 1;

            /* Only one plane, holds all bits */
            dataPlaneLeftShiftNumBits[0] = 0;

        }
        else
        {
            fclose (p_RLEFile);
            return MWRes_ProductHeaderFieldInvalid;
        }
    }
    else if
    (
        (rasterHeader.productID >= Min_SurfaceWindGrids_MWProductID) && (rasterHeader.productID <= Max_SurfaceWindGrids_MWProductID)
    )
    {
        /*
        //##############################################################
        // There are expected to be 2
        //##############################################################
        */
        numDataPlanesExpected = 2;

        if (planeNumber == 0)
        {
        	dataPlaneLeftShiftNumBits[0] = 0;
        	dataPlaneLeftShiftNumBits[1] = 8;
        }
        else if (planeNumber == 1)
        {
        	dataPlaneLeftShiftNumBits[0] = 8;
        	dataPlaneLeftShiftNumBits[1] = 0;
        }
        else if (planeNumber == 2)
        {
        	dataPlaneLeftShiftNumBits[0] = 0;
        	dataPlaneLeftShiftNumBits[1] = 0;
        }

    }
    else if
    (
        (rasterHeader.productID == Canadian_RADAR_MWProductID) ||
        (rasterHeader.productID == SeaSurfaceTemperatures_MWProductID) ||
        ((rasterHeader.productID >= Min_WaveGrids_MWProductID) && (rasterHeader.productID <= Max_WaveGrids_MWProductID))
    )
    {
        /*
        //##############################################################
        // There is only 1 plane, with no shift
        //##############################################################
        */
        numDataPlanesExpected = 1;
        dataPlaneLeftShiftNumBits[0] = 0;

    }
    else
    {
        /* Not recognized */
        fclose (p_RLEFile);
        return MWRes_OperationFailed;
    }

    /*
    //##############################################################
    // Does it have the expected number of planes?
    //##############################################################
    */
    if (rasterHeader.numPlanes != numDataPlanesExpected)
    {
        fclose (p_RLEFile);
        return MWRes_ProductHeaderFieldInvalid;
    }

    /*
    //##############################################################
    // Create memory to hold result for merging planes
    //##############################################################
    */
    numRasterSurfaceBytes = rasterHeader.numColumns * rasterHeader.numRows; /* No extra pitch */
    p_rasterSurface = (TByte_Unsigned*) malloc (numRasterSurfaceBytes);
    if (p_rasterSurface == NULL)
    {
        fclose (p_RLEFile);
        return MWRes_OutOfMemory;
    }

    /* Clear raster memory to all 0's */
    memset (p_rasterSurface, 0, numRasterSurfaceBytes);

    /*
    //##############################################################
    // Read in and merge each plane
    //##############################################################
    */

    for (planeIndex = 0; planeIndex < rasterHeader.numPlanes; planeIndex ++)
    {
        if (sourceCompressionType == WSI_RLE_MWCompressionType)
        {
            MWRes = MWTWSIRLEToRAW_Internal_processRasterPlane_RLE
            (
                rasterHeader.numColumns,
                rasterHeader.numRows,
                rasterHeader.numColumns, /* Pitch same as width */
                p_rasterSurface,
                p_RLEFile,
                dataPlaneLeftShiftNumBits[planeIndex]
            );
            if (MWRes != MWRes_Success)
            {
                fclose(p_RLEFile);
                free (p_rasterSurface);
                return MWRes;
            }
        }
        else if (sourceCompressionType == None_MWCompressionType)
        {
            MWRes = MWTWSIRLEToRAW_Internal_processRasterPlane_Raw
            (
                rasterHeader.numColumns,
                rasterHeader.numRows,
                rasterHeader.numColumns, /* Pitch same as width */
                p_rasterSurface,
                p_RLEFile,
                dataPlaneLeftShiftNumBits[planeIndex]
            );
            if (MWRes != MWRes_Success)
            {
                fclose(p_RLEFile);
                free (p_rasterSurface);
                return MWRes;
            }
        }
        else
        {
            return MWRes_BadParamValue;
        }
    }

    /*
    //##############################################################
    // Done with input
    //##############################################################
    */
    fclose (p_RLEFile);


    /*
    //##############################################################
    // Use internal method to write out image using the merged
    // plane
    //##############################################################
    */

    MWRes = MWTWSIRLEToRAW_Internal_write8BPPRAWFile
    (
        rasterHeader.numColumns,
        rasterHeader.numRows,
        rasterHeader.numColumns, /* Pitch same as width */
        p_rasterSurface,
        pstr_outputRAWFilePath,
        rasterHeader.productID
    );

    /*
    //##############################################################
    // Done with raster surface
    //##############################################################
    */
    free (p_rasterSurface);

    /*
    //##############################################################
    // Return result
    //##############################################################
    */
    return MWRes;

}
