/*
//====================================================================
//
// MWT_WSIRLEToPNG.h:
//
// TEST MODULE
//
// Copyright � 2010 - 2012 Sirius XM.  All rights reserved.
// Author: Alexander Pylchagin
//
// Implementation of the MWT_WSIRLEToPNG module.
// See .h file for interface and description.
//====================================================================
*/

#ifndef NO_SUPPORT_PNG

/*
//====================================================================
// Includes
//====================================================================
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "MWT_WSIRLEToPNG.h"
#include "MW_ProductHeaders.h"
#include "MWT_ProductIDs.h"
#include "png.h"
#include "osal.h"

/*
//====================================================================
//====================================================================
// Palettes for various product types shown as color images
//====================================================================
//====================================================================
*/

#define NUM_WSI_NOWRAD_PNG_PALETTE_COLORS     48
#define NUM_WSI_NOWRAD_PNG_TRANSPARENT_PALETTE_COLORS 3
#define NUM_WSI_SST_BMP_PALETTE_COLORS        60
#define NUM_WSI_CANRAD_BMP_PALETTE_COLORS     10
#define NUM_WSI_FULL_PALETTE_SIZE             256

#define BASE_COLOR_VAL 255
#define WHITENING_COLOR_VAL 64
#define DARKENING_COLOR_VAL 194

#define NO_COVERAGE_COLOR {128,128,128}
#define NO_ECHO_COLOR {0,0,0}

// These are the mm/hour rates
// Level    Rate
// 1        0.21 - 1.00 mm/hr
// 2        1.00 - 4.00 mm/hour
// 3        4.00 - 12.00 mm/hour
// 4        12.00 - 24.00 mm/hour
// 5        24.00 - 50.00 mm/hour
// 6        50.00 - 100.00 mm/hour
// 7        100.00 - 200.00 mm/hour
// 8        200.00 or greater mm/hour


#define CANADIAN_1 {0, 255, 48}    // up to 1 mm/hour is light green
#define CANADIAN_2 {0,150,0}        // up to 4 mm/hour is med green
#define CANADIAN_3 {0,51,0}        // up to 12 mm/hour is dark green
#define CANADIAN_4 {255,255,0}    // up to 24 mm/hour is high sat yellow
#define CANADIAN_5 {200,200,0}    // up to 50 mm/hour is med yellow
#define CANADIAN_6 {255,100,0}    // up to 100 mm/hour is orange
#define CANADIAN_7 {255,0,0}        // up to 200 mm/hour is high sat red
#define CANADIAN_8 {200,0,0}        // more is med. red

/*
//====================================================================
// WSI Palette colors Palette: 13 shades for precip types
//====================================================================
*/

#define PNG_MAKE_RGB(_Red, _Green, _Blue) \
    {(_Red), (_Green), (_Blue)}

#define WSIAV_TRANSPARENT PNG_MAKE_RGB(0, 0, 0)
#define WSIAV_PALE_GREEN PNG_MAKE_RGB(123, 176, 106)
#define WSIAV_PALE_LIGHT_GREEN PNG_MAKE_RGB(107, 159, 91)
#define WSIAV_LIGHT_GREEN1 PNG_MAKE_RGB(90, 142, 75)
#define WSIAV_LIGHT_GREEN2 PNG_MAKE_RGB(74, 125, 60)
#define WSIAV_MED_GREEN1 PNG_MAKE_RGB(58, 109, 45)
#define WSIAV_MED_GREEN2 PNG_MAKE_RGB(41, 92, 29)
#define WSIAV_DARK_GREEN PNG_MAKE_RGB(25, 75, 14)
#define WSIAV_YELLOW PNG_MAKE_RGB(215, 215, 0)
#define WSIAV_ORANGE PNG_MAKE_RGB(245, 166, 15)
#define WSIAV_BEIGE PNG_MAKE_RGB(225, 157, 95)
#define WSIAV_BROWN PNG_MAKE_RGB(178, 77, 0)
#define WSIAV_RED PNG_MAKE_RGB(206, 32, 25)
#define WSIAV_MED_RED PNG_MAKE_RGB(176, 24, 18)
#define WSIAV_DARK_RED PNG_MAKE_RGB(146, 16, 28)
#define WSIAV_PURPLE PNG_MAKE_RGB(255, 32, 223)
#define WSIAV_PINK2RED_01 PNG_MAKE_RGB(255, 220, 220)
#define WSIAV_PINK2RED_02 PNG_MAKE_RGB(255, 204, 204)
#define WSIAV_PINK2RED_03 PNG_MAKE_RGB(255, 188, 188)
#define WSIAV_PINK2RED_04 PNG_MAKE_RGB(255, 172, 172)
#define WSIAV_PINK2RED_05 PNG_MAKE_RGB(255, 156, 156)
#define WSIAV_PINK2RED_06 PNG_MAKE_RGB(255, 140, 140)
#define WSIAV_PINK2RED_07 PNG_MAKE_RGB(255, 124, 124)
#define WSIAV_PINK2RED_08 PNG_MAKE_RGB(255, 108, 108)
#define WSIAV_PINK2RED_09 PNG_MAKE_RGB(255, 96, 96)
#define WSIAV_PINK2RED_10 PNG_MAKE_RGB(255, 80, 80)
#define WSIAV_PINK2RED_11 PNG_MAKE_RGB(255, 63, 63)
#define WSIAV_PINK2RED_12 PNG_MAKE_RGB(255, 51, 51)
#define WSIAV_PINK2RED_13 PNG_MAKE_RGB(255, 38, 38)
#define WSIAV_PINK2RED_14 PNG_MAKE_RGB(255, 16, 16)
#define WSIAV_PINK2RED_15 PNG_MAKE_RGB(255, 0, 0)
#define WSIAV_LBLUE2DBLUE_01 PNG_MAKE_RGB(0, 239, 245)
#define WSIAV_LBLUE2DBLUE_02 PNG_MAKE_RGB(0, 212, 255)
#define WSIAV_LBLUE2DBLUE_03 PNG_MAKE_RGB(0, 191, 255)
#define WSIAV_LBLUE2DBLUE_04 PNG_MAKE_RGB(0, 173, 255)
#define WSIAV_LBLUE2DBLUE_05 PNG_MAKE_RGB(0, 148, 255)
#define WSIAV_LBLUE2DBLUE_06 PNG_MAKE_RGB(0, 123, 255)
#define WSIAV_LBLUE2DBLUE_07 PNG_MAKE_RGB(0, 104, 255)
#define WSIAV_LBLUE2DBLUE_08 PNG_MAKE_RGB(0, 85, 255)
#define WSIAV_LBLUE2DBLUE_09 PNG_MAKE_RGB(0, 67, 255)
#define WSIAV_LBLUE2DBLUE_10 PNG_MAKE_RGB(0, 38, 255)
#define WSIAV_LBLUE2DBLUE_11 PNG_MAKE_RGB(0, 14, 255)
#define WSIAV_LBLUE2DBLUE_12 PNG_MAKE_RGB(0, 0, 255)
#define WSIAV_LBLUE2DBLUE_13 PNG_MAKE_RGB(0, 0, 223)
#define WSIAV_LBLUE2DBLUE_14 PNG_MAKE_RGB(0, 0, 204)
#define WSIAV_LBLUE2DBLUE_15 PNG_MAKE_RGB(0, 0, 137)
/*
//====================================================================
// WSI NOWRad Palette with No Coverage Mask Bits
//====================================================================
*/

static png_color WSI_NOWRAD_NO_COVERAGE_MASK_PNG_PALETTE_COLORS[NUM_WSI_NOWRAD_PNG_PALETTE_COLORS] =
{
    //----------------------
    // SNOW
    //----------------------
    WSIAV_TRANSPARENT, // <5 (32)
    WSIAV_LBLUE2DBLUE_01, // 5-9
    WSIAV_LBLUE2DBLUE_02, // 10-14
    WSIAV_LBLUE2DBLUE_03, // 15-19
    WSIAV_LBLUE2DBLUE_04, // 20-24
    WSIAV_LBLUE2DBLUE_05, // 25-29
    WSIAV_LBLUE2DBLUE_06, // 30-34
    WSIAV_LBLUE2DBLUE_07, // 35-39
    WSIAV_LBLUE2DBLUE_08, // 40-44
    WSIAV_LBLUE2DBLUE_09, // 45-49
    WSIAV_LBLUE2DBLUE_10, // 50-54
    WSIAV_LBLUE2DBLUE_11, // 55-59
    WSIAV_LBLUE2DBLUE_12, // 60-64
    WSIAV_LBLUE2DBLUE_13, // 65-69
    WSIAV_LBLUE2DBLUE_14, // 70-74
    WSIAV_LBLUE2DBLUE_15, // > 75

    //----------------------
    // MIX
    //----------------------
    WSIAV_TRANSPARENT, // < 5 (16)
    WSIAV_PINK2RED_01, // 5-9
    WSIAV_PINK2RED_02, // 10-14
    WSIAV_PINK2RED_03, // 15-19
    WSIAV_PINK2RED_04, // 20-24
    WSIAV_PINK2RED_05, // 25-29
    WSIAV_PINK2RED_06, // 30-34
    WSIAV_PINK2RED_07, // 35-39
    WSIAV_PINK2RED_08, // 40-44
    WSIAV_PINK2RED_09, // 45-49
    WSIAV_PINK2RED_10, // 50-54
    WSIAV_PINK2RED_11, // 55-59
    WSIAV_PINK2RED_12, // 60-64
    WSIAV_PINK2RED_13, // 65-69
    WSIAV_PINK2RED_14, // 70-74
    WSIAV_PINK2RED_15, // > 75

    //----------------------
    // RAIN
    //----------------------
    WSIAV_TRANSPARENT, // < 5 (0)
    WSIAV_PALE_GREEN, // 5-9
    WSIAV_PALE_LIGHT_GREEN, // 10-14
    WSIAV_LIGHT_GREEN1, // 15-19
    WSIAV_LIGHT_GREEN2, // 20-24
    WSIAV_MED_GREEN1, // 25-29
    WSIAV_MED_GREEN2, // 30-34
    WSIAV_DARK_GREEN, // 35-39
    WSIAV_YELLOW, // 40-44
    WSIAV_ORANGE, // 45-49
    WSIAV_BEIGE, // 50-54
    WSIAV_BROWN, // 55-59
    WSIAV_RED, // 60-64
    WSIAV_MED_RED, // 65-69
    WSIAV_DARK_RED, // 70-74
    WSIAV_PURPLE // > 75
};

static png_byte WSI_NOWRAD_PALETTE_TRANSPARENT_COLORS[NUM_WSI_NOWRAD_PNG_PALETTE_COLORS] = {
    0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

/*
//====================================================================
// Canadian Radar (CANRAD) palette
//====================================================================
*/

static png_color WSI_CANRAD_BMP_PALETTE_COLORS [NUM_WSI_CANRAD_BMP_PALETTE_COLORS] =
{
    //----------------------
    // Simple 9 level palette
    // Rest are unused
    //----------------------

    // No echo
    NO_ECHO_COLOR,

    // No echo
    NO_ECHO_COLOR,

    // Canadian precip levels 1 through 8
    CANADIAN_1,
    CANADIAN_2,
    CANADIAN_3,
    CANADIAN_4,
    CANADIAN_5,
    CANADIAN_6,
    CANADIAN_7,
    CANADIAN_8

};

//====================================================================
// SST Palette
//====================================================================


//###################################
// BGR Color Table
//###################################

static png_color WSI_SST_BMP_PALETTE_COLORS[NUM_WSI_SST_BMP_PALETTE_COLORS] =
{
{8,8,8}, // 0, Land
{8,8,8}, // 1, not used
{8,8,8}, // 2, not used
{8,8,8}, // 3, not used
{8,8,8}, // 4, not used
{8,8,8}, // 5, not used
{8,8,8}, // 6, not used

{0,0,104}, // 7, < 4.444 C (40 F)
{0,0,112}, // 8, 4.444 C (40 F)
{0,0,124}, // 9
{0,0,142}, // 10
{0,0,160}, // 11
{0,0,178}, // 12
{0,0,196}, // 13
{0,0,214}, // 14
{0,0,232}, // 15
{0,7,255}, // 16
{0,24,255}, // 17
{0,42,255}, // 18
{0,60,255}, // 19
{0,78,255}, // 20
{0,96,255}, // 21
{0,114,255}, // 22
{0,132,255}, // 23
{0,150,255}, // 24
{0,168,255}, // 25
{0,186,255}, // 26
{0,204,255}, // 27
{0,222,255}, // 28
{0,240,255}, // 29
{0,255,251}, // 30
{12,255,236}, // 31
{30,255,218}, // 32
{50,255,200}, // 33
{68,255,182}, // 34
{86,255,164}, // 35
{104,255,146}, // 36
{122,255,128}, // 37
{142,255,110}, // 38
{160,255,92}, // 39
{178,255,74}, // 40
{196,255,56}, // 41
{214,255,38}, // 42
{232,255,20}, // 43
{250,255,2}, // 44
{255,242,0}, // 45
{255,224,0}, // 46
{255,206,0}, // 47
{255,188,0}, // 48
{255,170,0}, // 49
{255,152,0}, // 50
{255,134,0}, // 51
{255,128,0}, // 52
{255,122,0}, // 53
{255,116,0}, // 54
{255,110,0}, // 55
{255,104,0}, // 56
{255,98,0}, // 57
{255,92,0}, // 58, 32.222 C (90 F)
{255,74,0} // 59 > 32.222 C (90 F)

};


/*
//====================================================================
//====================================================================
// Internal methods
//====================================================================
//====================================================================
*/

/*
//====================================================================
// MWTWSIRLEToPNG_Internal_readRun:
//
// This method reads the WSI RLE run codes.
//
//
// PARAMETERS:
//
//    p_sourceFile:
//        The file holding the RLE codes.  It should currently be
//        positioned at the start of the next RLE code.  Calls to this
//        method automatically update it to point at the start of the
//        next RLE code in the file.
//
//    p_inout_unshiftedPixelValue:
//        Pointer to the byte that holds the previous unshifted pixel
//        value.  It will be replaced by the one indicated for the
//        new run.  If this is the first time this method is being
//        called in a raster plane, the pixel value should be set
//        to 0 before calling.
//
//        Note this is the value without any shifting for data plane
//        merging having been applied.
//
//    p_out_runLength:
//        Pointer to a variable which receives the number of consequetive
//        pixels, in scanline order, that occur of the unshifted pixel
//        value.
//
// RETURNS:
//
//    MWRes_Success
//
//    MWRes_NullPointer
//
//    MWRes_CommunicationsError:
//        On file read error
//
//    MWRes_ProductContentsInvalid:
//        On end of file
//
//====================================================================
*/
static TMW_ResultCode MWTWSIRLEToPNG_Internal_readRun
(
    FILE* p_sourceFile,
    TByte_Unsigned* p_inout_unshiftedPixelValue,
    TFourByteInteger_Unsigned* p_out_runLength
)
{
    TByte_Unsigned pixelHighNibble;
    TByte_Unsigned codeBytes[4];
    TFourByteInteger_Unsigned numBytesRead;


    /*
    //##############################################################
    // Test parameters
    //##############################################################
    */
    if ((p_sourceFile == NULL) || (p_inout_unshiftedPixelValue == NULL) || (p_out_runLength == NULL))
    {
        return MWRes_NullPointer;
    }

    /*
    //##############################################################
    // What was last high nibble of pixel value?
    //##############################################################
    */
    pixelHighNibble = *p_inout_unshiftedPixelValue & 0xF0;

    /*
    //##############################################################
    // Consume any high nibble change codes
    //##############################################################
    */

    while (1)
    {

        /* Read a byte */
        numBytesRead = 0;
        numBytesRead = fread (&codeBytes[0], sizeof(TByte_Unsigned), 1, p_sourceFile);
        if (numBytesRead != 1)
        {
            if (feof(p_sourceFile))
            {
                /* Not supposed to end here */
                return MWRes_ProductContentsInvalid;
            }
            else
            {
                return MWRes_CommunicationsError;
            }
        }

        /* Is it a high nibble change */
        if ((codeBytes[0] & 0xF0) == 0xF0)
        {
            /*
            // Its a pixel value high nibble change code
            // Low nibble gives new high nibble of pixel
            */
            pixelHighNibble = (codeBytes[0] & 0x0F) << 4;
        }
        else
        {
            /* Done with high nibble changes */
            break;
        }
    }

    /*
    //##############################################################
    // Branch based on high nibble of first byte
    //##############################################################
    */

    if ((codeBytes[0] & 0xF0) == 0xF0)
    {
        /* Code error */
        return MWRes_CodeFault;
    }
    else if ((codeBytes[0] & 0xF0) == 0xE0)
    {
        /*
        // Pixel value is current pixel high nibble and low nibble of this
        // byte.
        */
        *p_inout_unshiftedPixelValue = pixelHighNibble | (codeBytes[0] & 0x0F);

        /* Run length is (unsigned byte following this one) + 1*/
        /* Read a byte */
        numBytesRead = 0;
        numBytesRead = fread (&(codeBytes[1]), sizeof(TByte_Unsigned), 1, p_sourceFile);
        if (numBytesRead != 1)
        {
            if (feof(p_sourceFile))
            {
                /* Not supposed to end here */
                return MWRes_ProductContentsInvalid;
            }
            else
            {
                return MWRes_CommunicationsError;
            }
        }

        /* Set run length */
        *p_out_runLength = ((TFourByteInteger_Unsigned) codeBytes[1]) +1;


    }
    else if ((codeBytes[0] & 0xF0) == 0xD0)
    {
        /*
        // Pixel value is current pixel high nibble and low nibble of this
        // byte.
        */
        *p_inout_unshiftedPixelValue = pixelHighNibble | (codeBytes[0] & 0x0F);

        /* Run length is (two byte lohi unsigned int following this byte) + 1*/
        /* Read a byte */
        numBytesRead = 0;
        numBytesRead = fread (&(codeBytes[1]), sizeof(TByte_Unsigned), 2, p_sourceFile);
        if (numBytesRead != 2)
        {
            if (feof(p_sourceFile))
            {
                /* Not supposed to end here */
                return MWRes_ProductContentsInvalid;
            }
            else
            {
                return MWRes_CommunicationsError;
            }
        }

        /* Set run length */
        *p_out_runLength = (TFourByteInteger_Unsigned) (codeBytes[1]);
        *p_out_runLength = (*p_out_runLength) +
            ( ((TFourByteInteger_Unsigned) codeBytes[2]) * 256);
        *p_out_runLength = (*p_out_runLength) + 1;
    }
    else
    {
        /*
        // Pixel value is current pixel high nibble and low nibble of this
        // byte.
        */
        *p_inout_unshiftedPixelValue = pixelHighNibble | (codeBytes[0] & 0x0F);

        /*
        // Run length is high nibble +1
        */
        *p_out_runLength = ((codeBytes[0] & 0xF0) >> 4) + 1;

    }

    /*
    //##############################################################
    // Success
    //##############################################################
    */
    return MWRes_Success;

}

/*
//====================================================================
// MWTWSIRLEToPNG_Internal_processRasterPlane_RLE:
//
// This method reads the RLE codes for the plane and merges
// the pixels with the data already in the raster plane.
//
// The merge is achieved by leftshifting the run pixel value by
// the parameterized amount, and then bitwise orring the resulting
// pixel value with the value already in the raster surface.
//
// The RLE codes are assumed to start at the upper left of the
// image and encompass the entire raster body.
//
// PARAMETERS:
//
//    numColumns:
//        The number of columns in the raster
//
//    numRows:
//        The number of rows in the raster
//
//    numPitchBytes:
//        The number of bytes per row.  This must be
//        >= (numColumns * sizeof(TByte_Unsigned) )
//
//    p_rasterSurface:
//        The upper left pixel of the raster surface as an
//        array in scanline order.
//
//    p_RLEFile:
//        The source file from which the RLE codes will be read.
//        It should be positioned at the start of the first RLE code
//        of the plane before this method is called.
//
//    dataPlaneLeftShiftNumBits:
//        The number of bits to leftshift a run pixel value from
//        this plane before or-ing it with the merger value already
//        in the respective pixel of the merge raster surface.
//
// RETURNS:
//
//    MWRes_Success
//
//    MWRes_ProductContentsInvalid:
//        If reaches end of file mid run, or the runs do not line
//        up with the number of pixels in the plane.
//
//    MWRes_NullPointer:
//        If a pointer parameter is NULL
//
//    MWRes_BadParamValue:
//        If numPitchBytes <  (numColumns * sizeof(TByte_Unsigned) )
//
//====================================================================
*/
static TMW_ResultCode MWTWSIRLEToPNG_Internal_processRasterPlane_RLE
(
    TTwoByteInteger_Unsigned numColumns,
    TTwoByteInteger_Unsigned numRows,
    TFourByteInteger_Unsigned numPitchBytes,
    TByte_Unsigned* p_rasterSurface,
    FILE* p_RLEFile,
    TByte_Unsigned dataPlaneLeftShiftNumBits
)
{
    TMW_ResultCode MWRes;
    TFourByteInteger_Unsigned numPixelsInRaster;
    TFourByteInteger_Unsigned pixelWriteCount;
    TFourByteInteger_Unsigned runLengthRemaining;
    TByte_Unsigned unshiftedPixelValue;
    TByte_Unsigned shiftedPixelValue;
    TFourByteInteger_Unsigned extraPitchBytesPerRow;

    TByte_Unsigned *p_pixelCursor;
    TTwoByteInteger_Unsigned columnIndex;
    TTwoByteInteger_Unsigned rowIndex;

    /*
    //##############################################################
    // Test parameters
    //##############################################################
    */
    if (numPitchBytes < (numColumns * sizeof(TByte_Unsigned)) )
    {
        return MWRes_BadParamValue;
    }
    else if ((p_rasterSurface == NULL) || (p_RLEFile == NULL))
    {
        return MWRes_NullPointer;
    }

    /*
    //##############################################################
    // Calculate extra pitch bytes per row
    //##############################################################
    */
    extraPitchBytesPerRow = numPitchBytes - (numColumns * sizeof(TByte_Unsigned));

    /*
    //##############################################################
    // Iterate destination until it is full
    //##############################################################
    */
    numPixelsInRaster = numColumns * numRows;
    p_pixelCursor = p_rasterSurface;
    runLengthRemaining = 0;
    unshiftedPixelValue = 0;
    shiftedPixelValue = 0;

    pixelWriteCount = 0;
    for (rowIndex = 0; rowIndex < numRows; rowIndex ++)
    {
        for (columnIndex = 0; columnIndex < numColumns; columnIndex ++)
        {
            /*
            //##############################################################
            // If we need more run data, get some
            //##############################################################
            */
            if (runLengthRemaining == 0)
            {
                MWRes = MWTWSIRLEToPNG_Internal_readRun
                (
                    p_RLEFile,
                    &unshiftedPixelValue,
                    &runLengthRemaining
                );
                if (MWRes != MWRes_Success)
                {
                    return MWRes;
                }

                /* Check that run is not longer than end of raster */
                /* If it is, we have a problem */
                if (runLengthRemaining > (numPixelsInRaster - pixelWriteCount))
                {
                    return MWRes_ProductContentsInvalid;
                }

                /* Shift pixel value*/
                shiftedPixelValue = unshiftedPixelValue << dataPlaneLeftShiftNumBits;
            }

            /*
            //##############################################################
            // Encode run into the raster
            //
            //##############################################################
            */

            *p_pixelCursor = (*p_pixelCursor) | shiftedPixelValue;

            runLengthRemaining --;
            pixelWriteCount ++;

            /*
            //##############################################################
            // Bump up pixel cursor
            //##############################################################
            */
            p_pixelCursor += 1;


        } /* Next column */

        /* Bump up pixel cursor by extra pitch at end of row */
        p_pixelCursor += extraPitchBytesPerRow;


    } /* Next row */

    /*
    //##############################################################
    // Test for extra run length
    //##############################################################
    */
    if (runLengthRemaining != 0)
    {

        return MWRes_ProductContentsInvalid;
    }

    /*
    //##############################################################
    // Success
    //##############################################################
    */
    return MWRes_Success;

}


/*
//====================================================================
// MWTWSIRLEToPNG_Internal_processRasterPlane_Raw:
//
// This method reads the raw pixel codes for the plane and merges
// the pixels with the data already in the raster plane.
//
// The merge is achieved by leftshifting the run pixel value by
// the parameterized amount, and then bitwise orring the resulting
// pixel value with the value already in the raster surface.
//
// The pixels are assumed to start at the upper left of the
// image and encompass the entire raster body.
//
// PARAMETERS:
//
//    numColumns:
//        The number of columns in the raster
//
//    numRows:
//        The number of rows in the raster
//
//    numPitchBytes:
//        The number of bytes per row.  This must be
//        >= (numColumns * sizeof(TByte_Unsigned) )
//
//    p_rasterSurface:
//        The upper left pixel of the raster surface as an
//        array in scanline order.
//
//    p_sourceFile:
//        The source file from which the pixel codes will be read.
//        It should be positioned at the start of the first pixel code
//        of the plane before this method is called.
//
//    dataPlaneLeftShiftNumBits:
//        The number of bits to leftshift a run pixel value from
//        this plane before or-ing it with the merger value already
//        in the respective pixel of the merge raster surface.
//
// RETURNS:
//
//    MWRes_Success
//
//    MWRes_ProductContentsInvalid:
//        If reaches end of file mid run, or the runs do not line
//        up with the number of pixels in the plane.
//
//    MWRes_NullPointer:
//        If a pointer parameter is NULL
//
//    MWRes_BadParamValue:
//        If numPitchBytes <  (numColumns * sizeof(TByte_Unsigned) )
//
//====================================================================
*/
TMW_ResultCode MWTWSIRLEToPNG_Internal_processRasterPlane_Raw
(
    TTwoByteInteger_Unsigned numColumns,
    TTwoByteInteger_Unsigned numRows,
    TFourByteInteger_Unsigned numPitchBytes,
    TByte_Unsigned* p_rasterSurface,
    FILE* p_sourceFile,
    TByte_Unsigned dataPlaneLeftShiftNumBits
)
{
    //TMW_ResultCode MWRes;
    TFourByteInteger_Unsigned numPixelsInRaster;
    TFourByteInteger_Unsigned pixelWriteCount;
    TFourByteInteger_Unsigned runLengthRemaining;
    TByte_Unsigned unshiftedPixelValue;
    TByte_Unsigned shiftedPixelValue;
    TFourByteInteger_Unsigned extraPitchBytesPerRow;

    TByte_Unsigned *p_pixelCursor;
    TTwoByteInteger_Unsigned columnIndex;
    TTwoByteInteger_Unsigned rowIndex;
    TFourByteInteger_Signed ReadRes;

#define BYTE_BUFFER_SIZE 1024
    TByte_Unsigned byteBuffer[BYTE_BUFFER_SIZE];
    TFourByteInteger_Unsigned numBytesInByteBuffer = 0;
    TFourByteInteger_Unsigned numBytesInByteBufferRead = 0;

    /*
    //##############################################################
    // Test parameters
    //##############################################################
    */
    if (numPitchBytes < (numColumns * sizeof(TByte_Unsigned)) )
    {
        return MWRes_BadParamValue;
    }
    else if ((p_rasterSurface == NULL) || (p_sourceFile == NULL))
    {
        return MWRes_NullPointer;
    }

    /*
    //##############################################################
    // Calculate extra pitch bytes per row
    //##############################################################
    */
    extraPitchBytesPerRow = numPitchBytes - (numColumns * sizeof(TByte_Unsigned));

    /*
    //##############################################################
    // Iterate destination until it is full
    //##############################################################
    */
    numPixelsInRaster = numColumns * numRows;
    p_pixelCursor = p_rasterSurface;
    runLengthRemaining = 0;
    unshiftedPixelValue = 0;
    shiftedPixelValue = 0;

    pixelWriteCount = 0;
    for (rowIndex = 0; rowIndex < numRows; rowIndex ++)
    {
        for (columnIndex = 0; columnIndex < numColumns; columnIndex ++)
        {
            /*
            //##############################################################
            // If we need more run data, get some
            //##############################################################
            */
            if (runLengthRemaining == 0)
            {
                if (numBytesInByteBufferRead >= numBytesInByteBuffer)
                {
                    // Read next pixels
                    ReadRes = fread
                    (
                        byteBuffer,
                        1,
                        BYTE_BUFFER_SIZE,
                        p_sourceFile
                    );
                    if (ReadRes < 1)
                    {
                        return MWRes_ProductContentsInvalid;
                    }
                    numBytesInByteBuffer = ReadRes;
                    numBytesInByteBufferRead = 0;
                }

                // Pull byte from byte buffer
                unshiftedPixelValue = byteBuffer[numBytesInByteBufferRead];
                numBytesInByteBufferRead ++;

                // We read in 1 pixel
                runLengthRemaining = 1;

                /* Check that run is not longer than end of raster */
                /* If it is, we have a problem */
                if (runLengthRemaining > (numPixelsInRaster - pixelWriteCount))
                {
                    return MWRes_ProductContentsInvalid;
                }

                /* Shift pixel value*/
                shiftedPixelValue = unshiftedPixelValue << dataPlaneLeftShiftNumBits;
            }

            /*
            //##############################################################
            // Encode run into the raster
            //
            //##############################################################
            */

            *p_pixelCursor = (*p_pixelCursor) | shiftedPixelValue;

            runLengthRemaining --;
            pixelWriteCount ++;

            /*
            //##############################################################
            // Bump up pixel cursor
            //##############################################################
            */
            p_pixelCursor += 1;


        } /* Next column */

        /* Bump up pixel cursor by extra pitch at end of row */
        p_pixelCursor += extraPitchBytesPerRow;


    } /* Next row */

    /*
    //##############################################################
    // Test for extra run length
    //##############################################################
    */
    if (runLengthRemaining != 0)
    {

        return MWRes_ProductContentsInvalid;
    }

    /*
    //##############################################################
    // Success
    //##############################################################
    */
    return MWRes_Success;
}

/*
//====================================================================
// MWTWSIRLEToPNG_Internal_writeData:
//
// Provides writing PNG file data using specified stream instead of
// laying on commong libpng writing process
//
// PARAMETERS:
//
//    p_sStruct:
//        The PNG writing structure
//    p_data:
//        The buffer with data to be written
//    numBytes:
//        Number from bytes requested to be written from the buffer
//
//====================================================================
*/

void MWTWSIRLEToPNG_Internal_writeData(
    png_structp p_sStruct,
    png_bytep p_data,
    png_size_t numBytes)
{
    FILE* p_outFile = png_get_io_ptr(p_sStruct);

    if (p_outFile != NULL)
    {
        fwrite(p_data, 1, numBytes, p_outFile);
    }
}

/*
//====================================================================
// MWTWSIRLEToPNG_Internal_flushData:
//
// Provides flushing PNG file data using specified stream
//
// PARAMETERS:
//
//    p_sStruct:
//        The PNG writing structure
//
//====================================================================
*/

void MWTWSIRLEToPNG_Internal_flushData(
    png_structp p_sStruct)
{
    FILE* p_outFile = png_get_io_ptr(p_sStruct);

    if (p_outFile != NULL)
    {
        fflush(p_outFile);
    }
}
/*
//====================================================================
// MWTWSIRLEToPNG_Internal_write8BPPPNGFile:
//
// Given a raster in surface memory, writes out an 8BPP PNG
// file using a palette determined from the given ProductID.
//
// PARAMETERS:
//
//    numColumns:
//        The number of columns in the raster
//
//    numRows:
//        The number of rows in the raster
//
//    numSourcePitchBytes:
//        The number of bytes per row.  This must be
//        >= (numColumns * sizeof(TByte_Unsigned) )
//
//    p_rasterSurface:
//        The upper left pixel of the raster surface as an
//        array in scanline order.
//
//    pstr_pngFilePath:
//        The path of the file to be created/replaced with the
//        resulting bitmap image file.
//
//    productID:
//        The product ID from the product header. The palette used
//        in the image is dependent on this.
//
//
//====================================================================
*/
static TMW_ResultCode MWTWSIRLEToPNG_Internal_write8BPPPNGFile
(
    TTwoByteInteger_Unsigned numColumns,
    TTwoByteInteger_Unsigned numRows,
    TFourByteInteger_Unsigned numSourcePitchBytes,
    TByte_Unsigned* p_rasterSurface,
    const TASCIIChar* pstr_pngFilePath,
    TTwoByteInteger_Unsigned productID
)
{
    FILE* p_outFile;
    TMW_ResultCode resultCode = MWRes_OperationFailed;
    TFourByteInteger_Unsigned numPaletteEntries;
    TTwoByteInteger_Unsigned iRow;
    TTwoByteInteger_Unsigned iPalette;
    BOOLEAN bNeedTransparency = FALSE;
    png_color* p_palette;
    png_color grayscalePalette[NUM_WSI_FULL_PALETTE_SIZE];
    png_color p_outputPalette[NUM_WSI_FULL_PALETTE_SIZE];
    png_structp p_sStruct = NULL;
    png_infop p_sInfo = NULL;
    png_bytep* ppaRows = NULL;

    /*
    //##############################################################
    //  Test parameters
    //##############################################################
    */
    if ((pstr_pngFilePath == NULL) || (p_rasterSurface == NULL))
    {
        return MWRes_NullPointer;
    }

    /*
    //##############################################################
    // Get palette for product ID
    //##############################################################
    */

    if (productID == CompositePrecipNEXRAD_MWProductID)
    {
        numPaletteEntries = NUM_WSI_NOWRAD_PNG_PALETTE_COLORS;
        p_palette = WSI_NOWRAD_NO_COVERAGE_MASK_PNG_PALETTE_COLORS;
        bNeedTransparency = TRUE;
    }
    else if
    (
        (productID == Canadian_RADAR_MWProductID)
    )
    {
        numPaletteEntries = NUM_WSI_CANRAD_BMP_PALETTE_COLORS;
        p_palette = WSI_CANRAD_BMP_PALETTE_COLORS;

    }
    else if
    (
        (productID == SeaSurfaceTemperatures_MWProductID)
    )
    {
        numPaletteEntries = NUM_WSI_SST_BMP_PALETTE_COLORS;
        p_palette = WSI_SST_BMP_PALETTE_COLORS;
    }
    else if
    (
        ((productID >= Min_WaveGrids_MWProductID) &&
         (productID <= Max_WaveGrids_MWProductID)) ||
        ((productID >= Min_SurfaceWindGrids_MWProductID) &&
         (productID <= Max_SurfaceWindGrids_MWProductID))
    )
    {
        // Make grayscale palette
        for (numPaletteEntries = 0;
             numPaletteEntries < NUM_WSI_FULL_PALETTE_SIZE;
             numPaletteEntries ++)
        {
            grayscalePalette[numPaletteEntries].red = (TByte_Unsigned) numPaletteEntries;
            grayscalePalette[numPaletteEntries].green = (TByte_Unsigned) numPaletteEntries;
            grayscalePalette[numPaletteEntries].blue = (TByte_Unsigned) numPaletteEntries;
        }

        /* That is now our palette */
        numPaletteEntries = NUM_WSI_FULL_PALETTE_SIZE; /* redundant assignment, but just in case*/
        p_palette = grayscalePalette;
    }
    else
    {
        /* Not recognized */
        return MWRes_OperationFailed;
    }

    do
    {
        /*
        //##############################################################
        // Open the output file
        //##############################################################
        */
        p_outFile = fopen (pstr_pngFilePath, "wb");
        if (p_outFile == NULL)
        {
            resultCode = MWRes_CommunicationsError;
            break;
        }

        /*
        //#################################################################
        // Copy into our palette
        //#################################################################
        */
        for (iPalette = 0; iPalette < numPaletteEntries; ++iPalette)
        {
            p_outputPalette[iPalette] = p_palette[iPalette];
        }
        for (; iPalette < NUM_WSI_FULL_PALETTE_SIZE; iPalette ++)
        {
            p_outputPalette[iPalette].blue = 194;
            p_outputPalette[iPalette].green = 194;
            p_outputPalette[iPalette].red = 194;
        }

        //#################################################################
        // Fill out headers and writing them
        //#################################################################

        p_sStruct = png_create_write_struct(
                        PNG_LIBPNG_VER_STRING,
                        NULL, NULL, NULL
                            );
        if (p_sStruct == NULL)
        {
            printf("Failed to create PNG write structure");
            resultCode = MWRes_OutOfMemory;
            break;
        }

        p_sInfo = png_create_info_struct(p_sStruct);
        if (p_sInfo == NULL)
        {
            printf("Failed to create PNG info write structure");
            resultCode = MWRes_OutOfMemory;
            break;
        }

        if (setjmp(png_jmpbuf(p_sStruct)))
        {
            printf("Failed during write function initialization");
            resultCode = MWRes_CommunicationsError;
            break;
        }

        png_set_write_fn(p_sStruct, (png_voidp)p_outFile,
                MWTWSIRLEToPNG_Internal_writeData,
                MWTWSIRLEToPNG_Internal_flushData
                    );

        /* write header */
        if (setjmp(png_jmpbuf(p_sStruct)))
        {
            printf("Failed during writing PNG header");
            resultCode = MWRes_CommunicationsError;
            break;
        }

        // Sets the image header
        png_set_IHDR(p_sStruct, p_sInfo,
                     numColumns, numRows,
                     8, PNG_COLOR_TYPE_PALETTE,
                     PNG_INTERLACE_NONE,
                     PNG_COMPRESSION_TYPE_DEFAULT,
                     PNG_FILTER_TYPE_DEFAULT
                         );

        // Sets the image palette
        png_set_PLTE(p_sStruct, p_sInfo, p_outputPalette,
                     NUM_WSI_FULL_PALETTE_SIZE
                        );

        if (bNeedTransparency == TRUE)
        {
            png_set_tRNS(p_sStruct, p_sInfo,
                WSI_NOWRAD_PALETTE_TRANSPARENT_COLORS,
                NUM_WSI_NOWRAD_PNG_PALETTE_COLORS,
                NULL);
        }

        png_write_info(p_sStruct, p_sInfo);

        /* write bytes */
        if (setjmp(png_jmpbuf(p_sStruct)))
        {
            printf("Failed during writing PNG bytes");
            resultCode = MWRes_CommunicationsError;
            break;
        }

        //#################################################################
        // Write out the run encoded image bytes
        //#################################################################

        ppaRows = malloc(sizeof(png_bytep) * numRows);
        for (iRow = 0; iRow < numRows; ++iRow)
        {
            ppaRows[iRow] = (png_bytep)&p_rasterSurface[iRow * numColumns];
        }

        png_write_image(p_sStruct, ppaRows);

        //##############################################################
        // Done with output file
        //##############################################################
        if (setjmp(png_jmpbuf(p_sStruct)))
        {
            printf("Failed during end of PNG write");
            resultCode = MWRes_CommunicationsError;
            break;
        }

        png_write_end(p_sStruct, NULL);

        //##############################################################
        // Success
        //##############################################################
        resultCode = MWRes_Success;
    } while (FALSE);

    if (ppaRows != NULL)
    {
        free(ppaRows);
        ppaRows = NULL;
    }

    if (p_sStruct != NULL)
    {
        png_destroy_write_struct(&p_sStruct, &p_sInfo);
        p_sStruct = NULL;
        p_sInfo = NULL;
    }

    if (p_outFile != NULL)
    {
        fclose(p_outFile);
        p_outFile = NULL;
    }

    if (resultCode != MWRes_Success)
    {
        // Remove file if the compression has failed
        remove(pstr_pngFilePath);
    }

    return resultCode;
}

/*
//====================================================================
//====================================================================
// Public interface
//====================================================================
//====================================================================
*/

TMW_ResultCode MWTWSIRLEToPNG_convertRasterProductFile
(
    const TASCIIChar* pstr_rasterProductPath,
    const TASCIIChar* pstr_outputPNGFilePath,
    TMW_CompressionType sourceCompressionType,
    TByte_Unsigned planeNumber
)
{
    TMW_ResultCode MWRes;

    /* The number of data planes expected for this ProductID */
    TByte_Unsigned numDataPlanesExpected;

    /* The plane parsing loop*/
    TByte_Unsigned planeIndex;

    /* The amount to leftshift each data value for plane merge */
    TByte_Unsigned dataPlaneLeftShiftNumBits[4];

    /* The source file*/
    FILE* p_RLEFile;

    /* The expected header in the source file*/
    TMW_GenericRasterProductHeader_Output rasterHeader;
    TMW_GenericRasterProductHeader_FixedScaleSpecifier fixedScaleSpecifiers[MAX_PLANES_IN_RASTER];

    /* The number of bytes read */
    TFourByteInteger_Unsigned numBytesRead;

    /* The raster surface memory */
    TByte_Unsigned* p_rasterSurface;
    TFourByteInteger_Unsigned numRasterSurfaceBytes;


    /*
    //##############################################################
    // Test parameters
    //##############################################################
    */
    if ((pstr_rasterProductPath == NULL) || (pstr_outputPNGFilePath == NULL))
    {
        return MWRes_NullPointer;
    }

    /*
    //##############################################################
    // Open the source file and read in the raster header
    //##############################################################
    */

    p_RLEFile = fopen (pstr_rasterProductPath, "rb");
    if (p_RLEFile == NULL)
    {
        return MWRes_BadParamValue;
    }

    /* Read the header */
    numBytesRead = 0;
    memset (&rasterHeader, 0, sizeof(rasterHeader));

    numBytesRead = fread
    (
        &rasterHeader,
        1,
        sizeof(rasterHeader),
        p_RLEFile
    );
    if (numBytesRead != sizeof(rasterHeader))
    {
        if (ferror(p_RLEFile))
        {
            fclose (p_RLEFile);
            return MWRes_CommunicationsError;
        }
        else
        {
            fclose (p_RLEFile);
            return MWRes_OperationFailed;
        }
    }

    /* Make sure it is a raster header */
    if (rasterHeader.headerType != GenericRaster_ProductHeaderType)
    {
        fclose (p_RLEFile);
        return MWRes_OperationFailed;
    }

    /* Ensure that the bpp is <= 8 */
    if (rasterHeader.bitsPerPixel > 8)
    {
        fclose (p_RLEFile);
        return MWRes_OperationFailed;
    }

    /*
    //##############################################################
    // Read in the fixed scale specifier for each plane
    //##############################################################
    */
    if (rasterHeader.numPlanes > MAX_PLANES_IN_RASTER)
    {
        fclose (p_RLEFile);
        return MWRes_ProductHeaderFieldInvalid;
    }

    for (planeIndex = 0; planeIndex < rasterHeader.numPlanes; planeIndex ++)
    {
        /* Clear struct */
        memset
        (
            &fixedScaleSpecifiers[planeIndex],
            0,
            sizeof(TMW_GenericRasterProductHeader_FixedScaleSpecifier)
        );

        /* Read struct */
        numBytesRead = fread
        (
            &fixedScaleSpecifiers[planeIndex],
            1,
            sizeof(TMW_GenericRasterProductHeader_FixedScaleSpecifier),
            p_RLEFile
        );
        if (numBytesRead != sizeof(TMW_GenericRasterProductHeader_FixedScaleSpecifier))
        {
            if (ferror(p_RLEFile))
            {
                fclose (p_RLEFile);
                return MWRes_CommunicationsError;
            }
            else
            {
                fclose (p_RLEFile);
                return MWRes_OperationFailed;
            }
        }

    } /* Next plane fixed-scale specifier */

    /*
    //##############################################################
    // Based on the product ID, determine the expected number of
    // data planes, and the masks and shifts for merging each data
    // plane.
    //##############################################################
    */

    if
    (
        (rasterHeader.productID == CompositePrecipNEXRAD_MWProductID)
    )
    {
        /*
        //##############################################################
        // There are expected to be 2 planes
        //##############################################################
        */
        if (rasterHeader.numPlanes == 2)
        {
            numDataPlanesExpected = 2;

            /* First plane is precip intensity and does not shift */
            dataPlaneLeftShiftNumBits[0] = 0;

            /* Second plane is intensity and shifts to the left by 4 bytes */
            dataPlaneLeftShiftNumBits[1] = 4;
        }
        else if (rasterHeader.numPlanes == 1)
        {
            numDataPlanesExpected = 1;

            /* Only one plane, holds all bits */
            dataPlaneLeftShiftNumBits[0] = 0;

        }
        else
        {
            fclose (p_RLEFile);
            return MWRes_ProductHeaderFieldInvalid;
        }
    }
    else if
    (
        (rasterHeader.productID >= Min_SurfaceWindGrids_MWProductID) && (rasterHeader.productID <= Max_SurfaceWindGrids_MWProductID)
    )
    {
        /*
        //##############################################################
        // There are expected to be 2
        //##############################################################
        */
        numDataPlanesExpected = 2;

        if (planeNumber == 0)
        {
        	dataPlaneLeftShiftNumBits[0] = 0;
        	dataPlaneLeftShiftNumBits[1] = 8;
        }
        else if (planeNumber == 1)
        {
        	dataPlaneLeftShiftNumBits[0] = 8;
        	dataPlaneLeftShiftNumBits[1] = 0;
        }
        else if (planeNumber == 2)
        {
        	dataPlaneLeftShiftNumBits[0] = 0;
        	dataPlaneLeftShiftNumBits[1] = 0;
        }

    }
    else if
    (
        (rasterHeader.productID == Canadian_RADAR_MWProductID) ||
        (rasterHeader.productID == SeaSurfaceTemperatures_MWProductID) ||
        ((rasterHeader.productID >= Min_WaveGrids_MWProductID) && (rasterHeader.productID <= Max_WaveGrids_MWProductID))
    )
    {
        /*
        //##############################################################
        // There is only 1 plane, with no shift
        //##############################################################
        */
        numDataPlanesExpected = 1;
        dataPlaneLeftShiftNumBits[0] = 0;

    }
    else
    {
        /* Not recognized */
        fclose (p_RLEFile);
        return MWRes_OperationFailed;
    }

    /*
    //##############################################################
    // Does it have the expected number of planes?
    //##############################################################
    */
    if (rasterHeader.numPlanes != numDataPlanesExpected)
    {
        fclose (p_RLEFile);
        return MWRes_ProductHeaderFieldInvalid;
    }

    /*
    //##############################################################
    // Create memory to hold result for merging planes
    //##############################################################
    */
    numRasterSurfaceBytes = rasterHeader.numColumns * rasterHeader.numRows; /* No extra pitch */
    p_rasterSurface = malloc (numRasterSurfaceBytes);
    if (p_rasterSurface == NULL)
    {
        fclose (p_RLEFile);
        return MWRes_OutOfMemory;
    }

    /* Clear raster memory to all 0's */
    memset (p_rasterSurface, 0, numRasterSurfaceBytes);

    /*
    //##############################################################
    // Read in and merge each plane
    //##############################################################
    */

    for (planeIndex = 0; planeIndex < rasterHeader.numPlanes; planeIndex ++)
    {
        if (sourceCompressionType == WSI_RLE_MWCompressionType)
        {
            MWRes = MWTWSIRLEToPNG_Internal_processRasterPlane_RLE
            (
                rasterHeader.numColumns,
                rasterHeader.numRows,
                rasterHeader.numColumns, /* Pitch same as width */
                p_rasterSurface,
                p_RLEFile,
                dataPlaneLeftShiftNumBits[planeIndex]
            );
            if (MWRes != MWRes_Success)
            {
                fclose(p_RLEFile);
                free (p_rasterSurface);
                return MWRes;
            }
        }
        else if (sourceCompressionType == None_MWCompressionType)
        {
            MWRes = MWTWSIRLEToPNG_Internal_processRasterPlane_Raw
            (
                rasterHeader.numColumns,
                rasterHeader.numRows,
                rasterHeader.numColumns, /* Pitch same as width */
                p_rasterSurface,
                p_RLEFile,
                dataPlaneLeftShiftNumBits[planeIndex]
            );
            if (MWRes != MWRes_Success)
            {
                fclose(p_RLEFile);
                free (p_rasterSurface);
                return MWRes;
            }
        }
        else
        {
            return MWRes_BadParamValue;
        }
    }

    /*
    //##############################################################
    // Done with input
    //##############################################################
    */
    fclose (p_RLEFile);


    /*
    //##############################################################
    // Use internal method to write out image using the merged
    // plane
    //##############################################################
    */

    MWRes = MWTWSIRLEToPNG_Internal_write8BPPPNGFile
    (
        rasterHeader.numColumns,
        rasterHeader.numRows,
        rasterHeader.numColumns, /* Pitch same as width */
        p_rasterSurface,
        pstr_outputPNGFilePath,
        rasterHeader.productID
    );

    /*
    //##############################################################
    // Done with raster surface
    //##############################################################
    */
    free (p_rasterSurface);

    /*
    //##############################################################
    // Return result
    //##############################################################
    */
    return MWRes;

}

#endif /* NO_SUPPORT_PNG */

