/*
//====================================================================
//
// MWT_WSIRLEToBMP.h:
//
// TEST MODULE
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//
// Implementation of the MWT_WSIRLEToBMP module.
// See .h file for interface and description.
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "MWT_WSIRLEToBMP.h"
#include "MW_ProductHeaders.h"
#include "MWT_ProductIDs.h"


/*
//====================================================================
//====================================================================
// For windows bitmap file structures
//====================================================================
//====================================================================
*/

#ifndef WINDOWS

#define BYTE TByte_Unsigned
#define WORD TTwoByteInteger_Unsigned
#define DWORD TFourByteInteger_Unsigned
#define LONG TFourByteInteger_Signed

	#pragma pack(push, 1)
	typedef struct tagBITMAPFILEHEADER
	{
		WORD bfType;
		DWORD bfSize;
		WORD bfReserved1;
		WORD bfReserved2;
		DWORD bfOffBits;
	} BITMAPFILEHEADER;


	//-------------------------------------------------------------------

	typedef struct tagBITMAPINFOHEADER
	{
		DWORD biSize;
		LONG biWidth;
		LONG biHeight;
		WORD biPlanes;
		WORD biBitCount;
		DWORD biCompression;
		DWORD biSizeImage;
		LONG biXPelsPerMeter;
		LONG biYPelsPerMeter;
		DWORD biClrUsed;
		DWORD biClrImportant;
	} BITMAPINFOHEADER;

	//-------------------------------------------------------------------

	typedef struct tagRGBQUAD
	{
		BYTE b;
		BYTE g;
		BYTE r;
		BYTE a;
	} RGBQUAD;

	#pragma pack(pop)


	//-------------------------------------------------------------------

#define BI_RGB        0L
#define BI_RLE8       1L
#define BI_RLE4       2L
#define BI_BITFIELDS  3L

#endif

/*
//====================================================================
//====================================================================
// Palettes for various product types shown as color images
//====================================================================
//====================================================================
*/


#define NUM_WSI_NOWRAD_BMP_PALETTE_COLORS	64
#define NUM_WSI_SST_BMP_PALETTE_COLORS		60
#define NUM_WSI_CANRAD_BMP_PALETTE_COLORS	10

#define BASE_COLOR_VAL 255
#define WHITENING_COLOR_VAL 64
#define DARKENING_COLOR_VAL 194

#define NO_COVERAGE_COLOR {128,128,128,0}
#define NO_ECHO_COLOR {0,0,0,0}

// These are the mm/hour rates
// Level	Rate
// 1		0.21 - 1.00 mm/hr
// 2		1.00 - 4.00 mm/hour
// 3		4.00 - 12.00 mm/hour
// 4		12.00 - 24.00 mm/hour
// 5		24.00 - 50.00 mm/hour
// 6		50.00 - 100.00 mm/hour
// 7		100.00 - 200.00 mm/hour
// 8		200.00 or greater mm/hour


#define CANADIAN_1 {48, 255, 0, 0}	// up to 1 mm/hour is light green
#define CANADIAN_2 {0,150,0,0}		// up to 4 mm/hour is med green
#define CANADIAN_3 {0,51,0,0}		// up to 12 mm/hour is dark green
#define CANADIAN_4 {0,255,255,0}	// up to 24 mm/hour is high sat yellow
#define CANADIAN_5 {0,200,200,0}	// up to 50 mm/hour is med yellow
#define CANADIAN_6 {0,100,255,0}	// up to 100 mm/hour is orange
#define CANADIAN_7 {0,0,255,0}		// up to 200 mm/hour is high sat red
#define CANADIAN_8 {0,0,200,0}		// more is med. red

/*
//====================================================================
// WSI Palette colors Palette: 13 shades for precip types
//====================================================================
*/

#define BMP_MAKE_RGB(_Red, _Green, _Blue) \
    {(_Blue), (_Green), (_Red), 0}

#define WSIAV_TRANSPARENT BMP_MAKE_RGB(0, 0, 0)
#define WSIAV_PALE_GREEN BMP_MAKE_RGB(123, 176, 106)
#define WSIAV_PALE_LIGHT_GREEN BMP_MAKE_RGB(107, 159, 91)
#define WSIAV_LIGHT_GREEN1 BMP_MAKE_RGB(90, 142, 75)
#define WSIAV_LIGHT_GREEN2 BMP_MAKE_RGB(74, 125, 60)
#define WSIAV_MED_GREEN1 BMP_MAKE_RGB(58, 109, 45)
#define WSIAV_MED_GREEN2 BMP_MAKE_RGB(41, 92, 29)
#define WSIAV_DARK_GREEN BMP_MAKE_RGB(25, 75, 14)
#define WSIAV_YELLOW BMP_MAKE_RGB(215, 215, 0)
#define WSIAV_ORANGE BMP_MAKE_RGB(245, 166, 15)
#define WSIAV_BEIGE BMP_MAKE_RGB(225, 157, 95)
#define WSIAV_BROWN BMP_MAKE_RGB(178, 77, 0)
#define WSIAV_RED BMP_MAKE_RGB(206, 32, 25)
#define WSIAV_MED_RED BMP_MAKE_RGB(176, 24, 18)
#define WSIAV_DARK_RED BMP_MAKE_RGB(146, 16, 28)
#define WSIAV_PURPLE BMP_MAKE_RGB(255, 32, 223)
#define WSIAV_PINK2RED_01 BMP_MAKE_RGB(255, 220, 220)
#define WSIAV_PINK2RED_02 BMP_MAKE_RGB(255, 204, 204)
#define WSIAV_PINK2RED_03 BMP_MAKE_RGB(255, 188, 188)
#define WSIAV_PINK2RED_04 BMP_MAKE_RGB(255, 172, 172)
#define WSIAV_PINK2RED_05 BMP_MAKE_RGB(255, 156, 156)
#define WSIAV_PINK2RED_06 BMP_MAKE_RGB(255, 140, 140)
#define WSIAV_PINK2RED_07 BMP_MAKE_RGB(255, 124, 124)
#define WSIAV_PINK2RED_08 BMP_MAKE_RGB(255, 108, 108)
#define WSIAV_PINK2RED_09 BMP_MAKE_RGB(255, 96, 96)
#define WSIAV_PINK2RED_10 BMP_MAKE_RGB(255, 80, 80)
#define WSIAV_PINK2RED_11 BMP_MAKE_RGB(255, 63, 63)
#define WSIAV_PINK2RED_12 BMP_MAKE_RGB(255, 51, 51)
#define WSIAV_PINK2RED_13 BMP_MAKE_RGB(255, 38, 38)
#define WSIAV_PINK2RED_14 BMP_MAKE_RGB(255, 16, 16)
#define WSIAV_PINK2RED_15 BMP_MAKE_RGB(255, 0, 0)
#define WSIAV_LBLUE2DBLUE_01 BMP_MAKE_RGB(0, 239, 245)
#define WSIAV_LBLUE2DBLUE_02 BMP_MAKE_RGB(0, 212, 255)
#define WSIAV_LBLUE2DBLUE_03 BMP_MAKE_RGB(0, 191, 255)
#define WSIAV_LBLUE2DBLUE_04 BMP_MAKE_RGB(0, 173, 255)
#define WSIAV_LBLUE2DBLUE_05 BMP_MAKE_RGB(0, 148, 255)
#define WSIAV_LBLUE2DBLUE_06 BMP_MAKE_RGB(0, 123, 255)
#define WSIAV_LBLUE2DBLUE_07 BMP_MAKE_RGB(0, 104, 255)
#define WSIAV_LBLUE2DBLUE_08 BMP_MAKE_RGB(0, 85, 255)
#define WSIAV_LBLUE2DBLUE_09 BMP_MAKE_RGB(0, 67, 255)
#define WSIAV_LBLUE2DBLUE_10 BMP_MAKE_RGB(0, 38, 255)
#define WSIAV_LBLUE2DBLUE_11 BMP_MAKE_RGB(0, 14, 255)
#define WSIAV_LBLUE2DBLUE_12 BMP_MAKE_RGB(0, 0, 255)
#define WSIAV_LBLUE2DBLUE_13 BMP_MAKE_RGB(0, 0, 223)
#define WSIAV_LBLUE2DBLUE_14 BMP_MAKE_RGB(0, 0, 204)
#define WSIAV_LBLUE2DBLUE_15 BMP_MAKE_RGB(0, 0, 137)

/*
//====================================================================
// WSI NOWRad Palette with No Coverage Mask Bits
// 13 shades for precip types
//====================================================================
*/


static RGBQUAD WSI_NOWRAD_NO_COVERAGE_MASK_BMP_PALETTE_COLORS[NUM_WSI_NOWRAD_BMP_PALETTE_COLORS] =
{
    //----------------------
    // SNOW
    //----------------------
    WSIAV_TRANSPARENT, // <5 (32)
    WSIAV_LBLUE2DBLUE_01, // 5-9
    WSIAV_LBLUE2DBLUE_02, // 10-14
    WSIAV_LBLUE2DBLUE_03, // 15-19
    WSIAV_LBLUE2DBLUE_04, // 20-24
    WSIAV_LBLUE2DBLUE_05, // 25-29
    WSIAV_LBLUE2DBLUE_06, // 30-34
    WSIAV_LBLUE2DBLUE_07, // 35-39
    WSIAV_LBLUE2DBLUE_08, // 40-44
    WSIAV_LBLUE2DBLUE_09, // 45-49
    WSIAV_LBLUE2DBLUE_10, // 50-54
    WSIAV_LBLUE2DBLUE_11, // 55-59
    WSIAV_LBLUE2DBLUE_12, // 60-64
    WSIAV_LBLUE2DBLUE_13, // 65-69
    WSIAV_LBLUE2DBLUE_14, // 70-74
    WSIAV_LBLUE2DBLUE_15, // > 75

    //----------------------
    // MIX
    //----------------------
    WSIAV_TRANSPARENT, // < 5 (16)
    WSIAV_PINK2RED_01, // 5-9
    WSIAV_PINK2RED_02, // 10-14
    WSIAV_PINK2RED_03, // 15-19
    WSIAV_PINK2RED_04, // 20-24
    WSIAV_PINK2RED_05, // 25-29
    WSIAV_PINK2RED_06, // 30-34
    WSIAV_PINK2RED_07, // 35-39
    WSIAV_PINK2RED_08, // 40-44
    WSIAV_PINK2RED_09, // 45-49
    WSIAV_PINK2RED_10, // 50-54
    WSIAV_PINK2RED_11, // 55-59
    WSIAV_PINK2RED_12, // 60-64
    WSIAV_PINK2RED_13, // 65-69
    WSIAV_PINK2RED_14, // 70-74
    WSIAV_PINK2RED_15, // > 75

    //----------------------
    // RAIN
    //----------------------
    WSIAV_TRANSPARENT, // < 5 (0)
    WSIAV_PALE_GREEN, // 5-9
    WSIAV_PALE_LIGHT_GREEN, // 10-14
    WSIAV_LIGHT_GREEN1, // 15-19
    WSIAV_LIGHT_GREEN2, // 20-24
    WSIAV_MED_GREEN1, // 25-29
    WSIAV_MED_GREEN2, // 30-34
    WSIAV_DARK_GREEN, // 35-39
    WSIAV_YELLOW, // 40-44
    WSIAV_ORANGE, // 45-49
    WSIAV_BEIGE, // 50-54
    WSIAV_BROWN, // 55-59
    WSIAV_RED, // 60-64
    WSIAV_MED_RED, // 65-69
    WSIAV_DARK_RED, // 70-74
    WSIAV_PURPLE // > 75
};

/*
//====================================================================
// Canadian Radar (CANRAD) palette
//====================================================================
*/

static RGBQUAD WSI_CANRAD_BMP_PALETTE_COLORS [NUM_WSI_CANRAD_BMP_PALETTE_COLORS] =
{
	//----------------------
	// Simple 9 level palette
	// Rest are unused
	//----------------------

	// No echo
	NO_ECHO_COLOR,

	// No echo
	NO_ECHO_COLOR,

	// Canadian precip levels 1 through 8
	CANADIAN_1,
	CANADIAN_2,
	CANADIAN_3,
	CANADIAN_4,
	CANADIAN_5,
	CANADIAN_6,
	CANADIAN_7,
	CANADIAN_8

};

//====================================================================
// SST Palette
//====================================================================


//###################################
// BGR Color Table
//###################################

static RGBQUAD WSI_SST_BMP_PALETTE_COLORS[NUM_WSI_SST_BMP_PALETTE_COLORS] =
{
{8,8,8,0}, // 0, Land
{8,8,8,0}, // 1, not used
{8,8,8,0}, // 2, not used
{8,8,8,0}, // 3, not used
{8,8,8,0}, // 4, not used
{8,8,8,0}, // 5, not used
{8,8,8,0}, // 6, not used

{104,0,0,0}, // 7, < 4.444 C (40 F)
{112,0,0,0}, // 8, 4.444 C (40 F)
{124,0,0,0}, // 9
{142,0,0,0}, // 10
{160,0,0,0}, // 11
{178,0,0,0}, // 12
{196,0,0,0}, // 13
{214,0,0,0}, // 14
{232,0,0,0}, // 15
{255,7,0,0}, // 16
{255,24,0,0}, // 17
{255,42,0,0}, // 18
{255,60,0,0}, // 19
{255,78,0,0}, // 20
{255,96,0,0}, // 21
{255,114,0,0}, // 22
{255,132,0,0}, // 23
{255,150,0,0}, // 24
{255,168,0,0}, // 25
{255,186,0,0}, // 26
{255,204,0,0}, // 27
{255,222,0,0}, // 28
{255,240,0,0}, // 29
{251,255,0,0}, // 30
{236,255,12,0}, // 31
{218,255,30,0}, // 32
{200,255,50,0}, // 33
{182,255,68,0}, // 34
{164,255,86,0}, // 35
{146,255,104,0}, // 36
{128,255,122,0}, // 37
{110,255,142,0}, // 38
{92,255,160,0}, // 39
{74,255,178,0}, // 40
{56,255,196,0}, // 41
{38,255,214,0}, // 42
{20,255,232,0}, // 43
{2,255,250,0}, // 44
{0,242,255,0}, // 45
{0,224,255,0}, // 46
{0,206,255,0}, // 47
{0,188,255,0}, // 48
{0,170,255,0}, // 49
{0,152,255,0}, // 50
{0,134,255,0}, // 51
{0,128,255,0}, // 52
{0,122,255,0}, // 53
{0,116,255,0}, // 54
{0,110,255,0}, // 55
{0,104,255,0}, // 56
{0,98,255,0}, // 57
{0,92,255,0}, // 58, 32.222 C (90 F)
{0,74,255,0} // 59 > 32.222 C (90 F)

};


/*
//====================================================================
//====================================================================
// Internal methods
//====================================================================
//====================================================================
*/

/*
//====================================================================
// MWTWSIRLEToBMP_Internal_readRun:
//
// This method reads the WSI RLE run codes.
//
//
// PARAMETERS:
//
//	p_sourceFile:
//		The file holding the RLE codes.  It should currently be
//		positioned at the start of the next RLE code.  Calls to this
//		method automatically update it to point at the start of the
//		next RLE code in the file.
//
//	p_inout_unshiftedPixelValue:
//		Pointer to the byte that holds the previous unshifted pixel
//		value.  It will be replaced by the one indicated for the
//		new run.  If this is the first time this method is being
//		called in a raster plane, the pixel value should be set
//		to 0 before calling.
//
//		Note this is the value without any shifting for data plane
//		merging having been applied.
//
//	p_out_runLength:
//		Pointer to a variable which receives the number of consequetive
//		pixels, in scanline order, that occur of the unshifted pixel
//		value.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_NullPointer
//
//	MWRes_CommunicationsError:
//		On file read error
//
//	MWRes_ProductContentsInvalid:
//		On end of file
//
//====================================================================
*/
TMW_ResultCode MWTWSIRLEToBMP_Internal_readRun
(
	FILE* p_sourceFile,
	TByte_Unsigned* p_inout_unshiftedPixelValue,
	TFourByteInteger_Unsigned* p_out_runLength
)
{
	TByte_Unsigned pixelHighNibble;
	TByte_Unsigned codeBytes[4];
	TFourByteInteger_Unsigned numBytesRead;


	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if ((p_sourceFile == NULL) || (p_inout_unshiftedPixelValue == NULL) || (p_out_runLength == NULL))
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// What was last high nibble of pixel value?
	//##############################################################
	*/
	pixelHighNibble = *p_inout_unshiftedPixelValue & 0xF0;

	/*
	//##############################################################
	// Consume any high nibble change codes
	//##############################################################
	*/

	while (1)
	{

		/* Read a byte */
		numBytesRead = 0;
		numBytesRead = fread (&codeBytes[0], sizeof(TByte_Unsigned), 1, p_sourceFile);
		if (numBytesRead != 1)
		{
			if (feof(p_sourceFile))
			{
				/* Not supposed to end here */
				return MWRes_ProductContentsInvalid;
			}
			else
			{
				return MWRes_CommunicationsError;
			}
		}

		/* Is it a high nibble change */
		if ((codeBytes[0] & 0xF0) == 0xF0)
		{
			/*
			// Its a pixel value high nibble change code
			// Low nibble gives new high nibble of pixel
			*/
			pixelHighNibble = (codeBytes[0] & 0x0F) << 4;
		}
		else
		{
			/* Done with high nibble changes */
			break;
		}
	}

	/*
	//##############################################################
	// Branch based on high nibble of first byte
	//##############################################################
	*/

	if ((codeBytes[0] & 0xF0) == 0xF0)
	{
		/* Code error */
		return MWRes_CodeFault;
	}
	else if ((codeBytes[0] & 0xF0) == 0xE0)
	{
		/*
		// Pixel value is current pixel high nibble and low nibble of this
		// byte.
		*/
		*p_inout_unshiftedPixelValue = pixelHighNibble | (codeBytes[0] & 0x0F);

		/* Run length is (unsigned byte following this one) + 1*/
		/* Read a byte */
		numBytesRead = 0;
		numBytesRead = fread (&(codeBytes[1]), sizeof(TByte_Unsigned), 1, p_sourceFile);
		if (numBytesRead != 1)
		{
			if (feof(p_sourceFile))
			{
				/* Not supposed to end here */
				return MWRes_ProductContentsInvalid;
			}
			else
			{
				return MWRes_CommunicationsError;
			}
		}

		/* Set run length */
		*p_out_runLength = ((TFourByteInteger_Unsigned) codeBytes[1]) +1;


	}
	else if ((codeBytes[0] & 0xF0) == 0xD0)
	{
		/*
		// Pixel value is current pixel high nibble and low nibble of this
		// byte.
		*/
		*p_inout_unshiftedPixelValue = pixelHighNibble | (codeBytes[0] & 0x0F);

		/* Run length is (two byte lohi unsigned int following this byte) + 1*/
		/* Read a byte */
		numBytesRead = 0;
		numBytesRead = fread (&(codeBytes[1]), sizeof(TByte_Unsigned), 2, p_sourceFile);
		if (numBytesRead != 2)
		{
			if (feof(p_sourceFile))
			{
				/* Not supposed to end here */
				return MWRes_ProductContentsInvalid;
			}
			else
			{
				return MWRes_CommunicationsError;
			}
		}

		/* Set run length */
		*p_out_runLength = (TFourByteInteger_Unsigned) (codeBytes[1]);
		*p_out_runLength = (*p_out_runLength) +
			( ((TFourByteInteger_Unsigned) codeBytes[2]) * 256);
		*p_out_runLength = (*p_out_runLength) + 1;
	}
	else
	{
		/*
		// Pixel value is current pixel high nibble and low nibble of this
		// byte.
		*/
		*p_inout_unshiftedPixelValue = pixelHighNibble | (codeBytes[0] & 0x0F);

		/*
		// Run length is high nibble +1
		*/
		*p_out_runLength = ((codeBytes[0] & 0xF0) >> 4) + 1;

	}

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;



}


/*
//====================================================================
// MWTWSIRLEToBMP_Internal_processRasterPlane_RLE:
//
// This method reads the RLE codes for the plane and merges
// the pixels with the data already in the raster plane.
//
// The merge is achieved by leftshifting the run pixel value by
// the parameterized amount, and then bitwise orring the resulting
// pixel value with the value already in the raster surface.
//
// The RLE codes are assumed to start at the upper left of the
// image and encompass the entire raster body.
//
// PARAMETERS:
//
//	numColumns:
//		The number of columns in the raster
//
//	numRows:
//		The number of rows in the raster
//
//	numPitchBytes:
//		The number of bytes per row.  This must be
//		>= (numColumns * sizeof(TByte_Unsigned) )
//
//	p_rasterSurface:
//		The upper left pixel of the raster surface as an
//		array in scanline order.
//
//	p_RLEFile:
//		The source file from which the RLE codes will be read.
//		It should be positioned at the start of the first RLE code
//		of the plane before this method is called.
//
//	dataPlaneLeftShiftNumBits:
//		The number of bits to leftshift a run pixel value from
//		this plane before or-ing it with the merger value already
//		in the respective pixel of the merge raster surface.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_ProductContentsInvalid:
//		If reaches end of file mid run, or the runs do not line
//		up with the number of pixels in the plane.
//
//	MWRes_NullPointer:
//		If a pointer parameter is NULL
//
//	MWRes_BadParamValue:
//		If numPitchBytes <  (numColumns * sizeof(TByte_Unsigned) )
//
//====================================================================
*/
TMW_ResultCode MWTWSIRLEToBMP_Internal_processRasterPlane_RLE
(
	TTwoByteInteger_Unsigned numColumns,
	TTwoByteInteger_Unsigned numRows,
	TFourByteInteger_Unsigned numPitchBytes,
	TByte_Unsigned* p_rasterSurface,
	FILE* p_RLEFile,
	TByte_Unsigned dataPlaneLeftShiftNumBits
)
{
	TMW_ResultCode MWRes;
	TFourByteInteger_Unsigned numPixelsInRaster;
	TFourByteInteger_Unsigned pixelWriteCount;
	TFourByteInteger_Unsigned runLengthRemaining;
	TByte_Unsigned unshiftedPixelValue;
	TByte_Unsigned shiftedPixelValue;
	TFourByteInteger_Unsigned extraPitchBytesPerRow;

	TByte_Unsigned *p_pixelCursor;
	TTwoByteInteger_Unsigned columnIndex;
	TTwoByteInteger_Unsigned rowIndex;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if (numPitchBytes < (numColumns * sizeof(TByte_Unsigned)) )
	{
		return MWRes_BadParamValue;
	}
	else if ((p_rasterSurface == NULL) || (p_RLEFile == NULL))
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Calculate extra pitch bytes per row
	//##############################################################
	*/
	extraPitchBytesPerRow = numPitchBytes - (numColumns * sizeof(TByte_Unsigned));

	/*
	//##############################################################
	// Iterate destination until it is full
	//##############################################################
	*/
	numPixelsInRaster = numColumns * numRows;
	p_pixelCursor = p_rasterSurface;
	runLengthRemaining = 0;
	unshiftedPixelValue = 0;
	shiftedPixelValue = 0;

	pixelWriteCount = 0;
	for (rowIndex = 0; rowIndex < numRows; rowIndex ++)
	{
		for (columnIndex = 0; columnIndex < numColumns; columnIndex ++)
		{
			/*
			//##############################################################
			// If we need more run data, get some
			//##############################################################
			*/
			if (runLengthRemaining == 0)
			{
				MWRes = MWTWSIRLEToBMP_Internal_readRun
				(
					p_RLEFile,
					&unshiftedPixelValue,
					&runLengthRemaining
				);
				if (MWRes != MWRes_Success)
				{
					return MWRes;
				}

				/* Check that run is not longer than end of raster */
				/* If it is, we have a problem */
				if (runLengthRemaining > (numPixelsInRaster - pixelWriteCount))
				{
					return MWRes_ProductContentsInvalid;
				}

				/* Shift pixel value*/
				shiftedPixelValue = unshiftedPixelValue << dataPlaneLeftShiftNumBits;
			}

			/*
			//##############################################################
			// Encode run into the raster
			//
			//##############################################################
			*/

			*p_pixelCursor = (*p_pixelCursor) | shiftedPixelValue;

			runLengthRemaining --;
			pixelWriteCount ++;

			/*
			//##############################################################
			// Bump up pixel cursor
			//##############################################################
			*/
			p_pixelCursor += 1;


		} /* Next column */

		/* Bump up pixel cursor by extra pitch at end of row */
		p_pixelCursor += extraPitchBytesPerRow;


	} /* Next row */

	/*
	//##############################################################
	// Test for extra run length
	//##############################################################
	*/
	if (runLengthRemaining != 0)
	{

		return MWRes_ProductContentsInvalid;
	}

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;



}


/*
//====================================================================
// MWTWSIRLEToBMP_Internal_processRasterPlane_Raw:
//
// This method reads the raw pixel codes for the plane and merges
// the pixels with the data already in the raster plane.
//
// The merge is achieved by leftshifting the run pixel value by
// the parameterized amount, and then bitwise orring the resulting
// pixel value with the value already in the raster surface.
//
// The pixels are assumed to start at the upper left of the
// image and encompass the entire raster body.
//
// PARAMETERS:
//
//	numColumns:
//		The number of columns in the raster
//
//	numRows:
//		The number of rows in the raster
//
//	numPitchBytes:
//		The number of bytes per row.  This must be
//		>= (numColumns * sizeof(TByte_Unsigned) )
//
//	p_rasterSurface:
//		The upper left pixel of the raster surface as an
//		array in scanline order.
//
//	p_sourceFile:
//		The source file from which the pixel codes will be read.
//		It should be positioned at the start of the first pixel code
//		of the plane before this method is called.
//
//	dataPlaneLeftShiftNumBits:
//		The number of bits to leftshift a run pixel value from
//		this plane before or-ing it with the merger value already
//		in the respective pixel of the merge raster surface.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_ProductContentsInvalid:
//		If reaches end of file mid run, or the runs do not line
//		up with the number of pixels in the plane.
//
//	MWRes_NullPointer:
//		If a pointer parameter is NULL
//
//	MWRes_BadParamValue:
//		If numPitchBytes <  (numColumns * sizeof(TByte_Unsigned) )
//
//====================================================================
*/
TMW_ResultCode MWTWSIRLEToBMP_Internal_processRasterPlane_Raw
(
	TTwoByteInteger_Unsigned numColumns,
	TTwoByteInteger_Unsigned numRows,
	TFourByteInteger_Unsigned numPitchBytes,
	TByte_Unsigned* p_rasterSurface,
	FILE* p_sourceFile,
	TByte_Unsigned dataPlaneLeftShiftNumBits
)
{
	//TMW_ResultCode MWRes;
	TFourByteInteger_Unsigned numPixelsInRaster;
	TFourByteInteger_Unsigned pixelWriteCount;
	TFourByteInteger_Unsigned runLengthRemaining;
	TByte_Unsigned unshiftedPixelValue;
	TByte_Unsigned shiftedPixelValue;
	TFourByteInteger_Unsigned extraPitchBytesPerRow;

	TByte_Unsigned *p_pixelCursor;
	TTwoByteInteger_Unsigned columnIndex;
	TTwoByteInteger_Unsigned rowIndex;
	TFourByteInteger_Signed ReadRes;

#define BYTE_BUFFER_SIZE 1024
	TByte_Unsigned byteBuffer[BYTE_BUFFER_SIZE];
	TFourByteInteger_Unsigned numBytesInByteBuffer = 0;
	TFourByteInteger_Unsigned numBytesInByteBufferRead = 0;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if (numPitchBytes < (numColumns * sizeof(TByte_Unsigned)) )
	{
		return MWRes_BadParamValue;
	}
	else if ((p_rasterSurface == NULL) || (p_sourceFile == NULL))
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Calculate extra pitch bytes per row
	//##############################################################
	*/
	extraPitchBytesPerRow = numPitchBytes - (numColumns * sizeof(TByte_Unsigned));

	/*
	//##############################################################
	// Iterate destination until it is full
	//##############################################################
	*/
	numPixelsInRaster = numColumns * numRows;
	p_pixelCursor = p_rasterSurface;
	runLengthRemaining = 0;
	unshiftedPixelValue = 0;
	shiftedPixelValue = 0;

	pixelWriteCount = 0;
	for (rowIndex = 0; rowIndex < numRows; rowIndex ++)
	{
		for (columnIndex = 0; columnIndex < numColumns; columnIndex ++)
		{
			/*
			//##############################################################
			// If we need more run data, get some
			//##############################################################
			*/
			if (runLengthRemaining == 0)
			{
				if (numBytesInByteBufferRead >= numBytesInByteBuffer)
				{
					// Read next pixels
					ReadRes = fread
					(
						byteBuffer,
						1,
						BYTE_BUFFER_SIZE,
						p_sourceFile
					);
					if (ReadRes < 1)
					{
						return MWRes_ProductContentsInvalid;
					}
					numBytesInByteBuffer = ReadRes;
					numBytesInByteBufferRead = 0;
				}

				// Pull byte from byte buffer
				unshiftedPixelValue = byteBuffer[numBytesInByteBufferRead];
				numBytesInByteBufferRead ++;

				// We read in 1 pixel
				runLengthRemaining = 1;

				/* Check that run is not longer than end of raster */
				/* If it is, we have a problem */
				if (runLengthRemaining > (numPixelsInRaster - pixelWriteCount))
				{
					return MWRes_ProductContentsInvalid;
				}

				/* Shift pixel value*/
				shiftedPixelValue = unshiftedPixelValue << dataPlaneLeftShiftNumBits;
			}

			/*
			//##############################################################
			// Encode run into the raster
			//
			//##############################################################
			*/

			*p_pixelCursor = (*p_pixelCursor) | shiftedPixelValue;

			runLengthRemaining --;
			pixelWriteCount ++;

			/*
			//##############################################################
			// Bump up pixel cursor
			//##############################################################
			*/
			p_pixelCursor += 1;


		} /* Next column */

		/* Bump up pixel cursor by extra pitch at end of row */
		p_pixelCursor += extraPitchBytesPerRow;


	} /* Next row */

	/*
	//##############################################################
	// Test for extra run length
	//##############################################################
	*/
	if (runLengthRemaining != 0)
	{

		return MWRes_ProductContentsInvalid;
	}

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;



}


/*
//====================================================================
// MWTWSIRLEToBMP_Internal_write8BPPBitmapFile:
//
// Given a raster in surface memory, writes out an 8BPP Windows Bitmap
// file using a palette determined from the given ProductID.
//
// PARAMETERS:
//
//	numColumns:
//		The number of columns in the raster
//
//	numRows:
//		The number of rows in the raster
//
//	numSourcePitchBytes:
//		The number of bytes per row.  This must be
//		>= (numColumns * sizeof(TByte_Unsigned) )
//
//	p_rasterSurface:
//		The upper left pixel of the raster surface as an
//		array in scanline order.
//
//	pstr_bitmapFilePath:
//		The path of the file to be created/replaced with the
//		resulting bitmap image file.
//
//	productID:
//		The product ID from the product header. The palette used
//		in the image is dependent on this.
//
//
//====================================================================
*/
TMW_ResultCode MWTWSIRLEToBMP_Internal_write8BPPBitmapFile
(
	TTwoByteInteger_Unsigned numColumns,
	TTwoByteInteger_Unsigned numRows,
	TFourByteInteger_Unsigned numSourcePitchBytes,
	TByte_Unsigned* p_rasterSurface,
	const TASCIIChar* pstr_bitmapFilePath,
	TTwoByteInteger_Unsigned productID
)
{
	FILE* p_outFile;

	TFourByteInteger_Unsigned numPaletteEntries;
	RGBQUAD* p_palette;
	RGBQUAD grayscalePalette[256];
	TFourByteInteger_Unsigned PitchBytes;
	TFourByteInteger_Unsigned BMPPitchPad;

	TFourByteInteger_Unsigned i;

	/* For building RLE code */
	TByte_Unsigned* p_RLEBytes;
	TFourByteInteger_Unsigned Col;
	TFourByteInteger_Signed Row;
	TFourByteInteger_Unsigned ROffset;
	TFourByteInteger_Unsigned RLEOffset;
	TByte_Unsigned  RunLength;
	TByte_Unsigned CurrentPixelByte;
	TFourByteInteger_Unsigned TotalRowLength;
	TFourByteInteger_Unsigned RLEBytesThisRow;
	TByte_Unsigned NewPixelByte;
	TFourByteInteger_Unsigned NumRLEBytes;

	/* For writing out the file */
	TByte_Unsigned padByte = 0;
	TFourByteInteger_Unsigned numBytesWritten;
	TFourByteInteger_Unsigned UnpaddedFileSize;
	TFourByteInteger_Unsigned TotalImageBytes;
	TFourByteInteger_Unsigned PaddingNeeded;

	//#################################################################
	// Some structures we will use
	//#################################################################
	#pragma pack(push, 1)
		BITMAPFILEHEADER FileHeader;
		BITMAPINFOHEADER BMPHeader;
		RGBQUAD p_outputPalette[256];
	#pragma pack(pop)

	/*
	//##############################################################
	//  Test parameters
	//##############################################################
	*/
	if ((pstr_bitmapFilePath == NULL) || (p_rasterSurface == NULL))
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Get palette for product ID
	//##############################################################
	*/

	if (productID == CompositePrecipNEXRAD_MWProductID)
	{
		numPaletteEntries = NUM_WSI_NOWRAD_BMP_PALETTE_COLORS;
		p_palette = WSI_NOWRAD_NO_COVERAGE_MASK_BMP_PALETTE_COLORS;
	}
	else if
	(
		(productID == Canadian_RADAR_MWProductID)
	)
	{
		numPaletteEntries = NUM_WSI_CANRAD_BMP_PALETTE_COLORS;
		p_palette = WSI_CANRAD_BMP_PALETTE_COLORS;

	}
	else if
	(
		(productID == SeaSurfaceTemperatures_MWProductID)
	)
	{
		numPaletteEntries = NUM_WSI_SST_BMP_PALETTE_COLORS;
		p_palette = WSI_SST_BMP_PALETTE_COLORS;
	}
	else if
	(
		((productID >= Min_WaveGrids_MWProductID) && (productID <= Max_WaveGrids_MWProductID)) ||
		((productID >= Min_SurfaceWindGrids_MWProductID) && (productID <= Max_SurfaceWindGrids_MWProductID))
	)
	{
		// Make grayscale palette
		for (numPaletteEntries = 0; numPaletteEntries < 256; numPaletteEntries ++)
		{
			grayscalePalette[numPaletteEntries].r = (TByte_Unsigned) numPaletteEntries;
			grayscalePalette[numPaletteEntries].g = (TByte_Unsigned) numPaletteEntries;
			grayscalePalette[numPaletteEntries].b = (TByte_Unsigned) numPaletteEntries;
			grayscalePalette[numPaletteEntries].a = (TByte_Unsigned) 255; // Opaque
		}

		/* That is now our palette */
		numPaletteEntries = 256; /* redundant assignment, but just in case*/
		p_palette = grayscalePalette;
	}
	else
	{
		/* Not recognized */
		return MWRes_OperationFailed;
	}

	/*
	//##############################################################
	// Create output file
	//##############################################################
	*/

	PitchBytes = numColumns * sizeof(TByte_Unsigned);
	BMPPitchPad = (PitchBytes %4);

	//###############################################################
	// Prepare RLE memory: Should always require at most 2 bytes per
	// pixel, for totally random noise.
	//###############################################################
	p_RLEBytes = malloc (2 * numColumns * numRows * sizeof(TByte_Unsigned));

	if (p_RLEBytes == NULL)
	{
		printf ("Out of memory, failed to allocate RLE byte buffer\n");
		return MWRes_OutOfMemory;
	}

	//###############################################################
	// Do RLE8 encode: See documentation for Windows style bmp files
	// for details of RLE8 format.
	//
	// In a BMP, rows are stored from bottom to top
	//###############################################################
	ROffset = PitchBytes * (numRows-1);
	RLEOffset = 0;
	RunLength = 0;
	CurrentPixelByte = 0;
	for (Row = numRows-1; Row >= 0 ; Row --)
	{
		// Recalculate Row offset
		ROffset = numSourcePitchBytes * Row;

		// Total Row length, for error detection
		TotalRowLength = 0;

		// Number of RLE bytes this row
		RLEBytesThisRow = 0;

		//##############################################################
		// Encode columns
		//##############################################################

		for (Col = 0; Col < numColumns; Col ++)
		{

			//#################################################################
			// Get next source pixel
			//#################################################################


			// Get new high nibble
			NewPixelByte = p_rasterSurface[ROffset];

			// Special case, new run
			if (RunLength == 0)
			{
				CurrentPixelByte = NewPixelByte;
			}

			//#################################################################
			// Bump up run length based on pixel matches
			//#################################################################

			if (CurrentPixelByte == NewPixelByte)
			{
				// If the run would be too long, write it out
				if (RunLength >= 254)
				{
					// Encode this run
					p_RLEBytes[RLEOffset] = RunLength;
					p_RLEBytes[RLEOffset +1] = CurrentPixelByte;
					RLEOffset += 2;

					TotalRowLength += RunLength;
					RunLength = 0;
				}

				// Bump up run length
				RunLength += 1;
			}
			else
			{
				//#################################################################
				// Run is ending so write
				// out the run
				//#################################################################

				// Encode this run
				p_RLEBytes[RLEOffset] = RunLength;
				p_RLEBytes[RLEOffset +1] = CurrentPixelByte;
				RLEOffset += 2;

				TotalRowLength += RunLength;

				// Starting new run with this pixel
				RunLength = 1;
				CurrentPixelByte = NewPixelByte;
			}

			//#################################################################
			// Handle end of run due to end of scan line
			//#################################################################

			if ((numColumns == 1) || (Col >= (TFourByteInteger_Unsigned) (numColumns-1)) )
			{
				// If we have a run, record it
				if (RunLength > 0)
				{
					p_RLEBytes[RLEOffset] = RunLength;

					p_RLEBytes[RLEOffset +1] = CurrentPixelByte;
					RLEOffset += 2;

					TotalRowLength += RunLength;

					// Starting new run, no run yet
					RunLength = 0;
				}


				// Need RLE8 compliant EOL or EOF escape sequence
				if (Row == 0)
				{
					// End of BMP, use EOF sequence
					p_RLEBytes[RLEOffset] = 0;
					p_RLEBytes[RLEOffset+1] = 1;
				}
				else
				{
					// End of row, use EOL sequence
					p_RLEBytes[RLEOffset] = 0;
					p_RLEBytes[RLEOffset+1] = 0;
				}

				RLEOffset +=2;
			}


			//##############################################################
			// Advance offset
			//##############################################################
			ROffset +=1;



		} // Next source column

		//##############################################################
		// Check total row length
		//##############################################################

		if (TotalRowLength != numColumns)
		{
			free (p_RLEBytes);
			printf ("TotalRowLength found was not == numColumns\n");
			return MWRes_ProductContentsInvalid;
		}

	} // Next source row


	/*
	//###############################################################
	// RLEOffset is the number of RLE bytes
	//###############################################################
	*/
	NumRLEBytes = RLEOffset;

	/*
	//##############################################################
	// Open the output file
	//##############################################################
	*/
	p_outFile = fopen (pstr_bitmapFilePath, "wb");
	if (p_outFile == NULL)
	{
		free (p_RLEBytes);
		return MWRes_CommunicationsError;
	}

	/*
	//#################################################################
	// Copy into our palette
	//#################################################################
	*/
	for (i = 0; i < numPaletteEntries; i ++)
	{
		p_outputPalette[i] = p_palette[i];
	}
	for (; i < 256; i ++)
	{
		p_outputPalette[i].b = 194;
		p_outputPalette[i].g = 194;
		p_outputPalette[i].r = 194;
		p_outputPalette[i].a = 0;
	}

	//#################################################################
	// Fill out headers
	//#################################################################

	// Fill out headers
	memset (&BMPHeader, 0, sizeof (BMPHeader));
	BMPHeader.biSize = sizeof (BMPHeader);
	BMPHeader.biWidth = numColumns;
	BMPHeader.biHeight = numRows;
	BMPHeader.biPlanes = 1;
	BMPHeader.biBitCount = 8;
	BMPHeader.biCompression = BI_RLE8;
	BMPHeader.biSizeImage = NumRLEBytes;
	BMPHeader.biClrUsed = 256;
	BMPHeader.biClrImportant = 256;

	// File header
	memset (&FileHeader, 0, sizeof (FileHeader));

	// Set id field to "BM" : requirement
	// Note this is NOT unicode
	((TByte_Unsigned*) &(FileHeader.bfType))[0] = 'B';
	((TByte_Unsigned*) &(FileHeader.bfType))[1] = 'M';

	// Size of file is size of pixel bytes, pallete, and headers
	TotalImageBytes = NumRLEBytes;

	UnpaddedFileSize =
		sizeof (FileHeader) +
		sizeof (BMPHeader) +
		(sizeof (RGBQUAD) * 256) +
		TotalImageBytes;

	PaddingNeeded	= UnpaddedFileSize % 4;

	FileHeader.bfSize = UnpaddedFileSize + PaddingNeeded;
	FileHeader.bfReserved1 = 0;
	FileHeader.bfReserved2 = 0;

	// Set offset to the bits.
	FileHeader.bfOffBits = sizeof (BMPHeader) + (sizeof (RGBQUAD) * 256);

	// Write out the file header and then the BMP header
	numBytesWritten = fwrite (&FileHeader, sizeof(FileHeader), 1, p_outFile);
	if (numBytesWritten != 1)
	{
		if (p_RLEBytes != NULL)
		{
			free(p_RLEBytes);
		}
		if (p_outFile != NULL)
		{
			fclose(p_outFile);;
		}
		printf ("Failed to write BMP File Header to file\n");
		return MWRes_CommunicationsError;
	}

	numBytesWritten = fwrite (&BMPHeader, sizeof(BMPHeader), 1, p_outFile);
	if (numBytesWritten != 1)
	{
		if (p_RLEBytes != NULL)
		{
			free(p_RLEBytes);
		}
		if (p_outFile != NULL)
		{
			fclose(p_outFile);;
		}
		printf ("Failed to write BMP Bitmap Header to file\n");
		return MWRes_CommunicationsError;
	}


	//#################################################################
	// Write out the pallette
	//#################################################################

	numBytesWritten = fwrite (p_outputPalette, sizeof(RGBQUAD), 256, p_outFile);
	if (numBytesWritten != 256)
	{
		if (p_RLEBytes != NULL)
		{
			free(p_RLEBytes);
		}
		if (p_outFile != NULL)
		{
			fclose(p_outFile);;
		}
		printf ("Failed to write BMP Palette to file\n");
		return MWRes_CommunicationsError;
	}

	//#################################################################
	// Write out the run encoded image bytes
	//#################################################################

	numBytesWritten = fwrite
	(
		p_RLEBytes,
		1,
		NumRLEBytes,
		p_outFile
	);
	if (numBytesWritten != NumRLEBytes)
	{
		if (p_RLEBytes != NULL)
		{
			free(p_RLEBytes);
		}
		if (p_outFile != NULL)
		{
			fclose(p_outFile);;
		}
		printf ("Failed to write BMP RLE codes to file\n");
		return MWRes_CommunicationsError;
	}

	//#################################################################
	// Done with RLE bytes
	//#################################################################
	free(p_RLEBytes);
	p_RLEBytes = NULL;

	//#################################################################
	// Write out file size padding
	//#################################################################
	padByte = 0;
	for (i = 0; i < PaddingNeeded; i ++)
	{
		numBytesWritten = fwrite (&padByte, 1, 1, p_outFile);
		if (numBytesWritten != 1)
		{
			if (p_RLEBytes != NULL)
			{
				free(p_RLEBytes);
			}
			if (p_outFile != NULL)
			{
				fclose(p_outFile);;
			}
			printf ("Failed to add required alignment padding to end of file\n");
			return MWRes_CommunicationsError;
		}

	};

	//##############################################################
	// Done with output file
	//##############################################################
	if (p_outFile != NULL)
	{
		fclose(p_outFile);;
	}

	//##############################################################
	// Success
	//##############################################################
	return MWRes_Success;



}

/*
//====================================================================
//====================================================================
// Public interface
//====================================================================
//====================================================================
*/

TMW_ResultCode MWTWSIRLEToBitmap_convertRasterProductFile
(
	const TASCIIChar* pstr_rasterProductPath,
	const TASCIIChar* pstr_outputBitmapFilePath,
	TMW_CompressionType sourceCompressionType,
    TByte_Unsigned planeNumber
)
{
	TMW_ResultCode MWRes;

	/* The number of data planes expected for this ProductID */
	TByte_Unsigned numDataPlanesExpected;

	/* The plane parsing loop*/
	TByte_Unsigned planeIndex;

	/* The amount to leftshift each data value for plane merge */
	TByte_Unsigned dataPlaneLeftShiftNumBits[4];

	/* The source file*/
	FILE* p_RLEFile;

	/* The expected header in the source file*/
	TMW_GenericRasterProductHeader_Output rasterHeader;
	TMW_GenericRasterProductHeader_FixedScaleSpecifier fixedScaleSpecifiers[MAX_PLANES_IN_RASTER];

	/* The number of bytes read */
	TFourByteInteger_Unsigned numBytesRead;

	/* The raster surface memory */
	TByte_Unsigned* p_rasterSurface;
	TFourByteInteger_Unsigned numRasterSurfaceBytes;


	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if ((pstr_rasterProductPath == NULL) || (pstr_outputBitmapFilePath == NULL))
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Open the source file and read in the raster header
	//##############################################################
	*/

	p_RLEFile = fopen (pstr_rasterProductPath, "rb");
	if (p_RLEFile == NULL)
	{
		return MWRes_BadParamValue;
	}

	/* Read the header */
	numBytesRead = 0;
	memset (&rasterHeader, 0, sizeof(rasterHeader));

	numBytesRead = fread
	(
		&rasterHeader,
		1,
		sizeof(rasterHeader),
		p_RLEFile
	);
	if (numBytesRead != sizeof(rasterHeader))
	{
		if (ferror(p_RLEFile))
		{
			fclose (p_RLEFile);
			return MWRes_CommunicationsError;
		}
		else
		{
			fclose (p_RLEFile);
			return MWRes_OperationFailed;
		}
	}

	/* Make sure it is a raster header */
	if (rasterHeader.headerType != GenericRaster_ProductHeaderType)
	{
		fclose (p_RLEFile);
		return MWRes_OperationFailed;
	}

	/* Ensure that the bpp is <= 8 */
	if (rasterHeader.bitsPerPixel > 8)
	{
		fclose (p_RLEFile);
		return MWRes_OperationFailed;
	}

	/*
	//##############################################################
	// Read in the fixed scale specifier for each plane
	//##############################################################
	*/
	if (rasterHeader.numPlanes > MAX_PLANES_IN_RASTER)
	{
		fclose (p_RLEFile);
		return MWRes_ProductHeaderFieldInvalid;
	}

	for (planeIndex = 0; planeIndex < rasterHeader.numPlanes; planeIndex ++)
	{
		/* Clear struct */
		memset
		(
			&fixedScaleSpecifiers[planeIndex],
			0,
			sizeof(TMW_GenericRasterProductHeader_FixedScaleSpecifier)
		);

		/* Read struct */
		numBytesRead = fread
		(
			&fixedScaleSpecifiers[planeIndex],
			1,
			sizeof(TMW_GenericRasterProductHeader_FixedScaleSpecifier),
			p_RLEFile
		);
		if (numBytesRead != sizeof(TMW_GenericRasterProductHeader_FixedScaleSpecifier))
		{
			if (ferror(p_RLEFile))
			{
				fclose (p_RLEFile);
				return MWRes_CommunicationsError;
			}
			else
			{
				fclose (p_RLEFile);
				return MWRes_OperationFailed;
			}
		}

	} /* Next plane fixed-scale specifier */

	/*
	//##############################################################
	// Based on the product ID, determine the expected number of
	// data planes, and the masks and shifts for merging each data
	// plane.
	//##############################################################
	*/

	if
	(
		(rasterHeader.productID == CompositePrecipNEXRAD_MWProductID)
	)
	{
		/*
		//##############################################################
		// There are expected to be 2 planes
		//##############################################################
		*/
		if (rasterHeader.numPlanes == 2)
		{
			numDataPlanesExpected = 2;

			/* First plane is precip intensity and does not shift */
			dataPlaneLeftShiftNumBits[0] = 0;

			/* Second plane is intensity and shifts to the left by 4 bytes */
			dataPlaneLeftShiftNumBits[1] = 4;
		}
		else if (rasterHeader.numPlanes == 1)
		{
			numDataPlanesExpected = 1;

			/* Only one plane, holds all bits */
			dataPlaneLeftShiftNumBits[0] = 0;

		}
		else
		{
			fclose (p_RLEFile);
			return MWRes_ProductHeaderFieldInvalid;
		}
	}
	else if
	(
		(rasterHeader.productID >= Min_SurfaceWindGrids_MWProductID) && (rasterHeader.productID <= Max_SurfaceWindGrids_MWProductID)
	)
	{
		/*
		//##############################################################
		// There are expected to be 2
		//##############################################################
		*/
		numDataPlanesExpected = 2;
		
        if (planeNumber == 0)
        {
            dataPlaneLeftShiftNumBits[0] = 0;
            dataPlaneLeftShiftNumBits[1] = 8;
        }
        else if (planeNumber == 1)
        {
            dataPlaneLeftShiftNumBits[0] = 8;
            dataPlaneLeftShiftNumBits[1] = 0;
        }
        else if (planeNumber == 2)
        {
            dataPlaneLeftShiftNumBits[0] = 0;
            dataPlaneLeftShiftNumBits[1] = 0;
        }
	}
	else if
	(
		(rasterHeader.productID == Canadian_RADAR_MWProductID) ||
		(rasterHeader.productID == SeaSurfaceTemperatures_MWProductID) ||
		((rasterHeader.productID >= Min_WaveGrids_MWProductID) && (rasterHeader.productID <= Max_WaveGrids_MWProductID))
	)
	{
		/*
		//##############################################################
		// There is only 1 plane, with no shift
		//##############################################################
		*/
		numDataPlanesExpected = 1;
		dataPlaneLeftShiftNumBits[0] = 0;

	}
	else
	{
		/* Not recognized */
		fclose (p_RLEFile);
		return MWRes_OperationFailed;
	}

	/*
	//##############################################################
	// Does it have the expected number of planes?
	//##############################################################
	*/
	if (rasterHeader.numPlanes != numDataPlanesExpected)
	{
		fclose (p_RLEFile);
		return MWRes_ProductHeaderFieldInvalid;
	}

	/*
	//##############################################################
	// Create memory to hold result for merging planes
	//##############################################################
	*/
	numRasterSurfaceBytes = rasterHeader.numColumns * rasterHeader.numRows; /* No extra pitch */
	p_rasterSurface = malloc (numRasterSurfaceBytes);
	if (p_rasterSurface == NULL)
	{
		fclose (p_RLEFile);
		return MWRes_OutOfMemory;
	}

	/* Clear raster memory to all 0's */
	memset (p_rasterSurface, 0, numRasterSurfaceBytes);

	/*
	//##############################################################
	// Read in and merge each plane
	//##############################################################
	*/

	for (planeIndex = 0; planeIndex < rasterHeader.numPlanes; planeIndex ++)
	{
		if (sourceCompressionType == WSI_RLE_MWCompressionType)
		{
			MWRes = MWTWSIRLEToBMP_Internal_processRasterPlane_RLE
			(
				rasterHeader.numColumns,
				rasterHeader.numRows,
				rasterHeader.numColumns, /* Pitch same as width */
				p_rasterSurface,
				p_RLEFile,
				dataPlaneLeftShiftNumBits[planeIndex]
			);
			if (MWRes != MWRes_Success)
			{
				fclose(p_RLEFile);
				free (p_rasterSurface);
				return MWRes;
			}
		}
		else if (sourceCompressionType == None_MWCompressionType)
		{
			MWRes = MWTWSIRLEToBMP_Internal_processRasterPlane_Raw
			(
				rasterHeader.numColumns,
				rasterHeader.numRows,
				rasterHeader.numColumns, /* Pitch same as width */
				p_rasterSurface,
				p_RLEFile,
				dataPlaneLeftShiftNumBits[planeIndex]
			);
			if (MWRes != MWRes_Success)
			{
				fclose(p_RLEFile);
				free (p_rasterSurface);
				return MWRes;
			}
		}
		else
		{
			return MWRes_BadParamValue;
		}
	}

	/*
	//##############################################################
	// Done with input
	//##############################################################
	*/
	fclose (p_RLEFile);


	/*
	//##############################################################
	// Use internal method to write out image using the merged
	// plane
	//##############################################################
	*/

	MWRes = MWTWSIRLEToBMP_Internal_write8BPPBitmapFile
	(
		rasterHeader.numColumns,
		rasterHeader.numRows,
		rasterHeader.numColumns, /* Pitch same as width */
		p_rasterSurface,
		pstr_outputBitmapFilePath,
		rasterHeader.productID
	);

	/*
	//##############################################################
	// Done with raster surface
	//##############################################################
	*/
	free (p_rasterSurface);

	/*
	//##############################################################
	// Return result
	//##############################################################
	*/
	return MWRes;

}
